﻿// *************************************************************************************************************************
// DivObj (Javascript Object)
//
// Wrapper around a 'div' HTML Element.  Provides getters and setters for formatting a div object.  
// To add a raw HTML Element to a DivObj, call addObject():
//
// var divWrapper = new DivObj('ID');
// var newElement = document.createElement('div');
// divWrapper.addObject(newElement);
//
// To add a wrapper object to a DivObj, call draw() on the warpper object and give DivObj.getElement() as the parameter:
//
// var divWrapper = new DivObj('ID');
// var otherWrapper = new DivObj('OtherID');
// otherWrapper.draw(divWrapper.getElement());
//
// To add a DivObj to a raw HTML Element, call the draw() function:
//
// var parentElement = document.getElementById('ExistingElement');
// var divWrapper = new DivObj('ID');
// divWrapper.draw(parentElement);
//
// *************************************************************************************************************************

// Dependencies:
/// <reference path="uicore.js" />

DivObj = function(id)
{
    /// <summary>Constructor for DivObj</summary>
    /// <param name="id">[OPTIONAL] (string) unique id for this element. If null, generates random ID</param>
    /// <returns>A new DivObj</returns>
    
    this.m_strID = (id != undefined) ? id : 'Div' + UIHelper.generateRandomString(10);
    
    this.m_bVisible = true;
    this.m_bHidden = false;
    this.m_bEnabled = true;
    
    // Elements
    this.m_elContainer = null;
    this.m_elDiv = document.createElement('div');
    
    this.m_elDiv.id = this.m_strID;
}

DivObj.prototype =
{
    draw:function(elementDiv)
    {
        /// <summary>Calls appendChild() on the provided element, appending this div element</summary>
        /// <param name="elementDiv">(HTML Element) HTML element to append this div to</param>
        if(elementDiv != undefined)
        {
            this.m_elContainer = elementDiv;

            this.m_elContainer.appendChild(this.m_elDiv);
        }
    },
    
    addObject:function(element)
    {
        /// <summary>Adds provided HTML element to this div</summary>
        /// <param name="element">(HTML Element) HTML element to append to this div</param>
        if((element != undefined) && (this.m_elDiv != undefined))
            this.m_elDiv.appendChild(element);
    },
    
    getElement:function()
    {
        /// <summary>Returns this 'div' HTML Element</summary>
        return this.m_elDiv;
    },
    
    setProp:function(valueName, value)
    {
        /// <summary>Sets the provided property name to the provided value</summary>
        /// <param name="valueName">(string) Name of the property to change</param>
        /// <param name="value">(string/int) New value of the property</param>
        if((valueName != undefined) && (typeof valueName == 'string'))
        {
            valueName = valueName.toLowerCase();
            switch(valueName)
            {
                case 'float':
                case 'stylefloat':
                    this.setFloat(value);
                    break;
                case 'height':
                    this.setHeight(value);
                    break;
                case 'width':
                    this.setWidth(value);
                    break;
                case 'id':
                    this.setID(value);
                    break;
                case 'overflow':
                    this.setOverflow(value);
                    break;
                case 'classname':
                    this.setClassName(value);
                    break;
                case 'html':
                case 'innerhtml':
                    this.setHTML(value);
                    break;
                case 'text':
                case 'innertext':
                    this.setText(value);
                    break;
                case 'position':
                    this.setPosition(value);
                    break;
                case 'filter':
                    this.setFilter(value);
                    break;
                case 'cursor':
                    this.setCursor(value);
                    break;
                case 'align':
                    this.setAlign(value);
                    break;
                case 'bgcolor':
                case 'backgroundcolor':
                    this.setBGColor(value);
                    break;
                case 'bordercolor':
                    this.setBorderColor(value);
                    break;
                case 'tabindex':
                    this.setTabIndex(value);
                    break;
            }
        }
    },
    
    setTabIndex:function(value)
    {
        /// <summary>Sets the tab index of this div</summary>
        /// <param name="value">(int) tab index of this div</param>
        if((this.m_elDiv != undefined) && (value != undefined))
            this.m_elDiv.tabIndex = value;
    },
    
    getTabIndex:function()
    {
        /// <summary>Returns the tabIndex value of this div</summary>
        if(this.m_elDiv != undefined)
            return this.m_elDiv.tabIndex;
        return 'undefined';
    },
    
    setFloat:function(floatValue)
    {
        /// <summary>Sets the style.float property</summary>
        /// <param name="floatValue">(string) [none] | [left] | [right]</param>
        if((this.m_elDiv != undefined) && (floatValue != undefined))
            this.m_elDiv.style.styleFloat = floatValue;
    },
    
    getFloat:function()
    {
        /// <summary>Returns the style.float value of this div</summary>
        var toReturn = 'undefined';
        if(this.m_elDiv != undefined)
            toReturn = this.m_elDiv.style.styleFloat;
        return toReturn;
    },
    
    setCursor:function(value)
    {
        /// <summary>Sets the style.cursor property</summary>
        /// <param name="value">(string) [all-scroll] | [auto] | [col-resize] | [crosshair] | [default] | 
        /// [hand] | [help] | [move] | [no-drop] | [not-allowed] | [pointer] | [progress] | [row-resize] |
        /// [text] | [url(uri)] | [vertical-text] | [wait] | [*-resize]</param>
        if((this.m_elDiv != undefined) && (value != undefined))
            this.m_elDiv.style.cursor = value;
    },
    
    getCursor:function()
    {
        /// <summary>Returns the style.cursor value of this div</summary>
        var toReturn = 'undefined';
        if(this.m_elDiv != undefined)
            toReturn = this.m_elDiv.style.cursor;
        return toReturn;
    },
    
    setHeight:function(value)
    {
        /// <summary>Sets the style.height property</summary>
        /// <param name="value">(int/string) height</param>
        if((this.m_elDiv != undefined) && (value != undefined))
            this.m_elDiv.style.height = value;
    },
    
    getHeight:function()
    {
        /// <summary>Returns the style.height value of this div</summary>
        var toReturn = 'undefined';
        if(this.m_elDiv != undefined)
            toReturn = this.m_elDiv.style.height
        return toReturn;
    },
    
    setWidth:function(value)
    {
        /// <summary>Sets the style.width property</summary>
        /// <param name="value">(int/string) width</param>
        if((this.m_elDiv != undefined) && (value != undefined))
            this.m_elDiv.style.width = value;
    },
    
    getWidth:function()
    {
        /// <summary>Returns the style.width value of this div</summary>
        var toReturn = 'undefined';
        if(this.m_elDiv != undefined)
            toReturn = this.m_elDiv.style.width
        return toReturn;
    },
    
    setAllPadding:function(top, right, bottom, left)
    {
        /// <summary>Sets the padding on all sides of this div</summary>
        /// <param name="top">(int/string) top padding</param>
        /// <param name="right">(int/string) right padding</param>
        /// <param name="bottom">(int/string) bottom padding</param>
        /// <param name="left">(int/string) left padding</param>
        if(this.m_elDiv != undefined)
        {
            this.m_elDiv.style.paddingTop = top;
            this.m_elDiv.style.paddingRight = right;
            this.m_elDiv.style.paddingBottom = bottom;
            this.m_elDiv.style.paddingLeft = left;
        }
    },
    
    setPadding:function(dimension, value)
    {
        /// <summary>Sets the padding for the specified side of this div</summary>
        /// <param name="dimension">(string) [top] | [right] | [bottom] | [left]</param>
        /// <param name="value">(int/string) padding</param>
        if((this.m_elDiv != undefined) && (dimension != undefined) && (typeof dimension == 'string') && (value != undefined))
        {
            dimension = dimension.toLowerCase();
            switch(dimension)
            {
                case 'top':
                    this.m_elDiv.style.paddingTop = value;
                    break;
                case 'bottom':
                    this.m_elDiv.style.paddingBottom = value;
                    break;
                case 'right':
                    this.m_elDiv.style.paddingRight = value;
                    break;
                case 'left':
                    this.m_elDiv.style.paddingLeft = value;
                    break;
            }
        }
    },
    
    getPadding:function(dimension)
    {
        /// <summary>Returns the padding of the provided side in this div</summary>
        /// <param name="dimension">(string) [top] | [right] | [bottom] | [left]</param>
        var toReturn = 'undefined';
        if((this.m_elDiv != undefined) && (dimension != undefined) && (typeof dimension == 'string'))
        {
            dimension = dimension.toLowerCase();
            switch(dimension)
            {
                case 'top':
                    toReturn = this.m_elDiv.style.paddingTop;
                    break;
                case 'bottom':
                    toReturn = this.m_elDiv.style.paddingBottom;
                    break;
                case 'right':
                    toReturn = this.m_elDiv.style.paddingRight;
                    break;
                case 'left':
                    toReturn = this.m_elDiv.style.paddingLeft;
                    break;
            }
        }
        return toReturn;
    },
    
    setID:function(value)
    {
        /// <summary>Sets the id of this div</summary>
        /// <param name="value">(string) new ID</param>
        if((this.m_elDiv != undefined) && (value != undefined))
        {
            this.m_elDiv.id = value;
            this.m_strID = value;
        }
    },
    
    getID:function()
    {
        /// <summary>Returns the id of this div</summary>
        return this.m_strID;
    },
    
    setOverflow:function(value)
    {
        /// <summary>Sets the style.overflow property</summary>
        /// <param name="value">(string) [visible] | [scroll] | [hidden] | [auto]</param>
        if((this.m_elDiv != undefined) && (value != undefined))
            this.m_elDiv.style.overflow = value;
    },
    
    getOverflow:function()
    {
        /// <summary>Returns the style.overflow value of this div</summary>
        var toReturn = 'undefined';
        if(this.m_elDiv != undefined)
            toReturn = this.m_elDiv.style.overflow;
        return toReturn;
    },
    
    setClassName:function(value)
    {
        /// <summary>Sets the className property</summary>
        /// <param name="value">(string) new className</param>
        if((this.m_elDiv != undefined) && (value != undefined))
            this.m_elDiv.className = value;
    },
    
    getClassName:function()
    {
        /// <summary>Returns the class name of this div</summary>
        var toReturn = 'undefined';
        if(this.m_elDiv != undefined)
            toReturn = this.m_elDiv.className;
        return toReturn;
    },
    
    setHTML:function(value)
    {
        /// <summary>Sets the innerHTML property</summary>
        /// <param name="value">(string) new innerHTML</param>
        if((this.m_elDiv != undefined) && (value != undefined))
            this.m_elDiv.innerHTML = value;
    },
    
    getHTML:function()
    {
        /// <summary>Returns the innerHTML of this div</summary>
        var toReturn = 'undefined';
        if(this.m_elDiv != undefined)
            toReturn = this.m_elDiv.innerHTML;
        return toReturn;
    },
    
    setText:function(value)
    {
        /// <summary>Sets the innerText property</summary>
        /// <param name="value">(string) new innerText</param>
        if((this.m_elDiv != undefined) && (value != undefined))
            this.m_elDiv.innerText = value;
    },
    
    getText:function()
    {
        /// <summary>Returns the innerText of this div</summary>
        var toReturn = 'undefined';
        if(this.m_elDiv != undefined)
            toReturn = this.m_elDiv.innerText;
        return toReturn;
    },
    
    addBreak:function()
    {
        /// <summary>Adds a 'br' HTML Element to this div</summary>
        if(this.m_elDiv)
            this.m_elDiv.appendChild(document.createElement('br'));
    },
    
    setPosition:function(value)
    {
        /// <summary>Sets the style.position property</summary>
        /// <param name="value">(string) [static] | [absolute] | [relative] | [fixed]</param>
        if((this.m_elDiv != undefined) && (value != undefined))
            this.m_elDiv.style.position = value;
    },
    
    getPosition:function()
    {
        /// <summary>Returns the style.position value of this div</summary>
        var toReturn = 'undefined';
        if(this.m_elDiv != undefined)
            toReturn = this.m_elDiv.style.position;
        return toReturn;
    },
    
    setFilter:function(value)
    {
        /// <summary>Sets the style.filter property</summary>
        /// <param name="value">(string) new filter</param>
        if((this.m_elDiv != undefined) && (value != undefined))
            this.m_elDiv.style.filter = value;
    },
    
    getFilter:function()
    {
        /// <summary>Returns the style.filter value of this div</summary>
        var toReturn = 'undefined';
        if(this.m_elDiv != undefined)
            toReturn = this.m_elDiv.style.filter;
        return toReturn;
    },
    
    getFilters:function()
    {
        /// <summary>Returns the filters array of this div</summary>
        var toReturn = new Array();
        if(this.m_elDiv != undefined)
            toReturn = this.m_elDiv.filters;
        return toReturn;
    },
    
    setAlign:function(value)
    {
        /// <summary>Sets the align property</summary>
        /// <param name="value">(string) [center] | [justify] | [left] | [right]</param>
        if((this.m_elDiv != undefined) && (value != undefined))
            this.m_elDiv.align = value;
    },
    
    getAlign:function()
    {
        /// <summary>Returns the align value of this div</summary>
        var toReturn = 'undefined';
        if(this.m_elDiv != undefined)
            toReturn = this.m_elDiv.align;
        return toReturn;
    },
    
    setBGColor:function(value)
    {
        /// <summary>Sets the style.backgroundColor property</summary>
        /// <param name="value">(string) new color</param>
        if((this.m_elDiv != undefined) && (value != undefined))
            this.m_elDiv.style.backgroundColor = value;
    },
    
    getBGColor:function()
    {
        /// <summary>Returns the style.backgroundColor value of this div</summary>
        var toReturn = 'undefined';
        if(this.m_elDiv != undefined)
            toReturn = this.m_elDiv.style.backgroundColor;
        return toReturn;
    },
    
    setBorderColor:function(value)
    {
        /// <summary>Sets the style.borderColor property</summary>
        /// <param name="value">(string) new color</param>
        if((this.m_elDiv != undefined) && (value != undefined))
            this.m_elDiv.style.borderColor = value;
    },
    
    getBorderColor:function()
    {
        /// <summary>Returns the style.borderColor value of this div</summary>
        var toReturn = 'undefined';
        if(this.m_elDiv != undefined)
            toReturn = this.m_elDiv.style.borderColor;
        return toReturn;
    },
    
    setOffset:function(dimension, value)
    {
        /// <summary>Sets the offest property for the provided side</summary>
        /// <param name="dimension">(string) [top] | [bottom] | [left] | [right]</param>
        /// <param name="value">(int/string) offest value</param>
        if((this.m_elDiv != undefined) && (dimension != undefined) && (typeof dimension == 'string') && (value != undefined))
        {
            dimension = dimension.toLowerCase();
            switch(dimension)
            {
                case 'top':
                    this.m_elDiv.style.top = value;
                    break;
                case 'bottom':
                    this.m_elDiv.style.bottom = value;
                    break;
                case 'left':
                    this.m_elDiv.style.left = value;
                    break;
                case 'right':
                    this.m_elDiv.style.right = value;
                    break;
            }
        }
    },
    
    getOffset:function(dimension)
    {
        /// <summary>Returns the offset value of the provided side of this div</summary>
        /// <param name="dimension">(string) [top] | [right] | [bottom] | [left]</param>
        var toReturn = 'undefined';
        if((this.m_elDiv != undefined) && (dimension != undefined) && (typeof dimension == 'string'))
        {
            dimension = dimension.toLowerCase();
            switch(dimension)
            {
                case 'top':
                    toReturn = this.m_elDiv.style.top;
                    break;
                case 'bottom':
                    toReturn = this.m_elDiv.style.bottom;
                    break;
                case 'left':
                    toReturn = this.m_elDiv.style.left;
                    break;
                case 'right':
                    toReturn = this.m_elDiv.style.right;
                    break;
            }
        }
        return toReturn;
    },
    
    isVisible:function()
    {
        /// <summary>Returns the visible state of this div</summary>
        return this.m_bVisible;
    },
    
    hide:function(complete)
    {
        /// <summary>Hides this div, either through style.display or style.visibility property</summary>
        /// <param name="complete">(bool) True = Set style.display to 'none', False = Set style.visibility to 'hidden'</param>
        if(this.m_elDiv)
        {
            this.m_bVisible = false;
            if(complete)
            {
                this.m_elDiv.style.display = 'none';
                this.m_bHidden = true;
            }
            else
            {
                this.m_elDiv.style.visibility = 'hidden';
                this.m_bHidden = false;
            }
        }
    },
    
    show:function()
    {
        /// <summary>Makes the div visible (does the opposite of the previous hide() call)</summary>
        if((this.m_elDiv != undefined) && !this.m_bVisible)
        {
            if(this.m_bHidden)
                this.m_elDiv.style.display = 'block';
            else
                this.m_elDiv.style.visibility = 'visible';
            this.m_bVisible = true;
            this.m_bHidden = false;
        }
    },
    
    isEnabled:function()
    {
        /// <summary>Returns the enabled state of this div</summary>
        return this.m_bEnabled;
    },
    
    disable:function()
    {
        /// <summary>Disables this div</summary>
        if((this.m_elDiv != undefined) && this.m_bEnabled)
        {
            this.m_elDiv.disabled = true;
            this.m_bEnabled = false;
        }
    },
    
    enable:function()
    {
        /// <summary>Enables this div</summary>
        if((this.m_elDiv != undefined) && !this.m_bEnabled)
        {
            this.m_elDiv.disabled = false;
            this.m_bEnabled = true;
        }
    }
}

