﻿// *************************************************************************************************************************
// TextObj (Javascript Object)
//
// A TextObj is a wrapper around a 'span' HTML element.  The ID, Text, and Style Sheet Class Name are specified
// through the constructor
// To add the text element to an element, call the draw() function providing the actual HTML element to which
// appenChild() will be called to attach the text.
// *************************************************************************************************************************

// Dependencies:
/// <reference path="uicore.js">

TextObj = function(uniqueID, text, className, forceWrap)
{
    /// <summary>Constructor for TextObj.  [Text = 'span' HTML Element]</summary>
    /// <param name="uniqueID">[OPTIONAL] (string) Actual ID of this element.  If null, generates random unique ID.</param>
    /// <param name="text">(string) Text to display in this element</param>
    /// <param name="className">[OPTIONAL] (string) Style Sheet class name for this 'span' element</param>
    /// <returns>A new TextObj</returns>
    this.m_strID = (uniqueID != undefined) ? uniqueID : 'TextObj' + UIHelper.generateRandomString(10);
    this.m_strText = text;
    this.m_strClassName = className;
    this.m_strTitle = null;    
    this.m_bVisible = true;
    
    // Elements
    this.m_elContainer = null;
    this.m_elText = null;
    if ( null == forceWrap || undefined == forceWrap )
        this.m_elText = document.createElement('span');
    else
    {
        this.m_elText = document.createElement('textArea');
        this.m_elText.readOnly = true;
    }
        
    this.m_elText.id = this.m_strID;
    if((this.m_strClassName != undefined) && (this.m_strClassName.length > 0))
        this.m_elText.className = this.m_strClassName;
 
    this.m_elText.innerText = this.m_strText;
}

TextObj.prototype = 
{
	draw:function(elementDiv)
	{
	    /// <summary>Calls appendChild() on the provided element, appending this span element</summary>
        /// <param name="elementDiv">(HTML Element) HTML element to append this span to</param>
	    if(elementDiv != undefined)
	    {
	        this.m_elContainer = elementDiv;
	        
	        this.m_elContainer.appendChild(this.m_elText);
	    }
	},
	
	setID:function(value)
	{
	    /// <summary>Sets the ID for this 'span' element</summary>
        /// <param name="value">(string) New ID for this element</param>
	    if((this.m_elText != undefined) && (value != undefined))
	    {
	        this.m_strID = value;
	        this.m_elText.id = this.m_strID;
	    }
	},
	
	getID:function()
	{
	    /// <summary>Gets the ID for this 'span' element</summary>
	    return this.m_strID;
	},
	
	setFloat:function(value)
	{
	    /// <summary>Sets the style.float property for this 'span' element</summary>
        /// <param name="value">(string) [none] | [left] | [right]</param>
	    if((this.m_elText != undefined) && (value != undefined))
	        this.m_elText.style.styleFloat = value;
	},

	setColor:function(value)
	{
	    /// <summary>Sets the style.color property for this 'span' element</summary>
            /// <param name="value">(string) new color string</param>
	    if((this.m_elText != undefined) && (value != undefined))
	        this.m_elText.style.color = value;
	},
    
	setClassName:function(value)
	{
	    if(this.m_elText && value!=null)
	    {
	        this.m_strClassName = value;
	        this.m_elText.className = this.m_strClassName;
	    }
	},

        getElement:function()
        {
            /// <summary>Returns the actual 'input' HTML element in this TextBoxObj</summary>
            return this.m_elText;
        },	
	getText:function()
	{
	    /// <summary>Gets the inner text for this 'span' element</summary>
	    return this.m_strText;
	},
	
	setText:function(text, maxChars)
	{
	    /// <summary>Sets the inner text for this 'span' element</summary>
        /// <param name="text">(string) New inner text for this element</param>
            if ( null == text || undefined == text )
                this.m_strText = '';
            else
                this.m_strText = text;
                
            if(this.m_elText != undefined)
            {
                if ( null == maxChars || undefined == maxChars )
                    this.m_elText.innerText = this.m_strText;
                else
                    this.m_elText.innerText = UIHelper.truncateName( this.m_strText, maxChars );
            }
	},
	
	hide:function()
    {
        /// <summary>Hides this 'span' element (sets style.display = 'none')</summary>
        if(this.m_elText != undefined)
        {
            this.m_elText.style.display = 'none';
            this.m_bVisible = false;
        }
    },
    
    setTitle:function(value)
    {
        /// <summary>Sets the title for this 'span' element</summary>
        /// <param name="value">(string) New title for this element</param>
        if((this.m_elText != undefined) && (value != undefined))
        {
            this.m_strTitle = value;        
            this.m_elText.title = value;
        }
    },
    
    setWordWrap:function(value)
    {
        /// <summary>Sets the style.wordWrap property for this 'span' element</summary>
            /// <param name="value">(string) new wordwrap value</param>
        if((this.m_elText != undefined) && (value != undefined))
            this.m_elText.style.wordWrap = value;
    },
    
    show:function()
    {
        /// <summary>Makes this 'span' element visible (set style.display = 'block')</summary>
        if((this.m_elText != undefined) && !this.m_bVisible)
        {
            this.m_elText.style.display = 'block';
            this.m_bVisible = true;
        }
    }
}
