﻿// *************************************************************************************************************************
// TextBoxObj (Javascript Object)
//
// A TextBoxObj is a wrapper around a 'input' HTML element.  This can also be a password text box.
// The ID, Style Sheet class name, initial text, and enabled state are all set through the constructor.
// The textbox also stores its original value, and tracks if the user has ever modifieid its text.
// To add the textbox element to another element, call the draw() function providing the actual HTML element
// to which appenChild() will be called to attach the textbox
// *************************************************************************************************************************

// Dependencies:
/// <reference path="uicore.js">

TextBoxObj = function(uniqueID, isPassword, className, text, enabled, cobrandObj, callback, outfocuscallback, infocuscallback, capslocknotify)
{
    /// <summary>Constructor for TextBoxObj.  [TextBox = 'input' HTML Element (for password type='password')]</summary>
    /// <param name="uniqueID">[OPTIONAL] (string) Actual ID of this element.  If null, generates random unique ID.</param>
    /// <param name="isPassword">(bool) True = Password Textbox, False = Regular Textbox</param>
    /// <param name="className">[OPTIONAL] (string) Style Sheet class name for this 'input' element</param>
    /// <param name="text">(string) Text to display initially in this element</param>
    /// <param name="enabled">(bool) True = Textbox is initially enabled, False = Not enabled</param>
    /// <returns>A new TextBoxObj</returns>
    this.m_objEx = new ExternalWrapperObj();
    this.m_strID = (uniqueID != undefined) ? uniqueID : 'TextBoxObj' + UIHelper.generateRandomString(10);
    this.m_bPassword = isPassword;
    this.m_strClassName = className;
    this.m_strText = (text != undefined) ? text : '';
    this.m_bEnabled = enabled;
    this.m_fnCallback = (callback!=undefined) ? callback : null;
    this.m_fnoutfocusCallback = (outfocuscallback!=undefined && outfocuscallback!=null) ? outfocuscallback : null;
    this.m_fninfocusCallback = (infocuscallback!=undefined && infocuscallback!=null) ? infocuscallback : null;
    this.m_divCapsLocknotify = (capslocknotify!=undefined && capslocknotify!=null )? capslocknotify : null;
    
    // Member vars
    this.m_strOriginalText = this.m_strText;
    this.m_bChanged = false;
    this.m_bVisible = true;
    // Elements
    this.m_elContainer = null;
    this.m_elTextBox = null;
 
    // Create textbox
    this.m_elTextBox = document.createElement('input');
    this.m_elTextBox.id = this.m_strID;
    if(this.m_bPassword)
        this.m_elTextBox.type = 'password';
    this.m_elTextBox.value = this.m_strText;
    if((this.m_strClassName != undefined) && (this.m_strClassName.length > 0))
        this.m_elTextBox.className = this.m_strClassName;
    if(!this.m_bEnabled)
        this.m_elTextBox.disabled = true;
    this.m_elTextBox.autocomplete = true;

    // render cobranding
    if ( cobrandObj!= undefined && cobrandObj!=null )
    {
        if ( cobrandObj.m_strFieldBdColor.length > 0 )
        {
            this.m_elTextBox.style.borderColor = cobrandObj.m_strFieldBdColor;
        }
    }
    
}

TextBoxObj.prototype = 
{
    draw:function(elementDiv)
    {
        /// <summary>Calls appendChild() on the provided element, appending this input element</summary>
        /// <param name="elementDiv">(HTML Element) HTML element to append this texbox to</param>
        if(elementDiv != undefined)
        {
            var self = this;
            
            this.m_elContainer = elementDiv;

            this.m_elTextBox.onkeypress = function() {self.keyPressed()};            
            this.m_elTextBox.oncontextmenu = function() { return false; };
            if (this.m_fnCallback != null) {
              this.m_elTextBox.onkeydown= function() {self.checkEnterPress()};
            }
            this.m_elTextBox.onkeyup=function() {self.checkKeyUp()}
            this.m_elTextBox.onfocusin = function() {self.inFocus()}
            this.m_elTextBox.onfocusout=function() {self.outFocus()}
            this.m_elContainer.appendChild(this.m_elTextBox);
        }
    },
    
    getElement:function()
    {
        /// <summary>Returns the actual 'input' HTML element in this TextBoxObj</summary>
        return this.m_elTextBox;
    },
    
    setTabIndex:function(value)
    {
        /// <summary>Sets the tab index of this textbox</summary>
        /// <param name="value">(int) Tab index of this textobx</param>
        if((this.m_elTextBox != undefined) && (value != undefined))
            this.m_elTextBox.tabIndex = value;
    },

    setMaxLength:function(value)
    {
        /// <summary>Sets max length of this input</summary>
        /// <param name="value">(int) max length of this input</param>
        if((this.m_elTextBox != undefined) && (value != undefined))
            this.m_elTextBox.maxLength = value;
    },
    
    keyPressed:function()
    {
        this.changed();
    },
    
    changed:function()
    {
        /// <summary>Sets the 'changed' state of this textbox to true</summary>
        this.m_bChanged = true;
    },
    capsLockOnNotify:function( on )
    {
        if ( this.m_divCapsLocknotify != null )
        {
            if ( on && !this.m_divCapsLocknotify.isVisible() )
            {
                this.m_divCapsLocknotify.show();
                this.m_objEx.adjustWindowSize( null, null, true);                        
            } else if ( !on && this.m_divCapsLocknotify.isVisible() )
            {
                this.m_divCapsLocknotify.hide(false);
                this.m_objEx.adjustWindowSize( null, null, true);                        
            }
        }
    },
    checkKeyUp:function()
    {
        var caplockon = this.m_objEx.isCapsLockOn();
        this.capsLockOnNotify( caplockon );
    },
    outFocus:function()
    {
        this.capsLockOnNotify( false );    
        
        if ( this.m_fnoutfocusCallback != null )
            this.m_fnoutfocusCallback();
    },
    inFocus:function()
    {
        var caplockon = this.m_objEx.isCapsLockOn();
        this.capsLockOnNotify( caplockon );
        if ( this.m_fninfocusCallback != null )
            this.m_fninfocusCallback();
    },
    checkEnterPress:function()
    {
        if(window.event != undefined)
        {
            var keycode = window.event.keyCode;
            if((keycode == KEY_CODE_ENTER) && (this.m_fnCallback != null))
            {
                window.event.returnValue = false;
                this.m_fnCallback();
            }
        }

    },
    
    hasBeenTyped:function()
    {
        /// <summary>Returns true if the textbox text has ever been changed, false otherwise</summary>
        return this.m_bChanged;
    },
    
    textIsDifferent:function()
    {
        /// <summary>Returns true if the current text is different than the initial text, false otherwise</summary>
        if((this.m_elTextBox != undefined) && (this.m_elTextBox.value != this.m_strOriginalText))
            return true;
        return false;
    },
    
    getText:function()
    {
        /// <summary>Returns the current text in this textbox</summary>
        if(this.m_elTextBox != undefined)
            return this.m_elTextBox.value;
        return null;
    },

    setText:function(newValue)
    {
        if ( null == newValue || undefined == newValue )
            return;
        if(this.m_elTextBox != undefined)
             this.m_elTextBox.value = newValue;
    },
    
    disable:function()
    {
        /// <summary>Disables this textbox</summary>
        this.m_bEnabled = false;
        this.updateState();
    },
    
    enable:function()
    {
        /// <summary>Enables this textbox</summary>
        this.m_bEnabled = true;
        this.updateState();
    },
    isEnable:function()
    {
        return this.m_bEnabled;
    },
    updateState:function()
    {   
        /// <summary>Syncs the 'disabled' state of the 'input' HTML element with the state of the TextBoxObj</summary>
        if(this.m_elTextBox != undefined)
            this.m_elTextBox.disabled = !this.m_bEnabled;
    },
    
    isVisible:function()
    {
        return this.m_bVisible;
    },
    
    hide:function()
    {
        /// <summary>Hides this 'span' element (sets style.display = 'none')</summary>
        if(this.m_elTextBox != undefined)
        {
            this.m_elTextBox.style.display = 'none';
            this.m_bVisible = false;
        }
    },
    select:function()
    {
        if(this.m_elTextBox != undefined)
        {
            this.m_elTextBox.select();
        }
    },
    show:function()
    {
        /// <summary>Makes this 'span' element visible (set style.display = 'block')</summary>
        if((this.m_elTextBox != undefined) && !this.m_bVisible)
        {
            this.m_elTextBox.style.display = 'block';
            this.m_bVisible = true;
        }
    }}

