﻿// *************************************************************************************************************************
// ButtonObj (Javascript Object)
//
// A ButtonObj is a wrapper around a div element, which has 3 apperance states (standard, clicked, disabled).
// The ID, Text, Height, Width, Text Padding, enabled state, and call back function are specified through the constructor.
// The callback function is called when the button had both been pushed down and up in the same mouse click.
// To add the button to the form, call the draw() function providing the actual HTML element to which appendChild() will
// be called to attach this button
// *************************************************************************************************************************

// Dependencies:
/// <reference path="uicore.js">

ButtonObj = function(uniqueID, text, height, width, textTopPadding, classNamePrefix, enabled, cobrandObj, callback)
{
    /// <summary>Constructor for ButtonObj.  [Button = 'div' HTML Element]  [Button Text = 'TextObj' Wrapper]</summary>
    /// <param name="uniqueID">[OPTIONAL] (string) ID of this element.  If null, generates random unique ID. (Button ID = 'ButtonObj[uniqueID]')</param>
    /// <param name="text">(string) Text to display in the button</param>
    /// <param name="height">[OPTIONAL] (string/int) HTML height of this button</param>
    /// <param name="width">[OPTIONAL] (string/int) HTML width of this button</param>
    /// <param name="textTopPadding">[OPTIONAL] (string/int) HTML spacing between the text and the top border of the button</param>
    /// <param name="classNamePrefix">[OPTIONAL] (string) Style Sheet class name prefix for this button (Default: 'combobox')</param>
    /// <param name="enabled">(bool) True = enabled Button, false/null = disabled</param>
    /// <param name="callback">[OPTIONAL] (function) Function to call when this button is clicked (No Parameters)</param>
    /// <returns>A new ButtonObj</returns>
    
    this.m_strUniqueID = (uniqueID!=undefined) ? uniqueID : UIHelper.generateRandomString(10);
    this.m_strText = text;
    this.m_varHeight = height;
    this.m_varWidth = width;
    this.m_varTextTopPadding = textTopPadding;
    this.m_strClassNamePrefix = classNamePrefix;
    this.m_bEnabled = (enabled == true) ? true : false;
    this.m_fnCallback = (callback!=undefined) ? callback : null;
    
    this.m_strID = 'ButtonObj' + this.m_strUniqueID;
    this.m_strTextObjID = this.m_strID + 'TextObj';
    this.m_bPushedDown = false;
    if(this.m_strClassNamePrefix!=undefined)
    {
        this.m_strStandardClassName = this.m_strClassNamePrefix + '-standard';
        this.m_strClickedClassName = this.m_strClassNamePrefix + '-clicked';
        this.m_strDisabledClassName = this.m_strClassNamePrefix + '-disabled';
    }
    else
    {
        this.m_strStandardClassName = null;
        this.m_strClickedClassName = null;
        this.m_strDisabledClassName = null;
    }
    
    // Elements
    this.m_elContainer = null;
    this.m_elDivButton = null;
    
    //Objects
    this.m_objText = null;
    
    this.m_elDivButton = document.createElement('div');
    this.m_elDivButton.id = this.m_strID;
    this.m_objText = new TextObj(this.m_strTextObjID, this.m_strText, 'buttontext');
    
    if(this.m_varWidth != undefined)
        this.m_elDivButton.style.width = this.m_varWidth;
    if(this.m_varHeight != undefined)
        this.m_elDivButton.style.height = this.m_varHeight;
    if(this.m_varTextTopPadding != undefined)
        this.m_elDivButton.style.paddingTop = this.m_varTextTopPadding;
        
    this.m_objText.draw(this.m_elDivButton);
    if(!this.m_bEnabled)
    {
        this.m_elDivButton.disabled = true;
        if(this.m_strDisabledClassName != undefined)
            this.m_elDivButton.className = this.m_strDisabledClassName;
    }
    else if(this.m_strStandardClassName != undefined)
        this.m_elDivButton.className = this.m_strStandardClassName;

    // render cobranding
    if ( cobrandObj!= undefined && cobrandObj!=null )
    {
        if ( cobrandObj.m_strBtTxtColor.length > 0 )
        {
            this.m_objText.setColor(cobrandObj.m_strBtTxtColor);
        }
    }
}

ButtonObj.prototype = 
{
    draw:function(elementDiv)
    {
        /// <summary>Calls appendChild() on the provided element, appending this div element</summary>
        /// <param name="elementDiv">(HTML Element) HTML element to append this button to</param>
        if(elementDiv)
        {
            var self = this;
            
            this.m_elContainer = elementDiv;
            
            this.m_elContainer.appendChild(this.m_elDivButton);
            
            this.m_elDivButton.onmousedown = function() {self.pushed(true, false)};
            this.m_elDivButton.onmouseup = function() {self.pushed(false, true)};
            this.m_elDivButton.onkeydown = function() {self.keyPress()};
            document.body.attachEvent("onmouseup", function() {self.pushed(false, false)});
        }
    },
    getElement:function()
    {
        return this.m_elDivButton;
    },    
    enable:function()
    {
        /// <summary>Enables this button</summary>
        this.m_bEnabled = true;
        this.updateState();
    },
    
    disable:function()
    {
        /// <summary>Disables this button</summary>
        this.m_bEnabled = false;
        this.updateState();
    },
    
    setTabIndex:function(value)
    {
        /// <summary>Sets the tab index of this button</summary>
        /// <param name="value">(int) Tab index of this button</param>
        if((this.m_elDivButton != undefined) && (value != undefined))
            this.m_elDivButton.tabIndex = value;
    },
    
    setTextClass:function(className)
    {
        /// <summary>Changes the class name for the button text (span.[classname])</summary>
        /// <param name="className">(string) Class name to update button text to</param>
        if((this.m_objText != undefined) && (className != undefined))
            this.m_objText.setClassName(className);
    },
    
    setText:function(text)
    {
        /// <summary>Changes the text in this button</summary>
        /// <param name="text">(string) Text to update the button text with</param>
        if((this.m_objText != undefined) && (text != undefined))
            this.m_objText.setText(text);
    },
    
    // **********************************************************************************
    // * Internal methods
    // **********************************************************************************
    keyPress:function()
    {
        if(window.event != undefined)
        {
            var keycode = window.event.keyCode;
            if(keycode == KEY_CODE_SPACE || keycode == KEY_CODE_ENTER) {
                if(this.m_fnCallback != null)
                {
                    window.event.returnValue = false;
                    this.m_fnCallback();
                }
            }
        }
    },
    
    pushed:function(push, execute)
    {
        /// <summary>Called to show the button as pushed or raised</summary>
        /// <param name="push">(bool) true if button is pushed down, false if raised</param>
        /// <param name="execute">(bool) true if callback should be called on button raise, false otherwise</param>
        if((this.m_elDivButton != undefined) && this.m_bEnabled)
        {
            if(push && !this.m_bPushedDown)
            {
                this.m_elDivButton.className = this.m_strClickedClassName;
                this.m_bPushedDown = true;
            }
            else if(!push && this.m_bPushedDown)
            {
                this.m_elDivButton.className = this.m_strStandardClassName;
                this.m_bPushedDown = false;
                if(execute && (this.m_fnCallback != undefined))
                    this.m_fnCallback();
            }
        }
    },
    
    updateState:function()
    {
        /// <summary>Syncs the button's appearance with it's enabled/disabled state</summary>
        
        if((this.m_elDivButton != undefined) && (this.m_elDivButton.disabled == this.m_bEnabled))
        {
            this.m_elDivButton.disabled = !this.m_bEnabled;
            if(!this.m_bEnabled && (this.m_strDisabledClassName != undefined))
                this.m_elDivButton.className = this.m_strDisabledClassName;

            else if(this.m_bEnabled && (this.m_strStandardClassName != undefined))
                this.m_elDivButton.className = this.m_strStandardClassName;
        }
    }
}

