﻿// *************************************************************************************************************************
// CheckBoxObj (Javascript Object)
//
// Wrapper around an 'input' type='checkbox' HTML Element.  The label for the checkbox is a 'label' HTML Element. 
// This tracks the original value of the checkbox, and if the user has ever changed the state of the checkbox.
// To add the checkbox to the form, call the draw() function.  The checkbox will be added to the element provided throught
// the appendChild() method.
// *************************************************************************************************************************

// Dependencies:
/// <reference path="uicore.js">

CheckBoxObj = function(uniqueID, text, checked, checkBoxClassName, textClassName, enabled, cobrandObj, clickCallback, enterCallback)
{
    /// <summary>Constructor for CheckBoxObj.  [Checkbox = 'input' type='checkbox' HTML Element]  [Label = 'label' HTML Element]</summary>
    /// <param name="uniqueID">[OPTIONAL] (string) unique identifies used in the ID of this element. If null, generates random ID. (CheckBox ID = 'CheckBoxObj[uniqueID]') (Label ID = 'CheckBoxObj[uniqueID]TextObj')</param>
    /// <param name="text">(string) Text to display next to the checkbox</param>
    /// <param name="checked">(bool) True = checkbox checked initially, null/false = unchecked</param>
    /// <param name="checkBoxClassName">[OPTIONAL] (string) The Stylesheet class name for the checkbox (Default: 'checkbox')</param>
    /// <param name="textClassName">[OPTIONAL] (string) The Stylesheet class name for the text (Default: 'checkbox')</param>
    /// <param name="enabled">(bool) True = Checkbox enabled, null/false = disabled</param>
    /// <returns>A new CheckBoxObj</returns>
    
    this.m_strUniqueID = (uniqueID != undefined) ? uniqueID : UIHelper.generateRandomString(10);
    this.m_strText = text;
    this.m_bChecked = (checked == true) ? true : false;
    this.m_strCheckBoxClassName = (checkBoxClassName != undefined) ? checkBoxClassName : 'checkbox';
    this.m_strTextClassName = (textClassName != undefined) ? textClassName : 'checkbox';
    this.m_bEnabled = (enabled == true) ? true : false;
	    
    this.m_strID = 'CheckBoxObj' + this.m_strUniqueID;
    this.m_strLabelID = this.m_strID + 'TextObj';
    this.m_bOriginalChecked = this.m_bChecked;
    this.m_bWasChanged = false;
    
    // Elements
    this.m_elContainer = null;
    this.m_elInputCheckBox = null;
    this.m_elLabelText = null;

    // click call back
    this.m_fnClickCallback = null;
    this.m_fnEnterCallback = (enterCallback!=undefined) ? enterCallback : null;

    // Create CheckBox
    this.m_elInputCheckBox = document.createElement('input');
    this.m_elInputCheckBox.id = this.m_strID;
    this.m_elInputCheckBox.type = 'checkbox';
    this.m_elInputCheckBox.className = this.m_strCheckBoxClassName;    

    if(!this.m_bEnabled)
        this.m_elInputCheckBox.disabled = true;
    
    // Create CheckBox Label
    this.m_elLabelText = document.createElement('label');
    this.m_elLabelText.id = this.m_strLabelID;
    this.m_elLabelText.htmlFor = this.m_strID;
    this.m_elLabelText.innerText = ' ' + this.m_strText;
    this.m_elLabelText.className = this.m_strTextClassName;

    if(!this.m_bEnabled)
        this.m_elLabelText.disabled = true;

    // render cobranding
    if ( cobrandObj!= undefined && cobrandObj!=null )
    {
        if ( cobrandObj.m_strCheckboxLbColor.length > 0 )
        {
            this.m_elLabelText.style.color = cobrandObj.m_strCheckboxLbColor;
        }
    }
}

CheckBoxObj.prototype = 
{
    draw:function(elementDiv)
    {
        /// <summary>Calls appendChild() on the provided element, appending this checkbox element</summary>
        /// <param name="elementDiv">(HTML Element) HTML element to append this checkbox to</param>
        if(elementDiv != undefined)
        {
            var self = this;
            
            this.m_elContainer = elementDiv;
            
            this.m_elContainer.appendChild(this.m_elInputCheckBox);

            if ( this.m_fnClickCallback != null )
                this.m_elInputCheckBox.onclick = this.m_fnClickCallback;

            if (this.m_fnEnterCallback != null) {
                this.m_elInputCheckBox.onkeydown= function() {self.checkEnterKey()};
            }

            this.m_elInputCheckBox.checked = this.m_bChecked;

            this.m_elContainer.appendChild(this.m_elLabelText);
        }
    },
    // in xp preset this.m_elInputCheckBox.checked is not honored
    // need to do a refersh after the element is drawn
    refresh:function()
    {
        var element = document.getElementById(this.m_strID);
        if ( null != element && undefined != element )
            element.checked = this.m_bChecked;
    },
    setClickCallback:function(callback)
    {
        this.m_fnClickCallback = callback;
    },   	

    checkEnterKey:function()
    {
        if(window.event != undefined)
        {
            var keycode = window.event.keyCode;
            if((keycode == KEY_CODE_ENTER) && (this.m_fnEnterCallback != null))
            {
                window.event.returnValue = false;
                this.m_fnEnterCallback();
            }
        }
    },
    
    getChecked:function()
    {
        /// <summary>Returns true if the checkbox is checked, false if unchecked</summary>
        if(this.m_elInputCheckBox != undefined)
            this.m_bChecked = this.m_elInputCheckBox.checked;

        return this.m_bChecked;
    },

    setChecked:function(checked)
    {
        if(this.m_elInputCheckBox != undefined)
        {
            this.m_elInputCheckBox.checked = checked;
            this.m_bChecked = this.m_elInputCheckBox.checked;
        }
    },
    
    getOriginalChecked:function()
    {
        /// <summary>Returns true if the checkbox was checked at creation time, false if unchecked</summary>
        return this.m_bOriginalChecked;
    },
    
    disable:function()
    {
        /// <summary>Disables the checkbox</summary>
        this.m_bEnabled = false;
        this.updateState();
    },
    
    enable:function()
    {
        /// <summary>Enables the checkbox</summary>
        this.m_bEnabled = true;
        this.updateState();
    },
    isEnabled:function()
    {
        return this.m_bEnabled;
    },
    setTabIndex:function(value)
    {
        /// <summary>Sets the tab index of this checkbox</summary>
        /// <param name="value">(int) Tab index of this checkbox</param>
        if((value != undefined) && (this.m_elInputCheckBox != undefined))
            this.m_elInputCheckBox.tabIndex = value;
    },
    
    updateState:function()
    {
        /// <summary>Sync the appearance of the checkbox & label with the enabled/disabled state of the checkbox</summary>
    
        if(this.m_elInputCheckBox != undefined)
            this.m_elInputCheckBox.disabled = !this.m_bEnabled;

        if(this.m_elLabelText != undefined)
            this.m_elLabelText.disabled = !this.m_bEnabled;
    }
}
