﻿// *************************************************************************************************************************
// StringLocalObj (Javascript Object)
//
// A StringLocalObj is an object used for retrieving and cacheing Localized String and Errors.  The object can also create
// an error message div element for displaying errors.
// *************************************************************************************************************************

// Dependencies
/// <reference path="UIResource.js" />
/// <reference path="divwrapper.js" />
/// <reference path="image.js" />
/// <reference path="text.js" />
/// <reference path="link.js" />
/// <reference path="externalwrapper.js" />

var WLID_ERROR_GENERIC = 0x00000000;
var WLID_ERROR_EMPTY_USERNAME = 0x8004882D; // msidcrl.h    PPCRL_REQUEST_E_INVALID_MEMBER_NAME
var WLID_ERROR_EMPTY_PASSWORD = 0x8004882E; // msidcrl.h    PPCRL_REQUEST_E_MISSING_PRIMARY_CREDENTIAL
var WLID_ERROR_EMPTY_PIN = 0x8010002A ; //  SCARD_E_INVALID_CHV           
var PPCRL_REQUEST_E_PASSWORD_LOCKED_OUT = 0x80048823;
var PPCRL_REQUEST_E_PASSWORD_LOCKED_OUT_BAD_PASSWORD_OR_HIP = 0x80048824;

var WLID_ERROR_TYPE_APP = 1;
var WLID_ERROR_TYPE_USER_GEN = 2;
var WLID_ERROR_TYPE_USER_TIL = 3;
var ICONSIZE = 18;
var REALM_INFO_MAX_LENGTH = 50;

StringLocalObj = function()
{
    /// <summary>Constructor for StringLocalObj.  Initializes the object and loads all error codes</summary>
    this.m_arrString = new Array();
    this.m_arrErrors = new Array();
    this.loadErrors();
    this.m_strGenericErrorMsg = this.getErrMsg(WLID_ERROR_GENERIC);
    this.m_ExObj = new ExternalWrapperObj();

    this.m_iLangID = this.m_ExObj.getLCID();

    //Set the Default Globalization values
    this.m_strStyleSheet = "global_default.css";
    this.m_bIsRTL = false;
    this.m_iMaxChars = 24;
    this.urlPrivacy = this.m_ExObj.getURL(WLID_URL_PRIVACY)

    this.setGlobalizedValues();
}

StringLocalObj.getLocalString = function(stringID)
{
    /// <summary>Static method for retrieving localized strings without cacheing</summary>
    /// <param name="stringID">(int) ID of localize string (ID's in UIResource.js)</param>
    var toReturn = '';
    if(stringID != undefined)
    {
        var exObj = new ExternalWrapperObj();
        if(exObj != undefined)
            toReturn = exObj.getString(stringID);
    }
    return toReturn;
}

StringLocalObj.prototype = 
{
    loadErrors:function()
    {   
        //ErrMap[error code] = [error code, string id, type ];
    },    getErrMsg:function(errCode)
    {
        /// <summary>Retrieves the localized error message for the specified error code</summary>
        /// <param name="errCode">(error code) error code to look up</param>
        var errMsg = this.m_strGenericErrorMsg;
        if((errCode != undefined) && (this.m_ExObj != undefined))
            errMsg = this.m_ExObj.getErrMsg( errCode );

        return errMsg;
    },
    
    getErrType:function(errCode)
    {
        /// <summary>Gets the error type for the specified error code (WLID_ERROR_TYPE_APP, WLID_ERROR_TYPE_USER_GEN, or WLID_ERROR_TYPE_USER_TIL)</summary>
        /// <param name="errCode">(error code) error code to look up</param>
        var errType = WLID_ERROR_TYPE_APP;
        if((errCode != undefined) && ( this.m_ExObj!= undefined))
            errType = this.m_ExObj.getErrType( errCode );
        
        return errType;
    },

    getlink:function(strmsg, strmark )
    {
        var strStart = "{" + strmark + "}";
        var strEnd = "{/" + strmark + "}";
        var strlink = "";
        var idxStart = strmsg.indexOf(strStart);
        var idxEnd = strmsg.indexOf(strEnd);
        if ( -1 == idxStart || -1 == idxEnd || idxStart >= idxEnd )
            return strlink;
        
        // aaabbbccc, {l}1234{/l} --> 1234
        strlink = strmsg.substring( idxStart + strStart.length , idxEnd );
        
        return strlink;
    },
    getText:function(strmsg, strmark, firstpart )
    {
        var strStart = "{" + strmark + "}";
        var strEnd = "{/" + strmark + "}";
        var strTxt = strmsg;
        var idxStart = strmsg.indexOf(strStart);
        var idxEnd = strmsg.indexOf(strEnd);
        if ( -1 == idxStart || -1 == idxEnd || idxStart >= idxEnd )
            return strTxt;
        
        // aaabbbccc, {l}1234{/l}. dddd. --> aaabbbccc,  or . dddd
        if ( firstpart )
            strTxt = strmsg.substring( 0, idxStart );
        else
            strTxt = strmsg.substring( idxEnd + strEnd.length, strmsg.length);
        return strTxt;
    },
    createErrorTable:function(tableID, errCode, username)
    {
        if ( null == tableID || undefined == tableID )
            tableID = "ErrorTable" + UIHelper.generateRandomString(12);

        if ( null == errCode || undefined == errCode )
            return null;

        if ( null == this.m_ExObj || undefined == this.m_ExObj  )
            return null;

        var resolvelinktext = this.getString(L_TEXT_LINK_T);
        // creates a <table> element and a <tbody> element
        var tbl = document.createElement("table");
        var tblBody = document.createElement("tbody");
        {
            //    <tr>
            //        <td width='ICONSIZE'/><td width='2'/><td witdth='100%'/>
            //    </tr>
            var row = document.createElement("tr");
            var cell = document.createElement("td");

            // 1) icon        
            cell.setAttribute("width", ICONSIZE);
            cell.setAttribute("vAlign", "top");
            var ErrIcon = new ImageObj('__TestHook_ErrorIcon__', 'WLID_Icon_Error.gif', '16px', '16px', null, null);
            ErrIcon.draw(cell);            
            row.appendChild(cell);

            // 2) space
            cell = document.createElement("td");
            cell.setAttribute("width", 2);
            row.appendChild(cell);

            // 3) text + flow url
            cell = document.createElement("td");
            cell.setAttribute("width", "100%");

            var errorTextDiv = new DivObj('DivText'+tableID);                

            {
                var errorText = this.getErrMsg(errCode);
                var errorType = this.getErrType(errCode);
                var errorURL = this.m_ExObj.getFlowURL();

                // 4) substitution {p} --> partner name
                if ( errorText.indexOf( '{p}' ) >= 0 )  {
                    var partnerName = this.getString(L_TEXT_DEFAULT_PARTNER_NAME_T);
                    if((username != undefined ) && (username != null ) && (username != '' ) )
                    {
                        var partner = this.m_ExObj.getNamespaceProperty(username, WLID_REALM_PROP_FED_BRAND_NAME);
                        if ( partner != undefined && partner != null && partner != '' )
                            partnerName = UIHelper.truncateName(partner, REALM_INFO_MAX_LENGTH );
                    }
                    errorText = errorText.replace(/{p}/g, partnerName);
                }

                // 5) process special errors, replace the error message and link text
                if ( PPCRL_REQUEST_E_AUTHTYPE_NOT_ASSOCIATED == errCode )
                {
                    var errorMsg = errorText;
                    errorText = this.getText( errorMsg, "l", true);
                    resolvelinktext = this.getlink( errorMsg, "l" );
                    errorURL = this.m_ExObj.getURL(WLID_URL_MANAGEEID);
                } 
                
                {
                    var errorTextObj = new TextObj(null, errorText + '  ', 'errortext');                                    
                    errorTextObj.draw( errorTextDiv.getElement() );
                    
                    if((errorURL != undefined) && errorURL != null && (errorURL!='') )
                    {
                        var errorLink = new LinkObj('DirectionLink'+tableID, errorURL, resolvelinktext, '_blank');
                        errorLink.draw(errorTextDiv.getElement() );
                    } 
                }
            }

            errorTextDiv.draw(cell);
            row.appendChild(cell);

            tblBody.appendChild( row );
        }            
        // put the <tbody> in the <table>
        tbl.appendChild(tblBody);
        tbl.setAttribute("className", "internallayout");
        return tbl;        
    },
    
    createInfoTable:function(tableID, info)
    {
        if ( null == tableID || undefined == tableID )
            tableID = "InfoTable" + UIHelper.generateRandomString(12);

        if ( null == info || undefined == info )
            return null;
            
        // creates a <table> element and a <tbody> element
        var tbl = document.createElement("table");
        var tblBody = document.createElement("tbody");
        
        {
            //    <tr>
            //        <td width='ICONSIZE'/><td width='2'/><td witdth='100%'/>
            //    </tr>
            var row = document.createElement("tr");
            var cell = document.createElement("td");
            cell.setAttribute("width", ICONSIZE);
            cell.setAttribute("vAlign", "top");
            var InfoIcon = new ImageObj('__TestHook_InfoIcon__', 'ic_alert_low_16x.gif', '16px', '16px', null, null);
            InfoIcon.draw(cell);            
            
            row.appendChild(cell);

            cell = document.createElement("td");
            cell.setAttribute("width", 2);
            row.appendChild(cell);
            
            cell = document.createElement("td");
            cell.setAttribute("width", "100%");
            var InfoTextObj = new TextObj(null, info, 'regtext');
            InfoTextObj.draw(cell);
            
            row.appendChild(cell);

            tblBody.appendChild( row );
        }            
        // put the <tbody> in the <table>
        tbl.appendChild(tblBody);
        tbl.setAttribute("className", "internallayout");
        return tbl;        
    },
    getString:function(stringID)
    {
        /// <summary>Retrieves the localized string for the provided string ID</summary>
        /// <param name="stringID">(int) ID of the localized string to retrieve</param>
        var localizedString = '(Not found)';
        if(stringID != undefined)
            localizedString = (this.m_arrString[stringID]) ? this.m_arrString[stringID] : this.lookupString(stringID);
        return localizedString;
    },

    lookupString:function(stringID)
    {
        var stringValue = '(Not found)';
        if(stringID != undefined)
        {
            this.m_arrString[stringID] = this.m_ExObj.getString(stringID);
            stringValue = this.m_arrString[stringID];
        }
        return stringValue;
    },

    getStyleSheet:function()
    {
        return this.m_strStyleSheet;
    },

    getMaxChars:function()
    {
        return this.m_iMaxChars;
    },

    getPrivacyURL:function()
    {
        return this.urlPrivacy;
    },
    
    isRtlLanguage:function()
    {
        var retval = this.m_bIsRTL;

        //Override the RTL value if RTLTestMode is enabled.
        if (this.m_ExObj.isRtlTest())
            retval = true;

        return retval;
    },

    setRtl:function(uielementid )
    {
        if ( null == uielementid || undefined == uielementid )
            return;
        var elem =  document.getElementById( uielementid );
        if ( null == elem || undefined == elem )
            return;
        if ( null == elem.dir || undefined == elem.dir )
            return;

        //Verify this is a RTL language.
        if (!this.isRtlLanguage())
            return;
            
        elem.dir = 'rtl';            
    },

    setGlobalizedValues:function()
    {
        if(this.m_iLangID != undefined && this.m_iLangID != null)
        {
            //Use this function to override the default globalization values
            switch(this.m_iLangID)
            {
                case 1025:   //ar-SA
                    this.m_bIsRTL = true;
                    this.m_strStyleSheet = "global_1025.css";
                    break;  

                case 1026:   //bg-BG
                    break;  

                case 1027:   //ca-ES
                    break;

                case 1028:   //zh-TW
                    this.m_strStyleSheet = "global_1028.css";
                    break;  

                case 1029:   //cs-CZ
                    break;   

                case 1030:   //da-DK
                    break;

                case 1031:   //de-DE
                    break;

                case 1032:   //el-GR
                    break;

                case 1035:   //fi-FI
                    break;

                case 1036:   //fr-FR
                    break;
                
                case 1037:   //he-IL
                    this.m_bIsRTL = true;
                    this.m_strStyleSheet = "global_1037.css";
                    break;

                case 1038:   //hu-HU
                    this.m_strStyleSheet = "global_1038.css";
                    break;

                case 1040:   //it-IT
                    break;

                case 1041:   //ja-JP
                    this.m_strStyleSheet = "global_1041.css";
                    this.m_iMaxChars = 21;
                    break;
                
                case 1042:   //ko-KR
                    this.m_strStyleSheet = "global_1042.css";
                    break;

                case 1043:   //nl-NL
                    break;

                case 1044:   //nb-NO
                    break;

                case 1045:   //pl-PL
                    break;

                case 1046:   //pt-br
                    break;

                case 1048:   //ro-RO
                    break;

                case 1049:   //ru-RU
                    break;

                case 1050:   //hr-HR
                    break;

                case 1051:   //sk-SK
                    break;

                case 1053:   //sv-SE
                    break;

                case 1054:   //th-TH
                    break;

                case 1055:   //tr-TR
                    break;

                /*case 1057:   //id-ID
                    this.urlPrivacy = "http://g.msn.com/2privacy/idid";
                    break;*/

                case 1058:   //uk-UA
                    break;

                case 1060:   //sl-SI
                    break;

                case 1061:   //et-EE
                    break;

                case 1062:   //lv-LV
                    break;

                case 1063:   //lt-LT
                    break;

                /*case 1066:   //vi-VN
                    this.urlPrivacy = "http://g.msn.com/2privacy/vivn";
                    break;

                case 1069:   //eu-ES
                    this.urlPrivacy = "http://g.msn.com/2privacy/eues";
                    break;

                case 1081:   //hi-IN
                    this.m_strStyleSheet = "global_1081.css";
                    this.urlPrivacy = "http://g.msn.co.in/2privacy/hiin";
                    break;

                case 1086:   //ms-MY
                    this.urlPrivacy = "http://g.msn.com/2privacy/msmy";
                    break;
                
                case 1095:   //gu-IN
                    this.m_strStyleSheet = "global_1095.css";
                    this.urlPrivacy = "http://g.msn.com/2privacy/guin";
                    break;

                case 1097:   //ta-IN
                    this.m_strStyleSheet = "global_1097.css";
                    this.urlPrivacy = "http://g.msn.com/2privacy/tain";
                    break;
                
                case 1098:   //te-IN
                    this.m_strStyleSheet = "global_1098.css";
                    this.urlPrivacy = "http://g.msn.com/2privacy/tein";
                    break;

                case 1099:   //kn-IN
                    this.m_strStyleSheet = "global_1099.css";
                    this.urlPrivacy = "http://g.msn.com/2privacy/knin";
                    break;
                
                case 1100:   //ml-IN
                    this.m_strStyleSheet = "global_1100.css";
                    this.urlPrivacy = "http://g.msn.com/2privacy/mlin";
                    break;

                case 1102:   //mr-IN
                    this.m_strStyleSheet = "global_1102.css";
                    this.urlPrivacy = "http://g.msn.com/2privacy/mrin";
                    break;*/
                
                case 2052:   //zh-CN
                    this.m_strStyleSheet = "global_2052.css";
                    this.m_iMaxChars = 21;
                    break;

                case 2070:   //pt-PT
                    break;

                case 2074:   //sr-Latn-BA
                    break;

                case 3082:   //es-ES
                    break;

                case 3098:   //sr-Cyrl-CS
                    this.m_strStyleSheet = "global_3098.css";
                    break;
            }
        }
    }
}

ErrorObj = function(errorCode, errorID, errorType)
{
    this.m_errErrorCode = errorCode;
    this.m_strErrorStringID = errorID;
    this.m_iErrorType = errorType;
    this.m_strErrorMsg = null;
}

ErrorObj.prototype =
{
    getCode:function()
    {
        return this.m_errErrorCode;
    },
    
    setCode:function(value)
    {
        this.m_errErrorCode = value;
    },
    
    getStringID:function()
    {
        return this.m_strErrorStringID;
    },
    
    setStringID:function(value)
    {
        this.m_strErrorStringID = value;
    },
    
    getType:function()
    {
        return this.m_iErrorType;
    },
    
    setType:function(value)
    {
        this.m_iErrorType = value;
    },
    
    getMessage:function()
    {
        if((this.m_strErrorMsg == undefined) && (this.m_strErrorStringID != undefined))
            this.m_strErrorMsg = StringLocalObj.getLocalString(this.m_strErrorStringID);
        return this.m_strErrorMsg;
    }
}

