﻿// *************************************************************************************************************************
// UserTileObj (Javascript Object)
//
// A UserTileObj is a wrapper around a 'div' HTML element which contains buttons, images, textboxes, checkboxes, and other
// inputs for a user to choose and submit credentials.  A UserTileObj takes a SavedUserObj so it will represent the user
// including its last used cred type displayed in the cred type, and adjusts its visible input fields based on the cred
// type.  Signed In users will be signed in on first click, otherwise the tile will expand and become highlighted when
// the user clicks it.  The user tile depends on a TileBoxObj being its container, as its remove functionality and certain
// member variables depend on the parent existing.  See TileBoxObj for a way to display UserTileObj's on a page.
// *************************************************************************************************************************

// Dependencies:
/// <reference path="button.js" />
/// <reference path="checkbox.js" />
/// <reference path="combobox.js" />
/// <reference path="divwrapper.js" />
/// <reference path="externalwrapper.js" />
/// <reference path="image.js" />
/// <reference path="link.js" />
/// <reference path="localization.js" />
/// <reference path="saveduser.js" />
/// <reference path="text.js" />
/// <reference path="textbox.js" />
/// <reference path="uicore.js" />
/// <reference path="UIResource.js" />

UserTileObj = function(uniqueID, savedUser, animate, errorDiv, activeBG, activeBorder, tileLabelColor, forceExpand, divCapLockNotify)
{
    /// <summary>Constructor for UserTileObj.  [UserTile = 'div' HTML Element]</summary>
    /// <param name="uniqueID">[OPTIONAL] (string) Unique string to user in the ID of all the elements in this usertile.
    /// If null, generates a random ID</param>
    /// <param name="savedUser">(SavedUserObj) SavedUserObj to create this user tile from</param>
    /// <param name="animate">(bool) True = Animate the activation/deactivation of this tile when clicked/unclicked, False = No Animation</param>
    /// <param name="errorDiv">[OPTIONAL] (DivObj) Error div to display in this tile if one is defined</param>
    /// <param name="activeBG">[OPTIONAL] (string) Background color of this tile when activated (default: '#e1f3fc')</param>
    /// <param name="activeBorder">[OPTIONAL] (string) Border color of this tile when activated (default: '#bae4fc')</param>
    /// <returns>A new TileBoxObj</returns>

    // Member objects
    this.m_objLocalizer = new StringLocalObj();
    this.m_objEx = new ExternalWrapperObj();

    this.m_strUniqueID = (uniqueID != undefined) ? uniqueID : UIHelper.generateRandomString(10);
    this.m_iMaxChars = this.m_objLocalizer.getMaxChars();
    this.m_obSavedUser = savedUser;
    this.m_bShowAnimation = animate;
    this.m_objDivError = errorDiv;
    this.m_strActiveBGColor = (activeBG != undefined) ? activeBG : '#e1f3fc';
    this.m_strActiveBorderColor = (activeBorder != undefined) ? activeBorder : '#bae4fc';
    if (tileLabelColor != undefined)
        this.m_strTileLabelColor = tileLabelColor;
    this.m_divCapsLockNotify = (divCapLockNotify!=undefined && divCapLockNotify != null ) ? divCapLockNotify : null;
    
    // Saved User variables
    this.m_bPersisted = this.m_obSavedUser.getRememberMe(true);
    this.m_strCredType = this.m_obSavedUser.getCredType();
    this.m_strUsername = this.m_obSavedUser.getUsername();
    this.m_strUserImage = this.m_obSavedUser.getImage();
    this.m_bSavedPassword = this.m_obSavedUser.getSavePassword(true);
    if((this.m_strCredType != undefined) && 
        (this.m_strCredType == WLID_LAST_USED_CRED_FED_EID || this.m_strCredType == WLID_LAST_USED_CRED_FED_PASSWORD))
        this.m_bFederated = true;
    else
        this.m_bFederated = false;


    // honor WLIDUI_FORCE_SAVEPASSWORD even if this user is already signed in (actived) in other application
    this.m_bExpands = (this.m_obSavedUser.getSignedIn() == true) ? false : true;
    if ( x_wstrUIOptionValueTRUE == this.m_objEx.getProperty( k_wstrInProp_ForceSavePassword )  && !this.m_bSavedPassword)
        this.m_bExpands = true;
    if ( true == forceExpand)    
        this.m_bExpands = true;
    
    // set fed partner name, which will be shown in the top of the tile when actived
    if ( this.m_bFederated )
    {
        var fedpartnername = this.m_objEx.getNamespaceProperty(this.m_strUsername, WLID_REALM_PROP_FED_BRAND_NAME);
        var fedpartnerlabel = this.m_objLocalizer.getString(L_TEXT_FED_PARTNER_SUBLBL_T);
        if ( fedpartnername != undefined && fedpartnername != null ) 
        {
            fedpartnername = UIHelper.truncateName(fedpartnername, REALM_INFO_MAX_LENGTH );        
            this.m_objFedPartnerInfoDiv = new DivObj('InfoDiv' + this.m_strUniqueID );
            this.m_objFedPartnerInfoDiv.setPadding('left', '8px');
            this.m_objFedPartnerInfoDiv.setPadding('top', '8px');
            this.m_objFedPartnerInfoDiv.setPadding('right', '8px');
            this.m_objFedPartnerInfoDiv.setPadding('bottom', '2px');            
            this.m_objFedPartnerInfoDiv.hide(true);

            var tabFedInfo = this.m_objLocalizer.createInfoTable('FedInfo' + this.m_strUniqueID, fedpartnerlabel + ' ' + fedpartnername);
            if ( null != tabFedInfo && undefined != tabFedInfo )
                this.m_objFedPartnerInfoDiv.getElement().appendChild( tabFedInfo );
        }
    }
    
    // Member variables
    this.m_strID = 'UserTile' + this.m_strUniqueID;
    this.m_strPasswordText = (this.m_bSavedPassword == true) ? 'Password' : null;
    this.m_bDummyPass = (this.m_bSavedPassword == true);
    this.m_strBGColor = document.body.style.backgroundColor;
    this.m_strBorderColor = document.body.style.backgroundColor;
    this.m_bActive = false;
    this.m_bDeleting = false;
    this.m_objDispCurrentCred = null;
    this.m_objDispCurrentCert = null;
    this.m_iIndex = 0;
    this.m_bHideSavePassword = this.m_objEx.getHideStatus(x_wstrUIOptionSavePassword);
    if(this.m_bHideSavePassword)
    this.m_iScrollTop = 0;
    
    // Parent references
    this.m_elContainer = null;
    this.m_objTileBoxParent = null;
    
    this.m_elImgBorder = null;
    
    // DivObj
    this.m_objDivUserTile = null;
    this.m_objDivImageColumn = null;
    this.m_objDivImage = null;
    this.m_objDivForgetLink = null;
    this.m_objDivContentColumn = null;
    this.m_objDivActive = null;
    this.m_objDivSavePassword = null;
    this.m_objDivSavedUserSigninHolder = null;  // the signin button for the saveduser, when the tile is still inactived
    
    // LinkObj
    this.m_objLinkForgetMe = null;
    this.m_objLinkForgotPwd = null;
    this.m_objLinkRemPassHelp = null;
    // TextObj
    this.m_objTextUsernameLbl = null;
    this.m_objTextPINLbl = null;
    this.m_objTextCertLbl = null;
    this.m_objTextCert = null; // real thumbprint if only 1 cert is found
    
    // TextBoxObj
    this.m_objTBoxPIN = null;
    this.m_objTBoxPassword = null;
    
    // CheckBoxObj
    this.m_objChkSavePassword = null;
    
    // Other UI Objects
    this.m_objComboCredPicker = null;
    this.m_objComboCertPicker = null;
    
    this.m_objImgImage = null;
    this.m_objBtnSignIn0 = null;    // button for saved user single click signin on
    this.m_objBtnSignIn1 = null;    // button for password submit
    this.m_objBtnSignIn2 = null;    // button for eid submit
    
    this.idPassTb = 'divPass';
    this.idForgetPassLk = 'divForgetPass';
    this.idSigninBt1 = 'divSigninBt1';
    this.idSigninBt2 = 'divSigninBt2';    
    this.idCredPickerLbl = 'divCPLbl';
    this.idCredPickerCbo = 'divCPCombobox';
    this.idCertLbl = 'divCertLbl';
    this.idCertCbo = 'divCertCombobox';
    this.idPinLbl = 'divPinLbl';
    this.idPinTb = 'divPin';
    this.idForgetPin = 'divForgetPin';
    this.idRemPasswordCb = 'divRemPass';
    this.idTileInfo = 'ly_tileInfo';
    this.idUserImage = 'ly_UserImage';
    this.idUserName = 'ly_UserName';
    this.idActive = 'ly_Active';
    this.idSavedUserSignin = 'div_SavedUserSignin';
    
    this.idPassTable = 'tb_password';
    this.idSmartCardTable = 'tb_smartcard';

    this.m_bActiveRendered = false;
}

UserTileObj.prototype = 
{
	draw:function(elementDiv)
	{
	    /// <summary>Calls appendChild() on the provided element, appending this user tile 'div' element</summary>
        /// <param name="elementDiv">(HTML Element) HTML element to append this user tile to</param>
        if(elementDiv != undefined)
        {        
            var self = this;
            this.m_elContainer = elementDiv;
            
            // @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@  USER TILE DIV OBJECT @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
            // Global User Tile Object
            this.m_objDivUserTile = new DivObj(this.m_strID );
            this.m_objDivUserTile.setWidth('100%');
            this.m_objDivUserTile.setClassName('usertile');
            this.m_objDivUserTile.setCursor('hand');
            this.m_objDivUserTile.setBGColor(this.m_strBGColor);
            this.m_objDivUserTile.setBorderColor(this.m_strBorderColor);
            this.m_objDivUserTile.getElement().onmouseover = function() {self.mouseOn(true)};
            this.m_objDivUserTile.getElement().onmouseout = function() {self.mouseOn(false)};
            this.m_objDivUserTile.setTabIndex(0);
            this.m_bActive = false;

            this.__setUserTileLayout();
            
            // Animation when expanding the tile
            if(this.m_bShowAnimation)
                this.m_objDivUserTile.setFilter('progid:DXImageTransform.Microsoft.GradientWipe(duration=.2);');
            
            // Add tile to container object
            this.m_objDivUserTile.draw(this.m_elContainer);
        }	
	},
	getUniqueID:function()
	{
	    /// <summary>Gets the unique ID used in creating the ID of each element in the user tile</summary>
	    return this.m_strUniqueID;
	},
	
	
	setTabIndex:function(value)
	{
	    /// <summary>Sets the tab index of this usertile</summary>
        /// <param name="value">(int) Tab index of this usertile</param>
        if((this.m_objDivUserTile != undefined) && (value != undefined))
            this.m_objDivUserTile.setTabIndex(value);
	},
	
	switchCreds:function()
	{
	    /// <summary>Switches which credential inputs are shown, depending on which credential type was chosen</summary>
	    if(this.m_objComboCredPicker != undefined)
	    {
	        var currCred = this.m_objComboCredPicker.getSelectedObj();
	        if(!this.m_objDispCurrentCred || (this.m_objDispCurrentCred.Value != currCred.Value))
	        {
	            this.m_objDispCurrentCred = currCred;
	            switch(this.m_objDispCurrentCred.Value)
	            {
	                case WLID_LAST_USED_CRED_EID:
	                case WLID_LAST_USED_CRED_FED_EID:
	                    this.showPasswordFields(false);
	                    this.showEIDFields(true);
	                    break;
	                case WLID_LAST_USED_CRED_PASSWORD:
	                case WLID_LAST_USED_CRED_FED_PASSWORD:
	                    this.showPasswordFields(true);
	                    this.showEIDFields(false);
	                    break;
	            }

                    if ( this.m_objTileBoxParent != undefined && this.m_objTileBoxParent != null )
                        this.m_objTileBoxParent.refreshTile();
                        
                    if( this.m_objEx != undefined && this.m_objEx != null )
                        this.m_objEx.adjustWindowSize( null, null, true);
 
	        }
	    }
	},

    setFocusForActiveTile:function()
    {
        /// <summary>Set the correct focus depending on which credential type is current</summary>
        if(this.m_objDispCurrentCred)
        {
            switch(this.m_objDispCurrentCred.Value)
            {
                case WLID_LAST_USED_CRED_EID:
                case WLID_LAST_USED_CRED_FED_EID:
                    if (this.m_objTBoxPIN != undefined && this.m_objTBoxPIN.isEnable() && this.m_objTBoxPIN.isVisible())
                    {
                        this.m_objTBoxPIN.getElement().focus();
                    }
                    break;
                case WLID_LAST_USED_CRED_PASSWORD:
                case WLID_LAST_USED_CRED_FED_PASSWORD:
                    if (this.m_objTBoxPassword != undefined && this.m_objTBoxPassword.isEnable() && this.m_objTBoxPassword.isVisible())
                    {
                        this.m_objTBoxPassword.getElement().focus();
                    }
                    break;
            }
        } else {
            if (this.m_objTBoxPassword != undefined && this.m_objTBoxPassword.isEnable() && this.m_objTBoxPassword.isVisible())
            {
                this.m_objTBoxPassword.getElement().focus();
            }
        }        
    },

    switchCert:function()
    {
        this.m_objTBoxPIN.hide();
        this.m_objTextPINLbl.hide();        

        if ( this.m_objComboCertPicker != undefined )
        {
            this.m_objDispCurrentCert = this.m_objComboCertPicker.getSelectedObj();
            if ( this.m_objDispCurrentCert  != null && this.m_objDispCurrentCert  != undefined )
            {
                if ( this.m_objEx.isSmartCardCert( this.m_objDispCurrentCert.Value ) )
                {
                    this.m_objTBoxPIN.show();
                    this.m_objTextPINLbl.show();
                    this.m_objTBoxPIN.setText('');
                }
            }
        }

        if ( this.m_objTileBoxParent != undefined && this.m_objTileBoxParent != null )
            this.m_objTileBoxParent.refreshTile();
        
    },   

	showPasswordFields:function(visible)
	{
        /// <summary>Sets the visibility state of all Password related fields</summary>
        /// <param name="visible">(bool) True = make fields visible, False = make fields hidden</param>
            var tbpass = document.getElementById(this.idPassTable+ this.m_strUniqueID );        
            if ( tbpass != undefined && tbpass != null )
            {
                if ( visible )
                    tbpass.style.display = 'block';
                else
                    tbpass.style.display = 'none';                
            }
	},
	
	showEIDFields:function(visible)
	{
	    /// <summary>Sets the visibility state of all EID related fields</summary>
        /// <param name="visible">(bool) True = make fields visible, False = make fields hidden</param>
            var tbsmartcard = document.getElementById( this.idSmartCardTable + this.m_strUniqueID );
            if ( tbsmartcard != undefined && tbsmartcard != null )
            {
                if ( visible )
                    tbsmartcard.style.display = 'block';
                else
                    tbsmartcard.style.display = 'none';                
            }
	},
	
	mouseOn:function(mouseOnTop)
	{
	    /// <summary>Sets the highlight state of this user tile</summary>
        /// <param name="mouseOnTop">(bool) True = make tile highlighted, False = make tile unhighlighted</param>
	    if(!this.m_bActive)
	        this.highlight(mouseOnTop);
	},
	
	getExpands:function()
	{
	    /// <summary>Returns whether this is an expandable tile (true) or a single-click-submit tile (false)</summary>
	    return this.m_bExpands;
	},
	
	getUsername:function()
	{
	    /// <summary>Returns the username for this tile</summary>
	    if(this.m_strUsername != undefined)
	        return this.m_strUsername;
	    return null;
	},
	
	getPasswordTextbox:function()
	{
	    /// <summary>Returns the password TextBoxObj for this user tile</summary>
	    return this.m_objTBoxPassword;
	},
	
	getPasswordCheckbox:function()
	{
	    /// <summary>Returns the password CheckBoxObj for this user tile</summary>
	    return this.m_objChkSavePassword;
	},
	
	isActive:function()
	{
	    /// <summary>Returns whether this user is activated (true) or not (false)</summary>
	    return this.m_bActive;
	},
	
	setIndex:function(newIndex)
	{
	    if(newIndex != undefined)
	        this.m_iIndex = newIndex;
	},
	
	removeFromParent:function()
	{
	    /// <summary>Removes this user tile from the tile box</summary>
	    if(this.m_objTileBoxParent != undefined)
	        this.m_objTileBoxParent.removeTile(this.m_iIndex);
	},
	
	removeMe:function( lasttile )
	{
	    if((this.m_strUsername != undefined) && (this.m_objEx != undefined))
	        this.m_objEx.removeUser(this.m_strUsername);
	    if((this.m_elContainer != undefined) && (this.m_objDivUserTile != undefined))
	    {
                this.m_bDeleting = true;
	        // for the lasttile, don't remove it from ui, we need to to 
	        // seamlessly switch to newuser page
	        if ( undefined == lasttile || lasttile != true )
        	      this.m_elContainer.removeChild(this.m_objDivUserTile.getElement());
	    }
	},
	
	attachClickEvent:function(parent)
	{
	    if(this.m_objDivUserTile != undefined)
	    {
	        this.m_objTileBoxParent = parent;
	        var self = this;
	        this.m_objDivUserTile.getElement().onclick = function() {self.callParentClick()};
	        this.m_objDivUserTile.getElement().onkeydown = function() {self.checkKey()};
	        this.m_objDivUserTile.getElement().onkeypress = function() {self.checkKey()};
	    }
	},
	
	detachClickEvent:function()
	{
	    if(this.m_objDivUserTile != undefined)
	        this.m_objDivUserTile.getElement().onclick = null;
	},
	
	callParentClick:function()
	{
	    if(this.m_objTileBoxParent != undefined)
	    {
	        if(this.m_bExpands)
	            this.m_objTileBoxParent.switchTiles(this.m_iIndex);
	        else
	            this.submitCredentials();
	    }
	},
	
	checkKey:function()
    {
        if(window.event != undefined)
        {
            var keynum = window.event.keyCode;
            if(keynum == KEY_CODE_SPACE)
            {
                if(!this.isActive())
                    this.callParentClick();
            }
            else if(keynum == KEY_CODE_ENTER)
            {
                if ((this.m_objLinkForgetMe == null) || (this.m_objLinkForgetMe.isOnFocus() == false))
                {
                    if(this.isActive())
                        this.submitCredentials();                    
                    else
                        this.callParentClick();
                }
            }
        }
    },

    // create a table as a layout in for the user tile
    __setUserTileLayout:function()
    {
        var holder =  this.m_objDivUserTile.getElement();
        if ( null == holder || undefined == holder )
            return;

        var self = this;
        
        // creates a <table> element and a <tbody> element
        var tbl     = document.createElement("table");
        var tblBody = document.createElement("tbody");

        {
            //    <tr>
            //        <td width='78'/><td width='210'/>
            //    </tr>
            var row = document.createElement("tr");
            var cell = document.createElement("td");
            cell.setAttribute("width", "75");
            row.appendChild(cell);
            cell = document.createElement("td");
            row.appendChild(cell);
            tblBody.appendChild( row );
            
            // fed title or error info
            //    <tr>
            //        <td colspan='2' id='ly_Tileinfo'/>
            //    </tr>
            
            var row = document.createElement("tr");
            var cell = document.createElement("td");
            cell.setAttribute("colSpan", "2");

            {
                // set title info (error message or fed partner name)
                if(this.m_objDivError != undefined)
                    cell.appendChild( this.m_objDivError.getElement() );
                else if ( this.m_objFedPartnerInfoDiv != undefined )
                    cell.appendChild( this.m_objFedPartnerInfoDiv.getElement());
            }

            row.appendChild(cell);
            tblBody.appendChild( row );

            // image column + username
            //    <tr>
            //        <td id='ly_userimage'/>
            //        <td> ... </td>
            //    </tr>
            row = document.createElement("tr");
            cell = document.createElement("td");
            cell.setAttribute("vAlign","top");
            {
                // Create Div Column which contains User Image and Border Image
                this.m_objDivImageColumn = new DivObj('UserTileColumnImage' + this.m_strUniqueID);
                this.m_objDivImageColumn.setClassName('imagecolumn');

                if ( this.m_objLocalizer.isRtlLanguage() )
                {
                    this.m_objDivImageColumn.setFloat('right');
                    this.m_objDivImageColumn.setAllPadding('0px', '0px', '0px', '8px');
                } else {
                    this.m_objDivImageColumn.setFloat('left');
                    this.m_objDivImageColumn.setAllPadding('0px', '8px', '0px', '0px');
                }

                // !!!!!! Div Object to hold User Image !!!!!!!
                this.m_objDivImage = new DivObj('UserTileDivUserImage' + this.m_strUniqueID);
                if ( this.m_objLocalizer.isRtlLanguage() )
                {
                    this.m_objDivImage.setClassName('tileframe_rtl');
                } else {
                    this.m_objDivImage.setClassName('tileframe');
                }         
                
                // User Image inside the frame
                this.m_objImgImage = new ImageObj('UserTileUserImage' + this.m_strUniqueID, this.m_strUserImage);
                this.m_objImgImage.draw(this.m_objDivImage.getElement());
                this.m_objDivImageColumn.addObject(this.m_objDivImage.getElement());
                if ( this.m_objLocalizer.isRtlLanguage() )
                {
                    this.m_objImgImage.setClassName('tileimage_rtl');
                } else {
                    this.m_objImgImage.setClassName('tileimage');
                }

                cell.appendChild( this.m_objDivImageColumn.getElement() );                
            }

            row.appendChild( cell );
            
            cell = document.createElement("td");
            cell.setAttribute("valign", "middle");
            {
                // creates a <table> element and a <tbody> element
                var tbl_inner     = document.createElement("table");
                var tblBody_inner = document.createElement("tbody");

                // <tr><td width='190'>user name </td><td>signinbutton0</td></tr> 
                var tr_inner = document.createElement("tr");
                var td_inner = document.createElement("td");

                {
                    // Create username text and add to Column
                    this.m_objTextUsernameLbl = new TextObj('UserTileUsernameLabel' + this.m_strUniqueID, UIHelper.truncateName(this.m_strUsername, this.m_objLocalizer.getMaxChars()-4 ), 'username');

                    if (this.m_strUsername.length > 20)
                        this.m_objTextUsernameLbl.setTitle(this.m_strUsername);
                    
                    if (this.m_strTileLabelColor != undefined && this.m_strTileLabelColor.length > 0)
                        this.m_objTextUsernameLbl.setColor(this.m_strTileLabelColor);
                    
                    td_inner.appendChild( this.m_objTextUsernameLbl.getElement() );
                    td_inner.setAttribute("width", "190px");
                }
                tr_inner.appendChild( td_inner );

                td_inner = document.createElement("td");

                // sign in button only appears for unsign in saved user
                if(this.m_bExpands)
                {
                    // create signin button for the saved user SSO
                    this.m_objDivSavedUserSigninHolder = new DivObj(this.idSavedUserSignin + this.m_strUniqueID);
                    td_inner.appendChild( this.m_objDivSavedUserSigninHolder.getElement() );
                    
                    this.m_objBtnSignIn0 = 
                        new ButtonObj('UserTileButtonSignIn0' + this.m_strUniqueID, 
                                                 this.m_objLocalizer.getString(L_LINK_SIGN_IN_T),
                                                 18, 
                                                 65, 
                                                 0, 
                                                 'button_save', 
                                                 true, 
                                                 this.m_objEx.getCobranding(), 
                                                 function() {self.submitCredentials()});
                    this.m_objBtnSignIn0.setTabIndex(0);
                    this.m_objBtnSignIn0.draw( this.m_objDivSavedUserSigninHolder.getElement()  );
                    // hide this button first, only show it when the tile is highlighted and saved password is true
                    this.m_objDivSavedUserSigninHolder.hide( true );
                }
                
                tr_inner.appendChild( td_inner );
                tblBody_inner.appendChild( tr_inner );
                
                // <tr><td colspan='2'>active div </td></tr> 
                tr_inner = document.createElement("tr");
                td_inner = document.createElement("td");
                td_inner.setAttribute("colSpan", "2");
                 {
                    this.m_objDivContentColumn = new DivObj('UserTileColumnContent' + this.m_strUniqueID);
                    
                    // ################# ACTIVE DIV #################
                    // initialize it first, the rendering won't happen until clicked to activate
                    this.m_objDivActive = new DivObj('UserTileDivActive' + this.m_strUniqueID);
                    this.m_objDivActive.setPadding('bottom', '6px');
                    this.m_objDivActive.hide(true);
                    this.setActiveLayout();
                    this.m_objDivContentColumn.addObject(this.m_objDivActive.getElement());                
                    td_inner.appendChild( this.m_objDivContentColumn.getElement() );
                }
                
                tr_inner.appendChild( td_inner );
                tblBody_inner.appendChild( tr_inner );
                tbl_inner.setAttribute("className", "internallayout");

                tbl_inner.appendChild(tblBody_inner);
                cell.appendChild(tbl_inner);                
            }
            
            row.appendChild( cell );
            tblBody.appendChild( row );

            // Draw the "Forget Me" link if m_bExpands is true.
            // Else, draw a spacer.

            row = document.createElement("tr");
            cell = document.createElement("td");
            cell.setAttribute("colSpan", "2");  

            if (this.m_bExpands)
            {
                // !!!!!! Div Object to hold 'Forget Me' link !!!!
                this.m_objDivForgetLink = new DivObj('UserTileDivForgetLink' + this.m_strUniqueID);

                if ( this.m_objLocalizer.isRtlLanguage() )
                   this.m_objDivForgetLink.setAllPadding('0px', '18px', '0px', '0px');
                else
                   this.m_objDivForgetLink.setAllPadding('0px', '0px', '0px', '14px');

                // 'Forget Me' link                    
                this.m_objLinkForgetMe = new LinkObj('UserTileForgetLink' + this.m_strUniqueID, null, this.m_objLocalizer.getString(L_LINK_FORGET_ME_T), null);
                this.m_objLinkForgetMe.setClassName( 'forgetlink');
                this.m_objLinkForgetMe.getElement().onclick = function() {self.removeFromParent(); return false;};
                this.m_objLinkForgetMe.setTabIndex(0);
                this.m_objLinkForgetMe.draw(this.m_objDivForgetLink.getElement());

                cell.appendChild(this.m_objDivForgetLink.getElement());
            }
            else
            { 
                cell.style.height = 8;
            }

            row.appendChild( cell );
            tblBody.appendChild( row )
        }
        
        // put the <tbody> in the <table>
        tbl.appendChild(tblBody);
        // appends <table> into 'holder'
        holder.appendChild(tbl);
        tbl.setAttribute("className", "internallayout");
    },
         __renderActive:function( display )
        {
            if ( this.m_bActiveRendered )
                return;

            var self = this;
            
            this.__drawCredPicker();
            this.__drawPasswordElements();                
            this.__drawSmartCardElements();

            if ( display != false ) // default we need to should the elements in the active area
            {
                if ( this.isDefaultEID() )
                {
                        this.showPasswordFields(false);
                        this.showEIDFields(true); 
                        this.m_strCredType = WLID_LAST_USED_CRED_EID;
                }
            } 
            else    // this is for single signin case, we don't need to display the elements. only 
                        // need they are there, and their value can be extracted.
            {
                this.showPasswordFields(false);
                this.showEIDFields(false);    
            }

            this.refreshElements();
            this.m_bActiveRendered = true;
        },

        // draw cred picker for fed user
        getSupportedAuthType:function()
        {
            var authTypes = new Array();
            if ( this.m_bFederated )
            {
                // application disable eid, last signin type is not fed eid, only show password
                if ( WLID_LAST_USED_CRED_FED_EID != this.m_strCredType 
                  && x_wstrUIOptionValueTRUE == this.m_objEx.getProperty( k_wstrInProp_AppDisableSmartCard ))
                    authTypes.push(new DisplayObj(this.m_strID+'CredTypePassword', this.m_objLocalizer.getString(L_CRED_TYPE_PASSWORD_T), WLID_LAST_USED_CRED_PASSWORD));
                else 
                    authTypes =  this.m_objEx.getSupportedAuthTypes(this.m_strUsername);
                    
                return authTypes;
            }

            // no fed, last signin is smartcard
            if ( this.m_strCredType == WLID_LAST_USED_CRED_EID )
            {
                authTypes.push(new DisplayObj(this.m_strID+'CredTypePassword', this.m_objLocalizer.getString(L_CRED_TYPE_PASSWORD_T), WLID_LAST_USED_CRED_PASSWORD));
                authTypes.push(new DisplayObj(this.m_strID+'CredTypeEID', this.m_objLocalizer.getString(L_CRED_TYPE_EID_T), WLID_LAST_USED_CRED_EID));
                return authTypes;
            }
            
            // no fed case, app disable smartcard
            if ( x_wstrUIOptionValueTRUE == this.m_objEx.getProperty( k_wstrInProp_AppDisableSmartCard ) )
            {
                authTypes.push(new DisplayObj(this.m_strID+'CredTypePassword', this.m_objLocalizer.getString(L_CRED_TYPE_PASSWORD_T), WLID_LAST_USED_CRED_PASSWORD));
                return authTypes;
            }

            // no fed case, force smartcard
            if ( x_wstrUIOptionValueTRUE == this.m_objEx.getProperty( k_wstrInProp_SmartCardForce ) )
            {
                authTypes.push(new DisplayObj(this.m_strID+'CredTypePassword', this.m_objLocalizer.getString(L_CRED_TYPE_PASSWORD_T), WLID_LAST_USED_CRED_PASSWORD));
                authTypes.push(new DisplayObj(this.m_strID+'CredTypeEID', this.m_objLocalizer.getString(L_CRED_TYPE_EID_T), WLID_LAST_USED_CRED_EID));
                return authTypes;
            }


            // no fed, user disable smartcard 
            if (  x_wstrUIOptionValueTRUE != this.m_objEx.getProperty( k_wstrInProp_SmartCardEnabled )  )
            {
                authTypes.push(new DisplayObj(this.m_strID+'CredTypePassword', this.m_objLocalizer.getString(L_CRED_TYPE_PASSWORD_T), WLID_LAST_USED_CRED_PASSWORD));
                return authTypes;
            }

            // no fed, user enable smartcard, there is no additional flags set
            authTypes.push(new DisplayObj(this.m_strID+'CredTypePassword', this.m_objLocalizer.getString(L_CRED_TYPE_PASSWORD_T), WLID_LAST_USED_CRED_PASSWORD));
            authTypes.push(new DisplayObj(this.m_strID+'CredTypeEID', this.m_objLocalizer.getString(L_CRED_TYPE_EID_T), WLID_LAST_USED_CRED_EID));
            return authTypes;            
            
        },
        __drawCredPicker:function()
        {

            var authTypes = this.getSupportedAuthType();
            if ( null == authTypes || authTypes.length <= 1 )
                return;
                
            var self = this;            

            // Create Div that holds cred picker
            var callbackFunction = function() {self.switchCreds()};
            var credToShow = 0;

            if( this.isDefaultEID())
                credToShow = 1;
            
            this.m_objComboCredPicker = new ComboBoxObj('UserTileCredPicker' + this.m_strUniqueID, authTypes, credToShow, '108px', 'DownArrow00.gif', true, null, true, null, 'combobox', 4, 4*18, this.m_iMaxChars-10);
            this.m_objComboCredPicker.setCallback(callbackFunction);
            this.m_objComboCredPicker.setTabIndex(0);
            this.m_objComboCredPicker.draw(document.getElementById( this.idCredPickerCbo + this.m_strUniqueID));
             
            this.m_objDispCurrentCred = this.m_objComboCredPicker.getSelectedObj();
        },   
        __drawPasswordElements:function()
        {
            var self = this;
            // Create password textbox and add to BoxLink Div
            this.m_objTBoxPassword = new TextBoxObj('UserTileTextBoxPassword' + this.m_strUniqueID, 
                                                                                true, 
                                                                                'tilepasswordsmall', 
                                                                                this.m_strPasswordText, 
                                                                                true, 
                                                                                this.m_objEx.getCobranding(), 
                                                                                null,
                                                                                null,
                                                                                null,
                                                                                this.m_divCapsLockNotify );
            this.m_objTBoxPassword.setTabIndex(0);
            this.m_objTBoxPassword.draw(  document.getElementById( this.idPassTb + this.m_strUniqueID));
            if (!this.m_bFederated)
            {
                this.m_objTBoxPassword.setMaxLength(max_Password);
                // Create Forgot Password link and add to BoxLink Div
                this.m_objLinkForgotPwd = new LinkObj('UserTileLinkForgotPassword' + this.m_strUniqueID, 
                                                                            this.m_objEx.getURL(WLID_URL_FORGOT_PWD), 
                                                                            this.m_objLocalizer.getString(L_LINK_FORGOT_PASSWORD_T),
                                                                            '_blank');
                this.m_objLinkForgotPwd.setClassName( 'forgetlink');
                this.m_objLinkForgotPwd.setTabIndex(0);
                this.m_objLinkForgotPwd.draw(document.getElementById(this.idForgetPassLk + this.m_strUniqueID) );                
            }

            // Create Save Password Checkbox and add to Active Div
            this.m_objDivSavePassword = new DivObj('UserTileDivSavePasswordCheck' + this.m_strUniqueID);
            this.m_objChkSavePassword = 
                new CheckBoxObj('UserTileCheckSavePassword' + this.m_strUniqueID, 
                                            this.m_objLocalizer.getString(L_CHECKBOX_SAVED_PWD_T), 
                                            this.m_bSavedPassword, 
                                            'checkbox', 
                                            'checkbox', 
                                            true, 
                                            this.m_objEx.getCobranding());
            if ( this.m_bSavedPassword )
            {
                this.m_objTBoxPassword.disable();
                this.m_objChkSavePassword.setClickCallback( function() {self.savePasswordClicked()} );
            }
            this.m_objChkSavePassword.setTabIndex(0);
            this.m_objChkSavePassword.draw(this.m_objDivSavePassword.getElement());

            // (?) help link
            var spacer = new TextObj('TextSpacer' + this.m_strUniqueID, '  ', 'regtext');            
            spacer.draw( this.m_objDivSavePassword.getElement());
            this.m_objLinkRemPassHelp = new LinkObj( 'LinkPasswordQ' + this.m_strUniqueID, this.m_objEx.getURL(WLID_URL_HELP_REMEMBER_PWD), '(?)', '_blank');
            this.m_objLinkRemPassHelp.draw( this.m_objDivSavePassword.getElement());
            
            this.m_objDivSavePassword.draw(document.getElementById(this.idRemPasswordCb + this.m_strUniqueID));
            if(this.m_bHideSavePassword || this.m_bFederated)
                this.m_objDivSavePassword.hide(true);

            if ( x_wstrUIOptionValueTRUE == this.m_objEx.getProperty( k_wstrInProp_ForceSavePassword ) )
            {
                this.m_objChkSavePassword.setChecked(true);
                this.m_objChkSavePassword.disable();
            }

            // Create Sign in button and add to Button Div
            this.m_objBtnSignIn1 = 
                new ButtonObj('UserTileButtonSignIn1' + this.m_strUniqueID, 
                                         this.m_objLocalizer.getString(L_LINK_SIGN_IN_T), 
                                         18, 
                                         65, 
                                         0, 
                                         'button_save', 
                                         true, 
                                         this.m_objEx.getCobranding(), 
                                         function() {self.submitCredentials()});
            this.m_objBtnSignIn1.setTabIndex(0);
            this.m_objBtnSignIn1.draw( document.getElementById(this.idSigninBt1 + this.m_strUniqueID) );
    
        },
        __drawSmartCardElements:function()
        {
            var self = this;
            // Create Cert Label
            this.m_objTextCertLbl = new TextObj('UserTileTextCertLabel' + this.m_strUniqueID, this.m_objLocalizer.getString(L_TEXT_CERTIFICATE_T), 'textforcewrap', true /*force wrap*/); 
            this.m_objTextCertLbl.draw( document.getElementById( this.idCertLbl + this.m_strUniqueID) );


            //Create Cert dropdown

            var applyfilter = 'true';
            if ( x_wstrUIOptionValueTRUE == this.m_objEx.getProperty( k_wstrInProp_TestHookEnabled  ) ||
                  this.m_bFederated )
                applyfilter = 'false';

            var arrCertList = this.m_objEx.getCertList(applyfilter);
            var defaultindex = UIHelper.getIndex( arrCertList,  this.m_objEx.getExtendedProperty(this.m_strUsername,SAVED_USER_PROP_CERT_THUMB) );
            var callbackFunction = function() {self.switchCert()};
            
            this.m_objComboCertPicker = new ComboBoxObj('CertPicker' + this.m_strUniqueID, arrCertList, defaultindex, '108px', 'DownArrow00.gif', true, null, true, null, 'combobox', 3, 3*18, this.m_iMaxChars-9);
            this.m_objComboCertPicker.draw( document.getElementById( this.idCertCbo + this.m_strUniqueID) );
            this.m_objComboCertPicker.setCallback(callbackFunction);   
            this.m_objComboCertPicker.setTabIndex(0);
            this.m_objComboCertPicker.setComboBoxBorder(this.m_objEx.getCobranding()); 
            this.m_objComboCertPicker.setArrowAlign((this.m_objLocalizer.isRtlLanguage()) ? 'left' : 'right');
            this.m_objDispCurrentCert = this.m_objComboCertPicker.getSelectedObj();
            
            // Create PIN Label
            this.m_objTextPINLbl = new TextObj('UserTileTextPINLabel' + this.m_strUniqueID, this.m_objLocalizer.getString(L_TEXT_PIN_T), 'textforcewrap', true /*force wrap*/);
            this.m_objTextPINLbl.draw( document.getElementById( this.idPinLbl + this.m_strUniqueID) );
            
            // Create PIN Textbox
            this.m_objTBoxPIN = new TextBoxObj('UserTileTextBoxPIN' + this.m_strUniqueID, 
                                                                        true, 
                                                                        'tilepinsmall', 
                                                                        null, 
                                                                        true, 
                                                                        this.m_objEx.getCobranding(), 
                                                                        null,
                                                                        null,
                                                                        null,
                                                                        this.m_divCapsLockNotify);
            this.m_objTBoxPIN.setTabIndex(0);
            this.m_objTBoxPIN.draw( document.getElementById( this.idPinTb + this.m_strUniqueID) );            

            this.m_objBtnSignIn2 = 
                new ButtonObj('UserTileButtonSignIn2' + this.m_strUniqueID, 
                                         this.m_objLocalizer.getString(L_LINK_SIGN_IN_T), 
                                         18, 
                                         65, 
                                         0, 
                                         'button_save', 
                                         true, 
                                         this.m_objEx.getCobranding(), 
                                         function() {self.submitCredentials()});

            if ( this.m_objEx.hasDummyCertOnly(arrCertList) )
                this.m_objBtnSignIn2.disable();
            else
                this.m_objBtnSignIn2.setTabIndex(0);
            this.m_objBtnSignIn2.draw( document.getElementById(this.idSigninBt2 + this.m_strUniqueID) );
            
            // hide whole smartcard table by default
            document.getElementById( this.idSmartCardTable + this.m_strUniqueID ).style.display = 'none';
            this.switchCert();
        },        
        
	activate:function()
	{
	    /// <summary>Highlights and expands the tile showing the contents of its 'Active Div'</summary>
	    if((this.m_objDivUserTile != undefined) && (this.m_objDivActive != undefined) && !this.m_bActive && !this.m_bDeleting )
	    {
	        if(this.m_bShowAnimation)
	        {
                    this.m_objDivUserTile.getFilters()[0].Motion = 'forward';
                    this.m_objDivUserTile.getFilters()[0].WipeStyle = 1;
                    this.m_objDivUserTile.getFilters()[0].Apply();
                }

                this.__renderActive();
                this.m_objDivActive.show();
                if ( this.m_objFedPartnerInfoDiv != undefined && this.m_objFedPartnerInfoDiv != null )
                    this.m_objFedPartnerInfoDiv.show();
                this.m_objDivUserTile.setCursor('default');
                this.highlight(true);
                this.m_objDivContentColumn.setAllPadding('2px', '0px', '0px', '0px');
                
                if(this.m_bShowAnimation)
                    this.m_objDivUserTile.getFilters()[0].Play();

                // hide 'signin' button, only show it when the tile is deactived for a savedpassword user
                if ( this.m_objDivSavedUserSigninHolder != null )
                    this.m_objDivSavedUserSigninHolder.hide( true);

                // adjust default top position of the combobox
                // 1) trigger a scroll event in order to let parent div recalculate the top offset, this is particular 
                // for the cases the total size of tile changed
                if ( this.m_objTileBoxParent.getElement() != undefined && this.m_objTileBoxParent.getElement() != null )
                {
                    this.m_objTileBoxParent.getElement().doScroll("scrollbarDown");
                    this.m_objTileBoxParent.getElement().doScroll("scrollbarUp");
                }
                
                // 2) apply the top offset to the combobox
                if ( this.m_objComboCredPicker != undefined && this.m_objComboCredPicker != null )
                    this.m_objComboCredPicker.adjustPositionOnScroll(this.m_iScrollTop);

                if ( this.m_objComboCertPicker != undefined && this.m_objComboCertPicker != null )
                    this.m_objComboCertPicker.adjustPositionOnScroll(this.m_iScrollTop);           

                var nonActiveTileHeight = 91;
                var hasActiveBefore = (this.m_objDivUserTile.getElement().offsetTop%nonActiveTileHeight)!=0;
                var scrollDown = false;
                var loopbreaker = 50;
                var loop = 0;
                while ((this.m_iScrollTop - this.m_objDivUserTile.getElement().offsetTop) < nonActiveTileHeight && 
                           (this.m_iScrollTop > this.m_objDivUserTile.getElement().offsetTop) &&
                            loop++ < loopbreaker )
                {                    
                    this.m_objTileBoxParent.getElement().doScroll("scrollbarUp");
                    scrollDown = true;
                }

                if (hasActiveBefore && scrollDown)
                    this.m_objTileBoxParent.getElement().doScroll("scrollbarUp");                    

                loop = 0;
                while ((this.m_objDivUserTile.getElement().offsetTop - this.m_iScrollTop) > 2*nonActiveTileHeight &&
                            loop++ < loopbreaker )
                {                    
                    this.m_objTileBoxParent.getElement().doScroll("scrollbarDown");
                }

                this.m_bActive = true;
                this.setFocusForActiveTile();
            }
            
        },
	
	deactivate:function()
	{
	    /// <summary>Unhighlights and contracts the tile, hiding the contents of its 'Active Div'</summary>
	    if((this.m_objDivUserTile != undefined) && (this.m_objDivActive != undefined) && this.m_bActive)
	    {
	        if(this.m_bShowAnimation)
	        {
	            this.m_objDivUserTile.getFilters().Motion = 'reverse';
                this.m_objDivUserTile.getFilters().WipeStyle = 1;
                this.m_objDivUserTile.getFilters().Apply();
            }
               this.m_objDivUserTile.setCursor('hand');
	        this.m_objDivActive.hide(true);
               if ( this.m_objFedPartnerInfoDiv != undefined && this.m_objFedPartnerInfoDiv != null )
                  this.m_objFedPartnerInfoDiv.hide(true);
	        this.highlight(false);
	        this.m_objDivContentColumn.setAllPadding('30px', '0px', '0px', '0px');
	        if(this.m_bShowAnimation)
	            this.m_objDivUserTile.getFilters().Play();

                if ( !this.m_objChkSavePassword.getChecked() ) // only clean no remember me password case
                    if (this.m_objTBoxPassword != undefined)
                        this.m_objTBoxPassword.setText('');
                    
                if (this.m_objTBoxPIN != undefined)
                    this.m_objTBoxPIN.setText('');                    

	        this.m_bActive = false;
	    }
	},
	
	highlight:function(mouseOnTop)
	{
	    /// <summary>Sets the highlighted state of this tile, True = highlighted, False = unhighlighted</summary>
	    this.m_objDivUserTile.setClassName((mouseOnTop == true) ? 'usertile-active' : 'usertile');
	    this.m_objDivUserTile.setBGColor((mouseOnTop == true) ? this.m_strActiveBGColor : this.m_strBGColor);
	    this.m_objDivUserTile.setBorderColor((mouseOnTop == true) ? this.m_strActiveBorderColor : this.m_strBorderColor);

            if ( this.m_objDivSavedUserSigninHolder != null )
            {
                if ( this.m_bSavedPassword && mouseOnTop)
                    this.m_objDivSavedUserSigninHolder.show();
                else
                    this.m_objDivSavedUserSigninHolder.hide(true);  
            }
	},
	
	submitCredentials:function()
	{
	    /// <summary>Calls tile box's submit function which will then send usertile to the main page through its callback</summary>
	    this.__renderActive( false );

	    if(this.m_objTileBoxParent != undefined)
	        this.m_objTileBoxParent.submit(this.m_iIndex);
	},
	savePasswordClicked:function()
	{
	    // this is the first time a saved password user's 'remember password' is clicked, this guy 
	    // will never be considered as a saved password user anymore
	    // we should,  1) persist this guy as remember me only user, even if no further action is performed
	    //                  2) empty password textbox
	    //                  3) enable it 
	    //                  4) set m_bSavedPassword = false,
	    this.m_bDummyPass = false;
	    if ( this.m_bSavedPassword )
	    {
	        this.m_objEx.removeUser(this.m_obSavedUser.getUsername());
	        this.m_objEx.persistCredential( this.m_obSavedUser.getUsername(), PPCRL_CREDTYPE_MEMBERNAMEONLY, 0 );
	        
        	 this.m_objTBoxPassword.setText('');
        	 this.m_objTBoxPassword.enable();
	        this.m_bSavedPassword = false;
	    }
	},
	createSavedUserObj:function()
	{
	    /// <summary>Returns a SavedUserObj which represents the state of the user represented by this user tile</summary>
	    var savedUser = null;
	    if(this.m_strUsername != undefined)
	    {
	        savedUser = new SavedUserObj(this.m_strUsername);
        
            // Set remember me based off what was originally provided
	        savedUser.setRememberMe(this.m_bPersisted, true);
	        
	        // Set Signed in based on if this is a single click tile or not
	        savedUser.setSignedIn((this.m_bExpands == true) ? false : true);
	        
	        // Set User Image
	        savedUser.setImage(this.m_strUserImage);
	        
	        // Set cred type to either originla value, or what value was chosen
	        savedUser.setCredType(this.m_strCredType);
	        if(this.m_objDispCurrentCred != undefined)
	        {
	            var currCredType = this.m_objDispCurrentCred.Value;
	            if(currCredType != this.m_strCredType)
	                savedUser.setCredType(currCredType);
	        }
            
	        var userCredType = savedUser.getCredType();

                // if the persisted user don't have last used credtype in the extended user properties
                // set the default type as password
                if ( undefined == userCredType || null == userCredType )
                    userCredType = WLID_LAST_USED_CRED_PASSWORD;
                
	        switch(userCredType)
	        {
	            // EID Users
	            case WLID_LAST_USED_CRED_EID:
	            case WLID_LAST_USED_CRED_FED_EID:
	                // Password = PIN, for federated users, there is no PIN
	                savedUser.setPassword((this.m_objTBoxPIN != undefined) ? this.m_objTBoxPIN.getText() : null);
	                savedUser.setNewPassword(true);
	                // No Saved Password
	                savedUser.setSavePassword(false, true);
	                // EID Users should have Cert Thumb and Cert Name stored as extra properties
	                //var certThumb = this.m_obSavedUser.getProperty(SAVED_USER_PROP_CERT_THUMB);
	                if ( this.m_objDispCurrentCert != null )
                       {
                            var certThumb = this.m_objDispCurrentCert.Value;
                            if(certThumb!=null )
                                savedUser.setProperty(SAVED_USER_PROP_CERT_THUMB, certThumb);
                       }   
	                break;
	            // Password Users
	            case WLID_LAST_USED_CRED_PASSWORD:
	            case WLID_LAST_USED_CRED_FED_PASSWORD:
                    if ( !this.m_bDummyPass )                        
    	                savedUser.setPassword((this.m_objTBoxPassword != undefined) ? this.m_objTBoxPassword.getText() : null);
                    else
                        savedUser.setPassword( null);
                    
                    savedUser.setNewPassword((this.m_objTBoxPassword != undefined) ? this.m_objTBoxPassword.textIsDifferent() : false);
                    savedUser.setSavePassword(this.m_obSavedUser.getSavePassword(true), true);
                    savedUser.setSavePassword((this.m_objChkSavePassword != undefined) ? this.m_objChkSavePassword.getChecked() : false, false);
                    break;
                default:
                    savedUser.setPassword(null);
                    savedUser.setNewPassword(false);
                    savedUser.setSavePassword(this.m_obSavedUser.getSavePassword(true), true);
                    break;
	        }
	    }
	    return savedUser;
	},

    // is default to EID
    isDefaultEID:function()
    {
        // error handling case, last attempt should be first honored
        //
        if ( this.m_objDivError != undefined && this.m_objDivError != null )
        {
            var errorUser = this.m_objEx.getProperty(k_wstrInProp_DefaultUser);
            if ( errorUser == this.m_strUsername ) // this is the tile cause the error
            {
                // 1) k_wstrInProp_LastLoginType will be set by the last visited page (saveduser.htm)
                // 2) this.m_strCredType is the value from registry, it is used to handle the error from confirm page
                //
                if ( WLID_LAST_USED_CRED_EID == this.m_objEx.getProperty(k_wstrInProp_LastLoginType)  || 
                      WLID_LAST_USED_CRED_EID == this.m_strCredType ||
                      WLID_LAST_USED_CRED_FED_EID == this.m_strCredType )
                    return true;
                else
                    return false;
            }
        }

        // feration case
        if ( this.m_bFederated && this.m_strCredType == WLID_LAST_USED_CRED_FED_EID )
            return true;

        if ( this.m_bFederated && this.m_strCredType == WLID_LAST_USED_CRED_FED_PASSWORD )
            return false;
                
        // last visited page = null, not an error case, first honor force eid, then last signin type
        //
        if ( x_wstrUIOptionValueTRUE == this.m_objEx.getProperty( k_wstrInProp_SmartCardForce ) )
            return true;

        var arrCertList = this.m_objEx.getCertList('true');

        if ( (WLID_LAST_USED_CRED_FED_EID ==  this.m_strCredType || WLID_LAST_USED_CRED_EID == this.m_strCredType )
            //&& x_wstrUIOptionValueTRUE == this.m_objEx.getProperty( k_wstrInProp_SmartCardEnabled ) 
            && arrCertList.length  > 0 )
            return true;
        
        return false;        
    },

    // adjust tile position when scroll 
    adjustCombobox:function(scrolltopOffset)
    {
        if ( null == scrolltopOffset )
            return;
        if (undefined == scrolltopOffset )
            return;

        this.m_iScrollTop = scrolltopOffset;
        
        if ( this.m_objComboCredPicker != undefined && this.m_objComboCredPicker != null )
            this.m_objComboCredPicker.adjustPositionOnScroll(scrolltopOffset);

        if ( this.m_objComboCertPicker != undefined && this.m_objComboCertPicker != null )
            this.m_objComboCertPicker.adjustPositionOnScroll(scrolltopOffset);
    },

    // some preset properties are not honored in XP, need to refresh themm
    // after the elements have been rendered
    refreshElements:function()
    {
        this.m_objChkSavePassword.refresh();
    },

    // setfocus to this tile
    setFocus:function()
    {
        if(this.m_objDivUserTile != undefined)
        {
            this.m_objDivUserTile.setTabIndex(0);
            this.m_objDivUserTile.getElement().focus();
        }
    },     

    setActiveLayout:function()
    {
        var layouttable =
        "<table  class='internallayout' cellpadding='0' cellspacing='0'>" + 
        "<tbody>" + 
            "<tr>" + 
                "<td colspan='3' class='nonepadding'>" + 
                "<table  class='internallayout' cellpadding='0' cellspacing='0'>" + 
                    "<tbody>" + 
                        "<tr>" + 
                            "<td class='nonepadding'>" + 
                                "<div id='"+this.idCredPickerCbo+this.m_strUniqueID + "'>" +"</div>" + 
                            "</td>" + 
                        "</tr>" +      
                        "<tr>" + 
                            "<td style='height: 3px;' />" +
                        "</tr>" +                         
                    "</tbody>" + 
                "</table>" + 
                "</td>" + 
            "</tr>" +         
            "<tr>" + 
                "<td colspan='3'  class='nonepadding'>" + 
                "<table class='internallayout' cellpadding='0' cellspacing='0' id='" + this.idPassTable+ this.m_strUniqueID + "'>" + 
                    "<tbody>" +                      
                        "<tr>" + 
                            "<td width='188' style='vertical-align:top;' class='nonepadding'>" +
                                "<div id='" + this.idPassTb + this.m_strUniqueID +"'>" +"</div>" + 
                            "</td>" + 
                            "<td width='2'/>" +                             
                            "<td class='nonepadding'>" + 
                                "<div id='" + this.idSigninBt1 + this.m_strUniqueID +  "'>" +"</div>" + 
                            "</td>" + 
                            "<td width='2'/>" +                             
                        "</tr>" + 
                        "<tr>" + 
                            "<td class='nonepadding'>" + 
                                "<div id='" + this.idForgetPassLk + this.m_strUniqueID + "'>" +"</div>" + 
                            "</td>" + 
                            "<td width='2'/>" +                             
                            "<td class='nonepadding'>" + 
                            "</td>" + 
                            "<td width='2'/>" +                             
                        "</tr>" + 
                        "<tr>" + 
                            "<td class='nonepadding' colspan='3'>" + 
                                "<div id='"+this.idRemPasswordCb+this.m_strUniqueID + "'>" +"</div>" + 
                            "</td>" + 
                            "<td width='2'/>" +                             
                        "</tr>" +                         
                    "</tbody>" + 
                "</table>" + 
                "</td>" + 
            "</tr>" +         
            "<tr>" + 
                "<td colspan='3' class='nonepadding'>" + 
                "<table class='internallayout' cellpadding='0' cellspacing='0' id='" + this.idSmartCardTable+ this.m_strUniqueID + "'>" + 
                    "<tbody>" +                     
                        "<tr>" + 
                            "<td width='76' class='nonepadding'>" + 
                                "<div id='"+this.idCertLbl+this.m_strUniqueID + "'>" +"</div>" + 
                            "</td>" + 
                            "<td width='2'/>" +                             
                            "<td width='110' style='vertical-align:top;' class='nonepadding'>" + 
                                "<div id='"+this.idCertCbo+this.m_strUniqueID + "'>" +"</div>" + 
                            "</td>" + 
                            "<td width='2'/>" +                             
                            "<td class='nonepadding' rowspan='4' style='vertical-align:top;'>" + 
                                "<div id='" + this.idSigninBt2 + this.m_strUniqueID +  "'>" +"</div>" + 
                            "</td>" + 
                        "</tr>" + 
                        "<tr>" + 
                            "<td colspan='4' style='height: 3px;' />" +
                        "</tr>" +                         
                        "<tr>" + 
                            "<td class='nonepadding'>" + 
                                "<div id='"+this.idPinLbl+this.m_strUniqueID + "'>" +"</div>" + 
                            "</td>" + 
                            "<td width='2'/>" +                             
                            "<td class='nonepadding'>" + 
                                "<div id='"+this.idPinTb+this.m_strUniqueID + "'>" +"</div>" + 
                            "</td>" + 
                            "<td width='2'/>" +                             
                        "</tr>" + 
                        "<tr>" + 
                            "<td colspan='4' class='nonepadding'>" + 
                                "<div id='"+this.idForgetPin+this.m_strUniqueID + "'>" +"</div>" + 
                            "</td>" + 
                        "</tr>" + 
                    "</tbody>" + 
                "</table>" + 
                "</td>" + 
            "</tr>" +                                         
        "</tbody>" +             
        "</table>";
        this.m_objDivActive.setHTML( layouttable );
    }  
}
