/*****************************************************************************
Created by John Hsieh 2008/3/21 (Copyright)
GetDhtPeers is a program of Linux to get DHT peers of a torrent.
Usage: ./GetDhtPeers torrent info_hash peers_file start_port end_port timeout idle_timeout node_server_port

Modify: 2008/3/27 add GoodDhtNodes server support
******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <errno.h>
#include <sys/wait.h>
#include <time.h>
#include <openssl/sha.h>
#include <sys/fcntl.h>
#include "GetDhtPeers.h"

#define SAVE_ACTIVE_NODE 1
#define SHOW_MSG 1
//#define DEBUG 1
#ifdef DEBUG
#define DEBUG_TORRENT 1
#define DEBUG_RESPONSE 1
#endif

unsigned char g_caInfoHash[SHA_DIGEST_LENGTH];
unsigned short g_nStartPort,g_nEndPort,GOOD_NODE_SERVER_PORT=7999; // allowed port range
time_t g_nDueTime,g_nIdleTimeout;

BtDhtNode g_aDhtNode[MAX_DHT_NODE];

int main(int argc, char **argv)
{
	//int fd=0,nLen=0;
	//char buffer[8192];
	
	// check and process command line arguments
	if(ProcessArgument(argc,argv))
		exit(1); // invalid argument
	
	// read torrent
	if(ReadTorrent(argv[ARG_TORRENT]))
		exit(1); // invalid torrent
		
	// add GoodDhtNodes
	if(AddDhtNode(GOOD_NODE_SERVER_IP,GOOD_NODE_SERVER_PORT)<=0)
	{
		printf("(%s) warning: fail to add GoodDhtNodes to node list\n",THIS_APP);
	}

/*
	//if((fd = open("dht_get_peer_response_003", O_RDONLY) ) < 0)
	if((fd = open("dht_get_peers_response004", O_RDONLY) ) < 0)
	{
		printf("(%s) Error: failed to open torrent file: %s\n",THIS_APP,"dht_get_peers_response004");
		return 1;
	}
	if((nLen = read(fd, buffer, sizeof(buffer))) > 0)
	{
		ProcessDhtResponse(buffer,nLen,argv[3],1,1,1);
	}
	close(fd);
	exit(0);
*/	
	if(GetDhtPeers(g_caInfoHash,argv[ARG_PEERS_FILE],g_nStartPort,g_nEndPort,g_nDueTime,g_nIdleTimeout)==1)
	{
		exit(1); // error happened
	}

	printf("(%s) finished\n",THIS_APP);
	
}

// get and process command line arguments
// return: 0->success; 1->error
int ProcessArgument(int argc, char **argv)
{
	int nTimeout,nLen,i;
	char c;
	
	if (argc != ARG_NUMBER)
	{
		printf("Usage: ./GetDhtPeers torrent info_hash peers_file start_port end_port timeout idle_timeout node_server_port\n");
		return 1;
	}
	
	// get info hash
	bzero(g_caInfoHash, sizeof(g_caInfoHash));
	if((nLen=strlen(argv[ARG_INFO_HASH]))!=(SHA_DIGEST_LENGTH*2))
	{
		printf("(%s) Error: invalid info hash\n",THIS_APP);
		return 1;
	}
	for(i=0;i<SHA_DIGEST_LENGTH*2;i++)
	{
		c=argv[ARG_INFO_HASH][i];
		if(c>='0' && c<='9')
			g_caInfoHash[i/2]=g_caInfoHash[i/2]*16+(unsigned char) (c-'0');
		else if(c>='a' && c<='z')
			g_caInfoHash[i/2]=g_caInfoHash[i/2]*16+(unsigned char) (c-'a'+10);
		else if(c>='A' && c<='Z')
			g_caInfoHash[i/2]=g_caInfoHash[i/2]*16+(unsigned char) (c-'A'+10);
		else
		{
			printf("(%s) Error: invalid info hash\n",THIS_APP);
			return 1;
		}
	}
	
	g_nStartPort=(unsigned short) atoi(argv[ARG_START_PORT]); // get start port
	g_nEndPort=(unsigned short) atoi(argv[ARG_END_PORT]); // get end port
	
	//get due time
	if((nTimeout=atoi(argv[ARG_TIMEOUT]))<1) // timeout in seconds
	{
		printf("(%s) Warning: invalid timeout \"%s\" assigned, use default %d\n",THIS_APP,argv[ARG_TIMEOUT],DEFAULT_TIMEOUT);
		nTimeout=DEFAULT_TIMEOUT;
	}
	if((g_nDueTime=time(NULL))==-1)
	{
		printf("(%s) Error: failed to get system time\n",THIS_APP);
		return 1;
	}
	g_nDueTime+=nTimeout;
	
	//get idle timeout
	if((g_nIdleTimeout=atoi(argv[ARG_IDLE_TIMEOUT]))<1) // idle timeout in seconds
	{
		printf("(%s) Warning: invalid idle timeout \"%s\" assigned, use default %d\n",THIS_APP,argv[ARG_IDLE_TIMEOUT],DEFAULT_IDLE_TIMEOUT);
		g_nIdleTimeout=DEFAULT_IDLE_TIMEOUT;
	}
	
	// get GoodDhtNodes server port
	GOOD_NODE_SERVER_PORT=(unsigned short) atoi(argv[ARG_NODE_PORT]);
	
#ifdef DEBUG
/*	printf("Start port: %d\nEnd port: %d\nTimeout: %d\nDue time: %d\nIdle timeout: %d\n",g_nStartPort,g_nEndPort,nTimeout,g_nDueTime,g_nIdleTimeout);
	printf("Info hash: ");
	for(i=0;i<SHA_DIGEST_LENGTH;i++)
	{
		printf("%.2X ",g_caInfoHash[i]);
	}
	printf("\n");*/
#endif
	
	return 0;
}

// Initialize GetDhtPeers
void Init()
{
	bzero(g_aDhtNode, sizeof(g_aDhtNode));
}

// read torrent file and get nodes
// return: 0->success; 1->error
int ReadTorrent(char *strTorrent)
{
	char	buffer[MAX_BUFFER]; // buffer to keep data
	int nBufferPtr=0;
	int	nLen,fd=0;
	char token[BT_MAX_TOKEN+10];
	int nTokenPtr=0,nTokenPtr2=0,nTokenLen=0,nTokenLeft=0,nFilePos=0;
	int nDepth=0;
	char caType[BT_MAX_DEPTH];
	char caSubType[BT_MAX_DEPTH];
	char c,*pStr=0;
	char bReadNode=0,bBigToken=0;
	char strIP[16]; // IP of node
	unsigned short nPort; // Port of node
	
	if ((fd = open(strTorrent, O_RDONLY) ) < 0)
	{
		printf("(%s) Error: failed to open torrent file: %s\n",THIS_APP,strTorrent);
		return 1;
	}
	
	while((nLen = read(fd, buffer, sizeof(buffer))) > 0)
	{
		nBufferPtr=0;
		
			while(nBufferPtr<nLen)
			{
				// read next byte
				c=buffer[nBufferPtr];
				if((bBigToken==0)||(nTokenPtr==0)) // prevent big token makes buffer over run
					token[nTokenPtr]=c;
#ifdef DEBUG_TORRENT
/*				token[nTokenPtr+1]='\0';
				printf("Debug(%d): %s\n",nFilePos,token);*/
#endif
				
				if(nTokenPtr==0)
				{ // first char of token
					bBigToken=0;
					if(c=='d')
					{ // directory
#ifdef DEBUG_TORRENT
						printf("Debug(%d): Start of directory, Depth(%d)\n",nFilePos,nDepth);
#endif
						caType[nDepth]=BT_TYPE_DIR;
						caSubType[nDepth]=BT_TYPE_NULL;
						nDepth++;
						if(nDepth>BT_MAX_DEPTH)
						{
							printf("(%s) Depth of torrent \"%s\" is too deep\n",THIS_APP,strTorrent);
							close(fd);
							return 1;
						}
						caType[nDepth]=BT_TYPE_NULL;
						caSubType[nDepth]=BT_TYPE_NULL;
						nTokenPtr=0;
					}
					else if(c=='l')
					{ // list
#ifdef DEBUG_TORRENT
						printf("Debug(%d): Start of list, Depth(%d)\n",nFilePos,nDepth);
#endif
						caType[nDepth]=BT_TYPE_LIST;
						caSubType[nDepth]=BT_TYPE_NULL;
						nDepth++;
						if(nDepth>BT_MAX_DEPTH)
						{
							printf("(%s) Depth of torrent \"%s\" is too deep\n",THIS_APP,strTorrent);
							close(fd);
							return 1;
						}
						caType[nDepth]=BT_TYPE_NULL;
						caSubType[nDepth]=BT_TYPE_NULL;
						nTokenPtr=0;
						
					} //else if(c=='l')
					else if(c=='i')
					{ // int
#ifdef DEBUG_TORRENT
						printf("Debug(%d): Start of int, Depth(%d)\n",nFilePos,nDepth);
#endif
						if(caType[nDepth-1]==BT_TYPE_DIR)
						{ // start of a directory key
							caType[nDepth]=BT_TYPE_KEY;
							caSubType[nDepth]=BT_TYPE_INT;
						}
						else if((caType[nDepth-1]==BT_TYPE_LIST) || (caType[nDepth-1]==BT_TYPE_KEY))
						{ // start of a list element or start of a directory value
							caType[nDepth]=BT_TYPE_VALUE;
							caSubType[nDepth]=BT_TYPE_INT;
						}
						nTokenPtr++;
					}
					else if((c>='0') && (c<='9'))
					{ // string
#ifdef DEBUG_TORRENT
						printf("Debug(%d): Start of string, Depth(%d)\n",nFilePos,nDepth);
#endif
						if(caType[nDepth-1]==BT_TYPE_DIR)
							caType[nDepth]=BT_TYPE_KEY; // start of a directory key
						else if((caType[nDepth-1]==BT_TYPE_LIST) || (caType[nDepth-1]==BT_TYPE_KEY))
							caType[nDepth]=BT_TYPE_VALUE; // start of a list value or start of a directory value
						
						caSubType[nDepth]=BT_TYPE_STR1;
						nTokenPtr++;
					} //else if((c>='0') && (c<='9'))
					else if(c=='e')
					{ // end of dictionary or list
						//decrease level
						nDepth--;
#ifdef DEBUG_TORRENT
						printf("Debug(%d): End of dictionary or list, Depth(%d)\n",nFilePos,nDepth);
#endif
						if(caType[nDepth-1]==BT_TYPE_KEY)
						{ // should decrease further level
							nDepth--;
#ifdef DEBUG_TORRENT
							printf("Debug(%d): Meet end of dictionary value when end of dictionary or list, Depth(%d)\n",nFilePos,nDepth);
#endif
						}
						
						if((bReadNode==1) && (nDepth==2))
						{ // end to read node items
#ifdef DEBUG
							printf("Debug(%d): Finished read node, Depth(%d)\n",nFilePos,nDepth);
#endif
							bReadNode=0;
						}
						else if((bReadNode==1) && (nDepth==3))
						{ // get one complete node IP and Port, add node
							AddDhtNode(strIP,nPort);
						}
						nTokenPtr=0;
					} //else if(c=='e')
				} //if(nTokenPtr==0)
				else
				{ // nTokenPtr>0
					if((c==':') && (caSubType[nDepth]==BT_TYPE_STR1))
					{ // get complete length of string
						caSubType[nDepth]=BT_TYPE_STR2;
						nTokenPtr2=nTokenPtr+1; // start of token content
						token[nTokenPtr]='\0';
						nTokenLen=atoi(token); // token length
						nTokenLeft=nTokenLen; // left char to read in token
#ifdef DEBUG_TORRENT
//						printf("Debug(%d): string length (%d), Depth(%d) BT_TYPE_STR2 nTokenPtr2=%d\n",nFilePos,nTokenLen,nDepth,nTokenPtr2);
#endif
						if(nTokenLen>BT_MAX_TOKEN)
						{ // token too long
							bBigToken=1;
							//printf("Error(%d): string token is too long\n",nFilePos);
							//return 1;
						}
						nTokenPtr++;
					} //if((c==':') && (caSubType[nDepth]==BT_TYPE_STR1))
					else if((c=='e') && (caSubType[nDepth]==BT_TYPE_INT))
					{ // get complete int
						token[nTokenPtr]='\0';
						pStr=token+1;
						if((bReadNode==1) && (nDepth==4))
						{ // get port of a node
							nPort=(unsigned short) atoi(pStr);
						}
#ifdef DEBUG_TORRENT
						printf("Debug(%d): Read int \"%s\"\n",nFilePos,pStr);
#endif
						if(caType[nDepth]==BT_TYPE_KEY)
						{ // end of a key, next is value -> increase level
							nDepth++;
							if(nDepth>BT_MAX_DEPTH)
							{
								printf("(%s) Depth of torrent \"%s\" is too deep\n",THIS_APP,strTorrent);
								close(fd);
								return 1;
							}
							caType[nDepth]=BT_TYPE_NULL;
							caSubType[nDepth]=BT_TYPE_NULL;
						} //if(caType[nDepth]==BT_TYPE_KEY)
						else if((caType[nDepth]==BT_TYPE_VALUE) && (caType[nDepth-1]==BT_TYPE_KEY))
						{ // end of a key value and decrease level
							nDepth--;
							caType[nDepth]=BT_TYPE_NULL;
							caSubType[nDepth]=BT_TYPE_NULL;
						} // else if((caType[nDepth]==BT_TYPE_VALUE) && (caType[nDepth-1]==BT_TYPE_KEY))
						else if((caType[nDepth]==BT_TYPE_VALUE) && (caType[nDepth-1]==BT_TYPE_LIST))
						{ // end of a list element
							caType[nDepth]=BT_TYPE_NULL;
							caSubType[nDepth]=BT_TYPE_NULL;
						}
								
						nTokenPtr=nTokenPtr2=nTokenLeft=nTokenLen=0; // start a new token
					} //else if((c=='e') && (caSubType[nDepth]==BT_TYPE_INT))
					else if((caSubType[nDepth]==BT_TYPE_STR2) && (nTokenLeft==0))
					{ // read a complete string token
						if(bBigToken==0)
						{ // prevent big token from making buffer over run
							token[nTokenPtr2+nTokenLen]='\0';
							pStr=token+nTokenPtr2;
#ifdef DEBUG_TORRENT
							printf("Debug(%d): Read token \"%s\" (%d,%d)\n",nFilePos,pStr,nTokenPtr2,nTokenLen);
#endif
							if(nDepth==1)
							{ // first level token
								if(strcmp(pStr,"nodes")==0)
								{ // bingo, reach nodes
#ifdef DEBUG
									printf("Debug(%d): Read start of nodes\n======================================\n",nFilePos);
#endif
									bReadNode=1;
								}
							} //if(nDepth==1)
							else if((bReadNode==1) && (nDepth==4))
							{ // get IP of a node
								if(nTokenLen>15)
								{ // invalid IP
									strIP[0]='\0';
									printf("Debug(%d): Invalid IP \"%s\"\n",nFilePos,pStr);
								}
								else
								{ // may be an IP, keep it
									strcpy(strIP,pStr);
								}
							}
						} //if(bBigToken==0)
								
						if(caType[nDepth]==BT_TYPE_KEY)
						{ // end of a key, next is value -> increase level
							nDepth++;
							if(nDepth>BT_MAX_DEPTH)
							{
								printf("(%s) Depth of torrent \"%s\" is too deep\n",THIS_APP,strTorrent);
								close(fd);
								return 1;
							}
							caType[nDepth]=BT_TYPE_NULL;
							caSubType[nDepth]=BT_TYPE_NULL;
						} //if(caType[nDepth]==BT_TYPE_KEY)
						else if((caType[nDepth]==BT_TYPE_VALUE) && (caType[nDepth-1]==BT_TYPE_KEY))
						{ // end of a key value and decrease level
							nDepth--;
							caType[nDepth]=BT_TYPE_NULL;
							caSubType[nDepth]=BT_TYPE_NULL;
						} // else if((caType[nDepth]==BT_TYPE_VALUE) && (caType[nDepth-1]==BT_TYPE_KEY))
						else if((caType[nDepth]==BT_TYPE_VALUE) && (caType[nDepth-1]==BT_TYPE_LIST))
						{ // end of a list element
							caType[nDepth]=BT_TYPE_NULL;
							caSubType[nDepth]=BT_TYPE_NULL;
						}
								
						nTokenPtr=nTokenPtr2=nTokenLeft=nTokenLen=0; // start a new token
						
					} //else if((caSubType[nDepth]==BT_TYPE_STR2) && (nTokenLeft==1))
					else
					{
						nTokenPtr++;
					}
					
					if(caSubType[nDepth]==BT_TYPE_STR2)
					{
#ifdef DEBUG_TORRENT
/*						token[nTokenPtr]='\0';
						pStr=token+nTokenPtr2;
						printf("Debug(%d)(%d): %s\n",nFilePos,nTokenLeft,pStr);*/
#endif
						nTokenLeft--;
					}
						
				} //if(nTokenPtr==0) else
				nBufferPtr++; // buffer position
				nFilePos++; // file position
			} //while(nBufferPtr<nLen)
		
	} //while((nLen = read(fd, buffer, sizeof(buffer))) > 0)
	
	close(fd);
	
	return 0;
}

// Add a node
// Input:
//         strIP: the ip of node
//         port: the port of node in host endian
// return: >0: success and the value is the index in the node list
//         -1: invalid ip, -2: node list full
int AddDhtNode(char *strIP, unsigned short port)
{
	unsigned long nIP;
	unsigned char *pIP;
	unsigned short nPort;
	int i,nFirstEmpty=-1;
	char bFound=0;
	
	nPort=htons(port);
	
	if(inet_pton(AF_INET, strIP, &nIP)<=0)
	{ // not a valid IP address
#ifdef DEBUG
		printf("Error: \"%s\" is not a valid IP address\n",strIP);
#endif
		return -1;
	}
	
#ifdef DEBUG
	printf("Try to add a node: IP=\"%s\" (%X), Port=\"%d\"\n",strIP,nIP,port);
#endif

	pIP=(unsigned char *)&nIP;
	
	// find an empty node, and check if the node has already existed
	for(i=0;i<MAX_DHT_NODE;i++)
	{
		if((g_aDhtNode[i].status==BT_NODE_NULL) && (nFirstEmpty == -1))
			nFirstEmpty=i; // find the first empty node entry
		if((g_aDhtNode[i].ip==nIP) && (g_aDhtNode[i].port==nPort))
		{ // the node already in the node list
			bFound=1;
			break;
		}
	}

	if(bFound==1)
	{
#ifdef DEBUG
		printf("Warning: Node \"%s:%d\" has already been in the node list\n",strIP,port);
#endif
		return i;
	}
	
	// the node is not in the node list
	// add the node into the node list if the list is not full
	
	if(nFirstEmpty == -1)
	{ // the node list is full
		
		// Hint by John:
		//// We can consider to reclaim all node in the state BT_NODE_RETURN_NODE and BT_NODE_RETURN_PEER,
		//// because they have already replied.
		
#ifdef DEBUG
		printf("Warning: Node list is full and \"%s:%d\" is not added.\n",strIP,port);
#endif
		return -2;
	}
	
	// add the node into the node list
	g_aDhtNode[nFirstEmpty].ip=nIP;
	g_aDhtNode[nFirstEmpty].port=nPort;
	g_aDhtNode[nFirstEmpty].status=BT_NODE_NEW;
	

#ifdef SHOW_MSG
	printf("(%s) Add a new node %d.%d.%d.%d:%d at position (%d) in the node list\n",THIS_APP,pIP[0],pIP[1],pIP[2],pIP[3],ntohs(nPort),nFirstEmpty);
#endif

	return nFirstEmpty;
}

// Add a node from buffer get from a DHT response
// return: >=0: number of nodes added
//         -1: invalid format
int AddDhtResponseNode(char *buffer, int bufferLen)
{
	int nAddedNode=0;
	int nPtr=0;
	unsigned char *pcIP;
	unsigned short *pnPort;
	char strIP[16];

	if((bufferLen%26)!=0)
		return -1;
	
	while(nPtr<bufferLen)
	{
		nPtr+=20; // skip node id
		// get ip of node
		pcIP=(unsigned char *) (buffer+nPtr);
		sprintf(strIP,"%d.%d.%d.%d",pcIP[0],pcIP[1],pcIP[2],pcIP[3]);
		nPtr+=4;
		// get port of node
		pnPort=(unsigned short *) (buffer+nPtr);
		
		// add node to node list
		if(AddDhtNode(strIP,ntohs(*pnPort)) > 0)
			nAddedNode++;
		nPtr+=2;
	} //while(nPtr<bufferLen)
	
	return nAddedNode;
}

// Get all available DHT peers from nodes stored in the node list g_aDhtNode.
// The UDP server will only use one available port in the range from nStartPort to nEndPort.
// Return: 0->successful, 1->fetal error
int GetDhtPeers(unsigned char *pcInfoHash, char *strPeersFile, unsigned short nStartPort, unsigned short nEndPort,time_t nDueTime,time_t nIdleTimeout)
{
	int	udpfd, maxfdp1, outfd=0;
	int nReady;
	char mesg[MAX_BUFFER]; // buffer to keep packet data
	char request[128]; // request buffer
	char addNodeBuffer[10];
	char strTemp[256];
	fd_set	rset;
	ssize_t	nPacketLen; // packet length
	socklen_t	len;
	struct sockaddr_in nodeAddress, serverAddress, nodeServerAddress;
	unsigned long nGoodNodesServerIP,*pnIP;
	//unsigned int nRemoteAddress;
	unsigned short int nRemotePort,nServerPort,*pnPort;
	unsigned char *pnAddress,*pcRequest,*pcTemp;
	struct timeval idleTimeout;
	int i,nRequestLen,nParentID;
	unsigned char macAddress[6]={0x00,0x00,0xE2,0x64,0xCD,0x9E};
	//void	sig_chld(int);
	
	//for(i=0;i<6;i++)
	//	printf("%.2X",macAddress[i]);
	//printf("\n");
	
#ifdef DEBUG
/*
	int infd=0;
	if ((infd = open("dht_get_peers_request", O_RDONLY) ) < 0)
	{
		printf("Debug Error: failed to open request file: dht_get_peers_request\n");
		return 1;
	}
	
	if((nRequestLen = read(infd, request, sizeof(request))) > 0)
	{
		printf("Debug Error: read request file \"dht_get_peers_request\" ok\n");
	}
	else
	{
		printf("Debug Error: failed to read request file: \"dht_get_peers_request\"\n");
	}
	close(infd);
*/
#endif

		
	// prepare get_peers request packet
	strcpy(request,"d1:ad2:id20:");
	nRequestLen=strlen("d1:ad2:id20:");
	pcRequest=(unsigned char *) (request+nRequestLen);
	// add node id (user MAC address as node id)
	if(GetMacAddress(macAddress) == 1)
		printf("(%s) Error: failed to get MAC address, use default value\n",THIS_APP);
	strcpy(request+nRequestLen,NODE_ID_PREFIX);
	for(i=strlen(NODE_ID_PREFIX);i<14;i++)
		pcRequest[i]=0;
	//for(i=0;i<14;i++)
	//	pcRequest[i]=0;
	pcRequest+=14;
	for(i=0;i<6;i++)
		pcRequest[i]=macAddress[i];
	pcRequest+=6;
	nRequestLen+=20;
	// add info hash
	strcpy(request+nRequestLen,"9:info_hash20:");
	nRequestLen+=strlen("9:info_hash20:");
	pcRequest=(unsigned char *) (request+nRequestLen);
	for(i=0;i<SHA_DIGEST_LENGTH;i++)
		pcRequest[i]=pcInfoHash[i]; // add info hash
	nRequestLen+=SHA_DIGEST_LENGTH;
	// get_peers command
	strcpy(request+nRequestLen,"e1:q9:get_peers1:t4:");
	nRequestLen+=strlen("e1:q9:get_peers1:t4:");
	// transaction id (use parent process id as transaction id)
	nParentID=(int)getppid();
	pcTemp=(unsigned char *) &nParentID;
	pcRequest=(unsigned char *) (request+nRequestLen);
	for(i=0;i<4;i++)
		pcRequest[i]=pcTemp[i];
	nRequestLen+=4;
	// request
	strcpy(request+nRequestLen,"1:y1:qe");
	nRequestLen+=strlen("1:y1:qe");
	// Finished preparing request pakcet

#ifdef DEBUG
/*
	printf("Request packet length (%d)\n",nRequestLen);
	if((outfd = open(strPeersFile, O_WRONLY|O_CREAT|O_APPEND,0644))!=-1)
	{
		sprintf(strTemp,"====== Request packet ======");
		write(outfd,strTemp,strlen(strTemp));
		write(outfd,request,nRequestLen);
		close(outfd);
	}
	else
		printf("(%s) Error: failed to open file \"%s\"\n",THIS_APP,strPeersFile);
*/
#endif

	// create UDP server socket
	udpfd = socket(AF_INET, SOCK_DGRAM, 0);

	// prepare server address
	bzero(&serverAddress, sizeof(serverAddress));
	serverAddress.sin_family = AF_INET;
	serverAddress.sin_addr.s_addr = htonl(INADDR_ANY); // server listen to all interface
	
	// prepare node address
	bzero(&nodeAddress, sizeof(nodeAddress));
	nodeAddress.sin_family = AF_INET;
	//nodeAddress.sin_port = htons(SERV_PORT);
	//inet_pton(AF_INET, argv[1], &nodeAddress.sin_addr);
	
	//nodeServerAddress
	// prepare node address
	bzero(&nodeServerAddress, sizeof(nodeServerAddress));
	nodeAddress.sin_family = AF_INET;
	nodeServerAddress.sin_port = htons(GOOD_NODE_SERVER_PORT);
	inet_pton(AF_INET,"127.0.0.1",&nGoodNodesServerIP);
	nodeServerAddress.sin_addr.s_addr=nGoodNodesServerIP;

	// bind an available port for server
	nServerPort=nStartPort;
	while(nServerPort<=nEndPort)
	{
#ifdef DEBUG
		printf("(%s) Try to bind port %d\n",THIS_APP,nServerPort);
#endif
		serverAddress.sin_port = htons(nServerPort);

		if(bind(udpfd, (SA *) &serverAddress, sizeof(serverAddress)) == 0) // try to bind a port
			break; // success
		else
		{ // fail to bind this port
			if(errno == EADDRINUSE)
			{ // this port is used
#ifdef DEBUG
				printf("(%s) Warning: port %d has already been used\n",THIS_APP,nServerPort);
#endif
			}
			else
			{
				printf("(%s) Error: Socket bind error number (%d)\n",THIS_APP,errno);
				return 1;
			}
		}
		nServerPort++; // try next port
	} //while(nServerPort<=nEndPort)
	if(nServerPort>nEndPort)
	{ /* no available port for server */
		printf("Error: No available server port to bind\n");
		return 1;
	}
	
	//Bind(udpfd,(struct sockaddr *) &serverAddress, sizeof(serverAddress));
	//Signal(SIGCHLD, sig_chld);/* must call waitpid() */
	//signal(SIGCHLD,SIG_IGN); // needless waitpid()

	printf("(%s) Started on port: %d\n",THIS_APP,nServerPort);
	
	FD_ZERO(&rset);
	//maxfdp1 = max(listenfd, udpfd) + 1;
	maxfdp1 = udpfd + 1;
	
	// start to get DHT peers until timeout or idle timeout
	//for ( ; ; )
	while(time(NULL)<=nDueTime)
	{
		// send request to all new nodes
		for(i=0;i<MAX_DHT_NODE;i++)
		{
			if(g_aDhtNode[i].status != BT_NODE_NEW)
				continue; // not a new node
			
			// this is a new node
			// send request to this node
			nodeAddress.sin_addr.s_addr=g_aDhtNode[i].ip;
			nodeAddress.sin_port=g_aDhtNode[i].port;
			len = sizeof(nodeAddress);
			sendto(udpfd, request, nRequestLen, 0, (SA *) &nodeAddress, len);
#ifdef SHOW_MSG
			pnAddress= (unsigned char *) &(g_aDhtNode[i].ip);
			printf("(%s) Send request to node %d.%d.%d.%d:%d\n",THIS_APP,*pnAddress,*(pnAddress+1),*(pnAddress+2),*(pnAddress+3),ntohs(g_aDhtNode[i].port));
#endif
			g_aDhtNode[i].status=BT_NODE_REQUEST;
		} //for(i=0;i<MAX_DHT_NODE;i++)
		
		// setup read fd of select
		FD_SET(udpfd, &rset);
		
		// setup select idle timeout
		idleTimeout.tv_sec=nIdleTimeout;
		idleTimeout.tv_usec=0;
		
		if ( (nReady = select(maxfdp1, &rset, NULL, NULL, &idleTimeout)) < 0)
		{ /* may be an error occurs */
			printf("(%s) Get signal\n",THIS_APP);
			if (errno == EINTR)
				continue;               /* back to for() */
			else
			{ /* select error */
				printf("(%s) Select error\n",THIS_APP);
				exit(1);
			}
		}
		
		if(nReady==0)
		{ // idle timeout: select return due to timeout
			printf("(%s) warning: idle timeout %d seconds\n",THIS_APP,nIdleTimeout);
			break;
		}
		
		if (FD_ISSET(udpfd, &rset))
		{ // a UDP packet arrived
			
			// receive packet
			len = sizeof(nodeAddress);
			nPacketLen = recvfrom(udpfd, mesg, MAX_BUFFER, 0, (SA *) &nodeAddress, &len);
			
			pnAddress=(unsigned char *) &nodeAddress.sin_addr.s_addr;
			nRemotePort = nodeAddress.sin_port;
#ifdef SHOW_MSG
			printf("(%s) Receive packet(%d) from %d.%d.%d.%d:%d\n",THIS_APP,nPacketLen,*pnAddress,*(pnAddress+1),*(pnAddress+2),*(pnAddress+3),ntohs(nRemotePort));
#endif

			// add node to GoodDhtNodes server
			if(nGoodNodesServerIP != nodeAddress.sin_addr.s_addr)
			{ // the reply node is not GoodDhtNodes server
				addNodeBuffer[0]='-';
				pnIP=(unsigned long *) (addNodeBuffer+1);
				*pnIP=nodeAddress.sin_addr.s_addr;
				pnPort=(unsigned short *) (addNodeBuffer+5);
				*pnPort=nodeAddress.sin_port;
				sendto(udpfd, addNodeBuffer,7, 0, (SA *) &nodeServerAddress, sizeof(nodeServerAddress));
			}
				
#ifdef SAVE_ACTIVE_NODE
			if((outfd = open("ActiveNodes.txt", O_WRONLY|O_CREAT|O_APPEND,0644))!=-1)
			{
				//printf("outfd=%d\n",outfd);
				sprintf(strTemp,"%d.%d.%d.%d:%d",*pnAddress,*(pnAddress+1),*(pnAddress+2),*(pnAddress+3),ntohs(nRemotePort));
				write(outfd,strTemp,strlen(strTemp));
				close(outfd);
			}
			else
				printf("(%s) Error: failed to open file \"ActiveNode.txt\"\n",THIS_APP);
#endif

			ProcessDhtResponse(mesg,nPacketLen,strPeersFile,nodeAddress.sin_addr.s_addr,nodeAddress.sin_port,nParentID);
			
#ifdef DEBUG
/*
			//printf("(%s) Get packet(%d) from %d.%d.%d.%d:%d\n",THIS_APP,nPacketLen,*pnAddress,*(pnAddress+1),*(pnAddress+2),*(pnAddress+3),ntohs(nRemotePort));
			//outfd = open(strPeersFile, O_WRONLY|O_CREAT|O_TRUNC|O_APPEND,0644);
			if((outfd = open(strPeersFile, O_WRONLY|O_CREAT|O_APPEND,0644))!=-1)
			{
				//printf("outfd=%d\n",outfd);
				sprintf(strTemp,"====== (%d bytes) from %d.%d.%d.%d:%d ======",nPacketLen,*pnAddress,*(pnAddress+1),*(pnAddress+2),*(pnAddress+3),ntohs(nRemotePort));
				write(outfd,strTemp,strlen(strTemp));
				write(outfd,mesg,nPacketLen);
				close(outfd);
			}
			else
				printf("(%s) Error: failed to open file \"%s\"\n",THIS_APP,strPeersFile);
			//nRemoteAddress = ntohl(nodeAddress.sin_addr.s_addr);
			//pnAddress= (unsigned char *) &nRemoteAddress;
*/
#endif
		} //if (FD_ISSET(udpfd, &rset))
	} //while(time(NULL)<=nDueTime)
	return 0;
}

// get the MAC address of this device
// return: 0->success, 1->fail
int GetMacAddress(unsigned char *macAddress)
{
	printf("(%s) Hint: you have not implement function GetMacAddress()\n",THIS_APP);
	return 1;
}

// process response of DHT request to node
// if the node return nodes, add them into node list
// if the node return values, add them to the peers file
// Input: buffer -> response packet content
//        bufferLen -> length of packet content
//        strPeersFile -> file to store peers
//        ip -> ip of node who respond
//        port -> port of node who respond
// return: 0:success, 1:fail
int ProcessDhtResponse(char *buffer,int bufferLen,char *strPeersFile,unsigned long ip, unsigned short port, int transactionId)
{
	//char buffer[MAX_BUFFER]; // buffer to keep data
	int nBufferPtr=0;
	//int	nLen,fd=0;
	char token[BT_MAX_TOKEN+10];
	int nTokenPtr=0,nTokenPtr2=0,nTokenLen=0,nTokenLeft=0,nFilePos=0;
	int nDepth=0;
	char caType[BT_MAX_DEPTH];
	char caSubType[BT_MAX_DEPTH];
	char c,*pStr=0;
	char bReadNode=0,bReadValue=0,bBigToken=0;
	char strIP[16]; // IP of node
	unsigned short nPort; // Port of node
	
	nBufferPtr=0;
		
			while(nBufferPtr<bufferLen)
			{
				// read next byte
				c=buffer[nBufferPtr];
				if((bBigToken==0)||(nTokenPtr==0)) // prevent big token makes buffer over run
					token[nTokenPtr]=c;
#ifdef DEBUG_RESPONSE
/*				token[nTokenPtr+1]='\0';
				printf("Debug(%d): %s\n",nFilePos,token);*/
#endif
				
				if(nTokenPtr==0)
				{ // first char of token
					bBigToken=0;
					if(c=='d')
					{ // directory
#ifdef DEBUG_RESPONSE
						printf("Debug(%d): Start of directory, Depth(%d)\n",nFilePos,nDepth);
#endif
						caType[nDepth]=BT_TYPE_DIR;
						caSubType[nDepth]=BT_TYPE_NULL;
						nDepth++;
						if(nDepth>BT_MAX_DEPTH)
						{
							printf("(%s) Depth of response is too deep\n",THIS_APP);
							return 1;
						}
						caType[nDepth]=BT_TYPE_NULL;
						caSubType[nDepth]=BT_TYPE_NULL;
						nTokenPtr=0;
					}
					else if(c=='l')
					{ // list
#ifdef DEBUG_RESPONSE
						printf("Debug(%d): Start of list, Depth(%d)\n",nFilePos,nDepth);
#endif
						caType[nDepth]=BT_TYPE_LIST;
						caSubType[nDepth]=BT_TYPE_NULL;
						nDepth++;
						if(nDepth>BT_MAX_DEPTH)
						{
							printf("(%s) Depth of response is too deep\n",THIS_APP);
							return 1;
						}
						caType[nDepth]=BT_TYPE_NULL;
						caSubType[nDepth]=BT_TYPE_NULL;
						nTokenPtr=0;
						
					} //else if(c=='l')
					else if(c=='i')
					{ // int
#ifdef DEBUG_RESPONSE
						printf("Debug(%d): Start of int, Depth(%d)\n",nFilePos,nDepth);
#endif
						if(caType[nDepth-1]==BT_TYPE_DIR)
						{ // start of a directory key
							caType[nDepth]=BT_TYPE_KEY;
							caSubType[nDepth]=BT_TYPE_INT;
						}
						else if((caType[nDepth-1]==BT_TYPE_LIST) || (caType[nDepth-1]==BT_TYPE_KEY))
						{ // start of a list element or start of a directory value
							caType[nDepth]=BT_TYPE_VALUE;
							caSubType[nDepth]=BT_TYPE_INT;
						}
						nTokenPtr++;
					}
					else if((c>='0') && (c<='9'))
					{ // string
#ifdef DEBUG_RESPONSE
						printf("Debug(%d): Start of string, Depth(%d)\n",nFilePos,nDepth);
#endif
						if(caType[nDepth-1]==BT_TYPE_DIR)
							caType[nDepth]=BT_TYPE_KEY; // start of a directory key
						else if((caType[nDepth-1]==BT_TYPE_LIST) || (caType[nDepth-1]==BT_TYPE_KEY))
							caType[nDepth]=BT_TYPE_VALUE; // start of a list value or start of a directory value
						
						caSubType[nDepth]=BT_TYPE_STR1;
						nTokenPtr++;
					} //else if((c>='0') && (c<='9'))
					else if(c=='e')
					{ // end of dictionary or list
						//decrease level
						nDepth--;
#ifdef DEBUG_RESPONSE
						printf("Debug(%d): End of dictionary or list, Depth(%d)\n",nFilePos,nDepth);
#endif
						if(caType[nDepth-1]==BT_TYPE_KEY)
						{ // should decrease further level
							nDepth--;
#ifdef DEBUG_RESPONSE
							printf("Debug(%d): Meet end of dictionary value when end of dictionary or list, Depth(%d)\n",nFilePos,nDepth);
#endif
						}
						
						if((bReadValue==1) && (nDepth==3))
						{ // end to read value items
#ifdef DEBUG_RESPONSE
							printf("Debug(%d): Finished read value, Depth(%d)\n",nFilePos,nDepth);
#endif
							bReadValue=0;
						}
						/*else if((bReadNode==1) && (nDepth==3))
						{ // get one complete node IP and Port, add node
							AddDhtNode(strIP,nPort);
						}*/
						nTokenPtr=0;
					} //else if(c=='e')
				} //if(nTokenPtr==0)
				else
				{ // nTokenPtr>0
					if((c==':') && (caSubType[nDepth]==BT_TYPE_STR1))
					{ // get complete length of string
						caSubType[nDepth]=BT_TYPE_STR2;
						nTokenPtr2=nTokenPtr+1; // start of token content
						token[nTokenPtr]='\0';
						nTokenLen=atoi(token); // token length
						nTokenLeft=nTokenLen; // left char to read in token
#ifdef DEBUG_RESPONSE
//						printf("Debug(%d): string length (%d), Depth(%d) BT_TYPE_STR2 nTokenPtr2=%d\n",nFilePos,nTokenLen,nDepth,nTokenPtr2);
#endif
						if(nTokenLen>BT_MAX_TOKEN)
						{ // token too long
							bBigToken=1;
							//printf("Error(%d): string token is too long\n",nFilePos);
							//return 1;
						}
						nTokenPtr++;
					} //if((c==':') && (caSubType[nDepth]==BT_TYPE_STR1))
					else if((c=='e') && (caSubType[nDepth]==BT_TYPE_INT))
					{ // get complete int
						token[nTokenPtr]='\0';
						pStr=token+1;
						/*if((bReadNode==1) && (nDepth==4))
						{ // get port of a node
							nPort=(unsigned short) atoi(pStr);
						}*/
#ifdef DEBUG_RESPONSE
						printf("Debug(%d): Read int \"%s\"\n",nFilePos,pStr);
#endif
						if(caType[nDepth]==BT_TYPE_KEY)
						{ // end of a key, next is value -> increase level
							nDepth++;
							if(nDepth>BT_MAX_DEPTH)
							{
								printf("(%s) Depth of response is too deep\n",THIS_APP);
								return 1;
							}
							caType[nDepth]=BT_TYPE_NULL;
							caSubType[nDepth]=BT_TYPE_NULL;
						} //if(caType[nDepth]==BT_TYPE_KEY)
						else if((caType[nDepth]==BT_TYPE_VALUE) && (caType[nDepth-1]==BT_TYPE_KEY))
						{ // end of a key value and decrease level
							nDepth--;
							caType[nDepth]=BT_TYPE_NULL;
							caSubType[nDepth]=BT_TYPE_NULL;
						} // else if((caType[nDepth]==BT_TYPE_VALUE) && (caType[nDepth-1]==BT_TYPE_KEY))
						else if((caType[nDepth]==BT_TYPE_VALUE) && (caType[nDepth-1]==BT_TYPE_LIST))
						{ // end of a list element
							caType[nDepth]=BT_TYPE_NULL;
							caSubType[nDepth]=BT_TYPE_NULL;
						}
								
						nTokenPtr=nTokenPtr2=nTokenLeft=nTokenLen=0; // start a new token
					} //else if((c=='e') && (caSubType[nDepth]==BT_TYPE_INT))
					else if((caSubType[nDepth]==BT_TYPE_STR2) && (nTokenLeft==0))
					{ // read a complete string token
						if(bBigToken==0)
						{ // prevent big token from making buffer over run
							token[nTokenPtr2+nTokenLen]='\0';
							pStr=token+nTokenPtr2;
#ifdef DEBUG_RESPONSE
							printf("Debug(%d): Read token \"%s\" (%d,%d), Depth=(%d)\n",nFilePos,pStr,nTokenPtr2,nTokenLen,nDepth);
#endif
							if(nDepth==3)
							{ // 3rd level token
								if(strcmp(pStr,"values")==0)
								{ // bingo, reach values
#ifdef DEBUG_RESPONSE
									printf("Debug(%d): Read start of values\n======================================\n",nFilePos);
#endif
									bReadValue=1;
								}
								else if(strcmp(pStr,"nodes")==0)
								{ // bingo, reach nodes
#ifdef DEBUG_RESPONSE
									printf("Debug(%d): Read start of nodes\n======================================\n",nFilePos);
#endif
									bReadNode=1;
								}
							} //if(nDepth==3)
							else if((bReadValue==1) && (nDepth==5))
							{ // get a peer
								if(nTokenLen!=6)
								{ // invalid peer
									printf("Debug(%d): Invalid peer length (%d)\n",nFilePos,nTokenLen);
								}
								else
								{ // pStr is a peer value
									AddDhtPeer(pStr,strPeersFile); // output to peers file
									SetDhtNodeStatus(ip,port,BT_NODE_RETURN_PEER);
								}
							} //else if((bReadValue==1) && (nDepth==5))
							else if((bReadNode==1) && (nDepth==4))
							{ // get a node, node length is nTokenLen, node value is pStr
								AddDhtResponseNode(pStr,nTokenLen);
								SetDhtNodeStatus(ip,port,BT_NODE_RETURN_NODE);
								bReadNode=0;
#ifdef DEBUG_RESPONSE
								printf("Debug(%d): Finished read node, Depth(%d)\n",nFilePos,nDepth);
#endif
							} //else if((bReadNode==1) && (nDepth==4))
						} //if(bBigToken==0)
								
						if(caType[nDepth]==BT_TYPE_KEY)
						{ // end of a key, next is value -> increase level
							nDepth++;
							if(nDepth>BT_MAX_DEPTH)
							{
								printf("(%s) Depth of response is too deep\n",THIS_APP);
								return 1;
							}
							caType[nDepth]=BT_TYPE_NULL;
							caSubType[nDepth]=BT_TYPE_NULL;
						} //if(caType[nDepth]==BT_TYPE_KEY)
						else if((caType[nDepth]==BT_TYPE_VALUE) && (caType[nDepth-1]==BT_TYPE_KEY))
						{ // end of a key value and decrease level
							nDepth--;
							caType[nDepth]=BT_TYPE_NULL;
							caSubType[nDepth]=BT_TYPE_NULL;
						} // else if((caType[nDepth]==BT_TYPE_VALUE) && (caType[nDepth-1]==BT_TYPE_KEY))
						else if((caType[nDepth]==BT_TYPE_VALUE) && (caType[nDepth-1]==BT_TYPE_LIST))
						{ // end of a list element
							caType[nDepth]=BT_TYPE_NULL;
							caSubType[nDepth]=BT_TYPE_NULL;
						}
								
						nTokenPtr=nTokenPtr2=nTokenLeft=nTokenLen=0; // start a new token
						
					} //else if((caSubType[nDepth]==BT_TYPE_STR2) && (nTokenLeft==1))
					else
					{
						nTokenPtr++;
					}
					
					if(caSubType[nDepth]==BT_TYPE_STR2)
					{
#ifdef DEBUG_RESPONSE
/*						token[nTokenPtr]='\0';
						pStr=token+nTokenPtr2;
						printf("Debug(%d)(%d): %s\n",nFilePos,nTokenLeft,pStr);*/
#endif
						nTokenLeft--;
					}
						
				} //if(nTokenPtr==0) else
				nBufferPtr++; // buffer position
				nFilePos++; // file position
			} //while(nBufferPtr<bufferLen)
	
	return 0;
}

// append DHT peer get from node to the output peers file
// Input:
//        peer: peer info get from the DHT response
//        strPeersFile: file to store peers
// Return: 0:success, 1:fail
int AddDhtPeer(char *peer,char *strPeersFile)
{
	int outfd;
	unsigned char *pcIP;
	unsigned short *pnPort;
	char strPeer[32];
	
	if((outfd = open(strPeersFile, O_WRONLY|O_CREAT|O_APPEND,0644)) ==-1)
		return 1;
	
	pcIP=(unsigned char *) peer;
	pnPort=(unsigned short *) (peer+4);
	sprintf(strPeer,"%d.%d.%d.%d:%d\n",pcIP[0],pcIP[1],pcIP[2],pcIP[3],ntohs(*pnPort));
#ifdef SHOW_MSG
	printf("(%s) Add peer %s",THIS_APP,strPeer);
#endif
	write(outfd,strPeer,strlen(strPeer));
	close(outfd);
	
	return 0;
}

// find the node according to input ip and port
// if found, update its status
void SetDhtNodeStatus(unsigned long ip, unsigned short port, int status)
{
	int i;
	
	for(i=0;i<MAX_DHT_NODE;i++)
	{
		if(g_aDhtNode[i].status == BT_NODE_NULL)
			continue; // empty entry
		if(g_aDhtNode[i].ip != ip)
			continue; // ip not the same
		if(g_aDhtNode[i].port != port)
			continue; // port not the same
		
		// match
		g_aDhtNode[i].status=status; // update status
		break;
	}
}
