/*****************************************************************************
Created by John Hsieh 2008/3/27 (Copyright)
GoodDhtNodes is a program of Linux to maintain good BT DHT nodes.
Usage: ./GoodDhtNodes default_nodes_file nodes_file port timeout
******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <errno.h>
#include <sys/wait.h>
#include <time.h>
//#include <openssl/sha.h>
#include <sys/fcntl.h>
#include "GoodDhtNodes.h"

//#define SAVE_ACTIVE_NODE 1
#define SHOW_MSG 1
#define DEBUG 1
#ifdef DEBUG
//#define DEBUG_READ_NODES_FILE 1
#endif

//unsigned char g_caInfoHash[SHA_DIGEST_LENGTH];
unsigned short g_nPort; // port in host endian format
time_t g_nTimeout; // timeout in seconds
BtDhtNode g_aDhtNode[MAX_DHT_NODE];

int main(int argc, char **argv)
{
	// check and process command line arguments
	if(ProcessArgument(argc,argv))
		exit(1); // invalid argument
	
	// initialize	
	Init();
	
	// read nodes file
	if(ReadNodesFile(argv[ARG_DEFAULT_NODES_FILE],argv[ARG_NODES_FILE]))
		printf("(%s) fail to read nodes file \"%s\"",THIS_APP,argv[ARG_NODES_FILE]);
	
	// start the node server
	if(RunNodeServer(argv[ARG_NODES_FILE],g_nPort,g_nTimeout))
	{
		printf("(%s) Stopped due to error",THIS_APP);
		exit(1);
	}
	
	if(SaveNodesFile(argv[ARG_NODES_FILE]))
		printf("(%s) fail to write nodes file \"%s\"",THIS_APP,argv[ARG_NODES_FILE]);

	printf("(%s) finished\n",THIS_APP);
	
}

// get and process command line arguments
// return: 0->success; 1->error
int ProcessArgument(int argc, char **argv)
{
	if (argc != ARG_NUMBER)
	{
		printf("Usage: ./GoodDhtNodes default_nodes_file nodes_file port timeout\n");
		return 1;
	}
	
	if((g_nPort=(unsigned short) atoi(argv[ARG_PORT]))<1) // get port
	{
		printf("(%s) Warning: invalid port \"%s\" assigned, use default %d\n",THIS_APP,argv[ARG_PORT],DEFAULT_PORT);
		g_nPort=DEFAULT_PORT;
	}
	//get timeout
	if((g_nTimeout=atoi(argv[ARG_TIMEOUT]))<1) // timeout in seconds
	{
		printf("(%s) Warning: invalid timeout \"%s\" assigned, use default %d\n",THIS_APP,argv[ARG_TIMEOUT],DEFAULT_TIMEOUT);
		g_nTimeout=DEFAULT_TIMEOUT;
	}

	return 0;
}

// Initialize GetDhtPeers
void Init()
{
	bzero(g_aDhtNode, sizeof(g_aDhtNode));
}

// read nodes file
// return: 0->success; 1->error
int ReadNodesFile(char *strDefaultNodesFile,char *strNodesFile)
{
	time_t nTime=0;
	
	// get current time
	if((nTime=time(NULL))==-1)
	{
		printf("(%s) warning: failed to get system time\n",THIS_APP);
		//return 1;
	}
	
	// read default node file and normal node file
	// default nodes have higher priority than normal nodes
	ReadNodesFile_0(strDefaultNodesFile,nTime+1);
	ReadNodesFile_0(strNodesFile,nTime);
}

// read nodes file and use time as update time for each node
// return: 0->success; 1->error
int ReadNodesFile_0(char *strNodesFile,time_t nTime)
{
	FILE *fp=0;
	char buffer[32];
	int nPos=0;
	unsigned long nIP;
	unsigned short nPort;
	char *pcPort=0;
		
	if((fp=fopen(strNodesFile,"r"))==NULL)
		return 1;
		
	// read and add one node a time
	while(fgets(buffer,sizeof(buffer),fp) != NULL)
	{
#ifdef DEBUG_READ_NODES_FILE
		printf("(%s) Debug: Read(%d) %s",THIS_APP,strlen(buffer),buffer);
#endif

		// trim new line
		while(buffer[strlen(buffer)-1]=='\n' || buffer[strlen(buffer)-1]=='\r')
			buffer[strlen(buffer)-1]='\0';
#ifdef DEBUG_READ_NODES_FILE
		printf("(%s) Debug: After trim \"%s\"\n",THIS_APP,buffer);
#endif

		for(nPos=0;nPos<strlen(buffer);nPos++)
			if(buffer[nPos]==':')
				break;
		if(nPos>=(strlen(buffer)-1))
			continue; // invalid node
		
		buffer[nPos]='\0';
		pcPort=buffer+nPos+1;
		if(inet_pton(AF_INET,buffer,&nIP)<=0)
		{ // not a valid IP address
#ifdef SHOW_MSG
			printf("(%s) warning: In nodes file [%s] \"%s\" is not a valid IP address\n",THIS_APP,strNodesFile,buffer);
#endif
			continue;
		}
		
		if((nPort=(unsigned short) atoi(pcPort))<1) // get port
		{
#ifdef SHOW_MSG
			printf("(%s) warning: In nodes file invalid port \"%s\"\n",THIS_APP,pcPort);
#endif
			continue;
		}
#ifdef DEBUG_READ_NODES_FILE
		printf("(%s) Debug: good address\n",THIS_APP);
#endif

		AddDhtNode(buffer,nPort,nTime); // add the node to the nodes list
		
	} //while(fgets(buffer,sizeof(buffer),fp) != NULL)
	
	fclose(fp);

	return 0;
}

// save nodes to file
// return: 0->success; 1->error
int SaveNodesFile(char *strNodesFile)
{
	char strTempFile[256];
	char strBuffer[520];
	FILE *fp=0;
	int i;
	unsigned char *pcIP=0;
	
	// generate temp file name
	strcpy(strTempFile,strNodesFile);
	strcat(strTempFile,".tmp");
	
	// write nodes to temp file
	if((fp=fopen(strTempFile,"w"))==NULL)
		return 1;
	
	for(i=0;i<MAX_DHT_NODE;i++)
	{
		if(g_aDhtNode[i].ip == 0)
			continue; // empty entry
		
		pcIP=(unsigned char *) &(g_aDhtNode[i].ip);
		fprintf(fp,"%d.%d.%d.%d:%d\n",pcIP[0],pcIP[1],pcIP[2],pcIP[3],ntohs(g_aDhtNode[i].port));
	}
	
	fclose(fp);
	
	// copy temp file to nodes file
	strcpy(strBuffer,"mv '");
	strcat(strBuffer,strTempFile);
	strcat(strBuffer,"' '");
	strcat(strBuffer,strNodesFile);
	strcat(strBuffer,"'");
#ifdef DEBUG
	printf("(%s) execute: %s\n",THIS_APP,strBuffer);
#endif
	system(strBuffer);
	
	return 0;
}

// Add a node
// This function will do ip format check
// Input:
//         strIP: ip of node
//         port: port of node in host endian
//         time: time of response from node
// return: >0: success and the value is the index in the node list
//         -1: invalid ip, -2: node list full
int AddDhtNode(char *strIP, unsigned short port, time_t time)
{
	unsigned long nIP;
	unsigned char *pIP;
	unsigned short nPort=0;
	int i,nFirstEmpty=-1;
	char bFound=0;
	time_t oldest_time=0;
	int nOldest=-1;
	int nInsert=-1;
	
	if(inet_pton(AF_INET, strIP, &nIP)<=0)
	{ // not a valid IP address
#ifdef DEBUG
		printf("(%s) Error: \"%s\" is not a valid IP address\n",THIS_APP,strIP);
#endif
		return -1;
	}
	
	nPort=htons(port);
	
#ifdef DEBUG
	printf("(%s) Debug: Try to add a node: IP=\"%s\" (%X), Port=\"%d\"\n",THIS_APP,strIP,nIP,port);
#endif

	pIP=(unsigned char *)&nIP;
	
	// find an empty node, and check if the node has already existed
	for(i=0;i<MAX_DHT_NODE;i++)
	{
		if((g_aDhtNode[i].ip==0) && (nFirstEmpty == -1))
			nFirstEmpty=i; // find the first empty node entry
		if((g_aDhtNode[i].ip==nIP) && (g_aDhtNode[i].port==nPort))
		{ // the node already in the node list
			if(g_aDhtNode[i].time < time)
				g_aDhtNode[i].time=time; // update the last active time of this node
			bFound=1;
			break;
		}
		if(g_aDhtNode[i].time<oldest_time)
		{ // find the oldest node
			oldest_time=g_aDhtNode[i].time;
			nOldest=i;
		}
	}
	if(bFound==1)
	{
#ifdef DEBUG
		printf("(%s) Warning: Node \"%s:%d\" has already been in the node list\n",THIS_APP,strIP,port);
#endif
		return i;
	}
	
	// the node is not in the node list
	// add the node into the node list if the list is not full
	if((nFirstEmpty == -1) && (nOldest == -1))
	{ // the node list is full
		
#ifdef DEBUG
		printf("(%s) warning: Node list is full and \"%s:%d\" is not added.\n",THIS_APP,strIP,ntohs(nPort));
#endif
		return -2;
	}
	else if(nFirstEmpty != -1)
		nInsert=nFirstEmpty;
	else
		nInsert=nOldest;
	
	// add the node into the node list
	g_aDhtNode[nInsert].ip=nIP;
	g_aDhtNode[nInsert].port=nPort;
	g_aDhtNode[nInsert].time=time;
	

#ifdef SHOW_MSG
	printf("(%s) Add a new node %d.%d.%d.%d:%d at position (%d) in the node list\n",THIS_APP,pIP[0],pIP[1],pIP[2],pIP[3],ntohs(nPort),nInsert);
#endif

	return nInsert;
}

// The Node Server has several functions:
// 1. maintains good DHT nodes by periodically send ping request to nodes and check if they are active
// 2. reply the nodes requests (get_peers) from local GetDhtPeers clients
// 3. save good nodes to file (strNodesFile)
// Return: 0:success, 1:fail
int RunNodeServer(char *strNodesFile,unsigned short nPort,time_t nTimeout)
{
	int	udpfd, maxfdp1, outfd=0;
	int nReady;
	char mesg[MAX_BUFFER]; // buffer to keep packet data
	char request[128]; // request buffer
	char response[MAX_BUFFER];
	char ping[64];
	int nNodesCount=0;
	char strTemp[256];
	fd_set	rset;
	ssize_t	nPacketLen; // packet length
	socklen_t	len;
	struct sockaddr_in nodeAddress, serverAddress;
	//unsigned int nRemoteAddress;
	unsigned short int nRemotePort,nServerPort;
	unsigned long nIP;
	unsigned char *pnAddress=0,*pcRequest=0,*pcResponse=0,*pcTemp=0;
	struct timeval idleTimeout;
	int i,nRequestLen,nResponseLen,nPingLen,nParentID,nPid;
	//unsigned char macAddress[6]={0x00,0x00,0xE2,0x64,0xCD,0x9E};
	int nPacketType=0;
	time_t nLastSave=0,nNow=0,nLastPing=0;
	
	// prepare get_peers response packet (nodes)
	PrepareNodesResponse(response,&nResponseLen);
	
	// prepare ping request packet
	PreparePingRequest(ping,&nPingLen);

	// create UDP server socket
	udpfd = socket(AF_INET, SOCK_DGRAM, 0);

	// prepare server address (only service local host)
	if(inet_pton(AF_INET,"127.0.0.1",&nIP)<=0)
	{ // not a valid IP address
#ifdef DEBUG
		printf("(%s) Error: fail to bind \"127.0.0.1\"\n",THIS_APP);
#endif
		return 1;
	}
	bzero(&serverAddress, sizeof(serverAddress));
	serverAddress.sin_family = AF_INET;
	serverAddress.sin_addr.s_addr = nIP;
	//serverAddress.sin_addr.s_addr = htonl(INADDR_ANY); // server listen to all interface
	serverAddress.sin_port = htons(nPort);
	
	// prepare node address
	bzero(&nodeAddress, sizeof(nodeAddress));
	nodeAddress.sin_family = AF_INET;
	//nodeAddress.sin_port = htons(SERV_PORT);
	//inet_pton(AF_INET, argv[1], &nodeAddress.sin_addr);

	if(bind(udpfd, (SA *) &serverAddress, sizeof(serverAddress)) != 0) // try to bind a port
	{ // fail to bind this port
		if(errno == EADDRINUSE)
			printf("(%s) Error: port %d has already been used\n",THIS_APP,nPort);
		else
			printf("(%s) Error: Socket bind error number (%d)\n",THIS_APP,errno);
		
		return 1;
	}
	
	printf("(%s) Started on port: %d\n",THIS_APP,nPort);

	FD_ZERO(&rset);
	maxfdp1 = udpfd + 1;
	
	nLastSave=time(NULL); // set last save nodes time to current
	
	// start to get service client
	//for ( ; ; )
	while(1)
	{
		// get current time
		nNow=time(NULL);
		
		// save nodes to file
		if(nNow > (nLastSave+SAVE_INTERVAL))
		{ // exceed save nodes interval time
			nLastSave=nNow;
			SaveNodesFile(strNodesFile);
		}
		
		// send ping requests to check good nodes
		if(nNow > (nLastPing+PING_NODE_INTERVAL))
		{ // exceed save nodes interval time
			nLastPing=nNow;
			// send ping request to all nodes
			//printf("(%s) Hint: you have not implemented ping nodes\n",THIS_APP);
			for(i=0;i<MAX_DHT_NODE;i++)
			{
				if(g_aDhtNode[i].ip != 0)
				{
					// send request to this node
					nodeAddress.sin_addr.s_addr=g_aDhtNode[i].ip;
					nodeAddress.sin_port=g_aDhtNode[i].port;
					len = sizeof(nodeAddress);
					sendto(udpfd, ping, nPingLen, 0, (SA *) &nodeAddress, len);
#ifdef SHOW_MSG
					pnAddress= (unsigned char *) &(g_aDhtNode[i].ip);
					printf("(%s) Send ping request to node %d.%d.%d.%d:%d\n",THIS_APP,*pnAddress,*(pnAddress+1),*(pnAddress+2),*(pnAddress+3),ntohs(g_aDhtNode[i].port));
#endif
				} //if(g_aDhtNode[i].ip != 0)
			} //for(i=0;i<MAX_DHT_NODE;i++)
		} //if(nNow > (nLastPing+PING_NODE_INTERVAL))
		
		// setup read fd of select
		FD_SET(udpfd, &rset);
		
		// setup select idle timeout
		idleTimeout.tv_sec=nTimeout;
		idleTimeout.tv_usec=0;
		
		if ( (nReady = select(maxfdp1, &rset, NULL, NULL, &idleTimeout)) < 0)
		{ // may be an error occurs
			printf("(%s) Get signal\n",THIS_APP);
			if (errno == EINTR)
				continue;
			else
			{ // select error
				printf("(%s) Select error\n",THIS_APP);
				exit(1);
			}
		}
		
		if(nReady==0)
		{ // idle timeout: select return due to timeout
#ifdef DEBUG
			printf("(%s) Debug: idle timeout %d seconds\n",THIS_APP,nTimeout);
#endif
			continue;
		}
		
		if (FD_ISSET(udpfd, &rset))
		{ // a UDP packet arrived
			
			// receive packet
			len = sizeof(nodeAddress);
			nPacketLen = recvfrom(udpfd, mesg, MAX_BUFFER, 0, (SA *) &nodeAddress, &len);
			
			pnAddress=(unsigned char *) &nodeAddress.sin_addr.s_addr;
			nRemotePort = nodeAddress.sin_port;
#ifdef SHOW_MSG
//			printf("(%s) Receive packet(%d) from %d.%d.%d.%d:%d\n",THIS_APP,nPacketLen,*pnAddress,*(pnAddress+1),*(pnAddress+2),*(pnAddress+3),ntohs(nRemotePort));
#endif

			nPacketType=ProcessPacket(mesg,nPacketLen);
#ifdef SHOW_MSG
			printf("(%s) Receive packet(%d) type=%d from %d.%d.%d.%d:%d\n",THIS_APP,nPacketLen,nPacketType,*pnAddress,*(pnAddress+1),*(pnAddress+2),*(pnAddress+3),ntohs(nRemotePort));
#endif
			
			if(nPacketType == BT_PACKET_GET_PEERS_REQUEST)
			{ // get get_peers request, send response
				sendto(udpfd, response,nResponseLen, 0, (SA *) &nodeAddress, len);
			}
			else if(nPacketType == BT_PACKET_PING_RESPONSE)
			{ // update time of node
				sprintf(strTemp,"%d.%d.%d.%d",*pnAddress,*(pnAddress+1),*(pnAddress+2),*(pnAddress+3));
#ifdef DEBUG
				printf("(%s) Debug: receive ping response from %s:%d\n",strTemp,ntohs(nRemotePort));
#endif
				AddDhtNode(strTemp,ntohs(nRemotePort),time(NULL));
			}
			else if(nPacketType == BT_PACKET_ADD_NODE)
			{ // good nodes may be changed, re-generate get_peers response packet
				PrepareNodesResponse(response,&nResponseLen);
			}
			else
			{
#ifdef DEBUG
				printf("(%s) Debug: receive unknown packet from %s:%d\n",strTemp,ntohs(nRemotePort));
#endif
			}
			
		} //if (FD_ISSET(udpfd, &rset))
	} //while(1)

	return 0;
}

// generate get_peers request response packet (reply nodes)
// return: 0: no node, 1: has nodes
int PrepareNodesResponse(char *buffer, int *nPacketLen)
{
	int nNodesCount=0;
	char strTemp[256];
	unsigned char *pcRequest=0,*pcBuffer=0,*pcTemp=0;
	int i,j,nBufferLen,nParentID,nPid;
	unsigned char macAddress[6]={0x00,0x00,0xE2,0x64,0xCD,0x9E};
	
	// count nodes
	for(i=0;i<MAX_DHT_NODE;i++)
		if(g_aDhtNode[i].ip != 0)
			nNodesCount++;
	
	if(nNodesCount==0)
	{ // no node
		*nPacketLen=0;
		return 0;
	}
		
	// prepare get_peers response nodes packet
	strcpy(buffer,"d1:rd2:id20:");
	nBufferLen=strlen("d1:ad2:id20:");
	pcBuffer=(unsigned char *) (buffer+nBufferLen);
	// add node id (user MAC address as node id)
	if(GetMacAddress(macAddress) == 1)
		printf("(%s) Error: failed to get MAC address, use default value\n",THIS_APP);
	strcpy(buffer+nBufferLen,NODE_ID_PREFIX);
	for(i=strlen(NODE_ID_PREFIX);i<14;i++)
		pcBuffer[i]=0;
	//for(i=0;i<14;i++)
	//	pcBuffer[i]=0;
	pcBuffer+=14;
	for(i=0;i<6;i++)
		pcBuffer[i]=macAddress[i];
	pcBuffer+=6;
	nBufferLen+=20;
	// add nodes
	if(nNodesCount>0)
	{
		strcpy(buffer+nBufferLen,"5:nodes");
		nBufferLen+=strlen("5:nodes");
		// add nodes number
		sprintf(strTemp,"%d:",nNodesCount*26);
		strcpy(buffer+nBufferLen,strTemp);
		nBufferLen+=strlen(strTemp);
		// add nodes content
		for(i=0;i<nNodesCount;i++)
		{
			pcBuffer=(unsigned char *) (buffer+nBufferLen);
			for(j=0;j<20;j++)
				pcBuffer[j]=0; // add node id
			nBufferLen+=20;
			// add ip
			pcBuffer=(unsigned char *) (buffer+nBufferLen);
			pcTemp=(unsigned char *) &(g_aDhtNode[i].ip);
			for(j=0;j<4;j++)
				pcBuffer[j]=pcTemp[j];
			nBufferLen+=4;
			// add port
			pcBuffer=(unsigned char *) (buffer+nBufferLen);
			pcTemp=(unsigned char *) &(g_aDhtNode[i].port);
			for(j=0;j<2;j++)
				pcBuffer[j]=pcTemp[j];
			nBufferLen+=2;
		} //for(i=0;i<nNodesCount;i++)
	} //if(nNodesCount>0)
	// add token
	strcpy(buffer+nBufferLen,"5:token8:");
	nBufferLen+=strlen("5:token8:");
	// token (use parent process id + current process id)
	// add parent process id
	nParentID=(int)getppid();
	pcTemp=(unsigned char *) &nParentID;
	pcBuffer=(unsigned char *) (buffer+nBufferLen);
	for(i=0;i<4;i++)
		pcBuffer[i]=pcTemp[i];
	nBufferLen+=4;
	// add process id
	nPid=(int)getpid();
	pcTemp=(unsigned char *) &nPid;
	pcBuffer=(unsigned char *) (buffer+nBufferLen);
	for(i=0;i<4;i++)
		pcBuffer[i]=pcTemp[i];
	nBufferLen+=4;
	// buffer command
	strcpy(buffer+nBufferLen,"e1:t4:");
	nBufferLen+=strlen("e1:t4:");
	// transaction id (use current process id as transaction id)
	nPid=(int)getpid();
	pcTemp=(unsigned char *) &nPid;
	pcBuffer=(unsigned char *) (buffer+nBufferLen);
	for(i=0;i<4;i++)
		pcBuffer[i]=pcTemp[i];
	nBufferLen+=4;
	// buffer
	strcpy(buffer+nBufferLen,"1:y1:re");
	nBufferLen+=strlen("1:y1:re");
	// Finished preparing response pakcet
	
#ifdef DEBUG
/*
	int outfd=0;
	if((outfd = open("response_packet", O_WRONLY|O_CREAT,0644))!=-1)
	{
		write(outfd,buffer,nBufferLen);
		close(outfd);
	}
	else
		printf("(%s) Error: failed to open file \"response_packet\"\n",THIS_APP);
*/
#endif

	*nPacketLen=nBufferLen;
	return 1;
}

// generate ping request DHT packet (to check good nodes)
void PreparePingRequest(char *buffer, int *nPacketLen)
{
	char strTemp[256];
	unsigned char *pcRequest=0,*pcBuffer=0,*pcTemp=0;
	int i,j,nBufferLen,nParentID,nPid;
	unsigned char macAddress[6]={0x00,0x00,0xE2,0x64,0xCD,0x9E};
	
	// prepare get_peers response nodes packet
	strcpy(buffer,"d1:ad2:id20:");
	nBufferLen=strlen("d1:ad2:id20:");
	pcBuffer=(unsigned char *) (buffer+nBufferLen);
	// add node id (user MAC address as node id)
	if(GetMacAddress(macAddress) == 1)
		printf("(%s) Error: failed to get MAC address, use default value\n",THIS_APP);
	strcpy(buffer+nBufferLen,NODE_ID_PREFIX);
	for(i=strlen(NODE_ID_PREFIX);i<14;i++)
		pcBuffer[i]=0;
	//for(i=0;i<14;i++)
	//	pcBuffer[i]=0;
	pcBuffer+=14;
	for(i=0;i<6;i++)
		pcBuffer[i]=macAddress[i];
	pcBuffer+=6;
	nBufferLen+=20;
	// buffer command
	strcpy(buffer+nBufferLen,"e1:q4:ping1:t4:");
	nBufferLen+=strlen("e1:q4:ping1:t4:");
	// transaction id (use current process id as transaction id)
	nPid=(int)getpid();
	pcTemp=(unsigned char *) &nPid;
	pcBuffer=(unsigned char *) (buffer+nBufferLen);
	for(i=0;i<4;i++)
		pcBuffer[i]=pcTemp[i];
	nBufferLen+=4;
	// buffer
	strcpy(buffer+nBufferLen,"1:y1:qe");
	nBufferLen+=strlen("1:y1:qe");
	// Finished preparing response pakcet
	
#ifdef DEBUG
/*
	int outfd=0;
	if((outfd = open("ping_packet", O_WRONLY|O_CREAT,0644))!=-1)
	{
		write(outfd,buffer,nBufferLen);
		close(outfd);
	}
	else
		printf("(%s) Error: failed to open file \"ping_packet\"\n",THIS_APP);
*/
#endif

	*nPacketLen=nBufferLen;
}

// get the MAC address of this device
// return: 0->success, 1->fail
int GetMacAddress(unsigned char *macAddress)
{
	printf("(%s) Hint: you have not implement function GetMacAddress()\n",THIS_APP);
	return 1;
}

// process received UDP packet
// Return: packet type
int ProcessPacket(char *buffer,int bufferLen)
{
	char strIP[16];
	unsigned char *pcIP;
	unsigned short nPort,*pnPort;
	
	//char buffer[MAX_BUFFER]; // buffer to keep data
	int nBufferPtr=0;
	//int	nLen,fd=0;
	char token[BT_MAX_TOKEN+10];
	int nTokenPtr=0,nTokenPtr2=0,nTokenLen=0,nTokenLeft=0,nFilePos=0;
	int nDepth=0;
	char caType[BT_MAX_DEPTH];
	char caSubType[BT_MAX_DEPTH];
	char c,*pStr=0;
	//char bReadNode=0,bReadValue=0;
	char bBigToken=0,bReadType=0,bReadRequest=0;
	char cPacketType=0,bGetPeersRequest=0;
	
	if(buffer[0]=='-')
	{ // Add node
		if(bufferLen!=7)
			return BT_PACKET_NULL; // must be invalid format
		
		pcIP=(unsigned char *) (buffer+1);
		sprintf(strIP,"%d.%d.%d.%d",pcIP[0],pcIP[1],pcIP[2],pcIP[3]);
		pnPort=(unsigned short *) (buffer+5);
		nPort=*pnPort;
		AddDhtNode(strIP,ntohs(nPort),time(NULL));
		return BT_PACKET_ADD_NODE;
	}
	
	nBufferPtr=0;
		
			while(nBufferPtr<bufferLen)
			{
				// read next byte
				c=buffer[nBufferPtr];
				if((bBigToken==0)||(nTokenPtr==0)) // prevent big token makes buffer over run
					token[nTokenPtr]=c;
#ifdef DEBUG_RESPONSE
/*				token[nTokenPtr+1]='\0';
				printf("Debug(%d): %s\n",nFilePos,token);*/
#endif
				
				if(nTokenPtr==0)
				{ // first char of token
					bBigToken=0;
					if(c=='d')
					{ // directory
#ifdef DEBUG_RESPONSE
						printf("Debug(%d): Start of directory, Depth(%d)\n",nFilePos,nDepth);
#endif
						caType[nDepth]=BT_TYPE_DIR;
						caSubType[nDepth]=BT_TYPE_NULL;
						nDepth++;
						if(nDepth>BT_MAX_DEPTH)
						{
							printf("(%s) Depth of response is too deep\n",THIS_APP);
							return 1;
						}
						caType[nDepth]=BT_TYPE_NULL;
						caSubType[nDepth]=BT_TYPE_NULL;
						nTokenPtr=0;
					}
					else if(c=='l')
					{ // list
#ifdef DEBUG_RESPONSE
						printf("Debug(%d): Start of list, Depth(%d)\n",nFilePos,nDepth);
#endif
						caType[nDepth]=BT_TYPE_LIST;
						caSubType[nDepth]=BT_TYPE_NULL;
						nDepth++;
						if(nDepth>BT_MAX_DEPTH)
						{
							printf("(%s) Depth of response is too deep\n",THIS_APP);
							return 1;
						}
						caType[nDepth]=BT_TYPE_NULL;
						caSubType[nDepth]=BT_TYPE_NULL;
						nTokenPtr=0;
						
					} //else if(c=='l')
					else if(c=='i')
					{ // int
#ifdef DEBUG_RESPONSE
						printf("Debug(%d): Start of int, Depth(%d)\n",nFilePos,nDepth);
#endif
						if(caType[nDepth-1]==BT_TYPE_DIR)
						{ // start of a directory key
							caType[nDepth]=BT_TYPE_KEY;
							caSubType[nDepth]=BT_TYPE_INT;
						}
						else if((caType[nDepth-1]==BT_TYPE_LIST) || (caType[nDepth-1]==BT_TYPE_KEY))
						{ // start of a list element or start of a directory value
							caType[nDepth]=BT_TYPE_VALUE;
							caSubType[nDepth]=BT_TYPE_INT;
						}
						nTokenPtr++;
					}
					else if((c>='0') && (c<='9'))
					{ // string
#ifdef DEBUG_RESPONSE
						printf("Debug(%d): Start of string, Depth(%d)\n",nFilePos,nDepth);
#endif
						if(caType[nDepth-1]==BT_TYPE_DIR)
							caType[nDepth]=BT_TYPE_KEY; // start of a directory key
						else if((caType[nDepth-1]==BT_TYPE_LIST) || (caType[nDepth-1]==BT_TYPE_KEY))
							caType[nDepth]=BT_TYPE_VALUE; // start of a list value or start of a directory value
						
						caSubType[nDepth]=BT_TYPE_STR1;
						nTokenPtr++;
					} //else if((c>='0') && (c<='9'))
					else if(c=='e')
					{ // end of dictionary or list
						//decrease level
						nDepth--;
#ifdef DEBUG_RESPONSE
						printf("Debug(%d): End of dictionary or list, Depth(%d)\n",nFilePos,nDepth);
#endif
						if(caType[nDepth-1]==BT_TYPE_KEY)
						{ // should decrease further level
							nDepth--;
#ifdef DEBUG_RESPONSE
							printf("Debug(%d): Meet end of dictionary value when end of dictionary or list, Depth(%d)\n",nFilePos,nDepth);
#endif
						}
						
						nTokenPtr=0;
					} //else if(c=='e')
				} //if(nTokenPtr==0)
				else
				{ // nTokenPtr>0
					if((c==':') && (caSubType[nDepth]==BT_TYPE_STR1))
					{ // get complete length of string
						caSubType[nDepth]=BT_TYPE_STR2;
						nTokenPtr2=nTokenPtr+1; // start of token content
						token[nTokenPtr]='\0';
						nTokenLen=atoi(token); // token length
						nTokenLeft=nTokenLen; // left char to read in token
#ifdef DEBUG_RESPONSE
//						printf("Debug(%d): string length (%d), Depth(%d) BT_TYPE_STR2 nTokenPtr2=%d\n",nFilePos,nTokenLen,nDepth,nTokenPtr2);
#endif
						if(nTokenLen>BT_MAX_TOKEN)
						{ // token too long
							bBigToken=1;
							//printf("Error(%d): string token is too long\n",nFilePos);
							//return 1;
						}
						nTokenPtr++;
					} //if((c==':') && (caSubType[nDepth]==BT_TYPE_STR1))
					else if((c=='e') && (caSubType[nDepth]==BT_TYPE_INT))
					{ // get complete int
						token[nTokenPtr]='\0';
						pStr=token+1;
						
#ifdef DEBUG_RESPONSE
						printf("Debug(%d): Read int \"%s\"\n",nFilePos,pStr);
#endif
						if(caType[nDepth]==BT_TYPE_KEY)
						{ // end of a key, next is value -> increase level
							nDepth++;
							if(nDepth>BT_MAX_DEPTH)
							{
								printf("(%s) Depth of response is too deep\n",THIS_APP);
								return 1;
							}
							caType[nDepth]=BT_TYPE_NULL;
							caSubType[nDepth]=BT_TYPE_NULL;
						} //if(caType[nDepth]==BT_TYPE_KEY)
						else if((caType[nDepth]==BT_TYPE_VALUE) && (caType[nDepth-1]==BT_TYPE_KEY))
						{ // end of a key value and decrease level
							nDepth--;
							caType[nDepth]=BT_TYPE_NULL;
							caSubType[nDepth]=BT_TYPE_NULL;
						} // else if((caType[nDepth]==BT_TYPE_VALUE) && (caType[nDepth-1]==BT_TYPE_KEY))
						else if((caType[nDepth]==BT_TYPE_VALUE) && (caType[nDepth-1]==BT_TYPE_LIST))
						{ // end of a list element
							caType[nDepth]=BT_TYPE_NULL;
							caSubType[nDepth]=BT_TYPE_NULL;
						}
								
						nTokenPtr=nTokenPtr2=nTokenLeft=nTokenLen=0; // start a new token
					} //else if((c=='e') && (caSubType[nDepth]==BT_TYPE_INT))
					else if((caSubType[nDepth]==BT_TYPE_STR2) && (nTokenLeft==0))
					{ // read a complete string token
						if(bBigToken==0)
						{ // prevent big token from making buffer over run
							token[nTokenPtr2+nTokenLen]='\0';
							pStr=token+nTokenPtr2;
#ifdef DEBUG_RESPONSE
							printf("Debug(%d): Read token \"%s\" (%d,%d), Depth=(%d)\n",nFilePos,pStr,nTokenPtr2,nTokenLen,nDepth);
#endif
							if(nDepth==1)
							{ // 1st level token
								if(strcmp(pStr,"y")==0)
								{ // bingo, reach packet type
#ifdef DEBUG_RESPONSE
									printf("Debug(%d): Read start of packet type\n======================================\n",nFilePos);
#endif
									bReadType=1;
								}
								else if(strcmp(pStr,"q")==0)
								{ // bingo, reach request type
#ifdef DEBUG_RESPONSE
									printf("Debug(%d): Read start of request type\n======================================\n",nFilePos);
#endif
									bReadRequest=1;
								}
							} //if(nDepth==1)
							else if((bReadType==1) && (nDepth==2))
							{ // get packet type
								if(nTokenLen!=1)
								{ // invalid packet type
									printf("Debug(%d): Invalid packet type length (%d)\n",nFilePos,nTokenLen);
								}
								else
								{ // pStr is a packet type
									if(strcmp(pStr,"r")==0)
									{ // this is a response
										cPacketType='r';
									}
									else if(strcmp(pStr,"q")==0)
									{ // this is a request
										cPacketType='q';
									}
									else
										printf("Debug(%d): Invalid packet type '%s'\n",nFilePos,pStr);
								}
								bReadType=0;
							} //else if((bReadType==1) && (nDepth==2))
							else if((bReadRequest==1) && (nDepth==2))
							{ // get request type
								if(strcmp(pStr,"get_peers")==0)
								{ // This is a DHT get_peers request packet
									bGetPeersRequest=1;
								}
								bReadRequest=0;
#ifdef DEBUG_RESPONSE
								printf("Debug(%d): Finished read request, Depth(%d)\n",nFilePos,nDepth);
#endif
							} //else if((bReadRequest==1) && (nDepth==2))
						} //if(bBigToken==0)
								
						if(caType[nDepth]==BT_TYPE_KEY)
						{ // end of a key, next is value -> increase level
							nDepth++;
							if(nDepth>BT_MAX_DEPTH)
							{
								printf("(%s) Depth of response is too deep\n",THIS_APP);
								return 1;
							}
							caType[nDepth]=BT_TYPE_NULL;
							caSubType[nDepth]=BT_TYPE_NULL;
						} //if(caType[nDepth]==BT_TYPE_KEY)
						else if((caType[nDepth]==BT_TYPE_VALUE) && (caType[nDepth-1]==BT_TYPE_KEY))
						{ // end of a key value and decrease level
							nDepth--;
							caType[nDepth]=BT_TYPE_NULL;
							caSubType[nDepth]=BT_TYPE_NULL;
						} // else if((caType[nDepth]==BT_TYPE_VALUE) && (caType[nDepth-1]==BT_TYPE_KEY))
						else if((caType[nDepth]==BT_TYPE_VALUE) && (caType[nDepth-1]==BT_TYPE_LIST))
						{ // end of a list element
							caType[nDepth]=BT_TYPE_NULL;
							caSubType[nDepth]=BT_TYPE_NULL;
						}
								
						nTokenPtr=nTokenPtr2=nTokenLeft=nTokenLen=0; // start a new token
						
					} //else if((caSubType[nDepth]==BT_TYPE_STR2) && (nTokenLeft==1))
					else
					{
						nTokenPtr++;
					}
					
					if(caSubType[nDepth]==BT_TYPE_STR2)
					{
#ifdef DEBUG_RESPONSE
/*						token[nTokenPtr]='\0';
						pStr=token+nTokenPtr2;
						printf("Debug(%d)(%d): %s\n",nFilePos,nTokenLeft,pStr);*/
#endif
						nTokenLeft--;
					}
						
				} //if(nTokenPtr==0) else
				nBufferPtr++; // buffer position
				nFilePos++; // file position
			} //while(nBufferPtr<bufferLen)
	// Finished parsing packet
	
	if((cPacketType=='q') && (bGetPeersRequest==1))
		return BT_PACKET_GET_PEERS_REQUEST;
	else if(cPacketType=='r')
		return BT_PACKET_PING_RESPONSE;
	
	return BT_PACKET_NULL;
}


