/*
 *      Operation routines for FLASH MIB access
 *
 *      Authors: David Hsu	<davidhsu@realtek.com.tw>
 *
 *      $Id: flash.c,v 1.43 2008/07/18 05:39:31 bradhuang Exp $
 *
 */


/* System include files */
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <stdlib.h>
#include <ctype.h>

#define noPARSE_TXT_FILE

#define WLAN_FAST_INIT

/* Local include files */
#include "apmib.h"
#include "mibtbl.h"

/* Constand definitions */
#define DEC_FORMAT	("%d")
#define BYTE5_FORMAT	("%02x%02x%02x%02x%02x")
#define BYTE6_FORMAT	("%02x%02x%02x%02x%02x%02x")
#define BYTE13_FORMAT	("%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x")
#define STR_FORMAT	("%s")
#define HEX_FORMAT	("%02x")

#ifdef HOME_GATEWAY
#define PORTFW_FORMAT	("%s, %d, %d, %d")
#define PORTFILTER_FORMAT ("%d, %d, %d")
#define IPFILTER_FORMAT	("%s, %d")
#define TRIGGERPORT_FORMAT ("%d, %d, %d, %d, %d, %d")
#endif
#define MACFILTER_FORMAT ("%02x%02x%02x%02x%02x%02x")
#define MACFILTER_COLON_FORMAT ("%02x:%02x:%02x:%02x:%02x:%02x")
#define WDS_FORMAT	MACFILTER_FORMAT

#ifdef HOME_GATEWAY
#ifdef VPN_SUPPORT
//#define IPSECTUNNEL_FORMAT ("%d, %d, %s, %d, %s, %d, %d, %s , %d, %s, %d, %d, %d, %d, %d, %d, %s, %d, %d, %d, %lu, %lu, %d, %s, %s, %s")
#define IPSECTUNNEL_FORMAT ("%d, %d, %s, %d, %s, %d, %d, %s , %d, %s, %d, %d,  %d, %d,  %s, %d, %d, %d, %lu, %lu, %d, %s, %s, %s, %d, %s, %s, %d, %d, %s")
#endif
#endif

#define SPACE	(' ')
#define EOL	('\n')

static int config_area;

/* Macro definition */
static int _is_hex(char c)
{
    return (((c >= '0') && (c <= '9')) ||
            ((c >= 'A') && (c <= 'F')) ||
            ((c >= 'a') && (c <= 'f')));
}

static int string_to_hex(char *string, unsigned char *key, int len)
{
	char tmpBuf[4];
	int idx, ii=0;
	for (idx=0; idx<len; idx+=2) {
		tmpBuf[0] = string[idx];
		tmpBuf[1] = string[idx+1];
		tmpBuf[2] = 0;
		if ( !_is_hex(tmpBuf[0]) || !_is_hex(tmpBuf[1]))
			return 0;

		key[ii++] = (unsigned char) strtol(tmpBuf, (char**)NULL, 16);
	}
	return 1;
}

static void convert_lower(char *str)
{	int i;
	int len = strlen(str);
	for (i=0; i<len; i++)
		str[i] = tolower(str[i]);
}

static int APMIB_GET(int id, void *val)
{
	if (config_area == 3 || config_area == 4)
		return apmib_getDef(id, val);
	else
		return apmib_get(id, val);
}

static int APMIB_SET(int id, void *val)
{
	if (config_area == 3 || config_area == 4)
		return apmib_setDef(id, val);
	else
		return apmib_set(id, val);
}

/* Local declarations routines */
static int flash_read(char *buf, int offset, int len);
static int writeDefault(int isAll);
static int searchMIB(char *token);
static void getMIB(char *name, int id, TYPE_T type, int num, int array_separate, char **val);
static void setMIB(char *name, int id, TYPE_T type, int len, int valnum, char **val);
static void dumpAll(void);
static void showHelp(void);
static void showAllMibName(void);
static void showSetACHelp(void);
static void showSetWdsHelp(void);

#ifndef COMPACK_SIZE
static int read_flash_webpage(char *prefix, char *file);
#endif
#ifdef TLS_CLIENT
static int read_flash_cert(char *prefix, char *certfile);
#endif
#ifdef VPN_SUPPORT
static int read_flash_rsa(char *prefix);
#endif
#ifdef HOME_GATEWAY
static void showSetPortFwHelp(void);
static void showSetPortFilterHelp(void);
static void showSetIpFilterHelp(void);
static void showSetMacFilterHelp(void);
static void showSetUrlFilterHelp(void);
static void showSetTriggerPortHelp(void);
#ifdef ROUTE_SUPPORT
static void showSetStaticRouteHelp(void);
#endif

#ifdef VPN_SUPPORT
static void showSetIpsecTunnelHelp(void);
#endif
static int generatePPPConf(int is_pppoe, char *option_file, char *pap_file, char *chap_file);
#endif

#ifdef TLS_CLIENT
static void showSetCertRootHelp(void);
static void showSetCertUserHelp(void);
#endif

static void generateWpaConf(char *outputFile, int isWds);

#ifdef WLAN_FAST_INIT
static int initWlan(char *ifname);
#endif

#ifdef WIFI_SIMPLE_CONFIG
static int updateWscConf(char *in, char *out, int genpin);
#endif

#ifdef PARSE_TXT_FILE
static int parseTxtConfig(char *filename, APMIB_Tp pConfig);
static int getToken(char *line, char *value);
static int set_mib(APMIB_Tp pMib, int id, void *value);
static void getVal2(char *value, char **p1, char **p2);
#ifdef HOME_GATEWAY
static void getVal3(char *value, char **p1, char **p2, char **p3);
static void getVal4(char *value, char **p1, char **p2, char **p3, char **p4);
static void getVal5(char *value, char **p1, char **p2, char **p3, char **p4, char **p5);
#endif

static int acNum;
static int wdsNum;

#ifdef HOME_GATEWAY
static int macFilterNum, portFilterNum, ipFilterNum, portFwNum, triggerPortNum, staticRouteNum;
static int urlFilterNum;
#endif

#ifdef TLS_CLIENT
static int certRootNum, certUserNum ;
#endif

static is_wlan_mib=0;

/////////////////////////////////////////////////////////////////////////////////////////
static char __inline__ *getVal(char *value, char **p)
{
	int len=0;

	while (*value == ' ' ) value++;

	*p = value;

	while (*value && *value!=',') {
		value++;
		len++;
	}

	if ( !len ) {
		*p = NULL;
		return NULL;
	}

	if ( *value == 0)
		return NULL;

	*value = 0;
	value++;

	return value;
}
#endif  // PARSE_TXT_FILE

//////////////////////////////////////////////////////////////////////
static char *get_token(char *data, char *token)
{
	char *ptr=data;
	int len=0, idx=0;

	while (*ptr && *ptr != '\n' ) {
		if (*ptr == '=') {
			if (len <= 1)
				return NULL;
			memcpy(token, data, len);

			/* delete ending space */
			for (idx=len-1; idx>=0; idx--) {
				if (token[idx] !=  ' ')
					break;
			}
			token[idx+1] = '\0';
			
			return ptr+1;
		}
		len++;
		ptr++;
	}
	return NULL;
}

//////////////////////////////////////////////////////////////////////
static int get_value(char *data, char *value)
{
	char *ptr=data;	
	int len=0, idx, i;

	while (*ptr && *ptr != '\n' && *ptr != '\r') {
		len++;
		ptr++;
	}

	/* delete leading space */
	idx = 0;
	while (len-idx > 0) {
		if (data[idx] != ' ') 
			break;	
		idx++;
	}
	len -= idx;

	/* delete bracing '"' */
	if (data[idx] == '"') {
		for (i=idx+len-1; i>idx; i--) {
			if (data[i] == '"') {
				idx++;
				len = i - idx;
			}
			break;
		}
	}

	if (len > 0) {
		memcpy(value, &data[idx], len);
		value[len] = '\0';
	}
	return len;
}

//////////////////////////////////////////////////////////////////////
static void readFileSetParam(char *file)
{
	FILE *fp;
	char line[200], token[40], value[100], *ptr;
	int idx, hw_setting_found=0, ds_setting_found=0, cs_setting_found=0;
	char *arrayval[2];
	mib_table_entry_T *pTbl;

	
	if ( !apmib_init()) {
		printf("Initialize AP MIB failed!\n");
		return;
	}

	fp = fopen(file, "r");
	if (fp == NULL) {
		printf("read file [%s] failed!\n", file);
		return;
	}

	arrayval[0] = value;

	while ( fgets(line, 200, fp) ) {
		ptr = get_token(line, token);
		if (ptr == NULL)
			continue;
		if (get_value(ptr, value)==0)
			continue;

		idx = searchMIB(token);
		if ( idx == -1 ) {
			printf("invalid param [%s]!\n", token);
			return;
		}
		if (config_area == 1 || config_area == 2) {
			hw_setting_found = 1;
			if (config_area == 1)
				pTbl = hwmib_table;
			else
				pTbl = hwmib_wlan_table;
		}
		else if (config_area == 3 || config_area == 4) {
			ds_setting_found = 1;
			if (config_area == 3)
				pTbl = mib_table;
			else
				pTbl = mib_wlan_table;
		}
		else {
			cs_setting_found = 1;
			if (config_area == 5)
				pTbl = mib_table;
			else
				pTbl = mib_wlan_table;
		}
		config_area = 0;
		setMIB(token, pTbl[idx].id, pTbl[idx].type, pTbl[idx].size, 1, arrayval);
	}
	fclose(fp);

	if (hw_setting_found)
		apmib_update(HW_SETTING);
	if (ds_setting_found)
		apmib_update(DEFAULT_SETTING);
	if (cs_setting_found)
		apmib_update(CURRENT_SETTING);
}

//////////////////////////////////////////////////////////////////////
static int resetDefault()
{
	char *defMib;
	int fh;

	fh = open(FLASH_DEVICE_NAME, O_RDWR);

#if CONFIG_APMIB_SHARED_MEMORY == 1	
    apmib_sem_lock();
#endif

	if ((defMib=apmib_dsconf()) == NULL) {
		printf("Default configuration invalid!\n");
#if CONFIG_APMIB_SHARED_MEMORY == 1	
        apmib_sem_unlock();
#endif
		return -1;
	}

	memcpy( dsHeader.signature, CURRENT_SETTING_HEADER_TAG, TAG_LEN);

	lseek(fh, CURRENT_SETTING_OFFSET, SEEK_SET);

	if ( write(fh, (const void *)&dsHeader, sizeof(dsHeader))!=sizeof(dsHeader) ) {
		printf("write cs header failed!\n");
#if CONFIG_APMIB_SHARED_MEMORY == 1
	    apmib_shm_free(defMib, DSCONF_SHM_KEY);
	    apmib_sem_unlock();
#else
		free(defMib);
#endif
		return -1;
	}
	if ( write(fh, (const void *)defMib, dsHeader.len) != dsHeader.len ) {
		printf("write cs MIB failed!\n");
#if CONFIG_APMIB_SHARED_MEMORY == 1
	    apmib_shm_free(defMib, DSCONF_SHM_KEY);
	    apmib_sem_unlock();
#else
		free(defMib);
#endif
		return -1;
	}
	close(fh);
	sync();

#if CONFIG_APMIB_SHARED_MEMORY == 1
    apmib_load_csconf(); //read current settings diectly from flash 
	apmib_shm_free(defMib, DSCONF_SHM_KEY);
	apmib_sem_unlock();
#else
	free(defMib);
#endif

	return 0;
}



//////////////////////////////////////////////////////////////////////
int main(int argc, char** argv)
{
	int argNum=1, action=0, idx, num, valNum=0;
	char mib[100]={0}, valueArray[170][100], *value[170], *ptr;
#ifdef PARSE_TXT_FILE
	char filename[100]={0};
	APMIB_T apmib;
#endif
	mib_table_entry_T *pTbl;

	if ( argc > 1 ) {
#ifdef PARSE_TXT_FILE
		if ( !strcmp(argv[argNum], "-f") ) {
			if (++argNum < argc)
				sscanf(argv[argNum++], "%s", filename);
		}
#endif
		if ( !strcmp(argv[argNum], "get") ) {
			action = 1;
			if (++argNum < argc) {
				if (argc > 3 && !memcmp(argv[argNum], "wlan", 4)) {
					int idx;
					idx = atoi(&argv[argNum++][4]);
					if (idx >= NUM_WLAN_INTERFACE) {
						printf("invalid wlan interface index number!\n");
						return 0;
					}
					wlan_idx = idx;
				}
				sscanf(argv[argNum], "%s", mib);
				while (++argNum < argc) {
					sscanf(argv[argNum], "%s", valueArray[valNum]);
					value[valNum] = valueArray[valNum];
					valNum++;
				}
				value[valNum]= NULL;
			}
		}
		else if ( !strcmp(argv[argNum], "set") ) {
			action = 2;
			if (++argNum < argc) {
				if (argc > 4 && !memcmp(argv[argNum], "wlan", 4)) {
					int idx;
					idx = atoi(&argv[argNum++][4]);
					if (idx >= NUM_WLAN_INTERFACE) {
						printf("invalid wlan interface index number!\n");
						return 0;
					}
					wlan_idx = idx;
				}
				sscanf(argv[argNum], "%s", mib);
				while (++argNum < argc) {
					sscanf(argv[argNum], "%s", valueArray[valNum]);
					value[valNum] = valueArray[valNum];
					valNum++;
				}
				value[valNum]= NULL;
			}
		}
		else if ( !strcmp(argv[argNum], "all") ) {
			action = 3;
		}
		else if ( !strcmp(argv[argNum], "default") ) {
			return writeDefault(1);
		}
		else if ( !strcmp(argv[argNum], "default-sw") ) {
			return writeDefault(0);
		}
		else if ( !strcmp(argv[argNum], "reset") ) {
			return resetDefault();
		}
#ifndef COMPACK_SIZE
		else if ( !strcmp(argv[argNum], "extr") ) {
			if (++argNum < argc) {
				char prefix[20], file[20]={0};
				int ret;
				sscanf(argv[argNum], "%s", prefix);
				if (++argNum < argc)
					sscanf(argv[argNum], "%s", file);
				ret  = read_flash_webpage(prefix, file);
				if (ret == 0) // success
					unlink(file);
				return ret;
			}
			printf("Usage: %s web prefix\n", argv[0]);
			return -1;
		}
#endif
#ifdef TLS_CLIENT
		else if ( !strcmp(argv[argNum], "cert") ) {
			if (++argNum < argc) {
				char prefix[20], file[20]={0};
				int ret;
				sscanf(argv[argNum], "%s", prefix);
				if (++argNum < argc)
					sscanf(argv[argNum], "%s", file);
				ret  = read_flash_cert(prefix,file);
				//if (ret == 0) // success
				//	unlink(file);
				return ret;
			}
			printf("Usage: %s cert prefix\n", argv[0]);
			return -1;
		}
#endif
#ifdef VPN_SUPPORT
		else if ( !strcmp(argv[argNum], "rsa") ) {
			if (++argNum < argc) {
				char prefix[20], file[20]={0};
				int ret;
				sscanf(argv[argNum], "%s", prefix);
				if (++argNum < argc)
					sscanf(argv[argNum], "%s", file);
				ret  = read_flash_rsa(prefix);
				//if (ret == 0) // success
				//	unlink(file);
				return ret;
			}
			printf("Usage: %s rsa prefix\n", argv[0]);
			return -1;
		}
#endif
		else if ( !strcmp(argv[argNum], "test-hwconf") ) {
#if CONFIG_APMIB_SHARED_MEMORY == 1	
            apmib_sem_lock();
#endif
			if ((ptr=apmib_hwconf()) == NULL) {
#if CONFIG_APMIB_SHARED_MEMORY == 1	
                apmib_sem_unlock();
#endif
				return -1;
			}
#if CONFIG_APMIB_SHARED_MEMORY == 1
		    apmib_shm_free(ptr, HWCONF_SHM_KEY);
		    apmib_sem_unlock();
#else
			free(ptr);
#endif
			return 0;
		}
		else if ( !strcmp(argv[argNum], "test-dsconf") ) {
#if CONFIG_APMIB_SHARED_MEMORY == 1	
            apmib_sem_lock();
#endif
			if ((ptr=apmib_dsconf()) == NULL) {
#if CONFIG_APMIB_SHARED_MEMORY == 1	
                apmib_sem_unlock();
#endif
				return -1;
			}
#if CONFIG_APMIB_SHARED_MEMORY == 1
		    apmib_shm_free(ptr, DSCONF_SHM_KEY);
		    apmib_sem_unlock();
#else
			free(ptr);
#endif
			return 0;
		}
		else if ( !strcmp(argv[argNum], "test-csconf") ) {
#if CONFIG_APMIB_SHARED_MEMORY == 1	
            apmib_sem_lock();
#endif
			if ((ptr=apmib_csconf()) == NULL) {
#if CONFIG_APMIB_SHARED_MEMORY == 1	
                apmib_sem_unlock();
#endif
				return -1;
			}
#if CONFIG_APMIB_SHARED_MEMORY == 1
		    apmib_shm_free(ptr, CSCONF_SHM_KEY);
		    apmib_sem_unlock();
#else
			free(ptr);
#endif
			return 0;
		}
		else if ( !strcmp(argv[argNum], "wpa") ) {
			int isWds = 0;
			if ((argNum+2) < argc) {
				if (memcmp(argv[++argNum], "wlan", 4)) {
					printf("Miss wlan_interface argument!\n");
					return 0;
				}
				wlan_idx = atoi(&argv[argNum][4]);
				if (wlan_idx >= NUM_WLAN_INTERFACE) {
					printf("invalid wlan interface index number!\n");
					return 0;
				}
				if (((argc-1) > (argNum+1)) && !strcmp(argv[argNum+2], "wds")) 
					isWds = 1;
				generateWpaConf(argv[argNum+1], isWds);
				return 0;
			}
			else {
				printf("Miss arguments [wlan_interface config_filename]!\n");
				return 0;
			}
		}
		// set flash parameters by reading from file
		else if ( !strcmp(argv[argNum], "-param_file") ) {
			if ((argNum+2) < argc) {
				if (memcmp(argv[++argNum], "wlan", 4)) {
					printf("Miss wlan_interface argument!\n");
					return 0;
				}
				wlan_idx = atoi(&argv[argNum][4]);
				if (wlan_idx >= NUM_WLAN_INTERFACE) {
					printf("invalid wlan interface index number!\n");
					return 0;
				}
				readFileSetParam(argv[argNum+1]);
			}
			else
				printf("Miss arguments [wlan_interface param_file]!\n");
			return 0;
		}
#ifdef WLAN_FAST_INIT
		else if ( !strcmp(argv[argNum], "set_mib") ) {
			if ((argNum+1) < argc) {
				if (memcmp(argv[++argNum], "wlan", 4)) {
					printf("Miss wlan_interface argument!\n");
					return -1;
				}
				wlan_idx = atoi(&argv[argNum][4]);
				if (wlan_idx >= NUM_WLAN_INTERFACE) {
					printf("invalid wlan interface index number!\n");
					return -1;
				}
				return initWlan(argv[argNum]);
			}
			else
				printf("Miss arguments [wlan_interface]!\n");
			return -1;
		}
#endif
#ifdef HOME_GATEWAY
		else if ( !strcmp(argv[argNum], "gen-pppoe") ) { // generate pppoe config file
			if ((argNum+3) < argc) {
				return generatePPPConf(1, argv[argNum+1], argv[argNum+2], argv[argNum+3]);
			}
			else {
				printf("Miss arguments [option-file pap-file chap-file]!\n");
				return -1;
			}
		}
		else if ( !strcmp(argv[argNum], "gen-pptp") ) { // generate pptp config file
			if ((argNum+3) < argc) {
				return generatePPPConf(0, argv[argNum+1], argv[argNum+2], argv[argNum+3]);
			}
			else {
				printf("Miss arguments [option-file pap-file chap-file]!\n");
				return -1;
			}
		}
#endif		

#ifdef WIFI_SIMPLE_CONFIG
		else if ( !strcmp(argv[argNum], "upd-wsc-conf") ) {
			return updateWscConf(argv[argNum+1], argv[argNum+2], 0);			
		}

		else if ( !strcmp(argv[argNum], "gen-pin") ) {
			return updateWscConf(0, 0, 1);			
		}		
#endif // WIFI_SIMPLE_CONFIG
	}

	if ( action == 0) {
		showHelp();
		return -1;
	}
	if ( (action==1 && !mib[0]) ||
	     (action==2 && !mib[0]) ) {
		showAllMibName();
		return -1;
	}

	if ( action==2 && (!strcmp(mib, "MACAC_ADDR") || !strcmp(mib, "DEF_MACAC_ADDR"))) {
		if (!valNum || (strcmp(value[0], "add") && strcmp(value[0], "del") && strcmp(value[0], "delall"))) {
			showSetACHelp();
			return -1;
		}
		if ( (!strcmp(value[0], "del") && !value[1]) || (!strcmp(value[0], "add") && !value[1]) ) {
			showSetACHelp();
			return -1;
		}
	}
	if ( action==2 && (!strcmp(mib, "WDS") || !strcmp(mib, "DEF_WDS"))) {
		if (!valNum || (strcmp(value[0], "add") && strcmp(value[0], "del") && strcmp(value[0], "delall"))) {
			showSetWdsHelp();
			return -1;
		}
		if ( (!strcmp(value[0], "del") && !value[1]) || (!strcmp(value[0], "add") && !value[1]) ) {
			showSetWdsHelp();
			return -1;
		}
	}

#ifdef HOME_GATEWAY
	if ( action==2 && (!strcmp(mib, "PORTFW_TBL") || !strcmp(mib, "DEF_PORTFW_TBL"))) {
		if (!valNum || (strcmp(value[0], "add") && strcmp(value[0], "del") && strcmp(value[0], "delall"))) {
			showSetPortFwHelp();
			return -1;
		}
	}
	if ( action==2 && (!strcmp(mib, "PORTFILTER_TBL") || !strcmp(mib, "DEF_PORTFILTER_TBL")) ) {
		if (!valNum || (strcmp(value[0], "add") && strcmp(value[0], "del") && strcmp(value[0], "delall"))) {
			showSetPortFilterHelp();
			return -1;
		}
	}
	if ( action==2 && (!strcmp(mib, "IPFILTER_TBL") || !strcmp(mib, "DEF_IPFILTER_TBL"))) {
		if (!valNum || (strcmp(value[0], "add") && strcmp(value[0], "del") && strcmp(value[0], "delall"))) {
			showSetIpFilterHelp();
			return -1;
		}
	}
	if ( action==2 && (!strcmp(mib, "MACFILTER_TBL") || !strcmp(mib, "DEF_MACFILTER_TBL")) ) {
		if (!valNum || (strcmp(value[0], "add") && strcmp(value[0], "del") && strcmp(value[0], "delall"))) {
			showSetMacFilterHelp();
			return -1;
		}
	}
	if ( action==2 && (!strcmp(mib, "URLFILTER_TBL") || !strcmp(mib, "DEF_URLFILTER_TBL")) ) {
		if (!valNum || (strcmp(value[0], "add") && strcmp(value[0], "del") && strcmp(value[0], "delall"))) {
			showSetUrlFilterHelp();
			return -1;
		}
	}
	if ( action==2 && (!strcmp(mib, "TRIGGERPORT_TBL") || !strcmp(mib, "DEF_TRIGGERPORT_TBL")) ) {
		if (!valNum || (strcmp(value[0], "add") && strcmp(value[0], "del") && strcmp(value[0], "delall"))) {
			showSetTriggerPortHelp();
			return -1;
		}
	}
#ifdef ROUTE_SUPPORT
	if ( action==2 && (!strcmp(mib, "STATICROUTE_TBL") || !strcmp(mib, "DEF_STATICROUTE_TBL")) ) {
		if (!valNum || (strcmp(value[0], "add") && strcmp(value[0], "del") && strcmp(value[0], "delall"))) {
			showSetStaticRouteHelp();
			return -1;
		}
	}
#endif //ROUTE
#endif
#ifdef HOME_GATEWAY
#ifdef VPN_SUPPORT
	if ( action==2 && (!strcmp(mib, "IPSECTUNNEL_TBL") || !strcmp(mib, "DEF_IPSECTUNNEL_TBL")) ) {
		if (!valNum || (strcmp(value[0], "add") && strcmp(value[0], "del") && strcmp(value[0], "delall"))) {
			showSetIpsecTunnelHelp();
			return -1;
		}
	}
#endif
#endif
#ifdef TLS_CLIENT
	if ( action==2 && (!strcmp(mib, "CERTROOT_TBL") || !strcmp(mib, "DEF_CERTROOT_TBL")) ) {
		if (!valNum || (strcmp(value[0], "add") && strcmp(value[0], "del") && strcmp(value[0], "delall"))) {
			showSetCertRootHelp();
			return -1;
		}
	}
	if ( action==2 && (!strcmp(mib, "CERTUSER_TBL") || !strcmp(mib, "DEF_CERTUSER_TBL")) ) {
		if (!valNum || (strcmp(value[0], "add") && strcmp(value[0], "del") && strcmp(value[0], "delall"))) {
			showSetCertUserHelp();
			return -1;
		}
	}	
#endif

	switch (action) {
	case 1: // get

#ifdef PARSE_TXT_FILE
		if ( filename[0] ) {
			if ( parseTxtConfig(filename, &apmib) < 0) {
				printf("Parse text file error!\n");
				return -1;
			}

			if ( !apmib_init(&apmib)) {
				printf("Initialize AP MIB failed!\n");
				return -1;
			}
		}
		else

#endif
		if ( !apmib_init()) {
			printf("Initialize AP MIB failed!\n");
			return -1;
		}

		idx = searchMIB(mib);
		if ( idx == -1 ) {
			showHelp();
			showAllMibName();
			return -1;
		}
		num = 1;
		if (config_area == 4 || config_area == 6) { // wlan default or current
			if (mib_wlan_table[idx].id == MIB_WLAN_AC_ADDR)
				APMIB_GET(MIB_WLAN_AC_NUM, (void *)&num);
			else if (mib_wlan_table[idx].id == MIB_WLAN_WDS)
				APMIB_GET(MIB_WLAN_WDS_NUM, (void *)&num);
		}
#ifdef HOME_GATEWAY
		else if (!strcmp(mib, "PORTFW_TBL"))
			APMIB_GET(MIB_PORTFW_NUM, (void *)&num);
		else if (!strcmp(mib, "PORTFILTER_TBL"))
			APMIB_GET(MIB_PORTFILTER_NUM, (void *)&num);
		else if (!strcmp(mib, "IPFILTER_TBL"))
			APMIB_GET(MIB_IPFILTER_NUM, (void *)&num);
		else if (!strcmp(mib, "MACFILTER_TBL"))
			APMIB_GET(MIB_MACFILTER_NUM, (void *)&num);
		else if (!strcmp(mib, "URLFILTER_TBL"))
			APMIB_GET(MIB_URLFILTER_NUM, (void *)&num);
		else if (!strcmp(mib, "TRIGGERPORT_TBL"))
			APMIB_GET(MIB_TRIGGERPORT_NUM, (void *)&num);
#ifdef ROUTE_SUPPORT
		else if (!strcmp(mib, "STATICROUTE_TBL"))
			APMIB_GET(MIB_STATICROUTE_NUM, (void *)&num);
#endif //ROUTE
#endif

#ifdef HOME_GATEWAY
#ifdef VPN_SUPPORT
		else if (!strcmp(mib, "IPSECTUNNEL_TBL"))
			APMIB_GET(MIB_IPSECTUNNEL_NUM, (void *)&num);
#endif
#endif

#ifdef TLS_CLIENT
		else if (!strcmp(mib, "CERTROOT_TBL"))
			APMIB_GET(MIB_CERTROOT_NUM, (void *)&num);
		else if (!strcmp(mib, "CERTUSER_TBL"))
			APMIB_GET(MIB_CERTUSER_NUM, (void *)&num);			
#endif

		if (config_area == 1)
			pTbl = hwmib_table;
		else if (config_area == 2)
			pTbl = hwmib_wlan_table;
		else if (config_area == 3  || config_area == 5)
			pTbl = mib_table;
		else
			pTbl = mib_wlan_table;

		getMIB(mib, pTbl[idx].id, pTbl[idx].type, num, 1 ,value);
		break;

	case 2: // set
		if ( !apmib_init()) {
			printf("Initialize AP MIB failed!\n");
			return -1;
		}
		idx = searchMIB(mib);
		if ( idx == -1 ) {
			showHelp();
			showAllMibName();
			return -1;
		}
		if ( valNum < 1) {
			showHelp();
			return -1;
		}
		if (config_area == 1)
			pTbl = hwmib_table;
		else if (config_area == 2)
			pTbl = hwmib_wlan_table;
		else if (config_area == 3  || config_area == 5)
			pTbl = mib_table;
		else
			pTbl = mib_wlan_table;
		setMIB(mib, pTbl[idx].id, pTbl[idx].type, pTbl[idx].size, valNum, value);
		break;

	case 3: // all
		dumpAll();
		break;
	}

	return 0;
}

//////////////////////////////////////////////////////////////////////////////////
static int writeDefault(int isAll)
{	PARAM_HEADER_T header;
	APMIB_T mib;
	APMIB_Tp pMib=&mib;
	HW_SETTING_T hwmib;
	char *data;
	int status, fh, offset, i, idx;
	unsigned char checksum;
//	unsigned char buff[sizeof(APMIB_T)+sizeof(checksum)+1];
	unsigned char *buff;

	buff=calloc(1, 0x6000);
	if ( buff == NULL ) {
		printf("Allocate buffer failed!\n");
		return -1;
	}

#ifdef __mips__
	fh = open(FLASH_DEVICE_NAME, O_RDWR);
#endif

#ifdef __i386__
	fh = open(FLASH_DEVICE_NAME, O_RDWR|O_CREAT|O_TRUNC);
	write(fh, buff, 0x6000);
#endif
	if ( fh == -1 ) {
		printf("create file failed!\n");
		return -1;
	}

	if (isAll) {
		// write hw setting
		sprintf(header.signature, "%s%02d", HW_SETTING_HEADER_TAG, HW_SETTING_VER);
		header.len = sizeof(hwmib) + sizeof(checksum);

		memset((char *)&hwmib, '\0', sizeof(hwmib));
		hwmib.boardVer = 1;
		memcpy(hwmib.nic0Addr, "\x0\xe0\x4c\x81\x86\xd1", 6);
		memcpy(hwmib.nic1Addr, "\x0\xe0\x4c\x81\x86", 6);
		hwmib.nic1Addr[5] = 0xd1 + NUM_WLAN_INTERFACE;		
	
		// set RF parameters
		for (idx=0; idx<NUM_WLAN_INTERFACE; idx++) {
			memcpy(hwmib.wlan[idx].macAddr, "\x0\xe0\x4c\x81\x86", 5);		
			hwmib.wlan[idx].macAddr[5] = 0xd1 + idx;			
			hwmib.wlan[idx].regDomain = FCC;
			hwmib.wlan[idx].rfType = (unsigned char)RF_ZEBRA;
			hwmib.wlan[idx].antDiversity = (unsigned char)0; // disabled
			hwmib.wlan[idx].txAnt = 0;
			hwmib.wlan[idx].initGain = 4;
			for (i=0; i<MAX_CCK_CHAN_NUM; i++)
				hwmib.wlan[idx].txPowerCCK[i] = 8;
			for (i=0; i<MAX_OFDM_CHAN_NUM; i++)
				hwmib.wlan[idx].txPowerOFDM[i] = 14;
			}
		data = (char *)&hwmib;
		checksum = CHECKSUM(data, header.len-1);

		lseek(fh, HW_SETTING_OFFSET, SEEK_SET);
		if ( write(fh, (const void *)&header, sizeof(header))!=sizeof(header) ) {
			printf("write hs header failed!\n");
			return -1;
		}
		if ( write(fh, (const void *)&hwmib, sizeof(hwmib))!=sizeof(hwmib) ) {
			printf("write hs MIB failed!\n");
			return -1;
		}
		if ( write(fh, (const void *)&checksum, sizeof(checksum))!=sizeof(checksum) ) {
			printf("write hs checksum failed!\n");
			return -1;
		}

		close(fh);
		sync();
		fh = open(FLASH_DEVICE_NAME, O_RDWR);
	}
	// write default & current setting
	memset(pMib, '\0', sizeof(APMIB_T));

	// give a initial value for testing purpose
	strcpy(pMib->deviceName, "Realtek Wireless AP");
	for (idx=0; idx<NUM_WLAN_INTERFACE; idx++) {
		strcpy(pMib->wlan[idx].ssid, "802.11g-SSID" );
		pMib->wlan[idx].channel = 11;
		pMib->wlan[idx].wep = WEP_DISABLED;
		pMib->wlan[idx].beaconInterval = 100;
		pMib->wlan[idx].basicRates = TX_RATE_1M|TX_RATE_2M|TX_RATE_5M|TX_RATE_11M;
		pMib->wlan[idx].rateAdaptiveEnabled = 1;
		pMib->wlan[idx].supportedRates = TX_RATE_1M|TX_RATE_2M|TX_RATE_5M |TX_RATE_11M|
			TX_RATE_6M|TX_RATE_9M|TX_RATE_12M|TX_RATE_18M|TX_RATE_24M|
				TX_RATE_36M|TX_RATE_48M|TX_RATE_54M;

		pMib->wlan[idx].preambleType = LONG_PREAMBLE;
		pMib->wlan[idx].rtsThreshold = 2347;
		pMib->wlan[idx].fragThreshold = 2346;
		pMib->wlan[idx].authType = AUTH_BOTH;
		pMib->wlan[idx].inactivityTime = 30000; // 300 sec
		pMib->wlan[idx].dtimPeriod = 3;

		pMib->wlan[idx].wpaGroupRekeyTime = 86400;
		pMib->wlan[idx].wpaAuth = (unsigned char)WPA_AUTH_PSK;
		pMib->wlan[idx].wpaCipher = (unsigned char)WPA_CIPHER_TKIP;
		pMib->wlan[idx].rsPort = 1812;
		pMib->wlan[idx].accountRsPort = 1813;
		pMib->wlan[idx].accountRsUpdateDelay = 60;
		pMib->wlan[idx].rsMaxRetry = 3;
		pMib->wlan[idx].rsIntervalTime = 5;
		pMib->wlan[idx].accountRsMaxRetry = 3;
		pMib->wlan[idx].accountRsIntervalTime = 5;
		pMib->wlan[idx].wlanBand = BAND_11BG;

#ifdef WLAN_EASY_CONFIG
		pMib->wlan[idx].acfMode = MODE_BUTTON;
		pMib->wlan[idx].acfAlgReq = ACF_ALGORITHM_WPA2_AES;
		pMib->wlan[idx].acfAlgSupp = ACF_ALGORITHM_WPA_TKIP  | ACF_ALGORITHM_WPA2_AES;
		strcpy(pMib->wlan[idx].acfScanSSID, "REALTEK_EASY_CONFIG");
#endif

#ifdef WIFI_SIMPLE_CONFIG
	pMib->wlan[0].wscMethod = 3;
	strcpy(pMib->wlan[0].wscPin, "12345670");
	pMib->wlan[0].wscAuth = WSC_AUTH_OPEN; //open
	pMib->wlan[0].wscEnc = WSC_ENCRYPT_NONE; //open
	pMib->wlan[0].wscUpnpEnabled = 1;
	pMib->wlan[0].wscRegistrarEnabled = 1;
	strcpy(pMib->wlan[0].wscSsid, pMib->wlan[0].ssid ); //must be the same as pMib->wlan[idx].ssid
#endif

	}

	pMib->ipAddr[0] = 192;
	pMib->ipAddr[1] = 168;
	pMib->ipAddr[2] = 1;
	pMib->ipAddr[3] = 254;

	pMib->subnetMask[0] = 255;
	pMib->subnetMask[1] = 255;
	pMib->subnetMask[2] = 0;
	pMib->subnetMask[3] = 0;

	pMib->dhcpClientStart[0] = 192;
	pMib->dhcpClientStart[1] = 168;
	pMib->dhcpClientStart[2] = 1;
	pMib->dhcpClientStart[3] = 100;

	pMib->dhcpClientEnd[0] = 192;
	pMib->dhcpClientEnd[1] = 168;
	pMib->dhcpClientEnd[2] = 1;
	pMib->dhcpClientEnd[3] = 200;
	pMib->dhcp = DHCP_DISABLED;

	strcpy(pMib->superName, "super");
	strcpy(pMib->superPassword, "super");

#ifdef HOME_GATEWAY
	pMib->wanIpAddr[0] = 192;
	pMib->wanIpAddr[1] = 168;
	pMib->wanIpAddr[2] = 1;
	pMib->wanIpAddr[3] = 1;

	pMib->wanSubnetMask[0] = 255;
	pMib->wanSubnetMask[1] = 255;
	pMib->wanSubnetMask[2] = 0;
	pMib->wanSubnetMask[3] = 0;

	pMib->wanDefaultGateway[0] = 192;
	pMib->wanDefaultGateway[1] = 168;
	pMib->wanDefaultGateway[2] = 1;
	pMib->wanDefaultGateway[3] = 254;

	pMib->wanDhcp = DHCP_CLIENT;
	pMib->dnsMode = DNS_AUTO;
	pMib->pppIdleTime = 300;
	
	pMib->pppMtuSize = 1412;
	pMib->pptpMtuSize = 1412;
	pMib->fixedIpMtuSize = 1412;
	pMib->dhcpMtuSize = 1412;
#endif
	// SNMP, Forrest added, 2007.10.25.
#ifdef CONFIG_SNMP
	pMib->snmpEnabled = 1;
	sprintf(pMib->snmpName, "%s", "Realtek");
	sprintf(pMib->snmpLocation, "%s", "AP");
	sprintf(pMib->snmpContact, "%s", "Router");
	sprintf(pMib->snmpRWCommunity, "%s", "private");
	sprintf(pMib->snmpROCommunity, "%s", "public");
	pMib->snmpTrapReceiver1[0] = 0;
	pMib->snmpTrapReceiver1[1] = 0;
	pMib->snmpTrapReceiver1[2] = 0;
	pMib->snmpTrapReceiver1[3] = 0;
	pMib->snmpTrapReceiver2[0] = 0;
	pMib->snmpTrapReceiver2[1] = 0;
	pMib->snmpTrapReceiver2[2] = 0;
	pMib->snmpTrapReceiver2[3] = 0;
	pMib->snmpTrapReceiver3[0] = 0;
	pMib->snmpTrapReceiver3[1] = 0;
	pMib->snmpTrapReceiver3[2] = 0;
	pMib->snmpTrapReceiver3[3] = 0;
#endif

	data = (char *)pMib;

	// write default setting
	sprintf(header.signature, "%s%02d", DEFAULT_SETTING_HEADER_TAG, DEFAULT_SETTING_VER);
	header.len = sizeof(APMIB_T) + sizeof(checksum);
	checksum = CHECKSUM(data, header.len-1);
	lseek(fh, DEFAULT_SETTING_OFFSET, SEEK_SET);
	if ( write(fh, (const void *)&header, sizeof(header))!=sizeof(header) ) {
		printf("write ds header failed!\n");
		return -1;
	}
	if ( write(fh, (const void *)pMib, sizeof(mib))!=sizeof(mib) ) {
		printf("write ds MIB failed!\n");
		return -1;
	}
	if ( write(fh, (const void *)&checksum, sizeof(checksum))!=sizeof(checksum) ) {
		printf("write ds checksum failed!\n");
		return -1;
	}
	close(fh);
	sync();
	fh = open(FLASH_DEVICE_NAME, O_RDWR);

	// write current setting
	sprintf(header.signature, "%s%02d", CURRENT_SETTING_HEADER_TAG, CURRENT_SETTING_VER);
	header.len = sizeof(APMIB_T) + sizeof(checksum);
	lseek(fh, CURRENT_SETTING_OFFSET, SEEK_SET);
	if ( write(fh, (const void *)&header, sizeof(header))!=sizeof(header) ) {
		printf("write cs header failed!\n");
		return -1;
	}
	if ( write(fh, (const void *)pMib, sizeof(mib))!=sizeof(mib) ) {
		printf("write cs MIB failed!\n");
		return -1;
	}
	if ( write(fh, (const void *)&checksum, sizeof(checksum))!=sizeof(checksum) ) {
		printf("write cs checksum failed!\n");
		return -1;
	}

	close(fh);
	sync();

	// check if hw, ds, cs checksum is ok
	offset = HW_SETTING_OFFSET;
	if ( flash_read((char *)&header, offset, sizeof(header)) == 0) {
		printf("read hs header failed!\n");
		return -1;
	}
	offset += sizeof(header);
	if ( flash_read(buff, offset, header.len) == 0) {
		printf("read hs MIB failed!\n");
		return -1;
	}
	status = CHECKSUM_OK(buff, header.len);
	if ( !status) {
		printf("hs Checksum error!\n");
		return -1;
	}

	offset = DEFAULT_SETTING_OFFSET;
	if ( flash_read((char *)&header, offset, sizeof(header)) == 0) {
		printf("read ds header failed!\n");
		return -1;
	}
	offset += sizeof(header);
	if ( flash_read(buff, offset, header.len) == 0) {
		printf("read ds MIB failed!\n");
		return -1;
	}
	status = CHECKSUM_OK(buff, header.len);
	if ( !status) {
		printf("ds Checksum error!\n");
		return -1;
	}

	offset = CURRENT_SETTING_OFFSET;
	if ( flash_read((char *)&header, offset, sizeof(header)) == 0) {
		printf("read cs header failed!\n");
		return -1;
	}
	offset += sizeof(header);
	if ( flash_read(buff, offset, header.len) == 0) {
		printf("read cs MIB failed!\n");
		return -1;
	}
	status = CHECKSUM_OK(buff, header.len);

	if ( !status) {
		printf("cs Checksum error!\n");
		return -1;
	}

	free(buff);

#if CONFIG_APMIB_SHARED_MEMORY == 1	
    apmib_sem_lock();
    apmib_load_hwconf();
    apmib_load_dsconf();
    apmib_load_csconf();
    apmib_sem_unlock();
#endif

	return 0;
}

///////////////////////////////////////////////////////////////////////////////
static int flash_read(char *buf, int offset, int len)
{
	int fh;
	int ok=1;

#ifdef __mips__
	fh = open(FLASH_DEVICE_NAME, O_RDWR);
#endif

#ifdef __i386__
	fh = open(FLASH_DEVICE_NAME, O_RDONLY);
#endif
	if ( fh == -1 )
		return 0;

	lseek(fh, offset, SEEK_SET);

	if ( read(fh, buf, len) != len)
		ok = 0;

	close(fh);

	return ok;
}

///////////////////////////////////////////////////////////////////////////////
static int searchMIB(char *token)
{
	int idx = 0;
	char tmpBuf[100];
	int desired_config=0;

	if (!memcmp(token, "HW_", 3)) {
		config_area = 1;
		if (!memcmp(&token[3], "WLAN", 4) && token[8] == '_') {
			wlan_idx = token[7] - '0';
			if (wlan_idx >= NUM_WLAN_INTERFACE)
				return -1;
			strcpy(tmpBuf, &token[9]);
			desired_config = config_area+1;
		}
		else
			strcpy(tmpBuf, &token[3]);
	}
	else if (!memcmp(token, "DEF_", 4)) {
		config_area = 3;

		if (!memcmp(&token[4], "WLAN", 4) && token[9] == '_') {
			wlan_idx = token[8] - '0';
			if (wlan_idx >= NUM_WLAN_INTERFACE)
				return -1;
			strcpy(tmpBuf, &token[10]);
			desired_config = config_area+1;
		}
		else
			strcpy(tmpBuf, &token[4]);
	}
	else {
		config_area = 5;

		if (!memcmp(&token[0], "WLAN", 4) && token[5] == '_') {
			wlan_idx = token[4] - '0';
			if (wlan_idx >= NUM_WLAN_INTERFACE)
				return -1;
			strcpy(tmpBuf, &token[6]);
			desired_config = config_area+1;
		}
		else
			strcpy(tmpBuf, &token[0]);
	}

	if ( config_area == 1 ) {
		while (hwmib_table[idx].id) {
			if ( !strcmp(hwmib_table[idx].name, tmpBuf)) {
				if (desired_config && config_area != desired_config)
					return -1;
				return idx;
			}
			idx++;
		}
		idx=0;
		while (hwmib_wlan_table[idx].id) {
			if ( !strcmp(hwmib_wlan_table[idx].name, tmpBuf)) {
				config_area++;
				if (desired_config && config_area != desired_config)
					return -1;
				return idx;
			}
			idx++;
		}
		return -1;
	}
	else {
		while (mib_table[idx].id) {
			if ( !strcmp(mib_table[idx].name, tmpBuf)) {
				if (desired_config && config_area != desired_config)
					return -1;
				return idx;
			}
			idx++;
		}
		idx=0;
		while (mib_wlan_table[idx].id) {
			if ( !strcmp(mib_wlan_table[idx].name, tmpBuf)) {
				config_area++;
				if (desired_config && config_area != desired_config)
					return -1;
				return idx;
			}
			idx++;
		}
		return -1;
	}
}

///////////////////////////////////////////////////////////////////////////////
static void getMIB(char *name, int id, TYPE_T type, int num, int array_separate, char **val)
{
	unsigned char array_val[2048];
	struct in_addr ia_val;
	void *value;
	unsigned char tmpBuf[1024]={0}, *format=NULL, *buf, tmp1[400];
	int int_val, i;
	int index=1, tbl=0;
	char mibName[100]={0};

	if (num ==0)
		goto ret;

	strcat(mibName, name);

getval:
	buf = &tmpBuf[strlen(tmpBuf)];
	switch (type) {
	case BYTE_T:
		value = (void *)&int_val;
		format = DEC_FORMAT;
		break;
	case WORD_T:
		value = (void *)&int_val;
		format = DEC_FORMAT;
		break;
	case IA_T:
		value = (void *)&ia_val;
		format = STR_FORMAT;
		break;
	case BYTE5_T:
		value = (void *)array_val;
		format = BYTE5_FORMAT;
		break;
	case BYTE6_T:
		value = (void *)array_val;
		format = BYTE6_FORMAT;
		break;
	case BYTE13_T:
		value = (void *)array_val;
		format = BYTE13_FORMAT;
		break;

	case STRING_T:
		value = (void *)array_val;
		format = STR_FORMAT;
		break;

	case BYTE_ARRAY_T:
		value = (void *)array_val;
		break;

	case DWORD_T:
		value = (void *)&int_val;
		format = DEC_FORMAT;
		break;

	case WLAC_ARRAY_T:
	case WDS_ARRAY_T:

#ifdef HOME_GATEWAY
	case PORTFW_ARRAY_T:
	case IPFILTER_ARRAY_T:
	case PORTFILTER_ARRAY_T:
	case MACFILTER_ARRAY_T:
	case URLFILTER_ARRAY_T:
	case TRIGGERPORT_ARRAY_T:
#ifdef ROUTE_SUPPORT
	case STATICROUTE_ARRAY_T:
#endif
#ifdef VPN_SUPPORT
	case IPSECTUNNEL_ARRAY_T:
#endif
#endif
#ifdef TLS_CLIENT
	case CERTROOT_ARRAY_T:
	case CERTUSER_ARRAY_T:
#endif
		tbl = 1;
		value = (void *)array_val;
		array_val[0] = index++;
		break;
	default: printf("invalid mib!\n"); return;
	}

	if ( !APMIB_GET(id, value)) {
		printf("Get MIB failed!\n");
		return;
	}

	if ( type == IA_T )
		value = inet_ntoa(ia_val);

	if (type == BYTE_T || type == WORD_T)
		sprintf(buf, format, int_val);
	else if ( type == IA_T || type == STRING_T ) {
		sprintf(buf, format, value);
		
		if (type == STRING_T ) {
			unsigned char tmpBuf1[1024];
			int srcIdx, dstIdx;
			for (srcIdx=0, dstIdx=0; buf[srcIdx]; srcIdx++, dstIdx++) {
				if ( buf[srcIdx] == '"' || buf[srcIdx] == '\\' || buf[srcIdx] == '$' || buf[srcIdx] == '`' || buf[srcIdx] == ' ' )
					tmpBuf1[dstIdx++] = '\\';

				tmpBuf1[dstIdx] = buf[srcIdx];
			}
			if (dstIdx != srcIdx) {
				memcpy(buf, tmpBuf1, dstIdx);
				buf[dstIdx] ='\0';
			}
		}
	}
	else if (type == BYTE5_T) {
		sprintf(buf, format, array_val[0],array_val[1],array_val[2],
			array_val[3],array_val[4],array_val[5]);
		convert_lower(buf);
	}
	else if (type == BYTE6_T ) {
		sprintf(buf, format, array_val[0],array_val[1],array_val[2],
			array_val[3],array_val[4],array_val[5],array_val[6]);
		convert_lower(buf);
	}
	else if (type == BYTE13_T) {
		sprintf(buf, format, array_val[0],array_val[1],array_val[2],
			array_val[3],array_val[4],array_val[5],array_val[6],
			array_val[7],array_val[8],array_val[9],array_val[10],
			array_val[11],array_val[12]);
		convert_lower(buf);
	}
	else if(type == BYTE_ARRAY_T ) {
		int max_chan_num, chan;
		max_chan_num = (id == MIB_HW_TX_POWER_CCK)? MAX_CCK_CHAN_NUM : MAX_OFDM_CHAN_NUM;
		if(val == NULL || val[0] == NULL){
			for(i=0 ;i< max_chan_num ;i++){
				sprintf(tmp1, "%02x", array_val[i]);
				strcat(buf, tmp1);
			}
			convert_lower(buf);
		}
		else{
			chan = atoi(val[0]);
			if(chan < 1 || chan > max_chan_num){
				if((chan<1) || (id==MIB_HW_TX_POWER_CCK) || ((id==MIB_HW_TX_POWER_OFDM) && (chan>216))){
					printf("invalid channel number\n");
					return;
				}
				else{
					if((chan >= 163) && (chan <= 181))
						chan -= 148;
					else // 182 ~ 216
						chan -= 117;
				}
			}
			sprintf(buf, "%d", *(((unsigned char *)value)+chan-1) );
		}
	}
	else if (type == DWORD_T)
		sprintf(buf, format, int_val);

#ifdef HOME_GATEWAY
	else if (type == PORTFW_ARRAY_T) {
		PORTFW_Tp pEntry=(PORTFW_Tp)array_val;
		sprintf(buf, PORTFW_FORMAT, inet_ntoa(*((struct in_addr*)pEntry->ipAddr)),
			 pEntry->fromPort, pEntry->toPort, pEntry->protoType);
		if ( strlen(pEntry->comment) ) {
			sprintf(tmp1, ", %s", pEntry->comment);
			strcat(buf, tmp1);
		}
	}
	else if (type == PORTFILTER_ARRAY_T) {
		PORTFILTER_Tp pEntry=(PORTFILTER_Tp)array_val;
		sprintf(buf, PORTFILTER_FORMAT,
			 pEntry->fromPort, pEntry->toPort, pEntry->protoType);
		if ( strlen(pEntry->comment) ) {
			sprintf(tmp1, ", %s", pEntry->comment);
			strcat(buf, tmp1);
		}
	}
	else if (type == IPFILTER_ARRAY_T) {
		IPFILTER_Tp pEntry=(IPFILTER_Tp)array_val;
		sprintf(buf, IPFILTER_FORMAT, inet_ntoa(*((struct in_addr*)pEntry->ipAddr)),
			 pEntry->protoType);
		if ( strlen(pEntry->comment) ) {
			sprintf(tmp1, ", %s", pEntry->comment);
			strcat(buf, tmp1);
		}
	}
	else if (type == MACFILTER_ARRAY_T) {
		MACFILTER_Tp pEntry=(MACFILTER_Tp)array_val;
		sprintf(buf, MACFILTER_COLON_FORMAT, pEntry->macAddr[0],pEntry->macAddr[1],pEntry->macAddr[2],
			 pEntry->macAddr[3],pEntry->macAddr[4],pEntry->macAddr[5]);
		if ( strlen(pEntry->comment) ) {
			sprintf(tmp1, ", %s", pEntry->comment);
			strcat(buf, tmp1);
		}
	}
	else if (type == URLFILTER_ARRAY_T) {
		URLFILTER_Tp pEntry=(URLFILTER_Tp)array_val;
		sprintf(buf, STR_FORMAT, pEntry->urlAddr);
		//if ( strlen(pEntry->comment) ) {
		//	sprintf(tmp1, ", %s", pEntry->comment);
		//	strcat(buf, tmp1);
		//}
	}
	else if (type == TRIGGERPORT_ARRAY_T) {
		TRIGGERPORT_Tp pEntry=(TRIGGERPORT_Tp)array_val;
		sprintf(buf, TRIGGERPORT_FORMAT,
			 pEntry->tri_fromPort, pEntry->tri_toPort, pEntry->tri_protoType,
			 pEntry->inc_fromPort, pEntry->inc_toPort, pEntry->inc_protoType);
		if ( strlen(pEntry->comment) ) {
			sprintf(tmp1, ", %s", pEntry->comment);
			strcat(buf, tmp1);
		}
	}
#ifdef ROUTE_SUPPORT
	else if (type == STATICROUTE_ARRAY_T) {
		char strIp[20], strMask[20], strGw[20];
		STATICROUTE_Tp pEntry=(STATICROUTE_Tp)array_val;
		strcpy(strIp, inet_ntoa(*((struct in_addr*)pEntry->dstAddr)));
                strcpy(strMask, inet_ntoa(*((struct in_addr*)pEntry->netmask)));
                strcpy(strGw, inet_ntoa(*((struct in_addr*)pEntry->gateway)));
		sprintf(buf, "%s, %s, %s",strIp, strMask, strGw);
	}
#endif //ROUTE
#endif
#ifdef HOME_GATEWAY
#ifdef VPN_SUPPORT
	else if (type == IPSECTUNNEL_ARRAY_T) {
		IPSECTUNNEL_Tp pEntry=(IPSECTUNNEL_Tp)array_val;
		char strLcIp[20], strRtIp[20], strRtGw[20];
		strcpy(strLcIp, inet_ntoa(*((struct in_addr*)pEntry->lc_ipAddr)));
		strcpy(strRtIp, inet_ntoa(*((struct in_addr*)pEntry->rt_ipAddr)));
		strcpy(strRtGw, inet_ntoa(*((struct in_addr*)pEntry->rt_gwAddr)));

		sprintf(buf, IPSECTUNNEL_FORMAT, pEntry->tunnelId, pEntry->enable,  pEntry->connName, 
		pEntry->lcType, strLcIp, pEntry->lc_maskLen, 
		pEntry->rtType, strRtIp, pEntry->rt_maskLen, 
		strRtGw, pEntry->keyMode, 
		pEntry->conType, pEntry->espEncr, pEntry->espAuth, 
		pEntry->psKey, pEntry->ikeEncr, pEntry->ikeAuth, pEntry->ikeKeyGroup, pEntry->ikeLifeTime, 
		pEntry->ipsecLifeTime, pEntry->ipsecPfs, pEntry->spi, pEntry->encrKey, pEntry->authKey, pEntry->authType, pEntry->lcId, pEntry->rtId, pEntry->lcIdType, pEntry->rtIdType, pEntry->rsaKey
		);
	}
#endif
#endif
#ifdef TLS_CLIENT
	else if (type == CERTROOT_ARRAY_T) {
		CERTROOT_Tp pEntry=(CERTROOT_Tp)array_val;
		sprintf(buf, "%s", pEntry->comment);
	}
	else if (type == CERTUSER_ARRAY_T) {
		CERTUSER_Tp pEntry=(CERTUSER_Tp)array_val;
		sprintf(buf, "%s,%s", pEntry->comment, pEntry->pass);
	}	
#endif
	else if (type == WLAC_ARRAY_T) {
		MACFILTER_Tp pEntry=(MACFILTER_Tp)array_val;
		sprintf(buf, MACFILTER_FORMAT, pEntry->macAddr[0],pEntry->macAddr[1],pEntry->macAddr[2],
			 pEntry->macAddr[3],pEntry->macAddr[4],pEntry->macAddr[5]);
		if ( strlen(pEntry->comment) ) {
			sprintf(tmp1, ", %s", pEntry->comment);
			strcat(buf, tmp1);
		}
	}
	else if (type == WDS_ARRAY_T) {
		WDS_Tp pEntry=(WDS_Tp)array_val;
		sprintf(buf, WDS_FORMAT, pEntry->macAddr[0],pEntry->macAddr[1],pEntry->macAddr[2],
			 pEntry->macAddr[3],pEntry->macAddr[4],pEntry->macAddr[5]);
		if ( strlen(pEntry->comment) ) {
			sprintf(tmp1, ", %s", pEntry->comment);
			strcat(buf, tmp1);
		}
	}
	if (--num > 0) {
		if (!array_separate)
			strcat(tmpBuf, " ");
		else {
			if (tbl){
				if(type == STRING_T)
					printf("%s%d=\"%s\"\n", mibName, index-1, tmpBuf);
				else
					printf("%s%d=%s\n", mibName, index-1, tmpBuf);
			}
			else{
				if(type == STRING_T)
					printf("%s=\"%s\"\n", mibName, tmpBuf);
				else
					printf("%s=%s\n", mibName, tmpBuf);
			}
			tmpBuf[0] = '\0';
		}
		goto getval;
	}
ret:
	if (tbl) {
		if(type == STRING_T)
			printf("%s%d=\"%s\"\n", mibName, index-1, tmpBuf);
		else
			printf("%s%d=%s\n", mibName, index-1, tmpBuf);
	}
	else{
		if(type == STRING_T)
                                printf("%s=\"%s\"\n", mibName, tmpBuf);
		else
			printf("%s=%s\n", mibName, tmpBuf);
	}
}

///////////////////////////////////////////////////////////////////////////////
static void setMIB(char *name, int id, TYPE_T type, int len, int valNum, char **val)
{
	unsigned char key[170];
	struct in_addr ia_val;
	void *value;
	int int_val, i;
	MACFILTER_T wlAc;
	WDS_T wds;

#ifdef HOME_GATEWAY
	PORTFW_T portFw;
	PORTFILTER_T portFilter;
	IPFILTER_T ipFilter;
	MACFILTER_T macFilter;
	URLFILTER_T urlFilter;
	TRIGGERPORT_T triggerPort;
#ifdef ROUTE_SUPPORT
	STATICROUTE_T staticRoute;
#endif
#endif

#ifdef HOME_GATEWAY
#ifdef VPN_SUPPORT
	IPSECTUNNEL_T ipsecTunnel;
#endif
#endif
#ifdef TLS_CLIENT
	CERTROOT_T certRoot;
	CERTUSER_T certUser;
#endif
	int entryNum;
	int max_chan_num, tx_power_cnt=0;

	switch (type) {
	case BYTE_T:
	case WORD_T:
		int_val = atoi(val[0]);
		value = (void *)&int_val;
		break;

	case IA_T:
		if ( !inet_aton(val[0], &ia_val) ) {
			printf("invalid internet address!\n");
			return;
		}
		value = (void *)&ia_val;
		break;

	case BYTE5_T:
		if ( strlen(val[0])!=10 || !string_to_hex(val[0], key, 10)) {
			printf("invalid value!\n");
			return;
		}
		value = (void *)key;
		break;

	case BYTE6_T:
		if ( strlen(val[0])!=12 || !string_to_hex(val[0], key, 12)) {
			printf("invalid value!\n");
			return;
		}
		value = (void *)key;
		break;

	case BYTE_ARRAY_T:
		if ( (id!=MIB_HW_TX_POWER_CCK) && (id!=MIB_HW_TX_POWER_OFDM)) {		
			printf("invalid mib!\n");
			return;
		}
		max_chan_num = (id==MIB_HW_TX_POWER_CCK)? MAX_CCK_CHAN_NUM : MAX_OFDM_CHAN_NUM ;
		for (i=0; i<max_chan_num ; i++) {
			if(val[i] == NULL) break;
			if ( !sscanf(val[i], "%d", &int_val) ) {
				printf("invalid value!\n");
				return;
			}
			key[i] = (unsigned char)int_val;
			tx_power_cnt ++;
		}		
		if(tx_power_cnt != 1 && tx_power_cnt !=2 && tx_power_cnt != max_chan_num){
			unsigned char key_tmp[170];
			memcpy(key_tmp, key, tx_power_cnt);		
			APMIB_GET(id, key);
			memcpy(key, key_tmp, tx_power_cnt);
		}
		if(tx_power_cnt == 1){
			for(i=1 ; i < max_chan_num; i++) {
				key[i] = key[0];
			}
		}
		else if(tx_power_cnt ==2){
			//key[0] is channel number to set
			//key[1] is tx power value
			if(key[0] < 1 || key[0] > max_chan_num){
				if((key[0]<1) || (id==MIB_HW_TX_POWER_CCK) || ((id==MIB_HW_TX_POWER_OFDM) && (key[0]>216))){
					printf("invalid channel number\n");
					return;
				}
				else{
					if ((key[0] >= 163) && (key[0] <= 181))
						key[0] -= 148;
					else // 182 ~ 216
						key[0] -= 117;
				}
			}
			key[2] = 0xff ;
		}
		value = (void *)key;
		break;
	case DWORD_T:
		int_val = atoi(val[0]);
		value = (void *)&int_val;
		break;

#ifdef HOME_GATEWAY
	case PORTFW_ARRAY_T:
		if ( !strcmp(val[0], "add")) {
			id = MIB_PORTFW_ADD;
			if ( valNum < 5 ) {
				printf("input argument is not enough!\n");
				return;
			}
			if ( !inet_aton(val[1], (struct in_addr *)&portFw.ipAddr)) {
				printf("invalid internet address!\n");
				return;
			}
			portFw.fromPort = atoi(val[2]);
			portFw.toPort = atoi(val[3]);
			portFw.protoType = atoi(val[4]);
			if ( valNum > 5)
				strcpy(portFw.comment, val[5]);
			else
				portFw.comment[0] = '\0';

		}
		else if ( !strcmp(val[0], "del")) {
			id = MIB_PORTFW_DEL;
			if ( valNum < 2 ) {
				printf("input argument is not enough!\n");
				return;
			}
			int_val = atoi(val[1]);
			if ( !APMIB_GET(MIB_PORTFW_NUM, (void *)&entryNum)) {
				printf("Get port forwarding entry number error!");
				return;
			}
			if ( int_val > entryNum ) {
				printf("Element number is too large!\n");
				return;
			}
			*((char *)&portFw) = (char)int_val;
			if ( !APMIB_GET(MIB_PORTFW, (void *)&portFw)) {
				printf("Get table entry error!");
				return;
			}
		}
		else if ( !strcmp(val[0], "delall"))
			id = MIB_PORTFW_DELALL;

		value = (void *)&portFw;
		break;

	case PORTFILTER_ARRAY_T:
		if ( !strcmp(val[0], "add")) {
			id = MIB_PORTFILTER_ADD;
			if ( valNum < 4 ) {
				printf("input argument is not enough!\n");
				return;
			}
			portFilter.fromPort = atoi(val[1]);
			portFilter.toPort = atoi(val[2]);
			portFilter.protoType = atoi(val[3]);
			if ( valNum > 4)
				strcpy(portFilter.comment, val[4]);
			else
				portFilter.comment[0] = '\0';

		}
		else if ( !strcmp(val[0], "del")) {
			id = MIB_PORTFILTER_DEL;
			if ( valNum < 2 ) {
				printf("input argument is not enough!\n");
				return;
			}
			int_val = atoi(val[1]);
			if ( !APMIB_GET(MIB_PORTFILTER_NUM, (void *)&entryNum)) {
				printf("Get port filter entry number error!");
				return;
			}
			if ( int_val > entryNum ) {
				printf("Element number is too large!\n");
				return;
			}
			*((char *)&portFilter) = (char)int_val;
			if ( !APMIB_GET(MIB_PORTFILTER, (void *)&portFilter)) {
				printf("Get table entry error!");
				return;
			}
		}
		else if ( !strcmp(val[0], "delall"))
			id = MIB_PORTFILTER_DELALL;

		value = (void *)&portFilter;
		break;

	case IPFILTER_ARRAY_T:
		if ( !strcmp(val[0], "add")) {
			id = MIB_IPFILTER_ADD;
			if ( valNum < 3 ) {
				printf("input argument is not enough!\n");
				return;
			}
			if ( !inet_aton(val[1], (struct in_addr *)&ipFilter.ipAddr)) {
				printf("invalid internet address!\n");
				return;
			}
			ipFilter.protoType = atoi(val[2]);
			if ( valNum > 3)
				strcpy(ipFilter.comment, val[3]);
			else
				ipFilter.comment[0] = '\0';

		}
		else if ( !strcmp(val[0], "del")) {
			id = MIB_IPFILTER_DEL;
			if ( valNum < 2 ) {
				printf("input argument is not enough!\n");
				return;
			}
			int_val = atoi(val[1]);
			if ( !APMIB_GET(MIB_IPFILTER_NUM, (void *)&entryNum)) {
				printf("Get port forwarding entry number error!");
				return;
			}
			if ( int_val > entryNum ) {
				printf("Element number is too large!\n");
				return;
			}
			*((char *)&ipFilter) = (char)int_val;
			if ( !APMIB_GET(MIB_IPFILTER, (void *)&ipFilter)) {
				printf("Get table entry error!");
				return;
			}
		}
		else if ( !strcmp(val[0], "delall"))
			id = MIB_IPFILTER_DELALL;
		
		value = (void *)&ipFilter;
		break;

	case MACFILTER_ARRAY_T:
		if ( !strcmp(val[0], "add")) {
			id = MIB_MACFILTER_ADD;
			if ( valNum < 2 ) {
				printf("input argument is not enough!\n");
				return;
			}
			if ( strlen(val[1])!=12 || !string_to_hex(val[1], wlAc.macAddr, 12)) {
				printf("invalid value!\n");
				return;
			}

			if ( valNum > 2)
				strcpy(macFilter.comment, val[2]);
			else
				macFilter.comment[0] = '\0';

		}
		else if ( !strcmp(val[0], "del")) {
			id = MIB_MACFILTER_DEL;
			if ( valNum < 2 ) {
				printf("input argument is not enough!\n");
				return;
			}
			int_val = atoi(val[1]);
			if ( !APMIB_GET(MIB_MACFILTER_NUM, (void *)&entryNum)) {
				printf("Get port forwarding entry number error!");
				return;
			}
			if ( int_val > entryNum ) {
				printf("Element number is too large!\n");
				return;
			}
			*((char *)&macFilter) = (char)int_val;
			if ( !APMIB_GET(MIB_MACFILTER, (void *)&macFilter)) {
				printf("Get table entry error!");
				return;
			}
		}
		else if ( !strcmp(val[0], "delall"))
			id = MIB_MACFILTER_DELALL;
		value = (void *)&macFilter;
		break;

	case URLFILTER_ARRAY_T:
		if ( !strcmp(val[0], "add")) {
			id = MIB_URLFILTER_ADD;
			if ( valNum < 2 ) {
				printf("input argument is not enough!\n");
				return;
			}
			//if ( strlen(val[1])!=12 || !string_to_hex(val[1], wlAc.macAddr, 12)) {
			//	printf("invalid value!\n");
			//	return;
			//}

			//if ( valNum > 2)
			//	strcpy(urlFilter.comment, val[2]);
			//else
			//	uslFilter.comment[0] = '\0';

		}
		else if ( !strcmp(val[0], "del")) {
			id = MIB_URLFILTER_DEL;
			if ( valNum < 2 ) {
				printf("input argument is not enough!\n");
				return;
			}
			int_val = atoi(val[1]);
			if ( !APMIB_GET(MIB_URLFILTER_NUM, (void *)&entryNum)) {
				printf("Get port forwarding entry number error!");
				return;
			}
			if ( int_val > entryNum ) {
				printf("Element number is too large!\n");
				return;
			}
			*((char *)&urlFilter) = (char)int_val;
			if ( !APMIB_GET(MIB_URLFILTER, (void *)&urlFilter)) {
				printf("Get table entry error!");
				return;
			}
		}
		else if ( !strcmp(val[0], "delall"))
			id = MIB_URLFILTER_DELALL;
		value = (void *)&urlFilter;
		break;
		
	case TRIGGERPORT_ARRAY_T:
		if ( !strcmp(val[0], "add")) {
			id = MIB_TRIGGERPORT_ADD;
			if ( valNum < 7 ) {
				printf("input argument is not enough!\n");
				return;
			}
			triggerPort.tri_fromPort = atoi(val[1]);
			triggerPort.tri_toPort = atoi(val[2]);
			triggerPort.tri_protoType = atoi(val[3]);
			triggerPort.inc_fromPort = atoi(val[4]);
			triggerPort.inc_toPort = atoi(val[5]);
			triggerPort.inc_protoType = atoi(val[6]);

			if ( valNum > 7)
				strcpy(triggerPort.comment, val[7]);
			else
				triggerPort.comment[0] = '\0';

		}
		else if ( !strcmp(val[0], "del")) {
			id = MIB_TRIGGERPORT_DEL;
			if ( valNum < 2 ) {
				printf("input argument is not enough!\n");
				return;
			}
			int_val = atoi(val[1]);
			if ( !APMIB_GET(MIB_TRIGGERPORT_NUM, (void *)&entryNum)) {
				printf("Get trigger-port entry number error!");
				return;
			}
			if ( int_val > entryNum ) {
				printf("Element number is too large!\n");
				return;
			}
			*((char *)&triggerPort) = (char)int_val;
			if ( !APMIB_GET(MIB_TRIGGERPORT, (void *)&triggerPort)) {
				printf("Get trigger-port table entry error!");
				return;
			}
		}
		else if ( !strcmp(val[0], "delall"))
			id = MIB_TRIGGERPORT_DELALL;

		value = (void *)&triggerPort;
		break;
#ifdef ROUTE_SUPPORT
		case STATICROUTE_ARRAY_T:
		if ( !strcmp(val[0], "add")) {
			id = MIB_STATICROUTE_ADD;
			if ( valNum < 3 ) {
				printf("input argument is not enough!\n");
				return;
			}
			if ( !inet_aton(val[1], (struct in_addr *)&staticRoute.dstAddr)) {
                                printf("invalid destination IP address!\n");
                                return;
                        }
			if ( !inet_aton(val[2], (struct in_addr *)&staticRoute.netmask)) {
                                printf("invalid netmask !\n");
                                return;
                        }
			if ( !inet_aton(val[3], (struct in_addr *)&staticRoute.gateway)) {
                                printf("invalid gateway address!\n");
                                return;
                        }
		}
		else if ( !strcmp(val[0], "del")) {
			id = MIB_STATICROUTE_DEL;
			if ( valNum < 2 ) {
				printf("input argument is not enough!\n");
				return;
			}
			int_val = atoi(val[1]);
			if ( !APMIB_GET(MIB_STATICROUTE_NUM, (void *)&entryNum)) {
				printf("Get trigger-port entry number error!");
				return;
			}
			if ( int_val > entryNum ) {
				printf("Element number is too large!\n");
				return;
			}
			*((char *)&staticRoute) = (char)int_val;
			if ( !APMIB_GET(MIB_STATICROUTE, (void *)&staticRoute)) {
				printf("Get trigger-port table entry error!");
				return;
			}
		}
		else if ( !strcmp(val[0], "delall"))
			id = MIB_STATICROUTE_DELALL;

		value = (void *)&staticRoute;
		break;
#endif //ROUTE

#endif

	case WLAC_ARRAY_T:
		if ( !strcmp(val[0], "add")) {
			id = MIB_WLAN_AC_ADDR_ADD;
			if ( valNum < 2 ) {
				printf("input argument is not enough!\n");
				return;
			}
			if ( strlen(val[1])!=12 || !string_to_hex(val[1], wlAc.macAddr, 12)) {
				printf("invalid value!\n");
				return;
			}

			if ( valNum > 2)
				strcpy(wlAc.comment, val[2]);
			else
				wlAc.comment[0] = '\0';
		}
		else if ( !strcmp(val[0], "del")) {
			id = MIB_WLAN_AC_ADDR_DEL;
			if ( valNum < 2 ) {
				printf("input argument is not enough!\n");
				return;
			}
			int_val = atoi(val[1]);
			if ( !APMIB_GET(MIB_WLAN_AC_NUM, (void *)&entryNum)) {
				printf("Get port forwarding entry number error!");
				return;
			}
			if ( int_val > entryNum ) {
				printf("Element number is too large!\n");
				return;
			}
			*((char *)&wlAc) = (char)int_val;
			if ( !APMIB_GET(MIB_WLAN_AC_ADDR, (void *)&wlAc)) {
				printf("Get table entry error!");
				return;
			}
		}
		else if ( !strcmp(val[0], "delall"))
			id = MIB_WLAN_AC_ADDR_DELALL;
		value = (void *)&wlAc;
		break;

	case WDS_ARRAY_T:
		if ( !strcmp(val[0], "add")) {
			id = MIB_WLAN_WDS_ADD;
			if ( valNum < 2 ) {
				printf("input argument is not enough!\n");
				return;
			}
			if ( strlen(val[1])!=12 || !string_to_hex(val[1], wds.macAddr, 12)) {
				printf("invalid value!\n");
				return;
			}

			if ( valNum > 2)
				strcpy(wds.comment, val[2]);
			else
				wds.comment[0] = '\0';
		}
		else if ( !strcmp(val[0], "del")) {
			id = MIB_WLAN_WDS_DEL;
			if ( valNum < 2 ) {
				printf("input argument is not enough!\n");
				return;
			}
			int_val = atoi(val[1]);
			if ( !APMIB_GET(MIB_WLAN_WDS_NUM, (void *)&entryNum)) {
				printf("Get wds number error!");
				return;
			}
			if ( int_val > entryNum ) {
				printf("Element number is too large!\n");
				return;
			}
			*((char *)&wds) = (char)int_val;
			if ( !APMIB_GET(MIB_WLAN_WDS, (void *)&wds)) {
				printf("Get table entry error!");
				return;
			}
		}
		else if ( !strcmp(val[0], "delall"))
			id = MIB_WLAN_WDS_DELALL;
		value = (void *)&wds;
		break;

#ifdef HOME_GATEWAY
#ifdef VPN_SUPPORT
	case IPSECTUNNEL_ARRAY_T:
		if ( !strcmp(val[0], "add")) {
			id = MIB_IPSECTUNNEL_ADD;
			if ( valNum < 27 ) {
				printf("input argument is not enough!\n");
				return;
			}
			if ( !inet_aton(val[5], (struct in_addr *)&ipsecTunnel.lc_ipAddr)) {
				printf("invalid local IP address!\n");
				return;
			}
			
			if ( !inet_aton(val[8], (struct in_addr *)&ipsecTunnel.rt_ipAddr)) {
				printf("invalid remote IP address!\n");
				return;
			}
			if ( !inet_aton(val[10], (struct in_addr *)&ipsecTunnel.rt_gwAddr)) {
				printf("invalid remote gateway address!\n");
				return;
			}
			ipsecTunnel.tunnelId =  atoi(val[1]); 
			ipsecTunnel.enable = atoi(val[2]);
			ipsecTunnel.lcType = atoi(val[4]);

			if(strlen(val[3]) > (MAX_NAME_LEN-1)){
				printf("Connection Name too long !\n");
				return;
			}else
				strcpy(ipsecTunnel.connName, val[3]); 
			ipsecTunnel.lc_maskLen = atoi(val[6]);
			ipsecTunnel.rt_maskLen  = atoi(val[9]);
			ipsecTunnel.keyMode= atoi(val[11]);
			ipsecTunnel.conType = atoi(val[12]);
			ipsecTunnel.espEncr = atoi(val[13]);
			ipsecTunnel.espAuth = atoi(val[14]);
			if(strlen(val[15]) >  (MAX_NAME_LEN-1)){
				printf("Preshared Key too long !\n");
				return;
			}else
				strcpy(ipsecTunnel.psKey, val[15]); 

			ipsecTunnel.ikeEncr = atoi(val[16]);
			ipsecTunnel.ikeAuth = atoi(val[17]);
			ipsecTunnel.ikeKeyGroup = atoi(val[18]);
			ipsecTunnel.ikeLifeTime= strtol(val[19], (char **)NULL, 10);

			ipsecTunnel.ipsecLifeTime= strtol(val[20], (char **)NULL, 10);
			ipsecTunnel.ipsecPfs= atoi(val[21]);
			if(strlen(val[22]) >  (MAX_SPI_LEN-1)){
				printf("SPI too long !\n");
				return;
			}else
				strcpy(ipsecTunnel.spi, val[22]); 

			if(strlen(val[23]) >  (MAX_ENCRKEY_LEN-1)){
				printf("Encryption key too long !\n");
				return;
			}else
				strcpy(ipsecTunnel.encrKey, val[23]); 

			if(strlen(val[24]) >  (MAX_AUTHKEY_LEN-1)){
				printf("Authentication key too long !\n");
				return;
			}else
				strcpy(ipsecTunnel.authKey, val[24]); 


		}
		else if ( !strcmp(val[0], "del")) {
			id = MIB_IPSECTUNNEL_DEL;
			if ( valNum < 2 ) {
				printf("input argument is not enough!\n");
				return;
			}
			int_val = atoi(val[1]);
			if ( !APMIB_GET(MIB_IPSECTUNNEL_NUM, (void *)&entryNum)) {
				printf("Get ipsec tunnel number error!");
				return;
			}
			if ( int_val > entryNum ) {
				printf("Element number is too large!\n");
				return;
			}
			*((char *)&ipsecTunnel) = (char)int_val;
			if ( !APMIB_GET(MIB_IPSECTUNNEL, (void *)&ipsecTunnel)) {
				printf("Get table entry error!");
				return;
			}
		}
		else if ( !strcmp(val[0], "delall"))
			id = MIB_IPSECTUNNEL_DELALL;
		value = (void *)&ipsecTunnel;
		break;

#endif
#endif
#ifdef TLS_CLIENT
	case CERTROOT_ARRAY_T:
	if ( !strcmp(val[0], "add")) {
		id = MIB_CERTROOT_ADD;
		strcpy(certRoot.comment, val[1]);
	}
	else if ( !strcmp(val[0], "del")) {
			id = MIB_CERTROOT_DEL;
			if ( valNum < 2 ) {
				printf("input argument is not enough!\n");
				return;
			}
			int_val = atoi(val[1]);
			if ( !APMIB_GET(MIB_CERTROOT_NUM, (void *)&entryNum)) {
				printf("Get cert ca number error!");
				return;
			}
			if ( int_val > entryNum ) {
				printf("Element number is too large!\n");
				return;
			}
			*((char *)&certRoot) = (char)int_val;
			if ( !APMIB_GET(MIB_CERTROOT, (void *)&certRoot)) {
				printf("Get table entry error!");
				return;
			}			
	}
	else if ( !strcmp(val[0], "delall"))
			id = MIB_CERTROOT_DELALL;
	value = (void *)&certRoot;
	break;
	case CERTUSER_ARRAY_T:
	if ( !strcmp(val[0], "add")) {
		id = MIB_CERTUSER_ADD;
		strcpy(certUser.comment, val[1]);
		strcpy(certUser.pass , val[2]);
	}
	else if ( !strcmp(val[0], "del")) {
			id = MIB_CERTUSER_DEL;
			if ( valNum < 2 ) {
				printf("input argument is not enough!\n");
				return;
			}
			int_val = atoi(val[1]);
			if ( !APMIB_GET(MIB_CERTUSER_NUM, (void *)&entryNum)) {
				printf("Get cert ca number error!");
				return;
			}
			if ( int_val > entryNum ) {
				printf("Element number is too large!\n");
				return;
			}
			*((char *)&certUser) = (char)int_val;
			if ( !APMIB_GET(MIB_CERTUSER, (void *)&certUser)) {
				printf("Get table entry error!");
				return;
			}			
	}
	else if ( !strcmp(val[0], "delall"))
			id = MIB_CERTUSER_DELALL;
	value = (void *)&certUser;
	break;	
#endif
	case BYTE13_T:
		if ( strlen(val[0])!=26 || !string_to_hex(val[0], key, 26)) {
			printf("invalid value!\n");
			return;
		}
		value = (void *)key;
		break;

	case STRING_T:
		if ( strlen(val[0]) > len) {
			printf("string value too long!\n");
			return;
		}
		value = (void *)val[0];
		break;
	default: printf("invalid mib!\n"); return;
	}

	if ( !APMIB_SET(id, value))
		printf("set MIB failed!\n");

	if (config_area) {
		if (config_area == 1 || config_area == 2)
			apmib_update(HW_SETTING);
		else if (config_area == 3 || config_area == 4)
			apmib_update(DEFAULT_SETTING);
		else
			apmib_update(CURRENT_SETTING);
	}
}


////////////////////////////////////////////////////////////////////////////////
static void dumpAll(void)
{
	int idx=0, num;
	mib_table_entry_T *pTbl=NULL;

	if ( !apmib_init()) {
		printf("Initialize AP MIB failed!\n");
		return;
	}

	 config_area=0;

next_tbl:
	if (++config_area > 6)
	 	return;
	if (config_area == 1)
		pTbl = hwmib_table;
	else if (config_area == 2)
		pTbl = hwmib_wlan_table;
	else if (config_area == 3 || config_area == 5)
		pTbl = mib_table;
	else if (config_area == 4 || config_area == 6)
		pTbl = mib_wlan_table;

next_wlan:
	while (pTbl[idx].id) {
		if ( pTbl[idx].id == MIB_WLAN_AC_ADDR)
			APMIB_GET(MIB_WLAN_AC_NUM, (void *)&num);
		else if ( pTbl[idx].id == MIB_WLAN_WDS)
			APMIB_GET(MIB_WLAN_WDS_NUM, (void *)&num);
#ifdef HOME_GATEWAY
		else if ( pTbl[idx].id == MIB_PORTFW)
			APMIB_GET(MIB_PORTFW_NUM, (void *)&num);
		else if ( pTbl[idx].id == MIB_PORTFILTER)
			APMIB_GET(MIB_PORTFILTER_NUM, (void *)&num);
		else if ( pTbl[idx].id == MIB_IPFILTER)
			APMIB_GET(MIB_IPFILTER_NUM, (void *)&num);
		else if ( pTbl[idx].id == MIB_MACFILTER)
			APMIB_GET(MIB_MACFILTER_NUM, (void *)&num);
		else if ( pTbl[idx].id == MIB_URLFILTER)
			APMIB_GET(MIB_URLFILTER_NUM, (void *)&num);
		else if ( pTbl[idx].id == MIB_TRIGGERPORT)
			APMIB_GET(MIB_TRIGGERPORT_NUM, (void *)&num);
#ifdef ROUTE_SUPPORT
		else if ( pTbl[idx].id == MIB_STATICROUTE)
			APMIB_GET(MIB_STATICROUTE_NUM, (void *)&num);
#endif //ROUTE
#ifdef VPN_SUPPORT
		else if ( pTbl[idx].id == MIB_IPSECTUNNEL)
			APMIB_GET(MIB_IPSECTUNNEL_NUM, (void *)&num);
#endif
#endif
#ifdef TLS_CLIENT
		else if ( pTbl[idx].id == MIB_CERTROOT)
			APMIB_GET(MIB_CERTROOT_NUM, (void *)&num);
		else if ( pTbl[idx].id == MIB_CERTUSER)
			APMIB_GET(MIB_CERTUSER_NUM, (void *)&num);			
#endif
		else
			num = 1;
		if (num >0) {
			if (config_area == 1 || config_area == 2)
				printf("HW_");
			else if (config_area == 3 || config_area == 4)
				printf("DEF_");
			if (config_area == 2 || config_area == 4 || config_area == 6)
				printf("WLAN%d_", wlan_idx);
			getMIB(pTbl[idx].name, pTbl[idx].id,
						pTbl[idx].type, num, 1 , NULL);
		}
		idx++;
	}
	idx = 0;

	if (config_area == 2 || config_area == 4 || config_area == 6) {
		if (++wlan_idx < NUM_WLAN_INTERFACE) 
			goto next_wlan;
		else
			wlan_idx = 0;		
	}
	
	goto next_tbl;
}

//////////////////////////////////////////////////////////////////////////////////
static void showHelp(void)
{
	printf("Usage: flash cmd\n");
	printf("option:\n");
	printf("cmd:\n");
	printf("      default -- write flash parameters to default.\n");
	printf("      get [wlan interface-index] mib-name -- get a specific mib from flash\n");
	printf("          memory.\n");
	printf("      set [wlan interface-index] mib-name mib-value -- set a specific mib into\n");
	printf("          flash memory.\n");
	printf("      all -- dump all flash parameters.\n");
	printf("      reset -- reset current setting to default.\n");
#ifdef WLAN_FAST_INIT
	printf("      set_mib -- get mib from flash and set to wlan interface.\n");
#endif
	printf("\n");
}


//////////////////////////////////////////////////////////////////////////////////
static void showAllMibName(void)
{
	int idx;
	mib_table_entry_T *pTbl;

	config_area = 0;
	while (config_area++ < 7) {
		idx = 0;
		if (config_area == 1 || config_area == 2) {
			if (config_area == 1)
				pTbl = hwmib_table;
			else
				pTbl = hwmib_wlan_table;
			while (pTbl[idx].id) {
				printf("HW_%s\n", pTbl[idx].name);
				idx++;
			}
		}
		else {
			if (config_area == 3 || config_area == 5)
				pTbl = mib_table;
			else
				pTbl = mib_wlan_table;

			if (config_area == 3 || config_area == 4)
				printf("DEF_");

			while (pTbl[idx].id) {
				printf("%s\n", pTbl[idx].name);
				idx++;
			}
		}
	}
}

///////////////////////////////////////////////////////////////////////////////////
static void showSetACHelp(void)
{
#if 0
	printf("flash set MACAC_ADDR cmd\n");
	printf("cmd:\n");
	printf("      add mac-addr comment -- append a filter mac address.\n");
	printf("      del entry-number -- delete a filter entry.\n");
	printf("      delall -- delete all filter mac address.\n");
#endif	
}

///////////////////////////////////////////////////////////////////////////////////
static void showSetWdsHelp(void)
{
#if 0
	printf("flash set WDS cmd\n");
	printf("cmd:\n");
	printf("      add mac-addr comment -- append a WDS mac address.\n");
	printf("      del entry-number -- delete a WDS entry.\n");
	printf("      delall -- delete all WDS mac address.\n");
#endif	
}

#ifdef HOME_GATEWAY
///////////////////////////////////////////////////////////////////////////////////
static void showSetPortFwHelp(void)
{
#if 0
	printf("flash set PORTFW_TBL cmd\n");
	printf("cmd:\n");
	printf("      add ip from-port to-port protocol comment -- add a filter.\n");
	printf("      del entry-number -- delete a filter.\n");
	printf("      delall -- delete all filter.\n");
#endif	
}


///////////////////////////////////////////////////////////////////////////////////
static void showSetPortFilterHelp(void)
{
#if 0
	printf("flash set PORTFILTER_TBL cmd\n");
	printf("cmd:\n");
	printf("      add from-port to-port protocol comment -- add a filter.\n");
	printf("      del entry-number -- delete a filter.\n");
	printf("      delall -- delete all filter.\n");
#endif	
}

///////////////////////////////////////////////////////////////////////////////////
static void showSetIpFilterHelp(void)
{
#if 0
	printf("flash set IPFILTER_TBL cmd\n");
	printf("cmd:\n");
	printf("      add ip protocol comment -- add a filter.\n");
	printf("      del entry-number -- delete a filter.\n");
	printf("      delall -- delete all filter.\n");
#endif	
}

///////////////////////////////////////////////////////////////////////////////////
static void showSetMacFilterHelp(void)
{
#if 0
	printf("flash set MACFILTER_TBL cmd\n");
	printf("cmd:\n");
	printf("      add mac-addr comment -- add a filter.\n");
	printf("      del entry-number -- delete a filter.\n");
	printf("      delall -- delete all filter.\n");
#endif	
}
///////////////////////////////////////////////////////////////////////////////////
static void showSetUrlFilterHelp(void)
{
#if 0
	printf("flash set URLFILTER_TBL cmd\n");
	printf("cmd:\n");
	printf("      add url-addr -- add a filter.\n");
	printf("      del entry-number -- delete a filter.\n");
	printf("      delall -- delete all filter.\n");
#endif	
}
///////////////////////////////////////////////////////////////////////////////////
static void showSetTriggerPortHelp(void)
{
#if 0
	printf("flash set TRIGGER_PORT cmd\n");
	printf("cmd:\n");
	printf("   add trigger-from trigger-to trigger-proto incoming-from incoming-to incoming-proto comment -- add a trigger-port.\n");
	printf("   del entry-number -- delete a trigger-port.\n");
	printf("   delall -- delete all trigger-port.\n");
#endif	
}
///////////////////////////////////////////////////////////////////////////////////
#ifdef ROUTE_SUPPORT
static void showSetStaticRouteHelp(void)
{
	printf("flash set STATICROUTE_TBL cmd\n");
	printf("cmd:\n");
	printf("   add dest_ip netmask gateway  -- add a static route.\n");
	printf("   del entry-number -- delete a static route.\n");
	printf("   delall -- delete all static route.\n");


}
#endif //ROUTE
#endif

#ifdef HOME_GATEWAY
#ifdef VPN_SUPPORT
static void  showSetIpsecTunnelHelp(void)
{
        printf("flash set IPSECTUNNEL_TBL cmd\n");
        printf("cmd:\n");
        printf("   add tunnel_id enable name local_type local_ip local_mask_len remote_type remote_ip remote_mask_len remote_gw keymode connectType espEncr espAuth psKey ike_encr ike_auth ike_keygroup ike_lifetime ipsec_lifetime ipsec_pfs spi encrKey authKey -- add a ipsec manual tunnel.\n");
        printf("   del entry-number -- delete a vpn tunnel.\n");
        printf("   delall -- delete all tunnel.\n");
}
#endif
#endif

#ifdef TLS_CLIENT
static void  showSetCertRootHelp(void)
{
        printf("flash set CERTROOT_TBL cmd\n");
        printf("cmd:\n");
        printf("   add comment.\n");
        printf("   del entry-number -- delete a certca .\n");
        printf("   delall -- delete all certca.\n");
}
static void  showSetCertUserHelp(void)
{
        printf("flash set CERTUSER_TBL cmd\n");
        printf("cmd:\n");
        printf("   add comment password.\n");
        printf("   del entry-number -- delete a certca .\n");
        printf("   delall -- delete all certca.\n");
}
#endif

#ifdef PARSE_TXT_FILE
////////////////////////////////////////////////////////////////////////////////
static int parseTxtConfig(char *filename, APMIB_Tp pConfig)
{
	char line[300], value[300];
	FILE *fp;
	int id;

	fp = fopen(filename, "r");
	if ( fp == NULL )
		return -1;

	acNum = 0;
	wdsNum = 0;

#ifdef HOME_GATEWAY
	portFilterNum = ipFilterNum = macFilterNum = portFwNum = staticRouteNum=0;
	urlFilterNum = 0;
#endif
#ifdef TLS_CLIENT
	certRootNum =  certUserNum = 0;
#endif

	while ( fgets(line, 100, fp) ) {
		id = getToken(line, value);
		if ( id == 0 )
			continue;
		if ( set_mib(pConfig, id, value) < 0) {
			printf("Parse MIB [%d] error!\n", id );
			fclose(fp);
			return -1;
		}
	}

	fclose(fp);
	return 0;
}

////////////////////////////////////////////////////////////////////////////////
static int getToken(char *line, char *value)
{
	char *ptr=line, *p1;
	char token[300]={0};
	int len=0, idx;

	if ( *ptr == ';' )	// comments
		return 0;

	// get token
	while (*ptr && *ptr!=EOL) {
		if ( *ptr == '=' ) {
			memcpy(token, line, len);

			// delete ending space
			for (idx=len-1; idx>=0; idx--) {
				if (token[idx]!= SPACE )
					break;
			}
			token[idx+1] = '\0';
			ptr++;
			break;
		}
		ptr++;
		len++;
	}
	if ( !token[0] )
		return 0;

	// get value
	len=0;
	while (*ptr == SPACE ) ptr++; // delete space

	p1 = ptr;
	while ( *ptr && *ptr!=EOL) {
		ptr++;
		len++;
	}
	memcpy(value, p1, len );
	value[len] = '\0';

	idx = 0;
	while (mib_table[idx].id) {
		if (!strcmp(mib_table[idx].name, token))
			return mib_table[idx].id;
		idx++;
	}
	return 0;
}


////////////////////////////////////////////////////////////////////////////////
static int set_mib(APMIB_Tp pMib, int id, void *value)
{
	unsigned char key[100];
	char *p1, *p2;
#ifdef HOME_GATEWAY
	char *p3, *p4, *p5;
#endif
	struct in_addr inAddr;
	int i;
	MACFILTER_Tp pWlAc;
	WDS_Tp pWds;

#ifdef HOME_GATEWAY
	PORTFW_Tp pPortFw;
	PORTFILTER_Tp pPortFilter;
	IPFILTER_Tp pIpFilter;
	MACFILTER_Tp pMacFilter;
	URLFILTER_Tp pUrlFilter;
#endif

#ifdef TLS_CLIENT
	CERTROOT_Tp pCertRoot;
	CERTUSER_Tp pCertUser;
#endif
	for (i=0; mib_table[i].id; i++) {
		if ( mib_table[i].id == id )
			break;
	}
	if ( mib_table[i].id == 0 )
		return -1;

	switch (mib_table[i].type) {
	case BYTE_T:
		*((unsigned char *)(((long)pMib) + mib_table[i].offset)) = (unsigned char)atoi(value);
		break;

	case WORD_T:
		*((unsigned short *)(((long)pMib) + mib_table[i].offset)) = (unsigned short)atoi(value);
		break;

	case STRING_T:
		if ( strlen(value)+1 > mib_table[i].size )
			return 0;
		strcpy((char *)(((long)pMib) + mib_table[i].offset), (char *)value);
		break;

	case BYTE5_T:
		if (strlen(value)!=10 || !string_to_hex(value, key, 10))
			return -1;
		memcpy((unsigned char *)(((long)pMib) + mib_table[i].offset), key, 5);
		break;

	case BYTE6_T:
		if (strlen(value)!=12 || !string_to_hex(value, key, 12))
			return -1;
		memcpy((unsigned char *)(((long)pMib) + mib_table[i].offset), key, 6);
		break;

	case BYTE13_T:
		if (strlen(value)!=26 || !string_to_hex(value, key, 26))
			return -1;
		memcpy((unsigned char *)(((long)pMib) + mib_table[i].offset), key, 13);
		break;
	
	case DWORD_T:
		*((unsigned long *)(((long)pMib) + mib_table[i].offset)) = (unsigned long)atoi(value);
		break;

	case IA_T:
		if ( !inet_aton(value, &inAddr) )
			return -1;
		memcpy((unsigned char *)(((long)pMib) + mib_table[i].offset), (unsigned char *)&inAddr,  4);
		break;

	case WLAC_ARRAY_T:
		getVal2((char *)value, &p1, &p2);
		if (p1 == NULL) {
			printf("Invalid WLAC in argument!\n");
			break;
		}
		if (strlen(p1)!=12 || !string_to_hex(p1, key, 12))
			return -1;

		pWlAc = (MACFILTER_Tp)(((long)pMib)+mib_table[i].offset+acNum*sizeof(MACFILTER_T));
		memcpy(pWlAc->macAddr, key, 6);
		if (p2 != NULL )
			strcpy(pWlAc->comment, p2);
		acNum++;
		break;

	case WDS_ARRAY_T:
		getVal2((char *)value, &p1, &p2);
		if (p1 == NULL) {
			printf("Invalid WDS in argument!\n");
			break;
		}
		if (strlen(p1)!=12 || !string_to_hex(p1, key, 12))
			return -1;

		pWds = (WDS_Tp)(((long)pMib)+mib_table[i].offset+wdsNum*sizeof(WDS_T));
		memcpy(pWds->macAddr, key, 6);
		if (p2 != NULL )
			strcpy(pWds->comment, p2);
		wdsNum++;
		break;

#ifdef HOME_GATEWAY
	case MACFILTER_ARRAY_T:
		getVal2((char *)value, &p1, &p2);
		if (p1 == NULL) {
			printf("Invalid MACFILTER in argument!\n");
			break;
		}
		if (strlen(p1)!=12 || !string_to_hex(p1, key, 12))
			return -1;

		pMacFilter = (MACFILTER_Tp)(((long)pMib)+mib_table[i].offset+macFilterNum*sizeof(MACFILTER_T));
		memcpy(pMacFilter->macAddr, key, 6);
		if (p2 != NULL )
			strcpy(pMacFilter->comment, p2);
		macFilterNum++;
		break;

	case URLFILTER_ARRAY_T:
		getVal2((char *)value, &p1, &p2);
		if (p1 == NULL) {
			printf("Invalid URLFILTER in argument!\n");
			break;
		}
		//if (strlen(p1)!=12 || !string_to_hex(p1, key, 12))
		//	return -1;

		pUrlFilter = (URLFILTER_Tp)(((long)pMib)+mib_table[i].offset+urlFilterNum*sizeof(URLFILTER_T));
		memcpy(pUrlFilter->urlAddr, key, 20);
		//if (p2 != NULL )
		//	strcpy(pMacFilter->comment, p2);
		urlFilterNum++;
		break;

	case PORTFW_ARRAY_T:
		getVal5((char *)value, &p1, &p2, &p3, &p4, &p5);
		if (p1 == NULL || p2 == NULL || p3 == NULL || p4 == NULL ) {
			printf("Invalid PORTFW arguments!\n");
			break;
		}
		if ( !inet_aton(p1, &inAddr) )
			return -1;

		pPortFw = (PORTFW_Tp)(((long)pMib)+mib_table[i].offset+portFwNum*sizeof(PORTFW_T));
		memcpy(pPortFw->ipAddr, (unsigned char *)&inAddr, 4);
		pPortFw->fromPort = (unsigned short)atoi(p2);
		pPortFw->toPort = (unsigned short)atoi(p3);
		pPortFw->protoType = (unsigned char)atoi(p4);
		if ( p5 )
			strcpy( pPortFw->comment, p5 );
		portFwNum++;
		break;

	case IPFILTER_ARRAY_T:
		getVal3((char *)value, &p1, &p2, &p3);
		if (p1 == NULL || p2 == NULL) {
			printf("Invalid IPFILTER arguments!\n");
			break;
		}
		if ( !inet_aton(p1, &inAddr) )
			return -1;
		pIpFilter = (IPFILTER_Tp)(((long)pMib)+mib_table[i].offset+ipFilterNum*sizeof(IPFILTER_T));
		memcpy(pIpFilter->ipAddr, (unsigned char *)&inAddr, 4);
		pIpFilter->protoType = (unsigned char)atoi(p2);
		if ( p3 )
			strcpy( pIpFilter->comment, p3 );
		ipFilterNum++;
		break;

	case PORTFILTER_ARRAY_T:
		getVal4((char *)value, &p1, &p2, &p3, &p4);
		if (p1 == NULL || p2 == NULL || p3 == NULL) {
			printf("Invalid PORTFILTER arguments!\n");
			break;
		}
		if ( !inet_aton(p1, &inAddr) )
			return -1;
		pPortFilter = (PORTFILTER_Tp)(((long)pMib)+mib_table[i].offset+portFilterNum*sizeof(PORTFILTER_T));
		pPortFilter->fromPort = (unsigned short)atoi(p1);
		pPortFilter->toPort = (unsigned short)atoi(p2);
		pPortFilter->protoType = (unsigned char)atoi(p3);
		if ( p4 )
			strcpy( pPortFilter->comment, p4 );
		portFilterNum++;
		break;
#ifdef ROUTE_SUPPORT
	case STATICROUTE_ARRAY_T:
		getVal3((char *)value, &p1, &p2, &p3);
		if (p1 == NULL || p2 == NULL || p3 == NULL) {
			printf("Invalid PORTFILTER arguments!\n");
			break;
		}
		if ( !inet_aton(p1, &inAddr) )
			return -1;
		pStaticRoute = (STATICROUTE_Tp)(((long)pMib)+mib_table[i].offset+staticRouteNum*sizeof(STATICROUTE_T));
		if( !inet_aton(p1, &pStaticRoute->destAddr))
			return -1 ;
		if( !inet_aton(p2, &pStaticRoute->netmask))
			return -1 ;
		if( !inet_aton(p3, &pStaticRoute->gateway))
			return -1 ;
		staticRouteNum++;
		break;
#endif // ROUTE_SUPPORT
#endif
#ifdef TLS_CLIENT
	case CERTROOT_ARRAY_T:
		getVal1((char *)value, &p1);
		if (p1 == NULL ) {
			printf("Invalid CERTCA arguments!\n");
			break;
		}
		pCertRoot = (CERTROOT_Tp)(((long)pMib)+mib_table[i].offset+certRootNum*sizeof(CERTROOT_T));
		strcpy( pCertRoot->comment, p1 );
		certRootNum++;
		break;
	case CERTUSER_ARRAY_T:
		getVal2((char *)value,&p1, &p2);
		if (p1 == NULL || p2 = NULL) {
			printf("Invalid CERTPR arguments!\n");
			break;
		}
		pCertUser = (CERTUSER_Tp)(((long)pMib)+mib_table[i].offset+certUserNum*sizeof(CERTUSER_T));
		strcpy( pCertUser->pass, p1 );
		strcpy( pCertUser->comment, p2 );
		certUserNum++;
		break;		
#endif

	default:
		return -1;
	}
	return 0;
}

////////////////////////////////////////////////////////////////////////////////
static void getVal2(char *value, char **p1, char **p2)
{
	value = getVal(value, p1);
	if ( value )
		getVal(value, p2);
	else
		*p2 = NULL;
}

#ifdef HOME_GATEWAY
////////////////////////////////////////////////////////////////////////////////
static void getVal3(char *value, char **p1, char **p2, char **p3)
{
	*p1 = *p2 = *p3 = NULL;

	value = getVal(value, p1);
	if ( !value )
		return;
	value = getVal(value, p2);
	if ( !value )
		return;
	getVal(value, p3);
}

////////////////////////////////////////////////////////////////////////////////
static void getVal4(char *value, char **p1, char **p2, char **p3, char **p4)
{
	*p1 = *p2 = *p3 = *p4 = NULL;

	value = getVal(value, p1);
	if ( !value )
		return;
	value = getVal(value, p2);
	if ( !value )
		return;
	value = getVal(value, p3);
	if ( !value )
		return;
	getVal(value, p4);
}

////////////////////////////////////////////////////////////////////////////////
static void getVal5(char *value, char **p1, char **p2, char **p3, char **p4, char **p5)
{
	*p1 = *p2 = *p3 = *p4 = *p5 = NULL;

	value = getVal(value, p1);
	if ( !value )
		return;
	value = getVal(value, p2);
	if ( !value )
		return;
	value = getVal(value, p3);
	if ( !value )
		return;
	value = getVal(value, p4);
	if ( !value )
		return;
	getVal(value, p5);
}
#endif // HOME_GATEWAY

#endif // PARSE_TXT_FILE

#ifndef COMPACK_SIZE
////////////////////////////////////////////////////////////////////////////////
static int getdir(char *fullname, char *path, int loop)
{
	char tmpBuf[100], *p, *p1;

	strcpy(tmpBuf, fullname);
	path[0] = '\0';

	p1 = tmpBuf;
	while (1) {
		if ((p=strchr(p1, '/'))) {
			if (--loop == 0) {
				*p = '\0';
				strcpy(path, tmpBuf);
				return 0;
			}
			p1 = ++p;
		}
		else
			break;
	}
	return -1;
}

////////////////////////////////////////////////////////////////////////////////
static int read_flash_webpage(char *prefix, char *webfile)
{
	WEB_HEADER_T header;
	char *buf, tmpFile[100], tmpFile1[100], tmpBuf[100];
	int fh=0, i, loop, size;
	FILE_ENTRY_Tp pEntry;
	struct stat sbuf;
	char *file;

	if (webfile[0])
		file = webfile;
	else
		file = NULL;

	if (!file) {
		if ( flash_read((char *)&header, WEB_PAGE_OFFSET, sizeof(header)) == 0) {
			printf("Read web header failed!\n");
			return -1;
		}
	}
	else {
		if ((fh = open(file, O_RDONLY)) < 0) {
			printf("Can't open file %s\n", file);
			return -1;
		}
		lseek(fh, 0L, SEEK_SET);
		if (read(fh, &header, sizeof(header)) != sizeof(header)) {
			printf("Read web header failed %s!\n", file);
			close(fh);
			return -1;
		}
	}
#ifndef __mips__
	header.len = DWORD_SWAP(header.len);
#endif

	if (memcmp(header.signature, WEB_HEADER, SIGNATURE_LEN)) {
		printf("Invalid web image!\n");
		return -1;
	}

// for debug
//printf("web size=%ld\n", header.len);
	buf = malloc(header.len);
	if (buf == NULL) {
		sprintf(tmpBuf, "Allocate buffer failed %ld!\n", header.len);
		printf(tmpBuf);
		return -1;
	}

	if (!file) {
		if ( flash_read(buf, WEB_PAGE_OFFSET+sizeof(header), header.len) == 0) {
			printf("Read web image failed!\n");
			return -1;
		}
	}
	else {
		if (read(fh, buf, header.len) != header.len) {
			printf("Read web image failed!\n");
			return -1;
		}
		close(fh);
	}

	if ( !CHECKSUM_OK(buf, header.len) ) {
		printf("Web image invalid!\n");
		free(buf);
		return -1;
	}
// for debug
//printf("checksum ok!\n");

	// save to a file
	strcpy(tmpFile, "flashweb.bz2");
	fh = open(tmpFile, O_RDWR|O_CREAT|O_TRUNC);
	if (fh == -1) {
		printf("Create output file error %s!\n", tmpFile );
		return -1;
	}
	if ( write(fh, buf, header.len-1) != header.len -1) {
		printf("write file error %s!\n", tmpFile);
		return -1;
	}
	close(fh);
	free(buf);
	sync();

	// decompress file
	sprintf(tmpFile1, "%sXXXXXX", tmpFile);
	mkstemp(tmpFile1);

	sprintf(tmpBuf, "bunzip2 -c %s > %s", tmpFile, tmpFile1);
	system(tmpBuf);

	unlink(tmpFile);
	sync();

	if (stat(tmpFile1, &sbuf) != 0) {
		printf("Stat file error %s!\n", tmpFile1);
		return -1;
	}
	if (sbuf.st_size < sizeof(FILE_ENTRY_T) ) {
		sprintf(tmpBuf, "Invalid decompress file size %ld!\n", sbuf.st_size);
		printf(tmpBuf);
		unlink(tmpFile1);
		return -1;
	}
// for debug
//printf("decompress size=%ld\n", sbuf.st_size);

	buf = malloc(sbuf.st_size);
	if (buf == NULL) {
		sprintf(tmpBuf,"Allocate buffer failed %ld!\n", sbuf.st_size);
		printf(tmpBuf);
		return -1;
	}
	if ((fh = open(tmpFile1, O_RDONLY)) < 0) {
		printf("Can't open file %s\n", tmpFile1);
		return -1;
	}
	lseek(fh, 0L, SEEK_SET);
	if ( read(fh, buf, sbuf.st_size) != sbuf.st_size) {
		printf("Read file error %ld!\n", sbuf.st_size);
		return -1;
	}
	close(fh);
	unlink(tmpFile1);
	sync();
	size = sbuf.st_size;
	for (i=0; i<size; ) {
		pEntry = (FILE_ENTRY_Tp)&buf[i];

#ifndef __mips__
		pEntry->size = DWORD_SWAP(pEntry->size);
#endif

		strcpy(tmpFile, prefix);
		strcat(tmpFile, "/");
		strcat(tmpFile, pEntry->name);

		loop = 0;
		while (1) {
			if (getdir(tmpFile, tmpBuf, ++loop) < 0)
				break;
			if (tmpBuf[0] && stat(tmpBuf, &sbuf) < 0) { // not exist
 				if ( mkdir(tmpBuf, S_IREAD|S_IWRITE) < 0) {
					printf("Create directory %s failed!\n", tmpBuf);
					return -1;
				}
			}
		}
// for debug
//printf("write file %s, size=%ld\n", tmpFile, pEntry->size);

		fh = open(tmpFile, O_RDWR|O_CREAT|O_TRUNC);
		if (fh == -1) {
			printf("Create output file error %s!\n", tmpFile );
			return -1;
		}
// for debug
//if ( (i+sizeof(FILE_ENTRY_T)+pEntry->size) > size ) {
//printf("error in size, %ld !\n", pEntry->size);
//}

		if ( write(fh, &buf[i+sizeof(FILE_ENTRY_T)], pEntry->size) != pEntry->size ) {
			printf("Write file error %s, len=%ld!\n", tmpFile, pEntry->size);
			return -1;
		}
		close(fh);
		// always set execuatble for script file
//		chmod(tmpFile,  S_IXUSR);

		i += (pEntry->size + sizeof(FILE_ENTRY_T));
	}

	return 0;
}
#endif // COMPACK_SIZE
#ifdef VPN_SUPPORT
static int read_flash_rsa(char *outputFile)
{
	int fh;
	char *rsaBuf;

	if ( !apmib_init()) {
		printf("Initialize AP MIB failed!\n");
		return -1;
	}

	fh = open(outputFile,  O_RDWR|O_CREAT);
	if (fh == -1) {
		printf("Create WPA config file error!\n");
		return -1;
	}
	rsaBuf = malloc(sizeof(unsigned char) * MAX_RSA_FILE_LEN);
	apmib_get( MIB_IPSEC_RSA_FILE, (void *)rsaBuf);
	write(fh, rsaBuf, MAX_RSA_FILE_LEN);
	close(fh);
	free(rsaBuf);
	chmod(outputFile,  DEFFILEMODE);
	return 0;
}
#endif
#ifdef TLS_CLIENT
////////////////////////////////////////////////////////////////////////////////
static int read_flash_cert(char *prefix, char *certfile)
{
	CERT_HEADER_T header;
	char *buf, tmpFile[100], tmpFile1[100], tmpBuf[100];
	int fh=0, i, loop, size;
	FILE_ENTRY_Tp pEntry;
	struct stat sbuf;
	char *file;

	if (certfile[0])
		file = certfile;
	else
		file = NULL;

	if (!file) {
		if ( flash_read((char *)&header, CERT_PAGE_OFFSET, sizeof(header)) == 0) {
			printf("Read web header failed!\n");
			return -1;
		}
	}
	else {
		if ((fh = open(file, O_RDONLY)) < 0) {
			printf("Can't open file %s\n", file);
			return -1;
		}
		lseek(fh, 0L, SEEK_SET);
		if (read(fh, &header, sizeof(header)) != sizeof(header)) {
			printf("Read web header failed %s!\n", file);
			close(fh);
			return -1;
		}
	}
#ifndef __mips__
	header.len = DWORD_SWAP(header.len);
#endif
	
	if (memcmp(header.signature, CERT_HEADER, SIGNATURE_LEN)) {
		printf("Invalid cert image!\n");
		return -1;
	}

// for debug
//printf("web size=%ld\n", header.len);
	buf = malloc(header.len);
	if (buf == NULL) {
		sprintf(tmpBuf, "Allocate buffer failed %ld!\n", header.len);
		printf(tmpBuf);
		return -1;
	}

	if (!file) {
		if ( flash_read(buf, CERT_PAGE_OFFSET+sizeof(header), header.len) == 0) {
			printf("Read web image failed!\n");
			return -1;
		}
	}
	else {
		if (read(fh, buf, header.len) != header.len) {
			printf("Read web image failed!\n");
			return -1;
		}
		close(fh);
	}

	if ( !CHECKSUM_OK(buf, header.len) ) {
		printf("Web image invalid!\n");
		free(buf);
		return -1;
	}
// for debug
//printf("checksum ok!\n");

	// save to a file
	strcpy(tmpFile, "/tmp/cert.tmp");
	fh = open(tmpFile, O_RDWR|O_CREAT|O_TRUNC);
	if (fh == -1) {
		printf("Create output file error %s!\n", tmpFile );
		return -1;
	}
	if ( write(fh, buf, header.len-1) != header.len -1) {
		printf("write file error %s!\n", tmpFile);
		return -1;
	}
	close(fh);
	free(buf);
	sync();

	// decompress file
	sprintf(tmpFile1, "%sXXXXXX", tmpFile);
	mkstemp(tmpFile1);

	//sprintf(tmpBuf, "bunzip2 -c %s > %s", tmpFile, tmpFile1);
	sprintf(tmpBuf, "cat %s  > %s", tmpFile, tmpFile1);
	system(tmpBuf);

	unlink(tmpFile);
	sync();

	if (stat(tmpFile1, &sbuf) != 0) {
		printf("Stat file error %s!\n", tmpFile1);
		return -1;
	}
	if (sbuf.st_size < sizeof(FILE_ENTRY_T) ) {
		sprintf(tmpBuf, "Invalid decompress file size %ld!\n", sbuf.st_size);
		printf(tmpBuf);
		unlink(tmpFile1);
		return -1;
	}
// for debug
//printf("decompress size=%ld\n", sbuf.st_size);

	buf = malloc(sbuf.st_size);
	if (buf == NULL) {
		sprintf(tmpBuf,"Allocate buffer failed %ld!\n", sbuf.st_size);
		printf(tmpBuf);
		return -1;
	}
	if ((fh = open(tmpFile1, O_RDONLY)) < 0) {
		printf("Can't open file %s\n", tmpFile1);
		return -1;
	}
	lseek(fh, 0L, SEEK_SET);
	if ( read(fh, buf, sbuf.st_size) != sbuf.st_size) {
		printf("Read file error %ld!\n", sbuf.st_size);
		return -1;
	}
	close(fh);
	unlink(tmpFile1);
	sync();
	size = sbuf.st_size;
	for (i=0; i<size; ) {
		pEntry = (FILE_ENTRY_Tp)&buf[i];

#ifndef __mips__
		pEntry->size = DWORD_SWAP(pEntry->size);
#endif

		strcpy(tmpFile, prefix);
		strcat(tmpFile, "/");
		strcat(tmpFile, pEntry->name);
		if(!strcmp(pEntry->name , ""))
			break;
		//printf("name = %s\n", pEntry->name);	
		loop = 0;
		while (1) {
			if (getdir(tmpFile, tmpBuf, ++loop) < 0)
				break;
			if (tmpBuf[0] && stat(tmpBuf, &sbuf) < 0) { // not exist
 				if ( mkdir(tmpBuf, S_IREAD|S_IWRITE) < 0) {
					printf("Create directory %s failed!\n", tmpBuf);
					return -1;
				}
			}
		}
// for debug
//printf("write file %s, size=%ld\n", tmpFile, pEntry->size);

		fh = open(tmpFile, O_RDWR|O_CREAT|O_TRUNC);
		if (fh == -1) {
			printf("Create output file error %s!\n", tmpFile );
			return -1;
		}
// for debug
//if ( (i+sizeof(FILE_ENTRY_T)+pEntry->size) > size ) {
//printf("error in size, %ld !\n", pEntry->size);
//}

		if ( write(fh, &buf[i+sizeof(FILE_ENTRY_T)], pEntry->size) != pEntry->size ) {
			printf("Write file error %s, len=%ld!\n", tmpFile, pEntry->size);
			return -1;
		}
		close(fh);
		// always set execuatble for script file
//		chmod(tmpFile,  S_IXUSR);

		i += (pEntry->size + sizeof(FILE_ENTRY_T));
	}

	return 0;
}
#endif
////////////////////////////////////////////////////////////////////////////////
static void __inline__ WRITE_WPA_FILE(int fh, unsigned char *buf)
{
	if ( write(fh, buf, strlen(buf)) != strlen(buf) ) {
		printf("Write WPA config file error!\n");
		close(fh);
		exit(1);
	}
}

////////////////////////////////////////////////////////////////////////////////
static void generateWpaConf(char *outputFile, int isWds)
{
	int fh, intVal, encrypt, enable1x, wep;
	unsigned char buf1[1024], buf2[1024];

#ifdef UNIVERSAL_REPEATER	
	int isVxd = 0;
	
	if (strstr(outputFile, "-vxd")) 
		isVxd = 1;	
#endif		
	
	if ( !apmib_init()) {
		printf("Initialize AP MIB failed!\n");
		return;
	}

	fh = open(outputFile, O_RDWR|O_CREAT|O_TRUNC);
	if (fh == -1) {
		printf("Create WPA config file error!\n");
		return;
	}
	if (!isWds) {
	apmib_get( MIB_WLAN_ENCRYPT, (void *)&encrypt);

#ifdef UNIVERSAL_REPEATER
	if (isVxd && (encrypt == ENCRYPT_WPA2_MIXED)) {
		apmib_get( MIB_WLAN_MODE, (void *)&intVal);
		if (intVal == AP_MODE || intVal == AP_WDS_MODE) 
			encrypt = ENCRYPT_WPA;		
	}
#endif			
	
	sprintf(buf2, "encryption = %d\n", encrypt);
	WRITE_WPA_FILE(fh, buf2);

#ifdef UNIVERSAL_REPEATER
	if (isVxd) {
		if (strstr(outputFile, "wlan0-vxd"))
			apmib_get( MIB_REPEATER_SSID1, (void *)buf1);		
		else			
			apmib_get( MIB_REPEATER_SSID2, (void *)buf1);	
	}
	else
#endif
	apmib_get( MIB_WLAN_SSID,  (void *)buf1);
	sprintf(buf2, "ssid = \"%s\"\n", buf1);
	WRITE_WPA_FILE(fh, buf2);

	apmib_get( MIB_WLAN_ENABLE_1X, (void *)&enable1x);
	sprintf(buf2, "enable1x = %d\n", enable1x);
	WRITE_WPA_FILE(fh, buf2);

	apmib_get( MIB_WLAN_ENABLE_MAC_AUTH, (void *)&intVal);
	sprintf(buf2, "enableMacAuth = %d\n", intVal);
	WRITE_WPA_FILE(fh, buf2);

	apmib_get( MIB_WLAN_ENABLE_SUPP_NONWPA, (void *)&intVal);
	if (intVal)
		apmib_get( MIB_WLAN_SUPP_NONWPA, (void *)&intVal);

	sprintf(buf2, "supportNonWpaClient = %d\n", intVal);
	WRITE_WPA_FILE(fh, buf2);

	apmib_get( MIB_WLAN_WEP, (void *)&wep);
	sprintf(buf2, "wepKey = %d\n", wep);
	WRITE_WPA_FILE(fh, buf2);

	if ( encrypt==1 && enable1x ) {
		if (wep == 1) {
			apmib_get( MIB_WLAN_WEP64_KEY1, (void *)buf1);
			sprintf(buf2, "wepGroupKey = \"%02x%02x%02x%02x%02x\"\n", buf1[0],buf1[1],buf1[2],buf1[3],buf1[4]);
		}
		else {
			apmib_get( MIB_WLAN_WEP128_KEY1, (void *)buf1);
			sprintf(buf2, "wepGroupKey = \"%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x\"\n",
				buf1[0],buf1[1],buf1[2],buf1[3],buf1[4],
				buf1[5],buf1[6],buf1[7],buf1[8],buf1[9],
				buf1[10],buf1[11],buf1[12]);
		}
	}
	else
		strcpy(buf2, "wepGroupKey = \"\"\n");
	WRITE_WPA_FILE(fh, buf2);

	apmib_get( MIB_WLAN_WPA_AUTH, (void *)&intVal);
	sprintf(buf2, "authentication = %d\n", intVal);
	WRITE_WPA_FILE(fh, buf2);

	apmib_get( MIB_WLAN_WPA_CIPHER_SUITE, (void *)&intVal);
	sprintf(buf2, "unicastCipher = %d\n", intVal);
	WRITE_WPA_FILE(fh, buf2);

	apmib_get( MIB_WLAN_WPA2_CIPHER_SUITE, (void *)&intVal);
	sprintf(buf2, "wpa2UnicastCipher = %d\n", intVal);
	WRITE_WPA_FILE(fh, buf2);

	apmib_get( MIB_WLAN_WPA2_PRE_AUTH, (void *)&intVal);
	sprintf(buf2, "enablePreAuth = %d\n", intVal);
	WRITE_WPA_FILE(fh, buf2);

	apmib_get( MIB_WLAN_WPA_PSK_FORMAT, (void *)&intVal);
	if (intVal==0)
		sprintf(buf2, "usePassphrase = 1\n");
	else
		sprintf(buf2, "usePassphrase = 0\n");
	WRITE_WPA_FILE(fh, buf2);

	apmib_get( MIB_WLAN_WPA_PSK, (void *)buf1);
	sprintf(buf2, "psk = \"%s\"\n", buf1);
	WRITE_WPA_FILE(fh, buf2);

	apmib_get( MIB_WLAN_WPA_GROUP_REKEY_TIME, (void *)&intVal);
	sprintf(buf2, "groupRekeyTime = %d\n", intVal);
	WRITE_WPA_FILE(fh, buf2);

	apmib_get( MIB_WLAN_RS_PORT, (void *)&intVal);
	sprintf(buf2, "rsPort = %d\n", intVal);
	WRITE_WPA_FILE(fh, buf2);

	apmib_get( MIB_WLAN_RS_IP, (void *)buf1);
	sprintf(buf2, "rsIP = %s\n", inet_ntoa(*((struct in_addr *)buf1)));
	WRITE_WPA_FILE(fh, buf2);

	apmib_get( MIB_WLAN_RS_PASSWORD, (void *)buf1);
	sprintf(buf2, "rsPassword = \"%s\"\n", buf1);
	WRITE_WPA_FILE(fh, buf2);

	apmib_get( MIB_WLAN_RS_RETRY, (void *)&intVal);
	sprintf(buf2, "rsMaxReq = %d\n", intVal);
	WRITE_WPA_FILE(fh, buf2);

	apmib_get( MIB_WLAN_RS_INTERVAL_TIME, (void *)&intVal);
	sprintf(buf2, "rsAWhile = %d\n", intVal);
	WRITE_WPA_FILE(fh, buf2);

	apmib_get( MIB_WLAN_ACCOUNT_RS_ENABLED, (void *)&intVal);
	sprintf(buf2, "accountRsEnabled = %d\n", intVal);
	WRITE_WPA_FILE(fh, buf2);

	apmib_get( MIB_WLAN_ACCOUNT_RS_PORT, (void *)&intVal);
	sprintf(buf2, "accountRsPort = %d\n", intVal);
	WRITE_WPA_FILE(fh, buf2);

	apmib_get( MIB_WLAN_ACCOUNT_RS_IP, (void *)buf1);
	sprintf(buf2, "accountRsIP = %s\n", inet_ntoa(*((struct in_addr *)buf1)));
	WRITE_WPA_FILE(fh, buf2);

	apmib_get( MIB_WLAN_ACCOUNT_RS_PASSWORD, (void *)buf1);
	sprintf(buf2, "accountRsPassword = \"%s\"\n", buf1);
	WRITE_WPA_FILE(fh, buf2);

	apmib_get( MIB_WLAN_ACCOUNT_UPDATE_ENABLED, (void *)&intVal);
	sprintf(buf2, "accountRsUpdateEnabled = %d\n", intVal);
	WRITE_WPA_FILE(fh, buf2);

	apmib_get( MIB_WLAN_ACCOUNT_UPDATE_DELAY, (void *)&intVal);
	sprintf(buf2, "accountRsUpdateTime = %d\n", intVal);
	WRITE_WPA_FILE(fh, buf2);

	apmib_get( MIB_WLAN_ACCOUNT_RS_RETRY, (void *)&intVal);
	sprintf(buf2, "accountRsMaxReq = %d\n", intVal);
	WRITE_WPA_FILE(fh, buf2);

	apmib_get( MIB_WLAN_ACCOUNT_RS_INTERVAL_TIME, (void *)&intVal);
	sprintf(buf2, "accountRsAWhile = %d\n", intVal);
	WRITE_WPA_FILE(fh, buf2);
	}

	else {
		apmib_get( MIB_WLAN_WDS_ENCRYPT, (void *)&encrypt);
		if (encrypt == WDS_ENCRYPT_TKIP)		
			encrypt = ENCRYPT_WPA;
		else if (encrypt == WDS_ENCRYPT_AES)		
			encrypt = ENCRYPT_WPA2;		
		else
			encrypt = 0;
	
		sprintf(buf2, "encryption = %d\n", encrypt);
		WRITE_WPA_FILE(fh, buf2);
		WRITE_WPA_FILE(fh, "ssid = \"REALTEK\"\n");
		WRITE_WPA_FILE(fh, "enable1x = 1\n");
		WRITE_WPA_FILE(fh, "enableMacAuth = 0\n");
		WRITE_WPA_FILE(fh, "supportNonWpaClient = 0\n");
		WRITE_WPA_FILE(fh, "wepKey = 0\n");
		WRITE_WPA_FILE(fh,  "wepGroupKey = \"\"\n");
		WRITE_WPA_FILE(fh,  "authentication = 2\n");

		if (encrypt == ENCRYPT_WPA)
			intVal = WPA_CIPHER_TKIP;
		else
			intVal = WPA_CIPHER_AES;
			
		sprintf(buf2, "unicastCipher = %d\n", intVal);
		WRITE_WPA_FILE(fh, buf2);

		sprintf(buf2, "wpa2UnicastCipher = %d\n", intVal);
		WRITE_WPA_FILE(fh, buf2);

		WRITE_WPA_FILE(fh, "enablePreAuth = 0\n");

		apmib_get( MIB_WLAN_WDS_PSK_FORMAT, (void *)&intVal);
		if (intVal==0)
			sprintf(buf2, "usePassphrase = 1\n");
		else
			sprintf(buf2, "usePassphrase = 0\n");
		WRITE_WPA_FILE(fh, buf2);

		apmib_get( MIB_WLAN_WDS_PSK, (void *)buf1);
		sprintf(buf2, "psk = \"%s\"\n", buf1);
		WRITE_WPA_FILE(fh, buf2);

		WRITE_WPA_FILE(fh, "groupRekeyTime = 0\n");
		WRITE_WPA_FILE(fh, "rsPort = 1812\n");
		WRITE_WPA_FILE(fh, "rsIP = 192.168.1.1\n");
		WRITE_WPA_FILE(fh, "rsPassword = \"\"\n");
		WRITE_WPA_FILE(fh, "rsMaxReq = 3\n");
		WRITE_WPA_FILE(fh, "rsAWhile = 10\n");
		WRITE_WPA_FILE(fh, "accountRsEnabled = 0\n");
		WRITE_WPA_FILE(fh, "accountRsPort = 1813\n");
		WRITE_WPA_FILE(fh, "accountRsIP = 192.168.1.1\n");
		WRITE_WPA_FILE(fh, "accountRsPassword = \"\"\n");
		WRITE_WPA_FILE(fh, "accountRsUpdateEnabled = 0\n");
		WRITE_WPA_FILE(fh, "accountRsUpdateTime = 1000\n");
		WRITE_WPA_FILE(fh, "accountRsMaxReq = 3\n");
		WRITE_WPA_FILE(fh, "accountRsAWhile = 1\n");
	}

	close(fh);
}

////////////////////////////////////////////////////////////////////////////////
#ifdef WLAN_FAST_INIT

#include <sys/socket.h>
#include <sys/ioctl.h>
#include <linux/wireless.h>
#include "../../../linux-2.4.18/drivers/net/wireless_ag/ieee802_mib.h"

static int initWlan(char *ifname)
{
	struct wifi_mib *pmib;
	int i, intVal, intVal2, encrypt, enable1x, wep, mode, enable1xVxd;
	unsigned char buf1[1024], buf2[1024], mac[6];
	int skfd;
	struct iwreq wrq;
	MACFILTER_T *pAcl;

	skfd = socket(AF_INET, SOCK_DGRAM, 0);
	strncpy(wrq.ifr_name, ifname, IFNAMSIZ);
	if (ioctl(skfd, SIOCGIWNAME, &wrq) < 0) {
		printf("Interface open failed!\n");
		return -1;
	}

	if ((pmib = (struct wifi_mib *)malloc(sizeof(struct wifi_mib))) == NULL) {
		printf("MIB buffer allocation failed!\n");
		return -1;
	}

	if (!apmib_init()) {
		printf("Initialize AP MIB failed!\n");
		return -1;
	}

	// Disable WLAN MAC driver and shutdown interface first
	sprintf(buf1, "ifconfig %s down", ifname);
	system(buf1);

	// shutdown all WDS interface
	for (i=0; i<8; i++) {
		sprintf(buf1, "ifconfig %s-wds%d down", ifname, i);
		system(buf1);
	}

	// kill wlan application daemon
	sprintf(buf1, "wlanapp.sh kill %s", ifname);
	system(buf1);

	// get mib from driver
	wrq.u.data.pointer = (caddr_t)pmib;
	wrq.u.data.length = sizeof(struct wifi_mib);
	if (ioctl(skfd, 0x8B42, &wrq) < 0) {
		printf("Get WLAN MIB failed!\n");
		return -1;
	}

	// check mib version
	if (pmib->mib_version != MIB_VERSION) {
		printf("WLAN MIB version mismatch!\n");
		return -1;
	}

	apmib_get(MIB_HW_RF_TYPE, (void *)&intVal);
	if (intVal == 0) {
		printf("RF type is NULL!\n");
		return 0;
	}

	apmib_get(MIB_WLAN_DISABLED, (void *)&intVal);
	if (intVal == 1)
		return 0;

	// Set parameters to driver
	apmib_get(MIB_HW_REG_DOMAIN, (void *)&intVal);
	pmib->dot11StationConfigEntry.dot11RegDomain = intVal;

	apmib_get(MIB_WLAN_MAC_ADDR, (void *)mac);
	if (!memcmp(mac, "\x00\x00\x00\x00\x00\x00", 6)) {
		apmib_get(MIB_HW_WLAN_ADDR, (void *)mac);
	}

	// ifconfig all wlan interface when not in WISP
	// ifconfig wlan1 later interface when in WISP mode, the wlan0  will be setup in WAN interface
	apmib_get(MIB_OP_MODE, (void *)&intVal);
	apmib_get(MIB_WISP_WAN_ID, (void *)&intVal2);
	sprintf(buf1, "wlan%d", intVal2);
	if ((intVal != 2) ||
		strcmp(ifname, buf1)) {
		sprintf(buf2, "ifconfig %s hw ether %02x%02x%02x%02x%02x%02x", ifname, mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);
		system(buf2);
		memcpy(&(pmib->dot11OperationEntry.hwaddr[0]), mac, 6);
	}

	apmib_get(MIB_HW_LED_TYPE, (void *)&intVal);
	pmib->dot11OperationEntry.ledtype = intVal;

	// set AP/client/WDS mode
	apmib_get(MIB_WLAN_SSID, (void *)buf1);
	intVal2 = strlen(buf1);
	pmib->dot11StationConfigEntry.dot11DesiredSSIDLen = intVal2;
	memset(pmib->dot11StationConfigEntry.dot11DesiredSSID, 0, 32);
	memcpy(pmib->dot11StationConfigEntry.dot11DesiredSSID, buf1, intVal2);
	if ((pmib->dot11StationConfigEntry.dot11DesiredSSIDLen == 3) &&
		((pmib->dot11StationConfigEntry.dot11DesiredSSID[0] == 'A') || (pmib->dot11StationConfigEntry.dot11DesiredSSID[0] == 'a')) &&
		((pmib->dot11StationConfigEntry.dot11DesiredSSID[1] == 'N') || (pmib->dot11StationConfigEntry.dot11DesiredSSID[1] == 'n')) &&
		((pmib->dot11StationConfigEntry.dot11DesiredSSID[2] == 'Y') || (pmib->dot11StationConfigEntry.dot11DesiredSSID[2] == 'y'))) {
		pmib->dot11StationConfigEntry.dot11SSIDtoScanLen = 0;
		memset(pmib->dot11StationConfigEntry.dot11SSIDtoScan, 0, 32);
	}
	else {
		pmib->dot11StationConfigEntry.dot11SSIDtoScanLen = intVal2;
		memset(pmib->dot11StationConfigEntry.dot11SSIDtoScan, 0, 32);
		memcpy(pmib->dot11StationConfigEntry.dot11SSIDtoScan, buf1, intVal2);
	}

	apmib_get(MIB_WLAN_MODE, (void *)&mode);
	if (mode == 1) {
		// client mode
		apmib_get(MIB_WLAN_NETWORK_TYPE, (void *)&intVal2);
		if (intVal2 == 0)
			pmib->dot11OperationEntry.opmode = 8;
		else {
			pmib->dot11OperationEntry.opmode = 32;
			apmib_get(MIB_WLAN_DEFAULT_SSID, (void *)buf1);
			intVal2 = strlen(buf1);
			pmib->dot11StationConfigEntry.dot11DefaultSSIDLen = intVal2;
			memset(pmib->dot11StationConfigEntry.dot11DefaultSSID, 0, 32);
			memcpy(pmib->dot11StationConfigEntry.dot11DefaultSSID, buf1, intVal2);
		}
	}
	else
		pmib->dot11OperationEntry.opmode = 16;

	if (mode == 2)	// WDS only
		pmib->dot11WdsInfo.wdsPure = 1;
	else
		pmib->dot11WdsInfo.wdsPure = 0;

	// set RF parameters
	apmib_get(MIB_HW_RF_TYPE, (void *)&intVal);
	pmib->dot11RFEntry.dot11RFType = intVal;

	apmib_get(MIB_HW_ANT_DIVERSITY, (void *)&intVal);
	pmib->dot11RFEntry.dot11DiversitySupport = intVal;

	apmib_get(MIB_HW_TX_ANT, (void *)&intVal);
	pmib->dot11RFEntry.defaultAntennaB = intVal;

	apmib_get(MIB_HW_INIT_GAIN, (void *)&intVal);
	pmib->dot11RFEntry.initialGain = intVal;

	// skip CCA

	apmib_get(MIB_HW_TX_POWER_CCK, (void *)buf1);
	memcpy(pmib->dot11RFEntry.pwrlevelCCK, buf1, 14);
	
	apmib_get(MIB_HW_TX_POWER_OFDM, (void *)buf1);
	memcpy(pmib->dot11RFEntry.pwrlevelOFDM, buf1, 162);

	// set output power scale
	if (pmib->dot11RFEntry.dot11RFType == 7) { // Zebra
		apmib_get(MIB_WLAN_RF_POWER, (void *)&intVal);
		intVal = intVal*3;
		if (intVal) {
			for (i=0; i<14; i++) {
				if (pmib->dot11RFEntry.pwrlevelCCK[i] >= intVal)
					pmib->dot11RFEntry.pwrlevelCCK[i] -= intVal;
				else
					pmib->dot11RFEntry.pwrlevelCCK[i] = 0;
			}
			for (i=0; i<162; i++) {
				if (pmib->dot11RFEntry.pwrlevelOFDM[i] >= intVal)
					pmib->dot11RFEntry.pwrlevelOFDM[i] -= intVal;
				else
					pmib->dot11RFEntry.pwrlevelOFDM[i] = 0;
			}		
		}
	}
	
	apmib_get(MIB_WLAN_BEACON_INTERVAL, (void *)&intVal);
	pmib->dot11StationConfigEntry.dot11BeaconPeriod = intVal;

	apmib_get(MIB_WLAN_CHAN_NUM, (void *)&intVal);
	pmib->dot11RFEntry.dot11channel = intVal;

	apmib_get(MIB_WLAN_BASIC_RATE, (void *)&intVal);
	pmib->dot11StationConfigEntry.dot11BasicRates = intVal;

	apmib_get(MIB_WLAN_SUPPORTED_RATE, (void *)&intVal);
	pmib->dot11StationConfigEntry.dot11SupportedRates = intVal;

	apmib_get(MIB_WLAN_RATE_ADAPTIVE_ENABLED, (void *)&intVal);
	if (intVal == 0) {
		pmib->dot11StationConfigEntry.autoRate = 0;
		apmib_get(MIB_WLAN_FIX_RATE, (void *)&intVal);
		pmib->dot11StationConfigEntry.fixedTxRate = intVal;
	}
	else
		pmib->dot11StationConfigEntry.autoRate = 1;

	apmib_get(MIB_WLAN_RTS_THRESHOLD, (void *)&intVal);
	pmib->dot11OperationEntry.dot11RTSThreshold = intVal;

	apmib_get(MIB_WLAN_FRAG_THRESHOLD, (void *)&intVal);
	pmib->dot11OperationEntry.dot11FragmentationThreshold = intVal;

	apmib_get(MIB_WLAN_INACTIVITY_TIME, (void *)&intVal);
	pmib->dot11OperationEntry.expiretime = intVal;

	apmib_get(MIB_WLAN_PREAMBLE_TYPE, (void *)&intVal);
	pmib->dot11RFEntry.shortpreamble = intVal;

	apmib_get(MIB_WLAN_HIDDEN_SSID, (void *)&intVal);
	pmib->dot11OperationEntry.hiddenAP = intVal;

	apmib_get(MIB_WLAN_DTIM_PERIOD, (void *)&intVal);
	pmib->dot11StationConfigEntry.dot11DTIMPeriod = intVal;

	pmib->dot11OperationEntry.dot11LongRetryLimit = 6;
	pmib->dot11OperationEntry.dot11ShortRetryLimit = 6;

	pmib->dot11StationConfigEntry.dot11AclNum = 0;
	apmib_get(MIB_WLAN_AC_ENABLED, (void *)&intVal);
	pmib->dot11StationConfigEntry.dot11AclMode = intVal;
	if (intVal != 0) {
		apmib_get(MIB_WLAN_AC_NUM, (void *)&intVal);
		if (intVal != 0) {
			for (i=0; i<intVal; i++) {
				buf1[0] = i+1;
				apmib_get(MIB_WLAN_AC_ADDR, (void *)buf1);
				pAcl = (MACFILTER_T *)buf1;
				memcpy(&(pmib->dot11StationConfigEntry.dot11AclAddr[i][0]), &(pAcl->macAddr[0]), 6);
				pmib->dot11StationConfigEntry.dot11AclNum++;
			}
		}
	}

	apmib_get(MIB_WLAN_AUTH_TYPE, (void *)&intVal);
	apmib_get(MIB_WLAN_ENCRYPT, (void *)&encrypt);
	if ((intVal == 1) && (encrypt != 1)) {
		// shared-key and not WEP enabled, force to open-system
		intVal = 0;
	}
	pmib->dot1180211AuthEntry.dot11AuthAlgrthm = intVal;

	if (encrypt == 0)
		pmib->dot1180211AuthEntry.dot11PrivacyAlgrthm = 0;
	else if (encrypt == 1) {
		// WEP mode
		apmib_get(MIB_WLAN_WEP, (void *)&wep);
		if (wep == 1) {
			pmib->dot1180211AuthEntry.dot11PrivacyAlgrthm = 1;
			apmib_get(MIB_WLAN_WEP64_KEY1, (void *)buf1);
			memcpy(&(pmib->dot11DefaultKeysTable.keytype[0]), buf1, 5);
			apmib_get(MIB_WLAN_WEP64_KEY2, (void *)buf1);
			memcpy(&(pmib->dot11DefaultKeysTable.keytype[1]), buf1, 5);
			apmib_get(MIB_WLAN_WEP64_KEY3, (void *)buf1);
			memcpy(&(pmib->dot11DefaultKeysTable.keytype[2]), buf1, 5);
			apmib_get(MIB_WLAN_WEP64_KEY4, (void *)buf1);
			memcpy(&(pmib->dot11DefaultKeysTable.keytype[3]), buf1, 5);
			apmib_get(MIB_WLAN_WEP_DEFAULT_KEY, (void *)&intVal);
			pmib->dot1180211AuthEntry.dot11PrivacyKeyIndex = intVal;
		}
		else {
			pmib->dot1180211AuthEntry.dot11PrivacyAlgrthm = 5;
			apmib_get(MIB_WLAN_WEP128_KEY1, (void *)buf1);
			memcpy(&(pmib->dot11DefaultKeysTable.keytype[0]), buf1, 13);
			apmib_get(MIB_WLAN_WEP128_KEY2, (void *)buf1);
			memcpy(&(pmib->dot11DefaultKeysTable.keytype[1]), buf1, 13);
			apmib_get(MIB_WLAN_WEP128_KEY3, (void *)buf1);
			memcpy(&(pmib->dot11DefaultKeysTable.keytype[2]), buf1, 13);
			apmib_get(MIB_WLAN_WEP128_KEY4, (void *)buf1);
			memcpy(&(pmib->dot11DefaultKeysTable.keytype[3]), buf1, 13);
			apmib_get(MIB_WLAN_WEP_DEFAULT_KEY, (void *)&intVal);
			pmib->dot1180211AuthEntry.dot11PrivacyKeyIndex = intVal;
		}
	}
	else {
		// WPA mode
		pmib->dot1180211AuthEntry.dot11PrivacyAlgrthm = 2;
	}

	// Set 802.1x flag
	enable1x = 0;
	if (encrypt < 2) {
		apmib_get(MIB_WLAN_ENABLE_1X, (void *)&intVal);
		apmib_get(MIB_WLAN_ENABLE_MAC_AUTH, (void *)&intVal2);
		if ((intVal != 0) || (intVal2 != 0))
			enable1x = 1;
	}
	else
		enable1x = 1;
	pmib->dot118021xAuthEntry.dot118021xAlgrthm = enable1x;

	// Set WDS
	apmib_get(MIB_WLAN_WDS_ENABLED, (void *)&intVal);
	apmib_get(MIB_WLAN_WDS_NUM, (void *)&intVal2);
	pmib->dot11WdsInfo.wdsNum = 0;
	if (((mode == 2) || (mode == 3)) &&
		(intVal != 0) &&
		(intVal2 != 0)) {
		for (i=0; i<intVal2; i++) {
			buf1[0] = i+1;
			apmib_get(MIB_WLAN_WDS, (void *)buf1);
			pAcl = (MACFILTER_T *)buf1;
			memcpy(&(pmib->dot11WdsInfo.wdsMacAddr[i][0]), &(pAcl->macAddr[0]), 6);
			pmib->dot11WdsInfo.wdsNum++;
			sprintf(buf2, "ifconfig %s-wds%d hw ether %02x%02x%02x%02x%02x%02x", ifname, i, mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);
			system(buf2);
		}
		pmib->dot11WdsInfo.wdsEnabled = intVal;
	}
	else
		pmib->dot11WdsInfo.wdsEnabled = 0;

	if (((mode == 2) || (mode == 3)) &&
		(intVal != 0)) {
		apmib_get(MIB_WLAN_WDS_ENCRYPT, (void *)&intVal);
		if (intVal == 0)
			pmib->dot11WdsInfo.wdsPrivacy = 0;
		else if (intVal == 1) {
			apmib_get(MIB_WLAN_WDS_WEP_KEY, (void *)buf1);
			pmib->dot11WdsInfo.wdsPrivacy = 1;
			string_to_hex(buf1, &(pmib->dot11WdsInfo.wdsWepKey[0]), 10);
		}
		else if (intVal == 2) {
			apmib_get(MIB_WLAN_WDS_WEP_KEY, (void *)buf1);
			pmib->dot11WdsInfo.wdsPrivacy = 5;
			string_to_hex(buf1, &(pmib->dot11WdsInfo.wdsWepKey[0]), 26);
		}
		else if (intVal == 3)
			pmib->dot11WdsInfo.wdsPrivacy = 2;
		else
			pmib->dot11WdsInfo.wdsPrivacy = 4;
	}

	// enable/disable the notification for IAPP
	apmib_get(MIB_WLAN_IAPP_DISABLED, (void *)&intVal);
	if (intVal == 0)
		pmib->dot11OperationEntry.iapp_enable = 1;
	else
		pmib->dot11OperationEntry.iapp_enable = 0;

	// set band
	apmib_get(MIB_WLAN_BAND, (void *)&intVal);
	apmib_get(MIB_WIFI_SPECIFIC, (void *)&intVal2);
	if ((mode != 1) && (intVal2 == 1) && (intVal == 2))
		intVal = 3;
	pmib->dot11BssType.net_work_type = intVal;

	// set nat2.5 disable when client and mac clone is set
	apmib_get(MIB_WLAN_NAT25_MAC_CLONE, (void *)&intVal);
	if ((intVal == 1) && (mode == 1)) {
		pmib->ethBrExtInfo.nat25_disable = 1;
		pmib->ethBrExtInfo.macclone_enable = 1;
	}
	else {
		pmib->ethBrExtInfo.nat25_disable = 0;
		pmib->ethBrExtInfo.macclone_enable = 0;
	}

	// set nat2.5 disable and macclone disable when wireless isp mode
	apmib_get(MIB_OP_MODE, (void *)&intVal);
	if (intVal == 2) {
		pmib->ethBrExtInfo.nat25_disable = 1;
		pmib->ethBrExtInfo.macclone_enable = 0;
	}

	// set 11g protection mode
	apmib_get(MIB_WLAN_PROTECTION_DISABLED, (void *)&intVal);
	pmib->dot11StationConfigEntry.protectionDisabled = intVal;

	// set block relay
	apmib_get(MIB_WLAN_BLOCK_RELAY, (void *)&intVal);
	pmib->dot11OperationEntry.block_relay = intVal;

	// set WiFi specific mode
	apmib_get(MIB_WIFI_SPECIFIC, (void *)&intVal);
	pmib->dot11OperationEntry.wifi_specific = intVal;

	// set turbo mode
	if (pmib->dot11StationConfigEntry.autoRate)
		apmib_get(MIB_TURBO_MODE, (void *)&intVal);
	else
		intVal = TURBO_OFF;	
	pmib->miscEntry.turbo_mode = intVal;

#ifdef WIFI_SIMPLE_CONFIG
	pmib->wscEntry.wsc_enable = 0;
#endif

// for WMM
	apmib_get(MIB_WLAN_WMM_ENABLED, (void *)&intVal);
	pmib->dot11QosEntry.dot11QosEnable = intVal;

	wrq.u.data.pointer = (caddr_t)pmib;
	wrq.u.data.length = sizeof(struct wifi_mib);
	if (ioctl(skfd, 0x8B43, &wrq) < 0) {
		printf("Set WLAN MIB failed!\n");
		return -1;
	}
	close(skfd);

#ifdef UNIVERSAL_REPEATER
	// set repeater interface
	if (!strcmp(ifname, "wlan0")) {
		apmib_get(MIB_REPEATER_ENABLED1, (void *)&intVal);
		apmib_get(MIB_WLAN_NETWORK_TYPE, (void *)&intVal2);		
		system("ifconfig wlan0-vxd down");
		if (intVal != 0 && mode != WDS_MODE && 
				!(mode==CLIENT_MODE && intVal2==ADHOC)) {
			skfd = socket(AF_INET, SOCK_DGRAM, 0);
			strncpy(wrq.ifr_name, "wlan0-vxd", IFNAMSIZ);
			if (ioctl(skfd, SIOCGIWNAME, &wrq) < 0) {
				printf("Interface open failed!\n");
				return -1;
			}

			wrq.u.data.pointer = (caddr_t)pmib;
			wrq.u.data.length = sizeof(struct wifi_mib);
			if (ioctl(skfd, 0x8B42, &wrq) < 0) {
				printf("Get WLAN MIB failed!\n");
				return -1;
			}

			apmib_get(MIB_REPEATER_SSID1, (void *)buf1);
			intVal2 = strlen(buf1);
			pmib->dot11StationConfigEntry.dot11DesiredSSIDLen = intVal2;
			memset(pmib->dot11StationConfigEntry.dot11DesiredSSID, 0, 32);
			memcpy(pmib->dot11StationConfigEntry.dot11DesiredSSID, buf1, intVal2);

			sprintf(buf1, "ifconfig %s-vxd hw ether %02x%02x%02x%02x%02x%02x", ifname, mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);
			system(buf1);

			enable1xVxd = 0;
			if (encrypt == 0)
				pmib->dot1180211AuthEntry.dot11PrivacyAlgrthm = 0;
			else if (encrypt == 1) {
				if (enable1x == 0) {
					if (wep == 1)
						pmib->dot1180211AuthEntry.dot11PrivacyAlgrthm = 1;
					else
						pmib->dot1180211AuthEntry.dot11PrivacyAlgrthm = 5;
				}
				else
					pmib->dot1180211AuthEntry.dot11PrivacyAlgrthm = 0;
			}
			else {
				apmib_get(MIB_WLAN_WPA_AUTH, (void *)&intVal2);
				if (intVal2 == 2) {
					pmib->dot1180211AuthEntry.dot11PrivacyAlgrthm = 2;
					enable1xVxd = 1;
				}
				else
					pmib->dot1180211AuthEntry.dot11PrivacyAlgrthm = 0;
			}
			pmib->dot118021xAuthEntry.dot118021xAlgrthm = enable1xVxd;
			
			wrq.u.data.pointer = (caddr_t)pmib;
			wrq.u.data.length = sizeof(struct wifi_mib);
			if (ioctl(skfd, 0x8B43, &wrq) < 0) {
				printf("Set WLAN MIB failed!\n");
				return -1;
			}
			close(skfd);
		}
	}

	if (!strcmp(ifname, "wlan1")) {
		apmib_get(MIB_REPEATER_ENABLED1, (void *)&intVal);
		system("ifconfig wlan1-vxd down");
		if (intVal != 0) {
			skfd = socket(AF_INET, SOCK_DGRAM, 0);
			strncpy(wrq.ifr_name, "wlan1-vxd", IFNAMSIZ);
			if (ioctl(skfd, SIOCGIWNAME, &wrq) < 0) {
				printf("Interface open failed!\n");
				return -1;
			}

			wrq.u.data.pointer = (caddr_t)pmib;
			wrq.u.data.length = sizeof(struct wifi_mib);
			if (ioctl(skfd, 0x8B42, &wrq) < 0) {
				printf("Get WLAN MIB failed!\n");
				return -1;
			}

			apmib_get(MIB_REPEATER_SSID1, (void *)buf1);
			intVal2 = strlen(buf1);
			pmib->dot11StationConfigEntry.dot11DesiredSSIDLen = intVal2;
			memset(pmib->dot11StationConfigEntry.dot11DesiredSSID, 0, 32);
			memcpy(pmib->dot11StationConfigEntry.dot11DesiredSSID, buf1, intVal2);

			sprintf(buf1, "ifconfig %s-vxd hw ether %02x%02x%02x%02x%02x%02x", ifname, mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);
			system(buf1);

			enable1xVxd = 0;
			if (encrypt == 0)
				pmib->dot1180211AuthEntry.dot11PrivacyAlgrthm = 0;
			else if (encrypt == 1) {
				if (enable1x == 0) {
					if (wep == 1)
						pmib->dot1180211AuthEntry.dot11PrivacyAlgrthm = 1;
					else
						pmib->dot1180211AuthEntry.dot11PrivacyAlgrthm = 5;
				}
				else
					pmib->dot1180211AuthEntry.dot11PrivacyAlgrthm = 0;
			}
			else {
				apmib_get(MIB_WLAN_WPA_AUTH, (void *)&intVal2);
				if (intVal2 == 2) {
					pmib->dot1180211AuthEntry.dot11PrivacyAlgrthm = 2;
					enable1xVxd = 1;
				}
				else
					pmib->dot1180211AuthEntry.dot11PrivacyAlgrthm = 0;
			}
			pmib->dot118021xAuthEntry.dot118021xAlgrthm = enable1xVxd;

			wrq.u.data.pointer = (caddr_t)pmib;
			wrq.u.data.length = sizeof(struct wifi_mib);
			if (ioctl(skfd, 0x8B43, &wrq) < 0) {
				printf("Set WLAN MIB failed!\n");
				return -1;
			}
			close(skfd);
		}
	}
#endif

	free(pmib);
	return 0;
}
#endif // WLAN_FAST_INIT

#ifdef HOME_GATEWAY
static int generatePPPConf(int is_pppoe, char *option_file, char *pap_file, char *chap_file)
{
	FILE *fd;
	char tmpbuf[200], buf1[100], buf2[100];
	int srcIdx, dstIdx;

	if ( !apmib_init()) {
		printf("Initialize AP MIB failed!\n");
		return -1;
	}

	if (is_pppoe) {
		apmib_get(MIB_PPP_USER, (void *)buf1);
		apmib_get(MIB_PPP_PASSWORD, (void *)buf2);

	}
	else {
		apmib_get(MIB_PPTP_USER, (void *)buf1);
		apmib_get(MIB_PPTP_PASSWORD, (void *)buf2);		
	}
	// delete '"' in the value
	for (srcIdx=0, dstIdx=0; buf1[srcIdx]; srcIdx++, dstIdx++) {
		if (buf1[srcIdx] == '"')
			tmpbuf[dstIdx++] = '\\';
		tmpbuf[dstIdx] = buf1[srcIdx];
	}
	if (dstIdx != srcIdx) {
		memcpy(buf1, tmpbuf, dstIdx);
		buf1[dstIdx] ='\0';
	}
	
	for (srcIdx=0, dstIdx=0; buf2[srcIdx]; srcIdx++, dstIdx++) {
		if (buf2[srcIdx] == '"')
			tmpbuf[dstIdx++] = '\\';
		tmpbuf[dstIdx] = buf2[srcIdx];
	}
	if (dstIdx != srcIdx) {
		memcpy(buf2, tmpbuf, dstIdx);
		buf2[dstIdx] ='\0';
	}
	
	fd = fopen(option_file, "w");
	if (fd == NULL) {
		printf("open file %s error!\n", option_file);
		return -1;
	}
	sprintf(tmpbuf, "name \"%s\"\n", buf1);
	fputs(tmpbuf, fd);
	fclose(fd);
	
	fd = fopen(pap_file, "w");
	if (fd == NULL) {
		printf("open file %s error!\n", pap_file);
		return -1;
	}
	fputs("#################################################\n", fd);
	sprintf(tmpbuf, "\"%s\"	*	\"%s\"\n", buf1, buf2);	
	fputs(tmpbuf, fd);

	fd = fopen(chap_file, "w");
	if (fd == NULL) {
		printf("open file %s error!\n", chap_file);
		return -1;
	}
	fputs("#################################################\n", fd);
	sprintf(tmpbuf, "\"%s\"	*	\"%s\"\n", buf1, buf2);	
	fputs(tmpbuf, fd);
	return 0;
}
#endif // HOME_GATEWAY

#ifdef WIFI_SIMPLE_CONFIG
enum { 
	MODE_AP_UNCONFIG=1, 			// AP unconfigured (enrollee)
	MODE_CLIENT_UNCONFIG=2, 		// client unconfigured (enrollee) 
	MODE_CLIENT_CONFIG=3,			// client configured (registrar) 
	MODE_AP_PROXY=4, 			// AP configured (proxy)
	MODE_AP_PROXY_REGISTRAR=5,		// AP configured (proxy and registrar)
	MODE_CLIENT_UNCONFIG_REGISTRAR=6		// client unconfigured (registrar) 
};

#define WRITE_WSC_PARAM(dst, tmp, str, val) {	\
	sprintf(tmp, str, val); \
	memcpy(dst, tmp, strlen(tmp)); \
	dst += strlen(tmp); \
}

static void convert_bin_to_str(unsigned char *bin, int len, char *out)
{
	int i;
	char tmpbuf[10];

	out[0] = '\0';

	for (i=0; i<len; i++) {
		sprintf(tmpbuf, "%02x", bin[i]);
		strcat(out, tmpbuf);
	}
}

static void convert_hex_to_ascii(unsigned long code, char *out)
{
	*out++ = '0' + ((code / 10000000) % 10);  
	*out++ = '0' + ((code / 1000000) % 10);
	*out++ = '0' + ((code / 100000) % 10);
	*out++ = '0' + ((code / 10000) % 10);
	*out++ = '0' + ((code / 1000) % 10);
	*out++ = '0' + ((code / 100) % 10);
	*out++ = '0' + ((code / 10) % 10);
	*out++ = '0' + ((code / 1) % 10);
	*out = '\0';
}

static int compute_pin_checksum(unsigned long int PIN)
{
	unsigned long int accum = 0;
	int digit;
	
	PIN *= 10;
	accum += 3 * ((PIN / 10000000) % 10); 	
	accum += 1 * ((PIN / 1000000) % 10);
	accum += 3 * ((PIN / 100000) % 10);
	accum += 1 * ((PIN / 10000) % 10); 
	accum += 3 * ((PIN / 1000) % 10); 
	accum += 1 * ((PIN / 100) % 10); 
	accum += 3 * ((PIN / 10) % 10);

	digit = (accum % 10);
	return (10 - digit) % 10;
}

static int updateWscConf(char *in, char *out, int genpin)
{
	int fh;
	struct stat status;
	char *buf, *ptr;
	int intVal, intVal2, is_client, is_config, is_registrar, len, is_wep=0, wep_key_type=0, wep_transmit_key=0;
	char tmpbuf[100], tmp1[100];
		
	if ( !apmib_init()) {
		printf("Initialize AP MIB failed!\n");
		return -1;
	}

	apmib_get(MIB_WSC_PIN, (void *)tmpbuf);
	if (genpin || !memcmp(tmpbuf, "\x0\x0\x0\x0\x0\x0\x0\x0", PIN_LEN)) {
		#include <sys/time.h>			
		struct timeval tod;
		unsigned long num;
		
		gettimeofday(&tod , NULL);

		apmib_get(MIB_HW_NIC0_ADDR, (void *)&tmp1);			
		tod.tv_sec += tmp1[4]+tmp1[5];		
		srand(tod.tv_sec);
		num = rand() % 10000000;
		num = num*10 + compute_pin_checksum(num);
		convert_hex_to_ascii((unsigned long)num, tmpbuf);

		apmib_set(MIB_WSC_PIN, (void *)tmpbuf);
		apmib_update(CURRENT_SETTING);		

		printf("Generated PIN = %s\n", tmpbuf);

		if (genpin)
			return 0;
	}

	if (stat(in, &status) < 0) {
		printf("stat() error [%s]!\n", in);
		return -1;
	}

	buf = malloc(status.st_size+2048);
	if (buf == NULL) {
		printf("malloc() error [%d]!\n", (int)status.st_size+2048);
		return -1;		
	}

	ptr = buf;
	apmib_get(MIB_WLAN_MODE, (void *)&is_client);
	apmib_get(MIB_WSC_CONFIGURED, (void *)&is_config);
	apmib_get(MIB_WSC_REGISTRAR_ENABLED, (void *)&is_registrar);	
#ifdef CONFIG_RTL8186_KLD_REPEATER
	int is_repeater_enabled;
	int wps_vxdAP_enabled=0;
	apmib_get(MIB_REPEATER_ENABLED1, (void *)&is_repeater_enabled);
#endif
	if (is_client == CLIENT_MODE) {
#ifdef CONFIG_RTL8186_KLD_REPEATER
		if (is_repeater_enabled && is_config) {
			intVal = MODE_AP_PROXY_REGISTRAR;
			wps_vxdAP_enabled = 1;
			WRITE_WSC_PARAM(ptr, tmpbuf, "disable_configured_by_exReg = %d\n", 1);
		}
		else
#endif
		{
			if (is_registrar) {
				if (!is_config)
					intVal = MODE_CLIENT_UNCONFIG_REGISTRAR;
				else
					intVal = MODE_CLIENT_CONFIG;			
			}
			else
				intVal = MODE_CLIENT_UNCONFIG;
		}
	}
	else {
		if (!is_config)
			intVal = MODE_AP_UNCONFIG;
		else
			intVal = MODE_AP_PROXY_REGISTRAR;
	}
	WRITE_WSC_PARAM(ptr, tmpbuf, "mode = %d\n", intVal);

	if (is_client) {
#ifdef CONFIG_RTL8186_KLD_REPEATER
		if (wps_vxdAP_enabled)
			apmib_get(MIB_WSC_UPNP_ENABLED, (void *)&intVal);
		else
#endif
			intVal = 0;
	}
	else
		apmib_get(MIB_WSC_UPNP_ENABLED, (void *)&intVal);
	WRITE_WSC_PARAM(ptr, tmpbuf, "upnp = %d\n", intVal);

	intVal = 0;
	apmib_get(MIB_WSC_METHOD, (void *)&intVal);
	//Ethernet(0x2)+Label(0x4)+PushButton(0x80) Bitwise OR
	if (intVal == 1) //Pin+Ethernet
		intVal = (CONFIG_METHOD_ETH | CONFIG_METHOD_PIN);
	else if (intVal == 2) //PBC+Ethernet
		intVal = (CONFIG_METHOD_ETH | CONFIG_METHOD_PBC);
	if (intVal == 3) //Pin+PBC+Ethernet
		intVal = (CONFIG_METHOD_ETH | CONFIG_METHOD_PIN | CONFIG_METHOD_PBC);
	WRITE_WSC_PARAM(ptr, tmpbuf, "config_method = %d\n", intVal);

	apmib_get(MIB_WSC_AUTH, (void *)&intVal2);
	WRITE_WSC_PARAM(ptr, tmpbuf, "auth_type = %d\n", intVal2);

	apmib_get(MIB_WSC_ENC, (void *)&intVal);
	WRITE_WSC_PARAM(ptr, tmpbuf, "encrypt_type = %d\n", intVal);
	if (intVal == WSC_ENCRYPT_WEP)
		is_wep = 1;

	if (is_client) {
#ifdef CONFIG_RTL8186_KLD_REPEATER
		if (wps_vxdAP_enabled)
			intVal = 1;
		else
#endif
		{
			apmib_get(MIB_WLAN_NETWORK_TYPE, (void *)&intVal);
			if (intVal == 0)
				intVal = 1;
			else
				intVal = 2;
		}
	}
	else
		intVal = 1;
	WRITE_WSC_PARAM(ptr, tmpbuf, "connection_type = %d\n", intVal);

	apmib_get(MIB_WSC_MANUAL_ENABLED, (void *)&intVal);
	WRITE_WSC_PARAM(ptr, tmpbuf, "manual_config = %d\n", intVal);

	if (is_wep) { // only allow WEP in none-MANUAL mode (configured by external registrar)
		apmib_get(MIB_WLAN_ENCRYPT, (void *)&intVal);
		if (intVal != ENCRYPT_WEP) {
			printf("WEP mismatched between WPS and host system\n");
			free(buf);
			return -1;
		}
		apmib_get(MIB_WLAN_WEP, (void *)&intVal);
		if (intVal <= WEP_DISABLED || intVal > WEP128) {
			printf("WEP encrypt length error\n");
			free(buf);
			return -1;
		}
		apmib_get(MIB_WLAN_WEP_KEY_TYPE, (void *)&wep_key_type);
		apmib_get(MIB_WLAN_WEP_DEFAULT_KEY, (void *)&wep_transmit_key);
		wep_transmit_key++;
		WRITE_WSC_PARAM(ptr, tmpbuf, "wep_transmit_key = %d\n", wep_transmit_key);
		if (intVal == WEP64) {
			apmib_get(MIB_WLAN_WEP64_KEY1, (void *)&tmpbuf);
			if (wep_key_type == KEY_ASCII) {
				memcpy(tmp1, tmpbuf, 5);
				tmp1[5] = '\0';
			}
			else {
				convert_bin_to_str(tmpbuf, 5, tmp1);
				tmp1[10] = '\0';
			}			
			WRITE_WSC_PARAM(ptr, tmpbuf, "network_key = %s\n", tmp1);

			apmib_get(MIB_WLAN_WEP64_KEY2, (void *)&tmpbuf);
			if (wep_key_type == KEY_ASCII) {
				memcpy(tmp1, tmpbuf, 5);
				tmp1[5] = '\0';
			}
			else {
				convert_bin_to_str(tmpbuf, 5, tmp1);
				tmp1[10] = '\0';
			}			
			WRITE_WSC_PARAM(ptr, tmpbuf, "wep_key2 = %s\n", tmp1);

			apmib_get(MIB_WLAN_WEP64_KEY3, (void *)&tmpbuf);
			if (wep_key_type == KEY_ASCII) {
				memcpy(tmp1, tmpbuf, 5);
				tmp1[5] = '\0';
			}
			else {
				convert_bin_to_str(tmpbuf, 5, tmp1);
				tmp1[10] = '\0';
			}			
			WRITE_WSC_PARAM(ptr, tmpbuf, "wep_key3 = %s\n", tmp1);


			apmib_get(MIB_WLAN_WEP64_KEY4, (void *)&tmpbuf);
			if (wep_key_type == KEY_ASCII) {
				memcpy(tmp1, tmpbuf, 5);
				tmp1[5] = '\0';
			}
			else {
				convert_bin_to_str(tmpbuf, 5, tmp1);
				tmp1[10] = '\0';
			}			
			WRITE_WSC_PARAM(ptr, tmpbuf, "wep_key4 = %s\n", tmp1);
		}
		else {
			apmib_get(MIB_WLAN_WEP128_KEY1, (void *)&tmpbuf);
			if (wep_key_type == KEY_ASCII) {
				memcpy(tmp1, tmpbuf, 13);
				tmp1[13] = '\0';
			}
			else {
				convert_bin_to_str(tmpbuf, 13, tmp1);
				tmp1[26] = '\0';
			}
			WRITE_WSC_PARAM(ptr, tmpbuf, "network_key = %s\n", tmp1);


			apmib_get(MIB_WLAN_WEP128_KEY2, (void *)&tmpbuf);
			if (wep_key_type == KEY_ASCII) {
				memcpy(tmp1, tmpbuf, 13);
				tmp1[13] = '\0';
			}
			else {
				convert_bin_to_str(tmpbuf, 13, tmp1);
				tmp1[26] = '\0';
			}
			WRITE_WSC_PARAM(ptr, tmpbuf, "wep_key2 = %s\n", tmp1);

			apmib_get(MIB_WLAN_WEP128_KEY3, (void *)&tmpbuf);
			if (wep_key_type == KEY_ASCII) {
				memcpy(tmp1, tmpbuf, 13);
				tmp1[13] = '\0';
			}
			else {
				convert_bin_to_str(tmpbuf, 13, tmp1);
				tmp1[26] = '\0';
			}
			WRITE_WSC_PARAM(ptr, tmpbuf, "wep_key3 = %s\n", tmp1);

			apmib_get(MIB_WLAN_WEP128_KEY4, (void *)&tmpbuf);
			if (wep_key_type == KEY_ASCII) {
				memcpy(tmp1, tmpbuf, 13);
				tmp1[13] = '\0';
			}
			else {
				convert_bin_to_str(tmpbuf, 13, tmp1);
				tmp1[26] = '\0';
			}
			WRITE_WSC_PARAM(ptr, tmpbuf, "wep_key4 = %s\n", tmp1);
		}
	}
	else {
		apmib_get(MIB_WLAN_WPA_PSK, (void *)&tmp1);		
		WRITE_WSC_PARAM(ptr, tmpbuf, "network_key = %s\n", tmp1);
	}

#ifdef CONFIG_RTL8186_KLD_REPEATER
	if (wps_vxdAP_enabled)
		apmib_get(MIB_REPEATER_SSID1, (void *)&tmp1);	
	else
#endif
		apmib_get(MIB_WLAN_SSID, (void *)&tmp1);	
	WRITE_WSC_PARAM(ptr, tmpbuf, "ssid = %s\n", tmp1);	

#if 0	
//	}
//	else {			
		apmib_get(MIB_WSC_PSK, (void *)&tmp1);
		WRITE_WSC_PARAM(ptr, tmpbuf, "network_key = %s\n", tmp1);		
		
		apmib_get(MIB_WSC_SSID, (void *)&tmp1);
		WRITE_WSC_PARAM(ptr, tmpbuf, "ssid = %s\n", tmp1);
//	}
#endif

	apmib_get(MIB_WSC_PIN, (void *)&tmp1);
	WRITE_WSC_PARAM(ptr, tmpbuf, "pin_code = %s\n", tmp1);

	apmib_get(MIB_WLAN_CHAN_NUM, (void *)&intVal);
	if (intVal > 14)
		intVal = 2;
	else
		intVal = 1;
	WRITE_WSC_PARAM(ptr, tmpbuf, "rf_band = %d\n", intVal);

/*
	apmib_get(MIB_HW_MODEL_NUM, (void *)&tmp1);	
	WRITE_WSC_PARAM(ptr, tmpbuf, "model_num = \"%s\"\n", tmp1);	

	apmib_get(MIB_HW_SERIAL_NUM, (void *)&tmp1);	
	WRITE_WSC_PARAM(ptr, tmpbuf, "serial_num = \"%s\"\n", tmp1);	
*/
	apmib_get(MIB_DEVICE_NAME, (void *)&tmp1);	
	WRITE_WSC_PARAM(ptr, tmpbuf, "device_name = \"%s\"\n", tmp1);	

	len = (int)(((long)ptr)-((long)buf));
	
	fh = open(in, O_RDONLY);
	if (fh == -1) {
		printf("open() error [%s]!\n", in);
		return -1;
	}

	lseek(fh, 0L, SEEK_SET);
	if (read(fh, ptr, status.st_size) != status.st_size) {		
		printf("read() error [%s]!\n", in);
		return -1;	
	}
	close(fh);

	// search UUID field, replace last 12 char with hw mac address
	ptr = strstr(ptr, "uuid =");
	if (ptr) {
		char tmp2[100];
		apmib_get(MIB_HW_NIC0_ADDR, (void *)&tmp1);	
		convert_bin_to_str(tmp1, 6, tmp2);
		memcpy(ptr+27, tmp2, 12);		
	}

	fh = open(out, O_RDWR|O_CREAT|O_TRUNC);
	if (fh == -1) {
		printf("open() error [%s]!\n", out);
		return -1;
	}

	if (write(fh, buf, len+status.st_size) != len+status.st_size ) {
		printf("Write() file error [%s]!\n", out);
		return -1;
	}
	close(fh);
	free(buf);

	return 0;
}
#endif

