#include "swplatform.h"
#include <timers.h>
#include "midi.h"

#include <stdio.h>
#include <unistd.h>
// #include <keyboard.h>
// #include <termio.h>

/*
 * system configuration
 */
#define C_SCAN_ROWS 8
#define C_SCAN_COLS 8
#define C_RET_MASK (1<<C_SCAN_COLS)-1   // bit mask containing a 1 for each valid keyboard return line
#define C_SCAN_MASK (1<<C_SCAN_ROWS)-1  // bit mask containing a 1 for each valid keyboard scan line
#define C_MASK_UB5  16        // bit mask selecting user button 5 from the byte-wide IO-port value containing the user button states
#define C_NUM_UBS 5           // number of user buttons
#define C_MASK_COL7 128       // bit mask selecting column 7 from the byte-wide scan return IO-port value
#define C_TICK_LENGTH 1000    // timer shall tick in 1msec intervals (1000 usec)
#define C_ALIVE_INTERVAL 250  // show alive sign every 250 msec (250 times the basic tick period of 1 msec)
#define C_DISP_PERIOD 500     // display engine status every 5 seconds (500 times the basic tick period of 1 msec)
#define C_MIDI_CHANNEL 1      // default MIDI channel is 1, will be encoded as 0 on the wire by the midi driver
#define C_NOTE_OFFSET 35      // offset to add to key number to determine MIDI note number,
                              // this will place the middle C (C4, note 64) at the center key (25th key on the keyboard of 49)
#define C_DEFAULT_VELOCITY 64 // default velocity value for keyboards without velocity sensors
#define C_MAX_PS2RETRY 100 // maximum count of retrying PS2 keyboard initialization
/*
 * event action codes
 */
#define C_ACT_CLOSE 129      // key in matrix closed
#define C_ACT_OPEN  130      // key in matrix opened
#define C_ACT_UBCLOSE 131    // user button closed
#define C_ACT_UBOPEN  132    // user button opened
#define C_ACT_DUMP  299      // dump of key matrix state requested
#define C_ACT_KCMS  298      // key row status change
#define C_ACT_PS2K    133    // PS2 keyboard scan code
#define C_ACT_DISP    134    // display update
/*
 * data type definitions
 */
typedef struct ent {         // abstract type for queue entry, contains the linkage attributes
    struct ent * nextptr;
    struct ent * prevptr;
} ent;

typedef struct keymsg {      // key event message, extends queue entry by attributes for key messages
    struct ent * nextptr;
    struct ent * prevptr;
    int row;
    int column;
    int action;
} keymsg;

typedef struct queue {       // queue, consisting of ..
    struct ent queptrs;      // .. head and tail pointers ..
    pthread_mutex_t mutx;     // .. and a mutex for synchronizing
} queue;

queue key_queue;
/*
 * global variables
 */
int ub_state = 0;
int timer_ticks = 0;
int alive_counter = 0;
int display_period = 0;
int alive_led_status = 0;
int disp_led_status = 0;
int scanrow = 0;
int scanstat[C_SCAN_ROWS];      // one bit for every key contact reflecting current, not-debounced state
int scanstat_prev[C_SCAN_ROWS]; // one bit for every key contact reflecting state previous to last change
int ub_save = 0;                // one bit for every user button reflecting current, not-debounced state
int ub_prev = 0;                // one bit for every user button reflecting state previous to last change
uint8_t display_status = 0;     // status of display
/*
 * error codes
 */
#define C_ERR_TREG 130 // error starting tick timer
#define C_ERR_KMEX 131 // key message buffer space exhausted
#define C_ERR_DMEX 132 // dump message buffer space exhausted
#define C_ERR_KCEX 133 // key change message buffer space exhausted
#define C_ERR_SKEX 134 // send key event message buffer space exhausted
#define C_ERR_PS2F 135 // PS2 initialization failure
/*
 * ======================= utility procedures ==================================
 */
void error_halt(int errcod) {
    led_turn_all_off(drv_led_1);
    int i;
    for (i=0; i<=7; i++) {
      if (errcod&1)
        led_turn_on(drv_led_1,LEDS_LED0_R+(i*3));
      else
        led_turn_off(drv_led_1,LEDS_LED0_R+(i*3));
      errcod /= 2;
    }
    while (1);
}
/*
 * initialize RGB LED state
 */
void init_leds() {
    led_turn_all_off(drv_led_1);
    led_turn_on(drv_led_1,LEDS_LED0_B);
    display_status = 4;
}
/*
 * initialize a queue
 */
void init_queue(queue * qp) {
    qp->queptrs.nextptr = 0;
    qp->queptrs.prevptr = 0;
    pthread_mutex_init(&qp->mutx,NULL);
}

int queue_has_entries(queue * qp) {
    if (qp->queptrs.nextptr!=0) {
        return true;
    }
    else {
        return false;
    }
}

void insert_queue_last(queue * qp, struct ent * entptr) {
    if (qp->queptrs.nextptr!=0) {            // there are already entries in the queue
        entptr->nextptr=qp->queptrs.prevptr; // prevptr of queue points to LAST entry, nextptr of any entry points towards FIRST one
        qp->queptrs.prevptr->prevptr=entptr; // prevptr of any entry points towards LAST one
        qp->queptrs.prevptr=entptr;          // new entry is new LAST one ..
        entptr->prevptr=0;                    // .. so new entry has no PREV one
    }
    else {                                    // the queue is empty, insert the first and only one
        qp->queptrs.nextptr=entptr;
        qp->queptrs.prevptr=entptr;
        entptr->nextptr=0;
        entptr->prevptr=0;
    }
}

ent * get_queue_first(queue * qp) {
    if (!queue_has_entries(qp)) {
        return 0;
    }
    else {
        struct ent * entptr;
        entptr = qp->queptrs.nextptr;         // nextptr of queue points to FIRST entry
        qp->queptrs.nextptr=entptr->prevptr;  // new first element is PREV of old first one ..
                                               // .. (may be NULL if this was the only element
        if (qp->queptrs.nextptr==NULL) {      // this was the last entry, so ..
            qp->queptrs.prevptr = NULL;       // .. make head and tail pointers NULL
        }
        return entptr;
    }
}

void send_key_event(int act, int row, int col) {
  struct keymsg * kmptr = malloc(sizeof(struct keymsg));
  if (!kmptr)
    error_halt(C_ERR_SKEX);
  kmptr->action=act;
  kmptr->row=row;
  kmptr->column=col;
  insert_queue_last(&key_queue,(struct ent*)kmptr);
}

void check_alive() {
    alive_counter++;
    display_period++;
    if (alive_counter>=C_ALIVE_INTERVAL) {
        alive_counter = 0;
        if (alive_led_status==0) {
            alive_led_status = 1;
            led_turn_on(drv_led_1, LEDS_LED7_G);
        }
        else {
            alive_led_status = 0;
            led_turn_off(drv_led_1, LEDS_LED7_G);
        }
    }
    if (display_period>=C_DISP_PERIOD) {
      display_period = 0;
      if (disp_led_status==0) {
        disp_led_status = 1;
        led_turn_on(drv_led_1,LEDS_LED6_G);
        send_key_event(C_ACT_DISP,0,0);
        }
      else {
        disp_led_status = 0;
        led_turn_off(drv_led_1,LEDS_LED6_G);
        }
    }
}

void init_scanpoints() {
    scanrow = 0;
    int i = 0;
    for (i=0;i<C_SCAN_ROWS;i++) {
        scanstat[i] = 0;
        scanstat_prev[i] = 0;
    }
    ioport_set_value(drv_ioport_1,KEY_MATRIX_SCAN,(1<<scanrow)^C_SCAN_MASK);
}

void check_scanpoints() {
    /*
     * first, check if the current key matrix row reflects a change in state
     */
    int krow = ioport_get_value(drv_ioport_1,KEY_MATRIX_RET); // get the return lines for current row
    krow ^= C_RET_MASK;    // invert low-active scan return lines
    krow &= C_RET_MASK;    // mask out bits in krow not representing real return lines
    scanstat[scanrow] &= krow;  // only accept a closed contact if it was closed in the last scan
    if (scanstat[scanrow]!=scanstat_prev[scanrow]) {  // check if it changed since the previous detected change
      int scancol=0;
      int keystat=0;
      int oldstat=0;
      int newstat=0;
      int colmask=0;
      for (scancol=0; scancol<C_SCAN_COLS; scancol++) { // iterate over all columns in the current row
        colmask = 1<<scancol;
        oldstat = scanstat_prev[scanrow] & colmask; // pick out the bit for the previous state
        newstat = scanstat[scanrow] & colmask;      // pick out the bit for the new (debounced) state
        if (oldstat!=newstat) {                     // did this very key change ?
          send_key_event(newstat?C_ACT_CLOSE:C_ACT_OPEN,scanrow,scancol); } // if oldstat!=newstat
      }  // for scancol ...
      scanstat_prev[scanrow]=scanstat[scanrow]; // make the current row state to the previous row state
    } // if scanstat!=prev
    scanstat[scanrow]=krow;      // memorize row state until next reading of this line
    scanrow++;                   // move on to next row at next scan interval
    if (scanrow>=C_SCAN_ROWS)
        scanrow=0;               // wrap around to first row after last one
    ioport_set_value(drv_ioport_1,KEY_MATRIX_SCAN,((1<<scanrow)^C_SCAN_MASK)); // drive the nex key row (active low)
    /*
     * now check user buttons
     */
    int ub_act = ioport_get_value(drv_ioport_2,0);
    ub_save &= ub_act;
    if (ub_save!=ub_prev) { // check if a user button has changed
      int ubnum=0;
      int oldstat=0;
      int newstat=0;
      int ubmask=0;
      for (ubnum=0;ubnum<C_NUM_UBS;ubnum++) { // check every user button for change
        ubmask=1<<ubnum;
        oldstat = ub_prev & ubmask;
        newstat = ub_save & ubmask;
        if (oldstat!=newstat) { // has this button changed?
          send_key_event(newstat?C_ACT_UBCLOSE:C_ACT_UBOPEN,ubnum,ub_save); }
      } // for ubnum
      ub_prev=ub_save;
    } // if ub_save!=ub_prev
    ub_save=ub_act;          // memorize actual user button state (not-debounced)
}

void tick(void * timerctx) {
    timer_ticks++;
    check_alive();
    check_scanpoints();
}

void init_ub() {
    ub_state = 0;
    timer_ticks=0;
    void * timer_id = timer_register_handler((void*)0,C_TICK_LENGTH,tick);
    if (!timer_id) {
        error_halt(C_ERR_TREG);
    }
    init_scanpoints();
    init_queue(&key_queue);
    /*
    int fd = fileno(stdin);
    posix_devctl_keyboard_blocking_t blocking;
    blocking.blocking = true;
    posix_devctl( fd, DEVCTL_KEYBOARD_BLOCKING, (void*)&blocking, sizeof( blocking ), NULL );
    */
}


void init_ps2kbd() {
    int ic = C_MAX_PS2RETRY;
    int rnok = 0;
    int kbstat = ps2kb_get_drvstat(drv_ps2kb_1);
    printf("initial PS2KB status is %d, OK would be %d, RESET_FAIL would be %d\n",kbstat,PS2KB_OK,PS2KB_RESET_FAILED);
    while (ic>0) {
        rnok = ps2kb_reset(drv_ps2kb_1);
        printf("PS2KB reset returned %d, %d attempts left.\n",rnok,ic);
        if (!rnok) {
          kbstat = ps2kb_get_drvstat(drv_ps2kb_1);
          printf("PS2KB status is %d\n",kbstat);
          if (kbstat==PS2KB_OK) {
            return;
          }
        }
        ic--;
    }
  error_halt(C_ERR_PS2F);
}

void update_leds_b() { // reflect the actual state of the key return lines on the blue LEDs
    int krow = ioport_get_value(drv_ioport_1,KEY_MATRIX_RET)^C_RET_MASK; // invert low-active return lines
    int i;
    for (i=0; i<C_SCAN_COLS; i++) {
        if ((krow & (C_MASK_COL7>>i)) != 0)
          led_turn_on(drv_led_1,LEDS_LED0_B+(3*i));
        else
          led_turn_off(drv_led_1,LEDS_LED0_B+(3*i));
    }
}

void check_poll() { // check polled peripherals (in foreground)
  /*
   * check PS2 keyboard for events
   */
  uint8_t * scodes = ps2kb_get_scancode(drv_ps2kb_1);
  int i;
  for (i=0;i<8;i++,scodes++) {
    if (*scodes) {
      send_key_event(C_ACT_PS2K,*scodes,0);
    }
  }
}

void update_display() {
    uint8_t dstat = (display_status==1?4:display_status>>1);
    ps2kb_setleds(drv_ps2kb_1,dstat);
    display_status = dstat;
}

void dump_matrix() {
  printf("Key Matrix Status:\n");
  int row=0;
  int col=0;
    for(row=0;row<C_SCAN_ROWS;row++) {
      printf("%d - ",row);
      for(col=0;col<C_SCAN_COLS;col++) {
        if (((1<<col)&scanstat[row])!=0) {
          printf("*");
          }
        else {
          printf("_");
          }
        }
      printf("\n");
    }
}

void process_key(int r, int c, int a) { // process key event row r, column c, action a
  // convert r,c to key number 01..49
  int colbase[8] = {49,43,37,31,25,19,13,7};
  int kn = colbase[r]-c;    // key number, starting left from 1 thru 49
  uint8_t note = kn+C_NOTE_OFFSET;
  uint8_t event = (a==C_ACT_CLOSE?MIDI_NOTE_ON:MIDI_NOTE_OFF);
  uint8_t velocity = C_DEFAULT_VELOCITY;
  uint8_t channel = C_MIDI_CHANNEL;
  // printf("Key %d, event 0x%x, channel %d, note %d, velocity %d\n",kn,event,channel,note,velocity);
  midi_msg_t * mp = midi_createvoicemsg((midi_msg_t*)0,0,event,channel,note,velocity);
  /*
  velocity = mp->event.note_off.velocity;
  note = mp->event.note_off.note;
  channel = mp->channel;
  event = mp->eventid;
  printf(" ... event=0x%x - channel=%d - note=%d - velocity=%d\n",event,channel,note,velocity);
  */
  midi_txmsg(midi_1,mp);
}

void process_ub(int row, int column, int action) {
      printf(" User button event %d, button=%d, mask=0x%x\n", action, row, column);
}

void process_ps2key(int scode) {
    printf("PS2 key scan code 0x%x\n",scode);
}

void check_event() {
    if (queue_has_entries(&key_queue)) {
      keymsg * kmp = (keymsg *) get_queue_first(&key_queue);
      if (kmp) {
        // printf("Key Event Message: row=%d, col=%d, action=%d\n",kmp->row,kmp->column,kmp->action);
        switch (kmp->action) {
        case C_ACT_DUMP:
          dump_matrix();
          break; // case C_ACT_DUMP
        case C_ACT_DISP:
          update_display();
          breake; // case A_ACT_DISP;
       case C_ACT_KCMS:
          printf("Key state change: old=%d, new=%d\n",kmp->row, kmp->column);
          break; // case C_ACT_KCMS
        case C_ACT_CLOSE:
        case C_ACT_OPEN:
          process_key(kmp->row, kmp->column, kmp->action);
          break;
        case C_ACT_UBOPEN:
        case C_ACT_UBCLOSE:
          process_ub(kmp->row, kmp->column, kmp->action);
          break;
        case C_ACT_PS2K:
          process_ps2key(kmp->row);
        } // switch
        free(kmp);
      } // if kmp
    } // if has_entries
} // check_event

void main() {
  // printf("MIDIKBD ...\n");
  swplatform_init_stacks();
  printf("MIDIKBD starting ...\n");
  init_leds();
  init_ub();
  led_turn_on(drv_led_1,LEDS_LED1_B);
  init_ps2kbd();
  led_turn_off(drv_led_1,LEDS_LED1_B);
  led_turn_off(drv_led_1,LEDS_LED0_B);
  led_turn_on(drv_led_1,LEDS_LED0_G);
  printf("MIDIKBD initialized\n");
  for (;;) {
    update_leds_b();
    check_poll();
    check_event();
  } // for
} // main
