#include "swplatform.h"

#include <timers.h>

#define C_SCAN_ROWS 8
#define C_SCAN_COLS 8
#define C_RET_MASK (1<<C_SCAN_COLS)-1   // bit mask containing a 1 for each valid keyboard return line
#define C_SCAN_MASK (1<<C_SCAN_ROWS)-1  // bit mask containing a 1 for each valid keyboard scan line
#define C_MASK_UB5  16    // bit mask selecting user button 5 from the byte-wide IO-port value containing the user button states
#define C_MASK_COL7 128   // bit mask selecting column 7 from the byte-wide scan return IO-port value

#define C_ACT_CLOSE 129
#define C_ACT_OPEN  130
#define C_ACT_DUMP  299
#define C_ACT_KCMS  298

typedef struct ent {
    struct ent * nextptr;
    struct ent * prevptr;
} ent;

typedef struct keymsg {
    struct ent * nextptr;
    struct ent * prevptr;
    int row;
    int column;
    int action;
} keymsg;

struct ent key_queue;

int ub_state = 0;
int timer_ticks = 0;
int alive_counter = 0;
int display_period = 0;
int alive_led_status = 0;
int disp_led_status = 0;
int scanrow = 0;
int scanstat[C_SCAN_ROWS];
int scanstat_prev[C_SCAN_ROWS];

// error starting tick timer
#define C_ERR_TREG 130
// key message buffer space exhausted
#define C_ERR_KMEX 131
// dump message buffer space exhausted
#define C_ERR_DMEX 132
// key chamge message buffer space exhausted
#define C_ERR_KCEX 133

//
// timer shall tick in 10msec intervals (10000 usec)
//
#define C_TICK_LENGTH 10000
//
// show alive sign every 500 msec (50 times the basic tick period of 10 msec)
//
#define C_ALIVE_INTERVAL 50
//
// display engine status every 5 seconds (500 times the basic tick period of 10 msec)
//
#define C_DISP_PERIOD 500

void error_halt(int errcod) {
    led_turn_all_off(drv_led_1);
    int i;
    for (i=0; i<=7; i++) {
      if (errcod&1)
        led_turn_on(drv_led_1,LEDS_LED0_R+(i*3));
      else
        led_turn_off(drv_led_1,LEDS_LED0_R+(i*3));
      errcod /= 2;
    }
    while (1);
}
void init_leds() {
    led_turn_all_off(drv_led_1);
    led_turn_on(drv_led_1,LEDS_LED0_B);
}

void init_queue(struct ent * queptr) {
    queptr->nextptr=0;
    queptr->prevptr=0;
}

int queue_has_entries(struct ent * queptr) {
    if (queptr->nextptr!=0) {
        return true;
    }
    else {
        return false;
    }
}

void insert_queue_last(struct ent * queptr, struct ent * entptr) {
    if (queptr->nextptr!=0) {
        entptr->nextptr=queptr->prevptr;
        queptr->prevptr->prevptr=entptr;
        queptr->prevptr=entptr;
        entptr->prevptr=0;
    }
    else {
        queptr->nextptr=entptr;
        queptr->prevptr=entptr;
        entptr->nextptr=0;
        entptr->prevptr=0;
    }
}

ent * get_queue_first(struct ent * queptr) {
    if (!queue_has_entries(queptr)) {
        return 0;
    }
    else {
        struct ent * entptr;
        entptr = queptr->nextptr;
        queptr->nextptr=queptr->nextptr->nextptr;
        return entptr;
    }
}

void check_alive() {
    alive_counter++;
    display_period++;
    if (alive_counter>=C_ALIVE_INTERVAL) {
        alive_counter = 0;
        if (alive_led_status==0) {
            alive_led_status = 1;
            led_turn_on(drv_led_1, LEDS_LED7_G);
        }
        else {
            alive_led_status = 0;
            led_turn_off(drv_led_1, LEDS_LED7_G);
        }
    }
    if (display_period>=C_DISP_PERIOD) {
      display_period = 0;
      struct keymsg * kmptr = malloc(sizeof(struct keymsg));
      if (!kmptr) {
        error_halt(C_ERR_DMEX); }
      kmptr->action=C_ACT_DUMP;
      kmptr->row=timer_ticks;
      kmptr->column=0;
      insert_queue_last(&key_queue,(struct ent*)kmptr);
      if (disp_led_status==0) {
        disp_led_status = 1;
        led_turn_on(drv_led_1,LEDS_LED6_G);
        }
      else {
        disp_led_status = 0;
        led_turn_off(drv_led_1,LEDS_LED6_G);
        }
    }
}

void init_scanpoints() {
    scanrow = 0;
    int i = 0;
    for (i=0;i<C_SCAN_ROWS;i++) {
        scanstat[i] = 0;
        scanstat_prev[i] = 0;
    }
    ioport_set_value(drv_ioport_1,KEY_MATRIX_SCAN,(1<<scanrow)^C_SCAN_MASK);
}

void check_scanpoints() {
    int krow = ioport_get_value(drv_ioport_1,KEY_MATRIX_RET); // get the return lines for current row
    krow ^= C_RET_MASK;    // invert low-active scan return lines
    krow &= C_RET_MASK;    // mask out bits in krow not representing real return lines
    scanstat[scanrow] &= krow;  // only accept a closed contact if it was closed in the last scan
    if (scanstat[scanrow]!=scanstat_prev[scanrow]) {  // check if it changed since the previous detected change
      struct keymsg * kmptr;
      /*
      kmptr = malloc(sizeof(struct keymsg));
      if (!kmptr)
        error_halt(C_ERR_KCEX);
      kmptr->action=C_ACT_KCMS;
      kmptr->row=scanstat[scanrow];
      kmptr->column=scanstat_prev[scanrow];
      insert_queue_last(&key_queue,(struct ent*)kmptr);
      */
      int scancol=0;
      int keystat=0;
      int oldstat=0;
      int newstat=0;
      int colmask=0;
      for (scancol=0; scancol<C_SCAN_COLS; scancol++) { // iterate over all columns in the current row
        colmask = 1<<scancol;
        oldstat = scanstat_prev[scanrow] & colmask; // pick out the bit for the previous state
        newstat = scanstat[scanrow] & colmask;      // pick out the bit for the new (debounced) state
        if (oldstat!=newstat) {                     // did this very key change ?
          kmptr = malloc(sizeof(struct keymsg));
          if (!kmptr)
            error_halt(C_ERR_KMEX);
          kmptr->row=scanrow;
          kmptr->column=scancol;
          if (newstat!=0)
            kmptr->action = C_ACT_CLOSE;
          else
            kmptr->action = C_ACT_OPEN;
          insert_queue_last(&key_queue,(struct ent*)kmptr);
        } // if oldstat!=newstat
      }  // for scancol ...
      scanstat_prev[scanrow]=scanstat[scanrow]; // make the current row state to the previous row state
    } // if scanstat!=prev
    scanstat[scanrow]=krow;      // memorize row state until next reading of this line
    scanrow++;                   // move on to next row at next scan interval
    if (scanrow>=C_SCAN_ROWS)
        scanrow=0;               // wrap around to first row after last one
    ioport_set_value(drv_ioport_1,KEY_MATRIX_SCAN,((1<<scanrow)^C_SCAN_MASK)); // drive the nex key row (active low)
}

void tick(void * timerctx) {
    timer_ticks++;
    check_alive();
    check_scanpoints();
}

void init_ub() {
    ub_state = 0;
    timer_ticks=0;
    void * timer_id = timer_register_handler((void*)0,C_TICK_LENGTH,tick);
    if (!timer_id) {
        error_halt(C_ERR_TREG);
    }
    init_scanpoints();
    init_queue(&key_queue);
}

void check_ub() {
    int ub_act = ioport_get_value(drv_ioport_2,0);
    int krow = ioport_get_value(drv_ioport_1,KEY_MATRIX_RET)^C_RET_MASK; // invert low-active return lines
    int i;
    for (i=0; i<C_SCAN_COLS; i++) {
        if ((krow & (C_MASK_COL7>>i)) != 0)
          led_turn_on(drv_led_1,LEDS_LED0_B+(3*i));
        else
          led_turn_off(drv_led_1,LEDS_LED0_B+(3*i));
    }
}

void dump_matrix() {
  printf("Key Matrix Status:\n");
  int row=0;
  int col=0;
    for(row=0;row<C_SCAN_ROWS;row++) {
      printf("%d - ",row);
      for(col=0;col<C_SCAN_COLS;col++) {
        if (((1<<col)&scanstat[row])!=0) {
          printf("*");
          }
        else {
          printf("_");
          }
        }
      printf("\n");
    }
}

void process_key(int r, int c, int a) { // process key event row r, column c, action a
  struct midi_msg_t * mp = midi_createvoicemsg((midi_msg_t*)0,0,0,0,0,0);
  
}

void check_event() {
    if (queue_has_entries(&key_queue)) {
      keymsg * kmp = (keymsg *) get_queue_first(&key_queue);
      if (kmp) {
        printf("Key Event Message: row=%d, col=%d, action=%d\n",kmp->row,kmp->column,kmp->action);
        switch (kmp->action) {
        case C_ACT_DUMP:
          dump_matrix();
          break; // case C_ACT_DUMP
        case C_ACT_KCMS:
          printf("Key state change: old=%d, new=%d\n",kmp->row, kmp->column);
          break; // case C_ACT_KCMS
        case C_ACT_CLOSE:
        case C_ACT_OPEN:
          process_key(kmp->row, kmp->column, kmp->action);
          break;
        } // switch
        free(kmp);
      } // if kmp
    } // if has_entries
} // check_event

void main() {
  printf("MIDIKBD ...\n");
  swplatform_init_stacks();
  printf("MIDIKBD starting ...\n");
  init_leds();
  init_ub();
  led_turn_off(drv_led_1,LEDS_LED0_B);
  led_turn_on(drv_led_1,LEDS_LED0_G);
  printf("MIDIKBD initialized\n");
  for (;;) {
    check_ub();
    check_event();
  } // for
} // main()
