//***********************************************************
//
// This is the SPI Rocker and Keyboard interface file for 
//
//  Written by:   ESC Inc.
//                47 Comstock Ave.
//                Ivoryton, CT 06442
//                860-767-2282
//                www.embedded-designer.com
//
//    Copyright (C) 2010-2010 by ESC Inc.
//
// Author: George F. Martin     Created 9 June 2010
//
// File: SpiRockerKbTasks.c
//
//***********************************************************

#include "inc/hw_gpio.h"
//#include "inc/hw_ints.h"
#include "inc/hw_memmap.h"
//#include "inc/hw_sysctl.h"
#include "inc/hw_types.h"
//#include "driverlib/debug.h"
#include "driverlib/gpio.h"
//#include "driverlib/interrupt.h"

//#include "inc/hw_nvic.h"
//#include "driverlib/cpu.h"
#include "driverlib/sysctl.h"

#include "SafeRTOS/SafeRTOS_API.h"


#include "../Consts.h"
#include "../DataBase.h"

//***********************************************************
//
//  Rocker Switch Port J Interface
//
//  Bit 7 6 5 4  3 2 1 0 
//      | | | |  | | | |
//      | | | |  | | | |---> SPI Data Out
//      | | | |  | | |-----> SPI Clock
//      | | | |  | |-------> SPI Latch
//      | | | |  |---------> nc
//      | | | |------------< Keyboard
//      | | |--------------< Switch Down
//      | |----------------< Switch Up
//      |------------------< Lane Lock
//
//***********************************************************

#define  RK_OUT      GPIO_PIN_0
#define  RK_CLK      GPIO_PIN_1
#define  RK_LATCH    GPIO_PIN_2

#define  KB_IN       GPIO_PIN_4
#define  RK_DOWN     GPIO_PIN_5
#define  RK_UP       GPIO_PIN_6
#define  RK_LL       GPIO_PIN_7

//**********************************************[ Public ]********
//
// Rocker Switches Port Interface Sets up the hardware
//
//*****************************************************************
void PortInitRockerSwitches(void) {

// enable the GPIO PortJ
SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOJ);

// Set these pins as outputs
GPIOPinTypeGPIOOutput(GPIO_PORTJ_BASE, 
                        GPIO_PIN_3 | 
                        RK_LATCH   | 
                        RK_CLK     | 
                        RK_OUT);
                        
// Set these pins as inputs
GPIOPinTypeGPIOInput(GPIO_PORTJ_BASE, 
                        KB_IN | 
                        RK_LL | 
                        RK_UP | 
                        RK_DOWN);

// Set Drive strength and                         
GPIOPadConfigSet(GPIO_PORTJ_BASE, 
                        GPIO_PIN_3 | 
                        RK_LATCH   | 
                        RK_CLK     | 
                        RK_OUT, 
                        GPIO_STRENGTH_2MA,
                        GPIO_PIN_TYPE_STD);
  
}  // end of void PortInitRockerSwitches(void) {

INT16 ReadRockerSwState;   // Variable the holds the state
INT16 RdShftCnt;   // Count the number of shifts

//**********************************************[ Private ]*******
//
// Support routines for the Rocker Switch interface
//
//****************************************************************
void RockerClkHi(void) {
GPIOPinWrite(GPIO_PORTJ_BASE, RK_CLK, RK_CLK);                       
}	// end of void RockerClkHi(void) 

void RockerClkLo(void) {
GPIOPinWrite(GPIO_PORTJ_BASE, RK_CLK, 0);                       
 }	// end of void RockerClkLo(void) 

void RockerTxHi(void) {
GPIOPinWrite(GPIO_PORTJ_BASE, RK_OUT, RK_OUT);     
}	// end of void RockerTxHi(void) 

void RockerTxLo(void) {
GPIOPinWrite(GPIO_PORTJ_BASE, RK_OUT, 0);     
}	// end of void RockerTxLo(void) 

void RockerLatchHi(void) {
GPIOPinWrite(GPIO_PORTJ_BASE, RK_LATCH, RK_LATCH);     
}	// end of void RockerLatchHi(void) 

void RockerLatchLo(void) {
GPIOPinWrite(GPIO_PORTJ_BASE, RK_LATCH, 0);     
}	// end of void RockerLatchLo(void) 

INT8 ReadRockerUp(void) {
return((INT8) GPIOPinRead(GPIO_PORTJ_BASE, RK_LL));  
}	// end of INT8 ReadRockerUp(void) 

INT8 ReadRockerDn(void) {
return((INT8) GPIOPinRead(GPIO_PORTJ_BASE, RK_DOWN));  
}	// end of INT8 ReadRockerDn(void) 

INT8 ReadLaneLock(void) {
return((INT8) GPIOPinRead(GPIO_PORTJ_BASE, RK_UP));  
}	// end of INT8 ReadLanLock(void) 

INT8 ReadKeyboard(void) {
return((INT8) GPIOPinRead(GPIO_PORTJ_BASE, KB_IN)); 
}  // end of INT16 ReadKeyboard(void) {

#define  ROCKER_KB_SPI_DELAY    1

//**********************************************[ RTOS ]**********
//
// This delay routine signal the RTOS that we are waiting
//
//****************************************************************
void RockerDelay(void) {
xTaskDelay(ROCKER_KB_SPI_DELAY / portTICK_RATE_MS);
}	// end of void RockerDelay(void) 

// States
#define	ROCKER_SETUP		  1
#define	ROCKER_AT_START	  2
#define	ROCKER_RUNNING		3
#define	ROCKER_AT_END		  4

//**********************************************[ Public ]********
//
// Read Rocker Switches
//    We need a walking pattern of ..1110111..
//    for a length of 24 bits and then repeating.
//    Start off by loading 24 1's then start the work.
//
//*****************************************************************
void ReadRockerSwitches(void) {
INT16 i, temp;

switch (ReadRockerSwState) {
   
   default: {
      PortInitRockerSwitches();
      ReadRockerSwState = ROCKER_SETUP;
   } break;
   
   case ROCKER_SETUP: {  // Setup
      RockerClkLo();	    // Clock Low
      RockerTxHi();      // Data Hi
      RockerLatchHi();   // Latch H1
      RockerDelay();		 // Min Clock Time 
      for (i = 0; i < 30; i++) {
         RockerClkHi();	 // Clock Hi
         RockerDelay();
      	 RockerClkLo();	 // Clock Low
         RockerDelay();
      }
      RockerLatchLo();   // Latch Lo
      RockerDelay();
      RockerLatchHi();   // Latch Hi
      RockerDelay();
      ReadRockerSwState = ROCKER_AT_START;
   } break;
   
   case ROCKER_AT_START: {  // Shift 0 one time
      RdShftCnt = 0;
      RockerTxLo();     // Data Lo
      RockerDelay();   
      RockerClkHi();	   // Clock Hi
      RockerDelay();   
    	RockerClkLo();	   // Clock Low
      RockerDelay();
      RockerTxHi();     // Data Hi
      RockerLatchLo();  // Latch Lo
      RockerDelay();
      RockerLatchHi();  // Latch Hi
      RockerDelay();
      ReadRockerSwState = ROCKER_RUNNING;
   } break;
   
   case ROCKER_RUNNING: {   // Read results shift out 1...24 times
      temp = ReadRockerUp();    
      if (temp == 0) {
      	MasterSwitchData.Rocker[RdShftCnt]   = 'U';
      }
      else {
         temp = ReadRockerDn();
         if (temp == 0) {
            MasterSwitchData.Rocker[RdShftCnt]   = 'D';
         }
         else {
            MasterSwitchData.Rocker[RdShftCnt] = 'O';
         }
      }

      temp = ReadLaneLock();
      if (temp == 0) {
         MasterSwitchData.LaneLock[RdShftCnt] = 'L';
      }
      else {
       	MasterSwitchData.LaneLock[RdShftCnt] = 'U';
      }
         
      temp = ReadKeyboard();    
      if (temp == 0) {
         MasterSwitchData.Keypad[RdShftCnt] = 'C';
      }
      else {
         MasterSwitchData.Keypad[RdShftCnt] = 'O';
      }
      if (++RdShftCnt >= CHNL_MAX) {	
         ReadRockerSwState = ROCKER_AT_END;
      }	      
      else {
	      RockerTxHi();     // Data Hi
        RockerDelay();   
	      RockerClkHi();	   // Clock Hi
   	    RockerDelay();   
	    	RockerClkLo();	   // Clock Low
	      RockerDelay();
	      RockerLatchLo();  // Latch Lo
   	    RockerDelay();
      	RockerLatchHi();  // Latch Hi
      	RockerDelay();       
      }      
   } break;

   case ROCKER_AT_END: {  // Shift 0 out past end
      RockerTxHi();     // Data Hi
      RockerDelay();   
      RockerClkHi();	   // Clock Hi
  	  RockerDelay();   
    	RockerClkLo();	   // Clock Low
      RockerDelay();
      RockerLatchLo();  // Latch Lo
  	  RockerDelay();
     	RockerLatchHi();  // Latch Hi
     	RockerDelay();       
		  ReadRockerSwState = ROCKER_AT_START;
   } break;
   
}  // end of switch (ReadRockerSwState) {
   
}  // end of void ReadRockerSwitches(void) 


//**********************************************[ RTOS ]**********
//
// Operate the Rocker Switch and Keyboard State Machines 
//     from an RTOS.
// This routine is the Task and is an infinite loop
//
//*****************************************************************
static void ReadRockerKeyboardTask(void *pvParameters) {
   
for (;;) {
   ReadRockerSwitches(); // operate the state machine
}

}  // end of static void ReadRockerKeyboardTask(void *pvParameters)


//*****************************************************************************
//
// The stack for the Switch task.
//
//*****************************************************************************
static unsigned long g_pulRckrKbTaskStack[128];


//**********************************************[ Public ]********
//
// Create the Rocker Switch and Keyboard Task
//
//*****************************************************************
unsigned long ReadRockerKeyboardTaskInit(void) {
   
PortInitRockerSwitches();  // Init the hardware
  
// Create the RockerKeyboard task.
if (xTaskCreate( ReadRockerKeyboardTask, 
                (signed portCHAR *)"ROCK_KB",
                (signed portCHAR *)g_pulRckrKbTaskStack,
                 sizeof(g_pulRckrKbTaskStack), 
                 NULL, 
                 PRIORITY_RKB_TASK,
                 NULL) != pdPASS)
{
   return(1);  // Failure
}

   
return(0);  // Success.

}  // end of unsigned long ReadRockerKeyboardTaskInit(void) {


// end of file SpiRockerKbTasks.c

