/***********************************************************************
 *
 * This file is part of the OnBoard C package.  For more information on 
 * using OnBoard C, see http://groups.yahoo.com/group/onboardc.  For 
 * more information on developing OnBoard C (including submission of 
 * bug reports), see http://sourceforge.net/projects/onboardc.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 ***********************************************************************/

// For PalmOS 4.0 compatibility
#define ALLOW_ACCESS_TO_INTERNALS_OF_WINDOWS
#define ALLOW_ACCESS_TO_INTERNALS_OF_FORMS

#include <PalmOS.h>
#include <PalmCompatibility.h>
#include <SysEvtMgr.h>


#include "OnBoardC.h"
#include "../../OnBoard/Src/debugTypes.h"
#include "OnBoardC_res.h"

#define PROJECT_VERSION 4
#define PROJECT_VERSION_NEW 5
#define PROJECT_VERSION_CURRENT 6
#define PROJECT_VERSION_NEWOBJ 7
#define PROJECT_TYPE 'OBPJ'

#define OffsetOf(type, member)	((UInt32) &(((type *) 0)->member))


#define EXECUTE_FLAG (0x1)
#define REBUILD_FLAG (0x2)
#define DEBUG_FLAG (0x4)
#define AUTOVERSION_FLAG (0x8)



DmOpenRef gExprDB;
DmOpenRef gGlobalsDB;


LocalID gProjectDBid;
ProjectDetails gProjectHeader;
DmOpenRef gProjectDB;

Handle outputHandle;
UInt outputRecordNumber;
ULong outputSize;

DmOpenRef gAsmDB;

PrefsData gPrefs;

char **gProjectListContents = NULL;
int gProjectListCount;

char **gProjectContentsListContents = NULL;
int gProjectContentsListCount;

RectangleType gFullScreen = { {0, 0}, {160, 160} };
int gSourceIndex;


int errorCount = 0;

char **memoSourceList = NULL;
long *memoSourceIndexList = NULL;
int memoSourceCount = 0;

char **pedit32SourceList = NULL;
long *pedit32SourceIndexList = NULL;
int pedit32SourceCount = 0;

char **docSourceDBList = NULL;
int docSourceDBCount;
char **resourceDBList = NULL;
int resourceDBCount;

char **memoSourceDisplayList = NULL;
long *memoSourceDisplayIndexList = NULL;
int memoSourceDisplayCount;

char **pedit32SourceDisplayList = NULL;
long *pedit32SourceDisplayIndexList = NULL;
int pedit32SourceDisplayCount;

char gFormTitle[64];

DWord gRomVersion;
#define version35 0x03500000

/*********************************************************************************************/

VoidPtr getObjectPtr(FormPtr frmP, Word objectID)
{
	return (FrmGetObjectPtr(frmP, FrmGetObjectIndex(frmP, objectID)));
}

static void setHandle(FormPtr frmP, int fieldID, Handle newH, Boolean editable)
{
	FieldAttrType attr;
	FieldPtr fld = (FieldPtr)getObjectPtr(frmP, fieldID);
	Handle oldH = (Handle)FldGetTextHandle(fld);

	MemHandleUnlock(newH);
	FldSetText(fld, newH, 0, MemHandleSize(newH));

	FldGetAttributes(fld, &attr);
	attr.editable = editable;
	FldSetAttributes(fld, &attr);

	if (frmP == FrmGetActiveForm()) FldDrawField(fld);
	if (oldH != NULL) MemHandleFree(oldH);
}

void setHandleFromChar4(FormPtr frmP, int fieldID, ULong value, Boolean editable)
{
	Handle h = (Handle)MemHandleNew(6);
	CharPtr p = (CharPtr)MemHandleLock(h);
	*((ULong *)p) = value;
	p[4] = 0;

	setHandle(frmP, fieldID, h, editable);
}

void setHandleFromText(FormPtr frmP, int fieldID, CharPtr text, Boolean editable)
{
	Handle h;
	CharPtr p;
	int size = 0;
	
	if (text != NULL) 
		size = StrLen(text);
	size++;
	h = (Handle)MemHandleNew(size);
	p = (CharPtr)MemHandleLock(h);
	if (text != NULL)
		StrCopy(p, text);
	else
		*p = 0;

	setHandle(frmP, fieldID, h, editable);
}

void setTextFromHandle(FormPtr frmP, int fieldID, char *value)
{
	FieldPtr fld = (FieldPtr)getObjectPtr(frmP, fieldID);
	Handle h = FldGetTextHandle(fld);
	if (h != NULL) {
		CharPtr p = (CharPtr)MemHandleLock(h);
		StrCopy(value, p);
		MemHandleUnlock(h);
	}
	else
		*value = '\0';
}

void setChar4FromHandle(FormPtr frmP, int fieldID, ULong *value)
{
	FieldPtr fld = (FieldPtr)getObjectPtr(frmP, fieldID);
	Handle h = FldGetTextHandle(fld);
	if (h != NULL) {
		CharPtr p = (CharPtr)MemHandleLock(h);
		*value = *((ULong *)p);
		MemHandleUnlock(h);
	}
	else {
		*value = '    ';
	}
}

void error(char *errMessage)
{
	char lineNum[10];
	StrPrintF(lineNum, "%d", gLineCount);
	FrmCustomAlert(ErrorAlert, errMessage, lineNum, NULL);
	errorCount++;
}

void sourceError(char *errMessage, CharPtr sourcePtr)
{
	if (sourcePtr == NULL)
		error(errMessage);
	else {
		char lineNum[10];
		int button;
		StrPrintF(lineNum, "%d", gLineCount);
		button = FrmCustomAlert(SourceErrorAlert, errMessage, lineNum, NULL);
		errorCount++;
		if (button == SourceErrorGoto) {
			/*
				if we're in a macro, try to point at the macro call site instead
			*/
			if (textStackTop > 0)
				sourcePtr = curTextPtrStack[0];
			if ((gIncludeTop > 0) &&
					(sourcePtr > includeData[gIncludeTop - 1].includePtr) 
						&& (sourcePtr < (includeData[gIncludeTop - 1].includePtr + includeData[gIncludeTop - 1].includeLength))) {
				if (includeData[gIncludeTop - 1].includeKind == MemoSource)
					gotoMemoPad(sourcePtr - includeData[gIncludeTop - 1].includePtr, 0, includeData[gIncludeTop - 1].includeDBid);
				else
					if (includeData[gIncludeTop - 1].includeKind == Pedit32Source)
						gotoPedit32(sourcePtr - includeData[gIncludeTop - 1].includePtr, 0, includeData[gIncludeTop - 1].includeDBid);
					else
						gotoEditor(sourcePtr, 0, 0, NULL);
			}
			else
				if (gProjectHeader.source[gSourceIndex].sourceKind == MemoSource)
					gotoMemoPad(sourcePtr - gMainSourceBasePtr, 0, getSourceDBid());
				else
					if (gProjectHeader.source[gSourceIndex].sourceKind == Pedit32Source)
						gotoPedit32(sourcePtr - gMainSourceBasePtr, 0, getSourceDBid());
					else
						gotoEditor(sourcePtr, 0, 0, NULL);
		}
	}
}

void sourceErrorWithLength(char *errMessage, CharPtr sourcePtr, int length)
{
	char buf[256];
	int i;
	int t = StrLen(errMessage);
	int button;
	char lineNum[10];
	StrPrintF(lineNum, "%d", gLineCount);
	
	if ((t + length + 4) < 256) {
		StrCopy(buf, errMessage);
		StrCat(buf, " '");
		for (i = 0; i < length; i++)
			buf[t + 2 + i] = sourcePtr[i];
		buf[t + 2 + i] = '\'';				// balance '
		buf[t + 2 + i + 1] = 0;
		button = FrmCustomAlert(SourceErrorAlert, buf, lineNum, NULL);
	}
	else
		button = FrmCustomAlert(SourceErrorAlert, errMessage, lineNum, NULL);
	errorCount++;
	if (button == SourceErrorGoto) {
		if ((gIncludeTop > 0) &&
				(sourcePtr > includeData[gIncludeTop - 1].includePtr) 
					&& (sourcePtr < (includeData[gIncludeTop - 1].includePtr + includeData[gIncludeTop - 1].includeLength))) {
			if (includeData[gIncludeTop - 1].includeKind == MemoSource)
				gotoMemoPad(sourcePtr - includeData[gIncludeTop - 1].includePtr, length, includeData[gIncludeTop - 1].includeDBid);
			else
				if (includeData[gIncludeTop - 1].includeKind == Pedit32Source)
					gotoPedit32(sourcePtr - includeData[gIncludeTop - 1].includePtr, length, includeData[gIncludeTop - 1].includeDBid);
				else
					gotoEditor(sourcePtr, length, 0, NULL);
		}
		else
			if (gProjectHeader.source[gSourceIndex].sourceKind == MemoSource)
				gotoMemoPad(sourcePtr - gMainSourceBasePtr, length, getSourceDBid());
			else
				if (gProjectHeader.source[gSourceIndex].sourceKind == Pedit32Source)
					gotoPedit32(sourcePtr - gMainSourceBasePtr, length, getSourceDBid());
				else
					gotoEditor(sourcePtr, length, 0, NULL);
	}
}

void eraseTitle()
{
	RectangleType r = {{0, 0}, {160, 15} };
	WinEraseRectangle(&r, 0);
}

LocalID setDefaultProject(Boolean untitled)
{
	DmOpenRef projDB;
	Handle h;
	LocalID dbID;
	int i = 0;
	Err err;
	UInt16	dbattrs;

	if (untitled) {
		while (1) {
			StrCopy(gPrefs.projectName, "Untitled");
			if (i > 0)
				StrIToA(gPrefs.projectName + 8, i);
			StrCat(gPrefs.projectName, ".proj");
			dbID = DmFindDatabase(0, gPrefs.projectName);
			if (dbID != 0)
				i++;
			else
				break;
		}
	}
	else {
		dbID = DmFindDatabase(0, gPrefs.projectName);
		if (dbID != 0)
			DmDeleteDatabase(0, dbID);
	}
			
	err = DmCreateDatabase(0, gPrefs.projectName, 'OnBD', 'Proj', true);			
	if (err != 0) return 0;	// can't happen?
	
	dbID = DmFindDatabase(0, gPrefs.projectName);
	// Set the Backup bit.
	if (DmDatabaseInfo (0, dbID, NULL, &dbattrs, NULL, NULL, NULL, NULL,
		NULL, NULL, NULL, NULL, NULL) == 0) {
		dbattrs |= dmHdrAttrBackup;
		(void) DmSetDatabaseInfo (0, dbID, NULL, &dbattrs, NULL, NULL, 
			NULL, NULL, NULL, NULL, NULL, NULL, NULL);
	}
	projDB = DmOpenDatabase(0, dbID, dmModeReadWrite);
	h = DmNewResource(projDB, PROJECT_TYPE, 1, sizeof(ProjectHeader_New));
	
	gProjectHeader.prj.mVersion = PROJECT_VERSION_NEWOBJ;
	gProjectHeader.prj.mSourceDBCount = 0;
	gProjectHeader.prj.mFlags = 0;
	gProjectHeader.prj.mType = 'appl';
	gProjectHeader.prj.mCreator = 'MyAP';
	StrCopy(gProjectHeader.prj.mAppName, "MyApplication");
	StrCopy(gProjectHeader.prj.mPrcName, "MyPRC");
	gProjectHeader.prj.mIncludeCount = 0;
	
	DmWrite(MemHandleLock(h), 0, &gProjectHeader, sizeof(ProjectHeader_New));
	
	MemHandleUnlock(h);
	DmReleaseResource(h);
	DmCloseDatabase(projDB);
	return dbID;
}

void deleteProjectFiles(ProjectHeader_New *projHdr, Boolean setDBid)
{
	int i;
	SourceFile *src = (SourceFile *)(projHdr + 1);
	for (i = 0; i < projHdr->mSourceDBCount; i++) {
		if (src[i].sourceKind != Resource) {
			LocalID dbID = DmFindDatabase(0, src[i].outputDB);
			if (dbID) DmDeleteDatabase(0, dbID);
			if (setDBid) src[i].outputDBid = 0;
		}		
	}
}

void deleteGlobalData()
{
	LocalID dbID = DmFindDatabase(0, "CompilerGlobals.OnBC");
	if (dbID != 0) DmDeleteDatabase(0, dbID);
}

void initGlobalData()
{
	UInt globalsAt = 0;
	int err;
	LocalID dbID;
		
	err = DmCreateDatabase(0, "CompilerGlobals.OnBC", 'OnBC', 'DAT1', false);
	if (err != 0) {
		char buf[64];
		StrPrintF(buf, "error on create = %d", err);
		error(buf);
	}
	else {
		dbID = DmFindDatabase(0, "CompilerGlobals.OnBC");
		gGlobalsDB = DmOpenDatabase(0, dbID, dmModeReadWrite);
		globalTableH = DmNewRecord(gGlobalsDB, &globalsAt, MaxGlobal * sizeof(Declaration));
		globalRecordIndex = globalsAt;
		if (globalTableH == NULL) {
			error("Couldn't allocate global table space");
		}
		else
			globalTable = MemHandleLock(globalTableH);
	}
    globalTop = 0;
    fullGlobalCount = 0;
}

Boolean startApplication()
{
	Word prefSize = sizeof(PrefsData);
	int err;

	LocalID dbID = DmFindDatabase(0, "CompilerExpr.OnBC");
	if (dbID != 0) DmDeleteDatabase(0, dbID);
	err = DmCreateDatabase(0, "CompilerExpr.OnBC", 'OnBC', 'DAT1', false);
	if (err != 0) {
		char buf[64];
		StrPrintF(buf, "error on create = %d", err);
		error(buf);
	}
	dbID = DmFindDatabase(0, "CompilerExpr.OnBC");
	gExprDB = DmOpenDatabase(0, dbID, dmModeReadWrite);

	deleteGlobalData();
	initGlobalData();	
	
	FtrGet(sysFtrCreator, sysFtrNumROMVersion, &gRomVersion);
	
	gProjectDBid = 0;

	//Check for OnBC preferences file if not found create new one with defaults,
	//docEditor = SrcEdit, memoEditor = Pedit, addSourceType = Resource.
	if ((PrefGetAppPreferences('OnBC', 1000, &gPrefs, &prefSize, false) == noPreferenceFound) 
			|| (prefSize != sizeof(PrefsData))) {
		gPrefs.docEditor = SrcEdit;
		gPrefs.memoEditor = Pedit;
		gPrefs.addSourceType = Resource;
	}
	else
		gProjectDBid = DmFindDatabase(0, gPrefs.projectName);
	
	//Verify docEditor preference settings, if not set, set to SrcEdit.
	if ((gPrefs.docEditor != SrcEdit) 
			&& (gPrefs.docEditor != QED) 
				&& (gPrefs.docEditor != ZDoc) 
					&& (gPrefs.docEditor != SmartEdit)
						&& (gPrefs.docEditor != WordSmith_Doc)
							&& (gPrefs.docEditor != QuickWord)
				) gPrefs.docEditor = SrcEdit;
	//Verify memoEditor preference settings, if not set, set to Pedit.
	if ((gPrefs.memoEditor != Pedit) 
			&& (gPrefs.memoEditor != MemoPad)
				&& (gPrefs.memoEditor != WordSmith_Memo)
			) gPrefs.memoEditor = Pedit;
	
	if (gProjectDBid != 0) {
		return loadProject();
	}
	else
		return false;
}

void closeProjectDB()
{
	if (gProjectDBid) {
//		UInt attr;
		char *base;
		int i;
		
		DmOpenRef proj = DmOpenDatabase(0, gProjectDBid, dmModeReadWrite);
		Handle h = DmGet1Resource(PROJECT_TYPE, 1);
		DmResizeResource(h, sizeof(ProjectHeader_New) + gProjectHeader.prj.mSourceDBCount * sizeof(SourceFile)
															+ gProjectHeader.prj.mIncludeCount * sizeof(IncludeFile));
		base = MemHandleLock(h);
		
		DmWrite(base, 0, &gProjectHeader.prj, sizeof(ProjectHeader_New));
		for (i = 0; i < gProjectHeader.prj.mSourceDBCount; i++)
			DmWrite(base, sizeof(ProjectHeader_New) + (i * sizeof(SourceFile)), &gProjectHeader.source[i], sizeof(SourceFile));
		for (i = 0; i < gProjectHeader.prj.mIncludeCount; i++)
			DmWrite(base, sizeof(ProjectHeader_New) + (gProjectHeader.prj.mSourceDBCount * sizeof(SourceFile)) + (i * sizeof(IncludeFile)),
												 &gProjectHeader.include[i], sizeof(IncludeFile));

		MemHandleUnlock(h);
		DmReleaseResource(h);
		DmCloseDatabase(proj);
	
//		DmDatabaseInfo(0, gProjectDBid, 0, &attr, 0, 0, 0, 0, 0, 0, 0, 0, 0);
//		attr |= dmHdrAttrBackup;
//		DmSetDatabaseInfo(0, gProjectDBid, 0, &attr, 0, 0, 0, 0, 0, 0, 0, 0, 0);
	}
	
}

static void deleteMemoSourceList()
{
	if (memoSourceList) {
		int i;
		for (i = 0; i < memoSourceCount; i++)		
			MemPtrFree(memoSourceList[i]);
		MemPtrFree(memoSourceList);
		memoSourceList = NULL;
	}
	if (memoSourceIndexList) {
		MemPtrFree(memoSourceIndexList);
		memoSourceIndexList = NULL;
	}
}

static void deletePedit32SourceList()
{
	if (pedit32SourceList) {
		int i;
		for (i = 0; i < pedit32SourceCount; i++)		
			MemPtrFree(pedit32SourceList[i]);
		MemPtrFree(pedit32SourceList);
		pedit32SourceList = NULL;
	}
	if (pedit32SourceIndexList) {
		MemPtrFree(pedit32SourceIndexList);
		pedit32SourceIndexList = NULL;
	}
}

void stopApplication()
{
	LocalID dbID;
	int i;
	closeProjectDB();
	
	DmCloseDatabase(gExprDB);
	dbID = DmFindDatabase(0, "CompilerExpr.OnBC");
	if (dbID != 0) DmDeleteDatabase(0, dbID);
	
	if (fullGlobalCount > 0) {
		for (i = 0; i < fullGlobalCount; i++) {
			MemHandleUnlock(fullGlobalTableH[i]);
			DmReleaseRecord(gGlobalsDB, fullGlobalRecordIndex[i], false);
		}
	}
	MemHandleUnlock(globalTableH);
	DmReleaseRecord(gGlobalsDB, globalRecordIndex, false);
	DmCloseDatabase(gGlobalsDB);
	deleteGlobalData();
	
	FrmCloseAllForms();

	if (docSourceDBList) {
		for (i = 0; i < docSourceDBCount; i++)
			MemPtrFree(docSourceDBList[i]);
		MemPtrFree(docSourceDBList);
	}
	if (resourceDBList) {
		for (i = 0; i < resourceDBCount; i++)
			MemPtrFree(resourceDBList[i]);
		MemPtrFree(resourceDBList);
	}
	if (memoSourceDisplayList) {
		for (i = 0; i < memoSourceDisplayCount; i++)
			MemPtrFree(memoSourceDisplayList[i]);
		MemPtrFree(memoSourceDisplayList);
		memoSourceDisplayList = NULL;
	}
	if (memoSourceDisplayIndexList)  {
		MemPtrFree(memoSourceDisplayIndexList);
		memoSourceDisplayIndexList = NULL;
	}
	if (gProjectContentsListContents) {
		for (i = 0; i < gProjectContentsListCount; i++)
			MemPtrFree(gProjectContentsListContents[i]);
		MemPtrFree(gProjectContentsListContents);
	}
	if (gProjectListContents) {
		for (i = 0; i < gProjectListCount; i++)
			MemPtrFree(gProjectListContents[i]);
		MemPtrFree(gProjectListContents);
	}
	deleteMemoSourceList();
	deletePedit32SourceList();
}

Boolean validateProjectEntries()
{
	FormPtr frmP = FrmGetActiveForm();
	ControlPtr ctlP = (ControlPtr)getObjectPtr(frmP, ProjectExecuteCheckbox);
	
	setTextFromHandle(frmP, ProjectPrcNameField, gProjectHeader.prj.mPrcName);
	// make sure they're not empty ? or use default values ?
	setChar4FromHandle(frmP, ProjectCreatorField, &gProjectHeader.prj.mCreator);
	setChar4FromHandle(frmP, ProjectTypeField, &gProjectHeader.prj.mType);
	
	gProjectHeader.prj.mFlags = 0;
	if (CtlGetValue(ctlP)) gProjectHeader.prj.mFlags |= EXECUTE_FLAG;
		
	ctlP = (ControlPtr)getObjectPtr(frmP, ProjectAlwaysRebuildCheckbox);
	if (CtlGetValue(ctlP)) gProjectHeader.prj.mFlags |= REBUILD_FLAG;
	
	ctlP = (ControlPtr)getObjectPtr(frmP, ProjectDebugCheckbox);
	if (CtlGetValue(ctlP)) gProjectHeader.prj.mFlags |= DEBUG_FLAG;
	
	ctlP = (ControlPtr)getObjectPtr(frmP, ProjectAutoVersionCheckbox);
	if (CtlGetValue(ctlP)) gProjectHeader.prj.mFlags |= AUTOVERSION_FLAG;
	
	return true;
}

void setStatus(CharPtr message, CharPtr status, int length)
{	
	FormPtr frmP = FrmGetActiveForm();
	char buf[64];
	int i, t;
	
	StrCopy(buf, message);
	t = StrLen(buf);
	for (i = 0; i < length; i++)
		buf[t + i] = status[i];
	buf[t + length] = '\0';
	
	setHandleFromText(frmP, ProjectStatusField, buf, false);
}

/*
LocalID openAsmFile(CharPtr asmName)
{
	LocalID dbID;
	dbID = DmFindDatabase(0, asmName);
	if (dbID != 0) {
		Err err = DmDeleteDatabase(0, dbID);
		if (err != 0) {
			error("Couldn't delete previous assembler source file");
		}
	}
	if (dbID != -1) {
		Err err;
		ULong type = 'OBJc';
		err = DmCreateDatabase(0, asmName, 'OnBC', type, false);
		if (err != 0)
			error("Couldn't create assembler source file");
		else {
			dbID = DmFindDatabase(0, asmName);
			gAsmDB = DmOpenDatabase(0, dbID, dmModeReadWrite);
			outputRecordNumber = 1;
			outputSize = 4096;
			outputHandle = DmNewRecord(gAsmDB, &outputRecordNumber, outputSize);
			if (outputHandle == NULL)  {
				error("Out of memory for output buffer");
				return 0;
			}
			outputBuffer = MemHandleLock(outputHandle);
			totalOutputSize = 0;
			outputTop = 0;
		}
	}
	return dbID;
}

void closeAsmFile()
{
	UInt at = 0;
	long fakeHeader[4];
	Handle h;
	CharPtr p;

	MemHandleUnlock(outputHandle);
	DmResizeRecord(gAsmDB, outputRecordNumber, outputTop);
	DmReleaseRecord(gAsmDB, outputRecordNumber, true);
	totalOutputSize += outputTop;

	h = DmNewRecord(gAsmDB, &at, 16);
	p = MemHandleLock(h);
	fakeHeader[0] = 0x00010000;
	fakeHeader[1] = totalOutputSize;
	fakeHeader[2] = ((long)(outputRecordNumber + 1) << 16) + 0x1000;
	fakeHeader[3] = 0x00000000;
	DmWrite(p, 0, &fakeHeader, 16);
	MemHandleUnlock(h);
	DmReleaseRecord(gAsmDB, at, true);
	DmCloseDatabase(gAsmDB);
}
*/
LocalID openAsmFile(CharPtr asmName)
{
	LocalID dbID;
	dbID = DmFindDatabase(0, asmName);
	if (dbID != 0) {
		Err err = DmDeleteDatabase(0, dbID);
		if (err != 0) {
			error("Couldn't delete previous assembler source file");
		}
	}
	if (dbID != -1) {
		Err err;
		ULong type = 'OBJc';
		err = DmCreateDatabase(0, asmName, 'OnBC', type, false);
		if (err != 0)
			error("Couldn't create assembler source file");
		else {
			dbID = DmFindDatabase(0, asmName);
			gAsmDB = DmOpenDatabase(0, dbID, dmModeReadWrite);
			outputRecordNumber = 0;
			outputSize = 4096;
			outputHandle = DmNewRecord(gAsmDB, &outputRecordNumber, outputSize);
			if (outputHandle == NULL)  {
				error("Out of memory for output buffer");
				return 0;
			}
			outputBuffer = MemHandleLock(outputHandle);
			outputTop = 0;
		}
	}
	return dbID;
}

void closeAsmFile()
{
	addTokenToOutput(Asm_NotACharacter);		// make sure it's zero-terminated
	MemHandleUnlock(outputHandle);
	DmResizeRecord(gAsmDB, outputRecordNumber, outputTop);
	DmReleaseRecord(gAsmDB, outputRecordNumber, true);
	DmCloseDatabase(gAsmDB);
}

char *getSourceName()
{
	return gProjectHeader.source[gSourceIndex].sourceDB;
}

LocalID getSourceDBid()
{
	return gProjectHeader.source[gSourceIndex].sourceDBid;
}

void getAsmName(char *sourceName, char *asmName, int asmIndex)
{
//
// XXX should length-check the result for dmDBNameLength
//
	char *extension = StrStr(sourceName, ".c");
	int length = StrLen(sourceName);
	StrCopy(asmName, sourceName);
	if (extension == (sourceName + length - 2))
		asmName[length - 2] = '\0';
	if (asmIndex > 0) {
		char buf[10];
		StrIToA(buf, asmIndex);
		StrCat(asmName, buf);
	}
	StrCat(asmName, ".obj");
}

Boolean launchAssembler(LocalID asmDBId)
{
	GoToParamsType *pBlock;
	UInt					cardNo;
	LocalID				dbID;
	DmSearchStateType	searchState;
	Err err;
	//UInt pblockAt = 0;

	pBlock = (GoToParamsType *)MemPtrNew(sizeof(GoToParamsType));
	MemPtrSetOwner(pBlock, 0);
	
	pBlock->searchStrLen = 0;		// length of search string.
	pBlock->dbCardNo = 0;			// card number of the database	
	pBlock->dbID = asmDBId;			// LocalID of the database
	pBlock->recordNum = -1;			// index of record that contain a match
	pBlock->matchPos = 0;			// postion in record of the match.
	pBlock->matchFieldNum = 0;		// field number string was found int
	pBlock->matchCustom = (((gProjectHeader.prj.mFlags & EXECUTE_FLAG) == EXECUTE_FLAG) << 1);
	
	err = DmGetNextDatabaseByTypeCreator(true, &searchState, 'appl', 'OnBA', true, &cardNo, &dbID);
	if (err != 0) {
		error("Can't find assembler");
		return false;
	}
	if (dbID) {
		err = SysUIAppSwitch(cardNo, dbID, sysAppLaunchCmdGoTo, (Ptr)pBlock);
		if (err != 0) {
			error("Can't launch assembler");
			return false;
		}
	}
	
	return true;
}

ULong getModDate(LocalID dbID)
{
	ULong result = 0;
	if (dbID)
		DmDatabaseInfo(0, dbID, 0, 0, 0, 0, &result, 0, 0, 0, 0, 0, 0);
	return result;
}

Boolean matchInclude(IncludeData *incD, IncludeFile *incF)
{
	if ((incD->includeKind == incF->includeKind)
			&& (incD->includeNameLength == incF->includeNameLength)) {
		int x;
		for (x = 0; x < incD->includeNameLength; x++) {
			if (incD->includeName[x] != incF->includeDB[x])
				return false;
		}
		return true;
	}
	return false;
}

void markInclude(IncludeData *incD)
{
	int i, x;
	IncludeFile *incF;
	SourceFile *sf = &gProjectHeader.source[gSourceIndex];
	for (i = 0; i < sf->includeCount; i++) {
		int index = sf->includeIndex[i];
		if (matchInclude(incD, &gProjectHeader.include[index]))
			return;		// already marked for this source file
	}
	
	for (i = 0; i < gProjectHeader.prj.mIncludeCount; i++) {
		if (matchInclude(incD, &gProjectHeader.include[i])) {
			sf->includeIndex[sf->includeCount++] = i;
			return;		// include file in use for some other source
		}
	}

	if ((gProjectHeader.prj.mIncludeCount < MAX_SOURCE)
			&& (sf->includeCount < MAX_SOURCE)) {
		sf->includeIndex[sf->includeCount++] = gProjectHeader.prj.mIncludeCount;
		incF = &gProjectHeader.include[gProjectHeader.prj.mIncludeCount++];
		
		for (x = 0; x < incD->includeNameLength; x++)
			incF->includeDB[x] = incD->includeName[x];
		incF->includeDB[x] = '\0';
		incF->includeNameLength = incD->includeNameLength;
		incF->includeKind = incD->includeKind;
	}
}

Boolean buildProject(ListPtr lstP)
{
	int i;
	Boolean result = true;
	Handle sourceH;

	stringPoolIndex = 0;	
	codeStringPoolIndex = 0;	

	for (i = 0; i < gProjectHeader.prj.mSourceDBCount; i++) {
		gSourceIndex = i;

		if (gProjectHeader.source[i].sourceKind == DocSource) {
			ULong srcMod, objMod;
			Boolean doBuild = (gProjectHeader.prj.mFlags & REBUILD_FLAG) == REBUILD_FLAG;
			if (!doBuild) {
				srcMod = getModDate(gProjectHeader.source[i].sourceDBid);
				objMod = getModDate(gProjectHeader.source[i].outputDBid);
				doBuild = (srcMod >= objMod);
			}
			if (!doBuild) {
				int j;
				for (j = 0; j < gProjectHeader.source[i].includeCount; j++) {
					int index = gProjectHeader.source[i].includeIndex[j];
					LocalID incID = DmFindDatabase(0, gProjectHeader.include[index].includeDB);
					if (incID) {
						srcMod = getModDate(incID);
						doBuild = (srcMod >= objMod);
					}
					else
						doBuild = true;
					if (doBuild) break;
				}
			}
			
			if (doBuild) {
				LstSetSelection(lstP, i);
				reallocateOutputDBRecords();	
				sourceH = getSourceText(gProjectHeader.source[i].sourceDBid);
				gProjectHeader.source[i].outputDBid = openAsmFile(gProjectHeader.source[i].outputDB);
				gProjectHeader.source[i].includeCount = 0;
#if DEBUG_SUPPORT
				result = doCompile(sourceH, NULL, NULL, ((gProjectHeader.prj.mFlags & DEBUG_FLAG) == DEBUG_FLAG) );
#else
				result = doCompile(sourceH, NULL, NULL, false );
#endif
				closeAsmFile();
				if (!result) {
					DmDeleteDatabase(0, gProjectHeader.source[i].outputDBid);
					gProjectHeader.source[i].outputDBid = 0;
					break;
				}
			}
			stringPoolIndex += 1024;	/* increment this irregardless of whether we compile or not */
			codeStringPoolIndex += 1024;	/* increment this irregardless of whether we compile or not */
		}
		else {
			if (gProjectHeader.source[i].sourceKind == MemoSource) {
				LstSetSelection(lstP, i);
				reallocateOutputDBRecords();	
				sourceH = getMemoText(gProjectHeader.source[i].sourceDBid);
				gProjectHeader.source[i].outputDBid = openAsmFile(gProjectHeader.source[i].outputDB);	
				gProjectHeader.source[i].includeCount = 0;
#if DEBUG_SUPPORT
				result = doCompile(sourceH, NULL, NULL, ((gProjectHeader.prj.mFlags & DEBUG_FLAG) == DEBUG_FLAG));
#else
				result = doCompile(sourceH, NULL, NULL, false);
#endif
				closeAsmFile();
				if (!result) {
					DmDeleteDatabase(0, gProjectHeader.source[i].outputDBid);
					gProjectHeader.source[i].outputDBid = 0;
					break;
				}
				stringPoolIndex += 1024;
				codeStringPoolIndex += 1024;
			}
			else
				if (gProjectHeader.source[i].sourceKind == Pedit32Source) {
					LstSetSelection(lstP, i);
					reallocateOutputDBRecords();	
					sourceH = getPedit32Text(gProjectHeader.source[i].sourceDBid);
					gProjectHeader.source[i].outputDBid = openAsmFile(gProjectHeader.source[i].outputDB);	
					gProjectHeader.source[i].includeCount = 0;
#if DEBUG_SUPPORT
					result = doCompile(sourceH, NULL, NULL, ((gProjectHeader.prj.mFlags & DEBUG_FLAG) == DEBUG_FLAG));
#else
					result = doCompile(sourceH, NULL, NULL, false);
#endif
					closeAsmFile();
					if (!result) {
						DmDeleteDatabase(0, gProjectHeader.source[i].outputDBid);
						gProjectHeader.source[i].outputDBid = 0;
						break;
					}
					stringPoolIndex += 1024;
					codeStringPoolIndex += 1024;
				}
		}
	
	}
	
	if (result)
		launchAssembler(gProjectDBid);
	
	return true;
}

int countCTextRecords(UInt numRecords, DmOpenRef memoDB, Boolean *isCText)
{
	int i, cTextCount = 0;
	for (i = 0; i < numRecords; i++) {
		Handle h = DmGetRecord(memoDB, i);
		isCText[i] = false;
		if (h != NULL) {
			CharPtr p = MemHandleLock(h);
			if ((p[0] == '/') && (p[1] == '*')) {
				isCText[i] = true;
				cTextCount++;
			}
			MemHandleUnlock(h);
			DmReleaseRecord(memoDB, i, false);
		}
	}
	return cTextCount;
}

int identifyCTextRecords(UInt numRecords, DmOpenRef memoDB, Boolean *isCText, char **memoSourceList, long *memoSourceIndexList)
{
	int i, memoSourceCount = 0;

	for (i = 0; i < numRecords; i++) {
		if (isCText[i]) {
			int length = 0;
			Handle h = DmGetRecord(memoDB, i);
			CharPtr p = MemHandleLock(h);
			CharPtr startPtr = NULL;
			p += 2;		// step over the opening comment "/*"
			while ((*p == ' ') || (*p == '\t')) p++;
			while (((*p >= 'A') && (*p <= 'Z')) 
							|| ((*p >= 'a') && (*p <= 'z'))
							|| ((*p >= '0') && (*p <= '9'))
							|| (*p == '_')
							|| (*p == '.')) {
				if (startPtr == NULL) startPtr = p;
				p++;
				length++;
			}
			if (length > 0) {
				int s;
				CharPtr name = (char *)MemPtrNew(length + 1);
				memoSourceList[memoSourceCount] = name;			
				memoSourceIndexList[memoSourceCount] = i;				
				for (s = 0; s < length; s++)
					name[s] = startPtr[s];
				name[s] = 0;
				memoSourceCount++;	
			}
			MemHandleUnlock(h);
			DmReleaseRecord(memoDB, i, false);
		}
	}
	return memoSourceCount;
}

void buildPedit32SourceList()
{
	DmOpenRef pedit32DB;
	
	if (pedit32SourceList) return;
	
	pedit32DB = DmOpenDatabaseByTypeCreator('DATA', 'pn32', dmModeReadOnly);
	if (pedit32DB != NULL) {
		UInt numRecords = DmNumRecords(pedit32DB);
		UInt cTextCount;
		if (numRecords) {
			Boolean *isCText = (Boolean *)MemPtrNew(numRecords);
			cTextCount = countCTextRecords(numRecords, pedit32DB, isCText);
			
			if (cTextCount > 0) {
				pedit32SourceList = (char **)MemPtrNew(cTextCount * sizeof(char *));
				pedit32SourceIndexList = (long *)MemPtrNew(cTextCount * sizeof(long));
				pedit32SourceCount = identifyCTextRecords(numRecords, pedit32DB, isCText, pedit32SourceList, pedit32SourceIndexList);
			}
			MemPtrFree(isCText);
		}
		DmCloseDatabase(pedit32DB);
	}
}

void buildMemoSourceList()
{
	DmOpenRef memoDB;
	
	if (memoSourceList) return;

	memoDB = DmOpenDatabaseByTypeCreator('DATA', 'memo', dmModeReadOnly);
	if (memoDB != NULL) {
		UInt numRecords = DmNumRecords(memoDB);
		UInt cTextCount;
		if (numRecords) {
			Boolean *isCText = (Boolean *)MemPtrNew(numRecords);
			cTextCount = countCTextRecords(numRecords, memoDB, isCText);
			
			if (cTextCount > 0) {
				memoSourceList = (char **)MemPtrNew(cTextCount * sizeof(char *));
				memoSourceIndexList = (long *)MemPtrNew(cTextCount * sizeof(long));
				memoSourceCount = identifyCTextRecords(numRecords, memoDB, isCText, memoSourceList, memoSourceIndexList);
			}
			MemPtrFree(isCText);
		}
		DmCloseDatabase(memoDB);
	}
}

void copyOldSource(SourceFile *newS, SourceFile_Old *oldS)
{
	int j;
	newS->sourceKind = oldS->sourceKind;
	newS->sourceDBid = oldS->sourceDBid;
	newS->includeCount = oldS->includeCount;
	newS->outputDBid = oldS->outputDBid;
	for (j = 0; j < dmDBNameLength; j++)
		newS->sourceDB[j] = oldS->sourceDB[j];
	for (j = 0; j < dmDBNameLength; j++)
		newS->outputDB[j] = oldS->outputDB[j];
	for (j = 0; j < 8; j++)
		newS->includeIndex[j] = oldS->includeIndex[j];
}

void convertOldProjectToCurrentProject(ProjectHeader *pHdr)
{
	int i;
	MemSet(&gProjectHeader, sizeof(ProjectDetails), 0);
	gProjectHeader.prj.mVersion = PROJECT_VERSION_NEWOBJ;
	gProjectHeader.prj.mSourceDBCount = pHdr->sourceDBCount;
	gProjectHeader.prj.mIncludeCount = pHdr->includeCount;
	gProjectHeader.prj.mFlags = (pHdr->execute) ? EXECUTE_FLAG : 0;
	gProjectHeader.prj.mFlags |= (pHdr->alwaysRebuild) ? REBUILD_FLAG : 0;
	gProjectHeader.prj.mCreator = pHdr->creator;
	gProjectHeader.prj.mType = pHdr->type;
	for (i = 0; i < dmDBNameLength; i++) {
		gProjectHeader.prj.mAppName[i] = pHdr->appName[i];
		gProjectHeader.prj.mPrcName[i] = pHdr->prcName[i];
	}
	for (i = 0; i < pHdr->sourceDBCount; i++) {
		copyOldSource(&gProjectHeader.source[i], &pHdr->source[i]);
	}
	for (i = 0; i < pHdr->includeCount; i++)
		gProjectHeader.include[i] = pHdr->include[i];
}

void convertNewProjectToCurrentProject(ProjectHeader_New *pHdr)
{
	int i;
	IncludeFile *inc;
	SourceFile_Old *src = (SourceFile_Old *)(pHdr + 1);
	gProjectHeader.prj = *pHdr;
	gProjectHeader.prj.mVersion = PROJECT_VERSION_NEWOBJ;
	for (i = 0; i < pHdr->mSourceDBCount; i++)
		copyOldSource(&gProjectHeader.source[i], &src[i]); 
	inc = (IncludeFile *)(src + pHdr->mSourceDBCount);
	for (i = 0; i < pHdr->mIncludeCount; i++)
		gProjectHeader.include[i] = inc[i];
}

Boolean loadProject()
{
	char errBuf[64];
	int i;
	ProjectHeader_New *pHdr;

	DmOpenRef proj = DmOpenDatabase(0, gProjectDBid, dmModeReadOnly);
	
	Handle h = DmGet1Resource(PROJECT_TYPE, 1);
	if (h == NULL) {
		error("Corrupt project file");
		gProjectDBid = 0;
		return false;
	}
	
	pHdr = MemHandleLock(h);
	
	if (pHdr->mVersion != PROJECT_VERSION_NEWOBJ) {
		if (pHdr->mVersion == PROJECT_VERSION_CURRENT) {
			int i;
			SourceFile *src;
			IncludeFile *inc;
			deleteProjectFiles(pHdr, false);
			gProjectHeader.prj = *pHdr;
			gProjectHeader.prj.mVersion = PROJECT_VERSION_NEWOBJ;
			src = (SourceFile *)(pHdr + 1);
			for (i = 0; i < pHdr->mSourceDBCount; i++) {
				gProjectHeader.source[i] = src[i];
				if (gProjectHeader.source[i].sourceKind != Resource) {
					gProjectHeader.source[i].outputDBid = 0;
				}
			}
			inc = (IncludeFile *)(src + pHdr->mSourceDBCount);
			for (i = 0; i < pHdr->mIncludeCount; i++)
				gProjectHeader.include[i] = inc[i];		
		}
		else {
			if (pHdr->mVersion == PROJECT_VERSION) {			
				convertOldProjectToCurrentProject((ProjectHeader *)pHdr);
			}
			else {
				if (pHdr->mVersion == PROJECT_VERSION_NEW) {
					convertNewProjectToCurrentProject(pHdr);
				}
				else {
					MemHandleUnlock(h);
					DmReleaseResource(h);
					DmCloseDatabase(proj);
					error("Corrupt project file");
					gProjectDBid = 0;
					return false;
				}
			}
		}
	}
	else {
		int i;
		SourceFile *src;
		IncludeFile *inc;
		gProjectHeader.prj = *pHdr;
		src = (SourceFile *)(pHdr + 1);
		for (i = 0; i < pHdr->mSourceDBCount; i++)
			gProjectHeader.source[i] = src[i];
		inc = (IncludeFile *)(src + pHdr->mSourceDBCount);
		for (i = 0; i < pHdr->mIncludeCount; i++)
			gProjectHeader.include[i] = inc[i];		
	}
	
	MemHandleUnlock(h);
	DmReleaseResource(h); 
		
	for (i = 0; i < gProjectHeader.prj.mSourceDBCount; i++) {
		UInt attr;
		ULong type;
		
		if (gProjectHeader.source[i].sourceKind == MemoSource) {
			int k;
			buildMemoSourceList();
			for (k = 0; k < memoSourceCount; k++) {
				if (StrCompare(gProjectHeader.source[i].sourceDB, memoSourceList[k]) == 0) {
					gProjectHeader.source[i].sourceDBid = memoSourceIndexList[k];
					break;
				}				
			}
			if (k == memoSourceCount) {
				StrPrintF(errBuf, "Couldn't locate (Memo) source %s", gProjectHeader.source[i].sourceDB);
				error(errBuf);
				DmCloseDatabase(proj);
				return false;
			}
		}
		else
			if (gProjectHeader.source[i].sourceKind == Pedit32Source) {
				int k;
				buildPedit32SourceList();
				for (k = 0; k < pedit32SourceCount; k++) {
					if (StrCompare(gProjectHeader.source[i].sourceDB, pedit32SourceList[k]) == 0) {
						gProjectHeader.source[i].sourceDBid = pedit32SourceIndexList[k];
						break;
					}				
				}
				if (k == pedit32SourceCount) {
					StrPrintF(errBuf, "Couldn't locate (pedit32) source %s", gProjectHeader.source[i].sourceDB);
					error(errBuf);
					DmCloseDatabase(proj);
					return false;
				}
			}
			else {
				gProjectHeader.source[i].sourceDBid = DmFindDatabase(0, gProjectHeader.source[i].sourceDB);
				if (gProjectHeader.source[i].sourceDBid == 0) {
					StrPrintF(errBuf, "Couldn't locate source %s", gProjectHeader.source[i].sourceDB);
					error(errBuf);
					DmCloseDatabase(proj);
					return false;
				}
				DmDatabaseInfo(0, gProjectHeader.source[i].sourceDBid, 0, &attr, 0, 0, 0, 0, 0, 0, 0, &type, 0);
				if (gProjectHeader.source[i].sourceKind == Resource) {
					if ((attr & dmHdrAttrResDB) != dmHdrAttrResDB) {
						StrPrintF(errBuf, "%s is not a resource DB", gProjectHeader.source[i].sourceDB);
						error(errBuf);
						DmCloseDatabase(proj);
						return false;
					}
				}
				else {
					if (type != 'TEXt') {
						StrPrintF(errBuf, "%s is not a text DB", gProjectHeader.source[i].sourceDB);
						error(errBuf);
						DmCloseDatabase(proj);
						return false;
					}
				}
			}
		gProjectHeader.source[i].outputDBid = DmFindDatabase(0, gProjectHeader.source[i].outputDB);

	}
	DmCloseDatabase(proj);
	return true;
}

void projectFormInit()
{
	int i;
	FormPtr frmP = FrmGetActiveForm();	
	ListPtr lstP = getObjectPtr(frmP, ProjectSourcesList);
	ControlPtr ctlP = (ControlPtr)getObjectPtr(frmP, ProjectExecuteCheckbox);

	FrmDrawForm(frmP);	
//	if (gRomVersion >= version35) WinSetBackColor(0);
//	WinEraseRectangle(&gFullScreen, 0);

 	StrPrintF(gFormTitle, "OnBoardC : %s", gPrefs.projectName);
	FrmSetTitle(frmP, gFormTitle);	
	eraseTitle();

	FrmDrawForm(frmP);
	
	if (gProjectContentsListContents) {
		for (i = 0; i < gProjectContentsListCount; i++)
			MemPtrFree(gProjectContentsListContents[i]);
		MemPtrFree(gProjectContentsListContents);
	}
	
	gProjectContentsListContents = (char **)MemPtrNew(sizeof(char *) * gProjectHeader.prj.mSourceDBCount);
	gProjectContentsListCount = 0;
	for (i = 0; i < gProjectHeader.prj.mSourceDBCount; i++) {
		gProjectContentsListContents[gProjectContentsListCount] = MemPtrNew(StrLen(gProjectHeader.source[i].sourceDB) + 1);
		StrCopy(gProjectContentsListContents[gProjectContentsListCount], gProjectHeader.source[i].sourceDB);
		gProjectContentsListCount++;
	}
	LstSetListChoices(lstP, gProjectContentsListContents, gProjectContentsListCount);
	LstSetSelection(lstP, -1);
	LstDrawList(lstP);
	
	setHandleFromText(frmP, ProjectPrcNameField, gProjectHeader.prj.mPrcName, true);
	setHandleFromChar4(frmP, ProjectCreatorField, gProjectHeader.prj.mCreator, true);
	setHandleFromChar4(frmP, ProjectTypeField, gProjectHeader.prj.mType, true);
	
	CtlSetValue(ctlP, (gProjectHeader.prj.mFlags & EXECUTE_FLAG) == EXECUTE_FLAG);
	ctlP = (ControlPtr)getObjectPtr(frmP, ProjectAlwaysRebuildCheckbox);
	CtlSetValue(ctlP, (gProjectHeader.prj.mFlags & REBUILD_FLAG) == REBUILD_FLAG);
	ctlP = (ControlPtr)getObjectPtr(frmP, ProjectDebugCheckbox);
	CtlSetValue(ctlP, (gProjectHeader.prj.mFlags & DEBUG_FLAG) == DEBUG_FLAG);
	ctlP = (ControlPtr)getObjectPtr(frmP, ProjectAutoVersionCheckbox);
	CtlSetValue(ctlP, (gProjectHeader.prj.mFlags & AUTOVERSION_FLAG) == AUTOVERSION_FLAG);

}

short sortFn(void *A, void *B, Long other)
{
	char *a = *((char **)A);
	char *b = *((char **)B);
	return StrCompare(a, b);
}

Boolean scrollList(ListPtr lst, char c, char **listContents, int count)
{
	Boolean result = false;
	int newTop = -1;
	int listSize = LstGetVisibleItems(lst);	
	switch (c) {
		case pageDownChr :
			LstScrollList(lst, winDown, listSize);
	   		result = true;
			break;
		case pageUpChr :
			LstScrollList(lst, winUp, listSize);
	   		result = true;
			break;
		default : {
				char cc = c;
				int i;		
				if ((c >= 'a') && (c <= 'z')) cc = c - ('a' - 'A');
				for (i = 0; i < count; i++) {							
					char x = listContents[i][0];
					if ((x == c) || (x == cc)) {
						newTop = i;
						result = true;
						break;
					}
				}
			}
			break;
	}
	if (newTop != -1) {
		LstSetSelection(lst, newTop);
		LstSetTopItem(lst, newTop);
		LstEraseList(lst);
		LstDrawList(lst);
	}
	return result;
}

void buildDBLists()
{
	int i;
	int dbCount = DmNumDatabases(0);
	
	if (docSourceDBList) {
		for (i = 0; i < docSourceDBCount; i++)
			MemPtrFree(docSourceDBList[i]);
		MemPtrFree(docSourceDBList);
	}
	if (resourceDBList) {
		for (i = 0; i < resourceDBCount; i++)
			MemPtrFree(resourceDBList[i]);
		MemPtrFree(resourceDBList);
	}
	if (memoSourceDisplayList) {
		for (i = 0; i < memoSourceDisplayCount; i++)
			MemPtrFree(memoSourceDisplayList[i]);
		MemPtrFree(memoSourceDisplayList);
		memoSourceDisplayList = NULL;
	}
	if (memoSourceDisplayIndexList)  {
		MemPtrFree(memoSourceDisplayIndexList);
		memoSourceDisplayIndexList = NULL;
	}
	if (pedit32SourceDisplayList) {
		for (i = 0; i < pedit32SourceDisplayCount; i++)
			MemPtrFree(pedit32SourceDisplayList[i]);
		MemPtrFree(pedit32SourceDisplayList);
		pedit32SourceDisplayList = NULL;
	}
	if (pedit32SourceDisplayIndexList)  {
		MemPtrFree(pedit32SourceDisplayIndexList);
		pedit32SourceDisplayIndexList = NULL;
	}
	
	docSourceDBList = (char **)MemPtrNew(sizeof(char *) * dbCount);
	docSourceDBCount = 0;
	resourceDBList = (char **)MemPtrNew(sizeof(char *) * dbCount);
	resourceDBCount = 0;

	for (i = 0; i < dbCount; i++) {
		ULong type;
		UInt attr;
		char dbName[dmDBNameLength];
		int length;
		int j;
		
		LocalID dbID = DmGetDatabase(0, i);
		DmDatabaseInfo(0, dbID, dbName, &attr, 0, 0, 0, 0, 0, 0, 0, &type, 0);
		length = StrLen(dbName);
		
		for (j = 0; j < gProjectHeader.prj.mSourceDBCount; j++) {
			if (StrCompare(dbName, gProjectHeader.source[j].sourceDB) == 0)
				break;
		}
		if (j == gProjectHeader.prj.mSourceDBCount) {
			if (((attr & dmHdrAttrResDB) == dmHdrAttrResDB)
					&& ((type == 'Rsrc') 
							|| (StrCaselessCompare(dbName, ".rsrc") == (length - 5))))
			
			 {
				resourceDBList[resourceDBCount] = MemPtrNew(length + 1);
				StrCopy(resourceDBList[resourceDBCount], dbName);
				resourceDBCount++;
			}
			else {
				if ((type == 'TEXt') && (dbName[length - 2] == '.') 
						&& ((dbName[length - 1] == 'c')
								|| (dbName[length - 1] == 'h'))) {
					docSourceDBList[docSourceDBCount] = MemPtrNew(length + 1);
					StrCopy(docSourceDBList[docSourceDBCount], dbName);
					docSourceDBCount++;
				}
			}
		}
	}
	SysQSort(docSourceDBList, docSourceDBCount, sizeof(char *), sortFn, 0);
	SysQSort(resourceDBList, resourceDBCount, sizeof(char *), sortFn, 0);
	
	buildMemoSourceList();
	buildPedit32SourceList();
	
	memoSourceDisplayList = (char **)MemPtrNew(sizeof(char *) * memoSourceCount);
	memoSourceDisplayIndexList = (long *)MemPtrNew(sizeof(long *) * memoSourceCount);
	memoSourceDisplayCount = 0;
	
	for (i = 0; i < memoSourceCount; i++) {
		int j;
		for (j = 0; j < gProjectHeader.prj.mSourceDBCount; j++) {
			if (StrCompare(memoSourceList[i], gProjectHeader.source[j].sourceDB) == 0)
				break;
		}
		if (j == gProjectHeader.prj.mSourceDBCount) {
			memoSourceDisplayList[memoSourceDisplayCount] = (char *)MemPtrNew(StrLen(memoSourceList[i]) + 1);
			StrCopy(memoSourceDisplayList[memoSourceDisplayCount], memoSourceList[i]);
			memoSourceDisplayIndexList[memoSourceDisplayCount] = memoSourceIndexList[i];
			memoSourceDisplayCount++;
		}
	}

	pedit32SourceDisplayList = (char **)MemPtrNew(sizeof(char *) * pedit32SourceCount);
	pedit32SourceDisplayIndexList = (long *)MemPtrNew(sizeof(long *) * pedit32SourceCount);
	pedit32SourceDisplayCount = 0;
	
	for (i = 0; i < pedit32SourceCount; i++) {
		int j;
		for (j = 0; j < gProjectHeader.prj.mSourceDBCount; j++) {
			if (StrCompare(pedit32SourceList[i], gProjectHeader.source[j].sourceDB) == 0)
				break;
		}
		if (j == gProjectHeader.prj.mSourceDBCount) {
			pedit32SourceDisplayList[pedit32SourceDisplayCount] = (char *)MemPtrNew(StrLen(pedit32SourceList[i]) + 1);
			StrCopy(pedit32SourceDisplayList[pedit32SourceDisplayCount], pedit32SourceList[i]);
			pedit32SourceDisplayIndexList[pedit32SourceDisplayCount] = pedit32SourceIndexList[i];
			pedit32SourceDisplayCount++;
		}
	}
}

Boolean addToProjectEventHandler(EventPtr eventP)
{
	Boolean handled = false;
	FormPtr frmP = FrmGetActiveForm();
	ListPtr lstP = getObjectPtr(frmP, AddToProjectSourceListList);
	SourceKind whichSource = gPrefs.addSourceType;
	
	if (FrmGetControlGroupSelection(frmP, AddToProjectFormGroupID) 
			== FrmGetObjectIndex(frmP, AddToProjectDOCSourcePushButton))
		whichSource = DocSource;
	else										
		if (FrmGetControlGroupSelection(frmP, AddToProjectFormGroupID) 
				== FrmGetObjectIndex(frmP, AddToProjectMemoSourcePushButton))
			whichSource = MemoSource;
		else										
			if (FrmGetControlGroupSelection(frmP, AddToProjectFormGroupID) 
					== FrmGetObjectIndex(frmP, AddToProjectPedit32SourcePushButton))
				whichSource = Pedit32Source;
			else										
				if (FrmGetControlGroupSelection(frmP, AddToProjectFormGroupID) 
						== FrmGetObjectIndex(frmP, AddToProjectResourcePushButton))
					whichSource = Resource;
	
	switch (eventP->eType) {
  		case frmOpenEvent:
	  		FrmDrawForm(frmP);
//			if (gRomVersion >= version35) WinSetBackColor(0);
//			WinEraseRectangle(&gFullScreen, 0);
	  		buildDBLists();
	  		
			if (whichSource == DocSource) {
		  		LstSetListChoices(lstP, docSourceDBList, docSourceDBCount);
	  			FrmSetControlGroupSelection(frmP, AddToProjectFormGroupID, AddToProjectDOCSourcePushButton);
			}
			else {
				if (whichSource == MemoSource) {
		  			LstSetListChoices(lstP, memoSourceDisplayList, memoSourceDisplayCount);
		  			FrmSetControlGroupSelection(frmP, AddToProjectFormGroupID, AddToProjectMemoSourcePushButton);
		  		}
				else {
					if (whichSource == Pedit32Source) {
			  			LstSetListChoices(lstP, pedit32SourceDisplayList, pedit32SourceDisplayCount);
			  			FrmSetControlGroupSelection(frmP, AddToProjectFormGroupID, AddToProjectPedit32SourcePushButton);
			  		}
					else {
			  			LstSetListChoices(lstP, resourceDBList, resourceDBCount);
			  			FrmSetControlGroupSelection(frmP, AddToProjectFormGroupID, AddToProjectResourcePushButton);
			  		}
			  	}
		  	}
	  		LstDrawList(lstP);
	   		handled = true;
  			break;
	   	case keyDownEvent: {
	   			if (whichSource == DocSource)
		   			handled = scrollList(lstP, eventP->data.keyDown.chr, docSourceDBList, docSourceDBCount);
				else
					if (whichSource == MemoSource)
			   			handled = scrollList(lstP, eventP->data.keyDown.chr, memoSourceDisplayList, memoSourceDisplayCount);
					else
						if (whichSource == Pedit32Source)
				   			handled = scrollList(lstP, eventP->data.keyDown.chr, pedit32SourceDisplayList, pedit32SourceDisplayCount);
						else
				   			handled = scrollList(lstP, eventP->data.keyDown.chr, resourceDBList, resourceDBCount);
			}
			break;
	   	case ctlSelectEvent:
	   		switch (eventP->data.ctlSelect.controlID) {
	   			case AddToProjectAddButton : {
	   					int index = LstGetSelection(lstP);
	   					if (index != -1) {
		   					SourceFile *sf = &gProjectHeader.source[gProjectHeader.prj.mSourceDBCount];
	   						sf->sourceKind = whichSource;
		   					if (whichSource == DocSource) {
		   						StrCopy(sf->sourceDB, docSourceDBList[index]);
		   						sf->sourceDBid = DmFindDatabase(0, docSourceDBList[index]);
		   						getAsmName(sf->sourceDB, sf->outputDB, 0);
		   						sf->outputDBid = DmFindDatabase(0, sf->outputDB);
		   					}
		   					else {
								if (whichSource == Pedit32Source) {
			   						StrCopy(sf->sourceDB, pedit32SourceDisplayList[index]);
			   						sf->sourceDBid = pedit32SourceDisplayIndexList[index];
			   						getAsmName(sf->sourceDB, sf->outputDB, 0);
			   						sf->outputDBid = DmFindDatabase(0, sf->outputDB);
								}
								else {
									if (whichSource == MemoSource) {
				   						StrCopy(sf->sourceDB, memoSourceDisplayList[index]);
				   						sf->sourceDBid = memoSourceDisplayIndexList[index];
				   						getAsmName(sf->sourceDB, sf->outputDB, 0);
				   						sf->outputDBid = DmFindDatabase(0, sf->outputDB);
									}
									else {
				   						StrCopy(sf->sourceDB, resourceDBList[index]);
				   						sf->sourceDBid = DmFindDatabase(0, resourceDBList[index]);
									}
								}
		   					}
							
	   						gProjectHeader.prj.mSourceDBCount++;
	   						buildDBLists();
		   					if (whichSource == DocSource)
						  		LstSetListChoices(lstP, docSourceDBList, docSourceDBCount);
							else
								if (whichSource == MemoSource)
						  			LstSetListChoices(lstP, memoSourceDisplayList, memoSourceDisplayCount);
								else
									if (whichSource == Pedit32Source)
							  			LstSetListChoices(lstP, pedit32SourceDisplayList, pedit32SourceDisplayCount);
									else
							  			LstSetListChoices(lstP, resourceDBList, resourceDBCount);
					  		LstDrawList(lstP);
					  	}
					}	   					
   					handled = true;
	   				break;
	   			
	   			case AddToProjectDoneButton :
	   				FrmGotoForm(ProjectForm);
   					handled = true;
	   				break;
	   				
	   			case AddToProjectDOCSourcePushButton :
   					if (eventP->data.ctlSelect.on) {
				  		LstSetListChoices(lstP, docSourceDBList, docSourceDBCount);
				  		LstDrawList(lstP);
				  		gPrefs.addSourceType = DocSource;
   					}
   					handled = true;
   					break;
   				case AddToProjectMemoSourcePushButton :
   					if (eventP->data.ctlSelect.on) {
				  		LstSetListChoices(lstP, memoSourceDisplayList, memoSourceDisplayCount);
				  		LstDrawList(lstP);
				  		gPrefs.addSourceType = MemoSource;
   					}
   					handled = true;
   					break;
   				case AddToProjectPedit32SourcePushButton :
   					if (eventP->data.ctlSelect.on) {
				  		LstSetListChoices(lstP, pedit32SourceDisplayList, pedit32SourceDisplayCount);
				  		LstDrawList(lstP);
				  		gPrefs.addSourceType = Pedit32Source;
   					}
   					handled = true;
   					break;
	   			case AddToProjectResourcePushButton :
   					if (eventP->data.ctlSelect.on) {
				  		LstSetListChoices(lstP, resourceDBList, resourceDBCount);
				  		LstDrawList(lstP);
				  		gPrefs.addSourceType = Resource;
   					}
   					handled = true;
	   				break;
	   		}
	   		break;
        default:
            break;
	}
	return handled;
}

void buildProjectList(ListPtr lstP)
{
	int num = DmNumDatabases(0);
	int i;

	if (gProjectListContents) {
		for (i = 0; i < gProjectListCount; i++)
			MemPtrFree(gProjectListContents[i]);
		MemPtrFree(gProjectListContents);
	}
	
	gProjectListCount = 0;
	gProjectListContents = (char **)MemPtrNew(sizeof(char *) * num);
	for (i = 0; i < num; i++) {
		LocalID dbID = DmGetDatabase(0, i);
		char dbName[dmDBNameLength];
		int length;
		DmDatabaseInfo(0, dbID, dbName, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
		length = StrLen(dbName);
		if (StrStr(dbName, ".proj") == dbName + (length - 5)) {
			gProjectListContents[gProjectListCount] = MemPtrNew(length + 1);
			StrCopy(gProjectListContents[gProjectListCount], dbName);
			gProjectListCount++;
		}
	}
	SysQSort(gProjectListContents, gProjectListCount, sizeof(char *), sortFn, 0);

	LstSetListChoices(lstP, gProjectListContents, gProjectListCount);
}
//                     0         1         2         3         4  
//                     012345678901234567890123456789012345678901
const char *baseStr = "iiiiiiiiiiiiiiiiiiiiiiiiiiiiiii";
const int baseStrLength = 31;
const char *aboutMess = "Built with OnBoardC";

static void copyResource(ULong type, int id, DmOpenRef newDB, char *rootName)
{
	CharPtr rep;
	Handle copyH;
	CharPtr p;
	
	Handle skeletonH = DmGetResource(type, id);
	ULong sz = MemHandleSize(skeletonH);
	
	copyH = DmNewResource(newDB, type, id, sz);
	p = MemHandleLock(copyH);
	DmWrite(p, 0, MemHandleLock(skeletonH), sz);
	
	if (type == 'MBAR') {
		FontID f = FntSetFont(boldFont);
		int width = FntCharsWidth(rootName, StrLen(rootName)) + 30;
		width += FntCharsWidth("About ", 6);
		rep = StrStr(p + (sz - baseStrLength) - 1 , baseStr);
		if (rep) DmWrite(p, rep - p, rootName, StrLen(rootName) + 1);
		DmWrite(p, 0x28, &width, sizeof(width));
		width = 6;
		DmWrite(p, 0x24, &width, sizeof(width));
		FntSetFont(f);
	}
	else
		if (type == 'tFRM') {
			rep = StrStr(p + (sz - baseStrLength) - 1, baseStr);
			if (rep) DmWrite(p, rep - p, rootName, StrLen(rootName) + 1);
		}

	MemHandleUnlock(copyH);
	MemHandleUnlock(skeletonH);
	DmReleaseResource(copyH);
	DmReleaseResource(skeletonH);
}

static void deleteMemoEntry(CharPtr name)
{
	int i;
	buildMemoSourceList();
	for (i = 0; i < memoSourceCount; i++) {
		if (StrCompare(memoSourceList[i], name) == 0) {
			DmOpenRef memoDB = DmOpenDatabaseByTypeCreator('DATA', 'memo', dmModeReadWrite);
			DmRemoveRecord(memoDB, memoSourceIndexList[i]);
			DmCloseDatabase(memoDB);
			break;
		}
	}
}

static void deletePedit32Entry(CharPtr name)
{
	int i;
	buildPedit32SourceList();
	for (i = 0; i < pedit32SourceCount; i++) {
		if (StrCompare(pedit32SourceList[i], name) == 0) {
			DmOpenRef pedit32DB = DmOpenDatabaseByTypeCreator('DATA', 'pn32', dmModeReadWrite);
			DmRemoveRecord(pedit32DB, pedit32SourceIndexList[i]);
			DmCloseDatabase(pedit32DB);
			break;
		}
	}
}

ULong stripLineFeeds(CharPtr p, int size)
{
	int index = 0;
	int newIndex = 0;
	while (index < size) {
		int startIndex = index;
		while ((index < size) && (p[index] != 0x0a)) index++;
		if (p[index] == 0x0a) index++;
		DmWrite(p, newIndex, &p[startIndex], (index - startIndex));
		newIndex += (index - startIndex);
		if (p[index] == 0x0a) index++;
	}
	return newIndex;
}

static Boolean doNewProject()
{
	UInt16 dbattrs;
	Boolean result = false;
	FormPtr frmP = FrmInitForm(NewProjectForm);

	if (FrmDoDialog(frmP) == NewProjectOKButton) {
		int rootLength;
		char projectRoot[32];
		ControlPtr ctlP = getObjectPtr(frmP, NewProjectSkeletonCheckbox);
		setTextFromHandle(frmP, NewProjectProjectNameField, projectRoot);
		rootLength = StrLen(projectRoot);
		if (rootLength == 0)
			StrCopy(projectRoot, "Untitled");
		else {
			if (StrStr(projectRoot, ".proj") == (projectRoot + (rootLength - 5))) {
				rootLength -= 5;
				projectRoot[rootLength] = '\0';
			}
		}
		StrCopy(gPrefs.projectName, projectRoot);
		StrCat(gPrefs.projectName, ".proj");
		gProjectDBid = setDefaultProject(false);
		if (CtlGetValue(ctlP)) {
			LocalID dbID;
			Err err;
			ProjectHeader_New *pHdr;
			UInt at = 0;
			DOCHeader hdr = { 1, 0, 0, 1, 4096, 0 };
			Handle copyH;
			CharPtr p;
			ULong hSize;
			
			AlertTemplateType alt = { informationAlert, 0, 1, 0 };
			SourceKind kind = DocSource;			
			DmOpenRef db = DmOpenDatabase(0, gProjectDBid, dmModeReadWrite);
			Handle h = DmGet1Resource(PROJECT_TYPE, 1);
			pHdr = MemHandleLock(h);

			if (FrmGetControlGroupSelection(frmP, NewProjectFormGroupID2)
											 == FrmGetObjectIndex(frmP, NewProjectMemoPushButton))
				kind = MemoSource;
			else
				if (FrmGetControlGroupSelection(frmP, NewProjectFormGroupID2)
												 == FrmGetObjectIndex(frmP, NewProjectPedit32PushButton))
					kind = Pedit32Source;
			
			gProjectHeader.prj = *pHdr;
			MemHandleUnlock(h);
			StrCopy(gProjectHeader.prj.mPrcName, projectRoot);			
			gProjectHeader.prj.mSourceDBCount = 2;
			if (rootLength < 4) {
				projectRoot[rootLength] = '_';
				projectRoot[rootLength + 1] = '_';
				projectRoot[rootLength + 2] = '_';
				gProjectHeader.prj.mCreator = *((long *)projectRoot);
				projectRoot[rootLength] = '\0';
			}
			else
				gProjectHeader.prj.mCreator = *((long *)projectRoot);
			gProjectHeader.prj.mType = 'appl';
			gProjectHeader.source[0].sourceKind = Resource;
			StrCopy(gProjectHeader.source[0].sourceDB, projectRoot);
			StrCat(gProjectHeader.source[0].sourceDB, ".Rsrc");

			gProjectHeader.source[1].sourceKind = kind;
			StrCopy(gProjectHeader.source[1].sourceDB, projectRoot);
			StrCat(gProjectHeader.source[1].sourceDB, ".c");
			getAsmName(gProjectHeader.source[1].sourceDB, gProjectHeader.source[1].outputDB, 0);

			DmResizeResource(h, sizeof(ProjectHeader_New) + (2 * sizeof(SourceFile)));
			pHdr = MemHandleLock(h);
			DmWrite(pHdr, 0, &gProjectHeader.prj, sizeof(ProjectHeader_New));
			DmWrite(pHdr, sizeof(ProjectHeader_New), &gProjectHeader.source[0], sizeof(SourceFile));
			DmWrite(pHdr, sizeof(ProjectHeader_New) + sizeof(SourceFile), &gProjectHeader.source[1], sizeof(SourceFile));
			MemHandleUnlock(h);
			DmReleaseResource(h);
			DmCloseDatabase(db);

			dbID = DmFindDatabase(0, gProjectHeader.source[0].sourceDB);
			if (dbID != 0) DmDeleteDatabase(0, dbID);
			err = DmCreateDatabase(0, gProjectHeader.source[0].sourceDB, 'OnBC', 'Rsrc', true);
			dbID = DmFindDatabase(0, gProjectHeader.source[0].sourceDB);
			// Set the Backup bit.
			if (DmDatabaseInfo (0, dbID, NULL, &dbattrs, NULL, NULL, NULL, NULL,
				NULL, NULL, NULL, NULL, NULL) == 0) {
				dbattrs |= dmHdrAttrBackup;
				(void) DmSetDatabaseInfo (0, dbID, NULL, &dbattrs, NULL, NULL, 
					NULL, NULL, NULL, NULL, NULL, NULL, NULL);
			}
			db = DmOpenDatabase(0, dbID, dmModeReadWrite);


			copyResource('tFRM', 1000, db, projectRoot);	
			copyResource('MBAR', 1000, db, projectRoot);	

			copyH = DmNewResource(db, 'Talt', 1000, sizeof(AlertTemplateType) 
												+ rootLength + 6 + 1			// About <name>
												+ StrLen(aboutMess) + 1			// 
												+ 3);							// OK
			p = MemHandleLock(copyH);
			DmWrite(p, 0, &alt, sizeof(AlertTemplateType));
			DmWrite(p, sizeof(AlertTemplateType), "About ", 6);						
			DmWrite(p, sizeof(AlertTemplateType) + 6, projectRoot, rootLength + 1);						;
			DmWrite(p, sizeof(AlertTemplateType) + 6 + rootLength + 1, aboutMess, StrLen(aboutMess) + 1);						;
			DmWrite(p, sizeof(AlertTemplateType) + 6 + rootLength + 1 + StrLen(aboutMess) + 1, "OK", 3);						;
			MemHandleUnlock(copyH);
			DmReleaseResource(copyH);
// AIN, vers, AIB ???
			DmCloseDatabase(db);

			h = DmGetResource('tSTR', 1000);
			p = MemHandleLock(h);
			hSize = MemHandleSize(h);
//			hSize = stripLineFeeds(p, (int)MemHandleSize(h));
			
//			DmWrite(p, 20, &gProjectHeader.prj.mCreator, 4);
			if (kind == MemoSource) {
				CharPtr mp;
				deleteMemoEntry(gProjectHeader.source[1].sourceDB);
				deleteMemoSourceList();				
				db = DmOpenDatabaseByTypeCreator('DATA', 'memo', dmModeReadWrite);
				at = -1;
				copyH = DmNewRecord(db, &at, hSize + rootLength + 9);
				mp = MemHandleLock(copyH);
				DmWrite(mp, 0, "/* ", 3);
				DmWrite(mp, 3, projectRoot, rootLength);
				DmWrite(mp, 3 + rootLength, ".c */\n", 6);
				DmWrite(mp, 9 + rootLength, p, hSize);
				// This line sets the creator but we've
				//   changed the skeleton program
				// DmWrite(mp, 9 + rootLength + 20, &gProjectHeader.prj.mCreator, 4);	// set the creator
				hSize += rootLength + 9;
				hSize = stripLineFeeds(mp, (int)hSize);
				MemHandleUnlock(copyH);
				DmResizeRecord(db, at, hSize);
				DmReleaseRecord(db, at, true);
			}
			else {
				if (kind == Pedit32Source) {
					CharPtr mp;
					deletePedit32Entry(gProjectHeader.source[1].sourceDB);
					deletePedit32SourceList();				
					db = DmOpenDatabaseByTypeCreator('DATA', 'pn32', dmModeReadWrite);
					at = -1;
					copyH = DmNewRecord(db, &at, hSize + rootLength + 9);
					mp = MemHandleLock(copyH);
					DmWrite(mp, 0, "/* ", 3);
					DmWrite(mp, 3, projectRoot, rootLength);
					DmWrite(mp, 3 + rootLength, ".c */\n", 6);
					DmWrite(mp, 9 + rootLength, p, hSize);
					// This line sets the creator but we've
					//   changed the skeleton program
					//DmWrite(mp, 9 + rootLength + 20, &gProjectHeader.prj.mCreator, 4);	// set the creator
					hSize += rootLength + 9;
					hSize = stripLineFeeds(mp, (int)hSize);
					MemHandleUnlock(copyH);
					DmResizeRecord(db, at, hSize);
					DmReleaseRecord(db, at, true);
				}
				else {		// DOC
					char *n;
					UInt att;
					dbID = DmFindDatabase(0, gProjectHeader.source[1].sourceDB);
					if (dbID != 0) DmDeleteDatabase(0, dbID);
					err = DmCreateDatabase(0, gProjectHeader.source[1].sourceDB, 'REAd', 'TEXt', false);
					dbID = DmFindDatabase(0, gProjectHeader.source[1].sourceDB);
					DmDatabaseInfo(0, dbID, NULL, &att,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL);
					att |= dmHdrAttrBackup;
					DmSetDatabaseInfo(0,dbID,NULL,&att,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL);
					db = DmOpenDatabase(0, dbID, dmModeReadWrite);
					at = 1;
					copyH = DmNewRecord(db, &at, hSize);
					n = MemHandleLock(copyH);
					DmWrite(n, 0, p, hSize);
					// This line sets the creator but we've
					//   changed the skeleton program
					//DmWrite(n, 20, &gProjectHeader.prj.mCreator, 4);	// set the creator
					hSize = stripLineFeeds(n, (int)hSize);
					MemHandleUnlock(copyH);			
					DmResizeRecord(db, at, hSize);
					DmReleaseRecord(db, at, true);
					
					at = 0;
					copyH = DmNewRecord(db, &at, sizeof(DOCHeader));
					hdr.size = hSize;
					DmWrite(MemHandleLock(copyH), 0, &hdr, sizeof(DOCHeader));
					MemHandleUnlock(copyH);			
					DmReleaseRecord(db, at, true);
				}
			}
			MemHandleUnlock(h);
			DmReleaseResource(h);
			DmCloseDatabase(db);
		}
		loadProject();
		result = true;
	}

	FrmDeleteForm(frmP);
	return result;
}

static Boolean openProjectHandler(EventPtr eventP)
{
    Boolean handled = false;
    FormPtr frmP = FrmGetActiveForm();
    
    if (eventP->eType == keyDownEvent) {
		ListPtr lstP = (ListPtr)getObjectPtr(frmP, OpenProjectProjectListList);
		handled = scrollList(lstP, eventP->data.keyDown.chr, gProjectListContents, gProjectListCount);
    }
    
	return handled;
}

static Boolean deleteProjectHandler(EventPtr eventP)
{
    Boolean handled = false;
    FormPtr frmP = FrmGetActiveForm();
    
    if (eventP->eType == keyDownEvent) {
		ListPtr lstP = (ListPtr)getObjectPtr(frmP, DeleteProjectProjectListList);
		handled = scrollList(lstP, eventP->data.keyDown.chr, gProjectListContents, gProjectListCount);
    }
    
	return handled;
}

static Boolean doOpenProject()
{
	int btn;
	Boolean result = false;
	FormPtr frmP = FrmInitForm(OpenProjectForm);	
	ListPtr lstP = getObjectPtr(frmP, OpenProjectProjectListList);
	buildProjectList(lstP);
	FrmSetEventHandler(frmP, openProjectHandler);
	
	btn = FrmDoDialog(frmP);
	if (btn == OpenProjectOpenButton) {
		int index = LstGetSelection(lstP);
		if (index != -1) {
			StrCopy(gPrefs.projectName, gProjectListContents[index]);
			gProjectDBid = DmFindDatabase(0, gPrefs.projectName);
			result = loadProject();
//			result = true;
		}
	}
	else {
		if (btn == OpenProjectNewButton) {
			result = doNewProject();
		}
	}
	FrmDeleteForm(frmP);
	return result;
}

Boolean pointInObject(FormPtr frm, short x, short y, WordPtr pObjIndex, RectangleType *pRect)
{
	Word i;
	RectangleType r;
				
	FrmGetFormBounds(frm, &r);
					
	//x -= r.topLeft.x; //x -= frm->window.windowBounds.topLeft.x;
	//y -= r.topLeft.y; //y -= frm->window.windowBounds.topLeft.y;
				
	for (i = 0; i < FrmGetNumberOfObjects(frm); i++)
		{
		FrmGetObjectBounds(frm, i, &r);
		if (RctPtInRectangle (x, y, &r))
			{
			*pObjIndex = i;
			*pRect = r;
			return (true);
			}
		}

	return (false);
}

#if 0
Boolean pointInObject(FormPtr frm, short x, short y, WordPtr pObjIndex, RectangleType *pRect)
{
	Word i;
	ListPtr lst;
	FieldPtr fld;
	TablePtr table;
	ControlPtr ctl;
	ScrollBarPtr scrollBar;
	FormLabelType *pLabel;
	FormObjectKind objType;
	FontID curFont;
	RectangleType r;
		
	for (i = 0; i < frm->numObjects; i++)
		{
		objType = frm->objects[i].objectType;

		if (objType == frmControlObj)
			{
			ctl = frm->objects[i].object.control;
			r = ctl->bounds;
			}			

		else if (objType == frmListObj)
			{
			lst = frm->objects[i].object.list;
			r = lst->bounds;
			}
						
		else if (objType == frmFieldObj)
			{
			fld = frm->objects[i].object.field;
			r = fld->rect;			
			}

		else if (objType == frmTableObj)
			{
			table = frm->objects[i].object.table;
			r = table->bounds;			
			}
/* s'ok, I know there aren't any.
		else if (objType == frmBitmapObj)
			{
			Handle h;
			DmOpenRef theDB;					
			FormBitmapType *pBitmap;
			pBitmap = frm->objects[i].object.bitmap;
			r.topLeft = pBitmap->pos;
			theDB = DmOpenDatabase(0, gDBId, dmModeReadOnly);					
			h = (Handle)DmGet1Resource('Tbmp', pBitmap->rscID);
			if (h) {
				BitmapType *bmpP = (BitmapType *)MemHandleLock(h);
				r.extent.x = bmpP->width;
				r.extent.y = bmpP->height;
				MemHandleUnlock(h);
				DmReleaseResource(h);
				DmCloseDatabase(theDB);
			}
			else {
				DmCloseDatabase(theDB);
				continue;
			}
			}
*/
		else if (objType == frmLabelObj)
			{
			int x;
			char *t, *n;
			pLabel = frm->objects[i].object.label;
			curFont = FntSetFont(pLabel->fontID);
			r.topLeft = pLabel->pos;
			t = pLabel->text;
			n = StrStr(t, "\r");
			r.extent.x = 0;
			r.extent.y = FntLineHeight();
			while (n) {
				r.extent.y += FntLineHeight();
				x = FntCharsWidth(t, n - t);
				if (x > r.extent.x) r.extent.x =  x;
				t = n + 1;
				n = StrStr(t, "\r");
			}
			x = FntCharsWidth(t, StrLen(t));
			if (x > r.extent.x) r.extent.x =  x;
			FntSetFont(curFont);		
			}

		else if (objType == frmScrollBarObj)
			{
			scrollBar = frm->objects[i].object.scrollBar;
			r = scrollBar->bounds;			
			}

		else	
			continue;

		if (RctPtInRectangle (x, y, &r))
			{
			*pObjIndex = i;
			*pRect = r;
			return (true);
			}
		}

	return (false);
}
#endif

char RsrcEditTitle[] = "RsrcEdit";
char OtherEditTitle[] = "Edit";

static Boolean projectEventHandler(EventPtr eventP)
{
	Boolean handled = false;
	FormPtr frmP = FrmGetActiveForm();
	
	switch (eventP->eType) {
		case lstSelectEvent : {
				ListPtr lstP = getObjectPtr(frmP, ProjectSourcesList);
				int index = LstGetSelection(lstP);
				ControlPtr ctlP = getObjectPtr(frmP, ProjectEditButton);
				if ((index == -1) || (gProjectHeader.source[index].sourceKind == Resource))
					CtlSetLabel(ctlP, RsrcEditTitle);
				else
					CtlSetLabel(ctlP, OtherEditTitle);
				handled = true;
			}
			break;
		case penUpEvent: {
				Word index;
				RectangleType rect;
				if (!pointInObject(frmP, eventP->screenX, eventP->screenY, &index, &rect)) {
					ListPtr lstP = getObjectPtr(frmP, ProjectSourcesList);
					ControlPtr ctlP = getObjectPtr(frmP, ProjectEditButton);
					CtlSetLabel(ctlP, RsrcEditTitle);
					LstSetSelection(lstP, -1);
				}
				handled = true;
			}
			break;
  		case frmOpenEvent:
  			FrmDrawForm(frmP);
			if (gProjectDBid == 0) {
				if (!doOpenProject()) {
					EventType newEvent;
			    	MemSet(&newEvent, sizeof(EventType), 0);
				   	newEvent.eType = appStopEvent;
				   	EvtAddEventToQueue(&newEvent);
					handled = true;
					break;
				}
			}
			projectFormInit();
	   		handled = true;
  			break;
  		case appStopEvent :
  			validateProjectEntries();
  			handled = true;
  			break;
  		case menuEvent : {
			int focusIndex = FrmGetFocus(frmP);
			FieldPtr fldP = NULL;
			if ((focusIndex != noFocus) && (FrmGetObjectType(frmP, focusIndex) == frmFieldObj))
				fldP = (FieldPtr)FrmGetObjectPtr(frmP, focusIndex);
  		
			switch (eventP->data.menu.itemID) {
					case EditCut:
						if (fldP) FldCut(fldP);
						break;
					case EditCopy:
						if (fldP) FldCopy(fldP);
						break;
					case EditPaste:
						if (fldP) FldPaste(fldP);
						break;
					case EditUndo:
						if (fldP) FldUndo(fldP);
						break;
					case OptionsDeleteHeader:				
						deleteHeader();
						break;
					case OptionsAboutOnBoardC: {
							FormPtr dlg = FrmInitForm(AboutForm);
							FrmDoDialog(dlg);
				 			FrmDeleteForm(dlg);
						}
						break;
					case OptionsChooseEditor: {
							FormPtr dlg = FrmInitForm(ChooseEditorForm);
					  		FrmSetControlGroupSelection(dlg, ChooseEditorFormGroupID, gPrefs.memoEditor + ChooseEditorPeditPushButton);
					  		FrmSetControlGroupSelection(dlg, ChooseEditorFormGroupID2, gPrefs.docEditor + ChooseEditorSrcEditPushButton);
							if (FrmDoDialog(dlg) == ChooseEditorOKButton) {
								int item = FrmGetControlGroupSelection(dlg, ChooseEditorFormGroupID2);
								if (item == FrmGetObjectIndex(dlg, ChooseEditorSrcEditPushButton))
									gPrefs.docEditor = SrcEdit;
								else if (item == FrmGetObjectIndex(dlg, ChooseEditorQEDPushButton))
									gPrefs.docEditor = QED;
								else if (item == FrmGetObjectIndex(dlg, ChooseEditorZDocPushButton))
									gPrefs.docEditor = ZDoc;
								else if (item == FrmGetObjectIndex(dlg, ChooseEditorSmartEditPushButton))
									gPrefs.docEditor = SmartEdit;
								else if (item == FrmGetObjectIndex(dlg, ChooseEditorWordSmith_DocPushButton))
									gPrefs.docEditor = WordSmith_Doc;
								else if (item == FrmGetObjectIndex(dlg, ChooseEditorQuickWordPushButton))
									gPrefs.docEditor = QuickWord;
								
								item = FrmGetControlGroupSelection(dlg, ChooseEditorFormGroupID);
								if (item == FrmGetObjectIndex(dlg, ChooseEditorPeditPushButton))
									gPrefs.memoEditor = Pedit;
								else if (item == FrmGetObjectIndex(dlg, ChooseEditorMemopadPushButton))
									gPrefs.memoEditor = MemoPad;
								else if (item == FrmGetObjectIndex(dlg, ChooseEditorWordSmith_MemoPushButton))
									gPrefs.memoEditor = WordSmith_Memo;
							}
							FrmDeleteForm(dlg);
						}
						break;
					case ProjectNewProject :
						closeProjectDB();
						if (!doNewProject()) {
							gProjectDBid = DmFindDatabase(0, gPrefs.projectName);
							loadProject();
						}
						projectFormInit();
						break;
					case ProjectOpenProject :
						closeProjectDB();
						if (!doOpenProject()) {
							gProjectDBid = DmFindDatabase(0, gPrefs.projectName);
							loadProject();
						}
						projectFormInit();
						break;
					case ProjectDeleteProject : {
							int i;
							FormPtr dlg = FrmInitForm(DeleteProjectForm);
							ListPtr lstP = getObjectPtr(dlg, DeleteProjectProjectListList);
							buildProjectList(lstP);
							FrmSetEventHandler(dlg, deleteProjectHandler);
							if (FrmDoDialog(dlg) == DeleteProjectDeleteButton) {
								int index = LstGetSelection(lstP);
								if (index != -1) {
									if (StrCompare(gPrefs.projectName, gProjectListContents[index]) == 0) {
										SndPlaySystemSound(sndError);
									}
									else {
										LocalID dbID = DmFindDatabase(0, gProjectListContents[index]);
										DmOpenRef delDB = DmOpenDatabase(0, dbID, dmModeReadWrite);
										Handle h = DmGet1Resource(PROJECT_TYPE, 1);
										if (h) {
											ProjectHeader_New *pHdr = (ProjectHeader_New *)MemHandleLock(h);
											deleteProjectFiles(pHdr, false);
											MemHandleUnlock(h);
											DmReleaseResource(h);
											DmCloseDatabase(delDB);
										}
										DmDeleteDatabase(0, dbID);
									}
								}
							}
							for (i = 0; i < gProjectListCount; i++)
								MemPtrFree(gProjectListContents[i]);
							MemPtrFree(gProjectListContents);
							gProjectListContents = NULL;
				 			FrmDeleteForm(dlg);
							projectFormInit();
						}
						break;
					case ProjectDeleteObjectCode:
						deleteProjectFiles(&gProjectHeader.prj, true);
						break;
					case ProjectRenameProject : {
							FormPtr dlg = FrmInitForm(RenameProjectForm);
							setHandleFromText(dlg, RenameProjectProjectNameField, gPrefs.projectName, true);
							if (FrmDoDialog(dlg) == RenameProjectOKButton) {
								int length;
								setTextFromHandle(dlg, RenameProjectProjectNameField, gPrefs.projectName);
								length = StrLen(gPrefs.projectName);
								if (StrStr(gPrefs.projectName, ".proj") != gPrefs.projectName + (length - 5)) {
									StrCat(gPrefs.projectName, ".proj");
								}
								DmSetDatabaseInfo(0, gProjectDBid, gPrefs.projectName, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
								gProjectDBid = DmFindDatabase(0, gPrefs.projectName);
							}
				 			FrmDeleteForm(dlg);
							StrPrintF(gFormTitle, "OnBoardC : %s", gPrefs.projectName);						
	//						FrmSetTitle(frmP, "");
	//						FrmDrawForm(frmP);
							FrmSetTitle(frmP, gFormTitle);
							eraseTitle();
							FrmDrawForm(frmP);
						}
						break;
				}
			}
			handled = true;
  			break;
	   	case ctlSelectEvent: {
				ListPtr lstP = getObjectPtr(frmP, ProjectSourcesList);
		   		switch (eventP->data.ctlSelect.controlID) {
		   			case ProjectBuildButton :
		   				if (validateProjectEntries()) {
		   					buildProject(lstP);
		   				}
				   		handled = true;
		   				break;
		   			case ProjectEditButton : {		   			
		   					int index = LstGetSelection(lstP);
		   					validateProjectEntries();
		   					if ((index == -1) || (gProjectHeader.source[index].sourceKind == Resource)) {
								GoToParamsType *pBlock = (GoToParamsType *)MemPtrNew(sizeof(GoToParamsType));
								MemPtrSetOwner(pBlock, 0);
								MemSet(pBlock, sizeof(GoToParamsType), 0);
								pBlock->dbID = gProjectHeader.source[index].sourceDBid;
								launchEditor('OnBR', pBlock, sysAppLaunchCmdGoTo, "RsrcEdit");
							}
							else
								if (gProjectHeader.source[index].sourceKind == MemoSource)
									gotoMemoPad(0, 0, gProjectHeader.source[index].sourceDBid);
								else
									if (gProjectHeader.source[index].sourceKind == Pedit32Source)
										gotoPedit32(0, 0, gProjectHeader.source[index].sourceDBid);
									else
										gotoEditor(NULL, 0, gProjectHeader.source[index].sourceDBid, gProjectHeader.source[index].sourceDB);
		   					handled = true;
		   				}
		   				break;
		   			case ProjectAddButton :
		   				validateProjectEntries();
		   				FrmGotoForm(AddToProjectForm);
				   		handled = true;
		   				break;
		   			case ProjectRemoveButton : {
		   					int i;
		   					int index = LstGetSelection(lstP);
			   				validateProjectEntries();
		   					if (index != -1) {
		   						for (i = index; i < gProjectHeader.prj.mSourceDBCount - 1; i++) {
		   							int j;
		   							gProjectHeader.source[i].sourceKind = gProjectHeader.source[i + 1].sourceKind;
		   							StrCopy(gProjectHeader.source[i].sourceDB, gProjectHeader.source[i + 1].sourceDB);
		   							gProjectHeader.source[i].sourceDBid = gProjectHeader.source[i + 1].sourceDBid;
		   							StrCopy(gProjectHeader.source[i].outputDB, gProjectHeader.source[i + 1].outputDB);
		   							gProjectHeader.source[i].outputDBid = gProjectHeader.source[i + 1].outputDBid;
		   							for (j = 0; j < MaxInclude; j++) {
			   							gProjectHeader.source[i].includeIndex[j] = gProjectHeader.source[i + 1].includeIndex[j];
		   							}
		   							gProjectHeader.source[i].includeCount = gProjectHeader.source[i + 1].includeCount;
		   						}
		   						gProjectHeader.prj.mSourceDBCount--;							
		   						projectFormInit();	
		   					}
				   			handled = true;
				   		}
		   				break;
		   		}
			}
	   		break;
        default:
            break;
	}
	return handled;
}

static void resetFieldUnderline(FormPtr frmP)
{
	int i;
	for (i = 0; i < FrmGetNumberOfObjects(frmP); i++) {
		if (FrmGetObjectType(frmP, i) == frmFieldObj) {
			FieldType *fldP = (FieldType *)FrmGetObjectPtr(frmP, i);
			FieldAttrType attr;
			FldGetAttributes(fldP, &attr);
			if (attr.underlined == solidUnderline) {
				attr.underlined = grayUnderline;
				FldSetAttributes(fldP, &attr);
			}
		}
	
	}
}
/*
static void resetFieldUnderline(FormPtr frmP)
{
	int i;
	for (i = 0; i < frmP->numObjects; i++) {
		if (frmP->objects[i].objectType == frmFieldObj) {
			FieldType *fldP = frmP->objects[i].object.field;
			if (fldP->attr.underlined == solidUnderline)
				fldP->attr.underlined = grayUnderline;
		}
	
	}
}
*/
Boolean applicationHandleEvent(EventPtr eventP)
{
	FormPtr	frmP;

	Int		formId;
	Boolean	handled = false;

	if (eventP->eType == frmLoadEvent) {
		formId = eventP->data.frmLoad.formID;
		frmP = FrmInitForm(formId);
		FrmSetActiveForm(frmP);
		resetFieldUnderline(frmP);
		switch (formId)	{
			case ProjectForm :
				FrmSetEventHandler(frmP, projectEventHandler);
				break;	
			case AddToProjectForm :
				FrmSetEventHandler(frmP, addToProjectEventHandler);
				break;	
		}
		handled = true;
	}	
	return handled;
}

void eventLoop()
{
	EventType event;
	Word error;
	
	do {
		EvtGetEvent(&event, -1);
		if (!SysHandleEvent(&event))
			if (!MenuHandleEvent(0, &event, &error))
				if (!applicationHandleEvent(&event))
					FrmDispatchEvent(&event);
	} while (event.eType != appStopEvent);
}

DWord PilotMain(Word cmd, Ptr cmdPBP, Word launchFlags)
{

	if (cmd == sysAppLaunchCmdNormalLaunch) {
 		startApplication();
		startParser();
		FrmGotoForm(ProjectForm);		
		eventLoop();
		PrefSetAppPreferences('OnBC', 1000, 1, &gPrefs, sizeof(PrefsData), false);
		stopParser();
		stopApplication();
	}
	
	return 0;
}



