/***********************************************************************
 *
 * This file is part of the OnBoard C package.  For more information on 
 * using OnBoard C, see http://groups.yahoo.com/group/onboardc.  For 
 * more information on developing OnBoard C (including submission of 
 * bug reports), see http://sourceforge.net/projects/onboardc.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 ***********************************************************************/

#include <PalmOS.h>
#include <PalmCompatibility.h>

#include "OnBoardC.h"

/* Multi-Segment Defines */
#ifdef __GNUC__
#include "codegen-seg.h"
#endif
/* End of Multi-segment defines */

char *registerName = "d0d1d2d3d4d5d6d7a0a1a2a3a4a5a6a7pc";

//Boolean rawData;

void writeToBodyBuffer(CharPtr stuff, int length)
{
/*
	if (rawData) {
		writeTokenToBuffer(Asm_EndRaw);
		rawData = false;
	}
*/

	char *p;
	while ((bodyTop + length) >= bodySize) {
		MemHandleUnlock(bodyContents);
		bodySize += 1024;
		if (MemHandleResize(bodyContents, bodySize)) error("Couldn't grow buffer");
		bodyBuffer = MemHandleLock(bodyContents);
	}
//	DmWrite(bodyBuffer, bodyTop, stuff, length);
	
	p = bodyBuffer + bodyTop;
	switch (length) {
		case 1 :
			*p = *stuff;
			break;
		case 2 :
			*p++ = *stuff++;
			*p = *stuff;
			break;
		case 3 :
			*p++ = *stuff++;
			*p++ = *stuff++;
			*p = *stuff;
			break;
		default :
			MemMove(p, stuff, length);
			break;
	}

	bodyTop += length;
}

#if 0

void emitRaw(int data)
{
	if (!rawData) {
		rawData = true;
		writeTokenToBuffer(Asm_Raw);
	}
	writeToBodyBuffer(&data, 2);
}

int genEA(Node *operand, unsigned int *ea, unsigned int *ext, int size)
{
	if (((operand->reg >= D0) && (operand->reg <= A4))
			|| (operand->reg == A7)) {
		switch (operand->op) {
			case Indirect :
			case Arrow : {
					if (operand->offset == 0) {
						*ea = 0x10 | (operand->reg - A0);
						return 0;
					}
					else {
						*ea = 0x28 | (operand->reg - A0);
						*ext = operand->offset;
						return 1;
					}
				}
				break;
			case LeftBracket : {
					if (operand->reg >= A0) {
						*ea = 0x30 | (operand->reg - A0);
						*ext = (operand->reg2 << 12);
						return 1;
					}
					else {
						*ea = 0x30 | (operand->reg2 - A0);
						*ext = (operand->reg << 12);
						return 1;
					}
				}
				break;				
			case PreDecrement : {
					*ea = 0x20 | (operand->reg - A0);
					return 0;
				}
				break;
			case PostIncrement : {
					*ea = 0x18 | (operand->reg - A0);
					return 0;
				}
				break;
			default : {
					*ea = operand->reg;
					return 0;
				}
				break;
		}
	}
	else {
		if (operand->reg == Immediate) {
			*ea = 0x3C;
			if ((size == 1) || (size == 2)) {
				*ext = operand->value;
				return 1;
			}
			else {
				*((ULong *)ext) = operand->value;
				return 2;
			}
		}
		else {
			if (operand->op == DoubleQuote) {
				return -1;
			}
			if ((operand->data.decl->depth > 0)) {
				if (operand->data.decl->data.value > 0) {
					*ea = operand->data.decl->data.value;
					return 0;
				}
				else {
					*ea = 0x28 | (operand->reg - A0);
					*ext = operand->data.decl->data.value + operand->offset;
					return 1;
				}
			}
			else {
				return -1;
			}
		}
	}
}

Boolean genMoveInstr(Node *operand1, Node *operand2, int size)
{
	int ea1, ea2;
	unsigned int ext1[2];
	unsigned int ext2[2];
	int sz1, sz2;
	int instr;
	
	sz1 = genEA(operand1, &ea1, ext1, size);
	if (sz1 == -1) return false;
	sz2 = genEA(operand2, &ea2, ext2, size);
	if (sz2 == -1) return false;
	
	ea2 = ((ea2 & 0x7) << 3) | ((ea2 >> 3) & 0x7);

	switch (size) {
		case 1: instr = 0x1000; break;
		case 2: instr = 0x3000; break;
		case 4: instr = 0x2000; break;
		default: return false;
	}
	
	emitRaw(instr + ea1 + (ea2 << 6));
	if (sz1) {
		if (sz1 == 1)
			emitRaw(ext1[0]);
		if (sz1 == 2)
			emitRaw(ext1[1]);
	}
	if (sz2) {
		if (sz2 == 1)
			emitRaw(ext2[0]);
		if (sz2 == 2)
			emitRaw(ext2[1]);
	}
	return true;
}

#endif


void writeTokenToBuffer(AsmToken token)
{
	writeToBodyBuffer((CharPtr)&token, 1);
}

void writeRegToBuffer(int reg)
{
	char r = (char)(Token_d0 + reg);
	writeToBodyBuffer(&r, 1);
}

void writeNumberToBuffer(long value)
{
	if (value == 0)
		writeTokenToBuffer(Asm_Zero);
	else {
		writeTokenToBuffer(Asm_Digit);
		writeToBodyBuffer((char *)&value, 4);
	}
}

void writeStringToBuffer(CharPtr str, int length)
{
	writeTokenToBuffer(Asm_DoubleQuote);
	writeToBodyBuffer(str, length);
//	writeTokenToBuffer(Asm_Whitespace);
}

void writeIdentifierToBuffer(CharPtr name, int length, int depth, Boolean isStatic)
{
	buf[0] = Asm_Identifier;
	if (depth >= -1) {
		buf[1] = length + 3;
		if (isStatic) buf[1] += 3;
		writeToBodyBuffer(buf, 2);
		writeToBodyBuffer(name, length);
		buf[0] = '_';
		if (depth == -1) depth = 0;	 	// forward declaration
		buf[1] = (depth >> 3) + '0';
		buf[2] = (depth & 0x7) + '0';
		writeToBodyBuffer(buf, 3);
		if (isStatic) {
			buf[1] = (gStaticIndex >> 3) + '0';
			buf[2] = (gStaticIndex & 0x7) + '0';
			writeToBodyBuffer(buf, 3);
		}
	}
	else {		// local static
		int functionIndex = -depth - 2;
		buf[1] = length + 6;
		writeToBodyBuffer(buf, 2);
		writeToBodyBuffer(name, length);
		writeToBodyBuffer("_f_", 3);
		buf[0] = '_';
		buf[1] = (functionIndex >> 3) + '0';
		buf[2] = (functionIndex & 0x7) + '0';
		writeToBodyBuffer(buf, 3);
	}
//	writeTokenToBuffer(Asm_Whitespace);
}

/*
void writeNumberToBodyBuffer(long value)
{
	int length;
	
	if (value == 0x80000000)
		StrCopy(buf, "$80000000");
	else
		StrIToA(buf, value);
	length = StrLen(buf);

	while ((bodyTop + length) >= bodySize) {
		MemHandleUnlock(bodyContents);
		bodySize += 1024;
		if (MemHandleResize(bodyContents, bodySize)) error("Couldn't grow buffer");
		bodyBuffer = MemHandleLock(bodyContents);
	}
	MemMove(bodyBuffer + bodyTop, buf, length);
	bodyTop += length;
}
*/

/*
Boolean operandHasReference(Node *operand)
{
	if (((operand->reg >= D0) && (operand->reg <= A4))
			|| (operand->reg == A7)) {
		return false;
	}
	else {
		if (operand->reg == Immediate) {
			return false;
		}
		else {
			if (operand->op == DoubleQuote) {
				return true;
			}
			if ((operand->data.decl->depth > 0)) {
				return false;
			}
			else {
				return true;
			}
		}
	}
}
*/

void emitOperand(Node *operand)
{
	if (((operand->reg >= D0) && (operand->reg <= A4))
			|| (operand->reg == A7)) {
		switch (operand->op) {
			case Indirect :
			case Arrow : {
					writeNumberToBuffer(operand->offset);
					writeTokenToBuffer(Asm_LeftParen);
					writeRegToBuffer(operand->reg);
					writeTokenToBuffer(Asm_RightParen);
				}
				break;
			case LeftBracket : {
					writeTokenToBuffer(Asm_LeftParen);
					writeRegToBuffer(operand->reg);
					writeTokenToBuffer(Asm_Comma);
					writeRegToBuffer(operand->reg2);
					writeTokenToBuffer(Asm_RightParen);
				}
				break;				
			case PreDecrement : {
					writeTokenToBuffer(Asm_Minus);
					writeTokenToBuffer(Asm_LeftParen);
					writeRegToBuffer(operand->reg);
					writeTokenToBuffer(Asm_RightParen);
				}
				break;
			case PostIncrement : {
					writeTokenToBuffer(Asm_LeftParen);
					writeRegToBuffer(operand->reg);
					writeTokenToBuffer(Asm_RightParen);
					writeTokenToBuffer(Asm_Plus);
				}
				break;
			default : {
					writeTokenToBuffer((AsmToken)(EA_d0 + operand->reg));//writeRegToBuffer(operand->reg);
				}
				break;
		}
	}
	else {
		if (operand->reg == Immediate) {
			writeTokenToBuffer(Asm_Number);
			writeNumberToBuffer(operand->value);
		}
		else {
			if (operand->op == DoubleQuote) {
				int nameLength;
				addCodeStringConstant(operand->data.identifier.name, operand->data.identifier.length,
																				buf, &nameLength);
/*
				addStringConstant(operand->data.identifier.name, operand->data.identifier.length,
																				buf, &nameLength);
*/
				writeToBodyBuffer(buf, nameLength);
				writeTokenToBuffer(Asm_LeftParen);
				writeTokenToBuffer(Token_PC);
//				writeTokenToBuffer(Token_a5);
				writeTokenToBuffer(Asm_RightParen);
			}
			else {
				if ((operand->data.decl->depth > 0)) {
					if (operand->data.decl->data.value > 0)
						writeTokenToBuffer((AsmToken)(EA_d0 + operand->data.decl->data.value));
					else {
						writeTokenToBuffer(EA_local);
						writeNumberToBuffer(operand->data.decl->data.value + operand->offset);
					}
				}
				else {
					int i;
					if (getPoolIndex(operand->data.decl, &i)) {
						buf[0] = (Char) PoolID;
						buf[1] = i >> 8;
						buf[2] = i & 0xFF;
						writeToBodyBuffer(&buf[0], 3);
					}
					else {
						if (i != -1) {
							buf[0] = (Char) DefinePoolID;
							buf[1] = i >> 8;
							buf[2] = i & 0xFF;
							writeToBodyBuffer(&buf[0], 3);
						}						
						writeIdentifierToBuffer(operand->data.decl->name, operand->data.decl->length, operand->data.decl->depth, operand->data.decl->isStatic);
					}
					if (operand->offset != 0) {
						writeTokenToBuffer(Asm_Plus);
						writeNumberToBuffer(operand->offset);
					}
					writeTokenToBuffer(Asm_LeftParen);
					writeRegToBuffer(operand->reg);
					writeTokenToBuffer(Asm_RightParen);
				}
			}
		}
	}
}

void writeLabelToBuffer(int labelNumber)
{
	int length;
	buf[0] = Asm_Identifier;
	buf[2] = 'L';
	buf[3] = '_';
	StrIToA(&buf[4], labelNumber);
	buf[1] = length = StrLen(&buf[2]);	
	writeToBodyBuffer(buf, length + 2);
}

void emitBranch(AsmToken token, int target)
{
	target &= 0x3FFF;
#ifdef DO_CODEGEN
	writeTokenToBuffer(token);
	writeLabelToBuffer(target);
#endif
}

void emitDBranch(AsmToken token, Node *operand, int target)
{
	target &= 0x3FFF;
#ifdef DO_CODEGEN
	writeTokenToBuffer(token);
	emitOperand(operand);
	writeTokenToBuffer(Asm_Comma);
	writeLabelToBuffer(target);
#endif
}

void emitUserLabel(CharPtr uLabel, int length)
{
#ifdef DO_CODEGEN
	buf[0] = Asm_Identifier;
	buf[1] = length;	
	writeToBodyBuffer(buf, 2);
	writeToBodyBuffer(uLabel, length);
#endif
}

void emitUserBranch(CharPtr uLabel, int length)
{
#ifdef DO_CODEGEN
	writeTokenToBuffer(Token_Bra);
	buf[0] = Asm_Identifier;
	buf[1] = length;	
	writeToBodyBuffer(buf, 2);
	writeToBodyBuffer(uLabel, length);
#endif
}

void emitLabel(int label)
{
	label &= 0x3FFF;
#ifdef DO_CODEGEN
	writeLabelToBuffer(label & 0x3FFF);
#endif
}

void emitUnary(AsmToken token, AsmToken sizeSpec, Node *operand, Boolean addOperandSize)
{
#ifdef DO_CODEGEN
	writeTokenToBuffer(token);
	if (sizeSpec != Unsized) writeTokenToBuffer(sizeSpec);
	if (addOperandSize) {
		switch (getTypeSize(operand->typeIndex)) {
			case 1 : 
				writeTokenToBuffer(ByteSpec);
				break;
			case 2 : 
				writeTokenToBuffer(WordSpec);
				break;
			case 4 : 
				writeTokenToBuffer(LongSpec);
				break;
			default : {
				int *p = (int*)3;
				*p = 1;
				error("bad case 6");
				}
				break;
		}
	}
	emitOperand(operand);	
#endif
}

void emitConstant(AsmToken token, AsmToken sizeSpec, int constant)
{
#ifdef DO_CODEGEN
	writeTokenToBuffer(token);
	writeTokenToBuffer(sizeSpec);
	writeNumberToBuffer(constant);
#endif
}

void emitBinary(AsmToken token, AsmToken sizeSpec, Node *operand1, Node *operand2, Boolean addOperandSize)
{
#ifdef DO_CODEGEN
	writeTokenToBuffer(token);
	if (sizeSpec != Unsized) writeTokenToBuffer(sizeSpec);
	if (addOperandSize) {
		switch (getTypeSize(operand1->typeIndex)) {
			case 1 : 
				writeTokenToBuffer(ByteSpec);
				break;
			case 2 : 
				writeTokenToBuffer(WordSpec);
				break;
			case 4 : 
				writeTokenToBuffer(LongSpec);
				break;
			default :
				error("bad case 7");
				break;
		}
	}
	emitOperand(operand1);	
	writeTokenToBuffer(Asm_Comma);
	emitOperand(operand2);	
#endif
}

void emitLibCall(char *libname, int libNameLength)
{
	int i;
	int length = 3;
	
	buf[0] = Token_Jsr;
	buf[1] = Asm_Identifier;
	buf[2] = libNameLength;
	for (i = 0; i < libNameLength; i++)
		buf[length++] = libname[i]; 
	buf[length++] = Asm_LeftParen;
	buf[length++] = Token_PC;
	buf[length++] = Asm_RightParen;								
	writeToBodyBuffer(buf, length);
}

Boolean registerInUse[13];

int getFreeRegister(int start, int end)
{
	int i;
	for (i = start; i <= end; i++) {
		if (!registerInUse[i]) {
			registerInUse[i] = true;
			return i;
		}
	}
	error("Out of registers");
	return -1;	
}

void freeRegister(Node *operand)
{
	if ((operand->reg >= D0) && (operand->reg <= A4))
		registerInUse[(int) operand->reg] = false;
	if ((operand->op == LeftBracket)
			&& (operand->reg2 >= D0) && (operand->reg2 <= A4))
		registerInUse[(int) operand->reg2] = false;
}

Node regTgt;

void emitMove(Node *operand1, Node *operand2, Boolean addOperandSize)
{
	if ((operand1->op == Decl) && (operand1->reg == Immediate)) {
		if ((operand1->value <= 127) && (operand1->value >= -128)
				 && (((operand2->op == Decl) && (operand2->reg >= D0) && (operand2->reg <= D7))
				 	|| ((operand2->op == Decl) 	// local variable assigned to register
				 			&& (operand2->reg == A6)
				 				&& (operand2->data.decl->data.value > 0) ))
				 ) {
			emitBinary(Token_Moveq, Unsized, operand1, operand2, false);
			return;
		}
		else {		
			if (operand1->value == 0) {
				if ((operand2->op == Decl) && (operand2->reg >= A0) && (operand2->reg <= A4))
					emitBinary(Token_Suba, Unsized, operand2, operand2, true);
				else
					emitUnary(Token_Clr, Unsized, operand2, addOperandSize);
				return;
			}
		}
	}
	emitBinary(Token_Move, Unsized, operand1, operand2, addOperandSize);
}

Boolean loadIntoDReg(Node *operand)
{
#ifdef DO_CODEGEN
	if ((operand->reg > D7) || (operand->reg < D0)) {
		int dReg = getFreeRegister(D0, D7);
		if (dReg == -1) return false;
		regTgt.op = Decl;
		regTgt.offset = 0;
		regTgt.reg = dReg;
		if ((operand->reg == Immediate)
				 && (operand->value <= 127)
				 && (operand->value >= -128))
			emitBinary(Token_Moveq, Unsized, operand, &regTgt, false);
		else
			emitMove(operand, &regTgt, true);
		freeRegister(operand);
		operand->reg = dReg;
		operand->op = Decl;
		operand->offset = 0;
	}
#endif	
	return true;
}

Boolean loadConstant(Node *node, int value)
{
	node->reg = getFreeRegister(D0, D7);
	if (node->reg == -1) return false;
	node->op = Decl;
	node->offset = 0;
	regTgt.reg = Immediate;
	regTgt.typeIndex = LongTypeIndex;
	regTgt.value = value;
	emitMove(&regTgt, node, true);
	return true;
}

Boolean loadIntoAReg(Node *operand)
{
#ifdef DO_CODEGEN
	if ((operand->reg >= A0) && (operand->reg <= A4)
			&& (operand->op == Decl))
		return true;
	else {
		int aReg;
		freeRegister(operand);
		aReg = getFreeRegister(A0, A4);
		if (aReg == -1) return false;
		regTgt.op = Decl;
		regTgt.offset = 0;
		regTgt.reg = aReg;
		emitMove(operand, &regTgt, true);
		operand->reg = aReg;
		operand->op = Decl;
		operand->offset = 0;
	}
#endif
	return true;
}

Boolean	emitConditionalBranch(Node *result, int trueBranch, int falseBranch)
{
#ifdef DO_CODEGEN
	if (result->op != Goto) {
		if (!loadIntoDReg(result)) return false;
		emitUnary(Token_Tst, Unsized, result, true);
		freeRegister(result);
		emitBranch(Token_Beq, falseBranch);
		emitBranch(Token_Bra, trueBranch);
	}
#endif
	return true;
}

// keeping these global so they're not on the stack - make sure not
// to assume they're not getting hit in recursive calls
Node stackOperand;
Node stackAdjust;

static int pushValue(Node *node)
{
    int result;
	stackOperand.op = PreDecrement;
	stackOperand.reg = A7;
	stackOperand.typeIndex = node->typeIndex;
    if (stackOperand.typeIndex == DoubleTypeIndex) {
        emitMoveDouble(node, &stackOperand);
        result = 8;
    }
    else {
    	emitMove(node, &stackOperand, true);
    	result = 2;
    	if (getTypeSize(stackOperand.typeIndex) == 4) result += 2;
    }
	freeRegister(node);
    return result;
}

int saveRegisters()
{
	int result = 0;
	int i;
	stackAdjust.reg = A7;
	stackAdjust.op = PreDecrement;
	stackAdjust.typeIndex = LongTypeIndex;
	regTgt.op = Decl;
	regTgt.offset = 0;
	regTgt.typeIndex = LongTypeIndex;
	for (i = D0; i <= A4; i++) {
		if (((i >= D0) && (i <= D3)) || ((i >= A0) && (i <= A2))) {
			if (registerInUse[i]) {
				result |= 1 << i;
				regTgt.reg = i;
				emitMove(&regTgt, &stackAdjust, true);
				registerInUse[i] = false;
			}
		}
	}
	return result;
}

void restoreRegisters(int savedRegs)
{
	int i;
	stackAdjust.reg = A7;
	stackAdjust.op = PostIncrement;
	stackAdjust.typeIndex = LongTypeIndex;
	regTgt.op = Decl;
	regTgt.offset = 0;
	regTgt.typeIndex = LongTypeIndex;
	for (i = A4; i >= D0; i--) {
		if (((i >= D0) && (i <= D3)) || ((i >= A0) && (i <= A2))) {
			registerInUse[i] = false;
			if ((savedRegs & (1 << i)) != 0) {
				regTgt.reg = i;
				emitMove(&stackAdjust, &regTgt, true);
				registerInUse[i] = true;
			}
		}
	}
}

int getFreeFromSavedRegisters(int start, int finish, int savedRegs)
{
	int i;
	for (i = start; i <= finish; i++) {
		if ((savedRegs & (1 << i)) == 0) {
			return i;
		}
	}
	error("Out of registers");
	return -1;	
}

static void emitFPcode(int index)
{
	writeTokenToBuffer(Token_Moveq);
	writeTokenToBuffer(Asm_Number);
	writeNumberToBuffer(index);
	writeTokenToBuffer(Asm_Comma);
	writeTokenToBuffer(Token_d2);
	emitConstant(Token_Dc, WordSpec, 0x4E4F);	
	emitConstant(Token_Dc, WordSpec, 0x0306);	
}

void setOperand(Node *operand, int reg, int op, int typeIndex)
{
	operand->op = op;
	if (reg == -1)
		operand->reg = getFreeRegister(D0, D7);
	else
		operand->reg = reg;
	operand->offset = 0;
	operand->typeIndex = typeIndex;
}

void cleanStack(int stackDepth)
{
	setOperand(&stackOperand, A7, Decl, LongTypeIndex);
    if (stackDepth <= 8) {
	    stackAdjust.reg = Immediate;
	    stackAdjust.value = stackDepth;
	    stackAdjust.typeIndex = LongTypeIndex;
	    emitBinary(Token_Addq, Unsized, &stackAdjust, &stackOperand, true);
	}
	else {
	    stackAdjust.reg = A7;
        stackAdjust.offset = stackDepth;
	    stackAdjust.op = Arrow;
	    stackAdjust.typeIndex = LongTypeIndex;
	    emitBinary(Token_Lea, Unsized, &stackAdjust, &stackOperand, false);
    }
}

/*
 one or two operands, a pointer to a temp is always passed.
 Return Double result in temp space pointed by node
*/
static Boolean emitFPSequence1(int opcode, Node *operand1, Node *operand2, Node *node)
{
    Node *tempSpace;
    int stackDepth = 4;                 // address of temp
	int savedRegs = saveRegisters();
	if (operand2 != NULL) stackDepth += pushValue(operand2);														
	stackDepth += pushValue(operand1);
    tempSpace = newTemp(DoubleTypeIndex, gStatementIndex);
	emitUnary(Token_Pea, Unsized, tempSpace, false);
	emitFPcode(opcode);
    cleanStack(stackDepth);
	// get the result reg from A0 into someplace safe
	stackOperand.reg = A0;
	stackOperand.op = Decl;
	stackOperand.offset = 0;
	stackOperand.typeIndex = LongTypeIndex;
	node->reg = getFreeFromSavedRegisters(A0, A4, savedRegs);
	node->typeIndex = LongTypeIndex;
	if (node->reg == -1) return false;
	node->op = Decl;
	if (node->reg != A0)
		emitMove(&stackOperand, node, true);
	restoreRegisters(savedRegs);
	node->op = Indirect;	// it's an indirect result
	node->typeIndex = DoubleTypeIndex;
	*node = *tempSpace;
    return true;
}

/*
	one or two operands, no temp.
	Result size (4 bytes or less) given, result in D register
*/
static Boolean emitFPSequence2(int opcode, Node *operand1, Node *operand2, Node *node, int resultSize)
{
	int savedRegs = saveRegisters();
    int stackDepth = 0;
	if (operand2 != NULL) stackDepth += pushValue(operand2);														
	stackDepth += pushValue(operand1);
	emitFPcode(opcode);
    cleanStack(stackDepth);
	// get the result reg from D0 into someplace safe
	stackOperand.reg = D0;
	stackOperand.op = Decl;
	stackOperand.offset = 0;
	stackOperand.typeIndex = resultSize;
	node->reg = getFreeFromSavedRegisters(D0, D7, savedRegs);
	node->typeIndex = resultSize;
	if (node->reg == -1) return false;
	if (node->reg != D0)
		emitMove(&stackOperand, node, true);
	restoreRegisters(savedRegs);
	node->op = Decl;
    return true;
}

Boolean emitConvert(Node *operand, int toTypeIndex)
{
    Node *tempSpace;
	Node tgt, imm;

	if (toTypeIndex == operand->typeIndex)
		return true;
		
	switch (toTypeIndex) {
		default :
			operand->typeIndex = toTypeIndex;
			return true;
		case FloatTypeIndex : {
                tempSpace = newTemp(LongTypeIndex, gStatementIndex);
				switch (operand->typeIndex) {
					case DoubleTypeIndex :
                        if (!emitFPSequence2(12, operand, NULL, operand, LongTypeIndex)) return false;
                        break;
					case ShortTypeIndex :
					case IntTypeIndex :
					case UnsignedCharTypeIndex :
					case UnsignedShortTypeIndex :
					case UnsignedIntTypeIndex :
					case CharTypeIndex :
						if (!emitConvert(operand, LongTypeIndex)) return false;
                                            // fall thru...
					case LongTypeIndex :
                        if (!emitFPSequence2(4, operand, NULL, operand, LongTypeIndex)) return false;
                        break;
					case UnsignedLongTypeIndex :
                        if (!emitFPSequence2(3, operand, NULL, operand, UnsignedLongTypeIndex)) return false;
                        break;
                    default :
						error("illegal combination of types");
						return false;
				}
                emitMove(operand, tempSpace, true);
                *operand = *tempSpace;
                operand->typeIndex = FloatTypeIndex;
                return true;
			}
		case DoubleTypeIndex : {
                tempSpace = newTemp(DoubleTypeIndex, gStatementIndex);
				switch (operand->typeIndex) {
					case FloatTypeIndex :
						if (!emitFPSequence1(11, operand, NULL, operand)) return false;
                            break;
					case UnsignedLongTypeIndex :
						if (!emitFPSequence1(7, operand, NULL, operand)) return false;
                            break;
					case ShortTypeIndex :
					case IntTypeIndex :
					case UnsignedCharTypeIndex :
					case UnsignedShortTypeIndex :
					case UnsignedIntTypeIndex :
					case CharTypeIndex :
                        if (!emitConvert(operand, LongTypeIndex)) return false;
                        // fall thru...
                    case LongTypeIndex :
						if (!emitFPSequence1(8, operand, NULL, operand)) return false;
                            break;
                    default :
						error("illegal combination of types");
						return false;
				}
                emitMoveDouble(operand, tempSpace);
                *operand = *tempSpace;
                return true;
			}
		case UnsignedLongTypeIndex :
		case LongTypeIndex : {
				switch (operand->typeIndex) {
					case FloatTypeIndex :
                        if (toTypeIndex == UnsignedLongTypeIndex) {
                            if (!emitFPSequence2(17, operand, NULL, operand, UnsignedLongTypeIndex)) return false;
                        }
                        else {
                            if (!emitFPSequence2(18, operand, NULL, operand, LongTypeIndex)) return false;
                        }
                        return true;
					case DoubleTypeIndex :
                        if (toTypeIndex == UnsignedLongTypeIndex) {
                            if (!emitFPSequence2(21, operand, NULL, operand, UnsignedLongTypeIndex)) return false;
                        }
                        else {
                            if (!emitFPSequence2(22, operand, NULL, operand, LongTypeIndex)) return false;
                        }
                        return true;
					case LongTypeIndex :
					case UnsignedLongTypeIndex :
						operand->typeIndex = toTypeIndex;
						return true;
					case ShortTypeIndex :
					case IntTypeIndex :
						if (operand->reg == Immediate) {
							operand->value = (int)(operand->value);
						}
						else {
							if (!loadIntoDReg(operand))
								return false;
							emitUnary(Token_Ext, LongSpec, operand, false);
						}
						operand->typeIndex = toTypeIndex;
						return true;
					case UnsignedCharTypeIndex :
					case UnsignedShortTypeIndex :
					case UnsignedIntTypeIndex : {
							if (operand->reg == Immediate) {
								if (operand->typeIndex == UnsignedCharTypeIndex)
									operand->value &= 0xFF;
								else
									operand->value &= 0xFFFF;
								operand->typeIndex = toTypeIndex;
							}
							else {
								tgt.op = Decl;
								tgt.offset = 0;
								tgt.reg = getFreeRegister(D0, D7);
								if (tgt.reg == -1) return false;
								tgt.typeIndex = toTypeIndex;
								imm.reg = Immediate;
								imm.typeIndex = LongTypeIndex;
								imm.value = 0;
								emitBinary(Token_Moveq, Unsized, &imm, &tgt, false);
								emitMove(operand, &tgt, true);
								freeRegister(operand);
								*operand = tgt;
							}
							return true;
						}
					case CharTypeIndex :
						if (operand->reg == Immediate) {
							operand->value = (char)(operand->value);
						}
						else {
							if (!loadIntoDReg(operand))
								return false;
							emitUnary(Token_Ext, WordSpec, operand, false);
							emitUnary(Token_Ext, LongSpec, operand, false);
						}
						operand->typeIndex = toTypeIndex;
						return true;
					default :
						// allow pointer to long & unsigned long
						if (isPointer(operand->typeIndex)) {
							operand->typeIndex = toTypeIndex;
							return true;
						}
						else {
							error("illegal combination of types");
							return false;
						}
				}
			}
		case UnsignedIntTypeIndex :
		case IntTypeIndex :
		case UnsignedShortTypeIndex :
		case ShortTypeIndex : {
				switch (operand->typeIndex) {
					case FloatTypeIndex :
                            if (!emitFPSequence2(18, operand, NULL, operand, LongTypeIndex)) return false;
                            return emitConvert(operand, toTypeIndex);
					case DoubleTypeIndex :
                            if (!emitFPSequence2(22, operand, NULL, operand, LongTypeIndex)) return false;
                            return emitConvert(operand, toTypeIndex);
					case LongTypeIndex :
					case UnsignedLongTypeIndex : {
							if (operand->reg == Immediate) {
								operand->value &= 0xFFFF;
							}
							else {
								if (!loadIntoDReg(operand)) 
									return false;
								imm.reg = Immediate;
								imm.typeIndex = LongTypeIndex;
								imm.value = 0xFFFF;
								emitBinary(Token_Andi, Unsized, &imm, operand, true);
							}
							operand->typeIndex = toTypeIndex;
							return true;
						}
					case ShortTypeIndex :
					case IntTypeIndex :
					case UnsignedShortTypeIndex :
					case UnsignedIntTypeIndex :
						return true;
					case UnsignedCharTypeIndex : {
							if (operand->reg == Immediate) {
								operand->value &= 0xFF;
							}
							else {
								tgt.op = Decl;
								tgt.offset = 0;
								tgt.reg = getFreeRegister(D0, D7);
								if (tgt.reg == -1) return false;
								tgt.typeIndex = toTypeIndex;
								imm.reg = Immediate;
								imm.typeIndex = IntTypeIndex;
								imm.value = 0;
								emitBinary(Token_Moveq, Unsized, &imm, &tgt, false);
								emitMove(operand, &tgt, true);
								freeRegister(operand);
								*operand = tgt;
							}
							return true;
						}
						break;
					case CharTypeIndex :
						if (operand->reg == Immediate) {
							operand->value = (char)(operand->value);
						}
						else {
							if (!loadIntoDReg(operand))
								break;
							emitUnary(Token_Ext, WordSpec, operand, false);
						}
						operand->typeIndex = toTypeIndex;
						return true;
					default :
						error("illegal combination of types");
						return false;
				}	
			}
		case UnsignedCharTypeIndex :
		case CharTypeIndex : {
				switch (operand->typeIndex) {
					case FloatTypeIndex :
                        if (!emitFPSequence2(18, operand, NULL, operand, LongTypeIndex)) return false;
                        return emitConvert(operand, toTypeIndex);
					case DoubleTypeIndex :
                        if (!emitFPSequence2(22, operand, NULL, operand, LongTypeIndex)) return false;
                        return emitConvert(operand, toTypeIndex);
					case LongTypeIndex :
					case UnsignedLongTypeIndex : {
							if (operand->reg == Immediate) {
								operand->value &= 0xFF;
							}
							else {
								if (!loadIntoDReg(operand)) 
									return false;
								imm.reg = Immediate;
								imm.typeIndex = LongTypeIndex;
								imm.value = 0xFF;
								emitBinary(Token_Andi, Unsized, &imm, operand, true);
							}
							operand->typeIndex = toTypeIndex;
							return true;
						}
					case ShortTypeIndex :
					case IntTypeIndex :
					case UnsignedShortTypeIndex :
					case UnsignedIntTypeIndex : {
							if (operand->reg == Immediate) {
								operand->value &= 0xFF;
							}
							else {
								if (!loadIntoDReg(operand)) 
									return false;
								imm.reg = Immediate;
								imm.typeIndex = IntTypeIndex;
								imm.value = 0xFF;
								emitBinary(Token_Andi, Unsized, &imm, operand, true);
							}
							operand->typeIndex = toTypeIndex;
							return true;
						}
					case UnsignedCharTypeIndex :
					case CharTypeIndex :
						return true;
					default :
						error("illegal combination of types");
						return false;
				}	
		}
	}
}

void emitInlineCode(Declaration *theFunction)
{
	int i;
	int inlineCount = getInlineCount(theFunction);
	for (i = 0; i < inlineCount; i++)
		emitConstant(Token_Dc, WordSpec, getInlineWord(theFunction, i));
}

void emitMoveDouble(Node *operand2, Node *operand1)
// the assumption is that all doubles are memory-based (or constants)
{
	operand2->typeIndex = LongTypeIndex;
	operand1->typeIndex = LongTypeIndex;
	if ((operand2->op == Decl) && (operand2->reg == Immediate)) {
		operand1->offset += 4;
		operand2->value = operand2->data.dBits.d2;
		emitMove(operand2, operand1, true);
		operand1->offset -= 4;
		operand2->value = operand2->data.dBits.d1;
		emitMove(operand2, operand1, true);
	}
	else {
		operand2->offset += 4;
		operand1->offset += 4;
		emitMove(operand2, operand1, true);
		operand2->offset -= 4;
		operand1->offset -= 4;
		emitMove(operand2, operand1, true);
	}
	operand1->typeIndex = DoubleTypeIndex;
	operand2->typeIndex = DoubleTypeIndex;
}

Node fpImmediate;

Boolean doBitfieldExtraction(Node *node)
{
	// bitWidth is the # bits in the field
	// depth is the offset of the hi-bit
	Declaration *d = (Declaration *)(node->data.children.node2);
	Boolean signedField = isSigned(node->typeIndex);
	node->typeIndex = UnsignedCharTypeIndex;
	stackOperand.op = Decl;
	stackOperand.reg = Immediate;
#if 0
	if ((d->depth >> 3) > (((d->depth + d->bitWidth - 1) >> 3) + 1)) {	// bitfield straddles 2 or more byte boundaries
		int dReg = getFreeRegister(D0, D7);
		if (dReg == -1) return false;
		regTgt.op = Decl;
		regTgt.offset = 0;
		regTgt.reg = dReg;
		emitMove(node, &regTgt, true);
		stackOperand.typeIndex = UnsignedLongTypeIndex;
		stackOperand.value = 0;		// shift 8

		for (i = 0; i < 3; i++) {
			emitBinary(Token_Lsl, Unsized, &stackOperand, &regTgt, true);
			node->offset++;
			emitMove(node, &regTgt, true);
		}

		if (signedField) {
			int i;
			stackOperand.value = d->depth & 0x7;
			if (stackOperand.value != 0)
				emitBinary(Token_Lsl, Unsized, &stackOperand, &regTgt, true);
			i = 32 - d->bitWidth;
			while (i >= 8) {
				stackOperand.value = 0;
				emitBinary(Token_Asr, Unsized, &stackOperand, &regTgt, true);
				i -= 8;
			}	
			stackOperand.value = i;			
			if (stackOperand.value != 0)
				emitBinary(Token_Asr, Unsized, &stackOperand, &regTgt, true);
			regTgt.typeIndex = IntTypeIndex;
		}
		else {
			int i = 32 - (d->depth & 0x7) - d->bitWidth;
			while (i > 8) {
				stackOperand.value = 0;
				emitBinary(Token_Lsr, Unsized, &stackOperand, &regTgt, true);
				i -= 8;
			}
			if (i != 0) {
				stackOperand.value = i;
				emitBinary(Token_Lsr, Unsized, &stackOperand, &regTgt, true);
			}
			stackOperand.value = (unsigned long)0xFFFFFFFF >> (32 - d->bitWidth);
			regTgt.typeIndex = UnsignedLongTypeIndex;
			stackOperand.typeIndex = LongTypeIndex;
			emitBinary(Token_Andi, Unsized, &stackOperand, &regTgt, true);
		}
		*node = regTgt;
	}
	else 
#endif	
		 {
		if ((d->depth >> 3) != ((d->depth + d->bitWidth - 1) >> 3)) {	// bitfield straddles byte boundary
			int dReg = getFreeRegister(D0, D7);
			if (dReg == -1) return false;
			regTgt.op = Decl;
			regTgt.offset = 0;
			regTgt.reg = dReg;
			emitMove(node, &regTgt, true);
	
			stackOperand.typeIndex = UnsignedShortTypeIndex;
			stackOperand.value = 0;		// shift 8
			emitBinary(Token_Lsl, Unsized, &stackOperand, &regTgt, true);
			node->offset++;
			emitMove(node, &regTgt, true);
	
			if (signedField) {
				stackOperand.value = d->depth & 0x7;
				if (stackOperand.value != 0)
					emitBinary(Token_Lsl, Unsized, &stackOperand, &regTgt, true);
				stackOperand.value = 16 - d->bitWidth;
				if (stackOperand.value >= 8) {
					stackOperand.value = 0;
					emitBinary(Token_Asr, Unsized, &stackOperand, &regTgt, true);
					stackOperand.value = (16 - d->bitWidth) - 8;
				}				
				if (stackOperand.value != 0)
					emitBinary(Token_Asr, Unsized, &stackOperand, &regTgt, true);
				regTgt.typeIndex = IntTypeIndex;
			}
			else {
				stackOperand.value = 16 - (d->depth & 0x7) - d->bitWidth;
				if (stackOperand.value != 0)
					emitBinary(Token_Lsr, Unsized, &stackOperand, &regTgt, true);
				stackOperand.value = (unsigned int)0xFFFF >> (16 - d->bitWidth);
				regTgt.typeIndex = UnsignedIntTypeIndex;
				stackOperand.typeIndex = IntTypeIndex;
				emitBinary(Token_Andi, Unsized, &stackOperand, &regTgt, true);
			}
			*node = regTgt;
		}
		else {
			if (!loadIntoDReg(node)) return false;
			stackOperand.typeIndex = UnsignedCharTypeIndex;
			if (signedField) {
				stackOperand.value = d->depth & 0x7;
				if (stackOperand.value != 0)
					emitBinary(Token_Lsl, Unsized, &stackOperand, &regTgt, true);
				stackOperand.value = 8 - d->bitWidth;
				if (stackOperand.value != 0)
					emitBinary(Token_Asr, Unsized, &stackOperand, &regTgt, true);
				node->typeIndex = CharTypeIndex;
			}
			else {
				stackOperand.value = 8 - (d->depth & 0x7) - d->bitWidth;
				if (stackOperand.value != 0)
					emitBinary(Token_Lsr, Unsized, &stackOperand, node, true);
				stackOperand.value = (unsigned int)0xFF >> (8 - d->bitWidth);
				node->typeIndex = UnsignedCharTypeIndex;
				stackOperand.typeIndex = IntTypeIndex;
				emitBinary(Token_Andi, Unsized, &stackOperand, node, true);
			}
		}
	}
	return true;
}

Boolean	doBitFieldInsertion(Node *node, Node *value)
{
	// assigning to a bitfield, suppress the bitfield-get in Arrow
	// and just generate into operand1 the byte pointer to the
	// start of the field
	Node *operand1;
	Declaration *d = (Declaration *)(node->data.children.node1->data.children.node2);
	node->data.children.node1->data.children.node2 = NULL;
	operand1 = generateCode(node->data.children.node1, -1, -1);
	if (operand1 == NULL) return false;
	node->data.children.node1->data.children.node2 = (Node *)d;
	stackAdjust.op = Decl;
	stackAdjust.offset = 0;
	stackAdjust.reg = Immediate;
	if (!loadIntoDReg(value)) return false;
	operand1->typeIndex = UnsignedCharTypeIndex;

	if ((d->depth >> 3) != ((d->depth + d->bitWidth - 1) >> 3)) {	// bitfield straddles byte boundary
		unsigned int mask = (unsigned int)0xFFFF >> (16 - d->bitWidth);
		int loBit = 16 - (d->depth & 0x7) - d->bitWidth;
		value->typeIndex = UnsignedShortTypeIndex;
		stackAdjust.typeIndex = UnsignedShortTypeIndex;
		stackAdjust.value = mask;
		emitBinary(Token_Andi, Unsized, &stackAdjust, value, true);
		stackAdjust.value = loBit;
		if (stackAdjust.value != 0)
			emitBinary(Token_Lsl, Unsized, &stackAdjust, value, true);

		stackAdjust.value = (~(mask << loBit)) & 0xFF;
		stackAdjust.typeIndex = UnsignedCharTypeIndex;
		value->typeIndex = UnsignedCharTypeIndex;
		operand1->offset++;
		emitBinary(Token_Andi, Unsized, &stackAdjust, operand1, true);
		emitBinary(Token_Or, Unsized, value, operand1, true);

		operand1->offset--;
		stackAdjust.value = (~(mask << loBit)) >> 8;
		emitBinary(Token_Andi, Unsized, &stackAdjust, operand1, true);
		stackAdjust.value = 0;			// shift 8
		stackAdjust.typeIndex = UnsignedIntTypeIndex;
		emitBinary(Token_Lsr, Unsized, &stackAdjust, value, true);
		emitBinary(Token_Or, Unsized, value, operand1, true);
	}
	else {
		unsigned int mask = (unsigned int)0xFF >> (8 - d->bitWidth);
		value->typeIndex = UnsignedCharTypeIndex;
		stackAdjust.typeIndex = UnsignedCharTypeIndex;
		stackAdjust.value = mask;
		emitBinary(Token_Andi, Unsized, &stackAdjust, value, true);
		stackAdjust.value = 8 - (d->depth & 0x7) - d->bitWidth;
		if (stackAdjust.value != 0)
			emitBinary(Token_Lsl, Unsized, &stackAdjust, value, true);
		stackAdjust.value = ~(mask << stackAdjust.value);
		emitBinary(Token_Andi, Unsized, &stackAdjust, operand1, true);
		emitBinary(Token_Or, Unsized, value, operand1, true);
	}	
	return true;
}

Node *generateCode(Node *node, int trueBranch, int falseBranch)
{
	Node *operand1 = NULL;
	Node *operand2 = NULL;
	Node *arg      = NULL;
	int interBranch;
	int savedRegs;
	int stackDepth;
	
	switch (node->op) {
		default :
			error("bad case 8");
			break;
		case MultiplyEqual : {
				operand1 = generateCode(node->data.children.node1, -1, -1);
				if (operand1 == NULL) return NULL;
				operand2 = generateCode(node->data.children.node2, -1, -1);
				if (operand2 == NULL) return NULL;
				switch (node->typeIndex) {
					case UnsignedLongTypeIndex :
					case LongTypeIndex : {
							freeRegister(operand2);
							savedRegs = saveRegisters();		
							longMultiplyUsed = true;
							setOperand(&stackOperand, A7, PreDecrement, LongTypeIndex);
							emitMove(operand1, &stackOperand, true);
							emitMove(operand2, &stackOperand, true);
							emitLibCall("LMUL", 4);
                            cleanStack(8);
							setOperand(&stackOperand, D0, Decl, node->typeIndex);
							emitMove(&stackOperand, operand1, true);
							return operand1;
						}
					case UnsignedIntTypeIndex :
					case IntTypeIndex :
					case UnsignedShortTypeIndex :
					case ShortTypeIndex :					
						if (!loadIntoDReg(operand2))
							return NULL;
						emitBinary(Token_Muls, Unsized, operand1, operand2, false);
						emitMove(operand2, operand1, true);
						freeRegister(operand2);
						return operand1;
					default :
						error("unsupported combination of operator and types");
						return NULL;
				}
			}
			break;
		case DivideEqual : {
				operand1 = generateCode(node->data.children.node1, -1, -1);
				if (operand1 == NULL) return NULL;
				operand2 = generateCode(node->data.children.node2, -1, -1);
				if (operand2 == NULL) return NULL;
				switch (node->typeIndex) {
					case UnsignedLongTypeIndex :
					case LongTypeIndex : {
							freeRegister(operand2);
							savedRegs = saveRegisters();		
							if (node->typeIndex == LongTypeIndex)
								longSignedDivideUsed = true;
							longUnsignedDivideUsed = true;
							setOperand(&stackOperand, A7, PreDecrement, LongTypeIndex);
							emitMove(operand2, &stackOperand, true);
							emitMove(operand1, &stackOperand, true);
							if (node->typeIndex == LongTypeIndex)
								emitLibCall("LDIV", 4);
							else
								emitLibCall("LUDIV", 5);
                            cleanStack(8);
							setOperand(&stackOperand, D0, Decl, node->typeIndex);
							emitMove(&stackOperand, operand1, true);
							return operand1;
						}
					case UnsignedIntTypeIndex :
					case UnsignedShortTypeIndex :
						setOperand(&stackOperand, -1, Decl, IntTypeIndex);
						emitMove(operand1, &stackOperand, true);
						stackAdjust.reg = Immediate;
						stackAdjust.typeIndex = LongTypeIndex;
						stackAdjust.value = 0xFFFF;
						emitBinary(Token_Andi, LongSpec, &stackAdjust, &stackOperand, false);
						emitBinary(Token_Divs, Unsized, operand2, &stackOperand, false);
						emitMove(&stackOperand, operand1, true);
						freeRegister(&stackOperand);
						freeRegister(operand2);
						return operand1;
					case IntTypeIndex :
					case ShortTypeIndex :
						setOperand(&stackOperand, -1, Decl, IntTypeIndex);
						emitMove(operand1, &stackOperand, true);
						emitUnary(Token_Ext, LongSpec, &stackOperand, false);
						emitBinary(Token_Divs, Unsized, operand2, &stackOperand, false);
						emitMove(&stackOperand, operand1, true);
						freeRegister(&stackOperand);
						freeRegister(operand2);
						return operand1;
					default :
						error("unsupported combination of operator and types");
						return NULL;
				}
			}

		case RemainderEqual : {
				operand1 = generateCode(node->data.children.node1, -1, -1);
				if (operand1 == NULL) return NULL;
				operand2 = generateCode(node->data.children.node2, -1, -1);
				if (operand2 == NULL) return NULL;
				switch (node->typeIndex) {
					case UnsignedLongTypeIndex :
					case LongTypeIndex : {
							freeRegister(operand2);
							savedRegs = saveRegisters();		
							if (node->typeIndex == LongTypeIndex)
								longSignedRemainderUsed = true;
							longUnsignedRemainderUsed = true;
							setOperand(&stackOperand, A7, PreDecrement, LongTypeIndex);
							emitMove(operand2, &stackOperand, true);
							emitMove(operand1, &stackOperand, true);
							if (node->typeIndex == LongTypeIndex)
								emitLibCall("LREM", 4);
							else
								emitLibCall("LUREM", 5);
                            cleanStack(8);
                            setOperand(&stackOperand, D0, Decl, node->typeIndex);
							emitMove(&stackOperand, operand1, true);
							return operand1;
						}
					case UnsignedIntTypeIndex :
					case UnsignedShortTypeIndex :
						setOperand(&stackOperand, -1, Decl, IntTypeIndex);
						emitMove(operand1, &stackOperand, true);
						stackAdjust.reg = Immediate;
						stackAdjust.typeIndex = LongTypeIndex;
						stackAdjust.value = 0xFFFF;
						emitBinary(Token_Andi, LongSpec, &stackAdjust, &stackOperand, false);
						emitBinary(Token_Divu, Unsized, operand2, &stackOperand, false);
						emitUnary(Token_Swap, Unsized, &stackOperand, false);
						emitMove(&stackOperand, operand1, true);
						freeRegister(&stackOperand);
						freeRegister(operand2);
						return operand1;
					case IntTypeIndex :
					case ShortTypeIndex :
						setOperand(&stackOperand, -1, Decl, IntTypeIndex);
						emitMove(operand1, &stackOperand, true);
						emitUnary(Token_Ext, LongSpec, &stackOperand, false);
						emitBinary(Token_Divu, Unsized, operand2, &stackOperand, false);
						emitUnary(Token_Swap, Unsized, &stackOperand, false);
						emitMove(&stackOperand, operand1, true);
						freeRegister(&stackOperand);
						freeRegister(operand2);
						return operand1;
					default :
						error("unsupported combination of operator and types");
						return NULL;
				}
			}
		case Decl :
		case Indirect :
		case DoubleQuote :
			return node;
		case Identifier :
			error("un-resolved identifier in tree");
			return NULL;
		case Convert : {
				operand1 = generateCode(node->data.children.node1, -1, -1);
				if (operand1 == NULL) return NULL;
				if (!emitConvert(operand1, node->typeIndex)) return NULL;
				return operand1;
			}
		case UnaryMinus : {
				operand1 = generateCode(node->data.children.node1, -1, -1);
				if (operand1 == NULL) return NULL;
                if (node->typeIndex == DoubleTypeIndex) {
                    if (!emitFPSequence1(50, operand1, NULL, node))
                        return NULL;
                    return node;
                }
                else {
	                if (node->typeIndex == FloatTypeIndex) { // can't happen ?
	                    if (!emitFPSequence2(45, operand1, NULL, node, FloatTypeIndex))
	                        return NULL;
	                    return node;
	                }
	                else {
					    if (!loadIntoDReg(operand1)) return NULL;
					    emitUnary(Token_Neg, Unsized, operand1, true);
					    return operand1;
	                }
	            }
			}
		case Tilde : {
				operand1 = generateCode(node->data.children.node1, -1, -1);
				if (operand1 == NULL) return NULL;
			    if (!loadIntoDReg(operand1)) return NULL;
			    emitUnary(Token_Not, Unsized, operand1, true);
			    return operand1;
			}
		case AddressOf : {
				operand1 = generateCode(node->data.children.node1, -1, -1);
				if (operand1 == NULL) return NULL;
				freeRegister(operand1);
				node->reg = getFreeRegister(A0, A4);
				if (node->reg == -1) return false;
				node->op = Decl;
				node->offset = 0;
				emitBinary(Token_Lea, Unsized, operand1, node, false);
				return node;
			} 
		case LeftBracket : {
				operand1 = generateCode(node->data.children.node1, -1, -1);
				if (operand1 == NULL) return NULL;
				if (!loadIntoAReg(operand1)) return NULL;
				node->reg = operand1->reg;
				operand2 = generateCode(node->data.children.node2, -1, -1);
				if (operand2 == NULL) return NULL;
				if (!loadIntoDReg(operand2)) return NULL;
				node->reg2 = operand2->reg;
				return node;
			}
		case Arrow : {
				operand1 = generateCode(node->data.children.node1, -1, -1);
				if (operand1 == NULL) return NULL;
				if (!loadIntoAReg(operand1)) return NULL;
				node->reg = operand1->reg;
				if ((node->data.children.node2 != NULL)
						&& ((((Declaration *)(node->data.children.node2))->bitWidth & 0x7) != 0)) {
					if (!doBitfieldExtraction(node)) return NULL;
				}
				return node;
			}
		case LeftParen : {
				int functionType;
				stackDepth = 0;
				savedRegs = saveRegisters();
				if ((node->data.children.node1->op != AddressOf)
						 || (node->data.children.node1->data.children.node1->op != Decl)) {
					operand1 = generateCode(node->data.children.node1, -1, -1);			
					functionType = getBaseType(operand1->typeIndex);
				}				
				else {
					operand1 = node->data.children.node1->data.children.node1;
					functionType = operand1->typeIndex;
				}
				if (operand1 == NULL) return NULL;
				operand2 = node->data.children.node2;
				while (operand2 != NULL) {
					arg = operand2->data.children.node2;
										// there's an addressOf but not a pointer type
										// when we're passing a struct - the address of
										// is required to start the struct copy.
					if ((arg->op == AddressOf) && isPointer(arg->typeIndex)) {
						arg = generateCode(arg->data.children.node1, -1, -1);
						if (arg == NULL) return NULL;
						emitUnary(Token_Pea, Unsized, arg, false);
						freeRegister(arg);
						stackDepth += 4;
					}
					else {
						if (isStruct(arg->typeIndex)) {
			                stackOperand.op = PreDecrement; 	// set stackOperand each time since it's global
			                stackOperand.reg = A7;
							if (getTypeSize(arg->typeIndex) == 2) {
								arg = generateCode(arg->data.children.node1, -1, -1);
								if (arg == NULL) return NULL;
								stackOperand.typeIndex = IntTypeIndex;
								arg->typeIndex = IntTypeIndex;
								emitMove(arg, &stackOperand, true);
								freeRegister(arg);
							}
							else {
								if (getTypeSize(arg->typeIndex) == 4) {
									arg = generateCode(arg->data.children.node1, -1, -1);
									if (arg == NULL) return NULL;
									stackOperand.typeIndex = LongTypeIndex;
									arg->typeIndex = LongTypeIndex;
									emitMove(arg, &stackOperand, true);
									freeRegister(arg);
								}
								else {
									arg = generateCode(arg, -1, -1);
									if (arg == NULL) return NULL;
									stackAdjust.reg = arg->reg;
									stackAdjust.offset = getTypeSize(arg->typeIndex);
									stackAdjust.op = Arrow;
									stackAdjust.typeIndex = LongTypeIndex;
									emitBinary(Token_Lea, Unsized, &stackAdjust, arg, false);
									stackOperand.typeIndex = IntTypeIndex;
									if (!loadConstant(node, (getTypeSize(arg->typeIndex) / 2) - 1)) return NULL;										
									interBranch = nextLabel++;
									emitLabel(interBranch);
									arg->op = PreDecrement;
									arg->typeIndex = IntTypeIndex;
									emitBinary(Token_Move, Unsized, arg, &stackOperand, false);
									emitDBranch(Token_Dbra, node, interBranch);
									freeRegister(arg);
									freeRegister(node);
								}
							}
							stackDepth += getTypeSize(arg->typeIndex);
						}
						else {
							arg = generateCode(arg, -1, -1);
							if (arg == NULL) return NULL;
	                    	stackDepth += pushValue(arg);
						}
					}
					operand2 = operand2->data.children.node1;					
				}
				if ((operand1->op == Decl) && !isPointer(operand1->typeIndex))
					if (isInlineFunction(operand1->data.decl))
						emitInlineCode(operand1->data.decl);
					else
						emitUnary(Token_Jsr, Unsized, operand1, false);
				else {
					if (!loadIntoAReg(operand1))
						return NULL;
					operand1->op = Indirect;
					operand1->offset = 0;
					emitUnary(Token_Jsr, Unsized, operand1, false);
				}
                                
                if (stackDepth > 0) {
                	cleanStack(stackDepth);
				}
				node->op = Decl;
				node->offset = 0;
				interBranch = getBaseType(functionType);	// return type
				if (interBranch != VoidTypeIndex) {
					if (isPointer(interBranch)) {
						setOperand(&stackOperand, A0, Decl, node->typeIndex /*interBranch*/);
						node->reg = getFreeFromSavedRegisters(A0, A4, savedRegs);
						if (node->reg == -1) return false;
						if (node->reg != A0)
							emitMove(&stackOperand, node, true);
					}
					else {
						if ((interBranch != DoubleTypeIndex) && (interBranch != FloatTypeIndex)) {
							setOperand(&stackOperand, D0, Decl, node->typeIndex /*interBranch*/);
							node->reg = getFreeFromSavedRegisters(D0, D7, savedRegs);
							if (node->reg == -1) return false;
							if (node->reg != D0)
								emitMove(&stackOperand, node, true);
						}
					}
					restoreRegisters(savedRegs);
					registerInUse[(int) node->reg] = true;
				}
				else {
					node->reg = None;
					restoreRegisters(savedRegs);
				}			
				return node;
			}
		case PreIncrement :
		case PreDecrement : {
				operand1 = generateCode(node->data.children.node1, -1, -1);
				if (operand1 == NULL) return NULL;
				if (operand1->typeIndex == FloatTypeIndex) {
					fpImmediate.op = Decl;
					fpImmediate.reg = Immediate;
					fpImmediate.data.fValue = 1.0;
					fpImmediate.value = fpImmediate.data.dBits.d1;
					fpImmediate.typeIndex = LongTypeIndex;
					if (node->op == PreDecrement)
						emitFPSequence2(48, operand1, &fpImmediate, node, LongTypeIndex);
					else
						emitFPSequence2(46, operand1, &fpImmediate, node, LongTypeIndex);
					emitMove(node, operand1, true);
					return operand1;
				}
				else {
					if (operand1->typeIndex == DoubleTypeIndex) {
						fpImmediate.op = Decl;
						fpImmediate.reg = Immediate;
						fpImmediate.data.dValue = 1.0;
						fpImmediate.typeIndex = DoubleTypeIndex;
						if (node->op == PreDecrement)
							emitFPSequence1(53, operand1, &fpImmediate, node);
						else
							emitFPSequence1(51, operand1, &fpImmediate, node);
						emitMoveDouble(node, operand1);
						return operand1;
					}
					else {
						interBranch = 1;		// Xcrement amount
						if (isPointer(node->typeIndex)) {
							interBranch = getTypeSize(getBaseType(node->typeIndex));
						}
						stackAdjust.reg = Immediate;
						stackAdjust.value = interBranch;
						stackAdjust.typeIndex = node->typeIndex;
						if (interBranch <= 8)
							if (node->op == PreDecrement)
								emitBinary(Token_Subq, Unsized, &stackAdjust, operand1, true);
							else
								emitBinary(Token_Addq, Unsized, &stackAdjust, operand1, true);
						else
							if (node->op == PreDecrement)
								emitBinary(Token_Subi, Unsized, &stackAdjust, operand1, true);
							else
								emitBinary(Token_Addi, Unsized, &stackAdjust, operand1, true);
						return operand1;
					}
				}
			}
		case PostIncrement :		
		case PostDecrement : {
				stackDepth = node->op;		// stash this now before it gets slapped
				operand1 = generateCode(node->data.children.node1, -1, -1);
				if (operand1 == NULL) return NULL;
				if (operand1->typeIndex == FloatTypeIndex) {
					arg = newTemp(FloatTypeIndex, gStatementIndex);
					emitMove(operand1, arg, true);
					fpImmediate.op = Decl;
					fpImmediate.reg = Immediate;
					fpImmediate.data.fValue = 1.0;
					fpImmediate.value = fpImmediate.data.dBits.d1;
					fpImmediate.typeIndex = LongTypeIndex;
					if (node->op == PostDecrement)
						emitFPSequence2(48, operand1, &fpImmediate, node, LongTypeIndex);
					else
						emitFPSequence2(46, operand1, &fpImmediate, node, LongTypeIndex);
					emitMove(node, operand1, true);
					return arg;
				}
				else {
					if (operand1->typeIndex == DoubleTypeIndex) {
						arg = newTemp(DoubleTypeIndex, gStatementIndex);
						emitMoveDouble(operand1, arg);
						fpImmediate.op = Decl;
						fpImmediate.reg = Immediate;
						fpImmediate.data.dValue = 1.0;
						fpImmediate.typeIndex = DoubleTypeIndex;
						if (node->op == PostDecrement)
							emitFPSequence1(53, operand1, &fpImmediate, node);
						else
							emitFPSequence1(51, operand1, &fpImmediate, node);
						emitMoveDouble(node, operand1);
						return arg;
					}
					else {
						interBranch = 1;		// Xcrement amount
						if (isPointer(node->typeIndex)) {
							node->reg = getFreeRegister(A0, A4);			
							interBranch = getTypeSize(getBaseType(node->typeIndex));
						}
						else
							node->reg = getFreeRegister(D0, D7);
						if (node->reg == -1) return NULL;
						node->op = Decl;
						node->offset = 0;
						emitMove(operand1, node, true);
						stackAdjust.reg = Immediate;
						stackAdjust.value = interBranch;
						stackAdjust.typeIndex = node->typeIndex;
						if (interBranch <= 8)
							if (stackDepth == PostDecrement)
								emitBinary(Token_Subq, Unsized, &stackAdjust, operand1, true);
							else
								emitBinary(Token_Addq, Unsized, &stackAdjust, operand1, true);
						else
							if (stackDepth == PostDecrement)
								emitBinary(Token_Subi, Unsized, &stackAdjust, operand1, true);
							else
								emitBinary(Token_Addi, Unsized, &stackAdjust, operand1, true);
						freeRegister(operand1);
						return node;
					}
				}
			}
		case Equal : {
				operand2 = generateCode(node->data.children.node2, -1, -1);
				if (operand2 == NULL) return NULL;
				if ((node->data.children.node1->op == Arrow)
						&& (node->data.children.node1->data.children.node2 != NULL)
						&& ((((Declaration *)(node->data.children.node1->data.children.node2))->bitWidth & 0x7) != 0)) {
					if (!doBitFieldInsertion(node, operand2)) return false;
					// !!! nested assignments ???

// boz_x@users.sourceforge.net (13/06/03): 
// Just need to return something non-NULL at this point, then bail out.
// Previously bug here, caused because the function returns operand1 at the end
// and in the case of bitfields, this is unused, and initialised to NULL
// This was causing the compiler to stop prematurely when compiling bitfields
//
// As far as I can tell, the return value is checked for !=NULL, then discarded
// so it doesn't matter what we return
				  return operand2;
			      
				}
				else {
					operand1 = generateCode(node->data.children.node1, -1, -1);
					if (operand1 == NULL) return NULL;
					// array used to initialize local char[]'s
					if (isArray(node->typeIndex) || isStruct(node->typeIndex)) {
						if (!loadIntoAReg(operand1)) return NULL;
						operand1->op = PostIncrement;
						if (!loadIntoAReg(operand2)) return NULL;
						operand2->op = PostIncrement;
						if (node->offset == 4) {
							emitBinary(Token_Move, LongSpec, operand2, operand1, false);
						}
						else {
							if (node->offset == 8) {
								emitBinary(Token_Move, LongSpec, operand2, operand1, false);
								emitBinary(Token_Move, LongSpec, operand2, operand1, false);
							}
							else {
//								node->reg = getFreeRegister(D0, D7);
//								node->op = Decl;
//								node->offset = 0;
//								stackAdjust.reg = Immediate;
//								stackAdjust.typeIndex = LongTypeIndex;
//								stackAdjust.value = node->offset - 1;	// dbra goes to -1
//								emitBinary(Token_Moveq, Unsized, &stackAdjust, node, false);
								if (!loadConstant(node, node->offset - 1)) return NULL;										
								interBranch = nextLabel++;
								emitLabel(interBranch);
								emitBinary(Token_Move, ByteSpec, operand2, operand1, false);
								emitDBranch(Token_Dbra, node, interBranch);
							}
						}
						freeRegister(operand2);
						freeRegister(node);
						// operand1 is bogus here !!!, but does anybody care ?
					}
					else {					
						if (operand2->typeIndex == DoubleTypeIndex) {
							emitMoveDouble(operand2, operand1);
							freeRegister(operand2);
						}
						else {					
							emitMove(operand2, operand1, true);
							freeRegister(operand2);
						}
					}
				}

				return operand1;
			}
		case Comma : {
				operand1 = generateCode(node->data.children.node1, -1, -1);
				if (operand1 == NULL) return NULL;
				freeRegister(operand1);
				operand2 = generateCode(node->data.children.node2, -1, -1);
				if (operand2 == NULL) return NULL;
				return operand2;
			}
		case AndEqual :
		case OrEqual :
		case XorEqual :
		case MinusEqual :
		case PlusEqual : {
				operand2 = generateCode(node->data.children.node2, -1, -1);
				if (operand2 == NULL) return NULL;
				operand1 = generateCode(node->data.children.node1, -1, -1);
				if (operand1 == NULL) return NULL;
				switch (node->typeIndex) {
					case FloatTypeIndex :
					case DoubleTypeIndex :
						error("Floating-point not supported 8");
						return NULL;
					case UnsignedLongTypeIndex :
					case LongTypeIndex :
					case UnsignedIntTypeIndex :
					case IntTypeIndex :
					case UnsignedShortTypeIndex :
					case ShortTypeIndex :
					case UnsignedCharTypeIndex :
					case CharTypeIndex :
						if (!loadIntoDReg(operand2))
							return NULL;
						switch (node->op) {
							case PlusEqual :
								emitBinary(Token_Add, Unsized, operand2, operand1, true);
								break;
							case MinusEqual :
								emitBinary(Token_Sub, Unsized, operand2, operand1, true);
								break;
							case AndEqual :
								emitBinary(Token_And, Unsized, operand2, operand1, true);
								break;
							case OrEqual :
								emitBinary(Token_Or, Unsized, operand2, operand1, true);
								break;
							case XorEqual :
								emitBinary(Token_Eor, Unsized, operand2, operand1, true);
								break;
							default :
								error("bad case 9");
								break;
						}
						freeRegister(operand2);
						return operand1;
					default :
						error("unsupported combination of operator and types");
						return NULL;
				}
				return operand1;
			}
		case LeftShiftEqual :
		case RightShiftEqual :
				operand1 = generateCode(node->data.children.node1, -1, -1);
				if (operand1 == NULL) return NULL;
				operand2 = generateCode(node->data.children.node2, -1, -1);
				if (operand2 == NULL) return NULL;
				setOperand(&stackOperand, -1, Decl, node->typeIndex);
				if (stackOperand.reg == -1) return NULL;
				emitMove(operand1, &stackOperand, true);
				if (!loadIntoDReg(operand2))
					return NULL;
				if (node->op == LeftShiftEqual)
					emitBinary(Token_Lsl, Unsized, operand2, &stackOperand, true);
				else
					if ((node->typeIndex == UnsignedLongTypeIndex)
							|| (node->typeIndex == UnsignedIntTypeIndex)
							|| (node->typeIndex == UnsignedShortTypeIndex))
						emitBinary(Token_Lsr, Unsized, operand2, &stackOperand, true);
					else
						emitBinary(Token_Asr, Unsized, operand2, &stackOperand, true);
				freeRegister(operand2);
				emitMove(&stackOperand, operand1, true);
				freeRegister(&stackOperand);
				return operand1;
		case LeftShift :
		case RightShift :
				operand1 = generateCode(node->data.children.node1, -1, -1);
				if (operand1 == NULL) return NULL;
				operand2 = generateCode(node->data.children.node2, -1, -1);
				if (operand2 == NULL) return NULL;
				if (!loadIntoDReg(operand1))
					return NULL;
				if (!loadIntoDReg(operand2))
					return NULL;
				operand2->typeIndex = operand1->typeIndex;	// hack to get the right type for the shift
				if (node->op == LeftShift)
					emitBinary(Token_Lsl, Unsized, operand2, operand1, true);
				else
					if ((node->typeIndex == UnsignedLongTypeIndex)
							|| (node->typeIndex == UnsignedIntTypeIndex)
							|| (node->typeIndex == UnsignedShortTypeIndex))
						emitBinary(Token_Lsr, Unsized, operand2, operand1, true);
					else
						emitBinary(Token_Asr, Unsized, operand2, operand1, true);
				freeRegister(operand2);
				return operand1;
		case And :
		case Or :
		case Xor :
		case Plus :
		case Minus : {
				operand1 = generateCode(node->data.children.node1, -1, -1);
				if (operand1 == NULL) return NULL;
				operand2 = generateCode(node->data.children.node2, -1, -1);
				if (operand2 == NULL) return NULL;
				if (isPointer(node->typeIndex)) {
					if (!loadIntoAReg(operand1))
						return NULL;
					if (node->op == Plus)
						emitBinary(Token_Adda, Unsized, operand2, operand1, true);
					else
						emitBinary(Token_Suba, Unsized, operand2, operand1, true);
					freeRegister(operand2);
					return operand1;
				}
				else {
					switch (node->typeIndex) {
//						case FloatTypeIndex :
						case DoubleTypeIndex : {
                                if (node->op == Plus) {
                                    if (!emitFPSequence1(51, operand1, operand2, node))
                                        return NULL;
                                }
                                else {
                                    if (!emitFPSequence1(53, operand1, operand2, node))
                                        return NULL;
                                }
							    return node;
							}
						case UnsignedLongTypeIndex :
						case LongTypeIndex :
						case UnsignedIntTypeIndex :
						case IntTypeIndex :
						case UnsignedShortTypeIndex :
						case ShortTypeIndex :
							if ((operand1->reg == Immediate) 
									&& ((node->op == Plus) || (node->op == And) || (node->op == Or)) ) {
								if (!loadIntoDReg(operand2))
									return NULL;
								switch (node->op) {
									case Plus :
										emitBinary(Token_Add, Unsized, operand1, operand2, true);
										break;
									case And :
										emitBinary(Token_And, Unsized, operand1, operand2, true);
										break;
									case Or :
										emitBinary(Token_Or, Unsized, operand1, operand2, true);
										break;
									default :
										error("bad case 10a");
										break;
								}
								return operand2;
							}
							else {							
								if (!loadIntoDReg(operand1))
									return NULL;
								switch (node->op) {
									case Plus :
										emitBinary(Token_Add, Unsized, operand2, operand1, true);
										break;
									case Minus :
										emitBinary(Token_Sub, Unsized, operand2, operand1, true);
										break;
									case And :
										emitBinary(Token_And, Unsized, operand2, operand1, true);
										break;
									case Or :
										emitBinary(Token_Or, Unsized, operand2, operand1, true);
										break;
									case Xor : 
										setOperand(&stackOperand, A7, PreDecrement, node->typeIndex);
										emitMove(operand2, &stackOperand, true);
										stackOperand.op = Arrow;
										stackOperand.offset = 0;
										emitBinary(Token_Eor, Unsized, operand1, &stackOperand, true);
										stackOperand.op = PostIncrement;
										emitMove(&stackOperand, operand1, true);
										break;
									default :
										error("bad case 10b");
										break;
								}
								freeRegister(operand2);
								return operand1;
							}
						default :
							error("unsupported combination of operator and types");
							return NULL;
					}
				}
				return NULL;
			}
		case AndAnd : {
				if (trueBranch != -1) {
					interBranch = nextLabel++;
					operand1 = generateCode(node->data.children.node1, interBranch | 0x4000, falseBranch);
					if (operand1 == NULL) return NULL;
			    	if (!emitConditionalBranch(operand1, interBranch | 0x4000, falseBranch))
			    		return NULL;
					emitLabel(interBranch);
					operand2 = generateCode(node->data.children.node2, trueBranch, falseBranch);
					if (operand2 == NULL) return NULL;
			    	if (!emitConditionalBranch(operand2, trueBranch, falseBranch))
			    		return NULL;
					operand1->op = Goto;
					return operand1;		
				}
				else {
					trueBranch = nextLabel++;
					interBranch = nextLabel++;
					falseBranch = nextLabel++;
					operand1 = generateCode(node->data.children.node1, interBranch | 0x4000, falseBranch);
					if (operand1 == NULL) return NULL;
			    	if (!emitConditionalBranch(operand1, interBranch | 0x4000, falseBranch))
			    		return NULL;
					emitLabel(interBranch);
					operand2 = generateCode(node->data.children.node2, trueBranch, falseBranch | 0x4000);
					if (operand2 == NULL) return NULL;
			    	if (!emitConditionalBranch(operand2, trueBranch, falseBranch | 0x4000))
			    		return NULL;
					emitLabel(falseBranch);
					operand1->reg = getFreeRegister(D0, D7);
					if (operand1->reg == -1) return false;
					operand1->typeIndex = IntTypeIndex;
					stackAdjust.reg = Immediate;
					stackAdjust.typeIndex = IntTypeIndex;
					stackAdjust.value = 0;
					emitBinary(Token_Moveq, Unsized, &stackAdjust, operand1, false);
					interBranch = nextLabel++;
					emitBranch(Token_Bra, interBranch);
					emitLabel(trueBranch);
					stackAdjust.value = 1;
					emitBinary(Token_Moveq, Unsized, &stackAdjust, operand1, false);
					emitLabel(interBranch);
					return operand1;		
				}
			}			
		case OrOr : {
				if (trueBranch != -1) {
					interBranch = nextLabel++;
					operand1 = generateCode(node->data.children.node1, trueBranch, interBranch | 0x4000);
					if (operand1 == NULL) return NULL;
			    	if (!emitConditionalBranch(operand1, trueBranch, interBranch | 0x4000))
			    		return NULL;
					emitLabel(interBranch);
					operand2 = generateCode(node->data.children.node2, trueBranch, falseBranch);
					if (operand2 == NULL) return NULL;
			    	if (!emitConditionalBranch(operand2, trueBranch, falseBranch))
			    		return NULL;
					operand1->op = Goto;
					return operand1;		
				}
				else {
					trueBranch = nextLabel++;
					interBranch = nextLabel++;
					falseBranch = nextLabel++;
					operand1 = generateCode(node->data.children.node1, trueBranch, interBranch | 0x4000);
					if (operand1 == NULL) return NULL;
			    	if (!emitConditionalBranch(operand1, trueBranch, interBranch | 0x4000))
			    		return NULL;
					emitLabel(interBranch);
					operand2 = generateCode(node->data.children.node2, trueBranch, falseBranch | 0x4000);
					if (operand2 == NULL) return NULL;
			    	if (!emitConditionalBranch(operand2, trueBranch, falseBranch | 0x4000))
			    		return NULL;
					emitLabel(falseBranch);
					operand1->reg = getFreeRegister(D0, D7);
					if (operand1->reg == -1) return false;
					operand1->typeIndex = IntTypeIndex;
					stackAdjust.reg = Immediate;
					stackAdjust.typeIndex = IntTypeIndex;
					stackAdjust.value = 0;
					emitBinary(Token_Moveq, Unsized, &stackAdjust, operand1, false);
					interBranch = nextLabel++;
					emitBranch(Token_Bra, interBranch);
					emitLabel(trueBranch);
					stackAdjust.value = 1;
					emitBinary(Token_Moveq, Unsized, &stackAdjust, operand1, false);
					emitLabel(interBranch);
					return operand1;		
				}
			}			
		case Not : {
				operand1 = node->data.children.node1;
				if ((trueBranch != -1)
						&& ((operand1->op == EqualEqual) 
							|| (operand1->op == NotEqual)
							|| (operand1->op == LessEqual)
							|| (operand1->op == GreaterEqual)
							|| (operand1->op == Less)
							|| (operand1->op == Greater)
							|| (operand1->op== AndAnd)
							|| (operand1->op == OrOr)))
					return generateCode(operand1, falseBranch, trueBranch);
				else {
					operand1 = generateCode(operand1, -1, -1);
					if (operand1->typeIndex == DoubleTypeIndex) {
						error("oops, forgot FP not");
					}
					else {
						if (!loadIntoDReg(operand1)) return NULL;
						emitUnary(Token_Tst, Unsized, operand1, true);
						if (trueBranch != -1) {
							freeRegister(operand1);
							emitBranch(Token_Bne, falseBranch);
							emitBranch(Token_Bra, trueBranch);
							operand1->op = Goto;
						}
						else {	// setne etc.
							emitUnary(Token_Seq, Unsized, operand1, false);
							operand1->typeIndex = IntTypeIndex;
							stackAdjust.reg = Immediate;
							stackAdjust.typeIndex = IntTypeIndex;
							stackAdjust.value = 1;
							emitBinary(Token_Andi, Unsized, &stackAdjust, operand1, true);
						}
						return operand1;
					}
				}
			}
		case Question : {
				trueBranch = nextLabel++;
				falseBranch = nextLabel++;
				operand1 = generateCode(node->data.children.node1, trueBranch, falseBranch);
				if (operand1 == NULL) return NULL;
		    	if (!emitConditionalBranch(operand1, trueBranch, falseBranch))
		    		return NULL;
				emitLabel(trueBranch);
				operand1 = generateCode(node->data.children.node2->data.children.node1, -1, -1);
				if(!loadIntoDReg(operand1)) return NULL;
				trueBranch = nextLabel++;
				emitBranch(Token_Bra, trueBranch);
				emitLabel(falseBranch);
				operand2 = generateCode(node->data.children.node2->data.children.node2, -1, -1);
                if ( operand2->typeIndex == DoubleTypeIndex ) {
                    emitMoveDouble(operand2, operand1);
                } else {
                    emitMove(operand2, operand1, true);
                }
				freeRegister(operand2);
				emitLabel(trueBranch);
				return operand1;
			}
		case Less :
		case LessEqual :
		case Greater :
		case GreaterEqual :
		case NotEqual :
		case EqualEqual : {
				operand1 = generateCode(node->data.children.node1, -1, -1);
				if (operand1 == NULL) return NULL;
				operand2 = generateCode(node->data.children.node2, -1, -1);
				if (operand2 == NULL) return NULL;
				interBranch = 0;				// use to signal signed/unsigned
				switch (operand1->typeIndex) {
//					case FloatTypeIndex :
					case DoubleTypeIndex :
                        switch (node->op) {
                            case Less :
                                if (!emitFPSequence2(39, operand1, operand2, node, CharTypeIndex))
                                    return NULL;
                                break;
                            case LessEqual :
                                if (!emitFPSequence2(40, operand1, operand2, node, CharTypeIndex))
                                    return NULL;
                                break;
                            case Greater :
                                if (!emitFPSequence2(41, operand1, operand2, node, CharTypeIndex))
                                    return NULL;
                                break;
                            case GreaterEqual :
                                if (!emitFPSequence2(42, operand1, operand2, node, CharTypeIndex))
                                    return NULL;
                                break;
                            case NotEqual :
                                if (!emitFPSequence2(38, operand1, operand2, node, CharTypeIndex))
                                    return NULL;
                                break;
                            case EqualEqual :
                                if (!emitFPSequence2(37, operand1, operand2, node, CharTypeIndex))
                                    return NULL;
                                break;
                        }
                        if (trueBranch != -1) {
                            emitUnary(Token_Tst, Unsized, node, true);
                            freeRegister(node);
						    emitBranch(Token_Beq, falseBranch);
						    emitBranch(Token_Bra, trueBranch);
						    operand1->op = Goto;
						    return operand1;
                        }
                        else {
							stackAdjust.reg = Immediate;
							stackAdjust.typeIndex = IntTypeIndex;
							stackAdjust.value = 1;
							emitBinary(Token_Andi, Unsized, &stackAdjust, node, true);
							node->typeIndex = IntTypeIndex;
                            return node;
						}
					case UnsignedLongTypeIndex :
					case UnsignedIntTypeIndex :
					case UnsignedShortTypeIndex :
							interBranch = 1;		
							// fall thru...
					case LongTypeIndex :
					case IntTypeIndex :
					case ShortTypeIndex :
						if (operand2->reg == Immediate) {
							if (operand2->value == 0) {
								if ((operand1->op == Decl)
										&& (operand1->reg >= A0) 
										&& (operand1->reg <= A7)) {
									if (!loadIntoDReg(operand1))
										return NULL;
								}
								else
									emitUnary(Token_Tst, Unsized, operand1, true);
							}
							else
								emitBinary(Token_Cmpi, Unsized, operand2, operand1, true);
						}
						else {
							if (!loadIntoDReg(operand1))
								return NULL;
							emitBinary(Token_Cmp, Unsized, operand2, operand1, true);
							freeRegister(operand2);						
						}
						if (trueBranch != -1) {
							freeRegister(operand1);
							switch (node->op) {
								case Greater :
									if (trueBranch & 0x4000)									
										emitBranch((interBranch) ? Token_Bls : Token_Ble, falseBranch);
									else {
										emitBranch((interBranch) ? Token_Bhi : Token_Bgt, trueBranch);
										emitBranch(Token_Bra, falseBranch);
									}
									break;
								case Less :
									if (trueBranch & 0x4000)									
										emitBranch((interBranch) ? Token_Bcc : Token_Bge, falseBranch);
									else {
										emitBranch((interBranch) ? Token_Bcs : Token_Blt, trueBranch);
										emitBranch(Token_Bra, falseBranch);
									}
									break;
								case GreaterEqual :
									if (trueBranch & 0x4000)									
										emitBranch((interBranch) ? Token_Bcs : Token_Blt, falseBranch);
									else {
										emitBranch((interBranch) ? Token_Bcc : Token_Bge, trueBranch);
										emitBranch(Token_Bra, falseBranch);
									}
									break;
								case LessEqual :
									if (trueBranch & 0x4000)									
										emitBranch((interBranch) ? Token_Bhi : Token_Bgt, falseBranch);
									else {
										emitBranch((interBranch) ? Token_Bls : Token_Ble, trueBranch);
										emitBranch(Token_Bra, falseBranch);
									}
									break;
								case EqualEqual :
									if (falseBranch & 0x4000)									
										emitBranch(Token_Beq, trueBranch);
									else {
										emitBranch(Token_Bne, falseBranch);
										emitBranch(Token_Bra, trueBranch);
									}
									break;
								case NotEqual :
									if (falseBranch & 0x4000)
										emitBranch(Token_Bne, trueBranch);
									else {
										emitBranch(Token_Beq, falseBranch);
										emitBranch(Token_Bra, trueBranch);
									}
									break;
								default :
									error("bad case 11");
									return NULL;
									
							}
							operand1->op = Goto;
							return operand1;
						}
						else {
							freeRegister(operand1);
							operand1->reg = getFreeRegister(D0, D7);
							if (operand1->reg == -1) return false;
							operand1->op = Decl;
							operand1->offset = 0;
							switch (node->op) {
								case Greater :
									emitUnary((interBranch) ? Token_Shi : Token_Sgt, Unsized, operand1, false);
									break;
								case Less :
									emitUnary((interBranch) ? Token_Scs : Token_Slt, Unsized, operand1, false);
									break;
								case GreaterEqual :
									emitUnary((interBranch) ? Token_Scc : Token_Sge, Unsized, operand1, false);
									break;
								case LessEqual :
									emitUnary((interBranch) ? Token_Sls : Token_Sle, Unsized, operand1, false);
									break;
								case EqualEqual :
									emitUnary(Token_Seq, Unsized, operand1, false);
									break;
								case NotEqual :
									emitUnary(Token_Sne, Unsized, operand1, false);
									break;
								default :
									error("bad case 12");
									return NULL;
							}
							operand1->typeIndex = IntTypeIndex;
							stackAdjust.reg = Immediate;
							stackAdjust.typeIndex = IntTypeIndex;
							stackAdjust.value = 1;
							emitBinary(Token_Andi, Unsized, &stackAdjust, operand1, true);
						}
						return operand1;
					default :
						error("unsupported combination of operator and types");
						return NULL;
				}
				return NULL;
			}
		case Multiply : {
				operand1 = generateCode(node->data.children.node1, -1, -1);
				if (operand1 == NULL) return NULL;
				operand2 = generateCode(node->data.children.node2, -1, -1);
				if (operand2 == NULL) return NULL;
				switch (node->typeIndex) {
//					case FloatTypeIndex :
					case DoubleTypeIndex :
                            if (!emitFPSequence1(52, operand1, operand2, node))
                                return NULL;
                            return node;
					case UnsignedLongTypeIndex :
					case LongTypeIndex : {
							freeRegister(operand1);			// safe to do these now provided we
							freeRegister(operand2);			// we don't do anything to re-use
															// these registers
							savedRegs = saveRegisters();		
							longMultiplyUsed = true;
							setOperand(&stackOperand, A7, PreDecrement, LongTypeIndex);
							emitMove(operand1, &stackOperand, true);
							emitMove(operand2, &stackOperand, true);
							emitLibCall("LMUL", 4);
							cleanStack(8);
							setOperand(&stackOperand, D0, Decl, node->typeIndex);
							node->op = Decl;
							node->offset = 0;
							node->reg = getFreeFromSavedRegisters(D0, D7, savedRegs);
							if (node->reg == -1) return false;
							if (node->reg != D0)
								emitMove(&stackOperand, node, true);
							restoreRegisters(savedRegs);
							registerInUse[(int) node->reg] = true;
							return node;
						}
					case UnsignedIntTypeIndex :
					case IntTypeIndex :
					case UnsignedShortTypeIndex :
					case ShortTypeIndex :
						if (!loadIntoDReg(operand1))
							return NULL;
						if ((operand2->reg == Immediate) && (operand2->value == 2))
							emitBinary(Token_Add, Unsized, operand1, operand1, true);
						else
							if ((operand2->reg == Immediate) && (operand2->value == 4)) {
								operand2->value = 2;
								emitBinary(Token_Lsl, Unsized, operand2, operand1, true);
							}
							else {
								emitBinary(Token_Muls, Unsized, operand2, operand1, false);
								freeRegister(operand2);
							}
						return operand1;
					default :
						error("unsupported combination of operator and types");
						return NULL;
				}
			}
		case Divide : {
				operand1 = generateCode(node->data.children.node1, -1, -1);
				if (operand1 == NULL) return NULL;
				operand2 = generateCode(node->data.children.node2, -1, -1);
				if (operand2 == NULL) return NULL;
				switch (node->typeIndex) {
//					case FloatTypeIndex :
					case DoubleTypeIndex :
                            if (!emitFPSequence1(54, operand1, operand2, node))
                                return NULL;
                            return node;
					case UnsignedLongTypeIndex :
					case LongTypeIndex : {
							freeRegister(operand2);
							freeRegister(operand1);
							savedRegs = saveRegisters();		
							if (node->typeIndex == LongTypeIndex) 
								longSignedDivideUsed = true;
							longUnsignedDivideUsed = true;
							setOperand(&stackOperand, A7, PreDecrement, LongTypeIndex);
							emitMove(operand2, &stackOperand, true);
							emitMove(operand1, &stackOperand, true);
							if (node->typeIndex == LongTypeIndex)
								emitLibCall("LDIV", 4);
							else
								emitLibCall("LUDIV", 5);
							cleanStack(8);
							setOperand(&stackOperand, D0, Decl, node->typeIndex);
							node->op = Decl;
							node->offset = 0;
							node->reg = getFreeFromSavedRegisters(D0, D7, savedRegs);
							if (node->reg == -1) return false;
							if (node->reg != D0)
								emitMove(&stackOperand, node, true);
							restoreRegisters(savedRegs);
							registerInUse[(int) node->reg] = true;
							return node;
						}
					case UnsignedIntTypeIndex :
					case UnsignedShortTypeIndex :
						if (!loadIntoDReg(operand1))
							return NULL;
						stackAdjust.reg = Immediate;
						stackAdjust.typeIndex = LongTypeIndex;
						stackAdjust.value = 0xFFFF;
						emitBinary(Token_Andi, LongSpec, &stackAdjust, operand1, false);
						emitBinary(Token_Divu, Unsized, operand2, operand1, false);
						freeRegister(operand2);
						return operand1;
					case IntTypeIndex :
					case ShortTypeIndex :
						if (!loadIntoDReg(operand1))
							return NULL;
						emitUnary(Token_Ext, LongSpec, operand1, false);
						emitBinary(Token_Divs, Unsized, operand2, operand1, false);
						freeRegister(operand2);
						return operand1;
					default :
						error("unsupported combination of operator and types");
						return NULL;
				}
			}
		case Remainder : {
				operand1 = generateCode(node->data.children.node1, -1, -1);
				if (operand1 == NULL) return NULL;
				operand2 = generateCode(node->data.children.node2, -1, -1);
				if (operand2 == NULL) return NULL;
				switch (node->typeIndex) {
					case UnsignedLongTypeIndex :
					case LongTypeIndex : {
							freeRegister(operand2);
							freeRegister(operand1);
							savedRegs = saveRegisters();		
							if (node->typeIndex == LongTypeIndex) 
								longSignedRemainderUsed = true;
							longUnsignedRemainderUsed = true;
							setOperand(&stackOperand, A7, PreDecrement, LongTypeIndex);
							emitMove(operand2, &stackOperand, true);
							emitMove(operand1, &stackOperand, true);
							if (node->typeIndex == LongTypeIndex)
								emitLibCall("LREM", 4);
							else
								emitLibCall("LUREM", 5);
							cleanStack(8);
							setOperand(&stackOperand, D0, Decl, node->typeIndex);
							node->op = Decl;
							node->offset = 0;
							node->reg = getFreeFromSavedRegisters(D0, D7, savedRegs);
							if (node->reg == -1) return false;
							if (node->reg != D0)
								emitMove(&stackOperand, node, true);
							restoreRegisters(savedRegs);
							registerInUse[(int) node->reg] = true;
							return node;
						}
					case UnsignedIntTypeIndex :
					case UnsignedShortTypeIndex :
						if (!loadIntoDReg(operand1))
							return NULL;
						stackAdjust.reg = Immediate;
						stackAdjust.typeIndex = LongTypeIndex;
						stackAdjust.value = 0xFFFF;
						emitBinary(Token_Andi, LongSpec, &stackAdjust, operand1, false);
						emitBinary(Token_Divu, Unsized, operand2, operand1, false);
						emitUnary(Token_Swap, Unsized, operand1, false);
						freeRegister(operand2);
						return operand1;
					case IntTypeIndex :
					case ShortTypeIndex :
						if (!loadIntoDReg(operand1))
							return NULL;
						emitUnary(Token_Ext, LongSpec, operand1, false);
						emitBinary(Token_Divs, Unsized, operand2, operand1, false);
						emitUnary(Token_Swap, Unsized, operand1, false);
						freeRegister(operand2);
						return operand1;
					default :
						error("unsupported combination of operator and types");
						return NULL;
				}
			}
	}
	return NULL;
}

Boolean genCodeForExpression(Node *node, int trueBranch, int falseBranch, Node **result)
{
//	rawData = false;
#ifdef DO_CODEGEN
	int i;
	Node *n;
	for (i = D0; i <= A4; i++)
		registerInUse[i] = false;
	for (i = D4; i <= D7; i++)
		registerInUse[i] = gRegisterAllocated[i];
	for (i = A3; i <= A4; i++)
		registerInUse[i] = gRegisterAllocated[i];

	n = generateCode(node, trueBranch, falseBranch);	

	if (result) *result = n;
	return (n != NULL);
#else
	return true;
#endif
}

