/***********************************************************************
 *
 * This file is part of the OnBoard C package.  For more information on 
 * using OnBoard C, see http://groups.yahoo.com/group/onboardc.  For 
 * more information on developing OnBoard C (including submission of 
 * bug reports), see http://sourceforge.net/projects/onboardc.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 ***********************************************************************/

#include <PalmOS.h>
#include <PalmCompatibility.h>

#include "OnBoardC.h"
#include "../../OnBoard/Src/debugTypes.h"


/* Multi-Segment Defines */
#include "compile-seg.h"

/* End of Multi-segment defines */

Handle argumentListData = NULL;
Declaration *argumentList;
int argumentListTop;
int argumentListMax;

Handle fieldListData = NULL;
Declaration *fieldList;
int fieldListTop;
int fieldListMax;

int inlineList[MaxInlines];
int inlineListTop;

#define MaxStack 32
typedef struct {
	int op;
	int precedence;
	CharPtr position;
} OperatorStackEntry;

OperatorStackEntry operatorStack[MaxStack];
int topOfOperatorStack;
Node *operandStack[MaxStack];
int topOfOperandStack;

CharPtr outputBuffer;
ULong outputTop;
//ULong totalOutputSize;
Handle bodyContents = NULL;
CharPtr bodyBuffer;
ULong bodyTop;
ULong bodySize;
/*
Handle dataContents;
CharPtr dataBuffer;
ULong dataTop;
UInt dataContentIndex;
*/
ULong gDataTop;
Handle gDataContentHStack[8];
CharPtr gDataContentStack[8];
UInt gDataContentIndexStack[8];
UInt gDataContentSizeStack[8];
int gDataContentStackTop;

Handle stringPoolContents;
CharPtr stringPoolBuffer;
ULong stringPoolTop;
ULong stringPoolSize;
int stringPoolIndex;
UInt stringPoolRecordIndex;

Handle codeStringPoolContents;
CharPtr codeStringPoolBuffer;
ULong codeStringPoolTop;
ULong codeStringPoolSize;
int codeStringPoolIndex;
UInt codeStringPoolRecordIndex;


Boolean longMultiplyUsed = false;
Boolean longUnsignedDivideUsed = false;
Boolean longSignedDivideUsed = false;
Boolean longUnsignedRemainderUsed = false;
Boolean longSignedRemainderUsed = false;

Handle gSysTrapHandle;
Declaration *gSysTrapList;
int gSysTrapSize;
int gSysTrapTop;

Node tree[MaxNodes];
int topNode;

int gFunctionCount;
int gFrameSize;

int gStatementIndex;	// used to distinguish among re-usable temps

int gStashCount = 0;

Boolean gRegisterAllocated[13];


Declaration *findGlobalDeclBase(Declaration *theDecl)
{
	int i;
	if ((theDecl >= globalTable) && (theDecl <= &globalTable[globalTop]))
		return globalTable;
	for (i = 0; i < fullGlobalCount; i++) {
		if ((theDecl >= fullGlobalTable[i]) && (theDecl < &fullGlobalTable[i][MaxGlobal]))
			return fullGlobalTable[i];
	}
	return NULL;
}

static UInt stashTree(Node *start)
{
	UInt at = -1;
	Handle h = DmNewRecord(gExprDB, &at, (topNode * sizeof(Node)) + sizeof(Node *));
	char *p = MemHandleLock(h);
	DmWrite(p, 0, tree, topNode * sizeof(Node));
	DmWrite(p, topNode * sizeof(Node), &start, sizeof(Node *));
	DmReleaseRecord(gExprDB, at, true);
	MemHandleUnlock(h);
	return at;
}

static Node *recoverTree(UInt i)
{
	Node *start;
	Handle h = DmQueryRecord(gExprDB, i);	
	char *p = MemHandleLock(h);
	ULong treeSize = MemHandleSize(h) - sizeof(Node *);
	MemMove(tree, p, treeSize);
	topNode = treeSize / sizeof(Node);
	start = *((Node **)(p + treeSize));
	MemHandleUnlock(h);
//	DmReleaseRecord(gExprDB, i, true);
	return start;
}

Boolean hasSideEffects(Node *node)
{
	switch (node->op) {
		default :
			error("bad case 3");
			return true;

		case Decl :
		case DoubleQuote :
		case Identifier :
			return false;
			
		case LeftParen :
		case PostDecrement :
		case PostIncrement :
		case PreDecrement :
		case PreIncrement :
		case Equal :
		case AndEqual :
		case OrEqual :
		case XorEqual :
		case MinusEqual :
		case PlusEqual :
		case MultiplyEqual :
		case DivideEqual :
		case RemainderEqual :
		case LeftShiftEqual :
		case RightShiftEqual :
			return true;
			
		case LeftBracket :
		case Comma :
		case LeftShift :
		case RightShift :
		case And :
		case Or :
		case Plus :
		case Minus :
		case AndAnd :
		case OrOr :
		case Colon :
		case Question :
		case Less :
		case LessEqual :
		case Greater :
		case GreaterEqual :
		case NotEqual :
		case EqualEqual :
		case Multiply :
		case Divide : 
		case Remainder :
			return hasSideEffects(node->data.children.node1)
						|| hasSideEffects(node->data.children.node2);

		case AddressOf :
		case Convert :
		case UnaryMinus :
		case Arrow :
		case Not :
			return hasSideEffects(node->data.children.node1);
	}
}

Node *addNode(int op, int typeIndex, Node *node1, Node *node2)
{
	Node *result;
	if (topNode == MaxNodes) {
		error("Ran out of node space");
		return NULL;
	}
	result = &tree[topNode++];
	result->op = op;
	result->reg = None;
	result->typeIndex = typeIndex;
	result->data.children.node1 = node1;
	result->data.children.node2 = node2;
	return result;
}

Node *addDoubleImmediateNode(double theValue)
{
	Node *result;
	if (topNode == MaxNodes) {
		error("Ran out of node space");
		return NULL;
	}
	result = &tree[topNode++];
	result->op = Decl;
	result->offset = 0;
	result->reg = Immediate;
	result->data.dValue = theValue;
	result->typeIndex = DoubleTypeIndex;
	return result;
}

Node *addStringConstantNode(CharPtr startPtr, int length)
{
	Node *result;
	if (topNode == MaxNodes) {
		error("Ran out of node space");
		return NULL;
	}
	result = &tree[topNode++];
	result->op = DoubleQuote;
	result->reg = A5;
	result->data.identifier.name = startPtr;
	result->data.identifier.length = length;
	result->typeIndex = CharPtrTypeIndex;
	return result;
}

Node *addImmediateNode(long theValue, int typeIndex)
{
	unsigned long uValue = theValue;
	Node *result;
	if (topNode == MaxNodes) {
		error("Ran out of node space");
		return NULL;
	}
	result = &tree[topNode++];
	result->op = Decl;
	result->offset = 0;
	result->reg = Immediate;
	result->value = theValue;
	if (typeIndex != -1) 
		result->typeIndex = typeIndex;
	else
		if (uValue < 32768)
			result->typeIndex = IntTypeIndex;
		else
			if (uValue < 0x80000000)
				result->typeIndex = LongTypeIndex;
			else
				result->typeIndex = UnsignedLongTypeIndex;
	return result;
}

Node *addIdentifierNode(CharPtr name, int length, unsigned char hash)
{
	Node *result;
	if (topNode == MaxNodes) {
		error("Ran out of node space");
		return NULL;
	}
	result = &tree[topNode++];
	result->op = Identifier;
	result->offset = 0;
	result->reg = None;
	result->data.identifier.name = name;
	result->data.identifier.hash = hash;
	result->data.identifier.length = length;
	return result;
}

Node *addClone(Node *node)
{
	Node *result;
	if (topNode == MaxNodes) {
		error("Ran out of node space");
		return NULL;
	}
	result = &tree[topNode++];
	*result = *node;
	return result;
}

int currentFunctionType;

int typeIndex;
int MaxTypeIndex;
Type *typeTable = NULL;
Handle typeHandle = NULL;

Type predefinedTypes[] = {

	{ -1, Base, 2, { -1 } },
	{ -1, Base, 2, { -1 } },
	{ -1, Base, 4, { -1 } },
	{ -1, Base, 4, { -1 } },
	{ -1, Base, 2, { -1 } },
	{ -1, Base, 2, { -1 } },
	{ -1, Base, 1, { -1 } },
	{ -1, Base, 1, { -1 } },
	{ -1, Base, 4, { -1 } },
	{ -1, Base, 8, { -1 } },
	{ -1, Base, 0, { -1 } },

	{ IntTypeIndex, Pointer, 4, { -1 } },
	{ UnsignedIntTypeIndex, Pointer, 4, { -1 } },
	{ LongTypeIndex, Pointer, 4, { -1 } },
	{ UnsignedLongTypeIndex, Pointer, 4, { -1 } },
	{ ShortTypeIndex, Pointer, 4, { -1 } },
	{ UnsignedShortTypeIndex, Pointer, 4, { -1 } },
	{ CharTypeIndex, Pointer, 4, { -1 } },
	{ UnsignedCharTypeIndex, Pointer, 4, { -1 } },
	{ FloatTypeIndex, Pointer, 4, { -1 } },
	{ DoubleTypeIndex, Pointer, 4, { -1 } },
	{ VoidTypeIndex, Pointer, 4, { -1 } },
	
	{ CharPtrTypeIndex, Pointer, 4, { -1 } }
};

Declaration *globalTable;
Handle globalTableH;
UInt globalRecordIndex;
int globalTop;
Declaration *fullGlobalTable[8];
Handle fullGlobalTableH[8];
UInt fullGlobalRecordIndex[8];
int fullGlobalCount;

Declaration localDeclaration[MaxLocals];
int localTop;

int blockTable[MaxBlocks];
int topBlock;
int blockIndex;

int nextLabel;

Token unaryOperator[] = {
	Token_Error,			//	NotACharacter,
	Token_Error,			//	HexDigit,
	Token_Error,			//	Letter,
	Token_Error,			//	Digit,
	Token_Error,			//	Whitespace,
	Token_Error,			//	Operator,
	Token_Error,			//	EndOfInput,
	Token_Error,			//	Identifier,	
	Token_Error,			//	Token_Error,
	Token_Error,			//	Token_EOF,
	Token_Error,			//	Number,
	Token_Error,			//	Dollar,
	AddressOf,				//	And,
	Token_Error,			//	Or,
	Token_Error,			//	Xor,
	Indirect,				//	Multiply,
	Token_Error,			//	Divide,
	Token_Error,	 		//	Remainder,
	UnaryMinus,				//	Minus,
	UnaryPlus,				//	Plus,
	Token_Error,	 		//	LeftShift,
	Token_Error,	 		//	RightShift,	

	Token_Error,	 		//	AndEqual,
	Token_Error,	 		//	OrEqual,
	Token_Error,	 		//	XorEqual,
	Token_Error,	 		//	MultiplyEqual,
	Token_Error,			//	DivideEqual
	Token_Error,	 		//	RemainderEqual,
	Token_Error,	 		//	MinusEqual,
	Token_Error,	 		//	PlusEqual,
	Token_Error,	 		//	LeftShiftEqual,
	Token_Error,	 		//	RightShiftEqual,

	Token_Error,			//	LeftParen,
	Token_Error,			//	RightParen,
	Token_Error,			//	LeftCurly,
	Token_Error,			//	RightCurly,
	Token_Error,			//	LeftBracket,
	Token_Error,			//	RightBracket,
	Token_Error,			//	Comma,
	Token_Error,			//	Period,
	Token_Error,			//	Question,
	Token_Error,			//	Colon,
	Token_Error,			//	SemiColon,
	Tilde,					//	Tilde,
	Not,					//	Not,
	Token_Error,			//	DoubleQuote,
	Token_Error,			//	Quote,
	Token_Error,			//	Less,
	Token_Error,			//	Greater,
	Token_Error,			//	Equal,
	Token_Error,			//	UnaryMinus,
	Token_Error,			//	UnaryPlus
	Token_Error,			//	Indirect
	Token_Error,			//	AddressOf
	Token_Error,	 		//	EqualEqual,
	PreIncrement,			//	PostIncrement
	PreDecrement,			//	PostDecrement
	Token_Error,	 		//	PreIncrement,
	Token_Error,	 		//	PreDecrement,
	Token_Error,	 		//	Arrow,
	Token_Error,			//	LessEqual,
	Token_Error,			//	GreaterEqual,
	Token_Error,			//	NotEqual,
	Token_Error,			//	AndAnd,
	Token_Error,			//	OrOr,
	Sizeof,					//	Sizeof
	Token_Error,			//	Typedef

};

char operatorPrecedence[] = {
	0,			//	NotACharacter,
	0,			//	HexDigit,
	0,			//	Letter,
	0,			//	Digit,
	0,			//	Whitespace,
	0,			//	Operator,
	0,			//	EndOfInput,
	0,			//	Identifier,	
	0,			//	Token_Error,
	0,			//	Token_EOF,
	0,			//	Number,
	0,			//	Dollar,

	8,			//	And,
	6,			//	Or,
	7,			//	Xor,
	13,			//	Multiply,
	13,			//	Divide,
	13,	 		//	Remainder,
	12,			//	Minus,
	12,			//	Plus,
	11,	 		//	LeftShift,
	11,	 		//	RightShift,

	2,			//	AndEqual,
	2,			//	OrEqual,
	2,			//	XorEqual,
	2,	 		//	MultiplyEqual,
	2,			//	DivideEqual,
	2,	 		//	RemainderEqual,
	2,			//	MinusEqual,
	2,			//	PlusEqual,
	2,			//	LeftShiftEqual,
	2,			//	RightShiftEqual,

	16,			//	LeftParen,
	0,			//	RightParen,
	0,			//	LeftCurly,
	0,			//	RightCurly,
	16,			//	LeftBracket,
	0,			//	RightBracket,
	1,			//	Comma,
	16,			//	Period,
	3,			//	Question,
	3,			//	Colon,
	0,			//	SemiColon,
	14,			//	Tilde,
	14,			//	Not,
	0,			//	DoubleQuote,
	0,			//	Quote,
	10,			//	Less,
	10,			//	Greater,
	2,			//	Equal,
	14,			//	UnaryMinus,
	14, 		//	UnaryPlus,
	14,			//	Indirect,
	14,			//	AddressOf,
	9,			//	EqualEqual,
	15,			//	PostIncrement
	15,			//	PostDecrement
	14,			//	PreIncrement
	14,			//	PreDecrement
	16,			//	Arrow
	10,			//	LessEqual
	10,			//	GreaterEqual
	9,			//	NotEqual
	5,			//	AndAnd
	4,			//	OrOr
	14,			//	Sizeof
	14,			//	Typedef	(cast)
};


CharPtr typeName(CharPtr curTextPtr, int *theType, int scope)
{
	ParseState ps;
	CharPtr savePtr;
	int newType;
	Declaration *theName;
	
	Boolean previouslyDeclared = false;
	Boolean isUnion = false;

	curTextPtr = getNextToken(curTextPtr, &ps);
	if (ps.token == Const)	// Not supporting Const !!!
		curTextPtr = getNextToken(curTextPtr, &ps);
	switch (ps.token) {
		default :		// not a recognized type token
			pushBack(&ps);
			break;
		case Enum : {
				long curValue = 0;
				previouslyDeclared = false;
			
				savePtr = curTextPtr;
				curTextPtr = getNextToken(curTextPtr, &ps);
				if (ps.token == Identifier) {
					newType = findEnum(ps.start, ps.value, scope);
					if (newType == -1) {
						newType = newEnumType();
						theName = makeDeclaration(ps.start, ps.value, newType, curTextPtr, scope, ps.hash, true);
						if (scope == LocalScope) {
							theName->depth = blockIndex;
						}
						savePtr = curTextPtr;
						curTextPtr = getNextToken(curTextPtr, &ps);					
					}
					else
						previouslyDeclared = true;
				}
				else
					newType = newEnumType();		// anonymous
				if (!previouslyDeclared) {
					long enumRangeLow = 0x7FFFFFFF;
					long enumRangeHigh = 0x80000000;
					if (ps.token != LeftCurly) {
						sourceError("Left curly expected after enum", savePtr);
						return NULL;
					}
					while (true) {
						savePtr = curTextPtr;
						curTextPtr = getNextToken(curTextPtr, &ps);
						if (ps.token != Identifier) {
							sourceError("Identifier expected in enum list", savePtr);
							return NULL;
						}
						if (curValue < enumRangeLow) enumRangeLow = curValue;
						if (curValue > enumRangeHigh) enumRangeHigh = curValue;
						theName = makeDeclaration(ps.start, ps.value, newType, curTextPtr, scope, ps.hash, false);
						setBitWidth(theName, scope, true);
						setDataValue(theName, scope, curValue);
						if (scope == LocalScope) {
							theName->depth = blockIndex;
						}
						savePtr = curTextPtr;
						curTextPtr = getNextToken(curTextPtr, &ps);
						if (ps.token == Equal) {
							curTextPtr = constantExpression(curTextPtr, &curValue, false, NULL);						
							if (curTextPtr == NULL) return NULL;
							setDataValue(theName, scope, curValue);
							if (curValue < enumRangeLow) enumRangeLow = curValue;
							if (curValue > enumRangeHigh) enumRangeHigh = curValue;
							savePtr = curTextPtr;
							curTextPtr = getNextToken(curTextPtr, &ps);
						}
						curValue++;
						if (ps.token != Comma) {
							if (ps.token != RightCurly) {
								sourceError("Comma or right curly expected in enum list", savePtr);
								return NULL;
							}
							else
								break;
						}
					}
					if (((enumRangeLow >= 0) && (enumRangeHigh <= 255))
							|| ((enumRangeLow >= -128) && (enumRangeHigh <= 127))) 
						setEnumBaseType(newType, CharTypeIndex);
					else
						if (((enumRangeLow >= 0) && (enumRangeHigh <= 32767))
								|| ((enumRangeLow >= -32768) && (enumRangeHigh <= 32767))) 
							setEnumBaseType(newType, IntTypeIndex);
						else
							setEnumBaseType(newType, LongTypeIndex);
/*					
					if ((enumRangeHigh > 32767) || (enumRangeLow < -32768))
						setEnumBaseType(newType, LongTypeIndex);
					else
						if ((enumRangeHigh > 127) || (enumRangeLow < -128))
							setEnumBaseType(newType, IntTypeIndex);
						else
							setEnumBaseType(newType, CharTypeIndex);
*/							
				}
				*theType = newType;
			}
			break;
		case Union :
				isUnion = true;
		case Struct : {
				int curOffset = 0;
				int totalSize = 0;
				//Boolean lastBitOffset = 0;
				previouslyDeclared = false;				
				savePtr = curTextPtr;
				curTextPtr = getNextToken(curTextPtr, &ps);
				if (ps.token == Identifier) {
					newType = findStruct(ps.start, ps.value, scope, ps.hash);
					if (newType == -1) {
						newType = newStructType();
						theName = makeDeclaration(ps.start, ps.value, newType, curTextPtr, scope, ps.hash, true);
						if (scope == LocalScope) {
							theName->depth = blockIndex;
						}
						savePtr = curTextPtr;
						curTextPtr = getNextToken(curTextPtr, &ps);					
					}
					else {
						// want to handle forward declarations of structs
						// and allow subsequent declaration of their contents
						previouslyDeclared = !isEmptyStruct(newType);
						if (!previouslyDeclared)
							curTextPtr = getNextToken(curTextPtr, &ps);					
					}
				}
				else
					newType = newStructType();		// anonymous
				if (!previouslyDeclared) {
					int previousField = -1;
					int fieldStart    = 0;
					if (ps.token != LeftCurly) {
						pushBack(&ps);
						*theType = newType;
						return curTextPtr;
/*											
						sourceError("Left curly expected after struct", savePtr);
						return NULL;
*/						
					}
					while (true) {
						int baseType = -1;
						while (true) {
							int fieldSize;
							theName = NULL;		
							savePtr = curTextPtr;
						    curTextPtr = declaration(curTextPtr, &baseType, &theName, FieldScope, false);
							if (curTextPtr == NULL) return NULL;
							if (theName == NULL) {
								sourceError("Struct member expected", savePtr);
								return NULL;
							}
							else {
								if (previousField != -1) {
									int fieldIndex = fieldStart;
									while (fieldIndex != -1) {
										Declaration *theField = &fieldList[fieldIndex];
										if ((theField->hash == theName->hash)
												&& (theField->length == theName->length)) {
											int i;
											for (i = 0; i < theField->length; i++) {									
												if (theField->name[i] != theName->name[i])
													break;
											}
											if (i == theField->length) {
												sourceError("Duplicate field name", savePtr);
												return NULL;
											}
										}
										fieldIndex = theField->data.nextField;
									}
								}
							}
							if (previousField != -1)
								fieldList[previousField].data.nextField = theName->data.nextField;	// theName->data.nextField is initialized to itself
							else
								fieldStart = theName->data.nextField;
							previousField = theName->data.nextField;
							theName->data.nextField = -1;
							savePtr = curTextPtr;
							curTextPtr = getNextToken(curTextPtr, &ps);
							if (ps.token == Colon) {
								savePtr = curTextPtr;
								curTextPtr = getNextToken(curTextPtr, &ps);
								if (ps.token != Digit) {
									sourceError("Bitfield width expected", savePtr);
									return NULL;
								}
								fieldSize = ps.value;
								savePtr = curTextPtr;
								curTextPtr = getNextToken(curTextPtr, &ps);
							}
							else
								fieldSize = getTypeSize(theName->typeIndex) << 3;
							theName->bitWidth = fieldSize;
							if (isUnion) {
								if (fieldSize > totalSize)
									totalSize = fieldSize;
								theName->depth = 0;
							}
							else {
								if ((fieldSize & 7) == 0) {
									if ((curOffset & 7) != 0)
										curOffset += 8 - (curOffset & 7);
									// round up to 16-bit offset unless the current field is byte sized, or an array of byte sized elements
									if ((fieldSize != 8)
											&& (! (isArray(theName->typeIndex) && (getTypeSize(getBaseType(theName->typeIndex)) == 1)) ))
									{
										if ((curOffset & 8) != 0)
											curOffset += 8;
									}
								}
								theName->depth = curOffset;
								curOffset += fieldSize;
								totalSize = curOffset;
							}
							if (ps.token == Comma) {
								curTextPtr = getNextToken(curTextPtr, &ps);
								continue;
							}
							else {
								if (ps.token != SemiColon) {
									sourceError("Semicolon or comma expected after field declaration", savePtr);
									return NULL;
								}
								curTextPtr = getNextToken(curTextPtr, &ps);
								if (ps.token == RightCurly)
									goto fieldsDone;
								else
									pushBack(&ps);
								break;
							}
						}
					}
fieldsDone:						
					if ((totalSize & 15) != 0)
						totalSize += 16 - (totalSize & 15);
					setFields(newType, fieldStart, totalSize >> 3);
				}
				*theType = newType;
			}
			break;
		case Int_Token :
			*theType = IntTypeIndex;
			break;
		case Long_Token :
			*theType = LongTypeIndex;
			break;
		case Float_Token :
			*theType = FloatTypeIndex;
			break;
		case Double_Token :
			*theType = DoubleTypeIndex;
			break;
		case Char_Token :
			*theType = CharTypeIndex;
			break;
		case Short_Token :
			*theType = ShortTypeIndex;
			break;
		case Void_Token :
			*theType = VoidTypeIndex;
			break;
		case Unsigned_Token : {
				savePtr = curTextPtr;
				curTextPtr = getNextToken(curTextPtr, &ps);
				switch (ps.token) {
					case Int_Token :
						*theType = UnsignedIntTypeIndex;
						break;
					case Long_Token :
						*theType = UnsignedLongTypeIndex;
						break;
					case Char_Token :
						*theType = UnsignedCharTypeIndex;
						break;
					case Short_Token :
						*theType = UnsignedShortTypeIndex;
						break;
					default :
						sourceError("Bad type following unsigned", savePtr);
						return NULL;
				}
			}
			break;
		case Identifier : {
				char reg;
				theName = findIdentifier(ps.start, ps.value, &reg, ps.hash, NULL);				
				if ((theName != NULL) && (theName->isTypedef)) {
					*theType = theName->typeIndex;
					return curTextPtr;
				}
				else
					pushBack(&ps);
			}
			break;
	}
	return curTextPtr;
}


void makeSpaceForArgument()
{
    if (argumentListTop == argumentListMax) {
//		Err err;
		void *newArgs;
		MemHandleUnlock(argumentListData);
		argumentListMax += 32;
		
		newArgs = MemHandleNew(argumentListMax * sizeof(Declaration));
		if (newArgs == NULL)
			error("ran out of memory");
		else {
			char *src = MemHandleLock(argumentListData);
			argumentList = MemHandleLock(newArgs);
			MemMove(argumentList, src, argumentListTop * sizeof(Declaration));
			MemHandleUnlock(argumentListData);
			MemHandleFree(argumentListData);
			argumentListData = newArgs;
		}
/*		
		err = MemHandleResize(argumentListData, argumentListMax * sizeof(Declaration));
		if (err != 0) {
			// try one more time, I've been getting a weird case where the MemMgr refuses
			// to allocate when it clearly can do so, but a second attempt succeeds.
			// ( III, 3.3 8mb debug-rom on Pose 3.0a6 )
			err = MemHandleResize(argumentListData, argumentListMax * sizeof(Declaration));
		}
		argumentList = MemHandleLock(argumentListData);
*/
    }
}

CharPtr doArgLoop(int fnType, CharPtr curTextPtr)
{
	Boolean singleVoidArgument = true;
	Boolean hadVarArg = false;
	ParseState ps;
	int argStart = argumentListTop;
	CharPtr savePtr;
		
	curTextPtr = getNextToken(curTextPtr, &ps);
	if (ps.token != RightParen) {
		pushBack(&ps);
		while (true) {
			Declaration *theDecl = NULL;
			int baseType = -1;				
			savePtr = curTextPtr;
			curTextPtr = getNextToken(curTextPtr, &ps);				
			if (ps.token == Period) {
			    curTextPtr = getNextToken(curTextPtr, &ps);
			    if (ps.token == Period) {
				    curTextPtr = getNextToken(curTextPtr, &ps);
				    if (ps.token == Period) {
						hadVarArg = true;
						singleVoidArgument = false;
					    curTextPtr = getNextToken(curTextPtr, &ps);
						if (ps.token == RightParen)
							break;
						else {	// balance '('
							sourceError("expected ')' after '...'", savePtr);
							return NULL;
						}
				    }
				}
				sourceError("'...' expected", savePtr);
				return NULL;
			}
			else {
			    int i = argumentListTop;

				pushBack(&ps);
			    curTextPtr = declaration(curTextPtr, &baseType, &theDecl, ArgumentScope, true);
				if (curTextPtr == NULL) return NULL;
				if (theDecl == NULL) {
					sourceError("Argument expected", savePtr);
					return NULL;
				}

				if (argumentListTop != (i + 1)) {
					// yuck, an embedded function type with its own args.
					// we have to move all our args beyond the
					// new argumentListTop and reset argStart
					int oldTop = i + 1;		// our last arg was inserted first
					int newTop = argumentListTop;
					for (i = argStart; i < oldTop; i++) {
						makeSpaceForArgument();
						argumentList[argumentListTop++] = argumentList[i];
					}  
					argStart = newTop;
				}
		
				if (isArray(theDecl->typeIndex)) 
					theDecl->typeIndex = newPointerType(getBaseType(theDecl->typeIndex));
				theDecl->depth = 0;
				if (theDecl->typeIndex != VoidTypeIndex) singleVoidArgument = false;
				curTextPtr = getNextToken(curTextPtr, &ps);
				if (ps.token != Comma) {
					if (ps.token == RightParen)
						break;
					else {
						sourceError("Unexpected token in argument list", savePtr);
						return NULL;
					}
				}
				else
					singleVoidArgument = false;
			}
		}
	}
	// blow off the single 'void' argument
	if (singleVoidArgument)
		argumentListTop = argStart;
	setArguments(fnType, argStart, hadVarArg);
	return curTextPtr;
}

CharPtr declarator(CharPtr curTextPtr, int startType, Declaration **theDecl, int declScope, Boolean allowAbstract)
{
	ParseState ps;
	CharPtr peekPtr;
	CharPtr savePtr = curTextPtr;	
	Declaration *result;
	int fnType, emptyType;
	int *typeIndexPtr;
	int arrayType;
		
	curTextPtr = getNextToken(curTextPtr, &ps);
	if (ps.token == Multiply) {
		if (startType == CharTypeIndex)
			startType = CharPtrTypeIndex;
		else
			startType = newPointerType(startType);
		curTextPtr = declarator(curTextPtr, startType, theDecl, declScope, allowAbstract);
		if (curTextPtr == NULL) return NULL;
		return curTextPtr;
	}
	if (ps.token == LeftParen) {
		// We'll be stitching types into the type constructed by this recursive call
		// depending on what follows the parentheses. We can't simply pass in the startType
		// since that would encourage discovery of pre-built or cached types - which we
		// are unable to subsequently modify.
		emptyType = newType();
		curTextPtr = declarator(curTextPtr, emptyType, theDecl, declScope, allowAbstract);
		if (curTextPtr == NULL) return NULL;
		curTextPtr = getNextToken(curTextPtr, &ps);
		if (ps.token != RightParen) {
			sourceError("Right parenthesis expected in declarator", savePtr);
			return NULL;
		}
		if (*theDecl == NULL) return NULL;
		replaceType((*theDecl)->typeIndex, emptyType, startType);
		// if the new type was an array type, it'll have a zero size, so re-compute now
		fixTypeSizes((*theDecl)->typeIndex);

		peekPtr = curTextPtr;
		curTextPtr = getNextToken(curTextPtr, &ps);
		if (ps.token == LeftBracket) {
			while (ps.token == LeftBracket) {
				typeIndexPtr = &((*theDecl)->typeIndex);
				while (*typeIndexPtr != startType)
					typeIndexPtr = getBaseTypePtr(*typeIndexPtr);
				arrayType = newArrayType(*typeIndexPtr);
				*typeIndexPtr = arrayType;
				typeIndexPtr = getBaseTypePtr(arrayType);
				savePtr = curTextPtr;
				curTextPtr = getNextToken(curTextPtr, &ps);
				if (ps.token != RightBracket) {
					pushBack(&ps);
					curTextPtr = constantExpression(curTextPtr, &ps.value, false, NULL);
					if (curTextPtr == NULL) {
						sourceError("Constant expression expected in array declaration", savePtr);
						return NULL;
					}
					setDimension(arrayType, ps.value);
					curTextPtr = getNextToken(curTextPtr, &ps);
					if (ps.token != RightBracket) {
						sourceError("Right bracket expected in array declaration", savePtr);
						return NULL;
					}
				}
				fixTypeSizes((*theDecl)->typeIndex);
				peekPtr = curTextPtr;
				curTextPtr = getNextToken(curTextPtr, &ps);
			}
			pushBack(&ps);
		}
		else {
			if (ps.token == LeftParen) {
				typeIndexPtr = &((*theDecl)->typeIndex);
				while (*typeIndexPtr != startType)
					typeIndexPtr = getBaseTypePtr(*typeIndexPtr);
				fnType = newFunctionType(*typeIndexPtr);
				*typeIndexPtr = fnType;							
				curTextPtr = doArgLoop(fnType, curTextPtr);
			}
			else
				pushBack(&ps);
		}
	}
	else {
		if (ps.token == Identifier) {
//			char reg;
			if (declScope == FieldScope) {
				result = makeDeclaration(ps.start, ps.value, startType, curTextPtr, declScope, ps.hash, false);
			}
			else {
/*			
				Declaration *d = findIdentifier(ps.start, ps.value, &reg, ps.hash);
				if (d != NULL) {
					if (d->isTypedef) {
						sourceError("Illegal redefinition of typedef", ps.start);
						return NULL;
					}
					else {
					// urk, only handling forward-declarations when in global scope
						if ((declScope == GlobalScope) && (d->depth == -1)) {
							result = d;				// REMIND - check for mismatch between forward & actual type.
													// Which is hard, because the type isn't complete here, just the
													// base type. Maybe leave the depth as -1 and have the callers do
													// the checking somehow ?????
							setDepth(d, declScope, 0);
							if (!d->isStatic)
								 setStatic(d, GlobalScope, -1);	// mark as forward extern
						}
						else {
							if (declScope == GlobalScope) {
								if ((reg == A5) || (reg == PC)) {
									sourceError("Duplicate global definition", ps.start);
									return NULL;
								}
							}
							else {
								if (reg == A6) {
									if (d->depth == topBlock) {
										sourceError("Duplicate local definition", ps.start);
										return NULL;
									}
								}
							}
							result = makeDeclaration(ps.start, ps.value, startType, curTextPtr, declScope, ps.hash, false);
						}
					}
				}
				else
*/				
					result = makeDeclaration(ps.start, ps.value, startType, curTextPtr, declScope, ps.hash, false);
			}
			*theDecl = result;
			peekPtr = curTextPtr;
			curTextPtr = getNextToken(curTextPtr, &ps);
			if (ps.token == LeftBracket) {
				int *typeIndexPtr = &result->typeIndex;	
				Boolean modifyDecl = true;
				while (ps.token == LeftBracket) {
					arrayType = newArrayType(*typeIndexPtr);
					if (modifyDecl) {
						setTypeIndex(result, declScope, arrayType);
						modifyDecl = false;
					}
					else
						*typeIndexPtr = arrayType;
					typeIndexPtr = getBaseTypePtr(arrayType);
					savePtr = curTextPtr;
					curTextPtr = getNextToken(curTextPtr, &ps);
					if (ps.token != RightBracket) {
						pushBack(&ps);
						curTextPtr = constantExpression(curTextPtr, &ps.value, false, NULL);
						if (curTextPtr == NULL) {
							sourceError("Constant expression expected in array declaration", savePtr);
							return NULL;
						}
						setDimension(arrayType, ps.value);
						curTextPtr = getNextToken(curTextPtr, &ps);
						if (ps.token != RightBracket) {
							sourceError("Right bracket expected in array declaration", savePtr);
							return NULL;
						}
					}
					fixTypeSizes(result->typeIndex);
					peekPtr = curTextPtr;
					curTextPtr = getNextToken(curTextPtr, &ps);
				}
				pushBack(&ps);
			}
			else {
				if (ps.token == LeftParen) {
					fnType = newFunctionType(startType);			// use startType rather than result->typeIndex
																	// since the latter may be a complex type having
																	// arisen from this being a re-encountered forward
																	// declaration. (In which case we really ought to be
																	// validating the type rather than building a new one)
					setTypeIndex(result, declScope, fnType);
					curTextPtr = doArgLoop(fnType, curTextPtr);
						
				}
				else
					pushBack(&ps);
			}
		}
		else {
			if (allowAbstract) {
				result = makeDeclaration(NULL, 0, startType, curTextPtr, declScope, 0, false);
				*theDecl = result;
				pushBack(&ps);
			}
			else {
				sourceError("Illegal declarator", savePtr);
				return NULL;
			}
		}
	}
	return curTextPtr;
}

Node *resolveIdentifier(Node *operand)
{
	if (operand == NULL) return operand;
	if (operand->op == Identifier) {
		Declaration *d = findIdentifier(operand->data.identifier.name,
	 										operand->data.identifier.length,
													&operand->reg, operand->data.identifier.hash, NULL);
		if (d == NULL) {
			sourceErrorWithLength("unknown identifier",
							 operand->data.identifier.name, operand->data.identifier.length);
			return NULL;
		}
		else {
			if (d->isTypedef) {
				sourceErrorWithLength("illegal use of typedef name",
							 operand->data.identifier.name, operand->data.identifier.length);
				return NULL;
			}
			else {
				int theType = d->typeIndex;
				if ((d->bitWidth != 0) && isEnum(theType)) {
					operand->op = Decl;
					operand->offset = 0;
					operand->reg = Immediate;
					operand->typeIndex = theType;
					operand->value = d->data.value;
				}
				else {
					operand->op = Decl;
					operand->offset = 0;
					operand->data.decl = d;
					operand->typeIndex = theType;
					if (isFunction(theType)) {
						operand = addNode(AddressOf, newPointerType(theType), operand, NULL);
					}
/*
					else {
						if (isArray(theType)) {
							operand = addNode(AddressOf, newPointerType(getBaseType(theType)), operand, NULL);
						}
					}
*/
				}
			}
		}
	}
	return operand;
}


Boolean usualBinaryConvert(Node *binaryNode)
{
	Node *operand1 = binaryNode->data.children.node1;
	Node *operand2 = binaryNode->data.children.node2;
	Node *node;
	
	if (isEnum(operand1->typeIndex))
		operand1->typeIndex = getBaseType(operand1->typeIndex);
	
	if (isEnum(operand2->typeIndex))
		operand2->typeIndex = getBaseType(operand2->typeIndex);
	
	switch (operand1->typeIndex) {
		case FloatTypeIndex : {
				node = addNode(Convert, DoubleTypeIndex, operand1, NULL);
				if (node == NULL) return false;
				binaryNode->data.children.node1 = node;
				// fall thru...
			}
		case DoubleTypeIndex : {
				switch (operand2->typeIndex) {
					case DoubleTypeIndex :
						return true;
					case FloatTypeIndex :
					case UnsignedLongTypeIndex :
					case LongTypeIndex :
					case UnsignedIntTypeIndex :
					case IntTypeIndex :
					case UnsignedShortTypeIndex :
					case ShortTypeIndex :
					case UnsignedCharTypeIndex :
					case CharTypeIndex :
						node = addNode(Convert, DoubleTypeIndex, operand2, NULL);
						if (node == NULL) return false;
						binaryNode->data.children.node2 = node;
						return true;
					default :
						sourceError("illegal combination of types", operatorStack[topOfOperatorStack].position);
						return false;
				}
			}
		case UnsignedLongTypeIndex :
		case LongTypeIndex : {
				switch (operand2->typeIndex) {
					case FloatTypeIndex :
						node = addNode(Convert, DoubleTypeIndex, operand2, NULL);
						if (node == NULL) return false;
						binaryNode->data.children.node2 = node;
						// fall thru...
					case DoubleTypeIndex :
						node = addNode(Convert, DoubleTypeIndex, operand1, NULL);
						if (node == NULL) return false;
						binaryNode->data.children.node1 = node;
						return true;
					case UnsignedLongTypeIndex :
					case LongTypeIndex :
						return true;
					case UnsignedIntTypeIndex :
					case IntTypeIndex :
					case UnsignedShortTypeIndex :
					case ShortTypeIndex :
					case UnsignedCharTypeIndex :
					case CharTypeIndex :
						node = addNode(Convert, operand1->typeIndex, operand2, NULL);
						if (node == NULL) return false;
						binaryNode->data.children.node2 = node;
						return true;
					default :
						sourceError("illegal combination of types", operatorStack[topOfOperatorStack].position);
						return false;
				}
			}
		case UnsignedShortTypeIndex :
		case ShortTypeIndex :
		case UnsignedIntTypeIndex :
		case IntTypeIndex : {
				switch (operand2->typeIndex) {
					case FloatTypeIndex :
						node = addNode(Convert, DoubleTypeIndex, operand2, NULL);
						if (node == NULL) return false;
						binaryNode->data.children.node2 = node;
						// fall thru...
					case DoubleTypeIndex :
						node = addNode(Convert, DoubleTypeIndex, operand1, NULL);
						if (node == NULL) return false;
						binaryNode->data.children.node1 = node;
						return true;
					case UnsignedLongTypeIndex :
					case LongTypeIndex :
						node = addNode(Convert, operand2->typeIndex, operand1, NULL);
						if (node == NULL) return false;
						binaryNode->data.children.node1 = node;
						return true;
					case UnsignedIntTypeIndex :
					case IntTypeIndex :
					case UnsignedShortTypeIndex :
					case ShortTypeIndex :
						return true;
					case UnsignedCharTypeIndex :
					case CharTypeIndex :
						node = addNode(Convert, operand1->typeIndex, operand2, NULL);
						if (node == NULL) return false;
						binaryNode->data.children.node2 = node;
						return true;
					default :
						sourceError("illegal combination of types", operatorStack[topOfOperatorStack].position);
						return false;
				}
			}
		case UnsignedCharTypeIndex :
		case CharTypeIndex : {
				switch (operand2->typeIndex) {
					case FloatTypeIndex :
						node = addNode(Convert, DoubleTypeIndex, operand2, NULL);
						if (node == NULL) return false;
						binaryNode->data.children.node2 = node;
						// fall thru...
					case DoubleTypeIndex :
						node = addNode(Convert, DoubleTypeIndex, operand1, NULL);
						if (node == NULL) return false;
						binaryNode->data.children.node1 = node;
						return true;
					case UnsignedLongTypeIndex :
					case LongTypeIndex :
						node = addNode(Convert, operand2->typeIndex, operand1, NULL);
						if (node == NULL) return false;
						binaryNode->data.children.node1 = node;
						return true;
					case UnsignedShortTypeIndex :
					case ShortTypeIndex :
					case UnsignedIntTypeIndex :
					case IntTypeIndex :
						node = addNode(Convert, operand2->typeIndex,
												binaryNode->data.children.node1, NULL);
						if (node == NULL) return false;
						binaryNode->data.children.node1 = node;
						return true;
					case UnsignedCharTypeIndex :
					case CharTypeIndex :
						node = addNode(Convert, IntTypeIndex, operand1, NULL);
						if (node == NULL) return false;
						binaryNode->data.children.node1 = node;
						node = addNode(Convert, IntTypeIndex, operand2, NULL);
						if (node == NULL) return false;
						binaryNode->data.children.node2 = node;					
						return true;
					default :
						sourceError("illegal combination of types", operatorStack[topOfOperatorStack].position);
						return false;
				}
			}
		default :
			sourceError("illegal combination of types", operatorStack[topOfOperatorStack].position);
			return false;
	}
}

Node *usualUnaryConvert(Node *unaryChild)
{	
	if (isArray(unaryChild->typeIndex)) {
		return addNode(AddressOf, newPointerType(getBaseType(unaryChild->typeIndex)), unaryChild, NULL);
	}

	switch (unaryChild->typeIndex) {
		case DoubleTypeIndex :
			return unaryChild;
		case FloatTypeIndex :
			return addNode(Convert, DoubleTypeIndex, unaryChild, NULL);
//			sourceError("Floating-point not supported", operatorStack[topOfOperatorStack].position);
			return NULL;
		case UnsignedLongTypeIndex :
		case LongTypeIndex :
		case UnsignedShortTypeIndex :
		case ShortTypeIndex :
		case UnsignedIntTypeIndex :
		case IntTypeIndex :
			return unaryChild;
		case UnsignedCharTypeIndex :
		case CharTypeIndex :
			return addNode(Convert, IntTypeIndex, unaryChild, NULL);
/*
		default :
			sourceError("illegal type", operatorStack[topOfOperatorStack].position);
			return NULL;
*/
		default :
			return unaryChild;
	}
}

CharPtr declaration(CharPtr curTextPtr, int *baseType, Declaration **theDecl, int declScope, Boolean allowAbstract)
{
	ParseState ps;
	CharPtr savePtr = curTextPtr;
	Boolean sawStatic = false;
	Boolean sawExtern = false;
	Boolean sawRegister = false;
	Boolean localStatic = false;
	
	savePtr = curTextPtr;
	curTextPtr = getNextToken(curTextPtr, &ps);
	if (ps.token == Static) {
		sawStatic = true;
		if (declScope != GlobalScope)
			localStatic = true;
	}
	else
		if (ps.token == Extern)
			sawExtern = true;
		else
			if (ps.token == Register)
				sawRegister = true;
			else
				if (ps.token != Auto)	// consume and ignore
					pushBack(&ps);

	if (*baseType == -1) {
		curTextPtr = typeName(curTextPtr, baseType, declScope);
		if (*baseType == -1) return curTextPtr;			// no type, no declaration
		if (curTextPtr == NULL) return NULL;
	}
	savePtr = curTextPtr;
	curTextPtr = getNextToken(curTextPtr, &ps);
	if (ps.token == SemiColon) {
		pushBack(&ps);
	}
	else {
		pushBack(&ps);
		if (localStatic) declScope = GlobalScope;	// promote local statics if necessary
		curTextPtr = declarator(curTextPtr, *baseType, theDecl, declScope, allowAbstract);
		if (*theDecl != NULL) {
			if ((declScope == LocalScope) && sawRegister)
				(*theDecl)->data.value = -1;
/*
			if ((*theDecl)->isStatic == 1)	// have to keep it so, it may already have been referenced
				setDepth(*theDecl, declScope, 0);
			else {				
				if (sawStatic) {
					setDepth(*theDecl, declScope, 0);
					if (!localStatic) {
						if ((*theDecl)->isStatic == -1)		// a forward-extern, can't make it static now
							setStatic(*theDecl, declScope, false);
						else
							setStatic(*theDecl, declScope, true);
					}
					else
						setStatic(*theDecl, declScope, false);
				}
				else {
					setDepth(*theDecl, declScope, -1);
					setStatic(*theDecl, declScope, false);
				}
			}
*/			
		}
	}
	return curTextPtr;
}

Node *addConvert(Node *node, int toTypeIndex)
{
	if (isEnum(toTypeIndex))
		toTypeIndex = getBaseType(toTypeIndex);

	if (isEnum(node->typeIndex))
		node->typeIndex = getBaseType(node->typeIndex);
		
	if (toTypeIndex == node->typeIndex) return node;

	switch (toTypeIndex) {
		case FloatTypeIndex :
		case DoubleTypeIndex : {
				switch (node->typeIndex) {
					case FloatTypeIndex :
					case DoubleTypeIndex :
					case UnsignedLongTypeIndex :
					case LongTypeIndex :
					case UnsignedIntTypeIndex :
					case IntTypeIndex :
					case UnsignedShortTypeIndex :
					case ShortTypeIndex :
					case UnsignedCharTypeIndex :
					case CharTypeIndex : {
							Node *result = addNode(Convert, toTypeIndex, node, NULL);
							if (result == NULL) return false;
							return result;
						}
					default :
						sourceError("illegal combination of types", operatorStack[topOfOperatorStack].position);
						return NULL;
				}
			}
		case UnsignedLongTypeIndex :
		case LongTypeIndex : {
				switch (node->typeIndex) {
					case UnsignedLongTypeIndex :
					case LongTypeIndex :
						return node;
					case FloatTypeIndex :
					case DoubleTypeIndex :
					case UnsignedIntTypeIndex :
					case IntTypeIndex :
					case UnsignedShortTypeIndex :
					case ShortTypeIndex :
					case UnsignedCharTypeIndex :
					case CharTypeIndex : {
							Node *result = addNode(Convert, toTypeIndex, node, NULL);
							if (result == NULL) return false;
							return result;
						}
					default :
						sourceError("illegal combination of types", operatorStack[topOfOperatorStack].position);
						return NULL;
				}
			}
		case UnsignedShortTypeIndex :
		case ShortTypeIndex :
		case UnsignedIntTypeIndex :
		case IntTypeIndex : {
				switch (node->typeIndex) {
					case FloatTypeIndex :
					case DoubleTypeIndex :
					case UnsignedCharTypeIndex :
					case CharTypeIndex :
					case UnsignedLongTypeIndex :
					case LongTypeIndex : {
							Node *result = addNode(Convert, toTypeIndex, node, NULL);
							if (result == NULL) return false;
							return result;
						}
					case UnsignedIntTypeIndex :
					case IntTypeIndex :
					case UnsignedShortTypeIndex :
					case ShortTypeIndex :
						return node;
					default :
						sourceError("illegal combination of types", operatorStack[topOfOperatorStack].position);
						return NULL;
				}
			}
		case UnsignedCharTypeIndex :
		case CharTypeIndex : {
				switch (node->typeIndex) {
					case FloatTypeIndex :
					case DoubleTypeIndex :
					case UnsignedShortTypeIndex :
					case ShortTypeIndex :
					case UnsignedIntTypeIndex :
					case IntTypeIndex :
					case UnsignedLongTypeIndex :
					case LongTypeIndex : {
							Node *result = addNode(Convert, toTypeIndex, node, NULL);
							if (result == NULL) return false;
							return result;
						}
					case UnsignedCharTypeIndex :
					case CharTypeIndex :
						return node;
					default :
						sourceError("illegal combination of types", operatorStack[topOfOperatorStack].position);
						return false;
				}
			}
		default :
			sourceError("illegal combination of types", operatorStack[topOfOperatorStack].position);
			return false;
	}
}

Node *constructOp(int op, Node *operand1, Node *operand2)
{
	Node *node;
	int typeSize;

	if (isArray(operand1->typeIndex))
		operand1 = addNode(AddressOf, newPointerType(getBaseType(operand1->typeIndex)), operand1, NULL);
	if (isArray(operand2->typeIndex))
		operand2 = addNode(AddressOf, newPointerType(getBaseType(operand2->typeIndex)), operand2, NULL);

	if ((op == Minus) && isPointer(operand1->typeIndex) && isPointer(operand2->typeIndex)) {
		if (!matchTypes(operand1->typeIndex, operand2->typeIndex)) {
			sourceError("Illegal combination of pointer types for '-'", operatorStack[topOfOperatorStack].position);
			return NULL;
		}
		typeSize = getTypeSize(getBaseType(operand1->typeIndex));
		operand1->typeIndex = LongTypeIndex;
		operand2->typeIndex = LongTypeIndex;
		node = addNode(op, LongTypeIndex, operand1, operand2);
		if (typeSize > 1)
			node = addNode(Divide, LongTypeIndex, node, addImmediateNode(typeSize, UnsignedLongTypeIndex));
	}
	else {
		if ((op == Plus) || (op == Minus)) {
			if (isPointer(operand1->typeIndex)) {
				if (!isIntegral(operand2->typeIndex)) {
					sourceError("Illegal combination of pointer and non-integral types", operatorStack[topOfOperatorStack].position);
					return NULL;
				}
				if ((operand2->typeIndex != LongTypeIndex)
						&& (operand2->typeIndex != UnsignedLongTypeIndex)) {
					operand2 = addConvert(operand2, LongTypeIndex);
					if (operand2 == NULL) return NULL;
				}
				if (getTypeSize(getBaseType(operand1->typeIndex)) > 1) {
					operand2 = addNode(Multiply, operand2->typeIndex, operand2, 
									addImmediateNode(getTypeSize(getBaseType(operand1->typeIndex)), UnsignedLongTypeIndex));
				}
				node = addNode(op, operand1->typeIndex, operand1, operand2);
			}
			else {
				if (isPointer(operand2->typeIndex)) {
					if (!isIntegral(operand1->typeIndex)) {
						sourceError("Illegal combination of pointer and non-integral types", operatorStack[topOfOperatorStack].position);
						return NULL;
					}
					if ((operand1->typeIndex != LongTypeIndex)
							&& (operand1->typeIndex != UnsignedLongTypeIndex)) {
						operand1 = addConvert(operand1, LongTypeIndex); 
						if (operand1 == NULL) return NULL;
					}
					if (getTypeSize(getBaseType(operand2->typeIndex)) > 1) {
						operand1 = addNode(Multiply, operand1->typeIndex, operand1, 
										addImmediateNode(getTypeSize(getBaseType(operand2->typeIndex)), UnsignedLongTypeIndex));
					}
					node = addNode(op, operand2->typeIndex, operand1, operand2);
				}
				else {
					node = addNode(op, IntTypeIndex, operand1, operand2);
					if (node == NULL) return NULL;
					if (!usualBinaryConvert(node))
						return NULL;
					node->typeIndex = node->data.children.node1->typeIndex;	// Not necessarily the same as operand1->typeIndex !
				}
			}
		}
		else {
			node = addNode(op, IntTypeIndex, operand1, operand2);
			if (node == NULL) return false;
			if (!usualBinaryConvert(node))
				return NULL;
			node->typeIndex = node->data.children.node1->typeIndex;	// Not necessarily the same as operand1->typeIndex !
		}
	}
	return node;
}

int tempIndex;				// temps allocated from start of pool
int tempCount;				// total numbers of temps allocated in function



Declaration *makeDeclaration(CharPtr startPtr, int length, int startType,
							 CharPtr curTextPtr, int declScope, unsigned char hash,
							 Boolean isTypeName)
{
	if (hash > 127) error("weird hash");

	if (declScope == GlobalScope) {
		Declaration glbl;
		Declaration *result = &glbl;
	    if (globalTop == MaxGlobal) {
	    	UInt globalsAt = -1;
	    	if (fullGlobalCount == 8) {
		        sourceError("WAY too many globals", curTextPtr);
		        return false;
	    	}
	    	fullGlobalTable[fullGlobalCount] = globalTable;
	    	fullGlobalTableH[fullGlobalCount] = globalTableH;
	    	fullGlobalRecordIndex[fullGlobalCount++] = globalRecordIndex;
			globalTableH = DmNewRecord(gGlobalsDB, &globalsAt, MaxGlobal * sizeof(Declaration));
			globalRecordIndex = globalsAt;
	    	globalTable = MemHandleLock(globalTableH);
	    	globalTop = 0;
	    }
//		result = &globalTable[globalTop];
//		globalTop++;

		result->data.nextField = -1;
		result->name = startPtr;
		result->hash = hash;
		result->depth = -1;
		result->bitWidth = 0;
		result->length = length;
		result->typeIndex = startType;
		result->isTypedef = false;
		result->isStatic = false;
		result->typeNameSpace = isTypeName;
		result->nextDecl = gGlobalHashTable[hash];
		gGlobalHashTable[hash] = &globalTable[globalTop];

		DmWrite(globalTable, globalTop * sizeof(Declaration), result, sizeof(Declaration));
		
		return &globalTable[globalTop++];//result;
	}
	else {
		Declaration *result = NULL;
		switch (declScope) {
			case LocalScope :
			    if (localTop == MaxLocals) {
			        sourceError("Too many locals", curTextPtr);
			        return NULL;
			    }
				result = &localDeclaration[localTop];
				result->data.value = 0;
				localTop++;
				break;
			case ArgumentScope :
				makeSpaceForArgument();
				result = &argumentList[argumentListTop];
				result->data.nextField = -1;
				argumentListTop++;
				break;
			case FieldScope :
			    if (fieldListTop == fieldListMax) {
					MemHandleUnlock(fieldListData);
					fieldListMax += 32;
					MemHandleResize(fieldListData, fieldListMax * sizeof(Declaration));
					fieldList = MemHandleLock(fieldListData);
			    }
				result = &fieldList[fieldListTop];
				result->data.nextField = fieldListTop;	// so that the previous field can link to this one 
				fieldListTop++;
				break;
			default :
				error("bad case 1");
				break;
		}
		result->name = startPtr;
		result->hash = hash;
		result->depth = -1;
		result->bitWidth = 0;
		result->length = length;
		result->typeIndex = startType;
		result->isTypedef = false;
		result->isStatic = false;
		result->typeNameSpace = isTypeName;
		if (declScope == LocalScope) {
			result->nextDecl = gLocalHashTable[hash];
			gLocalHashTable[hash] = result;
		}
		else
			result->nextDecl = NULL;
		return result;
	}
}

int operandOp(int op)
{
	return op - (AndEqual - And);
/*
	switch (op) {
		case AndEqual : return And;
		case OrEqual : return Or;
		case XorEqual : return Xor;
		case MultiplyEqual : return Multiply;
		case DivideEqual : return Divide;
		case RemainderEqual : return Remainder;
		case MinusEqual : return Minus;
		case PlusEqual : return Plus;
		case LeftShiftEqual : return LeftShift;
		case RightShiftEqual : return RightShift;
		default :
			error("bad case 2");
			return -1;
	}
*/
}

void constantFold(int op, Node *operand1, Node *operand2)
{
	if (op == Plus) {
		if (operand1->typeIndex == DoubleTypeIndex)
			if (operand2->typeIndex == DoubleTypeIndex)
				operand1->data.dValue += operand2->data.dValue;
			else
				operand1->data.dValue += operand2->value;
		else
			if (operand2->typeIndex == DoubleTypeIndex) {
				operand1->data.dValue = operand1->value + operand2->data.dValue;
				operand1->typeIndex = DoubleTypeIndex;
			}
			else
				operand1->value += operand2->value;
	}
	else
		if (operand1->typeIndex == DoubleTypeIndex)
			if (operand2->typeIndex == DoubleTypeIndex)
				operand1->data.dValue -= operand2->data.dValue;
			else
				operand1->data.dValue -= operand2->value;
		else
			if (operand2->typeIndex == DoubleTypeIndex) {
				operand1->data.dValue = operand1->value - operand2->data.dValue;
				operand1->typeIndex = DoubleTypeIndex;
			}
			else
				operand1->value -= operand2->value;
}

Boolean doOperator()
{
	int op = operatorStack[--topOfOperatorStack].op;
	Node *node     = NULL;
	Node *operand1 = NULL;
	Node *operand2 = NULL;
	Node *temp     = NULL;
	Declaration *f;
	int lhsType;

	switch (op) {
		default :
			error("bad case 4");
			break;
		case Sizeof :
			operand1 = operandStack[topOfOperandStack - 1];
			operand1 = resolveIdentifier(operand1);
			if (operand1 == NULL) return false;
			operandStack[topOfOperandStack - 1] = addImmediateNode(getTypeSize(operand1->typeIndex), -1);
			break;
		case PreDecrement :
		case PreIncrement :
		case PostIncrement :
		case PostDecrement : {
				if (topOfOperandStack < 1) {
					sourceError("Illegal expression construct", operatorStack[topOfOperatorStack].position);
					return false;
				}
				operand1 = operandStack[topOfOperandStack - 1];
				operand1 = resolveIdentifier(operand1);
				if (operand1 == NULL) return false;
				if ((operand1->op != Decl)
						&& (operand1->op != Arrow)
						&& (operand1->op != LeftBracket)) {
					sourceError("l-value required for prefix/postfix operator", operatorStack[topOfOperatorStack].position);
					return false;
				}
				if (isStruct(operand1->typeIndex)) {
					sourceError("Scalar type required for prefix/postfix operator", operatorStack[topOfOperatorStack].position);
					return false;
				}
				node = addNode(op, operand1->typeIndex, operand1, NULL);
				if (node == NULL) return false;
				operandStack[topOfOperandStack - 1] = node;
			}
			break;
		case Comma : {
				if (topOfOperandStack < 2) {
					sourceError("Illegal expression construct", operatorStack[topOfOperatorStack].position);
					return false;
				}
				operand1 = operandStack[topOfOperandStack - 2];
				operand1 = resolveIdentifier(operand1);
				if (operand1 == NULL) return false;
				operand2 = operandStack[topOfOperandStack - 1];
				operand2 = resolveIdentifier(operand2);
				if (operand2 == NULL) return false;
				node = addNode(Comma, operand2->typeIndex, operand1, operand2);
				operandStack[topOfOperandStack - 2] = node;
				topOfOperandStack--;
			}
			break;
		case Question : {
				if (topOfOperandStack < 2) {
					sourceError("Illegal expression construct", operatorStack[topOfOperatorStack].position);
					return false;
				}
				operand1 = operandStack[topOfOperandStack - 2];
				operand1 = resolveIdentifier(operand1);
				if (operand1 == NULL) return false;
				operand2 = operandStack[topOfOperandStack - 1];
				operand2 = resolveIdentifier(operand2);
				if (operand2 == NULL) return false;
				if (isStruct(operand1->typeIndex)) {
					sourceError("Scalar type required for conditional operator", operatorStack[topOfOperatorStack].position);
					return false;
				}
				node = addNode(Question, operand2->typeIndex, operand1, operand2);
				operandStack[topOfOperandStack - 2] = node;
				topOfOperandStack--;
			}
			break;
		case Colon : {
				if (topOfOperandStack < 2) {
					sourceError("Illegal expression construct", operatorStack[topOfOperatorStack].position);
					return false;
				}
				operand1 = operandStack[topOfOperandStack - 2];
				operand1 = resolveIdentifier(operand1);
				if (operand1 == NULL) return false;
				operand2 = operandStack[topOfOperandStack - 1];
				operand2 = resolveIdentifier(operand2);
				if (operand2 == NULL) return false;
				node = addNode(Colon, operand1->typeIndex, operand1, operand2);
				if (!matchTypes(operand1->typeIndex, operand2->typeIndex)) {
					if (isPointer(operand1->typeIndex) 
								&& (operand2->reg == Immediate)	
								&& (operand2->value == 0))
						operand2->typeIndex = operand1->typeIndex;
					else 
						if (isPointer(operand2->typeIndex) 
								&& (operand1->reg == Immediate)	
								&& (operand1->value == 0))
							operand1->typeIndex = operand2->typeIndex;
						else {
							sourceError("mis-matching types for conditional expression", operatorStack[topOfOperatorStack].position);
							return false;
						}
				}
				else
					if (!isStruct(operand1->typeIndex) && !isPointer(operand1->typeIndex)) {
						if (!usualBinaryConvert(node))
							return false;
					}
				node->typeIndex = node->data.children.node1->typeIndex;	// Not necessarily the same as operand1->typeIndex !
				operandStack[topOfOperandStack - 2] = node;
				topOfOperandStack--;
			}
			break;
		case Typedef : {
				if (topOfOperandStack < 2) {
					sourceError("Illegal expression construct", operatorStack[topOfOperatorStack].position);
					return false;
				}
				operand1 = operandStack[topOfOperandStack - 2];		// don't need to resolve, as it's just the type
				operand2 = operandStack[topOfOperandStack - 1];
				operand2 = resolveIdentifier(operand2);
				if (operand2 == NULL) return false;
				if (isStruct(operand1->typeIndex) || isStruct(operand2->typeIndex)) {
					sourceError("Cast to/from struct type not permitted", operatorStack[topOfOperatorStack].position);
					return false;
				}
				if (isEnum(operand1->typeIndex))
					operand1->typeIndex = getBaseType(operand1->typeIndex);
				if (isEnum(operand2->typeIndex))
					operand2->typeIndex = getBaseType(operand2->typeIndex);
// i don't know why these lines got commented out, though they are unncessary now since
// pointer to long is handled by emitConvert					
//				if (isPointer(operand2->typeIndex))
//					operand2->typeIndex = UnsignedLongTypeIndex;
				if (isPointer(operand1->typeIndex)) {
					if (isIntegral(operand2->typeIndex)) {
					
						if ((operand2->op == Decl) && (operand2->reg == Immediate)) {
							operand2->typeIndex = operand1->typeIndex;
							operandStack[topOfOperandStack - 2] = operand2;
							topOfOperandStack--;
						}
						else {
							node = addNode(Convert, UnsignedLongTypeIndex, operand2, NULL);
							node->typeIndex = operand1->typeIndex;
							node = addNode(Convert, operand1->typeIndex, operand2, NULL);
							operandStack[topOfOperandStack - 2] = node;
							topOfOperandStack--;
						}
					}
					else {
						if (!isPointer(operand2->typeIndex)) {
							sourceError("Illegal cast to pointer", operatorStack[topOfOperatorStack].position);
							return false;
						}
						operand2->typeIndex = operand1->typeIndex;
						operandStack[topOfOperandStack - 2] = operand2;
						topOfOperandStack--;
					}
				}
				else {
					node = addNode(Convert, operand1->typeIndex, operand2, NULL);
					operandStack[topOfOperandStack - 2] = node;
					topOfOperandStack--;
				}
			}
			break;
		case Not : {
				if (topOfOperandStack < 1) {
					sourceError("Illegal expression construct", operatorStack[topOfOperatorStack].position);
					return false;
				}
				operand1 = operandStack[topOfOperandStack - 1];
				operand1 = resolveIdentifier(operand1);
				if (operand1 == NULL) return false;
				if (isEnum(operand1->typeIndex))
					operand1->typeIndex = getBaseType(operand1->typeIndex);
				if (!isPointer(operand1->typeIndex)
						&& !isIntegral(operand1->typeIndex)
							 && (operand1->typeIndex != DoubleTypeIndex)
								  && (operand1->typeIndex != FloatTypeIndex)) {
						sourceError("Scalar type required", operatorStack[topOfOperatorStack].position);
						return false;
				}
				node = addNode(op, IntTypeIndex, operand1, NULL);
				if (node == NULL) return false;
				node = usualUnaryConvert(node);
				if (node == NULL) return false;
				operandStack[topOfOperandStack - 1] = node;
			}
			break;
		case UnaryMinus : {
				if (topOfOperandStack < 1) {
					sourceError("Illegal expression construct", operatorStack[topOfOperatorStack].position);
					return false;
				}
				operand1 = operandStack[topOfOperandStack - 1];
				operand1 = resolveIdentifier(operand1);
				if (operand1 == NULL) return false;
				if ((operand1->op == Decl) && (operand1->reg == Immediate)) {
					if (operand1->typeIndex == DoubleTypeIndex)
						operand1->data.dValue = - operand1->data.dValue;
					else
						operand1->value = - operand1->value;
					operandStack[topOfOperandStack - 1] = operand1;
				}
				else {
					node = addNode(op, operand1->typeIndex, operand1, NULL);
					if (node == NULL) return false;
					node = usualUnaryConvert(node);
					if (node == NULL) return false;
					operandStack[topOfOperandStack - 1] = node;
				}
			}
			break;
		case Tilde : {
				if (topOfOperandStack < 1) {
					sourceError("Illegal expression construct", operatorStack[topOfOperatorStack].position);
					return false;
				}
				operand1 = operandStack[topOfOperandStack - 1];
				operand1 = resolveIdentifier(operand1);
				if (operand1 == NULL) return false;
				if (isEnum(operand1->typeIndex))
					operand1->typeIndex = getBaseType(operand1->typeIndex);
				if (!isIntegral(operand1->typeIndex)) {
					sourceError("Integral type required", operatorStack[topOfOperatorStack].position);
					return false;
				}
				if ((operand1->op == Decl) && (operand1->reg == Immediate)) {
					operand1->value = ~ operand1->value;
					operandStack[topOfOperandStack - 1] = operand1;
				}
				else {
					node = addNode(op, operand1->typeIndex, operand1, NULL);
					if (node == NULL) return false;
					node = usualUnaryConvert(node);
					if (node == NULL) return false;
					operandStack[topOfOperandStack - 1] = node;
				}
			}
			break;
		case AddressOf : {
				if (topOfOperandStack < 1) {
					sourceError("Illegal expression construct", operatorStack[topOfOperatorStack].position);
					return false;
				}
				operand1 = operandStack[topOfOperandStack - 1];
				operand1 = resolveIdentifier(operand1);
				if (operand1 == NULL) return false;
				if (operand1->op == Decl) {
					if ((operand1->reg != A5) && (operand1->reg != A6)) {
						sourceError("not an l-value for address of", operatorStack[topOfOperatorStack].position);
						return false;
					}
					if ((operand1->reg == A6) && (operand1->data.decl->data.value > 0)) {
						sourceError("Can't take address of register variable", operatorStack[topOfOperatorStack].position);
						return false;
					}
					
					node = addNode(AddressOf, newPointerType(operand1->typeIndex), operand1, NULL);
				}
				else {
					if ((operand1->op == AddressOf)
							&& isFunction(getBaseType(operand1->typeIndex))) {
						node = operand1;	// all done, the function's address is already provided
					}
					else {				
						if ((operand1->op != Arrow) 
								&& (operand1->op != LeftBracket)) {
							sourceError("not an l-value for address of", operatorStack[topOfOperatorStack].position);
							return false;
						}
						node = addNode(AddressOf, newPointerType(operand1->typeIndex), operand1, NULL);
					}
				}
				if (node == NULL) return false;
				operandStack[topOfOperandStack - 1] = node;
			}
			break;
		case Period : {
				if (topOfOperandStack < 2) {
					sourceError("Illegal expression construct", operatorStack[topOfOperatorStack].position);
					return false;
				}
				operand1 = operandStack[topOfOperandStack - 2];
				operand2 = operandStack[topOfOperandStack - 1];
				operand1 = resolveIdentifier(operand1);
				if (operand1 == NULL) return false;
				lhsType = operand1->typeIndex;
				if (!isStruct(lhsType)) { 
					sourceError("expected struct type for '.'", operatorStack[topOfOperatorStack].position);
					return false;
				}
				f = findField(lhsType, operand2->data.identifier.name, operand2->data.identifier.length);
				if (f == NULL) {
					sourceError("unknown field in struct", operatorStack[topOfOperatorStack].position);
					return false;
				}
				switch (operand1->op) {
					case Arrow :
						operand1->offset += f->depth >> 3;
						operand1->typeIndex = f->typeIndex;
						operand1->data.children.node2 = (Node *)f;	// in case this new field is a bitfield
						topOfOperandStack--;
						break;						
					case LeftBracket :
					case Decl :	{		
							node = addNode(AddressOf, newPointerType(lhsType), operand1, NULL);
							node = addNode(Arrow, f->typeIndex, node, (Node *)f);
							node->offset = f->depth >> 3;
							operandStack[topOfOperandStack - 2] = node;
							topOfOperandStack--;
						}
						break;
					default :
						error("bad case 5");
						break;
				}
//				operand1 = operandStack[topOfOperandStack - 1];
			}
			break;
		case Indirect : {
				if (topOfOperandStack < 1) {
					sourceError("Illegal expression construct", operatorStack[topOfOperatorStack].position);
					return false;
				}
				operand1 = operandStack[topOfOperandStack - 1];
				operand1 = resolveIdentifier(operand1);
				if (operand1 == NULL) return false;
				operand1 = usualUnaryConvert(operand1);
				if (operand1 == NULL) return false;
				lhsType = operand1->typeIndex;
				if (!isPointer(lhsType)) {
					sourceError("expected pointer type for '*'", operatorStack[topOfOperatorStack].position);
					return false;
				}
				// if it's a pointer to a function, ignore the indirect
				if (!isFunction(getBaseType(lhsType))) {
					node = addNode(Arrow, getBaseType(lhsType), operand1, NULL);
					node->offset = 0;
					operandStack[topOfOperandStack - 1] = node;
				}
//				operand1 = operandStack[topOfOperandStack - 1];
			}
			break;
		case Arrow : {
				if (topOfOperandStack < 2) {
					sourceError("Illegal expression construct", operatorStack[topOfOperatorStack].position);
					return false;
				}
				operand1 = operandStack[topOfOperandStack - 2];
				operand2 = operandStack[topOfOperandStack - 1];
				operand1 = resolveIdentifier(operand1);
				if (operand1 == NULL) return false;
				lhsType = operand1->typeIndex;
				if (!isPointer(lhsType)) {
					sourceError("expected pointer type for '->'", operatorStack[topOfOperatorStack].position);
					return false;
				}
				lhsType = getBaseType(lhsType);
				if (!isStruct(lhsType)) {
					sourceError("expected struct type for '->'", operatorStack[topOfOperatorStack].position);
					return false;
				}
				f = findField(lhsType, operand2->data.identifier.name, operand2->data.identifier.length);
				if (f == NULL) {
					sourceError("unknown field in struct", operatorStack[topOfOperatorStack].position);
					return false;
				}
				node = addNode(Arrow, f->typeIndex, operand1, (Node *)f);
				node->offset = f->depth >> 3;
				operandStack[topOfOperandStack - 2] = node;
				topOfOperandStack--;
//				operand1 = operandStack[topOfOperandStack - 1];
			}
			break;
		case LeftBracket : {
				if (topOfOperandStack < 2) {
					sourceError("Illegal expression construct", operatorStack[topOfOperatorStack].position);
					return false;
				}
				operand1 = operandStack[topOfOperandStack - 2];	// the base
				operand1 = resolveIdentifier(operand1);
				if (operand1 == NULL) return false;
				operand1 = usualUnaryConvert(operand1);
				if (operand1 == NULL) return false;
				
				operand2 = operandStack[topOfOperandStack - 1];	// the index
				operand2 = resolveIdentifier(operand2);
				if (operand2 == NULL) return false;
				operand2 = usualUnaryConvert(operand2);
				if (operand2 == NULL) return false;
				if (!isPointer(operand1->typeIndex)) {
					if (isArray(operand1->typeIndex)) {
						operand1->typeIndex = newPointerType(getBaseType(operand1->typeIndex));
					}
					else {
						sourceError("Must have pointer type for index base", operatorStack[topOfOperatorStack].position);
						return false;
					}
				}
				if (isPointer(operand2->typeIndex) || isStruct(operand2->typeIndex)) {
					sourceError("Illegal type for index expression", operatorStack[topOfOperatorStack].position);
					return false;
				}
				if ((operand2->typeIndex != LongTypeIndex)
						&& (operand2->typeIndex != UnsignedLongTypeIndex)) {
					operand2 = addConvert(operand2, IntTypeIndex); 
					if (operand2 == NULL) return NULL;
				}
				if (getTypeSize(getBaseType(operand1->typeIndex)) > 1) {
					// force constant up to index expr. type
					node = addImmediateNode(getTypeSize(getBaseType(operand1->typeIndex)), operand2->typeIndex);
					operand2 = addNode(Multiply, operand2->typeIndex, operand2, node);
				}
				node = addNode(LeftBracket, getBaseType(operand1->typeIndex), operand1, operand2);
				operandStack[topOfOperandStack - 2] = node;
				topOfOperandStack--;
//				operand1 = operandStack[topOfOperandStack - 1];
			}
			break;
		case LeftParen : {
				int i;
				int argCount;
				int argBase;
				if (topOfOperandStack < 1) {
					sourceError("Illegal expression construct", operatorStack[topOfOperatorStack].position);
					return false;
				}
				argCount = operandStack[--topOfOperandStack]->value;
				argBase = topOfOperandStack - argCount;
				if (argBase <= 0) {
					sourceError("Wrong number of arguments for call", operatorStack[topOfOperatorStack].position);
					return false;
				}
				operand1 = operandStack[argBase - 1];		 // the callTarget
				operand1 = resolveIdentifier(operand1);
				if (operand1 == NULL) return false;
				if (!isPointer(operand1->typeIndex)
						|| !isFunction(getBaseType(operand1->typeIndex))) {
					sourceError("Expected function for call", operatorStack[topOfOperatorStack].position);
					return false;
				}
				lhsType = getBaseType(operand1->typeIndex);
				if (getTypeSize(lhsType) != argCount) {
					if ((getTypeSize(lhsType) & 0x8000) == 0) {	// if not ... function, then error
						sourceError("Wrong number of arguments for call", operatorStack[topOfOperatorStack].position);
						return false;
					}
				}
				operand2 = NULL;			// the argTree
				i = getBaseType(lhsType);	// the return type
				if (isStruct(i) || (i == DoubleTypeIndex) || (i == FloatTypeIndex)) {
					temp = newTemp(i, gStatementIndex);
					if (temp == NULL) return false;
					node = addNode(AddressOf, newPointerType(i), temp, NULL);
					if (node == NULL) return false;
					operand2 = addNode(Comma, -1, operand2, node);
					if (operand2 == NULL) return false;
				
				}
/*
				if (isStruct(getBaseType(lhsType))) {				
					temp = newTemp(getBaseType(lhsType));
					if (temp == NULL) return false;
					node = addNode(AddressOf, newPointerType(getBaseType(lhsType)), temp, NULL);
					if (node == NULL) return false;
					operand2 = addNode(Comma, -1, operand2, node);
					if (operand2 == NULL) return false;
				}
				else {
					if (getBaseType(lhsType) == DoubleTypeIndex) {
						temp = newTemp(DoubleTypeIndex);
						if (temp == NULL) return false;
						node = addNode(AddressOf, newPointerType(DoubleTypeIndex), temp, NULL);
						if (node == NULL) return false;
						operand2 = addNode(Comma, -1, operand2, node);
						if (operand2 == NULL) return false;
					}
					else
						if (getBaseType(lhsType) == FloatTypeIndex) {
							temp = newTemp(FloatTypeIndex);
							if (temp == NULL) return false;
							node = addNode(AddressOf, newPointerType(FloatTypeIndex), temp, NULL);
							if (node == NULL) return false;
							operand2 = addNode(Comma, -1, operand2, node);
							if (operand2 == NULL) return false;
						}
				}
*/				
				for (i = 0; i < argCount; i++) {
					node = operandStack[argBase + i];
					node = resolveIdentifier(node);
					if (node == NULL) return false;
					node = usualUnaryConvert(node);
					if (node == NULL) return false;
					// ... arguments detected
					if ((i >= (getTypeSize(lhsType) & 0x7FFF)) && ((getTypeSize(lhsType) & 0x8000) != 0)) {
						operand2 = addNode(Comma, -1, operand2, node);
					}
					else {
						int formalType = getArgTypeIndex(lhsType, i);
						if (!matchTypes(node->typeIndex, formalType)) {
							if (isPointer(formalType) 
										&& (node->reg == Immediate)	
										&& (node->value == 0)) {
								node->typeIndex = UnsignedLongTypeIndex;
							}
							else {
								if (isPointer(node->typeIndex)
										&& isArray(formalType)
										&& matchTypes(getBaseType(node->typeIndex), getBaseType(formalType))) {
									node->typeIndex = UnsignedLongTypeIndex;
								}
								else {
									sourceError("Illegal type for argument", operatorStack[topOfOperatorStack].position);
									return false;
								}
							}
						}
						if (!isPointer(formalType) 
								&& !isArray(formalType)) {
							if (isStruct(formalType)) {
								// build a pointer to the struct arg, but pass the type as struct
								// so that the codegen knows to copy the contents onto the stack
								node = addNode(AddressOf, node->typeIndex, node, NULL);
							}
							else 
								node = addConvert(node, formalType);
							if (node == NULL) return false;
						}
						operand2 = addNode(Comma, -1, operand2, node);
						if (operand2 == NULL) return false;
					}
				}
				topOfOperandStack = argBase;
				node = addNode(LeftParen, getBaseType(lhsType), operand1, operand2);
				if (node == NULL) return false;
				if (isStruct(getBaseType(lhsType))
						 || (getBaseType(lhsType) == DoubleTypeIndex) 
						 || (getBaseType(lhsType) == FloatTypeIndex)) {
					node = addNode(Comma, getBaseType(lhsType), node, temp);
					if (node == NULL) return false;
				}
				operandStack[topOfOperandStack - 1] = node;
			}
			break;
		case LeftShiftEqual :
		case RightShiftEqual :
		case AndEqual :
		case OrEqual :
		case XorEqual :
		case MultiplyEqual :
		case DivideEqual :
		case RemainderEqual :
		case MinusEqual :
		case PlusEqual : {
				if (topOfOperandStack < 2) {
					sourceError("Illegal expression construct", operatorStack[topOfOperatorStack].position);
					return false;
				}
				operand1 = operandStack[topOfOperandStack - 2];
				operand2 = operandStack[topOfOperandStack - 1];
				if (operand1 == NULL) return false;
				operand1 = resolveIdentifier(operand1);
				if ((operand1->op != Decl)
						&& (operand1->op != Arrow)
						&& (operand1->op != LeftBracket)) {
					sourceError("l-value required", operatorStack[topOfOperatorStack].position);
					return false;
				}
				operand2 = resolveIdentifier(operand2);
				if (operand2 == NULL) return false;
				if (isStruct(operand1->typeIndex) || isStruct(operand2->typeIndex)) {
					sourceError("Can't perform operations on structs", operatorStack[topOfOperatorStack].position);
					return NULL;
				}
				if ((((op != MultiplyEqual) && (op != DivideEqual) && (op != RemainderEqual))
							|| (getTypeSize(operand1->typeIndex) > 1))		// can't do in-place byte multiply
						&& !((operand1->op == Arrow) 
									&& (operand1->data.children.node2 != NULL)
									&& ((((Declaration *)(operand1->data.children.node2))->bitWidth & 0x7) != 0))
						&& !hasSideEffects(operand1)
						&& isIntegral(operand1->typeIndex)
						&& isIntegral(operand2->typeIndex)
						&& (getTypeSize(operand1->typeIndex) <= getTypeSize(operand2->typeIndex))) {
					operand2 = addConvert(operand2, operand1->typeIndex);
					if (operand2 == NULL) return NULL;
					node = addNode(op, operand1->typeIndex, operand1, operand2);
					operandStack[topOfOperandStack - 2] = node;
					topOfOperandStack--;
				}
				else {
				// for lhs <op>= rhs, (we can only reference the lhs tree once) construct :
				//		temp = &lhs
				//      *temp = *temp <op> rhs
				//      
				//	which we build via a comma operator, so from last to first.
				//
				//	Bitfields mess this up but...
				//		lhs.bf <op>= rhs	 will already been cg'd as (&lhs)->bf, so we can reuse lhs safely
				// -->
				//		(&lhs)->bf = (&lhs)->bf <op> rhs
				//
				//	[Note - we could do the same for the non-bitfield case when it's safe to reuse lhs]
				//
					if ((operand1->op == Arrow) 
								&& (operand1->data.children.node2 != NULL)
								&& ((((Declaration *)(operand1->data.children.node2))->bitWidth & 0x7) != 0)) {
						node = addClone(operand1);	// have to use a clone of operand1 since the operand1 node will get mutated
													// during codegen and we need to preserve the original addressing info.
						operand2 = constructOp(operandOp(op), operand1, operand2);
						if (operand2 == NULL) return false;										
						if (!isPointer(operand1->typeIndex)) {
							operand2 = addConvert(operand2, operand1->typeIndex);
							if (operand2 == NULL) return NULL;
						}
						if (!matchTypes(operand1->typeIndex, operand2->typeIndex)) {
							sourceError("Illegal types for assignment", operatorStack[topOfOperatorStack].position);
							return false;
						}
						// now operand2 is (&lhs)->bf <op> rhs

						node = addNode(Equal, operand1->typeIndex, node, operand2);
						if (node == NULL) return false;
						operandStack[topOfOperandStack - 2] = node;
						topOfOperandStack--;
					}
					else {
						lhsType = newPointerType(operand1->typeIndex);
						temp = newTemp(lhsType, gStatementIndex);
						if (temp == NULL) return false;
						node = addNode(Arrow, operand1->typeIndex, temp, NULL);
						if (node == NULL) return false;
						node->offset = 0;					// node is *temp
						
						operand2 = constructOp(operandOp(op), node, operand2);
						if (operand2 == NULL) return false;										
						if (!isPointer(operand1->typeIndex)) {
							operand2 = addConvert(operand2, operand1->typeIndex);
							if (operand2 == NULL) return NULL;
						}
						if (!matchTypes(operand1->typeIndex, operand2->typeIndex)) {
							sourceError("Illegal types for assignment", operatorStack[topOfOperatorStack].position);
							return false;
						}
						// now operand2 is *temp <op> rhs
						
						node = addNode(Arrow, operand1->typeIndex, addClone(temp), NULL);
						if (node == NULL) return false;
						node->offset = 0;					// node is *temp
						operand2 = addNode(Equal, operand1->typeIndex, node, operand2); // operand2 is *temp = *temp <op> rhs
						node = addNode(AddressOf, lhsType, operand1, NULL);  // node is &lhs
						if (node == NULL) return false;
						node = addNode(Equal, lhsType, addClone(temp), node);
						if (node == NULL) return false;					// node is temp = &lhs
						
						node = addNode(Comma, operand1->typeIndex, node, operand2);
						if (node == NULL) return false;
						operandStack[topOfOperandStack - 2] = node;
						topOfOperandStack--;
					}
				}
			}
			break;
		case Equal : {
				if (topOfOperandStack < 2) {
					sourceError("Illegal expression construct", operatorStack[topOfOperatorStack].position);
					return false;
				}
				operand1 = operandStack[topOfOperandStack - 2];
				operand2 = operandStack[topOfOperandStack - 1];
				operand1 = resolveIdentifier(operand1);
				if (operand1 == NULL) return false;
				if ((operand1->op != Decl)
						&& (operand1->op != Arrow)
						&& (operand1->op != LeftBracket)) {
					sourceError("l-value required", operatorStack[topOfOperatorStack].position);
					return false;
				}
				operand2 = resolveIdentifier(operand2);
				if (operand2 == NULL) return false;

				if (isEnum(operand1->typeIndex))
					operand1->typeIndex = getBaseType(operand1->typeIndex);
				if (isEnum(operand2->typeIndex))
					operand2->typeIndex = getBaseType(operand2->typeIndex);

				if (isArray(operand2->typeIndex)) {
					operand2 = addNode(AddressOf, newPointerType(getBaseType(operand2->typeIndex)), operand2, NULL);
				}
				if (!matchTypes(operand1->typeIndex, operand2->typeIndex)) {
					if (isPointer(operand1->typeIndex)
							&& (operand2->typeIndex == IntTypeIndex)
							&& (operand2->op == Decl)
							&& (operand2->reg == Immediate)
							&& (operand2->value == 0)) {
						node = addNode(op, UnsignedLongTypeIndex, operand1, operand2);
						operand2->typeIndex = UnsignedLongTypeIndex;
					}
					else
						if (isPointer(operand2->typeIndex)
								&& (operand1->typeIndex == IntTypeIndex)
								&& (operand1->op == Decl)
								&& (operand1->reg == Immediate)
								&& (operand1->value == 0)) {
							node = addNode(op, UnsignedLongTypeIndex, operand1, operand2);
							operand1->typeIndex = UnsignedLongTypeIndex;
						}
						else {				
							sourceError("Illegal types for assignment", operatorStack[topOfOperatorStack].position);
							return false;
						}
				}
				if (isStruct(operand1->typeIndex)) {
					operand1 = addNode(AddressOf, operand1->typeIndex, operand1, NULL);
					operand2 = addNode(AddressOf, operand2->typeIndex, operand2, NULL);
					node = addNode(op, operand1->typeIndex, operand1, operand2);
					node->offset = getTypeSize(operand1->typeIndex);
				}
				else {
					if (!isPointer(operand1->typeIndex)) {
						operand2 = addConvert(operand2, operand1->typeIndex);
						if (operand2 == NULL) return NULL;
					}
					node = addNode(op, operand1->typeIndex, operand1, operand2);
				}
				operandStack[topOfOperandStack - 2] = node;
				topOfOperandStack--;
			}
			break;
		case AndAnd :
		case OrOr : {
				if (topOfOperandStack < 2) {
					sourceError("Illegal expression construct", operatorStack[topOfOperatorStack].position);
					return false;
				}
				operand1 = operandStack[topOfOperandStack - 2];
				operand2 = operandStack[topOfOperandStack - 1];
				operand1 = resolveIdentifier(operand1);
				if (operand1 == NULL) return false;
				operand2 = resolveIdentifier(operand2);
				if (operand2 == NULL) return false;
				node = addNode(op, operand1->typeIndex, operand1, operand2);
				operandStack[topOfOperandStack - 2] = node;
				topOfOperandStack--;
			}
			break;
		case Plus :
		case Minus : {
				if (topOfOperandStack < 2) {
					sourceError("Illegal expression construct", operatorStack[topOfOperatorStack].position);
					return false;
				}
				operand1 = operandStack[topOfOperandStack - 2];
				operand2 = operandStack[topOfOperandStack - 1];
				operand1 = resolveIdentifier(operand1);
				if (operand1 == NULL) return false;
				operand2 = resolveIdentifier(operand2);
				if (operand2 == NULL) return false;
				if ((operand1->op == Decl) && (operand1->reg == Immediate)
						&& (operand2->op == Decl) && (operand2->reg == Immediate)) {
					constantFold(op, operand1, operand2);		
					operandStack[topOfOperandStack - 2] = operand1;
					topOfOperandStack--;
				}
				else {
					node = constructOp(op, operand1, operand2);
					if (node == NULL) return false;					
					operandStack[topOfOperandStack - 2] = node;
					topOfOperandStack--;
				}
			}
			break;		
		case LeftShift :
		case RightShift :
		case And :
		case Or : 
		case Xor :
				if (topOfOperandStack < 2) {
					sourceError("Illegal expression construct", operatorStack[topOfOperatorStack].position);
					return false;
				}
				operand1 = operandStack[topOfOperandStack - 2];
				operand1 = resolveIdentifier(operand1);
				if (operand1 == NULL) return false;
				operand2 = operandStack[topOfOperandStack - 1];
				operand2 = resolveIdentifier(operand2);
				if (operand2 == NULL) return false;
				if (isEnum(operand1->typeIndex))
					operand1->typeIndex = getBaseType(operand1->typeIndex);
				if (isEnum(operand2->typeIndex))
					operand2->typeIndex = getBaseType(operand2->typeIndex);
				if (!isIntegral(operand1->typeIndex) || !isIntegral(operand2->typeIndex)) {
					sourceError("Integral types required", operatorStack[topOfOperatorStack].position);
					return false;
				}
				if ((operand1->op == Decl) && (operand1->reg == Immediate)
						&& (operand2->op == Decl) && (operand2->reg == Immediate)) {
					switch (op) {
						case LeftShift :
							operand1->value = operand1->value << operand2->value; break;
						case RightShift :
							operand1->value = operand1->value >> operand2->value; break;
						case And :
							operand1->value = operand1->value & operand2->value; break;
						case Or : 
							operand1->value = operand1->value | operand2->value; break;
						case Xor : 
							operand1->value = operand1->value ^ operand2->value; break;
					}
					operandStack[topOfOperandStack - 2] = operand1;
					topOfOperandStack--;
				}
				else {
					if ((op == LeftShift) || (op == RightShift)) {
						operand1 = usualUnaryConvert(operand1);
						if (operand1 == NULL) return false;
						operand2 = usualUnaryConvert(operand2);
						if (operand2 == NULL) return false;
						node = addNode(op, operand1->typeIndex, operand1, operand2);
						if (node == NULL) return false;
					}
					else {
						node = addNode(op, IntTypeIndex, operand1, operand2);
						if (node == NULL) return false;
						if (!usualBinaryConvert(node)) return false;
					}
					node->typeIndex = node->data.children.node1->typeIndex;	// Not necessarily the same as operand1->typeIndex !
					operandStack[topOfOperandStack - 2] = node;
					topOfOperandStack--;
				}
			break;

		case LessEqual :
		case Less :
		case GreaterEqual :
		case Greater :
		case NotEqual :
		case EqualEqual :
		case Multiply :
		case Divide :
		case Remainder : {
				if (topOfOperandStack < 2) {
					sourceError("Illegal expression construct", operatorStack[topOfOperatorStack].position);
					return false;
				}
				operand1 = operandStack[topOfOperandStack - 2];
				operand2 = operandStack[topOfOperandStack - 1];
				operand1 = resolveIdentifier(operand1);
				if (operand1 == NULL) return false;
				operand2 = resolveIdentifier(operand2);
				if (operand2 == NULL) return false;
				
				if (isEnum(operand1->typeIndex))
					operand1->typeIndex = getBaseType(operand1->typeIndex);
				if (isEnum(operand2->typeIndex))
					operand2->typeIndex = getBaseType(operand2->typeIndex);

				if (isStruct(operand1->typeIndex) || isStruct(operand2->typeIndex)) {
					sourceError("Illegal operator for struct operand", operatorStack[topOfOperatorStack].position);
					return false;
				}
				
				if (op == Remainder) {
					if (!isIntegral(operand1->typeIndex) && !isIntegral(operand2->typeIndex)) {
						sourceError("% operator only valid on integral types", operatorStack[topOfOperatorStack].position);
						return false;
					}
				}
				
				if ((operand1->op == Decl) && (operand1->reg == Immediate)
						&& (operand2->op == Decl) && (operand2->reg == Immediate)) {
					if (operand1->typeIndex == DoubleTypeIndex) {
						if (operand2->typeIndex != DoubleTypeIndex) {
							operand2->data.dValue = operand2->value;
							operand2->typeIndex = DoubleTypeIndex;
						}
					}
					else {
						if (operand2->typeIndex == DoubleTypeIndex) {
							if (operand1->typeIndex != DoubleTypeIndex) {
								operand1->data.dValue = operand1->value;
								operand1->typeIndex = DoubleTypeIndex;
							}
						}
					}
					if (operand1->typeIndex == DoubleTypeIndex) {
						switch (op) {
							case LessEqual :						
								operand1->value = operand1->data.dValue <= operand2->data.dValue;
								operand1->typeIndex = IntTypeIndex;
								break;
							case Less :
								operand1->value = operand1->data.dValue < operand2->data.dValue;
								operand1->typeIndex = IntTypeIndex;
								break;
							case GreaterEqual :
								operand1->value = operand1->data.dValue >= operand2->data.dValue;
								operand1->typeIndex = IntTypeIndex;
								break;
							case Greater :
								operand1->value = operand1->data.dValue > operand2->data.dValue;
								operand1->typeIndex = IntTypeIndex;
								break;
							case NotEqual :
								operand1->value = operand1->data.dValue != operand2->data.dValue;
								operand1->typeIndex = IntTypeIndex;
								break;
							case EqualEqual :
								operand1->value = operand1->data.dValue == operand2->data.dValue;
								operand1->typeIndex = IntTypeIndex;
								break;
							case Multiply :
								operand1->data.dValue = operand1->data.dValue * operand2->data.dValue; break;
							case Divide :
								operand1->data.dValue = operand1->data.dValue / operand2->data.dValue; break;
							case Remainder :
								sourceError("% illegal for floating-point", operatorStack[topOfOperatorStack].position);
								return false;
						}
					}
					else {
						switch (op) {
							case LessEqual :						
								operand1->value = operand1->value <= operand2->value; break;
							case Less :
								operand1->value = operand1->value < operand2->value; break;
							case GreaterEqual :
								operand1->value = operand1->value >= operand2->value; break;
							case Greater :
								operand1->value = operand1->value > operand2->value; break;
							case NotEqual :
								operand1->value = operand1->value != operand2->value; break;
							case EqualEqual :
								operand1->value = operand1->value == operand2->value; break;
							case Multiply :
								operand1->value = operand1->value * operand2->value; break;
							case Divide :
								operand1->value = operand1->value / operand2->value; break;
							case Remainder :
								operand1->value = operand1->value % operand2->value; break;
						}
					}
					operandStack[topOfOperandStack - 2] = operand1;
					topOfOperandStack--;
				}
				else {					
					node = addNode(op, IntTypeIndex, operand1, operand2);
					if (node == NULL) return false;
					
					if (isPointer(operand1->typeIndex) || isPointer(operand2->typeIndex)) {
						if ((op == Multiply) || (op == Divide) || (op == Remainder)) {
							sourceError("Invalid operator for pointer types", operatorStack[topOfOperatorStack].position);
							return false;
						}
						if (!matchTypes(operand1->typeIndex, operand2->typeIndex)) {
							if ((!(isPointer(operand1->typeIndex)
									&& (operand2->typeIndex == IntTypeIndex)
									&& (operand2->op == Decl)
									&& (operand2->reg == Immediate)
									&& (operand2->value == 0)) )
								&& (!(isPointer(operand2->typeIndex)
										&& (operand1->typeIndex == IntTypeIndex)
										&& (operand1->op == Decl)
										&& (operand1->reg == Immediate)
										&& (operand1->value == 0)) )) {
								sourceError("Invalid combination of pointer types", operatorStack[topOfOperatorStack].position);
								return false;
							}
						}
						node->typeIndex = IntTypeIndex;
						operand2->typeIndex = UnsignedLongTypeIndex;
						operand1->typeIndex = UnsignedLongTypeIndex;
					}
					else {
						if (!usualBinaryConvert(node))
							return false;
						if ((op == Multiply) || (op == Divide) || (op == Remainder))
							node->typeIndex = node->data.children.node1->typeIndex;	// Not necessarily the same as operand1->typeIndex !
						else
							node->typeIndex = IntTypeIndex;	// relationals always return int type
					}
					operandStack[topOfOperandStack - 2] = node;
					topOfOperandStack--;
				}
			}
			break;
	}
	return true;
}


CharPtr parseExpression(CharPtr curTextPtr)
{
	ParseState ps;
	Boolean inExpression = true;
	int operatorStackStart = topOfOperatorStack;
	CharPtr savePtr;
	int baseType;
	Node *node;
	Declaration *d;
	int nestedStackStart;

	while (inExpression) {
		Boolean waitingForBinaryOperator;
		Boolean inUnaryMode = true;
		while (inUnaryMode) {
			savePtr = curTextPtr;
			curTextPtr = getNextToken(curTextPtr, &ps);
			switch (ps.token) {
				case Not :
				case Tilde :
				case Minus :
				case Plus :
				case Multiply :
				case And :
					ps.token = unaryOperator[ps.token];
					if ((topOfOperatorStack > operatorStackStart)
							&& (operatorStack[topOfOperatorStack - 1].precedence > operatorPrecedence[ps.token])) {
						if (!doOperator()) return NULL;
					}
					operatorStack[topOfOperatorStack].op = ps.token;
					operatorStack[topOfOperatorStack].precedence = operatorPrecedence[ps.token];
					operatorStack[topOfOperatorStack].position = savePtr;
					topOfOperatorStack++;
					if (topOfOperatorStack == MaxStack) {
						sourceError("Operator stack overflow", savePtr);
						return NULL;
					}
					break;

				case Sizeof :
					if ((topOfOperatorStack > operatorStackStart)
							&& (operatorStack[topOfOperatorStack - 1].precedence > operatorPrecedence[ps.token])) {
						if (!doOperator()) return NULL;
					}
					savePtr = curTextPtr;
					curTextPtr = getNextToken(curTextPtr, &ps);
					if (ps.token == LeftParen) {
						baseType = -1;
						//node;
						d = NULL;
						savePtr = curTextPtr;
						curTextPtr = declaration(curTextPtr, &baseType, &d, LocalScope, true);
						if (curTextPtr == NULL) return NULL;
						if ((d == NULL) || (d->name != NULL)) {
							// wasn't a declaration, try getting an expression
							ps.token = Sizeof;
							operatorStack[topOfOperatorStack].op = ps.token;
							operatorStack[topOfOperatorStack].precedence = operatorPrecedence[ps.token];
							operatorStack[topOfOperatorStack].position = savePtr;
							topOfOperatorStack++;
							if (topOfOperatorStack == MaxStack) {
								sourceError("Operator stack overflow", savePtr);
								return NULL;
							}
					    	nestedStackStart = topOfOperatorStack;
							curTextPtr = parseExpression(curTextPtr);						
							if (curTextPtr == NULL) return NULL;
							while (topOfOperatorStack > nestedStackStart) {
								if (!doOperator())
									return NULL;
							}								
							savePtr = curTextPtr;
							curTextPtr = getNextToken(curTextPtr, &ps);
							if (ps.token != RightParen) {
								sourceError("Right paren expected after sizeof expression", savePtr);
								return NULL;
							}
							
//							sourceError("Type expected after sizeof", savePtr);
//							return NULL;
						}
						else {
							savePtr = curTextPtr;
							curTextPtr = getNextToken(curTextPtr, &ps);
							if (ps.token != RightParen) {
								sourceError("Right paren expected after sizeof type", savePtr);
								return NULL;
							}
							node = addImmediateNode(getTypeSize(d->typeIndex), -1);
							if (node == NULL) return NULL;					
							operandStack[topOfOperandStack] = node;
							topOfOperandStack++;
							if (topOfOperandStack == MaxStack) {
								sourceError("Operand stack overflow", savePtr);
								return NULL;
							}
						}
						inUnaryMode = false;
					}
					else {	// sizeof unary-expression here !
						pushBack(&ps);
						ps.token = Sizeof;
						operatorStack[topOfOperatorStack].op = ps.token;
						operatorStack[topOfOperatorStack].precedence = operatorPrecedence[ps.token];
						operatorStack[topOfOperatorStack].position = savePtr;
						topOfOperatorStack++;
						if (topOfOperatorStack == MaxStack) {
							sourceError("Operator stack overflow", savePtr);
							return NULL;
						}
					}
					break;					

				case LeftParen : {
						baseType = -1;
						d = NULL;
						savePtr = curTextPtr;						
						curTextPtr = declaration(curTextPtr, &baseType, &d, LocalScope, true);
						if (curTextPtr == NULL) return NULL;
						if ((d == NULL) || (d->name != NULL)) {
					    	nestedStackStart = topOfOperatorStack;
							curTextPtr = parseExpression(curTextPtr);						
							if (curTextPtr == NULL) return NULL;
							while (topOfOperatorStack > nestedStackStart) {
								if (!doOperator())
									return NULL;
							}								
							savePtr = curTextPtr;
							curTextPtr = getNextToken(curTextPtr, &ps);
							if (ps.token != RightParen) {
								sourceError("Right paren expected after parenthesized expression", savePtr);
								return NULL;
							}
							inUnaryMode = false;
						}
						else {
							savePtr = curTextPtr;
							curTextPtr = getNextToken(curTextPtr, &ps);
							if (ps.token != RightParen) {
								sourceError("Right paren expected after cast type", savePtr);
								return NULL;
							}
							if ((topOfOperatorStack > operatorStackStart)
									&& (operatorStack[topOfOperatorStack - 1].precedence > operatorPrecedence[Typedef])) {
								if (!doOperator()) return NULL;
							}
							operandStack[topOfOperandStack] = addNode(Typedef, d->typeIndex, NULL, NULL);
							topOfOperandStack++;
							if (topOfOperandStack == MaxStack) {
								sourceError("Operand stack overflow", savePtr);
								return NULL;
							}
							operatorStack[topOfOperatorStack].op = Typedef;
							operatorStack[topOfOperatorStack].precedence = operatorPrecedence[Typedef];
							operatorStack[topOfOperatorStack].position = savePtr;
							topOfOperatorStack++;
							if (topOfOperatorStack == MaxStack) {
								sourceError("Operator stack overflow", savePtr);
								return NULL;
							}
						}						
					}				
					break;
					
				case PostIncrement :
				case PostDecrement :
					ps.token = unaryOperator[ps.token];	// convert to Pre.. version
					if ((topOfOperatorStack > operatorStackStart)
							&& (operatorStack[topOfOperatorStack - 1].precedence > operatorPrecedence[ps.token])) {
						if (!doOperator()) return NULL;
					}
					operatorStack[topOfOperatorStack].op = ps.token;
					operatorStack[topOfOperatorStack].precedence = operatorPrecedence[ps.token];
					operatorStack[topOfOperatorStack].position = savePtr;
					topOfOperatorStack++;
					if (topOfOperatorStack == MaxStack) {
						sourceError("Operator stack overflow", savePtr);
						return NULL;
					}
					break;
				
				case Digit : {
						node = addImmediateNode(ps.value, -1);
						if (node == NULL) return NULL;					
						operandStack[topOfOperandStack] = node;
						topOfOperandStack++;
						if (topOfOperandStack == MaxStack) {
							sourceError("Operand stack overflow", savePtr);
							return NULL;
						}
						inUnaryMode = false;
					}
					break;
				
				case FPDigit : {
						node = addDoubleImmediateNode(gFPDigit);
						if (node == NULL) return NULL;					
						operandStack[topOfOperandStack] = node;
						topOfOperandStack++;
						if (topOfOperandStack == MaxStack) {
							sourceError("Operand stack overflow", savePtr);
							return NULL;
						}
						inUnaryMode = false;
					}
					break;
				
				case DoubleQuote : {
						node = addStringConstantNode(ps.start, ps.value);
						if (node == NULL) return NULL;					
						node = addNode(AddressOf, CharPtrTypeIndex, node, NULL);
						if (node == NULL) return NULL;					
						operandStack[topOfOperandStack] = node;
						topOfOperandStack++;
						if (topOfOperandStack == MaxStack) {
							sourceError("Operand stack overflow", savePtr);
							return NULL;
						}
						inUnaryMode = false;
					}
					break;
				
				case Identifier : {
						node = addIdentifierNode(ps.start, ps.value, ps.hash);
						if (node == NULL) return NULL;					
						operandStack[topOfOperandStack] = node;
						topOfOperandStack++;
						if (topOfOperandStack == MaxStack) {
							sourceError("Operand stack overflow", savePtr);
							return NULL;
						}
						inUnaryMode = false;
					}
					break;
					
				case RightParen :										
				case RightBracket :
				case SemiColon :
				case Comma :
					inUnaryMode = false;
					inExpression = false;
					pushBack(&ps);
					break;
	
				default :
					sourceError("Unexpected token while waiting for a unary expression", savePtr);
					return NULL;
			}
		}
		
		if (!inExpression) break;
		
		waitingForBinaryOperator = true;
		
		while (waitingForBinaryOperator) {
			savePtr = curTextPtr;
			curTextPtr = getNextToken(curTextPtr, &ps);
			switch (ps.token) {				
				case Colon :
					if ((topOfOperatorStack > operatorStackStart)
							&& (operatorStack[topOfOperatorStack - 1].precedence > operatorPrecedence[ps.token])) {
						if (!doOperator()) return NULL;
					}
					if (topOfOperatorStack == operatorStackStart) {
						waitingForBinaryOperator = false;
						inExpression = false;
						pushBack(&ps);
					}
					else {
						if (operatorStack[topOfOperatorStack - 1].op != Question) {
							sourceError("Unexpected ':'", savePtr);
							return NULL;
						}
						else {
							operatorStack[topOfOperatorStack].op = Colon;
							operatorStack[topOfOperatorStack].precedence = operatorPrecedence[Colon];
							operatorStack[topOfOperatorStack].position = savePtr;
							topOfOperatorStack++;
							if (topOfOperatorStack == MaxStack) {
								sourceError("Operator stack overflow", savePtr);
								return NULL;
							}
							waitingForBinaryOperator = false;
						}
					}
					break;
				case Question :
				case NotEqual :
				case EqualEqual :
				case LessEqual :
				case GreaterEqual :
				case Less :
				case Greater :
				case PlusEqual :
				case MinusEqual :
				case AndEqual :
				case OrEqual :
				case XorEqual :
				case MultiplyEqual :
				case DivideEqual :
				case RemainderEqual :
				case LeftShiftEqual :
				case RightShiftEqual :
				case LeftShift :
				case RightShift :
				case Equal :
				case Period :
				case Multiply :
				case Divide :
				case Remainder :
				case Minus :
				case Plus :
				case AndAnd :
				case OrOr :
				case And :
				case Or :
				case Xor :
				case Arrow :
					while ((topOfOperatorStack > operatorStackStart)
							&& (operatorStack[topOfOperatorStack - 1].precedence >= operatorPrecedence[ps.token])) {
						if (!doOperator()) return NULL;
					}
					operatorStack[topOfOperatorStack].op = ps.token;
					operatorStack[topOfOperatorStack].precedence = operatorPrecedence[ps.token];
					operatorStack[topOfOperatorStack].position = savePtr;
					topOfOperatorStack++;
					if (topOfOperatorStack == MaxStack) {
						sourceError("Operator stack overflow", savePtr);
						return NULL;
					}
					waitingForBinaryOperator = false;
					break;

				case PostIncrement :
				case PostDecrement :
					if ((topOfOperatorStack > operatorStackStart)
							&& (operatorStack[topOfOperatorStack - 1].precedence > operatorPrecedence[ps.token])) {
						if (!doOperator()) return NULL;
					}
					operatorStack[topOfOperatorStack].op = ps.token;
					operatorStack[topOfOperatorStack].precedence = operatorPrecedence[ps.token];
					operatorStack[topOfOperatorStack].position = savePtr;
					topOfOperatorStack++;
					if (topOfOperatorStack == MaxStack) {
						sourceError("Operator stack overflow", savePtr);
						return NULL;
					}
					// stay waiting
					break;
				
				case LeftBracket : {
						if ((topOfOperatorStack > operatorStackStart)
								&& (operatorStack[topOfOperatorStack - 1].precedence >= operatorPrecedence[ps.token])) {
							if (!doOperator()) return NULL;
						}
				    	nestedStackStart = topOfOperatorStack;
						curTextPtr = parseExpression(curTextPtr);						
						if (curTextPtr == NULL) return NULL;
						while (topOfOperatorStack > nestedStackStart) {
							if (!doOperator())
								return NULL;
						}								
						savePtr = curTextPtr;
					    curTextPtr = getNextToken(curTextPtr, &ps);
						if (ps.token != RightBracket) {
							sourceError("Right bracket expected after index expression", savePtr);
							return NULL;
						}
						else {
							operatorStack[topOfOperatorStack].op = LeftBracket;
							operatorStack[topOfOperatorStack].precedence = operatorPrecedence[LeftBracket];
							operatorStack[topOfOperatorStack].position = savePtr;
							topOfOperatorStack++;
						}						
					}
					break;
					
				case LeftParen : {	// function call
					    int argumentCount = 0;				
						if ((topOfOperatorStack > operatorStackStart)
								&& (operatorStack[topOfOperatorStack - 1].precedence > operatorPrecedence[ps.token])) {
							if (!doOperator()) return NULL;
						}
					    curTextPtr = getNextToken(curTextPtr, &ps);
					    if (ps.token != RightParen) {
							pushBack(&ps);
						    while (true) {
						    	nestedStackStart = topOfOperatorStack;
								curTextPtr = parseExpression(curTextPtr);						
								if (curTextPtr == NULL) return NULL;
								while (topOfOperatorStack > nestedStackStart) {
									if (!doOperator())
										return NULL;
								}								
								argumentCount++;
								savePtr = curTextPtr;
								curTextPtr = getNextToken(curTextPtr, &ps);
								if (ps.token != Comma) {
									if (ps.token == RightParen)
										break;
									else {
										sourceError("Unexpected token in argument list", savePtr);
										return NULL;
									}
								}
						    }
						}
						operandStack[topOfOperandStack] = addImmediateNode(argumentCount, -1);
						topOfOperandStack++;
						if (topOfOperandStack == MaxStack) {
							sourceError("Operand stack overflow", savePtr);
							return NULL;
						}
						operatorStack[topOfOperatorStack].op = LeftParen;
						operatorStack[topOfOperatorStack].precedence = operatorPrecedence[LeftParen];
						operatorStack[topOfOperatorStack].position = savePtr;
						topOfOperatorStack++;
					}
					break;
				
				case Identifier :
					waitingForBinaryOperator = false;
					break;
	
				case RightBracket :
				case RightParen :
				case SemiColon :
				case Comma :
					waitingForBinaryOperator = false;
					inExpression = false;
					pushBack(&ps);
					break;

				default :
					waitingForBinaryOperator = false;
					inExpression = false;
					pushBack(&ps);
					break;
			}
			
		}
	}
	return curTextPtr;
}

CharPtr compileExpression(CharPtr curTextPtr, Node **result, Boolean allowComma)
{
	ParseState ps;
	int operatorStackStart;
	
	Node *expr;

	topOfOperatorStack = 0;
	topOfOperandStack = 0;
	topNode = 0;

	while (true) {
		operatorStackStart = topOfOperatorStack;
		curTextPtr = parseExpression(curTextPtr);
		if (curTextPtr == NULL) return NULL;	
		curTextPtr = getNextToken(curTextPtr, &ps);
		if ((ps.token == Comma) && allowComma) {
			while ((topOfOperatorStack > operatorStackStart)
					&& (operatorStack[topOfOperatorStack - 1].precedence > operatorPrecedence[ps.token])) {
				if (!doOperator()) return NULL;
			}
			operatorStack[topOfOperatorStack].op = ps.token;
			operatorStack[topOfOperatorStack].precedence = operatorPrecedence[ps.token];
			operatorStack[topOfOperatorStack].position = curTextPtr;
			topOfOperatorStack++;
		}
		else {
			pushBack(&ps);
			break;
		}
	}
					
	while (topOfOperatorStack > 0) {
		if (!doOperator())
			return NULL;
	}
	expr = resolveIdentifier(operandStack[0]);
	if (expr == NULL) return NULL;
			
	if (isArray(expr->typeIndex))
		expr = addNode(AddressOf, newPointerType(getBaseType(expr->typeIndex)), expr, NULL);
	if (expr == NULL) return NULL;
	*result = expr;
	return curTextPtr;
}

CharPtr expression(CharPtr curTextPtr, int trueBranch, int falseBranch, Node **result, Boolean allowComma)
{
	Node *expr;
	curTextPtr = compileExpression(curTextPtr, &expr, allowComma);
	if (curTextPtr == NULL) return NULL;
	if (result == NULL) {
	/* top of tree PostXcrement can be turned into PreXcrement since
		the result is discarded above here */
		if (expr->op == PostIncrement)
			expr->op = PreIncrement;
		else
		 	if (expr->op == PostDecrement)
		 		expr->op = PreDecrement;
	}
	if (!genCodeForExpression(expr, trueBranch, falseBranch, result))
		return NULL;
	else
		return curTextPtr;	
}

CharPtr exprCompile(CharPtr curTextPtr, UInt *savedIndex)
{
	Node *expr;
	curTextPtr = compileExpression(curTextPtr, &expr, true);
	if (curTextPtr == NULL) return NULL;
	*savedIndex = stashTree(expr);
	return curTextPtr;
}

Boolean exprGenCode(UInt index, int trueBranch, int falseBranch, Node **result)
{
	Node *top = recoverTree(index);
	if (result == NULL) {
	/* top of tree PostXcrement can be turned into PreXcrement since
		the result is discarded above here */
		if (top->op == PostIncrement)
			top->op = PreIncrement;
		else
		 	if (top->op == PostDecrement)
		 		top->op = PreDecrement;
	}
	return genCodeForExpression(top, trueBranch, falseBranch, result);
}

CharPtr constantExpression(CharPtr curTextPtr, long *theValue, Boolean expectFP, double *dValue)
{
	Node *expr;
	CharPtr savePtr = curTextPtr;

	topOfOperatorStack = 0;
	topOfOperandStack = 0;
	topNode = 0;
	curTextPtr = parseExpression(curTextPtr);
	if (curTextPtr == NULL) return NULL;
	while (topOfOperatorStack > 0) {
		if (!doOperator())
			return NULL;
	}
	expr = resolveIdentifier(operandStack[0]);
	if (expr == NULL) return false;
	
	if ((expr->op != Decl) || (expr->reg != Immediate)) {
		sourceError("Constant expression expected", savePtr);
		return NULL;
	}
	else {
		if (expr->typeIndex == DoubleTypeIndex) {
			if (expectFP)
				*dValue = expr->data.dValue;
			else {
				sourceError("FP constant not allowed", savePtr);
				return NULL;
			}
		}
		else
			if (expectFP)
				*dValue = 0.0;
			else
				*theValue = expr->value;
	}
	return curTextPtr;
}

void writeDeclaration(Declaration *theDecl)
{
	int typeSize = getTypeSize(theDecl->typeIndex);
	//char depthString[4] = "_xx";
	
	int i;
	if (getPoolIndex(theDecl, &i)) {
		buf[0] = (Char) PoolID;
		buf[1] = i >> 8;
		buf[2] = i & 0xFF;
		addToOutput(&buf[0], 3);
	}
	else {
		if (i != -1) {
			buf[0] = (Char) DefinePoolID;
			buf[1] = i >> 8;
			buf[2] = i & 0xFF;
			addToOutput(&buf[0], 3);
		}						
		addIdentifierToOutput(theDecl->name, theDecl->length, theDecl->depth, theDecl->isStatic);
	}
	switch (typeSize) {
		case 1 : 
			addTokenToOutput(ByteSpec);
			break;
		case 2 :
			addTokenToOutput(WordSpec);
			break;
		case 4 : 
			addTokenToOutput(LongSpec);
			break;
		default : {
				addTokenToOutput(Asm_Period);
				addNumberToOutput(typeSize);
			}
			break;
	}
}

Node initTarget[4];

CharPtr StructResultName = "__S_R";
int StructResultNameLength = 5;
unsigned char StructResultNameHash = '_' ^ '_' ^ 'S' ^ '_' ^ 'R';
Declaration *structResult;

int asmCount = 0;

Boolean checkForLocalDuplicate(Declaration *theDecl)
{
	char reg;
	Declaration *other = findIdentifier(theDecl->name, theDecl->length, &reg, theDecl->hash, theDecl);
	if (other) {
		if (other->isTypedef) {
			sourceError("Illegal redefinition of typedef", theDecl->name);
			return false;
		}
	 	if ((reg == A6) && (other->depth == theDecl->depth)) {
			sourceError("Duplicate local definition", theDecl->name);
			return false;
		}
	}
	return true;
}

Boolean checkForDuplicateTypedef(Declaration *theDecl)
{
	char reg;
	Declaration *other = findIdentifier(theDecl->name, theDecl->length, &reg, theDecl->hash, theDecl);
	if (other) {
		if (other->isTypedef) {
			if (!matchTypes(other->typeIndex, theDecl->typeIndex)) {
				sourceError("Illegal redefinition of typedef", theDecl->name);
				return false;
			}
		}
		sourceError("Illegal redefinition in typedef", theDecl->name);
		return false;
	}
	return true;
}

typedef struct SwitchData {
	long value;					// is label # for default if
	int label;					// this label is -1
	struct SwitchData *next;
} SwitchData;


Boolean gForDebug;

void emitDebugType(int typeIndex)
{
	if (typeIndex == CharPtrTypeIndex)
		addNumberToOutput(DBG_CharPointer);
	else
		if (isPointer(typeIndex)) {
			char reg;
			Declaration *formPtrTypeName = findIdentifier("FormPtr", 7, &reg, 'F' ^ 'o' ^ 'r' ^ 'm' ^ 'P' ^ 't' ^ 'r', NULL);
			if (formPtrTypeName 
					&& formPtrTypeName->isTypedef
					&& (typeIndex == formPtrTypeName->typeIndex))
				addNumberToOutput(DBG_FormPointer);
			else {
				Declaration *eventPtrTypeName = findIdentifier("EventPtr", 8, &reg, 'E' ^ 'v' ^ 'e' ^ 'n' ^ 't' ^ 'P' ^ 't' ^ 'r', NULL);
				if (eventPtrTypeName 
						&& eventPtrTypeName->isTypedef
						&& (typeIndex == eventPtrTypeName->typeIndex))
					addNumberToOutput(DBG_EventPointer);
				else
					addNumberToOutput(DBG_Pointer);
			}
		}
		else
		    if (isIntegral(typeIndex)) {
				if (isSigned(typeIndex))
					addNumberToOutput(DBG_Signed);
				else
					addNumberToOutput(DBG_Unsigned);
            }
		  
}


typedef struct {
	Node *result;
	UInt savedTestTreeIndex;
	UInt savedIncrTreeIndex;
	SwitchData **switchList;
	SwitchData *switchData;
	int label1;
	int label2;
	int label3;
	int label4;
	int breakTarget;
	int continueTarget;
	Token state;			// For, If, Else, Switch, Do, While, LeftCurly only
    Declaration *theDecl;
} StatementData;

#define MAX_NESTING 30
StatementData statementStack[MAX_NESTING];	// max statement nesting, sufficient?
int statementStackTop;

void pushStatementData(StatementData *sd)
{
	if (statementStackTop >= MAX_NESTING)
		error("Statements nested too deeply");
	statementStack[statementStackTop++] = *sd;
}

void popStatementData(StatementData *sd)
{
	*sd = statementStack[--statementStackTop];
}

CharPtr completeStatement(CharPtr curTextPtr, StatementData *sd, Boolean *doElse)
{
	Node *result;
	int i;
	CharPtr savePtr = curTextPtr;
	ParseState ps;
	
	switch (sd->state) {
	case For:
		// pop off the block for the init declaration
		for (i = localTop - 1; i >= blockTable[topBlock]; i--) {
			 gLocalPoolIndexList[i] = -1;
			 gLocalHashTable[localDeclaration[i].hash] = localDeclaration[i].nextDecl;
		}
		localTop = blockTable[topBlock];
		topBlock--;
		
		emitLabel(sd->label3);		// continue target
		if (!exprGenCode(sd->savedIncrTreeIndex, -1, -1, NULL))
			return NULL;
		emitLabel(sd->label1);		// test expr, branch after init to here
		if (!exprGenCode(sd->savedTestTreeIndex, sd->label2, sd->label4, &result))
			return NULL;
    	if (!emitConditionalBranch(result, sd->label2, sd->label4))
    		return NULL;
		emitLabel(sd->label4);
		break;
	case If:
        curTextPtr = getNextToken(curTextPtr, &ps);
		if (ps.token == Else) {
			sd->label3 = nextLabel++;
			emitBranch(Token_Bra, sd->label3);
			emitLabel(sd->label2);
			sd->state = Else;
			pushStatementData(sd);
			*doElse = true;
		}
		else {
			pushBack(&ps);
			emitLabel(sd->label2);
		}		
		break;
	case Else:
		emitLabel(sd->label3);							
		break;
	case While:		
		emitBranch(Token_Bra, sd->label2);
		emitLabel(sd->label3);
		break;
	case Do:
        curTextPtr = getNextToken(curTextPtr, &ps);
        if (ps.token != While) {
        	sourceError("While expected after 'do' body", savePtr);
        	return NULL;
        }
		savePtr = curTextPtr;
        curTextPtr = getNextToken(curTextPtr, &ps);
		if (ps.token != LeftParen) {
			sourceError("Left parenthesis expected after 'do..while'", savePtr);
			return NULL;
		}
		emitLabel(sd->label2);
		curTextPtr = expression(curTextPtr, sd->label1, sd->label3, &result, true);
    	if (curTextPtr == NULL) return NULL;
    	if (!emitConditionalBranch(result, sd->label1, sd->label3))
    		return NULL;
		savePtr = curTextPtr;
        curTextPtr = getNextToken(curTextPtr, &ps);
		if (ps.token == RightParen) {
			emitLabel(sd->label3);
		}
		else {
			sourceError("Right parenthesis expected after 'do..while' expression", savePtr);
			return NULL;
		}
        curTextPtr = getNextToken(curTextPtr, &ps);
		if (ps.token != SemiColon) {
			sourceError("Semi-colon expected after do statement", savePtr);
			return NULL;
		}
		break;
	case Switch:
		emitBranch(Token_Bra, sd->breakTarget);	// skip over in case the last case ends without a break
		emitLabel(sd->label4);			// the start of the decision sequence
		if (sd->switchData != NULL) {
			SwitchData *nextData;
			sd->label3 = -1;
			while (sd->switchData != NULL) {								
				if (sd->switchData->label == -1) {		// marks the default statement, if we have one
														// we add a branch to it at the end of the control
														// sequence.
					if (sd->label3 == -1) {
						sd->label3 = sd->switchData->value;
					}
					else {
						sourceError("Duplicate default statement", savePtr);
						return NULL;
					}									
				}
				else {						// add code to test the switch expr. against the expression value
											// and branch to the case code on a match, or fall thru to the next
											// test instruction if the match fails.				
					topNode = 0;
					operandStack[0] = addIdentifierNode(sd->theDecl->name, sd->theDecl->length, sd->theDecl->hash);
					operandStack[1] = addImmediateNode(sd->switchData->value, -1);
					operatorStack[0].op = NotEqual;				
					operatorStack[0].precedence = operatorPrecedence[NotEqual];				
					topOfOperatorStack = 1;
					topOfOperandStack = 2;
					if (!doOperator())
						return NULL;
					sd->label1 = nextLabel++;
					if (!genCodeForExpression(operandStack[0], sd->label1, sd->switchData->label, &result))
						return NULL;
			    	emitLabel(sd->label1);
				}
				nextData = sd->switchData->next;
				MemPtrFree(sd->switchData);
				sd->switchData = nextData;
			}
			if (sd->label3 != -1) {
	    		emitBranch(Token_Bra, sd->label3);
			}
		}
		emitLabel(sd->breakTarget);
		break;
    default:
        // do nothing;
        break;
	}
	return curTextPtr;
}

CharPtr statementOrDeclaration(CharPtr curTextPtr)
{
    Declaration *theDecl;	
	ParseState ps;
	
	StatementData sd = { 0 } ;
	Node *result;
	Node *switchTemp;
	SwitchData *switchData;

	int baseType, i;
	CharPtr savePtr = curTextPtr;
	statementStackTop = 0;
    
    curTextPtr = getNextToken(curTextPtr, &ps);
	if ((curTextPtr == NULL) || (ps.token != LeftCurly)) {
		sourceError("Empty body", savePtr);
		return NULL;
	}
	sd.state = LeftCurly;

    
    while (true) {
		savePtr = curTextPtr;
    	gStatementIndex++;
	    curTextPtr = getNextToken(curTextPtr, &ps);

		switch (ps.token) {
			case For : {
					pushStatementData(&sd);					
					sd.label4 = nextLabel++;		// the break label
					sd.label1 = nextLabel++;		// the test expression
					sd.label2 = nextLabel++;		// top-of-body
					sd.label3 = nextLabel++;		// increment expression
	
					savePtr = curTextPtr;
		            curTextPtr = getNextToken(curTextPtr, &ps);
					if (ps.token != LeftParen) {
						sourceError("Left parenthesis expected after 'for'", savePtr);
						return NULL;
					}
					// begin a new block for declarations in the init expression
					topBlock++;
					blockIndex++;
					if (topBlock == MaxBlocks) {
						sourceError("Too many blocks", curTextPtr);
						return NULL;
					}
					blockTable[topBlock] = localTop;
					
					baseType = -1;
					theDecl = NULL;				
				    curTextPtr = declaration(curTextPtr, &baseType, &theDecl, LocalScope, false);
				    if (curTextPtr == NULL) return NULL;
				    if (theDecl == NULL) { // not a declaration
						curTextPtr = expression(curTextPtr, -1, -1, NULL, true);
				    	if (curTextPtr == NULL) return NULL;
					}
					else {		// supporting a single declaration for for statements
				    	theDecl->depth = blockIndex;
						if (!checkForLocalDuplicate(theDecl)) return NULL;
			            savePtr = curTextPtr;

						theDecl->data.value = -(gFrameSize + getTypeSize(theDecl->typeIndex));
						gFrameSize += getTypeSize(theDecl->typeIndex);
						if (gFrameSize & 1) gFrameSize++;

						addTokenToOutput(Token_Local);
						writeDeclaration(theDecl);
			            curTextPtr = getNextToken(curTextPtr, &ps);
			            if (ps.token == Equal) {			            	
			            	curTextPtr = localInitializer(curTextPtr, theDecl, theDecl->typeIndex, false, false, theDecl, 0);
			            	if (curTextPtr == NULL)
			            		return NULL;
			            }
					}
					savePtr = curTextPtr;
		            curTextPtr = getNextToken(curTextPtr, &ps);
					if (ps.token != SemiColon) {
						sourceError("Semi-colon expected after 'for' initialization expression", savePtr);
						return NULL;
					}

					curTextPtr = exprCompile(curTextPtr, &sd.savedTestTreeIndex);
			    	if (curTextPtr == NULL) return NULL;
					savePtr = curTextPtr;
		            curTextPtr = getNextToken(curTextPtr, &ps);
					if (ps.token != SemiColon) {
						sourceError("Semi-colon expected after 'for' test expression", savePtr);
						return NULL;
					}
					curTextPtr = exprCompile(curTextPtr, &sd.savedIncrTreeIndex);
			    	if (curTextPtr == NULL) return NULL;
		            curTextPtr = getNextToken(curTextPtr, &ps);
					if (ps.token != RightParen) {
						sourceError("Right parenthesis expected after 'for' increment expression", savePtr);
						return NULL;
					}
					emitBranch(Token_Bra, sd.label1);
					emitLabel(sd.label2);
					sd.breakTarget = sd.label4;
					sd.continueTarget = sd.label3;

					sd.state = For;
				}
				break;
			case While : {
					pushStatementData(&sd);					
					sd.label1 = nextLabel++;
					sd.label2 = nextLabel++;
					sd.label3 = nextLabel++;
	
					savePtr = curTextPtr;
		            curTextPtr = getNextToken(curTextPtr, &ps);
					if (ps.token != LeftParen) {
						sourceError("Left parenthesis expected after 'while'", savePtr);
						return NULL;
					}
					emitLabel(sd.label2);
					curTextPtr = expression(curTextPtr, sd.label1, sd.label3, &result, true);
			    	if (curTextPtr == NULL) return NULL;
			    	if (!emitConditionalBranch(result, sd.label1, sd.label3))
			    		return NULL;
					savePtr = curTextPtr;
		            curTextPtr = getNextToken(curTextPtr, &ps);
					if (ps.token != RightParen) {
						sourceError("Right parenthesis expected after 'while' expression", savePtr);
						return NULL;
					}
					emitLabel(sd.label1);
					sd.breakTarget = sd.label3;
					sd.continueTarget = sd.label2;
					
					sd.state = While;
				}
				break;
			case Do : {
					pushStatementData(&sd);					
					sd.label1 = nextLabel++;
					sd.label2 = nextLabel++;
					sd.label3 = nextLabel++;
					emitLabel(sd.label1);
					sd.breakTarget = sd.label3;
					sd.continueTarget = sd.label2;
	
					sd.state = Do;
				}
				break;
			case If : {
					pushStatementData(&sd);
					sd.label1 = nextLabel++;
					sd.label2 = nextLabel++;
	
					savePtr = curTextPtr;				
				    curTextPtr = getNextToken(curTextPtr, &ps);
					if (ps.token == LeftParen) {
						curTextPtr = expression(curTextPtr, sd.label1, sd.label2, &result, true);					
				    	if (curTextPtr == NULL) return NULL;
				    	if (!emitConditionalBranch(result, sd.label1, sd.label2))
				    		return NULL;
			            curTextPtr = getNextToken(curTextPtr, &ps);
						if (ps.token != RightParen) {
							sourceError("Right parenthesis expected after 'if' expression", savePtr);
							return NULL;
						}
						else {
							emitLabel(sd.label1);
							sd.state = If;
						}
					}
					else {
						sourceError("Left parenthesis expected after 'if'", savePtr);
						return NULL;
					}
				}
				break;
			case Case : {
					if (sd.switchList == NULL) {
						sourceError("Case statement outside switch", savePtr);
						return NULL;
					}
					curTextPtr = constantExpression(curTextPtr, &ps.value, false, NULL);
					if (curTextPtr == NULL) {
						sourceError("Constant expression expected in case statement", savePtr);
						return NULL;
					}
					switchData = (SwitchData *)MemPtrNew(sizeof(SwitchData));
					switchData->next = *(sd.switchList);
					switchData->value = ps.value;
					switchData->label = nextLabel++;
			    	emitLabel(switchData->label);
					*(sd.switchList) = switchData;
					curTextPtr = getNextToken(curTextPtr, &ps);
					if (ps.token != Colon) {
						sourceError("Colon expected following case label", savePtr);
						return NULL;
					}
				}
				break;	// loop around for statement
			case Default : {
					if (sd.switchList == NULL) {
						sourceError("Default statement outside switch", savePtr);
						return NULL;
					}
					curTextPtr = getNextToken(curTextPtr, &ps);
					if (ps.token != Colon) {
						sourceError("Colon expected following default", savePtr);
						return NULL;
					}
					switchData = (SwitchData *)MemPtrNew(sizeof(SwitchData));
					switchData->next = *(sd.switchList);
					switchData->value = nextLabel++;
					switchData->label = -1;				// flags this as the default case
			    	emitLabel(switchData->value);
					*(sd.switchList) = switchData;
				}
				break;	// loop around for statement
			case Break : {
					if (sd.breakTarget == -1) {
						sourceError("No target for break statement", savePtr);
						return NULL;
					}
		    		emitBranch(Token_Bra, sd.breakTarget);
		            curTextPtr = getNextToken(curTextPtr, &ps);
					if (ps.token != SemiColon) {
						sourceError("Semi-colon expected after break", savePtr);
						return NULL;
					}
				}
				goto endOfStatement;
			case Continue : {
					if (sd.continueTarget == -1) {
						sourceError("No target for continue statement", savePtr);
						return NULL;
					}
		    		emitBranch(Token_Bra, sd.continueTarget);
		            curTextPtr = getNextToken(curTextPtr, &ps);
					if (ps.token != SemiColon) {
						sourceError("Semi-colon expected after continue", savePtr);
						return NULL;
					}
				}
				goto endOfStatement;
			case Switch : {
					pushStatementData(&sd);
					savePtr = curTextPtr;				
				    curTextPtr = getNextToken(curTextPtr, &ps);
					if (ps.token == LeftParen) {
						curTextPtr = expression(curTextPtr, -1, -1, &result, true);					
				    	if (curTextPtr == NULL) return NULL;
#ifdef DO_CODEGEN
				    	if (!isIntegral(result->typeIndex)) {
				    		if (isEnum(result->typeIndex))
				    			result->typeIndex = getBaseType(result->typeIndex);
				    		else {
				    			sourceError("Integral type required for switch expression", savePtr);
				    			return NULL;
				    		}
				    	}
				    	switchTemp = newTemp(result->typeIndex, -1);	// not available for reuse
				    	emitMove(result, switchTemp, true);
#else				    	
				    	switchTemp = newTemp(IntTypeIndex, -1);
#endif				    	
				    	sd.label4 = nextLabel++;
			    		emitBranch(Token_Bra, sd.label4);
				    	sd.theDecl = switchTemp->data.decl;
			            curTextPtr = getNextToken(curTextPtr, &ps);
						if (ps.token == RightParen) {
							sd.switchData = NULL;							
							statementStack[statementStackTop].switchData = NULL;				
							sd.breakTarget = nextLabel++;
							sd.state = Switch;
							sd.switchList = &statementStack[statementStackTop].switchData;
						}
						else {
							sourceError("Right parenthesis expected after 'switch' expression", savePtr);
							return NULL;
						}
					}
					else {
						sourceError("Left parenthesis expected after 'switch'", savePtr);
						return NULL;
					}
				}
				break;
			case Return : {
					baseType = getBaseType(currentFunctionType);
					savePtr = curTextPtr;
				    curTextPtr = getNextToken(curTextPtr, &ps);
					if (ps.token != SemiColon) {
						if (baseType == VoidTypeIndex) {
							sourceError("Return from void function", savePtr);
							return NULL;
						}
						pushBack(&ps);
				    	curTextPtr = expression(curTextPtr, -1, -1, &result, true);
				    	if (curTextPtr == NULL) return NULL;
#ifdef DO_CODEGEN
				    	if (isPointer(baseType)) {	
				    		if (!matchTypes(baseType, result->typeIndex)) {			    			
								if ((result->typeIndex == IntTypeIndex)
										&& (result->op == Decl)
										&& (result->reg == Immediate)
										&& (result->value == 0)) {
									result->typeIndex = UnsignedLongTypeIndex;
								}
								else {
					    			sourceError("Return of invalid pointer type", savePtr);
					    			return NULL;
					    		}
				    		}
				    		initTarget[0].reg = A0;
				    		initTarget[0].op = Decl;
							initTarget[0].offset = 0;
				    		initTarget[0].typeIndex = UnsignedLongTypeIndex;
				    		if ((result->reg != initTarget[0].reg) || (result->op != initTarget[0].op))
					    		emitMove(result, &initTarget[0], true);
				    	}
				    	else {
				    		if (result->typeIndex == VoidTypeIndex) {
				    			sourceError("Return of void expression", savePtr);
				    			return NULL;
				    		}
				    		else {
					    		if (isStruct(baseType)) {
					    			if (!matchTypes(baseType, result->typeIndex)) {
										sourceError("Incompatible return type", savePtr);
										return NULL;
									}
									constructStructResult(result, baseType);
									result = generateCode(&initTarget[2], -1, -1);
									if (result == NULL) return NULL;
					    		}
								else {
									if (isStruct(result->typeIndex)) {
										sourceError("Incompatible return type", savePtr);
										return NULL;
									}
									else {
										if (!emitConvert(result, baseType)) return NULL;
										if ((baseType == DoubleTypeIndex) || (baseType == FloatTypeIndex)) {
							    			if (!matchTypes(baseType, result->typeIndex)) {
												sourceError("Incompatible return type", savePtr);
												return NULL;
											}
											constructFPResult(result, baseType);
											result = generateCode(&initTarget[2], -1, -1);
											if (result == NULL) return NULL;
										}
										else {
								    		initTarget[0].reg = D0;
								    		initTarget[0].op = Decl;
											initTarget[0].offset = 0;
								    		initTarget[0].typeIndex = baseType;
								    		if (!matchTypes(baseType, result->typeIndex)) {
												if (isEnum(baseType))
													baseType = getBaseType(baseType);
												if (isEnum(result->typeIndex))
													result->typeIndex = getBaseType(result->typeIndex);
									    		if (!emitConvert(result, baseType))
									    			return NULL;
								    		}
								    		if ((result->reg != initTarget[0].reg) || (result->op != initTarget[0].op))
									    		emitMove(result, &initTarget[0], true);
									    }
									}
								}
				    		}
				    	}
#endif				    	
				    	savePtr = curTextPtr;
					    curTextPtr = getNextToken(curTextPtr, &ps);
						if (ps.token != SemiColon) {
							sourceError("Semicolon expected after return statement", savePtr);
							return NULL;
						}			    	
					}
					else {
						if (baseType != VoidTypeIndex) {
							sourceError("Return of void from non-void function", savePtr);
							return NULL;
						}				
					}
			    	emitBranch(Token_Bra, 0);
				}			
				goto endOfStatement;
			case RightCurly :
				if (statementStackTop == 0) {
					return curTextPtr;
				}
				if (sd.state != LeftCurly) {
					sourceError("Unexpected right brace", curTextPtr);
					return NULL;
				}
				// close off the block
				for (i = localTop - 1; i >= blockTable[topBlock]; i--) {
					 gLocalPoolIndexList[i] = -1;
					 gLocalHashTable[localDeclaration[i].hash] = localDeclaration[i].nextDecl;
				}
				localTop = blockTable[topBlock];
				topBlock--;
				
				popStatementData(&sd);
				goto endOfStatement;
			case LeftCurly :
				pushStatementData(&sd);
				topBlock++;
				blockIndex++;
				if (topBlock == MaxBlocks) {
					sourceError("Too many blocks", curTextPtr);
					return NULL;
				}
				blockTable[topBlock] = localTop;
				sd.state = LeftCurly;
				break;
			case SemiColon :
				goto endOfStatement;
			case Typedef : {
					baseType = -1;
				    theDecl = NULL;
				    savePtr = curTextPtr;
				    curTextPtr = declaration(curTextPtr, &baseType, &theDecl, LocalScope, false);
				    if (theDecl == NULL) {
				    	sourceError("Expected a declaration after typedef", savePtr);
				    	return NULL;
				    }
				    savePtr = curTextPtr;
		            curTextPtr = getNextToken(curTextPtr, &ps);
					if (ps.token != SemiColon) {
						sourceError("Semi-colon expected after typedef", savePtr);
						return NULL;
					}
				    theDecl->isTypedef = true;
				    if (!checkForDuplicateTypedef(theDecl)) return NULL;
				}
				goto endOfStatement;
			case Goto :
				savePtr = curTextPtr;
				curTextPtr = getNextToken(curTextPtr, &ps);
				if (ps.token == Identifier) {
					emitUserBranch(ps.start, ps.value);
				}
				else {
					sourceError("Identifier expected 'goto'", savePtr);
					return NULL;
				}
	            curTextPtr = getNextToken(curTextPtr, &ps);
				if (ps.token != SemiColon) {
					sourceError("Semi-colon expected after goto", savePtr);
					return NULL;
				}
				goto endOfStatement;
			default : {				
					if (ps.token == Identifier) {
					    ParseState lookahead; 
						curTextPtr = getNextToken(curTextPtr, &lookahead);
						if (lookahead.token == Colon) {
							emitUserLabel(ps.start, ps.value);
							break;	// loop around for statement
						}
						else {
							pushBack(&lookahead);
						}
					}
					baseType = -1;
				    theDecl = NULL;
					pushBack(&ps);
				    curTextPtr = declaration(curTextPtr, &baseType, &theDecl, LocalScope, false);
				    if (curTextPtr == NULL) return NULL;
				    if (theDecl == NULL) { // not a declaration
				    	curTextPtr = expression(curTextPtr, -1, -1, &result, true);
				    	if (curTextPtr == NULL) return NULL;
				    	savePtr = curTextPtr;
			            curTextPtr = getNextToken(curTextPtr, &ps);
						if (ps.token != SemiColon) {
							sourceError("Semi-colon expected after expression", savePtr);
							return NULL;
						}
				    }
				    else {
				    // seems like a wacky way to detect a global, but the depth isn't set yet
				    // and there's no other indicator. ??? can't we set the depth in declaration() ???
						if (findGlobalDeclBase(theDecl)) {
//						if ((theDecl >= globalTable) && (theDecl <= &globalTable[globalTop]))
							curTextPtr = handleLocalStatics(curTextPtr, theDecl, baseType);
							if (curTextPtr == NULL)
								return NULL;
						}
						else {
					        while (true) {
					            savePtr = curTextPtr;
						    	theDecl->depth = blockIndex;
								if (!checkForLocalDuplicate(theDecl)) return NULL;
								if (theDecl->data.value == -1) { // register
/*								
{								
			char errBuf[32] = "saw register for ";
			int t = StrLen(errBuf);
			int x;
			for (x = 0; x < theDecl->length; x++)
				errBuf[t + x] = theDecl->name[x];
			errBuf[t + x] = '\0';
			error(errBuf);
}								
*/
									theDecl->data.value = 0;
									if (isPointer(theDecl->typeIndex)) {
										for (i = A3; i <= A4; i++) {
											if (!gRegisterAllocated[i]) {
												gRegisterAllocated[i] = true;
												theDecl->data.value = i;
												break;
											}
										}
									}
									else {
										if (isStruct(theDecl->typeIndex) || isArray(theDecl->typeIndex)) {
											sourceError("Register can't be applied to struct or array types", savePtr);
											return NULL;
										}
										else {
											if ((theDecl->typeIndex != DoubleTypeIndex) && 
													(theDecl->typeIndex != FloatTypeIndex))
												for (i = D4; i <= D7; i++) {
													if (!gRegisterAllocated[i]) {
														gRegisterAllocated[i] = true;
														theDecl->data.value = i;
														break;
													}
												}
										}
									}
								}
								if (theDecl->data.value == 0) {
									theDecl->data.value = -(gFrameSize + getTypeSize(theDecl->typeIndex));
									gFrameSize += getTypeSize(theDecl->typeIndex);
									if (gFrameSize & 1) gFrameSize++;

									addTokenToOutput(Token_Local);
									writeDeclaration(theDecl);
									if (gForDebug)
										emitDebugType(theDecl->typeIndex);									
								}
					            curTextPtr = getNextToken(curTextPtr, &ps);
					            if (ps.token == Equal) {			
					            	curTextPtr = localInitializer(curTextPtr, theDecl, theDecl->typeIndex, false, false, theDecl, 0);
					            	if (curTextPtr == NULL)
					            		return NULL;
					            	curTextPtr = getNextToken(curTextPtr, &ps);
					            }
		
					            if (ps.token == SemiColon)
					                break;
					            else {
					                if (ps.token != Comma) {
					                    sourceError("Comma or semi-colon expected after declaration", savePtr);
					                    return NULL;
					                }
					                theDecl = NULL;
					                savePtr = curTextPtr;
								    curTextPtr = declaration(curTextPtr, &baseType, &theDecl, LocalScope, false);
					                if (curTextPtr == NULL) return NULL;
					                if (theDecl == NULL) {
										sourceError("Declaration expected", savePtr);
										return NULL;
					                }
					            }
					        }	
						}
				    }
				}
				goto endOfStatement;
		}
		continue;
endOfStatement:
		while (sd.state != LeftCurly) {	// finish all pending statements until
										// the next outermost block
			Boolean doElse = false;
			curTextPtr = completeStatement(curTextPtr, &sd, &doElse);
			if (curTextPtr == NULL) return NULL;
			if (statementStackTop == 0) {
				sourceError("Illegal stack state", curTextPtr);
				return NULL;	// can't do this! return must only come via final right curly
			}
			popStatementData(&sd);
			if (doElse) break;
		}
		continue;
	}
	return NULL;		
}


char buf[64];		// handy buffer for various things (not on the stack!)

char *VarDataName = "__VarData__";
int VarDataNameLength = 11;
unsigned char VarDataNameHash = '_' ^ '_' ^ 'V' ^ 'a' ^ 'r' ^ 'D' ^ 'a' ^ 't' ^ 'a' ^ '_' ^ '_';

Node gVarDataAddress;

void emitDebugEntry()
{
//	emitUnary(Token_Pea, Unsized, &gVarDataAddress, false);
	emitLibCall("DebugEntry", 10);
//	cleanStack(4);
}

CharPtr functionBody(CharPtr curTextPtr, Boolean isPilotMain, Boolean forDebug)
{
    int currentGlobalTop = globalTop;
    int i;
	char movemString[32];	
	int movemIndex = 0;

	bodyBuffer = MemHandleLock(bodyContents);
	bodyTop = 0;
	nextLabel = 1;			// label 0 is the exit label
	topBlock = 1;			//	make all local decls
	blockIndex = 1;			//  with depth >= 1
	localTop = 0;
	tempIndex = 0;
	tempCount = 0;
	tempNameSpace = tempNameBase;
	gFunctionCount++;
	gFrameSize = 0;
	gStatementIndex = 0;
	gForDebug = !isPilotMain && forDebug;
	
	for (i = 0; i < MaxLocals; i++)
		gLocalPoolIndexList[i] = -1;
	for (i = 0; i < 128; i++) 
		gLocalHashTable[i] = NULL;
	
	for (i = D0; i <= A4; i++)
		gRegisterAllocated[i] = false;		// start with everything available
		
	for (i = D0; i <= D3; i++)
		gRegisterAllocated[i] = true;		// exclude a working set of Dn...
	for (i = A0; i <= A2; i++)
		gRegisterAllocated[i] = true;		// ...and An registers
	
	if (forDebug) {
		if (isPilotMain) {
			gFrameSize = sizeof(DebugData);
			addTokenToOutput(Token_Local);
			addIdentifierToOutput("d000", 4, 1, false);
			addTokenToOutput(Asm_Period);
			addNumberToOutput(sizeof(DebugData));

			writeTokenToBuffer(Token_Pea);
			writeNumberToBuffer(-sizeof(DebugData));
			writeTokenToBuffer(Asm_LeftParen);
			writeTokenToBuffer((AsmToken)(Token_a6));
			writeTokenToBuffer(Asm_RightParen);
			emitLibCall("DebugInit", 9);
		}
		else {
			if (gCurSegment == 1) {
				emitLibCall("DebugEntry", 10);
			}
		}
	}

	curTextPtr = statementOrDeclaration(curTextPtr);
	if (curTextPtr == NULL) {
		MemHandleUnlock(bodyContents);
		return NULL;
	}
/*
	while (true) {
		CharPtr savePtr;
		savePtr = curTextPtr;
	    curTextPtr = getNextToken(curTextPtr, &ps);
		if (ps.token == RightCurly)
			break;
		else
			pushBack(&ps);
		curTextPtr = statementOrDeclaration(curTextPtr);
		if (curTextPtr == NULL) {
			MemHandleUnlock(bodyContents);
			return NULL;
		}
	}
*/
	emitLabel(0);		// (this goes to the body buffer)
	if (forDebug && isPilotMain) {
		writeTokenToBuffer(Token_Pea);
		writeNumberToBuffer(-sizeof(DebugData));
		writeTokenToBuffer(Asm_LeftParen);
		writeTokenToBuffer((AsmToken)(Token_a6));
		writeTokenToBuffer(Asm_RightParen);
		emitLibCall("DebugTerm", 9);
	}		


	addTokenToOutput(Token_BeginProc);

	for (i = D4; i <= D7; i++) {
		if (gRegisterAllocated[i]) {
			if (movemIndex > 0) 
				movemString[movemIndex++] = Asm_Divide;
				movemString[movemIndex++] = Token_d0 + i;	// have to use Token_ since we get the assembler to build
															// the register set mask, and it's only driven by the core
															// tokens, not the EA_.. kind
		}
	}		
	for (i = A3; i <= A4; i++) {
		if (gRegisterAllocated[i]) {
			if (movemIndex > 0) 
				movemString[movemIndex++] = Asm_Divide;
			movemString[movemIndex++] = Token_d0 + i;
		}
	}
	
	if (movemIndex > 0) {
		addTokenToOutput(Token_Movem);
		addTokenToOutput(LongSpec);
		for (i = 0; i < movemIndex; i++)
			addTokenToOutput((AsmToken)movemString[i]);
		addTokenToOutput(Asm_Comma);
		addTokenToOutput(Asm_Minus);
		addTokenToOutput(Asm_LeftParen);
		addTokenToOutput(Token_a7);
		addTokenToOutput(Asm_RightParen);
	}
/*		
	if (bodyTop >= 4096) {
		char *start = bodyBuffer;
		ULong length = bodyTop;
		while (length >= 4096) {
			ULong pieceLength;
			char *p = start + 4095;		// we want to include the '\n' if one
										// just happens to be at start + 4096
			while ((*p != Asm_Whitespace) && (p > start)) p--;			
			if (p == start) p = start + 4095;
			pieceLength = (p - start) + 1;
			addToOutput(start, pieceLength);
			start = p + 1;
			length -= pieceLength;
		}
		if (length > 0) addToOutput(start, length);
	}	
	else
*/	
	if ((outputTop + bodyTop) > 60000) {
		asmCount++;
		getAsmName(getSourceName(), buf, asmCount);
		addTokenToOutput(Asm_Extend);
		addStringToOutput(buf, StrLen(buf));
		closeAsmFile();
		openAsmFile(buf);
	}

	addToOutput(bodyBuffer, bodyTop);
	
	if (movemIndex > 0) {
		addTokenToOutput(Token_Movem);
		addTokenToOutput(LongSpec);
		addTokenToOutput(Asm_LeftParen);
		addTokenToOutput(Token_a7);
		addTokenToOutput(Asm_RightParen);
		addTokenToOutput(Asm_Plus);
		addTokenToOutput(Asm_Comma);
		for (i = 0; i < movemIndex; i++)
			addTokenToOutput((AsmToken)movemString[i]);
	}

	addTokenToOutput(Token_EndProc);
	
	addToOutput(codeStringPoolBuffer, codeStringPoolTop);
	codeStringPoolTop = 0;
	
	MemHandleUnlock(bodyContents);
	
	globalTop = currentGlobalTop;
	
	if (outputTop > 50000) {	// getting scary-sized, can't go over 64k
									// as the assembler wants to build a single
									// record out of the input
		asmCount++;
		getAsmName(getSourceName(), buf, asmCount);
//			StrPrintF(buf, "%s%d.asm", appName, asmCount);
//		addTokenToOutput(Token_Include);
		addTokenToOutput(Asm_Extend);
		addStringToOutput(buf, StrLen(buf));
//			StrPrintF(buf, "%s%d", appName, asmCount);
		closeAsmFile();
		openAsmFile(buf);		
	}
	
	return curTextPtr;
}

// can't clobber buf in addStringConstant as it may be being used as the identifier
// name return buffer passed in tBuf

char strBuf[8];

void addStringConstant(CharPtr startPtr, int strLength, CharPtr tBuf, int *length)
{
	int buflen;
	int entryLength;
	
	tBuf[0] = Asm_Identifier;
	StrCopy(tBuf + 2, "__S");
	StrIToA(tBuf + 5, stringPoolIndex++);
	tBuf[1] = StrLen(tBuf + 2);
	buflen = StrLen(tBuf);
	
	entryLength = buflen + 5 + strLength;
		
	if ((stringPoolTop + entryLength) >= stringPoolSize) {
		MemHandleUnlock(stringPoolContents);
		stringPoolSize += 2048;
		stringPoolContents = DmResizeRecord(gOutputDB, stringPoolRecordIndex, stringPoolSize);
		if (stringPoolContents == NULL) {
			error("Out of memory for string constants");
			return;
		}
		stringPoolBuffer = MemHandleLock(stringPoolContents);		
	}
	
	DmWrite(stringPoolBuffer, stringPoolTop, tBuf, buflen);
	stringPoolTop += buflen;
	strBuf[0] = Token_Dc;
	strBuf[1] = ByteSpec;
	strBuf[2] = Asm_DoubleQuote;
	strBuf[3] = (strLength >> 8) & 0xFF;
	strBuf[4] = strLength & 0xFF;
	DmWrite(stringPoolBuffer, stringPoolTop, strBuf, 5);
	stringPoolTop += 5;
	DmWrite(stringPoolBuffer, stringPoolTop, startPtr, strLength);
	stringPoolTop += strLength;
	*length = buflen;
}

void addCodeStringConstant(CharPtr startPtr, int strLength, CharPtr tBuf, int *length)
{
	int buflen;
	int entryLength;
	
	tBuf[0] = Asm_Identifier;
	StrCopy(tBuf + 2, "__X");
	StrIToA(tBuf + 5, codeStringPoolIndex++);
	tBuf[1] = StrLen(tBuf + 2);
	buflen = StrLen(tBuf);
	
	entryLength = buflen + 5 + strLength;
		
	if ((codeStringPoolTop + entryLength) >= codeStringPoolSize) {
		MemHandleUnlock(codeStringPoolContents);
		codeStringPoolSize += 2048;
		codeStringPoolContents = DmResizeRecord(gOutputDB, codeStringPoolRecordIndex, codeStringPoolSize);
		if (codeStringPoolContents == NULL) {
			error("Out of memory for string constants");
			return;
		}
		codeStringPoolBuffer = MemHandleLock(codeStringPoolContents);		
	}
	
	DmWrite(codeStringPoolBuffer, codeStringPoolTop, tBuf, buflen);
	codeStringPoolTop += buflen;
	strBuf[0] = Token_Dc;
	strBuf[1] = ByteSpec;
	strBuf[2] = Asm_DoubleQuote;
	strBuf[3] = (strLength >> 8) & 0xFF;
	strBuf[4] = strLength & 0xFF;
	DmWrite(codeStringPoolBuffer, codeStringPoolTop, strBuf, 5);
	codeStringPoolTop += 5;
	DmWrite(codeStringPoolBuffer, codeStringPoolTop, startPtr, strLength);
	codeStringPoolTop += strLength;
	*length = buflen;
}

Boolean checkForDuplicateDef(Declaration *theDecl)
{
	char reg;
	Declaration *other = findIdentifier(theDecl->name, theDecl->length, &reg, theDecl->hash, theDecl);
	if (other) {
		if (other->isTypedef) {
			sourceError("Illegal redefinition of typedef", theDecl->name);
			return false;
		}
	 	if (((reg == A5) || (reg == PC)) && (other->depth == 0)) {
			sourceError("Duplicate global definition", theDecl->name);
			return false;
		}
	}
	return true;
}

Boolean isName(CharPtr name, int length, CharPtr test)
{
	int i;
	for (i = 0; i < length; i++) {
		if (test[i] == '\0') return false;
		if (name[i] != test[i])
			return false;
	}
	if (test[i] != '\0') return false;
	return true;
}

Boolean compile(CharPtr curTextPtr, Boolean forDebug)
{
	ParseState ps;
    CharPtr savePtr;
    int i;
    int length;
	
	while (true) {
	    int baseType = -1;
	    Declaration *theDecl = NULL;
	
	    savePtr = curTextPtr;
	    curTextPtr = getNextToken(curTextPtr, &ps);
		if (ps.token == Typedef) {
		    savePtr = curTextPtr;
		    curTextPtr = declaration(curTextPtr, &baseType, &theDecl, GlobalScope, false);
		    if (curTextPtr == NULL) return false;
		    if (theDecl == NULL) {
		    	sourceError("Expected a declaration after typedef", savePtr);
		    	return false;
		    }
		    //		    setStatus("Typedef-ing ... ", theDecl->name, theDecl->length);
		    savePtr = curTextPtr;
            curTextPtr = getNextToken(curTextPtr, &ps);
			if (ps.token != SemiColon) {
				sourceError("Semi-colon expected after typedef", savePtr);
				return false;
			}
			setTypeDef(theDecl, GlobalScope, true);
			if (!checkForDuplicateTypedef(theDecl)) return false;
//		    theDecl->isTypedef = true;
		}
		else {
		  Boolean sawExtern = false;
			Boolean sawAsm = false;
			if (ps.token == Extern) {
				sawExtern = true;
				pushBack(&ps);
			}
			else {
				if (ps.token == Asm) {
					sawAsm = true;
				}
				else
					pushBack(&ps);
			}
/*		
			Boolean sawStatic = false;
			Boolean sawExtern = false;
			if (ps.token == Static)
				sawStatic = true;
			else
				if (ps.token == Extern)
					sawExtern = true;
				else
					if (ps.token == Asm) {
						sourceError("Can't use 'asm' syntax in this version, sorry", savePtr);
						return false;
					}
					else
						pushBack(&ps);
*/						
		    curTextPtr = declaration(curTextPtr, &baseType, &theDecl, GlobalScope, false);
		    if (curTextPtr == NULL) return false;
		    if (theDecl == NULL) {
		    	savePtr = curTextPtr;
		        curTextPtr = getNextToken(curTextPtr, &ps);
		        if (ps.token == Token_EOF)
		        	return true;
				if (ps.token != SemiColon) {
					sourceError("Declaration or semi-colon expected", savePtr);
					return false;
				}
		    }
		    else {
//						setStatus("Declaring ... ", theDecl->name, theDecl->length);
				setDepth(theDecl, GlobalScope, 0);
/*
				if (sawStatic) {
					if (theDecl->isStatic == -1)	// was previously an extern, keep it that way
						setStatic(theDecl, GlobalScope, false);
					else
						setStatic(theDecl, GlobalScope, true);
				}
				else
					setStatic(theDecl, GlobalScope, false);
*/
			    if (isFunction(theDecl->typeIndex)) {       
			        curTextPtr = getNextToken(curTextPtr, &ps);
			        if (ps.token == LeftCurly) {
				    	savePtr = curTextPtr;
			        	if (!checkForDuplicateDef(theDecl)) return false;
				        length = theDecl->length;				
						currentFunctionType = theDecl->typeIndex;
						baseType = getBaseType(currentFunctionType);
						setStatus("Compiling ... ", theDecl->name, length);
						addTokenToOutput(Token_Proc);
						addIdentifierToOutput(theDecl->name, length, 0, theDecl->isStatic);
						if (sawAsm) {
							// very cheap asm parser sucks everything up to close from the '{'
							while ((*curTextPtr != '}') && (*curTextPtr)) curTextPtr++; 
							// balance '{'
							if (*curTextPtr == '}') {
								addTokenToOutput(Token_Asm);
								length = curTextPtr - savePtr;
								addToOutput((char *)&length, 2);
								addToOutput(savePtr, length);
								length = 0;
								addToOutput((char *)&length, 1);
								addTokenToOutput(Token_EndAsm);
								curTextPtr++;
							}
							else {
		        				sourceError("Expected close curly in asm definition", savePtr);
								curTextPtr = NULL;
								return false;
							}
						}
						else {
							addTokenToOutput(Asm_LeftParen);
							if (isStruct(baseType) || (baseType == DoubleTypeIndex) || (baseType == FloatTypeIndex)) {
								addIdentifierToOutput(StructResultName, StructResultNameLength, structResult->depth, false);
								addTokenToOutput(LongSpec);
								if (getTypeSize(currentFunctionType) > 0)
									addTokenToOutput(Asm_Comma);
							}
							for (i = 0; i < getTypeSize(currentFunctionType); i++) {
								Declaration *parm = getArg(currentFunctionType, i);
								writeDeclaration(parm);
								if (forDebug)
									emitDebugType(parm->typeIndex);
								if (i < (getTypeSize(currentFunctionType) - 1)) {
									addTokenToOutput(Asm_Comma);
								}
							}
							addTokenToOutput(Asm_RightParen);
							addTokenToOutput(Asm_Return);
							if (forDebug)
								emitDebugType(getBaseType(currentFunctionType));
							pushBack(&ps);		// pass the left curly through to the statement loop
							curTextPtr = functionBody(curTextPtr, isName(theDecl->name, theDecl->length, "PilotMain"), forDebug );
							if (curTextPtr == NULL) return false;
						}
						currentFunctionType = -1;
			        }
			        else {
			        	if (ps.token == Equal) {
				        	int inlineCount = 0;
				        	int inlineBase  = 0;
				        	if (!checkForDuplicateDef(theDecl)) return false;
			        		savePtr = curTextPtr;
			        		curTextPtr = getNextToken(curTextPtr, &ps);
			        		if (ps.token == LeftCurly) {
			        			while (true) {
			        				savePtr = curTextPtr;
									curTextPtr = constantExpression(curTextPtr, &ps.value, false, NULL);
									if (curTextPtr == NULL) {
										sourceError("Constant expression expected in inline declaration", savePtr);
										return false;
									}
				        			if (inlineCount == 0) {
				        				inlineBase = addInline(ps.value);
		        						if (inlineBase == -1) return false;
		        					}
		        					else
		        						if (addInline(ps.value) == -1) return false;
									inlineCount++;
			        				savePtr = curTextPtr;
					        		curTextPtr = getNextToken(curTextPtr, &ps);
					        		if (ps.token != Comma) {
					        			if (ps.token != RightCurly) {
					        				sourceError("Expected comma or right curly in inline definition", savePtr);
					        				return false;
					        			}
					        			else
					        				break;
					        		}
			        			}
			        		}
			        		else {
			        			if (ps.token == Digit) {
			        				inlineBase = addInline(ps.value);
			        				if (inlineBase == -1) return false;
			        				inlineCount++;
			        			}
			        			else {
			        				sourceError("Expected number or left curly after = for inline", savePtr);
			        				return false;
			        			}
			        		}
			        		setInlineFunction(theDecl, inlineBase, inlineCount);
				        	savePtr = curTextPtr;
			        		curTextPtr = getNextToken(curTextPtr, &ps);
			        		if (ps.token != SemiColon) {
			        			sourceError("Semicolon expected after inline definition", savePtr);
			        			return false;
			        		}
			        	}
			        	else
			        		if (ps.token == SemiColon) {
//			        			if (sawExtern)
			        				setDepth(theDecl, GlobalScope, -1);
				        	}
			        }
			    }
			    else {
			        while (true) {
			            savePtr = curTextPtr;			            
			            curTextPtr = getNextToken(curTextPtr, &ps);
			            // if the var was 'extern ...' and isn't being initialized
			            // set it as a forward declaration
						if (sawExtern && (ps.token != Equal))
							setDepth(theDecl, GlobalScope, -1);
						else {
				        	if (!checkForDuplicateDef(theDecl)) return false;
							buf[0] = Asm_Identifier;
							buf[1] = theDecl->length + 3;
							if (theDecl->isStatic) buf[1] += 3;
							writeToDataBuffer(buf, 2);
//							DmWrite(dataBuffer, dataTop, buf, 2);
//							dataTop += 2;
							writeToDataBuffer(theDecl->name, theDecl->length);
//							DmWrite(dataBuffer, dataTop, theDecl->name, theDecl->length);
//							dataTop += theDecl->length;
							writeToDataBuffer("_00", 3);
//							DmWrite(dataBuffer, dataTop, "_00", 3);
//							dataTop += 3;
							if (theDecl->isStatic) {
								buf[0] = '_';
								buf[1] = (gStaticIndex >> 3) + '0';
								buf[2] = (gStaticIndex & 0x7) + '0';
								writeToDataBuffer(buf, 3);
//								DmWrite(dataBuffer, dataTop, buf, 3);
//								dataTop += 3;
							}
								
				            if (ps.token == Equal) {			            	
				            	curTextPtr = handleInitializer(curTextPtr, theDecl, theDecl->typeIndex, false);
				            	if (curTextPtr == NULL)	return false;
						        curTextPtr = getNextToken(curTextPtr, &ps);
							}
				            else {
			            		writeTokenToDataBuffer(Token_Dc);
								writeSize(getTypeSize(theDecl->typeIndex));
			            		writeTokenToDataBuffer(Asm_Zero);
//								DmWrite(dataBuffer, dataTop, "\0\0\0\0", 4);
//								dataTop += 4;
				            }
						}
			            if (ps.token == SemiColon)
			                break;
			            else {
			                if (ps.token != Comma) {
			                    sourceError("Comma or semi-colon expected after declaration", savePtr);
			                    return false;
			                }
			                theDecl = NULL;
			                savePtr = curTextPtr;
			                curTextPtr = declaration(curTextPtr, &baseType, &theDecl, GlobalScope, false);
			                if (curTextPtr == NULL) return false;
			                if (theDecl == NULL) {
			                    sourceError("Declaration expected", savePtr);
			                	return false;
			                }
			                setDepth(theDecl, GlobalScope, 0);
			            }
			        }	
				}
		    }
		}
	}
}

char dateMacroContents[11 + longDateStrLength + 1];

char timeMacroContents[11 + timeStringLength + 2];

// __OBC__ "OnBoardC"
char obcMacroContents[10 + 8 + 2]; // 10 for "__OBC__ \"", 8 for "OnBoardC", 2 for "\""

char* fileMacroContents = "__FILE__ "; 
char* lineMacroContents = "__LINE__ "; 

char *OnBoardMacroNew = "OnBoardC \"OnBoardC\"";

// See comments below in addOnBoardMacro(), but it's now been further dragged
// into the standard way of adding macros, and just done with the others in
// setDefaultMacros()
// S. Little
// 18/08/03

/*
char *OnBoardMacro = "OnBoardC";
unsigned char OnBoardMacroHash = 'O' ^ 'n' ^ 'B' ^ 'o' ^ 'a' ^ 'r' ^ 'd' ^ 'C';
*/

/*
void addOnBoardMacro()
{
  // This function appears to break some stuff when adding more macros. 
  // Not sure why, but OnBoardC is deprecated as a macro anyway, so it 
  // might be best to just scrap it.
  // For now, we just use addMacro to add a similar macro.
  // S. Little
  // 29/07/2003

  addMacro(OnBoardMacroNew);
*/
  /*	MacroDef md;
  error("addonboardmacro");
	md.name = OnBoardMacro;
	md.length = StrLen(OnBoardMacro);
	md.hash = OnBoardMacroHash;
	md.parameterCount = 0;
	md.macroStart = OnBoardMacro;
	md.macroEnd = md.macroStart + md.length;
	md.nextMacro = gMacroHashTable[OnBoardMacroHash];
	gMacroHashTable[OnBoardMacroHash] = macroListTop;
	DmWrite(macroList, macroListTop * sizeof(MacroDef), &md, sizeof(MacroDef));
	macroListTop++;
	error("done"); */
/*
}

*/
void setDefaultMacros()
{
	DateTimeType dt;
	TimSecondsToDateTime(TimGetSeconds(), &dt);

	StrCopy(dateMacroContents, "__DATE__ \"");
	DateToAscii(dt.month, dt.day, dt.year, dfDMYLong, dateMacroContents + 10);
	StrCat(dateMacroContents, "\"");
	addMacro(dateMacroContents);

	StrCopy(timeMacroContents, "__TIME__ \"");
	TimeToAscii(dt.hour, dt.minute, tfColonAMPM, timeMacroContents + 10);
	StrCat(timeMacroContents, "\"");
	addMacro(timeMacroContents);

	// need to add a version string really
	StrCopy(obcMacroContents, "__OBC__ \"OnBoardC\"\n");
	addMacro(obcMacroContents);
	//why use a special function for this macro at all? 
	//	addOnBoardMacro();
	addMacro(OnBoardMacroNew);

	addSpecialMacro(fileMacroContents,MFFileName); 
	addSpecialMacro(lineMacroContents,MFLineNumber);
}

void resetCompiler()
{
	int i;
	
	if (bodyContents != NULL) {
		MemHandleFree(bodyContents);
		bodyContents = NULL;
	}

 	if (gHB == NULL) {
    	MaxTypeIndex = 400;
		bodySize = 1024;
    	if (typeHandle != NULL) {
    		MemHandleUnlock(typeHandle);
    		MemHandleFree(typeHandle);
    	}
 		typeHandle = MemHandleNew(sizeof(Type) * MaxTypeIndex);
 		if (typeHandle == NULL) {
 			error("Couldn't allocate type table");
 			return;
 		}
 		typeTable = MemHandleLock(typeHandle);
	    typeIndex = NumberOfPredefinedTypes;
	    for (i = 0; i < typeIndex; i++) {
	    	typeTable[i] = predefinedTypes[i];
	    }
	}	   
    else {
    	MaxTypeIndex = gHB->typeCount + 50;
		bodySize = 4096;
    	if (typeHandle != NULL) {
    		MemHandleUnlock(typeHandle);
    		MemHandleFree(typeHandle);
    	}
 		typeHandle = MemHandleNew(sizeof(Type) * (MaxTypeIndex - gHB->typeCount));
 		if (typeHandle == NULL) {
 			error("Couldn't allocate type table");
 			return;
 		}
 		typeTable = MemHandleLock(typeHandle);
    	typeIndex = gHB->typeCount;
    }
    
	bodyContents = MemHandleNew(bodySize);		// this one on the heap for speed
	if ((bodyContents == NULL) && (bodySize > 1024)) {
		bodySize = 1024;
		bodyContents = MemHandleNew(bodySize);
	}
	if (bodyContents == NULL) {
		error("Couldn't allocate body contents (1K)");
		return;
	}

    
    gHasPushBackToken = 0;
	longMultiplyUsed = false;
	longUnsignedDivideUsed = false;
	longSignedDivideUsed = false;
	longUnsignedRemainderUsed = false;
	longSignedRemainderUsed = false;

	if (argumentListData == NULL) {
		argumentListMax = 32;
		argumentListData = MemHandleNew(argumentListMax * sizeof(Declaration));
		argumentList = MemHandleLock(argumentListData);
	}
    argumentListTop = 0;
	if (fieldListData == NULL) {
		fieldListMax = 32;
		fieldListData = MemHandleNew(fieldListMax * sizeof(Declaration));
		fieldList = MemHandleLock(fieldListData);
	}
    fieldListTop = 0;
	
    inlineListTop = 0;
    macroListTop = 0;
    textStackTop = 0;
    macroEndPtr = NULL;

	if (fullGlobalCount > 0) {
		for (i = 0; i < fullGlobalCount; i++) {
			MemHandleUnlock(fullGlobalTableH[i]);
			DmReleaseRecord(gGlobalsDB, fullGlobalRecordIndex[i], false);
		}
		MemHandleUnlock(globalTableH);
		DmReleaseRecord(gGlobalsDB, globalRecordIndex, false);
		DmCloseDatabase(gGlobalsDB);
		deleteGlobalData();
		initGlobalData();	
	}
	else
		globalTop = 0;
        
    currentFunctionType = -1;
    
    structResult = makeDeclaration(StructResultName, StructResultNameLength, -1, NULL, GlobalScope, StructResultNameHash, false);
	setDepth(structResult, GlobalScope, 0);

	if (tempNameBase != NULL) {
		TempNamePool *t = tempNameBase;
		while (t != NULL) {
			TempNamePool *next = t->next;
			MemPtrFree(t);
			t = next;
		}
		tempNameBase = NULL;
	}
	gFunctionCount = 0;
	
/*	if (gHB != NULL) {
		for (i = 0; i < 128; i++) {
			gMacroHashTable[i] = gHB->macroHashTable[i];
			gGlobalHashTable[i] = NULL;
		}
	}
	else {
		for (i = 0; i < 128; i++) {
			gMacroHashTable[i] = -1;
			gGlobalHashTable[i] = NULL;
		}
	}
*/
	for (i = 0; i < 128; i++) {
		gMacroHashTable[i] = -1;
		gGlobalHashTable[i] = NULL;
	}

	asmCount = 1;
	resetPoolIndexList();
	errorCount = 0;	
	gCurSegment = 1;		// the default code segment is code 1. code 0 is used for initialization data
	gLineCount = 0; // reset the line counter after a compile, 
	                //to avoid getting it wrong after building OnBoardHeader.

	gMatchReservedWords = true;
	gSkipTop = 0;
	gSkipState[gSkipTop] = DontSkip;		

	setDefaultMacros();
}

