/***********************************************************************
 *
 * This file is part of the OnBoard C package.  For more information on 
 * using OnBoard C, see http://groups.yahoo.com/group/onboardc.  For 
 * more information on developing OnBoard C (including submission of 
 * bug reports), see http://sourceforge.net/projects/onboardc.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 ***********************************************************************/

#include <PalmOS.h>
#include <PalmCompatibility.h>

#include "OnBoardC.h"

/* Multi-Segment Defines */
#include "types-seg.h"

/* End of Multi-segment defines */

Boolean isFunction(int index)
{
	if (gHB == NULL)
		return (typeTable[index].flags == Function);
	else
		if (index < gHB->typeCount)
			return (gHB->typeList[index].flags == Function);
		else
			return (typeTable[index - gHB->typeCount].flags == Function);
}

Boolean isPointer(int index)
{
	if (gHB == NULL)
		return (typeTable[index].flags == Pointer);
	else
		if (index < gHB->typeCount)
			return (gHB->typeList[index].flags == Pointer);
		else
			return (typeTable[index - gHB->typeCount].flags == Pointer);
}

Boolean isArray(int index)
{
	if (gHB == NULL)
		return (typeTable[index].flags == Array);
	else
		if (index < gHB->typeCount)
			return (gHB->typeList[index].flags == Array);
		else
			return (typeTable[index - gHB->typeCount].flags == Array);
}

Boolean isIntegral(int index)
{
	return ((index == IntTypeIndex)
				|| (index == UnsignedIntTypeIndex)
				|| (index == LongTypeIndex)
				|| (index == UnsignedLongTypeIndex)
				|| (index == ShortTypeIndex)
				|| (index == UnsignedShortTypeIndex)
				|| (index == CharTypeIndex)
				|| (index == UnsignedCharTypeIndex));
}

Boolean isSigned(int index)
{
	return ((index == IntTypeIndex)
				|| (index == LongTypeIndex)
				|| (index == ShortTypeIndex)
				|| (index == CharTypeIndex));
}

Boolean isStruct(int index)
{
	if (gHB == NULL)
		return (typeTable[index].flags == Structure);
	else
		if (index < gHB->typeCount)
			return (gHB->typeList[index].flags == Structure);
		else
			return (typeTable[index - gHB->typeCount].flags == Structure);
}

Boolean isEnum(int index)
{
	if (gHB == NULL)
		return (typeTable[index].flags == Enumeration);
	else
		if (index < gHB->typeCount)
			return (gHB->typeList[index].flags == Enumeration);
		else
			return (typeTable[index - gHB->typeCount].flags == Enumeration);
}

Boolean isStructOrEnum(int index)
{
	if (gHB == NULL)
		return ((typeTable[index].flags == Structure)
					|| (typeTable[index].flags == Enumeration));
	else
		if (index < gHB->typeCount)
			return ((gHB->typeList[index].flags == Structure)
						|| (gHB->typeList[index].flags == Enumeration));
		else
			return ((typeTable[index - gHB->typeCount].flags == Structure)
						|| (typeTable[index - gHB->typeCount].flags == Enumeration));
}

void fixTypeSizes(int index)
{
	if (isArray(index)) {
		if (gHB == NULL) {
			fixTypeSizes(typeTable[index].baseType);
			if (typeTable[index].data.value != -1)
				typeTable[index].size = typeTable[index].data.value * getTypeSize(typeTable[index].baseType);
		}
		else {
			if (index < gHB->typeCount) {
				fixTypeSizes(gHB->typeList[index].baseType);
				if (gHB->typeList[index].data.value != -1)
					gHB->typeList[index].size = gHB->typeList[index].data.value * getTypeSize(gHB->typeList[index].baseType);
			}
			else {
				fixTypeSizes(typeTable[index - gHB->typeCount].baseType);
				if (typeTable[index - gHB->typeCount].data.value != -1)
					typeTable[index - gHB->typeCount].size = typeTable[index - gHB->typeCount].data.value * getTypeSize(typeTable[index - gHB->typeCount].baseType);
			}
		}
	}
}

Boolean matchTypes(int index1, int index2)
{
	if (index1 == index2)
		return true;
	else {
		int base1 = getBaseType(index1);
		int base2 = getBaseType(index2);
		if (isPointer(index1))
			if (isPointer(index2)) {
				if ((base1 == VoidTypeIndex) || (base2 == VoidTypeIndex))
					return true;
				else
					return matchTypes(base1, base2);
			}
			else
				return false;
		else
			if (isPointer(index2))
				return false;
			else
				if (isStruct(index1) || isStruct(index2))
					return false;
				else
					if (isEnum(index1))
						return matchTypes(base1, index2);
					else
						if (isEnum(index2))
							return matchTypes(base2, index1);
						else
							if (isFunction(index1)) {
								int i;
								if (!isFunction(index2))
									return false;
								if (!matchTypes(base1, base2))
									return false;
								if (getTypeSize(index1) != getTypeSize(index2))
									return false;
								for (i = 0; i < getTypeSize(index1); i++) {
									if (!matchTypes(getArgTypeIndex(index1, i), getArgTypeIndex(index2, i)))
										return false;
								}
								return true;
							}
							else
								if (isFunction(index2))
									return false;
								else
									return true;
	}									
}

void setDimension(int index, int dimension)
{
	if (gHB == NULL) {
		typeTable[index].data.value = dimension;
		typeTable[index].size = dimension * getTypeSize(typeTable[index].baseType);
	}
	else
		if (index < gHB->typeCount) {
			gHB->typeList[index].data.value = dimension;
			gHB->typeList[index].size = dimension * getTypeSize(gHB->typeList[index].baseType);
		}
		else {
			typeTable[index - gHB->typeCount].data.value = dimension;
			typeTable[index - gHB->typeCount].size = dimension * getTypeSize(typeTable[index - gHB->typeCount].baseType);
		}
}

int getDimension(int index)
{
	if (gHB == NULL)
		return typeTable[index].data.value;
	else
		if ((gHB != NULL) && (index < gHB->typeCount))
			return gHB->typeList[index].data.value;
		else
			return typeTable[index - gHB->typeCount].data.value;
}

void setArguments(int index, int argStart, Boolean isVarArg)
{
	int argIndex = argStart;
	if (gHB != NULL) argIndex += gHB->argCount;
	
	if (gHB == NULL) {
		typeTable[index].data.index = argIndex;
		typeTable[index].size = argumentListTop - argStart;
		if (isVarArg) typeTable[index].size |= 0x8000;
	}
	else {
		if (index < gHB->typeCount) {
			gHB->typeList[index].data.index = argIndex;
			gHB->typeList[index].size = argumentListTop - argStart;
			if (isVarArg) gHB->typeList[index].size |= 0x8000;
		}
		else {
			typeTable[index - gHB->typeCount].data.index = argIndex;
			typeTable[index - gHB->typeCount].size = argumentListTop - argStart;
			if (isVarArg) typeTable[index - gHB->typeCount].size |= 0x8000;
		}
	}	
}

void setFields(int index, int fieldStart, int theSize)
{
	if (gHB == NULL) {
		typeTable[index].data.index = fieldStart;
		typeTable[index].baseType = fieldListTop - fieldStart;
		typeTable[index].size = theSize;
	}
	else {
		if (index < gHB->typeCount) {
			Type t = gHB->typeList[index];
			t.data.index = fieldStart;
			t.baseType = fieldListTop - fieldStart;
			t.size = theSize;
			DmWrite(gHB, (char *)(&gHB->typeList[index]) - (char *)gHB, &t, sizeof(Type));
/*			
			gHB->typeList[index].data.index = fieldStart;
			gHB->typeList[index].baseType = fieldListTop - fieldStart;
			gHB->typeList[index].size = theSize;
*/			
		}
		else {
			typeTable[index - gHB->typeCount].data.index = fieldStart;
			typeTable[index - gHB->typeCount].baseType = fieldListTop - fieldStart;
			typeTable[index - gHB->typeCount].size = theSize;
		}
	}
}

void setEnumBaseType(int index, int baseType)
{
	int baseSize = getTypeSize(baseType);
	if (gHB == NULL) {
		typeTable[index].baseType = baseType;
		typeTable[index].size = baseSize;
	}
	else {
		if (index < gHB->typeCount) {
			gHB->typeList[index].baseType = baseType;
			gHB->typeList[index].size = baseSize;
		}
		else {
			typeTable[index - gHB->typeCount].baseType = baseType;
			typeTable[index - gHB->typeCount].size = baseSize;
		}
	}
}

int getArgTypeIndex(int index, int argIndex)
{
	int baseIndex;
	if (gHB == NULL)
		baseIndex = typeTable[index].data.index;
	else
		if ((gHB != NULL) && (index < gHB->typeCount))
			baseIndex = gHB->typeList[index].data.index;
		else
			baseIndex = typeTable[index - gHB->typeCount].data.index;
	
	if (gHB != NULL) {
		if (baseIndex >= gHB->argCount)
			return argumentList[baseIndex - gHB->argCount + argIndex].typeIndex;
		else
			return gHB->argList[baseIndex + argIndex].typeIndex;
	}
	else
		return argumentList[baseIndex + argIndex].typeIndex;
}

Declaration *getArg(int index, int argIndex)
{
	int baseIndex;	
	if (gHB == NULL)
		baseIndex = typeTable[index].data.index;
	else
		if ((gHB != NULL) && (index < gHB->typeCount))
			baseIndex = gHB->typeList[index].data.index;
		else
			baseIndex = typeTable[index - gHB->typeCount].data.index;

	if (gHB != NULL) {
		if (baseIndex >= gHB->argCount)
			return &argumentList[baseIndex - gHB->argCount + argIndex];
		else
			return &gHB->argList[baseIndex + argIndex];
	}
	else
		return &argumentList[baseIndex + argIndex];
}

Type *getType(int index)
{
	if (gHB == NULL)
		return &typeTable[index];
	else
		if (index < gHB->typeCount)
			return &gHB->typeList[index];
		else
			return &typeTable[index - gHB->typeCount];
}

int getBaseType(int index)
{
	if (gHB == NULL)
		return typeTable[index].baseType;
	else
		if (index < gHB->typeCount)
			return gHB->typeList[index].baseType;
		else
			return typeTable[index - gHB->typeCount].baseType;
}

void setInlineFunction(Declaration *d, int inlineBase, int inlineSize)
{
	if (gHB != NULL)
		setDataField(d, GlobalScope, inlineBase + gHB->inlineCount);
	else
		setDataField(d, GlobalScope, inlineBase);
	setBitWidth(d, GlobalScope, inlineSize);
}

Boolean isInlineFunction(Declaration *d)
{
	return (d->data.nextField != -1);
}

int getInlineCount(Declaration *d)
{
	return d->bitWidth;
}

int getInlineWord(Declaration *d, int index)
{
	if (gHB != NULL) {
		if (d->data.nextField >= gHB->inlineCount) 
			return inlineList[d->data.nextField - gHB->inlineCount + index];
		else
			return gHB->inlineList[d->data.nextField + index];
	}
	else
		return inlineList[d->data.nextField + index];
}

int addInline(int inlineValue)
{
	int result = inlineListTop;
	if (inlineListTop == MaxInlines) {
		error("Too many inline values");
		return -1;
	}
	inlineList[inlineListTop++] = inlineValue;
	return result;
}

int newStructType()
{
	int result;
	int adjust = 0;
	if (gHB != NULL) adjust = gHB->typeCount;

	if (typeIndex == MaxTypeIndex) {
		MemHandleUnlock(typeHandle);
		MaxTypeIndex += 50;
		if (MemHandleResize(typeHandle, (MaxTypeIndex - adjust) * sizeof(Type)) != 0) {
			error("Out of type table entries");
			return NULL;
		}
		typeTable = MemHandleLock(typeHandle);
	}
	result = typeIndex++;
	typeTable[result - adjust].flags = Structure;
	typeTable[result - adjust].data.index = -1;
	typeTable[result - adjust].size = 0;
	newPointerType(result);
	return result;
}

int newEnumType()
{
	int result;
	int adjust = 0;
	if (gHB != NULL) adjust = gHB->typeCount;

	if (typeIndex == MaxTypeIndex) {
		MemHandleUnlock(typeHandle);
		MaxTypeIndex += 50;
		if (MemHandleResize(typeHandle, (MaxTypeIndex - adjust) * sizeof(Type)) != 0) {
			error("Out of type table entries");
			return NULL;
		}
		typeTable = MemHandleLock(typeHandle);
	}
	result = typeIndex++;
	typeTable[result - adjust].flags = Enumeration;
	typeTable[result - adjust].baseType = IntTypeIndex;
	typeTable[result - adjust].size = 2;
	return result;
}

int newType()
{
	int result;
	int adjust = 0;
	if (gHB != NULL) adjust = gHB->typeCount;

	if (typeIndex == MaxTypeIndex) {
		MemHandleUnlock(typeHandle);
		MaxTypeIndex += 50;
		if (MemHandleResize(typeHandle, (MaxTypeIndex - adjust) * sizeof(Type)) != 0) {
			error("Out of type table entries");
			return NULL;
		}
		typeTable = MemHandleLock(typeHandle);
	}
	result = typeIndex++;
	typeTable[result - adjust].flags = Base;
	typeTable[result - adjust].baseType = -1;
	typeTable[result - adjust].size = 0;
	return result;
}

int newPointerType(int theType)
{
	int result;
	int adjust = 0;
	if (gHB != NULL) adjust = gHB->typeCount;
	
	if (theType <= HighScalarType) {
		return theType + BaseScalarPtrTypeIndex;
	}
	else {
		if (theType == CharPtrTypeIndex)
			return CharPtrPtrTypeIndex;
		if (theType < (typeIndex - 1)) {
			if (isPointer(theType + 1)
					&& (getBaseType(theType + 1) == theType))
				return theType + 1;
		}		
		if (typeIndex == MaxTypeIndex) {
			MemHandleUnlock(typeHandle);
			MaxTypeIndex += 50;
			if (MemHandleResize(typeHandle, (MaxTypeIndex - adjust) * sizeof(Type)) != 0) {
				error("Out of type table entries");
				return NULL;
			}
			typeTable = MemHandleLock(typeHandle);
		}
		result = typeIndex++;
		typeTable[result - adjust].flags = Pointer;
		typeTable[result - adjust].baseType = theType;
		typeTable[result - adjust].size = 4;
		return result;
	}
}

int newArrayType(int theType)
{
	int result;
	int adjust = 0;
	if (gHB != NULL) adjust = gHB->typeCount;

	if (typeIndex == MaxTypeIndex) {
		MemHandleUnlock(typeHandle);
		MaxTypeIndex += 50;
		if (MemHandleResize(typeHandle, (MaxTypeIndex - adjust) * sizeof(Type)) != 0) {
			error("Out of type table entries");
			return NULL;
		}
		typeTable = MemHandleLock(typeHandle);
	}
	result = typeIndex++;
	typeTable[result - adjust].flags = Array;
	typeTable[result - adjust].baseType = theType;
	typeTable[result - adjust].size = 4;
	typeTable[result - adjust].data.value = -1;
	return result;
}

int newFunctionType(int theType)
{
	int result;
	int adjust = 0;
	if (gHB != NULL) adjust = gHB->typeCount;

	if (typeIndex == MaxTypeIndex) {
		MemHandleUnlock(typeHandle);
		MaxTypeIndex += 50;
		if (MemHandleResize(typeHandle, (MaxTypeIndex - adjust) * sizeof(Type)) != 0) {
			error("Out of type table entries");
			return NULL;
		}
		typeTable = MemHandleLock(typeHandle);
	}
	result = typeIndex++;
	typeTable[result - adjust].flags = Function;
	typeTable[result - adjust].baseType = theType;
	typeTable[result - adjust].size = -1;
	newPointerType(result);
	return result;
}

Declaration *getFirstField(int index)
{
	int fieldListStart;
	
	if (gHB == NULL)
		fieldListStart = typeTable[index].data.index;
	else
		if (index < gHB->typeCount)
			fieldListStart = gHB->typeList[index].data.index;
		else
			fieldListStart = typeTable[index - gHB->typeCount].data.index;
			
	if (fieldListStart == -1) return NULL;

	if ((gHB != NULL) && (index < gHB->typeCount))
		return &gHB->fieldList[fieldListStart];
	else
		return &fieldList[fieldListStart];
}

Boolean isEmptyStruct(int index)
{
	if (gHB == NULL)
		return (typeTable[index].data.index == -1);
	else
		if (index < gHB->typeCount)
			return (gHB->typeList[index].data.index == -1);
		else
			return (typeTable[index - gHB->typeCount].data.index == -1);
}

Declaration *getNextField(int index, Declaration *previousField)
{
	int nextField = previousField->data.nextField;
	if (nextField == -1)
		return NULL;
	else
		if ((gHB != NULL) && (index < gHB->typeCount))
			return &gHB->fieldList[nextField];
		else
			return &fieldList[nextField];
}

Declaration *findField(int index, CharPtr name, int length)
{
	Declaration *theFieldList = fieldList;	
	Declaration *theField;
	int fieldListStart;
	int fieldCount;

	if (gHB == NULL) {
		fieldListStart = typeTable[index - gHB->typeCount].data.index;
		fieldCount = typeTable[index - gHB->typeCount].baseType;
	}
	else {
		if (index < gHB->typeCount) {
			fieldListStart = gHB->typeList[index].data.index;
			fieldCount = gHB->typeList[index].baseType;
		}
		else {
			fieldListStart = typeTable[index - gHB->typeCount].data.index;
			fieldCount = typeTable[index - gHB->typeCount].baseType;
		}
	}

	if (fieldListStart == -1) return NULL;	
	
	if ((gHB != NULL) && (index < gHB->typeCount))
		theFieldList = gHB->fieldList;
	theField = &theFieldList[fieldListStart];
	while (true) {
		if (theField->length == length) {
			int x;
			for (x = 0; x < length; x++) {
				if (theField->name[x] != name[x])
					break;
			}
			if (x == length) {
				return theField;
			}
		}		
		if (theField->data.nextField == -1)
			break;
		theField = &theFieldList[theField->data.nextField];
	}
	return NULL;
}

int findEnum(CharPtr name, int length, int scope)
{
	int i;
	if (scope == LocalScope) {
		for (i = localTop - 1; i >= 0; i--) {
			if (localDeclaration[i].typeNameSpace
						&& isEnum(localDeclaration[i].typeIndex)) {
				if (length == localDeclaration[i].length) {
					int x;
					for (x = 0; x < length; x++)
						if (localDeclaration[i].name[x] != name[x])
							break;
					if (x == length) {
						return localDeclaration[i].typeIndex;
					}
				}
			}
		}
	}	
	for (i = 0; i < globalTop; i++) {		
		if (globalTable[i].typeNameSpace
					&& isEnum(globalTable[i].typeIndex)) {
			if (length == globalTable[i].length) {
				int x;
				for (x = 0; x < length; x++)
					if (globalTable[i].name[x] != name[x])
						break;
				if (x == length) {
					return globalTable[i].typeIndex;
				}
			}
		}
	}
	for (i = 0; i < fullGlobalCount; i++) {
		int j;
		for (j = 0; j < MaxGlobal; j++) {		
			if (fullGlobalTable[i][j].typeNameSpace
						&& isEnum(fullGlobalTable[i][j].typeIndex)) {
				if (length == fullGlobalTable[i][j].length) {
					int x;
					for (x = 0; x < length; x++)
						if (fullGlobalTable[i][j].name[x] != name[x])
							break;
					if (x == length) {
						return fullGlobalTable[i][j].typeIndex;
					}
				}
			}
		}
	}
	if (gHB != NULL) {
		for (i = 0; i < gHB->globalCount; i++) {		
			if (gHB->globalList[i].typeNameSpace
						&& isEnum(gHB->globalList[i].typeIndex)) {
				if (length == gHB->globalList[i].length) {
					int x;
					for (x = 0; x < length; x++)
						if (gHB->globalList[i].name[x] != name[x])
							break;
					if (x == length) {
						return gHB->globalList[i].typeIndex;
					}
				}
			}
		}
	}
	return -1;	
}

char SysTrapName[8] = "sysTrap";

Declaration *gLocalHashTable[128];
Declaration *gGlobalHashTable[128];

Declaration *findIdentifier(CharPtr name, int length, char *reg, unsigned char hash, Declaration *proxy)
{
	int i;
	int x;
	Declaration *dp;

	if (currentFunctionType != -1) {
		int currentFunctionSize;
/* */
		dp = gLocalHashTable[hash];
		while (dp != NULL) {
			if (dp != proxy)
				if (!dp->typeNameSpace) {
					if (length == dp->length)  {
						for (x = 0; x < length; x++)
							if (dp->name[x] != name[x])
								break;
						if (x == length) {
							*reg = A6;
							return dp;
						}
					}
				}
			dp = dp->nextDecl;
		}
/* */		
/*		
		for (i = localTop - 1; i >= 0; i--) {
			if (!localDeclaration[i].typeNameSpace) {
				if ((length == localDeclaration[i].length)
						&& (hash == localDeclaration[i].hash))  {
					for (x = 0; x < length; x++)
						if (localDeclaration[i].name[x] != name[x])
							break;
					if (x == length) {
						*reg = A6;
						return &localDeclaration[i];
					}
				}
			}
		}
*/		
		currentFunctionSize = getTypeSize(currentFunctionType);
		for (i = 0; i < currentFunctionSize; i++) {
			Declaration *arg = getArg(currentFunctionType, i);
			if (arg != proxy) {
				if (arg->length != 0) {
					if ((length == arg->length)
							&& (hash == arg->hash)) {
						for (x = 0; x < length; x++)
							if (arg->name[x] != name[x])
								break;
						if (x == length) {
							*reg = A6;
							return arg;
						}
					}
				}
			}
		}
	}
/*	*/
	dp = gGlobalHashTable[hash];
	while (dp != NULL) {
		if (dp != proxy)
			if (!dp->typeNameSpace) {
				if (length == dp->length) {
					for (x = 0; x < length; x++)
						if (dp->name[x] != name[x])
							break;
					if (x == length) {
						if (isFunction(dp->typeIndex))
							*reg = PC;
						else
							*reg = A5;
						return dp;
					}
				}
			}
		dp = dp->nextDecl;
	}
/*
	for (i = 0; i < globalTop; i++) {		
		if (!globalTable[i].typeNameSpace) {
			if ((length == globalTable[i].length)
					&& (hash == globalTable[i].hash)) {
				for (x = 0; x < length; x++)
					if (globalTable[i].name[x] != name[x])
						break;
				if (x == length) {
					if (isFunction(globalTable[i].typeIndex))
						*reg = PC;
					else
						*reg = A5;
					return &globalTable[i];
				}
			}
		}
	}
*/	
	if (gSysTrapHandle != NULL) {
		if (length > 7) {
			for (i = 0; i < 7; i++) {
				if (name[i] != SysTrapName[i])
					break;
			}
			if (i == 7) {
				int subLength = length - 7;
				for (i = 0; i < gSysTrapTop; i++) {
					if ((subLength == gSysTrapList[i].length)
							&& (hash == gSysTrapList[i].hash)) {
						for (x = 0; x < subLength; x++)
							if (gSysTrapList[i].name[x] != name[x + 7])
								break;
						if (x == subLength) {
							*reg = A5;
							return &gSysTrapList[i];
						}
					}
				}
			}
		}
	}
	if (gHB != NULL) {
		int hashStartIndex;
		int hashEndIndex;
		if (length > 7) {
			for (i = 0; i < 7; i++) {
				if (name[i] != SysTrapName[i])
					break;
			}
			if (i == 7) {
				int subLength = length - 7;
				for (i = 0; i < gHB->sysTrapCount; i++) {
					if ((subLength == gHB->sysTrapList[i].length)
							&& (hash == gHB->sysTrapList[i].hash)) {
						for (x = 0; x < subLength; x++)
							if (gHB->sysTrapList[i].name[x] != name[x + 7])
								break;
						if (x == subLength) {
							*reg = A5;
							return &gHB->sysTrapList[i];
						}
					}
				}
			}
		}

		hashStartIndex = gHB->hashEntryIndex[hash];
		if (hash == 127)
			hashEndIndex = gHB->globalCount;
		else
			hashEndIndex = gHB->hashEntryIndex[hash + 1];

		for (i = hashStartIndex; i < hashEndIndex; i++) {		
			if (!gHB->globalList[i].typeNameSpace) {
				if ((length == gHB->globalList[i].length)
						&& (hash == gHB->globalList[i].hash)) {
					for (x = 0; x < length; x++)
						if (gHB->globalList[i].name[x] != name[x])
							break;
					if (x == length) {
						if (isFunction(gHB->globalList[i].typeIndex))
							*reg = PC;
						else
							*reg = A5;
						return &gHB->globalList[i];
					}
				}
			}
		}
	}
	return NULL;	
}

int findStruct(CharPtr name, int length, int scope, unsigned char hash)
{
	int i;
	int x;
	if (scope == LocalScope) {
		for (i = localTop - 1; i >= 0; i--) {
			if (localDeclaration[i].typeNameSpace
						&& isStruct(localDeclaration[i].typeIndex)) {
				if ((hash == localDeclaration[i].hash)
						&& (length == localDeclaration[i].length)) {
					for (x = 0; x < length; x++)
						if (localDeclaration[i].name[x] != name[x])
							break;
					if (x == length) {
						return localDeclaration[i].typeIndex;
					}
				}
			}
		}
	}
	for (i = 0; i < globalTop; i++) {		
		if (globalTable[i].typeNameSpace
					&& isStruct(globalTable[i].typeIndex)) {
			if ((hash == globalTable[i].hash) &&
						(length == globalTable[i].length)) {
				for (x = 0; x < length; x++)
					if (globalTable[i].name[x] != name[x])
						break;
				if (x == length) {
					return globalTable[i].typeIndex;
				}
			}
		}
	}
	for (i = 0; i < fullGlobalCount; i++) {
		int j;
		for (j = 0; j < globalTop; j++) {		
			if (fullGlobalTable[i][j].typeNameSpace
						&& isStruct(fullGlobalTable[i][j].typeIndex)) {
				if ((hash == fullGlobalTable[i][j].hash) &&
							(length == fullGlobalTable[i][j].length)) {
					for (x = 0; x < length; x++)
						if (fullGlobalTable[i][j].name[x] != name[x])
							break;
					if (x == length) {
						return fullGlobalTable[i][j].typeIndex;
					}
				}
			}
		}
	}
	if (gHB != NULL) {
		for (i = 0; i < gHB->globalCount; i++) {		
			if (gHB->globalList[i].typeNameSpace
						&& isStruct(gHB->globalList[i].typeIndex)) {
				if ((hash == gHB->globalList[i].hash) &&
						(length == gHB->globalList[i].length)) {
					for (x = 0; x < length; x++)
						if (gHB->globalList[i].name[x] != name[x])
							break;
					if (x == length) {
						return gHB->globalList[i].typeIndex;
					}
				}
			}
		}
	}
	return -1;	
}

int getTypeSize(int index)
{
	if (gHB == NULL)
		return typeTable[index].size;
	else
		if (index < gHB->typeCount)
			return gHB->typeList[index].size;
		else
			return typeTable[index - gHB->typeCount].size;
}

void assureTypeTable()
{
	int adjust = 0;
	if (gHB != NULL) adjust = gHB->typeCount;

	if (typeIndex >= (MaxTypeIndex - 10)) {
		MemHandleUnlock(typeHandle);
		MaxTypeIndex += 50;
		if (MemHandleResize(typeHandle, (MaxTypeIndex - adjust) * sizeof(Type)) != 0) {
			error("Out of type table entries");
			return;
		}
		typeTable = MemHandleLock(typeHandle);
	}
}

int *getBaseTypePtr(int index)
{
	if (gHB == NULL) {
		assureTypeTable();
		return &typeTable[index].baseType;
	}
	else
		if (index < gHB->typeCount)
			return &gHB->typeList[index].baseType;
		else {
			assureTypeTable();
			return &typeTable[index - gHB->typeCount].baseType;
		}
}

void replaceType(int index, int oldType, int newType)
{
	while (true) {
		int *baseTypePtr = getBaseTypePtr(index);
		if (*baseTypePtr == -1) {
			error("invalid type replacement (internal error)");
			return;
		}
		else
			if (*baseTypePtr == oldType) {
				*baseTypePtr = newType;
				return;
			}
			else
				index = *baseTypePtr;
	}
}			
		
		
void setBitWidth(Declaration *declPtr, int declScope, char value)
{
	if (declScope == GlobalScope) {
		char *base = (char *)findGlobalDeclBase(declPtr);
		DmWrite(base, ((char *)&declPtr->bitWidth) - base, &value, sizeof(char));
	}
	else
		declPtr->bitWidth = value;
}

void setDataValue(Declaration *declPtr, int declScope, long value)
{
	if (declScope == GlobalScope) {
		char *base = (char *)findGlobalDeclBase(declPtr);
		DmWrite(base, ((char *)&declPtr->data.value) - base, &value, sizeof(long));
	}
	else
		declPtr->data.value = value;
}

void setDataField(Declaration *declPtr, int declScope, int value)
{
	if (declScope == GlobalScope) {
		char *base = (char *)findGlobalDeclBase(declPtr);
		DmWrite(base, ((char *)&declPtr->data.nextField) - base, &value, sizeof(int));
	}
	else
		declPtr->data.nextField = value;
}

void setDepth(Declaration *declPtr, int declScope, int value)
{
	if (declScope == GlobalScope) {
		char *base = (char *)findGlobalDeclBase(declPtr);
		DmWrite(base, ((char *)&declPtr->depth) - base, &value, sizeof(int));
	}
	else
		declPtr->depth = value;
}

void setStatic(Declaration *declPtr, int declScope, char value)
{
	if (declScope == GlobalScope) {
		char *base = (char *)findGlobalDeclBase(declPtr);
		DmWrite(base, ((char *)&declPtr->isStatic) - base, &value, sizeof(char));
	}
	else
		declPtr->isStatic = value;
}

void setTypeIndex(Declaration *declPtr, int declScope, int value)
{
	if (declScope == GlobalScope) {
		char *base = (char *)findGlobalDeclBase(declPtr);
		DmWrite(base, ((char *)&declPtr->typeIndex) - base, &value, sizeof(int));
	}
	else
		declPtr->typeIndex = value;
}

void setTypeDef(Declaration *declPtr, int declScope, Boolean value)
{
	if (declScope == GlobalScope) {
		char *base = (char *)findGlobalDeclBase(declPtr);
		DmWrite(base, ((char *)&declPtr->isTypedef) - base, &value, sizeof(Boolean));
	}
	else
		declPtr->isTypedef = value;
}

int gHeaderPoolIndexList[128];
int gGlobalPoolIndexList[128];
int gLocalPoolIndexList[MaxLocals];
int gArgumentPoolIndexList[128];
int gPoolIndex = 0;


void resetPoolIndexList()
{
	int i;
	for (i = 0; i < 128; i++) {
		gHeaderPoolIndexList[i] = -1;
		gGlobalPoolIndexList[i] = -1;
		gArgumentPoolIndexList[i] = -1;
	}
	gPoolIndex = 0;
}

Boolean getPoolIndex(Declaration *declPtr, int *poolIndex)
{
	int declIndex;
	*poolIndex = -1;
	
	if (declPtr->depth < -1) return false;
	
	if ((gHB != NULL) && ((char *)declPtr >= (char *)(gHB->globalList)) && ((char *)declPtr < (char *)(gHB->typeList))) {	
		declIndex = declPtr - gHB->globalList;
		if ((declIndex >= 0) && (declIndex < 128) && (declIndex < gHB->globalCount)) {
			*poolIndex = gHeaderPoolIndexList[declIndex];
			if (*poolIndex == -1) {
				gHeaderPoolIndexList[declIndex] = gPoolIndex++;
				*poolIndex = gHeaderPoolIndexList[declIndex];
				return false;
			}
			else
				return true;
		}
		else
			return false;
	}
	else {
		// don't support pool ids for globals beyond the initial global
		// data chunk.
		Declaration *declBase = findGlobalDeclBase(declPtr);
		if (declBase) {
			if ((fullGlobalCount == 0) || (declBase == fullGlobalTable[0])) {
				declIndex = declPtr - declBase;
				if ((declIndex >= 0) && (declIndex < 128) && (declIndex < globalTop)) {
					*poolIndex = gGlobalPoolIndexList[declIndex];
					if (*poolIndex == -1) {
						gGlobalPoolIndexList[declIndex] = gPoolIndex++;
						*poolIndex = gGlobalPoolIndexList[declIndex];
						return false;
					}
					else
						return true;
				}
			}
		}
		declIndex = declPtr - localDeclaration;
		if ((declIndex >= 0) && (declIndex < localTop)) {
			*poolIndex = gLocalPoolIndexList[declIndex];
			if (*poolIndex == -1) {
				gLocalPoolIndexList[declIndex] = gPoolIndex++;
				*poolIndex = gLocalPoolIndexList[declIndex];
				return false;
			}
			else
				return true;
		}
		declIndex = declPtr - argumentList;
		if ((declIndex >= 0) && (declIndex < argumentListTop) && (declIndex < 128)) {
			*poolIndex = gArgumentPoolIndexList[declIndex];
			if (*poolIndex == -1) {
				gArgumentPoolIndexList[declIndex] = gPoolIndex++;
				*poolIndex = gArgumentPoolIndexList[declIndex];
				return false;
			}
			else
				return true;
		}
		else
			return false;
	}
}
		
		
		
		
		
