/*
 *	File:		args.c
 *	Author:		Wade Guthrie
 *	Created:	Oh, a long time ago
 *
 *	Description:	
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>

#include "args.h"
#include "debug.h"

static int	gWhichArg	= 1;
static YesNo	gGoodGroup	= No;		/* flag: on if valid */
						/* argument group */

	ArgStatus
ArgsParse (
	ArgumentType	*typeList,
	int		argCount,
	char		**argVector)
{
#	define	FUNCTION	"ArgsParse"
	char		*arg		= 0;
	ArgStatus	status		= ARG_STAT_OKAY;

	if ((typeList == 0) || (argVector == 0))
	{
		status		= ARG_STAT_INTERNAL_ERROR;
	}
	else
	{
		ArgumentType	*type		= 0;
		int		i		= 0;
		YesNo		moreTypes	= Yes,
				foundOne	= No;

		// for each argument

		while ((status == ARG_STAT_OKAY)
			&& ((arg = ArgsGetArg	(argCount, 
						 argVector, 
						 &status)) != 0))
		{
			i		= 0;
			moreTypes	= Yes;
			foundOne	= No;

			// check argument against each argument type

			while (moreTypes)
			{
				type = &typeList[i++]; 

				if ((type->longname == 0) 
						&& (type->shortname == 0))
				{
					moreTypes	= No;
				}
				else if (strlen(arg) > 1)	// it's a long argument
				{
					if ((type->longname != 0)
						&& (strcmp (arg, type->longname) == 0)
						&& (strlen (type->longname) == strlen (arg)))
					{
						moreTypes	= No;
						foundOne	= Yes;
						status		= ArgsHandleType
									  (arg,
									   type,
									   typeList,
									   argCount,
									   argVector);
					}
				}
				else if ((type->shortname != '\0')
					&& (*arg == type->shortname))
				{
					moreTypes	= No;
					foundOne	= Yes;
					status		= ArgsHandleType
								  (arg,
								   type,
								   typeList,
								   argCount,
								   argVector);
				}

			}

			if (!foundOne)
			{
				status = ARG_STAT_UNKNOWN_ARG;
			}
		}
	}

	return status;
#	undef	FUNCTION
}




/*
 * This is the C way to do things rather than the C++ way.  That makes
 * it easier for vanilla C programs to use this.
 */

	ArgStatus
ArgsHandleType	(char const	*arg,
		 ArgumentType	*argType,
		 ArgumentType	*typeList,
		 int		argc,
		 char		**argv)
{
#	define	FUNCTION	"ArgsHandleType"
	YesNo		okSoFar		= Yes;
	ArgStatus	status		= ARG_STAT_OKAY;

	if (argType == 0)
	{
		status	= ARG_STAT_INTERNAL_ERROR;
	}
	else if (argType->optinfo == ARG_PRINT_HELP)
	{
		ArgsPrintHelp	(typeList);
	}
	else if (okSoFar && (argType->opt != 0))
	{
		char	*param	= 0;

		switch (argType->optinfo)
		{
		case ARG_INT_OPT:
			if ((param = ArgsGetParam(argc, argv)) == 0)
			{
				status	= ARG_STAT_NEED_PARAM;
			}
			else 
			{
				*(int *) (argType->opt)	
						= (int) strtol (param, 0, 0);
			}
			break;

		case ARG_LONG_OPT:
			if ((param = ArgsGetParam(argc, argv)) == 0)
			{
				status	= ARG_STAT_NEED_PARAM;
			}
			else 
			{
				*(long *) (argType->opt) 
						= strtol (param, 0, 0);
			}
			break;

		case ARG_STRING_OPT:
			if ((param = ArgsGetParam(argc, argv)) == 0)
			{
				status	= ARG_STAT_NEED_PARAM;
			}
			else 
			{
				*(char **) (argType->opt) = param;
			}
			break;

		case ARG_SET_FLAG:
			*(YesNo *) (argType->opt)	= Yes;
			break;

		default:
			status	= ARG_STAT_MALFORMED_TYPES;
			break;
		}
	}

	if ((status == ARG_STAT_OKAY) && (argType->func != 0))
	{
		if (!argType->func (argType->opt))
			status	= ARG_STAT_GENERAL_ERROR;
	}

	return	status;
#	undef	FUNCTION
}



	char *
ArgsGetArg	(int		argCount,
		 char		**argVector,
		 ArgStatus	*status)
{
#	define	FUNCTION	"ArgsGetArg"

	static char	charAnswer[3];
	char		*answer		= 0;
	YesNo		keepGoing	= Yes;
	char 		c;			/* the return argument */

	/* body of the program */

	if (status == 0) 			/* input arguments valid? */
	{
		keepGoing	= No;
	}
	else if (argVector == 0) 		/* input arguments valid? */
	{
		*status		= ARG_STAT_INTERNAL_ERROR;
		keepGoing	= No;
	} 
	else if (gWhichArg >= argCount) 	/* do more arguments exist? */
	{
		*status		= ARG_STAT_OKAY;	/* we're just done */
		keepGoing	= No;
	} 
	else if (argVector[gWhichArg] == 0) 	/* is current group ok? */
	{
		*status		= ARG_STAT_INTERNAL_ERROR;
		keepGoing	= No;
	}
	else if(*argVector[gWhichArg] == '-')	/* check for start of a 
						 * valid argument group */
	{
		gGoodGroup	= Yes;
		if(*++argVector[gWhichArg] == NULLchar) 
		{
			*status		= ARG_STAT_NEEDS_ARG;
			keepGoing	= No;
		}
	}

	/* get the argument */

	if (keepGoing)
	{
		*status	= (gGoodGroup) 
				? ARG_STAT_OKAY 
				: ARG_STAT_MALFORMED_ARGS;

		if ((c = *argVector[gWhichArg]++) == '-')
		{
			// We've got a '--' argument

			answer		= argVector[gWhichArg];
			gGoodGroup	= No;
			gWhichArg++;
		}
		else
		{
			// this is a single-character argument

			charAnswer[0]	= c;
			charAnswer[1]	= '\0';

			answer		= charAnswer;


			// are we at the end of the arg group?

			if(*argVector[gWhichArg] == NULLchar)
			{
				gWhichArg++;
				gGoodGroup = No;
			}
		}

		/* clean up and leave */

		if (*status != ARG_STAT_OKAY)
		{
			answer = 0;
		}
	}

	return answer;
#	undef	FUNCTION
}


/*
 * ArgsGetParam passes back the next argument group from its argument list
 * argVector (presumably extracted from argv).  This group is expected to
 * be a parameter to a previous argument.  argVector[argCount] is expected
 * to be pointing to the beginning of a legitimate parameter and not
 * a valid argument (a non-numeric preceded by a hyphen ('-').
 * The two possible input cases are:
 *	1. legitimate parameter -- ArgsGetParam returns a pointer to it.
 *	2. no legitimate parameter -- ArgsGetParam returns a NULL.
 */

	char *
ArgsGetParam	(int	argCount,
		 char	**argVector)
{
#	define	FUNCTION	"ArgsGetParam"


	if (argVector == 0) {			/* input arguments valid? */
		/* fprintf(stderr, "%s: argVector == 0, returning NULL\n", 
		 *	FUNCTION); 
		 */
		return (char *)NULL;

	} else if (gWhichArg >= argCount) {	/* do any (more) arguments 
						 * exist? */
		/* fprintf(stderr, "%s: gWhichArg (%d) >= argCount (%d), "
		 *	"returning NULL\n", FUNCTION, gWhichArg, argCount);
		 */
		return (char *)NULL;

	} else if (argVector[gWhichArg] == 0) {	/* is current group ok? */
		/* fprintf(stderr, 
		 *	"%s: argVector[%d] (%s) == 0, returning NULL\n", 
		 *	FUNCTION, gWhichArg, argVector[gWhichArg]);
		 */
		return (char *)NULL;

	} else if (gGoodGroup) {		
		/* parameter not valid if still in the middle of a group */
		/* fprintf(stderr, 
		 *	"%s: gGoodGroup == TRUE, returning NULL\n", 
		 *	FUNCTION); 
		 */
		return (char *)NULL;
	}

	/* check for start of a valid argument group -- those are bad */

	if(*argVector[gWhichArg] == '-') {
		if(!isdigit(*(argVector[gWhichArg]+1))) {
			/* fprintf(stderr, 
			 * "%s: *ArgV[%d] == '-' and *(ArgV[%d]+1 = '%s') "
			 *	"not a digit, returning NULL\n", 
			 *	FUNCTION, gWhichArg, gWhichArg, 
			 *	argVector[gWhichArg]+1);
			 */
			return (char *)NULL;
		}
	}

	/* get the argument */

	gGoodGroup = No;
	/* fprintf(stderr, "%s: returning '%s'\n", FUNCTION, 
	 *	argVector[gWhichArg]); 
	 */
	return argVector[gWhichArg++];
#	undef	FUNCTION
}


	void		
ArgsPrintHelp	(ArgumentType	*typeList)
{
#	define	FUNCTION	"ArgsPrintHelp"

	if (typeList != 0)
	{
		YesNo	done	= No;
		int	i	= 0;

		for (i=0; !done; i++)
		{
			if ((typeList[i].longname == 0) 
				&& (typeList[i].shortname == '\0'))
			{
				done	= Yes;
			}
			else
			{
				if (typeList[i].documentation != 0)
					printf ("%s", 
					    typeList[i].documentation);
			}
		}
	}
#	undef	FUNCTION
}



	void		
ArgsPrintError	(ArgStatus	status)
{
#	define	FUNCTION	"ArgsPrintError"

	switch (status)
	{
	case ARG_STAT_OKAY:
		printf ("No Error\n");
		break;
	case ARG_STAT_INTERNAL_ERROR:
		printf ("** Internal Error: some pointer got hosed\n");
		break;
	case ARG_STAT_GENERAL_ERROR:
		printf ("** Error: the 'doit' func returned error\n");
		break;
	case ARG_STAT_NEED_PARAM:
		printf ("** Error: an arg needs a parameter\n");
		break;
	case ARG_STAT_UNKNOWN_ARG:
		printf ("** Error: unknown argument\n");
		break;
	case ARG_STAT_NEEDS_ARG:
		printf ("** Error: there was a hypen without an arg\n");
		break;
	case ARG_STAT_MALFORMED_TYPES:
		printf ("** Error: arg TYPE list is messed up\n");
		break;
	case ARG_STAT_MALFORMED_ARGS:
		printf ("** Error: arg list is messed up \n");
		break;
	}
#	undef	FUNCTION
}
