/* 
 * This program converts OnBoard C programs to host programs (for use with the
 * PRC tools) and back again.
 */

/*
 * TODO: Known shortcomings
 *
 *	o  doesn't work between directories
 *	o  doesn't do host->palm
 *		- doesn't convert Makefile->project file
 *	o  doesn't do multi-segment stuff
 *	o  doesn't do icons/bitmaps
 */

#include <stdio.h>
#include <stdlib.h>	// for 'exit'
#include <errno.h>
#include <ctype.h>	// for isprint
#include <string.h>

#include "general.h"
#include "debug.h"
#include "args.h"

/*
 * Defines and typedefs
 */

#define	VERSION			"0.02"
#ifdef NDEBUG
#	define	SPECIAL_VERSION		"ALPHA 1 (No Debug)"
#else
#	define	SPECIAL_VERSION		"ALPHA 1 (Debug Build)"
#endif

#define BYTES_FOR_H_FILE		36	// records for header files are 36 bytes
#define	ExtraPcHeader			90
#define	FileHeader			112
#define FILENAME_SIZE			32
#define APP_NAME_SIZE			64
#define	BYTES_ON_LINE			16	// for showBuf

#define	PROJ_HEAD_FLAG_AUTOVERSION	0x08
#define	PROJ_HEAD_FLAG_DEBUG		0x04
#define	PROJ_HEAD_FLAG_REBUILD		0x02
#define	PROJ_HEAD_FLAG_AUTOEXECUTE	0x01

#define WINDOWS_NAMES 0
#define PILOT_LINK_NAMES 1

typedef unsigned char Byte;



/*
 * ProjectHeader
 *
 * Most of this stuff isn't used by _this_ program.  It's just here for
 * documentation.
 */

typedef struct ProjectHeader
{
	Byte	pcHeader[ExtraPcHeader];
	Byte	filler1[3];		// 24 bits: 00 07 00
	Byte	filesInProject;		// 8 bits: number of files in project
	Byte	filler2[5];		// 8 bits: 00
					// 8 bits: ? I've seen 00, 01, and 0F
					// 24 bits: 00 00 00
	Byte	flags;			// 8 bits: 08=autoversion, 04=debug, 
					//    02=rebuild, 01=execute
	char	creatorId[4];		// 32 bits: creator ID of generated 
					//    executable
	char	appl[4];		// 32 bits: "appl"
	char	filler3[32];		// 32x8 bits: "MyApplication" 00, fill 
					//    with 00
	char	appName[APP_NAME_SIZE];	// 64x8 bits: app name, 00 terminated, 
					//    fill with zero or garbage
} ProjectHeader;

/*
 * Files in project
 * 
 * Each file starts with a single, identifying byte:
 * 
 * 1x8 bits: 00=DOC source, 01=Memo source, 02=rsrc source, 03=pedit32
 * 
 * The next field contains the filename but the size seems to be file type
 * dependent.  
 * 
 * It seems I left out some stuff.  
 * 
 * There's an obj that OBC adds for you (my guess is that it interacts with 
 * the OS to feed events to the code).  I think you find it before any C file 
 * in the list.  That file sucks-up 36x8 bits (it doesn't seem to have an 
 * identifying byte):
 * 
 * 	32x8 bits: object file name, 00 terminated, garbage filled.
 * 	4x8 bits for some special field.  It looks like the other .obj 
 * 		special field.
 * 
 * RSRC files seem to be 71x8 bits (72 if you include the identifying byte).  
 * It's made up of the filename, 00 terminated, garbage filled.
 * 
 * C source files seem to be 107x8 bits (108x8 bits if you
 * include the lead byte identifying the file type).  Those bits seem to
 * be divided into:
 * 
 *       67x8 bits for source filename, 00 terminated, garbage filled 
 *       	followed by
 *       4x8 bits for some sort of special field, followed by
 *       32x8 bits for object filename, 00 teminated, garbage filled followed by
 *       4x8 bits for some sort of special field.
 * 
 * The special field for the source filename looks like 00 00 00 XX where XX
 * is 00, 03, 04, 05, or 06.  The special field for the object filename looks 
 * like 00 7F YY ZZ where YY and ZZ vary all over the map.
 * 
 * I have seen header files (at one time I naively included header files in my
 * project so this may describe what I'm seeing).  Header files seem to take
 * 36x8 bits.  Those bits are divided into 32x8 bits for header filename, 00
 * terminated, garbage filled followed by 4x8 bits for some sort of special
 * field that looks like 00 00 00 WW (where WW includes at least the
 * following values: 07. 09, 0A, 0B, and 0C).
 */

typedef struct FileStart
{
	unsigned char	type;
	char		name[32];
} FileStart;

// Structure representing the different types of files supported by OnBoard C.

typedef enum EnumFileType 
{
	DocFile, 
	MemoFile, 
	RsrcFile, 
	PEditFile
} EnumFileType;

typedef struct FileType
{
	int			byteCount;
	char			*name;
	enum EnumFileType	type;
	void			(*convert)		(char	*nameOnPalm,
							 char	*nameOnHost);
	char			*(*getNameOnHost)	(char 	*nameOnPalm, 
							 char 	*databaseName);
	char			*(*getNameOnPalm)	(char	*databaseName);
} FileType;


/*
 * Prototypes
 */

static char	*getVersion	(void);
static YesNo	printChanges	(void		*arg);
static void	showBuf		(unsigned char	*buf, 
				 int		bytes);
static char	*getWinNameOfPrc	(char		*databaseName);
static char	*getWinNameOfPdb	(char		*databaseName);
static char	*getPlinkNameOfPrc	(char		*databaseName);
static char	*getPlinkNameOfPdb	(char		*databaseName);
static void ConvertProject  (char	    *projectfile, FileType *typeList );
static YesNo	isHeader	(char		*filename);

extern int	SwapChoose	(void);
extern void	convertRsrc	(char		*nameOnPalm,
				 char		*nameOnHost);
extern char	*RsrcHostName	(char 		*nameOnPalm, 
				 char 		*databaseName);

extern void	Doc2Text_wrap	(char		*nameOnPalm,
				 char		*nameOnHost);
extern char	*DocHostName	(char 		*nameOnPalm, 
				 char 		*databaseName);

/*
 * Global variables (shudder)
 */

static YesNo	keepGoing		= Yes;
static YesNo	flag[FlagIndexCount]	= {0};

char const	changes [] =
"	0.01		Just gettin' warmed up\n"
"   0.02        Pilot-link naming scheme added\n"
;


static FileType winTypeList[4] =
{
    {108, "doc",	DocFile,	Doc2Text_wrap,	
                    DocHostName, 
                    getWinNameOfPdb},
    {108, "memo",	MemoFile,	0, 0, 0},
    {108, "rsrc",	RsrcFile,	convertRsrc,	
                    RsrcHostName, 
                    getWinNameOfPrc},
    {108, "pEdit",	PEditFile,	0, 0, 0}
};


static FileType plinkTypeList[4] =
{
    {108, "doc",	DocFile,	Doc2Text_wrap,	
                    DocHostName, 
                    getPlinkNameOfPdb},
    {108, "memo",	MemoFile,	0, 0, 0},
    {108, "rsrc",	RsrcFile,	convertRsrc,	
                    RsrcHostName, 
                    getPlinkNameOfPrc},
    {108, "pEdit",	PEditFile,	0, 0, 0}
};



/////////////////////////////


	int
main (int argc, char *argv[])
{
#	define	FUNCTION	"Sherpa (main)"

	// first, declare things that are in arglist

	char		*projfilename		= 0;
	int		debugLevel		= 0,
			warningLevel		= 4,
                        namingScheme            = 0;
	FileType *typeList = NULL;

	// then the command-line argument list

	ArgumentType	arglist [] =
	{
	 {"changes",	'c',	ARG_NONE,	0, 		printChanges,
	  "  --changes\t\t-c\t 	print the program's history & exit.\n"},

	 {"debug",	'd',	ARG_INT_OPT,	(void *) (&debugLevel),	0,
	  "  --debug <num>\t\t-d <num>	set debug level to num (0-100)\n"},

	 {"dryrun",	'n',	ARG_SET_FLAG,	(void *) (&flag[FlagDryRun]),	0,
	  "  --dryrun\t\t-n\t	print what the program WOULD do without doing it.\n"},

	 {"help",	'h',	ARG_SET_FLAG,	(void *) (&flag[FlagGotHelp]),	0,
	  "  --help\t\t-h\t	print this message & exit\n"},

	 {"makefile",	'm',	ARG_SET_FLAG,	(void *) (&flag[FlagBuildMakefile]),	0,
	  "  --makefile\t\t-m\t	build a makefile while you're at it.\n"},

	 {"overwrite",	'o',	ARG_SET_FLAG,	(void *) (&flag[FlagOverwriteFile]),	0,
	  "  --overwrite\t\t-o\t	overwrite existing files.\n"},

	 {"warning",	'w',	ARG_INT_OPT,	(void *) &warningLevel,	0,
	  "  --warning <num>\t-w <num>	set warning level to num (0-5)\n"},

	 {"namingscheme",	's',	ARG_INT_OPT,	(void *) &namingScheme,	0,
	  "  --namingscheme <num>\t-s <num>	0 = Windows, 1 = pilot-link\n"},

	 {"fixincludes",	'f',	ARG_SET_FLAG,	(void *)(&flag[FlagFixInclude]),	0,
	  "  --fixincludes\t\t-f\t\tForce include of palm headers\n"},

	 {0,		'\0',	ARG_NONE,	(void *) 0,		0, 0}
	};

	// then declare the rest of the variables

	ArgStatus	argStatus		= ARG_STAT_OKAY;
	YesNo		debuggingStarted	= No;
	int		exitStatus		= EXIT_SUCCESS;


	/*
	 * BODY OF THE CODE
	 */

	printf ("\nSherpa, Porting help between OnBoard C and the PRC Tools, "
		"Version %s\n", getVersion());
	printf ("This is free softare under GNU Public License.\n\n");

	/*
	 * Parse arguments and check a couple of minor things
	 */

	if (argc < 2)	// 1 required arg + name of program
	{
		flag[FlagGotHelp]	= Yes;
		exitStatus		= EXIT_FAILURE;
	}
	else
	{
		projfilename		= *++argv;	argc--;
	}

	if ((argStatus = ArgsParse (arglist, argc, argv)) != ARG_STAT_OKAY)
	{
		ArgsPrintError (argStatus);
		keepGoing	= No;
		exitStatus	= EXIT_FAILURE;
	}

	/*
	 * Set up debugging and byte swapping and the like. . .
	 */

	if (isTrue(FlagGotHelp))
	{
		printf ("Usage 'sherpa <projfile.prc> [options]' \n"
			"where 'options' can be zero or more of:\n\n");
		ArgsPrintHelp	(arglist);
		printf ("\n");
		keepGoing	= No;
	}
	else 
	{
		initializeDebug (	"sherpa.log",
					"w",
					debugLevel, 
					warningLevel);

		debuggingStarted	= Yes;

		if (isTrue(FlagDryRun))
		{
			// So we can print things out
			if (getDebugLevel() < (DEBUG_DRYRUN+1))
				setDebugLevel (DEBUG_DRYRUN+1);
		}


		if ( ! SwapChoose()) 
		{
			errorPrintf(
				"** ERROR (%s): failed to select proper byte "
				"swapping algorithm\n",
				FUNCTION);
			keepGoing		= No;
			exitStatus		= EXIT_FAILURE;
		}
	}

	switch ( namingScheme )
	{
	    case WINDOWS_NAMES:
		typeList = (FileType *)&winTypeList;
		break;

	    case PILOT_LINK_NAMES:
		typeList = (FileType *)&plinkTypeList;
		break;

	    default:
		errorPrintf( "** ERROR (%s): unknown naming scheme value\n",
			FUNCTION );
		exitStatus = EXIT_FAILURE;
		break;
	}

	/*
	 * . . .and then do the actual work of the code
	 */

	if (keepGoing)
	{
		if ( projfilename && typeList )
			ConvertProject (projfilename, typeList );
	}

	printDebugTally ();

	exit (exitStatus);

#	undef	FUNCTION
}



	static void
WriteStdIncludeHack ( void )
{
#	define	FUNCTION	"WriteStdIncludeHack"
	FILE *includeFile	= 0;

	includeFile = openNewFile ("StdInc.h", "w");

	if ( includeFile != 0 ) {
		fprintf (includeFile, "#include <PalmOS.h>\n" );
		fprintf (includeFile, "#include <PalmCompatibility.h>\n" );

		fclose( includeFile );
	}
#	undef	FUNCTION
}


	static YesNo
SimpleGrep ( char *file, char *string )
{
#	define	FUNCTION	"SimpleGrep"
	FILE *openFile;
	long fileLength;
	YesNo found = No;
	char *fileContent;

	openFile = fopen( file, "r" );
	if ( openFile == NULL ) {
		warningPrintf (1, "** Warning (%s): Unable to open file (%s) "
			       "for include scan\n", FUNCTION, file );
		return found;
	}
	
	if ( fseek( openFile, 0, SEEK_END ) != 0 ) {
		warningPrintf (1, "** Warning (%s): Unable to seek file (%s) "
			       "for include scan\n", FUNCTION, file );
		fclose( openFile );
		return found;
	}

	fileLength = ftell( openFile );
	if ( fileLength == 0 ) {
		warningPrintf (1, "** Warning (%s): File (%s) appears to be "
			       "zero length\n", FUNCTION, file );
		fclose( openFile );
		return found;
	}

	rewind( openFile );

	fileContent = (char *)malloc( fileLength + 1 );
	if ( fileContent == NULL ) {
		warningPrintf (1, "** Warning (%s): Unable to allocate memory "
			       "for include scan\n", FUNCTION );
		fclose( openFile );
		return found;
	}

	if ( fread( fileContent, fileLength, 1, openFile ) != 1 ) {
		warningPrintf (1, "** Warning (%s): Unable to read file (%s) "
			       "for include scan\n", FUNCTION, file );
		free( fileContent );
		fclose( openFile );
		return found;
	}
	fileContent[fileLength] = '\0';

	if ( strstr( fileContent, string ) != NULL ) {
	    found = Yes;
	}

	free( fileContent );
	fclose( openFile );

	return found;
#undef	FUNCTION
}

	static void 
ConvertProject (char	*projectfile, FileType *typeList )
{
#	define	FUNCTION	"ConvertProject"

	// Makefile building blocks

	static char	*make1 =
	"### app config\n"
	"# ICONTEXT is the name which will appear below your icon\n"
	"# in the Palm launcher when your app is installed. APPID is the\n"
	"# unique ID of your app that you registered with Palm.\n"
	"\n"
	;

	static char	*make2 = 
	"\n\n"
	"### tool definitions\n"
	"# These are specific to which version of gcc you use. If you are\n"
	"# using 0.5.0, you will need to change them\n"
	"\n"
	"CC		= m68k-palmos-gcc\n"
	"PILRC		= pilrc\n"
	"OBJRES		= m68k-palmos-obj-res\n"
	"BUILDPRC	= build-prc\n"
	"\n"
	"### flags\n"
	"\n"
	"BINDIR		= Pilrc\n"
	"\n"
	"### build optimization level and Warnings\n"
	"# Here I have specified -O2 for optimization. This is a good\n"
	"# level for Palms as it tries to optimize speed without\n"
	"# bloating the finished app size.\n"
	"# The -Wall option is discussed thoroughly, above.\n"
	"\n"
	"CFLAGS = -Wall -O2 $(STDINCLUDEHACK)\n"
	"#CFLAGS = -Wall\n"
	"\n"
	"\n"
	"all: $(PRC)\n"
	"\n"
	"$(PRC): $(APP)\n"
	"	$(PILRC) $(RCP) $(BINDIR)\n"
	"	$(OBJRES) $(APP)\n"
	"	$(BUILDPRC) $(PRC) $(ICONTEXT) $(APPID) *.grc $(BINDIR)/*.bin\n"
	"	dir *.PRC\n"
	"\n"
	"$(APP): $(SRC:.c=.o)\n"
	"	$(CC) $(CFLAGS) $^ -g -o $@\n"
	"\n"
	"%.o: %.c\n"
	"	$(CC) $(CFLAGS) -c $< -g -o $@ \n"
	"#	touch $<\n"
	"# to rebuild every time, uncomment above line\n"
	"\n"
	"depend dep:\n"
	"	$(CC) -M $(SRC) > .dependencies\n"
	"\n"
	"clean:\n"
	"	rm -rf *.o $(APP) $(BINDIR)/*.bin *.grc *.stamp *~\n"
	"\n"
	"veryclean: clean\n"
	"	rm -rf *.prc *.bak\n"
	"\n"
	"\n"
	"### end\n"
	;

	static char *stdincludehack =
	"STDINCLUDEHACK = -include StdInc.h\n";

	static char *palminclude = "#include <PalmOS.h>";

	// Then the file-type structure -- the main table of this whole program

    /*
     *
	static FileType typeList[4] =
	{
		{108, "doc",	DocFile,	Doc2Text_wrap,	
						DocHostName, 
						getNameOfPdb},
		{108, "memo",	MemoFile,	0, 0, 0},
		{108, "rsrc",	RsrcFile,	convertRsrc,	
						RsrcHostName, 
						getNameOfPrc},
		{108, "pEdit",	PEditFile,	0, 0, 0}
	};
     */

	// And the rest of the variables

	FILE		*file		= 0,
			*makefile	= 0;

	YesNo		foundSourceFile	= No;
	YesNo		foundPalmInclude	= No;

	int		bytesToRead	= 0,
			bytesRead	= 0,
			totalBytesRead	= 0;
	int		myType		= -1;
	char		buffer[1024];

	ProjectHeader	*projectHeader	= (ProjectHeader *) &buffer[0];
	FileStart	*fileStart	= (FileStart *) &buffer[0];
	
	// open the project file for input
  					
  	if ((file = fopen (projectfile, "r")) == 0)
  	{
 		errorPrintf ("** ERROR (%s): file '%s' didn't open, %s\n", 
			FUNCTION, projectfile, strerror(errno));
 		return;
  	}

	// open the makefile for output (if necessary)

	if (isTrue(FlagBuildMakefile))
	{
		if (isTrue (FlagDryRun))
		{
			warningPrintf (1, "** Warning: makefile not created on dryrun\n");
		}
		else if ((makefile = openNewFile ("Makefile", "w")) == 0)
		{
			// 'openNewFile' already generated the error message
			//errorPrintf ("** ERROR (%s): file '%s' didn't open, %s\n", 
			//	FUNCTION, "Makefile", strerror(errno));
		}
	}

	// Read the project file header

	bytesToRead	= FileHeader+ExtraPcHeader+2;
	if ((bytesRead	= fread (buffer, sizeof(char), bytesToRead, file)) != bytesToRead)
	{
		if (!feof (file))
		{
			errorPrintf("** ERROR (%s): Problem reading from file, %s\n", 
				FUNCTION, strerror(errno));
			debugPrintf ((0, "\twanted %d got %d\n", bytesToRead, bytesRead));
		}
	}

	totalBytesRead	+= bytesRead;

	debugPrintf ((DEBUG_MODERATE, "%s: read header at %d bytes\n", 
		FUNCTION,
		bytesRead));

	debugPrintf ((DEBUG_MODERATE, "Creator ID='%c%c%c%c', AppName='%s'\n",
		projectHeader->creatorId[0], 
		projectHeader->creatorId[1], 
		projectHeader->creatorId[2], 
		projectHeader->creatorId[3], 
		projectHeader->appName));

	// write the top part of the makefile

	if (makefile != 0)
	{
		fprintf (makefile, "%s", make1);
		fprintf (makefile, "APP	 	= %s\n", projectHeader->appName);
		fprintf (makefile, "ICONTEXT	= \"%s\"\n", projectHeader->appName);
		fprintf (makefile, "APPID 		= %c%c%c%c\n",
							projectHeader->creatorId[0], 
							projectHeader->creatorId[1], 
							projectHeader->creatorId[2], 
							projectHeader->creatorId[3]);
		fprintf (makefile, "PRC		= $(APP).prc\n");
	}

	// and now go through the individual files in the project file

	while	(!feof (file))
	{
		debugPrintf ((DEBUG_MODERATE, "File position=%d\n", 
			ftell(file)));

		// read the filename of this component of the project

		if ((bytesRead = fread (buffer, sizeof(FileStart), 1, file)) != 1)
		{
			if (!feof (file))
			{
				errorPrintf("** ERROR (%s): Problem reading from file, %s\n", 
					FUNCTION, strerror(errno));
				debugPrintf ((0, "\twanted %d got %d\n", sizeof(FileStart), bytesRead));
			}
		}

		totalBytesRead	+= bytesRead;

		debugPrintf ((DEBUG_EXTENSIVE, "\tafter file=%d\n", 
			ftell(file)));

		if (feof (file))
		{
			debugPrintf ((DEBUG_MODERATE, "End of project file\n"));
		}
		else
		{
			myType		= fileStart->type;

			/*
			 * Is this a legal file type?  If not, tell the user
			 */

			if ((myType < 0) || (myType >= 4))
			{
				errorPrintf (
					"** ERROR (%s): Type no %d invalid\n", 
					FUNCTION,
					myType);
				bytesToRead	= 100;
				if ((bytesRead	= fread (buffer, 
							 sizeof(char), 
							 bytesToRead, 
							 file)) != bytesToRead)
				{
					if (!feof (file))
					{
					errorPrintf(
						"** ERROR (%s): Problem reading "
						"from file, %s\n", 
						FUNCTION, strerror(errno));
					debugPrintf ((0, "\twanted %d got %d\n", 
							bytesToRead, bytesRead));
					}
				}

				totalBytesRead	+= bytesRead;
				showBuf (buffer, bytesRead);
				break;
			}

			/*
			 * It's a known file type, let's see if we can convert
			 * it to the host format
			 */

			else 
			{
				char	*databaseName	= fileStart->name;

				char	*nameOnPalm	= 0;
				char	*nameOnHost	= 0;

				nameOnPalm
					= (typeList[myType].getNameOnPalm == 0)
					? databaseName
					: typeList[myType].getNameOnPalm 
							(databaseName);

				nameOnHost	
					= (typeList[myType].getNameOnHost == 0)
					? nameOnPalm
					: typeList[myType].getNameOnHost
							(nameOnPalm, 
							 databaseName);

				// Output the makefile stuff

				if (makefile != 0) 
				{
					if (typeList[myType].type == RsrcFile)
					{
						if (foundSourceFile)
						{
							warningPrintf (0,
							"** ERROR (%s): source "
							"files found before "
							"RSRC file in project "
							"-- makefile is going "
							"to be honked-up\n",
							FUNCTION);
						}

						fprintf (makefile, 
							"RCP		= %s\n",
							nameOnHost);
						fprintf (makefile, 
							"SRC		= ");
					}
					else if (isHeader(nameOnHost))
					{
						debugPrintf ((DEBUG_EXTENSIVE, 
							"HEADER:%s\n", 
							nameOnHost));
					}
					else
					{
						debugPrintf ((DEBUG_EXTENSIVE, 
							"SOURCE:%s\n", 
							nameOnHost));
						fprintf (makefile, "%s ", 
							SafeChar(nameOnHost));
						foundSourceFile	= Yes;
					}
				}

				/*
				 * If we don't have function to convert the
				 * file, tell the user that we don't support
				 * the file type.
				 */

				if (typeList[myType].convert == 0)
				{
					warningPrintf (0,
						"** Warning (%s): File type '%s' "
						"not supported for file '%s'\n",
						FUNCTION,
						SafeChar(typeList[myType].name),
						SafeChar(fileStart->name));
				}

				/*
				 * If we _do_ know how to convert the file
				 * type, convert it!
				 */

				else
				{
					debugPrintf ((DEBUG_DRYRUN, 
						"  '%s' (%s), type %s ==> '%s'\n",
						SafeChar(nameOnPalm),
						SafeChar(databaseName),
						SafeChar(typeList[myType].name),
						SafeChar(nameOnHost)));


					if (!isTrue (FlagDryRun))
					{
						// Add the source file to the
						// makefile if the source file
						// isn't a header.


						// Convert the file we've found

						typeList[myType].convert
							(nameOnPalm, nameOnHost);

						if ( typeList[myType].type == DocFile ) 
						{
							foundPalmInclude |= SimpleGrep( nameOnHost, 
											palminclude );
						}
					}
				}

				/*
				 * Read the rest of the info about this file
				 * (and discard it).
				 *
				 * Figure out how much more to read for this
				 * file type (headers only require 36 bytes
				 * but the .c files are based on the file
				 * format).
				 */

				bytesToRead	= isHeader (fileStart->name)
					? BYTES_FOR_H_FILE
					: typeList[myType].byteCount;

				// already read the file start
				bytesToRead	-= sizeof(FileStart);	

				// Read remaining bytes of file specification

				if ((bytesRead = fread (buffer, sizeof(char), 
						bytesToRead,
						file)) != bytesToRead)
				{
					if (!feof(file))
					{
					errorPrintf("** ERROR (%s): Problem "
						"reading from file, %s\n", 
						FUNCTION, strerror(errno));
					debugPrintf ((0, "\twanted %d got %d\n", 
							bytesToRead, bytesRead));
					}
				}

				totalBytesRead	+= bytesRead;

			} // if valid file type
		} // if !eof
	} // while file isn't done

	if (makefile != 0)
	{
		if ( isTrue( FlagFixInclude ) )
		{
			WriteStdIncludeHack();
			fprintf (makefile, "\n%s", stdincludehack);
		}
		else if ( foundPalmInclude == No ) 
		{
			warningPrintf(0, "** WARNING (%s): No platform header "
				      "file include commands found\n**\t\t\t     You "
				      "may want to use --fixincludes\n", 
				      FUNCTION );
		}
		fprintf (makefile, "\n%s", make2);
		fclose (makefile);
	}
	
#undef	FUNCTION
}


	static char *
getWinNameOfPrc (char	*databaseName)
{
#	define	FUNCTION	"getWinNameOfPrc"
	static char	string[100];
	char		*ext	= ".PRC";
	char		*answer	= 0,
			*now	= 0,
			*end	= &string[sizeof(string)-1];

	if (databaseName != 0)
	{
		answer	= &string[0];

		// convert the file
		for	(now = &string[0];
			 (now <= end) && (*databaseName != '\0');
			 now++, databaseName++)
		{
			*now = (*databaseName == '.')
				? '_'
				: *databaseName;
		}

		// add the file extension
		while	(*ext != '\0')
			*now++ = *ext++;

		// terminate the name
		*now = '\0';
	}

	return answer;
#	undef	FUNCTION
}



	static char *
getWinNameOfPdb (char	*databaseName)
{
#	define	FUNCTION	"getWinNameOfPdb"
	static char	string[100];
	char		*ext	= ".PDB";
	char		*answer	= 0,
			*now	= 0,
			*end	= &string[sizeof(string)-1];

	if (databaseName != 0)
	{
		answer	= &string[0];

		// convert the file
		for	(now = &string[0];
			 (now <= end) && (*databaseName != '\0');
			 now++, databaseName++)
		{
			*now = (*databaseName == '.')
				? '_'
				: *databaseName;
		}

		// add the file extension
		while	(*ext != '\0')
			*now++ = *ext++;

		// terminate the name
		*now = '\0';
	}

	return answer;
#	undef	FUNCTION
}



	static char *
getPlinkNameOfPrc (char	*databaseName)
{
#	define	FUNCTION	"getPlinkNameOfPrc"
	static char	string[100];
	char		*ext	= ".prc";
	char		*answer	= 0,
			*now	= 0,
			*end	= &string[sizeof(string)-1];

	if (databaseName != 0)
	{
		answer	= &string[0];

		// convert the file
		for	(now = &string[0];
			 (now <= end) && (*databaseName != '\0');
			 now++, databaseName++)
		{
			*now = *databaseName;
		}

		// add the file extension
		while	(*ext != '\0')
			*now++ = *ext++;

		// terminate the name
		*now = '\0';
	}

	return answer;
#	undef	FUNCTION
}



	static char *
getPlinkNameOfPdb (char	*databaseName)
{
#	define	FUNCTION	"getPlinkNameOfPdb"
	static char	string[100];
	char		*ext	= ".pdb";
	char		*answer	= 0,
			*now	= 0,
			*end	= &string[sizeof(string)-1];

	if (databaseName != 0)
	{
		answer	= &string[0];

		// convert the file
		for	(now = &string[0];
			 (now <= end) && (*databaseName != '\0');
			 now++, databaseName++)
		{
			*now = *databaseName;
		}

		// add the file extension
		while	(*ext != '\0')
			*now++ = *ext++;

		// terminate the name
		*now = '\0';
	}

	return answer;
#	undef	FUNCTION
}


/*
 * UTILITY FUNCTIONS
 */



	static YesNo 
isHeader (char *filename)
{
#	define	FUNCTION	"isHeader"
	YesNo	answer	= No;

	if (filename != 0)
	{
		// find the '.' in the name -- presumably the character AFTER
		// that is 'h', 'H', 'c', or 'C'
		while ((*filename != '\0') && (*filename != '.'))
			++filename;

		if (*filename == '.')
		{
			++filename;
			if ((*filename == 'h') || (*filename == 'H'))
				answer	= Yes;
		}
	}

	return	answer;
#undef	FUNCTION
}



	static void 
showBuf (unsigned char *buf, int bytes)
{
#	define	FUNCTION	"showBuf"

	if (buf == 0)
	{
		errorPrintf ("** ERROR (%s): NULL buffer\n", FUNCTION);
	}
	else
	{
		int	charOnLine	= 0,
			line		= 0;	// counting in characters

		while (line < bytes)
		{
			// do the numbers first
			for	(charOnLine = 0; 
				(charOnLine < BYTES_ON_LINE) 
					&& (line+charOnLine < bytes); 
				charOnLine++)
			{
				debugPrintf	((DEBUG_ALWAYS, 
						"%02x ", 
						buf[line+charOnLine]));
			}

			// then, do the characters
			for	(charOnLine = 0; 
				(charOnLine < BYTES_ON_LINE) 
					&& (line+charOnLine < bytes); 
				charOnLine++)
			{
				if (isprint(buf[line+charOnLine]))
					debugPrintf ((DEBUG_ALWAYS, 
						"  %c ", 
						buf[line+charOnLine]));
				else
					debugPrintf ((DEBUG_ALWAYS, 
						"\\%02d ", 
						buf[line+charOnLine]));
			}

			line += charOnLine;
		}
	}

#undef	FUNCTION
}

///////////////////////////////////////

	static char *
getVersion (void)
{
#	define	FUNCTION	"getVersion"
	static char	version[] = {VERSION " " SPECIAL_VERSION};
	return version;
#	undef	FUNCTION
}



	static YesNo			
printChanges (void *arg)
{
#	define	FUNCTION	"printChanges"
	ignore ((int) arg);

	debugPrintf ((DEBUG_ALWAYS, "%s\n", changes));
	keepGoing	= No;

	return Yes;
#	undef	FUNCTION
}



	YesNo
isTrue (FlagIndex aIndex)
{
#	define	FUNCTION	"isTrue"
	YesNo	answer	= No;
	int	intIndex	= (int) aIndex;

	if ((intIndex >= 0) && (intIndex <FlagIndexCount))
	{
		answer	= flag[intIndex];
	}

	return answer;
#	undef	FUNCTION
}

