/***********************************************************************
 *
 * This file is part of the OnBoard C package.  For more information on 
 * using OnBoard C, see http://groups.yahoo.com/group/onboardc.  For 
 * more information on developing OnBoard C (including submission of 
 * bug reports), see http://sourceforge.net/projects/onboardc.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 ***********************************************************************/

#include <PalmCompatibility.h>
#include <PalmOS.h>
#include <SysEvtMgr.h>

#include "debug.h"

#if DEBUG_SUPPORT

static CharPtr getA6();
static CharPtr getA5();
static DebugData *getDebugData();
static CharPtr findMacsbugName(int *pc);
static char *findBase(int *pc);
static int truncateString(CharPtr s, int length, int max);
static void openTraceFile(DebugData *dbgP);
static void writeToTraceFile(DebugData *dbgP, char *message);
static void drawDebugWindow(Boolean functionList, char *msg);
static void drawFunctionList(SymbolData *symbol, int* functionIndexList, int functionTop, int functionCount);
static void drawStackTrace(SymbolData *symbol, CharPtr *stackTraceNames, int stackTraceTop, int stackTraceCount, int entry);
static int buildStackTrace(char *pc, ULong *a6, DebugData *dbgP, SymbolData *symbol, int symbolCount, int *stackTrace, CharPtr *stackTraceNames, ULong *frameStack);
static int findSymbol(DebugData *dbgP, char *pc, SymbolData *symbol, int symbolCount);
static void drawVarList(SymbolData *symbol, int symbolCount, int top, int varMin, int varMax, char *frameA6, ULong ramSize);
static void getVarValue(SymbolData *symbol, int index, char *buf, char *frameA6, ULong ramSize);
static void doFunctionList(DebugData *dbgP, SymbolData *symbol);
static int countVars(SymbolData *symbol, int symbolCount, int top);
static void doStackWindow(char *pc, DebugData *dbgP, SymbolData *symbol, int symbolCount, ULong *a6, char *msg, WinHandle offscreen);
static void *getCurAppInfo();
static void writeStackTrace(char *pc, ULong *a6, DebugData *dbgP, SymbolData *symbol, int symbolCount, char *message);

#endif

void StartOfDebugCode()
{
}


Handle gSymbolDataH = NULL;
SymbolData *gSymbolDataList;
int gMaxSymbolData = 64;
int gSymbolDataListTop = 0;
int gSymbolDataIndex;

Handle gSymbolStringH;
char *gSymbolString;
int gMaxSymbolString = 256;
int gSymbolStringTop;
int gSymbolStringIndex;

#if DEBUG_SUPPORT

char *DebugEntryName = "DebugEntry";
int DebugEntryNameLength = 10;
int DebugEntryNameHash = 'D' ^ 'e' ^ 'b' ^ 'u' ^ 'g' ^ 'E' ^ 'n' ^ 't' ^ 'r' ^ 'y';
char *DebugInitName = "DebugInit";
int DebugInitNameLength = 9;
int DebugInitNameHash = 'D' ^ 'e' ^ 'b' ^ 'u' ^ 'g' ^ 'I' ^ 'n' ^ 'i' ^ 't';
char *DebugExitName = "DebugExit";
int DebugExitNameLength = 9;
int DebugExitNameHash = 'D' ^ 'e' ^ 'b' ^ 'u' ^ 'g' ^ 'E' ^ 'x' ^ 'i' ^ 't';
char *DebugTermName = "DebugTerm";
int DebugTermNameLength = 9;
int DebugTermNameHash = 'D' ^ 'e' ^ 'b' ^ 'u' ^ 'g' ^ 'T' ^ 'e' ^ 'r' ^ 'm';

#define BUS_ERROR_TRAP (ULong*)(0x08)
#define ADDR_ERROR_TRAP (ULong*)(0x0C)
#define ILL_INSTR_TRAP (ULong*)(0x10)
#define DIVZ_TRAP (ULong*)(0x14)


#define STACK_TRACE_LEFT 10
#define STACK_TRACE_TOP 5
#define STACK_TRACE_WIDTH 140
#define STACK_TRACE_HEIGHT 60
#define STACK_TRACE_COUNT 5

#define VAR_LIST_LEFT 10
#define VAR_LIST_TOP 80
#define VAR_LIST_HEIGHT 60
#define VAR_LIST_WIDTH 140
#define VAR_LIST_COUNT 5

#define MSG_LEFT 10
#define MSG_TOP (STACK_TRACE_TOP + STACK_TRACE_HEIGHT + 1)

#define FNS_BTN_LEFT 25
#define FNS_BTN_TOP  145
#define FNS_BTN_WIDTH 30
#define FNS_BTN_HEIGHT 12

#define OK_BTN_LEFT 65
#define OK_BTN_TOP  145
#define OK_BTN_WIDTH 30
#define OK_BTN_HEIGHT 12

#define SCREEN_BTN_LEFT 105
#define SCREEN_BTN_TOP  145
#define SCREEN_BTN_WIDTH 31
#define SCREEN_BTN_HEIGHT 12

#define MAX_STACK_TRACE 10

void DebugEntry()
{
	DebugData *dbgP = getDebugData();
	if (dbgP) {
		ULong *a6 = (ULong *)getA6();			// frame for 'DebugEntry'
		char *frameA6 = (char *)(a6[0]);		// first frame of debuggee
		char *pc = (char *)(a6[1]);				// return address of 'DebugEntry' == debuggee function
		if (dbgP->symbH && dbgP->resBase) {
			int i;
			SymbolData *symbol = (SymbolData *)MemHandleLock(dbgP->symbH);
			int symbolCount = symbol[0].size;
			int index = findSymbol(dbgP, pc, symbol, symbolCount);
			if (index != -1) {
				if ((symbol[index].debugAction & TraceAction) == TraceAction) {
					writeToTraceFile(dbgP, symbol[index].name);
					writeToTraceFile(dbgP, "\n");
					i = index + 1;
					while ((symbol[i].flag == SymLocal) && (i < symbolCount)) {
						if (symbol[i].offset > 0) {
							char tBuf[64];
							writeToTraceFile(dbgP, "\t");
							getVarValue(symbol, i, tBuf, frameA6, dbgP->ramSize);
							writeToTraceFile(dbgP, tBuf);
							writeToTraceFile(dbgP, "\n");
						}
						i++;
					}
				}
#if DEBUG_BREAKPOINTS				
				if ((symbol[index].debugAction & BreakAction) == BreakAction) {
					UInt err;
					RectangleType fullScreen = { 0, 0, 160, 160 };
					WinHandle offscreen = NULL;
					WinHandle debugWindow = NULL;
					WinHandle onscreen = NULL;

					offscreen = WinCreateOffscreenWindow(160, 160, screenFormat, &err);
					if (offscreen) {
						debugWindow = WinCreateWindow(&fullScreen, noFrame, false, true, &err);
						if (debugWindow) {
							onscreen = WinSetDrawWindow(debugWindow);			
							if (onscreen) {
								WinCopyRectangle(0, offscreen, &fullScreen, 0, 0, scrCopy);
								
								doStackWindow(pc, dbgP, symbol, symbolCount, (ULong *)(frameA6), NULL, offscreen);
								
								WinSetDrawWindow(onscreen);
								WinCopyRectangle(offscreen, 0, &fullScreen, 0, 0, scrCopy);
							}
							WinDeleteWindow(debugWindow, false);
						}
						WinDeleteWindow(offscreen, false);
					}
				}
#endif				
			}
			MemHandleUnlock(dbgP->symbH);
		}
	}
}

#if DEBUG_BREAKPOINTS				

static void doStackWindow(char *pc, DebugData *dbgP, SymbolData *symbol, int symbolCount, ULong *a6, char *msg, WinHandle offscreen)
{
	int stackTraceCount, stackTraceTop;
	CharPtr stackTraceNames[MAX_STACK_TRACE];
	int stackTrace[MAX_STACK_TRACE];
	ULong frameStack[MAX_STACK_TRACE];
	int varTop, varMin, varMax, entry;
	RectangleType fullScreen = { 0, 0, 160, 160 };

	RectangleType stackTraceRect = { STACK_TRACE_LEFT, STACK_TRACE_TOP, STACK_TRACE_WIDTH, STACK_TRACE_HEIGHT };
	RectangleType okButton = { OK_BTN_LEFT, OK_BTN_TOP, OK_BTN_WIDTH, OK_BTN_HEIGHT };
	RectangleType fnsButton = { FNS_BTN_LEFT, FNS_BTN_TOP, FNS_BTN_WIDTH, FNS_BTN_HEIGHT };
	RectangleType screenButton = { SCREEN_BTN_LEFT, SCREEN_BTN_TOP, SCREEN_BTN_WIDTH, SCREEN_BTN_HEIGHT };
	
	RectangleType stackUpArrow = { STACK_TRACE_LEFT + STACK_TRACE_WIDTH - 10, STACK_TRACE_TOP, 10, 10 };
	RectangleType stackDownArrow = { STACK_TRACE_LEFT + STACK_TRACE_WIDTH - 10, STACK_TRACE_TOP + STACK_TRACE_HEIGHT - 10, 10, 10 };
	RectangleType varsUpArrow = { VAR_LIST_LEFT + VAR_LIST_WIDTH - 10, VAR_LIST_TOP, 10, 10 };
	RectangleType varsDownArrow = { VAR_LIST_LEFT + VAR_LIST_WIDTH - 10, VAR_LIST_TOP + VAR_LIST_HEIGHT - 10, 10, 10 };
	
	drawDebugWindow(false, msg);
	stackTraceCount = buildStackTrace(pc, a6, dbgP, symbol, symbolCount, stackTrace, stackTraceNames, frameStack);
	stackTraceTop = 0;
	entry = 0;
	drawStackTrace(symbol, stackTraceNames, stackTraceTop, stackTraceCount, entry);
	varMin = varTop = stackTrace[0] + 1;
	varMax = countVars(symbol, symbolCount, varTop);
	drawVarList(symbol, symbolCount, varTop, varMin, varMax, (char *)(frameStack[0]), dbgP->ramSize);
	
	while (true) {
		Boolean penDown;
		Int x,y;
		EvtGetPen(&x, &y, &penDown);
		if (penDown) {
			RectangleType *p = NULL;												
			if (RctPtInRectangle(x, y, &stackUpArrow)) {
				if (stackTraceTop > 0) {
					--stackTraceTop;
					drawStackTrace(symbol, stackTraceNames, stackTraceTop, stackTraceCount, entry);
					p = &stackUpArrow;
				}
				else
					continue;
			}
			if (RctPtInRectangle(x, y, &stackDownArrow)) {
				if (stackTraceTop < (stackTraceCount - STACK_TRACE_COUNT)) {
					++stackTraceTop;
					drawStackTrace(symbol, stackTraceNames, stackTraceTop, stackTraceCount, entry);
					p = &stackDownArrow;
				}
				else
					continue;
			}
			if (RctPtInRectangle(x, y, &varsUpArrow)) {
				if (varTop > varMin) {
					--varTop;
					drawVarList(symbol, symbolCount, varTop, varMin, varMax, (char *)(frameStack[entry]), dbgP->ramSize);
					p = &varsUpArrow;
				}
				else
					continue;
			}
			if (RctPtInRectangle(x, y, &varsDownArrow)) {
				if ((varTop + VAR_LIST_COUNT) < varMax) {
					++varTop;
					drawVarList(symbol, symbolCount, varTop, varMin, varMax, (char *)(frameStack[entry]), dbgP->ramSize);
					p = &varsDownArrow;
				}
				else
					continue;
			}
			if (p) {
				WinInvertRectangle(p, 1);
				do {
					EvtGetPen(&x, &y, &penDown);
				} while (penDown);
				WinInvertRectangle(p, 1);
				EvtFlushPenQueue();
			}
			else {
				if (RctPtInRectangle(x, y, &stackTraceRect)) {												
					int i = stackTraceTop + (y - STACK_TRACE_TOP) / 12;
					if (i < stackTraceCount) {
						entry = i;
						varMin = varTop = stackTrace[entry] + 1;
						varMax = countVars(symbol, symbolCount, varTop);
						drawStackTrace(symbol, stackTraceNames, stackTraceTop, stackTraceCount, entry);
						drawVarList(symbol, symbolCount, varTop, varMin, varMax, (char *)(frameStack[entry]), dbgP->ramSize);
						do {
							EvtGetPen(&x, &y, &penDown);
						} while (penDown);
						EvtFlushPenQueue();
					}
				}
				else {
					if (RctPtInRectangle(x, y, &okButton)) {
						WinInvertRectangle(&okButton, 1);
						do {
							EvtGetPen(&x, &y, &penDown);
						} while (penDown);
						WinInvertRectangle(&okButton, 1);
						EvtFlushPenQueue();
						break;
					}
					else {
						if (RctPtInRectangle(x, y, &fnsButton)) {
							WinInvertRectangle(&fnsButton, 1);
							do {
								EvtGetPen(&x, &y, &penDown);
							} while (penDown);
							WinInvertRectangle(&fnsButton, 1);
							EvtFlushPenQueue();
							doFunctionList(dbgP, symbol);
							drawDebugWindow(false, NULL);
							drawStackTrace(symbol, stackTraceNames, stackTraceTop, stackTraceCount, entry);
							drawVarList(symbol, symbolCount, varTop, varMin, varMax, (char *)(frameStack[entry]), dbgP->ramSize);
						}
						else {
							if (RctPtInRectangle(x, y, &screenButton)) {
								WinInvertRectangle(&screenButton, 1);
								do {
									EvtGetPen(&x, &y, &penDown);
								} while (penDown);
								WinInvertRectangle(&screenButton, 1);
								WinCopyRectangle(offscreen, 0, &fullScreen, 0, 0, scrCopy);
								do {
									EvtGetPen(&x, &y, &penDown);
								} while (!penDown);
								drawDebugWindow(false, NULL);
								drawStackTrace(symbol, stackTraceNames, stackTraceTop, stackTraceCount, entry);
								drawVarList(symbol, symbolCount, varTop, varMin, varMax, (char *)(frameStack[entry]), dbgP->ramSize);
								EvtFlushPenQueue();
							}
						}
					}
				}
			}
		}
	}
}
		
#define FN_LIST_LEFT 10
#define FN_LIST_WIDTH 140
#define FN_LIST_TOP 12
#define FN_LIST_HEIGHT 96
#define FN_LIST_COUNT 8
#define FN_LIST_FLAG_WIDTH 8
#define FN_LIST_TRACE_LEFT FN_LIST_LEFT
#define FN_LIST_BREAK_LEFT (FN_LIST_LEFT + FN_LIST_FLAG_WIDTH)
#define FN_LIST_NAME_LEFT (FN_LIST_BREAK_LEFT + FN_LIST_FLAG_WIDTH)

static void doFunctionList(DebugData *dbgP, SymbolData *symbol)
{
	int *functionIndexList;
	int symbolCount = symbol[0].size;
	int functionTop = 0;
	int functionCount = 0;
	int i;
	
	for (i = 0; i < symbolCount; i++)
		if (symbol[i].flag == SymCode) functionCount++;


	functionIndexList = MemPtrNew(functionCount * sizeof(int));
	if (functionIndexList) {
	
		RectangleType upArrow = { FN_LIST_LEFT + FN_LIST_WIDTH - 10, FN_LIST_TOP, 10, 10 };
		RectangleType downArrow = { FN_LIST_LEFT + FN_LIST_WIDTH - 10, FN_LIST_TOP + FN_LIST_HEIGHT - 10, 10, 10 };
		RectangleType okButton = { OK_BTN_LEFT, OK_BTN_TOP, OK_BTN_WIDTH, OK_BTN_HEIGHT };
		RectangleType breakRect = { FN_LIST_BREAK_LEFT, FN_LIST_TOP, FN_LIST_FLAG_WIDTH, FN_LIST_HEIGHT };
		RectangleType traceRect = { FN_LIST_TRACE_LEFT, FN_LIST_TOP, FN_LIST_FLAG_WIDTH, FN_LIST_HEIGHT };
		RectangleType breakARect = { FN_LIST_BREAK_LEFT, FN_LIST_TOP - 12, FN_LIST_FLAG_WIDTH, 12 };
		RectangleType traceARect = { FN_LIST_TRACE_LEFT, FN_LIST_TOP - 12, FN_LIST_FLAG_WIDTH, 12 };
		RectangleType breakNRect = { FN_LIST_BREAK_LEFT, FN_LIST_TOP + FN_LIST_HEIGHT, FN_LIST_FLAG_WIDTH, 12 };
		RectangleType traceNRect = { FN_LIST_TRACE_LEFT, FN_LIST_TOP + FN_LIST_HEIGHT, FN_LIST_FLAG_WIDTH, 12 };
		
		int count = 0;
		for (i = 0; i < symbolCount; i++) {
			if (symbol[i].flag == SymCode) {
				functionIndexList[count++] = i;
			}
		}
		drawDebugWindow(true, NULL);
		drawFunctionList(symbol, functionIndexList, functionTop, functionCount);
	
		while (true) {
			Boolean penDown;
			Int x,y;
			EvtGetPen(&x, &y, &penDown);
			if (penDown) {			
				RectangleType *p = NULL;				
				if (RctPtInRectangle(x, y, &upArrow)) {
					if (functionTop > 0) {
						--functionTop;
						drawFunctionList(symbol, functionIndexList, functionTop, functionCount);
						p = &upArrow;
					}
					goto handleButton;
				}
				if (RctPtInRectangle(x, y, &downArrow)) {
					if ((functionTop + FN_LIST_COUNT) < functionCount) {
						++functionTop;
						drawFunctionList(symbol, functionIndexList, functionTop, functionCount);
						p = &downArrow;
					}
					goto handleButton;
				}
				if (RctPtInRectangle(x, y, &breakARect)) {
					int i;
					for (i = 0; i < functionCount; i++) {
						int index = functionIndexList[i];
						char action = symbol[index].debugAction | BreakAction;
						DmWrite(symbol, (char *)(&symbol[index].debugAction) - (char *)symbol, &action, 1);
					}
					drawFunctionList(symbol, functionIndexList, functionTop, functionCount);
					p = &breakARect;
					goto handleButton;
				}
				if (RctPtInRectangle(x, y, &breakNRect)) {
					int i;
					for (i = 0; i < functionCount; i++) {
						int index = functionIndexList[i];
						char action = symbol[index].debugAction & ~BreakAction;
						DmWrite(symbol, (char *)(&symbol[index].debugAction) - (char *)symbol, &action, 1);
					}
					drawFunctionList(symbol, functionIndexList, functionTop, functionCount);
					p = &breakNRect;
					goto handleButton;
				}
				if (RctPtInRectangle(x, y, &traceARect)) {
					int i;
					for (i = 0; i < functionCount; i++) {
						int index = functionIndexList[i];
						char action = symbol[index].debugAction | TraceAction;
						DmWrite(symbol, (char *)(&symbol[index].debugAction) - (char *)symbol, &action, 1);
					}
					drawFunctionList(symbol, functionIndexList, functionTop, functionCount);
					p = &traceARect;
					goto handleButton;
				}
				if (RctPtInRectangle(x, y, &traceNRect)) {
					int i;
					for (i = 0; i < functionCount; i++) {
						int index = functionIndexList[i];
						char action = symbol[index].debugAction & ~TraceAction;
						DmWrite(symbol, (char *)(&symbol[index].debugAction) - (char *)symbol, &action, 1);
					}
					drawFunctionList(symbol, functionIndexList, functionTop, functionCount);
					p = &traceNRect;
					goto handleButton;
				}
				if (RctPtInRectangle(x, y, &breakRect)) {
					int entry = (y - FN_LIST_TOP) / 12;
					int index = functionIndexList[functionTop + entry];
					char action = symbol[index].debugAction;
					if ((action & BreakAction) == BreakAction) {
						action &= ~BreakAction;
						DmWrite(symbol, (char *)(&symbol[index].debugAction) - (char *)symbol, &action, 1);
						drawFunctionList(symbol, functionIndexList, functionTop, functionCount);
					}
					else {
						action |= BreakAction;
						DmWrite(symbol, (char *)(&symbol[index].debugAction) - (char *)symbol, &action, 1);
						drawFunctionList(symbol, functionIndexList, functionTop, functionCount);
					}
					do {
						EvtGetPen(&x, &y, &penDown);
					} while (penDown);
					EvtFlushPenQueue();
				}
				if (RctPtInRectangle(x, y, &traceRect)) {
					int entry = (y - FN_LIST_TOP) / 12;
					int index = functionIndexList[functionTop + entry];
					char action = symbol[index].debugAction;
					if ((action & TraceAction) == TraceAction) {
						action &= ~TraceAction;
						DmWrite(symbol, (char *)(&symbol[index].debugAction) - (char *)symbol, &action, 1);
						drawFunctionList(symbol, functionIndexList, functionTop, functionCount);
					}
					else {
						action |= TraceAction;
						DmWrite(symbol, (char *)(&symbol[index].debugAction) - (char *)symbol, &action, 1);
						drawFunctionList(symbol, functionIndexList, functionTop, functionCount);
					}
					do {
						EvtGetPen(&x, &y, &penDown);
					} while (penDown);
					EvtFlushPenQueue();
				}
				if (RctPtInRectangle(x, y, &okButton)) {
					p = &okButton;
				}
handleButton:				
				if (p) {				
					WinInvertRectangle(p, 1);
					do {
						EvtGetPen(&x, &y, &penDown);
					} while (penDown);
					WinInvertRectangle(p, 1);
					EvtFlushPenQueue();
					if (p == &okButton)
						break;
				}
			}
		}
		MemPtrFree(functionIndexList);
	}
}

#endif

static ULong installHandler(ULong *trapAddress, ULong trapHandler)
{
	ULong result;
	result = *trapAddress;
	*trapAddress = trapHandler;
	return result;
}

static void handleBusError(CharPtr pc, ULong *a6, ULong badAddress)
{
#if DEBUG_BREAKPOINTS
	UInt err;
	RectangleType fullScreen = { 0, 0, 160, 160 };
	WinHandle offscreen = NULL;
	WinHandle debugWindow = NULL;
	WinHandle onscreen = NULL;
#endif
	
	DebugData *dbgP = getDebugData();
	if (dbgP) {
		ULong curHandler = installHandler(BUS_ERROR_TRAP, dbgP->oldHandler_BusErr);
		if (dbgP->symbH && dbgP->resBase) {
			SymbolData *symbol = (SymbolData *)MemHandleLock(dbgP->symbH);
			int symbolCount = symbol[0].size;
			char buf[32];
			
#if DEBUG_BREAKPOINTS
			StrPrintF(buf, "Bus error : 0x%lx", badAddress);
			offscreen = WinCreateOffscreenWindow(160, 160, screenFormat, &err);
			if (offscreen) {
				debugWindow = WinCreateWindow(&fullScreen, noFrame, false, true, &err);
				if (debugWindow) {
					onscreen = WinSetDrawWindow(debugWindow);			
					if (onscreen) {
						WinCopyRectangle(0, offscreen, &fullScreen, 0, 0, scrCopy);
						doStackWindow(pc, dbgP, symbol, symbolCount, a6, buf, offscreen);
						WinSetDrawWindow(onscreen);
						WinCopyRectangle(offscreen, 0, &fullScreen, 0, 0, scrCopy);
					}
					WinDeleteWindow(debugWindow, false);
				}
				WinDeleteWindow(offscreen, false);
			}
#else
			StrPrintF(buf, "*** Bus error : 0x%lx ***\n", badAddress);
			writeStackTrace(pc, a6, dbgP, symbol, symbolCount, buf);
#endif
		}
		installHandler(BUS_ERROR_TRAP, curHandler);
	}
	SysReset();
}

static void handleAddressError(CharPtr pc, ULong *a6, ULong badAddress)
{
#if DEBUG_BREAKPOINTS
	UInt err;
	RectangleType fullScreen = { 0, 0, 160, 160 };
	WinHandle offscreen = NULL;
	WinHandle debugWindow = NULL;
	WinHandle onscreen = NULL;
#endif
	
	DebugData *dbgP = getDebugData();
	if (dbgP) {
		ULong curHandler = installHandler(ADDR_ERROR_TRAP, dbgP->oldHandler_AddrErr);
		
		if (dbgP->symbH && dbgP->resBase) {
			SymbolData *symbol = (SymbolData *)MemHandleLock(dbgP->symbH);
			int symbolCount = symbol[0].size;
			char buf[32];

#if DEBUG_BREAKPOINTS
			StrPrintF(buf, "Address error : 0x%lx", badAddress);
			offscreen = WinCreateOffscreenWindow(160, 160, screenFormat, &err);
			if (offscreen) {
				debugWindow = WinCreateWindow(&fullScreen, noFrame, false, true, &err);
				if (debugWindow) {
					onscreen = WinSetDrawWindow(debugWindow);			
					if (onscreen) {
						WinCopyRectangle(0, offscreen, &fullScreen, 0, 0, scrCopy);
						doStackWindow(pc, dbgP, symbol, symbolCount, a6, buf, offscreen);
						WinSetDrawWindow(onscreen);
						WinCopyRectangle(offscreen, 0, &fullScreen, 0, 0, scrCopy);
					}
					WinDeleteWindow(debugWindow, false);
				}
				WinDeleteWindow(offscreen, false);
			}
#else
			StrPrintF(buf, "*** Address error : 0x%lx ***\n", badAddress);
			writeStackTrace(pc, a6, dbgP, symbol, symbolCount, buf);
#endif
		}
		installHandler(ADDR_ERROR_TRAP, curHandler);
	}
	SysReset();
}

static void handleIllegalInstruction(CharPtr pc, ULong *a6)
{
#if DEBUG_BREAKPOINTS
	UInt err;
	RectangleType fullScreen = { 0, 0, 160, 160 };
	WinHandle offscreen = NULL;
	WinHandle debugWindow = NULL;
	WinHandle onscreen = NULL;
#endif
	
	DebugData *dbgP = getDebugData();
	if (dbgP) {
		ULong curHandler = installHandler(ILL_INSTR_TRAP, dbgP->oldHandler_AddrErr);
		
		if (dbgP->symbH && dbgP->resBase) {
			SymbolData *symbol = (SymbolData *)MemHandleLock(dbgP->symbH);
			int symbolCount = symbol[0].size;

#if DEBUG_BREAKPOINTS
			offscreen = WinCreateOffscreenWindow(160, 160, screenFormat, &err);
			if (offscreen) {
				debugWindow = WinCreateWindow(&fullScreen, noFrame, false, true, &err);
				if (debugWindow) {
					onscreen = WinSetDrawWindow(debugWindow);			
					if (onscreen) {
						WinCopyRectangle(0, offscreen, &fullScreen, 0, 0, scrCopy);
						doStackWindow(pc, dbgP, symbol, symbolCount, a6, "Illegal instruction", offscreen);
						WinSetDrawWindow(onscreen);
						WinCopyRectangle(offscreen, 0, &fullScreen, 0, 0, scrCopy);
					}
					WinDeleteWindow(debugWindow, false);
				}
				WinDeleteWindow(offscreen, false);
			}
#else
			writeStackTrace(pc, a6, dbgP, symbol, symbolCount, "*** Illegal instruction ***\n");
#endif
		}
		installHandler(ILL_INSTR_TRAP, curHandler);
	}
	SysReset();
}

static void handleDivideByZero(CharPtr pc, ULong *a6)
{
#if DEBUG_BREAKPOINTS
	UInt err;
	RectangleType fullScreen = { 0, 0, 160, 160 };
	WinHandle offscreen = NULL;
	WinHandle debugWindow = NULL;
	WinHandle onscreen = NULL;
#endif
	
	DebugData *dbgP = getDebugData();
	if (dbgP) {
		ULong curHandler = installHandler(DIVZ_TRAP, dbgP->oldHandler_Divz);
		if (dbgP->symbH && dbgP->resBase) {
			SymbolData *symbol = (SymbolData *)MemHandleLock(dbgP->symbH);
			int symbolCount = symbol[0].size;

#if DEBUG_BREAKPOINTS
			offscreen = WinCreateOffscreenWindow(160, 160, screenFormat, &err);
			if (offscreen) {
				debugWindow = WinCreateWindow(&fullScreen, noFrame, false, true, &err);
				if (debugWindow) {
					onscreen = WinSetDrawWindow(debugWindow);			
					if (onscreen) {
						WinCopyRectangle(0, offscreen, &fullScreen, 0, 0, scrCopy);
						
						doStackWindow(pc, dbgP, symbol, symbolCount, a6, "Divide by zero", offscreen);
						
						WinSetDrawWindow(onscreen);
						WinCopyRectangle(offscreen, 0, &fullScreen, 0, 0, scrCopy);
					}
					WinDeleteWindow(debugWindow, false);
				}
				WinDeleteWindow(offscreen, false);
			}
#else
			writeStackTrace(pc, a6, dbgP, symbol, symbolCount, "*** Divide by zero ***\n");
#endif
		}
		installHandler(DIVZ_TRAP, curHandler);
	}
	SysReset();
}

static asm trapHandler_BusErr()
{
	move.l	2(a7),d0		// the bad address
	move.l	10(a7),d1		// the return address
	move.l	d0,-(a7)
	move.l	a6,-(a7)
	move.l	d1,-(a7)
	jsr		handleBusError
	lea		0xc(a7),a7
	rte
}

static asm trapHandler_AddrErr()
{
	move.l	2(a7),d0		// the bad address
	move.l	10(a7),d1		// the return address
	move.l	d0,-(a7)
	move.l	a6,-(a7)
	move.l	d1,-(a7)
	jsr		handleAddressError
	lea		0xc(a7),a7
	rte
}

static asm trapHandler_IllInstr()
{
	move.l	2(a7),d0		// the return address	
	move.l	a6,-(a7)
	move.l	d0,-(a7)
	jsr		handleIllegalInstruction
	addq.l	#8,a7
	rte
}

static asm trapHandler_Divz()
{
	move.l	2(a7),d0		// the return address
	move.l	a6,-(a7)
	move.l	d0,-(a7)
	jsr		handleDivideByZero
	addq.l	#8,a7
	rte
}

#if DEBUG_BREAKPOINTS

static Boolean DebugSysHandleEvent(EventPtr eventP)
{
	DebugData *dbgP = getDebugData();
	if (dbgP) {
		if ((eventP->eType == keyDownEvent) &&
				(eventP->data.keyDown.chr == findChr)) {
			if (dbgP->symbH) {
				UInt err;
				SymbolData *symbol = (SymbolData *)MemHandleLock(dbgP->symbH);
				RectangleType fullScreen = { 0, 0, 160, 160 };
				WinHandle offscreen = NULL;
				WinHandle debugWindow = NULL;
				WinHandle onscreen = NULL;
				offscreen = WinCreateOffscreenWindow(160, 160, screenFormat, &err);
				if (offscreen) {
					debugWindow = WinCreateWindow(&fullScreen, noFrame, false, true, &err);
					if (debugWindow) {
						onscreen = WinSetDrawWindow(debugWindow);			
						if (onscreen) {
							WinCopyRectangle(0, offscreen, &fullScreen, 0, 0, scrCopy);
							doFunctionList(dbgP, symbol);
							WinSetDrawWindow(onscreen);
							WinCopyRectangle(offscreen, 0, &fullScreen, 0, 0, scrCopy);
						}
						WinDeleteWindow(debugWindow, false);
					}
					WinDeleteWindow(offscreen, false);
				}
				MemHandleUnlock(dbgP->symbH);
				return true;
			}
		}
		else {
			return (dbgP->oldNextEventHandler)(eventP);
		}
	}
	else
		;	// actually i don't know what to do in this case !!
	return false;
}
#endif

void DebugErrDisplayFileLineMsg(const Char * const filename, UInt lineNo, const Char * const msg)
{
	ULong *a6 = (ULong *)getA6();
	char *pc = (char *)(a6[1]); 
	DebugData *dbgP = getDebugData();
	if (dbgP->symbH && dbgP->resBase) {
		SymbolData *symbol = (SymbolData *)MemHandleLock(dbgP->symbH);
		int symbolCount = symbol[0].size;
		char buf[256];
		writeToTraceFile(dbgP, "*** ErrDisplayFileLineMsg ***\n");
		StrPrintF(buf, "File %s, line %d, '%s'\n", filename, lineNo, msg);
		writeStackTrace(pc, a6, dbgP, symbol, symbolCount, buf);
	}
	(dbgP->oldErrDisplayFileLineMsgEventHandler)(filename, lineNo, msg);
}

void DebugExit()
{
	DebugData *dbgP = getDebugData();
	if (dbgP) {
		ULong *a6 = (ULong *)getA6();			// frame for 'DebugEntry'
		char *frameA6 = (char *)(a6[0]);		// first frame of debuggee
		char *pc = (char *)(a6[1]);				// return address of 'DebugEntry' == debuggee function
		if (dbgP->symbH && dbgP->resBase) {
			SymbolData *symbol = (SymbolData *)MemHandleLock(dbgP->symbH);
			int symbolCount = symbol[0].size;
			int index = findSymbol(dbgP, pc, symbol, symbolCount);
			if (index != -1) {
			}
		}
	}
}

void DebugInit(DebugData *dbgP)
{
	RectangleType fullScreen = { 0, 0, 160, 160 };
	WinHandle offscreen = NULL;
	WinHandle debugWindow = NULL;
	WinHandle onscreen = NULL;
	ULong *a6 = (ULong *)getA6();
	int *frameA6 = (int *)(a6[0]);		// get frame of previous fn, should be PilotMain's
	char *pc = (char *)(a6[1]);			// return address, should be PilotMain + 8
	int cmd = frameA6[4];				// the incoming 'cmd' argument to PilotMain
	Err fErr;
	DWord value;
//	DbgBreak();
	if (cmd == sysAppLaunchCmdNormalLaunch) {
		SysAppInfoPtr appInfoP;
		dbgP->nullLink = 0;			
		dbgP->version = 'DBG1';			
		dbgP->resBase = 0;
		dbgP->ramSize = 2048L * 1024;
		dbgP->traceID = -1;
		dbgP->symbH = DmGet1Resource('Symb', 1);
		dbgP->a6Prime = *((ULong *)frameA6);
		*((ULong *)frameA6) = (ULong)dbgP;
		
#if DEBUG_BREAKPOINTS
		dbgP->oldNextEventHandler = SysGetTrapAddress(sysTrapSysHandleEvent);
		SysSetTrapAddress(sysTrapSysHandleEvent, DebugSysHandleEvent);
#endif
		dbgP->oldErrDisplayFileLineMsgEventHandler = SysGetTrapAddress(sysTrapErrDisplayFileLineMsg);
		SysSetTrapAddress(sysTrapErrDisplayFileLineMsg, DebugErrDisplayFileLineMsg);

		fErr = FtrGet('pose', 0, &value);
		if (fErr) {	// i.e. we couldn't get the feature, so not running under Pose
			dbgP->oldHandler_BusErr = installHandler(BUS_ERROR_TRAP, (ULong)trapHandler_BusErr);
			dbgP->oldHandler_AddrErr = installHandler(ADDR_ERROR_TRAP, (ULong)trapHandler_AddrErr);
			dbgP->oldHandler_IllInstr = installHandler(ILL_INSTR_TRAP, (ULong)trapHandler_IllInstr);
			dbgP->oldHandler_Divz = installHandler(DIVZ_TRAP, (ULong)trapHandler_Divz);
		}
		
		appInfoP = (SysAppInfoPtr)getCurAppInfo();
		appInfoP->extraP = (char *)dbgP;

		if (dbgP->symbH) {
			SymbolData *symbol = (SymbolData *)MemHandleLock(dbgP->symbH);
			int symbolCount = symbol[0].size;
			int i;
#if DEBUG_BREAKPOINTS
			UInt err;
#endif
			for (i = 0; i < symbolCount; i++) {
				if (symbol[i].flag == SymCode) {
#if !DEBUG_BREAKPOINTS
					char action = symbol[i].debugAction;
					action |= TraceAction;
					DmWrite(symbol, (char *)(&symbol[i].debugAction) - (char *)symbol, &action, 1);
#endif
					if (StrStr(symbol[i].name, "PilotMain") == symbol[i].name) {
						dbgP->resBase = ((ULong)pc - 12) - symbol[i].offset;
						break;
					}
				}
			}

#if DEBUG_BREAKPOINTS
			offscreen = WinCreateOffscreenWindow(160, 160, screenFormat, &err);
			if (offscreen) {
				debugWindow = WinCreateWindow(&fullScreen, noFrame, false, true, &err);
				if (debugWindow) {
					onscreen = WinSetDrawWindow(debugWindow);			
					if (onscreen) {
						WinCopyRectangle(0, offscreen, &fullScreen, 0, 0, scrCopy);
						doFunctionList(dbgP, symbol);
						WinSetDrawWindow(onscreen);
						WinCopyRectangle(offscreen, 0, &fullScreen, 0, 0, scrCopy);
					}
					WinDeleteWindow(debugWindow, false);
				}
				WinDeleteWindow(offscreen, false);
			}
			MemHandleUnlock(dbgP->symbH);
#endif			
		}
		openTraceFile(dbgP);
	}
}

void DebugTerm(DebugData *dbgP)
{
	Err err;
	DWord value;
	ULong *a6 = (ULong *)getA6();
	int *frameA6 = (int *)(a6[0]);		// get frame of previous fn, should be PilotMain's
	int cmd = frameA6[4];				// the incoming 'cmd' argument to PilotMain
//	DbgBreak();
	if (cmd == sysAppLaunchCmdNormalLaunch) {	
		if (dbgP->version == 'DBG1') {
			*((ULong *)frameA6) = dbgP->a6Prime;	
			if (dbgP->symbH) {
				DmReleaseResource(dbgP->symbH);
			}
			err = FtrGet('pose', 0, &value);
			if (err) {	// i.e. we couldn't get the feature, so not running under Pose
				installHandler(BUS_ERROR_TRAP, dbgP->oldHandler_BusErr);
				installHandler(ADDR_ERROR_TRAP, dbgP->oldHandler_AddrErr);
				installHandler(ILL_INSTR_TRAP, dbgP->oldHandler_IllInstr);
				installHandler(DIVZ_TRAP, dbgP->oldHandler_Divz);
			}
#if DEBUG_BREAKPOINTS
			SysSetTrapAddress(sysTrapSysHandleEvent, dbgP->oldNextEventHandler);
#endif
			SysSetTrapAddress(sysTrapErrDisplayFileLineMsg, dbgP->oldErrDisplayFileLineMsgEventHandler);
		}
	}
}

#if DEBUG_BREAKPOINTS

static void drawDebugWindow(Boolean functionList, char *msg)
{
	RectangleType okRect = { OK_BTN_LEFT, OK_BTN_TOP, OK_BTN_WIDTH, OK_BTN_HEIGHT };
	RectangleType fullScreen = { 0, 0, 160, 160 };
	WinEraseRectangle(&fullScreen, 0);	
	
	if (msg) {
		FontID curFont = FntSetFont(boldFont);
		WinDrawChars(msg, StrLen(msg), MSG_LEFT, MSG_TOP);
		FntSetFont(curFont);
	}
	
	if (functionList) {
		RectangleType functionListRect = { FN_LIST_LEFT, FN_LIST_TOP, FN_LIST_WIDTH, FN_LIST_HEIGHT };
		WinDrawRectangleFrame(simpleFrame, &functionListRect);
		WinDrawChar('A', FN_LIST_TRACE_LEFT + 1, FN_LIST_TOP - 12);
		WinDrawChar('A', FN_LIST_BREAK_LEFT + 2, FN_LIST_TOP - 12);
		WinDrawChar('N', FN_LIST_TRACE_LEFT + 1, FN_LIST_TOP + FN_LIST_HEIGHT + 1);
		WinDrawChar('N', FN_LIST_BREAK_LEFT + 2, FN_LIST_TOP + FN_LIST_HEIGHT + 1);
	}
	else {
		RectangleType fnsRect = { FNS_BTN_LEFT, FNS_BTN_TOP, FNS_BTN_WIDTH, FNS_BTN_HEIGHT };
		RectangleType screenRect = { SCREEN_BTN_LEFT, SCREEN_BTN_TOP, SCREEN_BTN_WIDTH, SCREEN_BTN_HEIGHT };
		RectangleType stackTraceRect = { STACK_TRACE_LEFT, STACK_TRACE_TOP, STACK_TRACE_WIDTH, STACK_TRACE_HEIGHT };
		RectangleType varListRect = { VAR_LIST_LEFT, VAR_LIST_TOP, VAR_LIST_WIDTH, VAR_LIST_HEIGHT };
		WinDrawRectangleFrame(simpleFrame, &stackTraceRect);
		WinDrawRectangleFrame(simpleFrame, &varListRect);
		WinDrawRectangleFrame(roundFrame, &fnsRect);
		WinDrawChars("Fns", 3, FNS_BTN_LEFT + 8, FNS_BTN_TOP);
		WinDrawRectangleFrame(roundFrame, &screenRect);
		WinDrawChars("Screen", 6, SCREEN_BTN_LEFT + 2, SCREEN_BTN_TOP);
	}
	WinDrawRectangleFrame(roundFrame, &okRect);
	WinDrawChars("OK", 2, OK_BTN_LEFT + 8, OK_BTN_TOP);
	
}

static void drawFunctionList(SymbolData *symbol, int* functionIndexList, int functionTop, int functionCount)
{
	int count = 0;
	FontID curFont;
	RectangleType functionListRect = { FN_LIST_LEFT, FN_LIST_TOP, FN_LIST_WIDTH, FN_LIST_HEIGHT };
	WinEraseRectangle(&functionListRect, 0);
	WinDrawLine(FN_LIST_TRACE_LEFT + FN_LIST_FLAG_WIDTH, FN_LIST_TOP, FN_LIST_TRACE_LEFT + FN_LIST_FLAG_WIDTH, FN_LIST_TOP + FN_LIST_HEIGHT);
	WinDrawLine(FN_LIST_BREAK_LEFT + FN_LIST_FLAG_WIDTH, FN_LIST_TOP, FN_LIST_BREAK_LEFT + FN_LIST_FLAG_WIDTH, FN_LIST_TOP + FN_LIST_HEIGHT);
	curFont = FntSetFont(symbolFont);
	if (functionTop > 0)
		WinDrawChar(0x8, FN_LIST_LEFT + FN_LIST_WIDTH - 11, FN_LIST_TOP + 1);
	if ((functionTop + FN_LIST_COUNT) < functionCount)
		WinDrawChar(0x7, FN_LIST_LEFT + FN_LIST_WIDTH - 11, FN_LIST_TOP + FN_LIST_HEIGHT - 10);
	FntSetFont(curFont);

	while (true) {
		int index = functionIndexList[functionTop];
		if ((symbol[index].debugAction & TraceAction) == TraceAction)
			WinDrawChar('T', FN_LIST_TRACE_LEFT + 1, FN_LIST_TOP + (count * 12));
		if ((symbol[index].debugAction & BreakAction) == BreakAction)
			WinDrawChar('B', FN_LIST_BREAK_LEFT + 2, FN_LIST_TOP + (count * 12));
		WinDrawChars(symbol[index].name, StrLen(symbol[index].name), FN_LIST_NAME_LEFT + 2, FN_LIST_TOP + (count * 12));
		count++;
		if (count == FN_LIST_COUNT)
			break;
		functionTop++;
		if (functionTop == functionCount)
			break;
	}
}

#endif

static Boolean goodPtr(void *p, Boolean oddOK, ULong ramSize, Boolean isEZ)
{
	if (!oddOK && ((ULong)p & 0x1))
		return false;

	if ( (ULong)p < 0x20000 ) {	// assume full 128k of DRAM !!
		return true;	
	}
	
	
	if (isEZ) {		// base at 0
		return ((ULong)p < ramSize);
	}
	else {			// base at 0x10000000
		return  ((ULong)p >= 0x10000000) && ( (ULong)p < (0x10000000 + ramSize));
	}


}

static void getVarValue(SymbolData *symbol, int index, char *buf, char *frameA6, ULong ramSize)
{
	char tBuf[64];
	char vBuf[64];
	Boolean isEZ = ((symbol[0].offset & 1) == 1);

	StrPrintF(tBuf, "%s = ", symbol[index].name);
	if (symbol[index].size == 1) {
		if (symbol[index].type == DBG_Signed)
			StrPrintF(vBuf, "%d", *((char *)(frameA6 + symbol[index].offset)) );
		else
			StrPrintF(vBuf, "%u", (long)(*((unsigned char *)(frameA6 + symbol[index].offset))) );
		StrCat(tBuf, vBuf);
	}
	else {
		if (symbol[index].size == 2) {
			if (symbol[index].type == DBG_Signed)
				StrPrintF(vBuf, "%d", *((int *)(frameA6 + symbol[index].offset)) );
			else
				StrPrintF(vBuf, "%u", (long)(*((unsigned int *)(frameA6 + symbol[index].offset))) );
			StrCat(tBuf, vBuf);
		}
		else {
			if (symbol[index].size == 4) {
				switch (symbol[index].type) {
					case DBG_CharPointer : {
							char *d = *((char **)(frameA6 + symbol[index].offset));
							if (d == NULL)
								StrPrintF(vBuf, "<null>");
							else {
								if (!goodPtr(d, true, ramSize, isEZ)) {
									StrPrintF(vBuf, "0x%lx", d);
								}
								else {
									if (StrLen(d) > 61) {
										int x = 1;
										vBuf[0] = '"';
										while (x < 63) {
											vBuf[x] = d[x - 1];
											x++;
										}
										vBuf[x] = '"';
										vBuf[x + 1] = 0;
									}
									else
										StrPrintF(vBuf, "\"%s\"", d);
								}
							}
						}
						break;
					case DBG_FormPointer : {
							FormPtr frmP = *((FormPtr *)(frameA6 + symbol[index].offset));
							if (frmP == NULL)
								StrPrintF(vBuf, "<null>");
							else {
								Boolean goodForm = false;
								if (goodPtr(frmP, false, ramSize, isEZ)) {
									if (goodPtr(frmP->objects, false, ramSize, isEZ)) {
										int x;
										for (x = 0; x < frmP->numObjects; x++) {
											if ((frmP->objects[x].objectType < frmFieldObj)
													|| (frmP->objects[x].objectType > frmScrollBarObj))
												break;
											else {
												if (frmP->objects[x].objectType == frmTitleObj) {
													FormTitleType *t = (FormTitleType *)(frmP->objects[x].object.title);
													if (goodPtr(t, false, ramSize, isEZ) && goodPtr(t->text, true, ramSize, isEZ)) {
														goodForm = true;
														StrPrintF(vBuf, "[%d, \"%s\"]", frmP->formId, t->text);
													}
												}
												break;
											}
										}
									}	
								}
								if (!goodForm)
									StrPrintF(vBuf, "0x%lx", frmP);
							}
						}
						break;
					case DBG_EventPointer : {
							EventPtr evtP = *((EventPtr *)(frameA6 + symbol[index].offset));
							if (evtP == NULL)
								StrPrintF(vBuf, "<null>");
							else {
								if (!goodPtr(evtP, false, ramSize, isEZ)) {
									StrPrintF(vBuf, "0x%lx", evtP);
								}
								else {
									switch (evtP->eType) {
										case appStopEvent :
											StrPrintF(vBuf, "[appStop]");
											break;
										case nilEvent :
											StrPrintF(vBuf, "[nilEvent]");
											break;
										case winEnterEvent :
											StrPrintF(vBuf, "[winEnter, 0x%lx]", evtP->data.winEnter.enterWindow);
											break;
										case winExitEvent :
											StrPrintF(vBuf, "[winExit, 0x%lx]", evtP->data.winExit.exitWindow);
											break;
										case keyDownEvent :
											StrPrintF(vBuf, "[keyDown, 0x%x]", evtP->data.keyDown.chr);
											break;
										case penDownEvent :
											StrPrintF(vBuf, "[penDown, %d, %d]", evtP->screenX, evtP->screenY);
											break;
										case menuEvent :
											StrPrintF(vBuf, "[menu, %d]", evtP->data.menu.itemID);
											break;
										case frmLoadEvent :
											StrPrintF(vBuf, "[frmLoad, %d]", evtP->data.frmLoad.formID);
											break;
										case frmOpenEvent :
											StrPrintF(vBuf, "[frmOpen, %d]", evtP->data.frmOpen.formID);
											break;
										case frmCloseEvent :
											StrPrintF(vBuf, "[frmClose, %d]", evtP->data.frmClose.formID);
											break;
										case ctlSelectEvent :
											StrPrintF(vBuf, "[ctlSelect, %d]", evtP->data.ctlSelect.controlID);
											break;
										case ctlRepeatEvent :
											StrPrintF(vBuf, "[ctlRepeat, %d]", evtP->data.ctlRepeat.controlID);
											break;
										default :
											StrPrintF(vBuf, "[event %d]", evtP->eType);
											break;
									}
								}
							}
						}
						break;
					case DBG_Signed :
						StrPrintF(vBuf, "%ld", *((long *)(frameA6 + symbol[index].offset)) );
						break;
					case DBG_Unsigned :
					case DBG_Pointer :
					default:
						StrPrintF(vBuf, "0x%lx", *((long *)(frameA6 + symbol[index].offset)) );
						break;
				}
				StrCat(tBuf, vBuf);
			}
			else {
				StrPrintF(vBuf, "@0x%lx", *((long *)(frameA6 + symbol[index].offset)) );
				StrCat(tBuf, vBuf);
			}
		}
	}
	StrCopy(buf, tBuf);
}

#if DEBUG_BREAKPOINTS

static int countVars(SymbolData *symbol, int symbolCount, int top)
{
	while ((top < symbolCount) && (symbol[top].flag == SymLocal)) {
		top++;
	}
	return top;
}

static void drawVarList(SymbolData *symbol, int symbolCount, int top, int varMin, int varMax, char *frameA6, ULong ramSize)
{
	int count = 0;
	FontID curFont;
	RectangleType varListRect = { VAR_LIST_LEFT, VAR_LIST_TOP, VAR_LIST_WIDTH, VAR_LIST_HEIGHT };
	WinEraseRectangle(&varListRect, 0);
	curFont = FntSetFont(symbolFont);
	if (top > varMin)
		WinDrawChar(0x8, VAR_LIST_LEFT + VAR_LIST_WIDTH - 11, VAR_LIST_TOP + 1);
	if ((top + VAR_LIST_COUNT) < varMax)
		WinDrawChar(0x7, VAR_LIST_LEFT + VAR_LIST_WIDTH - 11, VAR_LIST_TOP + VAR_LIST_HEIGHT - 10);
	FntSetFont(curFont);

	while ((count < VAR_LIST_COUNT) && (top < symbolCount)) {
		if (symbol[top].flag == SymLocal) {
			char tBuf[64];
			int length;
			getVarValue(symbol, top, tBuf, frameA6, ramSize);
			length = StrLen(tBuf);
			length = truncateString(tBuf, length, (VAR_LIST_WIDTH - 2) );
			WinDrawChars(tBuf, length, VAR_LIST_LEFT + 1, VAR_LIST_TOP + (count * 12));
			count++;
			top++;
		}
		else
			break;
	}

}

static void drawStackTrace(SymbolData *symbol, CharPtr *stackTraceNames, int stackTraceTop, int stackTraceCount, int entry)
{
	int count = 0;
	int i = stackTraceTop;
	FontID curFont;
	RectangleType stackTraceRect = { STACK_TRACE_LEFT, STACK_TRACE_TOP, STACK_TRACE_WIDTH, STACK_TRACE_HEIGHT };
	WinEraseRectangle(&stackTraceRect, 0);
	curFont = FntSetFont(symbolFont);
	if (stackTraceTop > 0)
		WinDrawChar(0x8, STACK_TRACE_LEFT + STACK_TRACE_WIDTH - 11, STACK_TRACE_TOP + 1);
	if ((stackTraceTop + STACK_TRACE_COUNT) < stackTraceCount)
		WinDrawChar(0x7, STACK_TRACE_LEFT + STACK_TRACE_WIDTH - 11, STACK_TRACE_TOP + STACK_TRACE_HEIGHT - 10);
	FntSetFont(curFont);
	
	while (i < stackTraceCount) {
		if (stackTraceNames[i])
			WinDrawChars(stackTraceNames[i], StrLen(stackTraceNames[i]), STACK_TRACE_LEFT + 2, STACK_TRACE_TOP + (count * 12));
		else
			WinDrawChars("<unknown>", 9, STACK_TRACE_LEFT + 2, STACK_TRACE_TOP + (count * 12));
		count++;
		if (count == STACK_TRACE_COUNT)
			break;
		i++;
	}
	if ((entry >= stackTraceTop) && (entry < (stackTraceTop + STACK_TRACE_COUNT))) {
		RectangleType r = { STACK_TRACE_LEFT, STACK_TRACE_TOP, STACK_TRACE_WIDTH, 12 };
		r.topLeft.y += (entry - stackTraceTop) * 12;
		WinInvertRectangle(&r, 0);
	}
}
#endif


static int buildStackTrace(char *pc, ULong *a6, DebugData *dbgP, SymbolData *symbol, int symbolCount, int *stackTrace, CharPtr *stackTraceNames, ULong *frameStack)
{
	int count = 0;
	while (a6 && count < MAX_STACK_TRACE) {
		int index = findSymbol(dbgP, pc, symbol, symbolCount);
		CharPtr name;
		frameStack[count] = (ULong)a6;
		if (index == -1)
			name = findMacsbugName((int *)pc);
		else
			name = symbol[index].name;
		stackTraceNames[count] = name;
		stackTrace[count++] = index;

		if (StrCompare(name, "PilotMain") == 0)
			break;
		// bump to next frame, but get the return address from this frame first
		pc = (char *)(a6[1]);
		a6 = (ULong *)(a6[0]);
	}
	return count;
}

static int findSymbol(DebugData *dbgP, char *pc, SymbolData *symbol, int symbolCount)
{
	int i;
	ULong pcOffset = (ULong)pc - dbgP->resBase;
	for (i = 0; i < symbolCount; i++) {
		if (symbol[i].flag == SymCode) {
			if ((pcOffset >= symbol[i].offset) 
				&& (pcOffset < (symbol[i].offset + symbol[i].size)))
			return i;
		}
	}
	return -1;
}
#define DbgHdrLength 16

static void openTraceFile(DebugData *dbgP)
{
	char *DebugHeader = "OnBoardC Trace:\n";	// pc-relative string literal, should be ok
	DmOpenRef memoDB = DmOpenDatabaseByTypeCreator('DATA', 'memo', dmModeReadWrite);
	if (memoDB != NULL) {
		int i, count = DmNumRecords(memoDB);
		Handle memoH;
		char *p;
		for (i = 0; i < count; i++) {
			Handle memoH = (Handle)DmQueryRecord(memoDB, i);
			if (memoH) {
				p = (char *)MemHandleLock(memoH);
				if (StrStr(p, DebugHeader) == p) {
					MemHandleUnlock(memoH);
					DmRemoveRecord(memoDB, i);
					break;					
				}
				MemHandleUnlock(memoH);
			}
		}
		memoH = (Handle)DmNewRecord(memoDB, &dbgP->traceID, DbgHdrLength + 1);
		if (memoH) {
			p = (char *)MemHandleLock(memoH);
			DmWrite(p, 0, DebugHeader, DbgHdrLength);						
			i = 0;
			DmWrite(p, DbgHdrLength, &i, 1);
			MemHandleUnlock(memoH);
			DmReleaseRecord(memoDB, dbgP->traceID, true);
		}
		DmCloseDatabase(memoDB);
	}				
}

void writeToTraceFile(DebugData *dbgP, char *message)
{
	if (dbgP->traceID != -1) {
		DmOpenRef memoDB = DmOpenDatabaseByTypeCreator('DATA', 'memo', dmModeReadWrite);
		if (memoDB != NULL) {
			int messageLength = StrLen(message);			
			Handle memoH = (Handle)DmGetRecord(memoDB, dbgP->traceID);
			if (memoH) {
				ULong curSize = MemHandleSize(memoH);
				if ((curSize + messageLength) >= 4096) {
					char *p = (char *)MemHandleLock(memoH);
					DmWrite(p, DbgHdrLength, p + 1024, (curSize - 1024 - DbgHdrLength));
					MemHandleUnlock(memoH);
					curSize -= 1024;					
				}				
				memoH = DmResizeRecord(memoDB, dbgP->traceID, curSize + messageLength);
				if (memoH) {
					char *p = (char *)MemHandleLock(memoH);
					DmWrite(p, curSize - 1, message, messageLength + 1);
					MemHandleUnlock(memoH);
				}
				DmReleaseRecord(memoDB, dbgP->traceID, true);
				DmCloseDatabase(memoDB);
			}
		}				
	}
}



static int truncateString(CharPtr s, int length, int max)
{
	while (FntCharsWidth(s, length) >= max ) {
		s[--length] = '\0';
	}
	return length;
}

static asm CharPtr getA6()
{
	move.l	a6,a0
	rts
}

static asm CharPtr getA5()
{
	move.l	a5,a0
	rts
}

static asm void *getCurAppInfo()
{
	move.l	(a5),a0
	rts
}

static DebugData *getDebugData()
{
	ULong *a6 = (ULong *)getA6();
	DebugData *dbgP;
	while (a6) { dbgP = (DebugData *)a6; a6 = (ULong *)(*a6); }
	if (dbgP->version == 'DBG1')
		return dbgP;
	else {
		SysAppInfoPtr appInfoP = (SysAppInfoPtr)getCurAppInfo();
		dbgP = (DebugData *)(appInfoP->extraP);
		if (dbgP->version == 'DBG1')
			return dbgP;
		else
			return NULL;
	}
}

static char *findBase(int *pc)
{
	int i = 0;
	while ((i > -4000) && (pc[i] != 0x4e56)) i--;
	if (pc[i] == 0x4e56)
		return (char *)(&pc[i]);
	else
		return NULL;
}

static CharPtr findMacsbugName(int *pc)
{
	int i = 0;
	while ((i < 4000) && (pc[i] != 0x4e75)) i++;
	if (pc[i] == 0x4e75) {
		CharPtr p = (CharPtr)(&pc[i + 1]);
		if (*p & 0x80) {
			return p + 1;
		}
		return p;
	}
	return NULL;
}

void writeStackTrace(char *pc, ULong *a6, DebugData *dbgP, SymbolData *symbol, int symbolCount, char *message)
{
	writeToTraceFile(dbgP, message);
	// build the stack trace and write all data to the trace file.
	
	while (a6) {
		int index = findSymbol(dbgP, pc, symbol, symbolCount);
		CharPtr name;
		if (index == -1)
			name = findMacsbugName((int *)pc);
		else
			name = symbol[index].name;

		writeToTraceFile(dbgP, name);
		writeToTraceFile(dbgP, "\n");
		if (index != -1) {
			int i = index + 1;
			while ((symbol[i].flag == SymLocal) && (i < symbolCount)) {
				if (symbol[i].offset > 0) {
					char tBuf[64];
					writeToTraceFile(dbgP, "\t");
					getVarValue(symbol, i, tBuf, (char *)a6, dbgP->ramSize);
					writeToTraceFile(dbgP, tBuf);
					writeToTraceFile(dbgP, "\n");
				}
				i++;
			}
		}
		
		if (StrCompare(name, "PilotMain") == 0)
			break;
		// bump to next frame, but get the return address from this frame first
		pc = (char *)(a6[1]);
		a6 = (ULong *)(a6[0]);
	}

}

void EndOfDebugCode()
{
}

#endif
