/***********************************************************************
 *
 * This file is part of the OnBoard C package.  For more information on 
 * using OnBoard C, see http://groups.yahoo.com/group/onboardc.  For 
 * more information on developing OnBoard C (including submission of 
 * bug reports), see http://sourceforge.net/projects/onboardc.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 ***********************************************************************/

#include <PalmOS.h>
#include <PalmCompatibility.h>

#define version20	0x02000000

#include "OnBoardC.h"
#include "OnBoardC_res.h"

//#include "LibTokens.h"

IncludeData includeData[MaxInclude];
int gIncludeTop;
int gLineCount;
DmOpenRef gOutputDB;

CharPtr macroEndPtrStack[MaxTextStack];
CharPtr curTextPtrStack[MaxTextStack];
int macroListTopStack[MaxTextStack];
CharPtr macroEndPtr;
int textStackTop;

int maxMacros;
int macroRecordIndex;
MacroDef *macroList;
Handle macroH = NULL;
int macroListTop;
int savedMacroListTop;

HeaderBlock *gHB;
Handle headerData;
Handle stringData;
Boolean freezeHeader;

SkipState gSkipState[MAXSKIP];
int gSkipTop;

int gStaticIndex = 0;


Boolean gMatchReservedWords = true;

char *HeaderName = "OnBoardHeader.h";

typedef struct {
	char *name;
	int length;
	Int token;
} ReservedWordEntry;

ReservedWordEntry reservedWordList[] = {



	{ "asm", 3, Asm },
	{ "auto", 4, Auto },
	{ "break", 5, Break },
	{ "case", 4, Case },
	{ "char", 4, Char_Token },
	{ "continue", 8, Continue },
	{ "const", 5, Const },
	{ "default", 7, Default },
	{ "do", 2, Do },
	{ "double", 6, Double_Token },
	{ "else", 4, Else },
	{ "enum", 4, Enum },
	{ "extern", 6, Extern },
	{ "float", 5, Float_Token },
	{ "for", 3, For },
	{ "goto", 4, Goto },
	{ "if", 2, If },
	{ "int", 3, Int_Token },
	{ "long", 4, Long_Token },
	{ "register", 8, Register },
	{ "return", 6, Return },
	{ "short", 5, Short_Token },
	{ "sizeof", 6, Sizeof },
	{ "static", 6, Static },
	{ "struct", 6, Struct },
	{ "switch", 6, Switch },
	{ "typedef", 7, Typedef },
	{ "union", 5, Union },
	{ "unsigned", 8, Unsigned_Token },
	{ "void", 4, Void_Token },
	{ "while", 5, While },
	{ "+", 0, Token_Error }
};

char reservedWordIndex[26] = 
//		  a   b   c   d   e   f   g   h   i   j   k   l   m
		{ 0,  2,  3,  7, 10, 13, 15, -1, 16, -1, -1, 18, -1,
		 -1, -1, -1, -1, 19, 21, 26, 27, 29, 30, -1, -1, -1 };
//		  n   o   p   q   r   s   t   u   v   w   x   y   z	 

Boolean reservedWordHashHit[128];

Token charTokenMap[256] = {
				EndOfInput, NotACharacter, NotACharacter, NotACharacter,
				NotACharacter, NotACharacter, NotACharacter, NotACharacter,
				NotACharacter,
				Whitespace,								// 09 (tab)
				Whitespace,								// 0A (LF)
				NotACharacter,		
				NotACharacter,		
				Whitespace,								// 0D (CR)
				NotACharacter,		
				NotACharacter,		
				NotACharacter, NotACharacter, NotACharacter, NotACharacter,	// 10
				NotACharacter, NotACharacter, NotACharacter, NotACharacter,
				NotACharacter, NotACharacter, NotACharacter, NotACharacter,
				NotACharacter, NotACharacter, NotACharacter, NotACharacter,
				Whitespace,		// 20 (space)
				Not, DoubleQuote, Number,
				Dollar, Remainder, And, Quote, 
				LeftParen, RightParen, Multiply, Plus, 
				Comma, Minus, Period, Divide,
				Digit,			// 30 (0)
				Digit,
				Digit,
				Digit,
				Digit,
				Digit,
				Digit,
				Digit,
				Digit,
				Digit,
				Colon, SemiColon, Less, Equal,
				Greater, Question,
				Operator,		// 40
				HexDigit, HexDigit, HexDigit, HexDigit,
				HexDigit, HexDigit, Letter, Letter,
				Letter, Letter, Letter, Letter,
				Letter, Letter, Letter, Letter,
				Letter, Letter, Letter, Letter,
				Letter, Letter, Letter, Letter,
				Letter, Letter,
				LeftBracket,		// 5B
				Backslash, RightBracket, Xor, Letter,
				Operator,		// 60
				HexDigit, HexDigit, HexDigit, HexDigit,
				HexDigit, HexDigit, Letter, Letter,
				Letter, Letter, Letter, Letter,
				Letter, Letter, Letter, Letter,
				Letter, Letter, Letter, Letter,
				Letter, Letter, Letter, Letter,
				Letter, Letter,
				LeftCurly,			// 7B
				Or, RightCurly, Tilde,
				NotACharacter,		// 7F (DEL)
				
				// treat all the others as letters for now - allows wacky identifers and such
				Letter, Letter, Letter, Letter, Letter, Letter, Letter, Letter,	// 80
				Letter, Letter, Letter, Letter, Letter, Letter, Letter, Letter, // 90
				Whitespace, Letter, Letter, Letter, Letter, Letter, Letter, Letter, // A0 (160 == nbsp)
				Letter, Letter, Letter, Letter, Letter, Letter, Letter, Letter,
				Letter, Letter, Letter, Letter, Letter, Letter, Letter, Letter,
				Letter, Letter, Letter, Letter, Letter, Letter, Letter, Letter,
				Letter, Letter, Letter, Letter, Letter, Letter, Letter, Letter,
				Letter, Letter, Letter, Letter, Letter, Letter, Letter, Letter,
				Letter, Letter, Letter, Letter, Letter, Letter, Letter, Letter,
				Letter, Letter, Letter, Letter, Letter, Letter, Letter, Letter,
				Letter, Letter, Letter, Letter, Letter, Letter, Letter, Letter,
				Letter, Letter, Letter, Letter, Letter, Letter, Letter, Letter,
				Letter, Letter, Letter, Letter, Letter, Letter, Letter, Letter,
				Letter, Letter, Letter, Letter, Letter, Letter, Letter, Letter,
				Letter, Letter, Letter, Letter, Letter, Letter, Letter, Letter,
				Letter, Letter, Letter, Letter, Letter, Letter, Letter, Letter
};

char *DefineName = "define";
int DefineLength = 6;
unsigned char DefineHash = 'd' ^ 'e' ^ 'f' ^ 'i' ^ 'n' ^ 'e';
char *IncludeName = "include";
int IncludeLength = 7;
unsigned char IncludeHash = 'i' ^ 'n' ^ 'c' ^ 'l' ^ 'u' ^ 'd' ^ 'e';
char *IfdefName = "ifdef";
int IfdefLength = 5;
unsigned char IfdefHash = 'i' ^ 'f' ^ 'd' ^ 'e' ^ 'f';
char *IfndefName = "ifndef";
int IfndefLength = 6;
unsigned char IfndefHash = 'i' ^ 'f' ^ 'n' ^ 'd' ^ 'e' ^ 'f';
char *EndifName = "endif";
int EndifLength = 5;
unsigned char EndifHash = 'e' ^ 'n' ^ 'd' ^ 'i' ^ 'f';
char *PragmaName = "pragma";
int PragmaLength = 6;
unsigned char PragmaHash = 'p' ^ 'r' ^ 'a' ^ 'g' ^ 'm' ^ 'a';
char *UndefName = "undef";
int UndefLength = 5;
unsigned char UndefHash = 'u' ^ 'n' ^ 'd' ^ 'e' ^ 'f';

void initReservedWordMap()
{
	int i;
	for (i = 0; i < sizeof(reservedWordHashHit); i++)
		reservedWordHashHit[i] = false;
		
	for (i = 0; i < (sizeof(reservedWordList) / sizeof(ReservedWordEntry)); i++) {
		int j;
		char *p = reservedWordList[i].name;
		unsigned char hash = p[0];
		for (j = 1; j < reservedWordList[i].length; j++)
			hash ^= p[j];
		reservedWordHashHit[hash] = true;
	}
}

Boolean findReservedWord(CharPtr startTextPtr, int length, int *token)
{
	Char initial = *startTextPtr;
	ReservedWordEntry *re;
	int index = reservedWordIndex[initial - 'a'];

	if (!gMatchReservedWords) return false;
	
	if ((initial < 'a') || (initial > 'z')) return false;	
	if (index == -1) return false;

	re = &reservedWordList[index];
	while (true) {
		if (re->length == length) {
			int x;
			for (x = 0; x < length; x++) {
				if (re->name[x] != startTextPtr[x])
					break;
			}
			if (x == length) {
				*token = re->token;
				return true;
			}
		}
		re++;
		if (*re->name != initial)
			break;
	}
	
	return false;
}

int gMacroHashTable[128];

CharPtr popMacro(CharPtr curTextPtr)
{
	int i;
	while (curTextPtr == macroEndPtr) {
		textStackTop--;
		curTextPtr = curTextPtrStack[textStackTop];
		macroEndPtr = macroEndPtrStack[textStackTop];
		for (i = macroListTop - 1; i >= macroListTopStack[textStackTop]; i--) {
			gMacroHashTable[macroList[i].hash] = macroList[i].nextMacro;
			if (gMacroHashTable[macroList[i].hash] != -1)
				if ((macroList + gMacroHashTable[macroList[i].hash])->nextMacro == macroList[i].hash)
					error("macro cycle");
		}
		macroListTop = macroListTopStack[textStackTop];
	}
	return curTextPtr;
}

double gFPDigit;
//Boolean gReturnWhitespace = false;

CharPtr getFPDigit(long value, CharPtr curTextPtr, int *token)
{
	double dValue = value;
	value = 10;
	curTextPtr++;
	while ((charTokenMap[(UChar)*curTextPtr] == Digit)) {
		dValue = (dValue * value) + (*curTextPtr - '0');
		dValue /= value;
		value *= 10;
		curTextPtr++;
		if (curTextPtr == macroEndPtr) 
			curTextPtr = popMacro(curTextPtr);
	}
	if ((charTokenMap[(UChar)*curTextPtr] == NotACharacter)) {
		*token = Token_Error;
		return curTextPtr;
	}
	if ((*curTextPtr == 'f') || (*curTextPtr == 'F')) {
		curTextPtr++;
		if (curTextPtr == macroEndPtr) 
			curTextPtr = popMacro(curTextPtr);
		dValue = (float)dValue;
	}
	*token = FPDigit;
	gFPDigit = dValue;			// rather than passing around everywhere we keep a global, hmmm.
	return curTextPtr;
}

CharPtr getNextLexeme(CharPtr curTextPtr, ParseState *ps)
{
	int i;
	int length;
	Boolean tFlag;
	long value;
	
	while (true) {
		CharPtr startPtr = curTextPtr;
		ps->start = startPtr;
		switch (charTokenMap[(UChar)(*curTextPtr++)]) {
			case Not :
				if (*curTextPtr == '=') {
					curTextPtr++;
					ps->token = NotEqual;
				}
				else {
					ps->token = Not;
				}
				ps->value = curTextPtr - startPtr;
				return curTextPtr;
			case Less :
				if (*curTextPtr == '=') {
					curTextPtr++;
					ps->token = LessEqual;
				}
				else {
					if (*curTextPtr == '<') {
						curTextPtr++;
						if (*curTextPtr == '=') {
							curTextPtr++;
							ps->token = LeftShiftEqual;
						}
						else
							ps->token = LeftShift;
					}
					else
						ps->token = Less;
				}
				ps->value = curTextPtr - startPtr;
				return curTextPtr;
			case Greater :
				if (*curTextPtr == '=') {
					curTextPtr++;
					ps->token = GreaterEqual;
				}
				else {
					if (*curTextPtr == '>') {
						curTextPtr++;
						if (*curTextPtr == '=') {
							curTextPtr++;
							ps->token = RightShiftEqual;
						}
						else
							ps->token = RightShift;
					}
					else
						ps->token = Greater;
				}
				ps->value = curTextPtr - startPtr;
				return curTextPtr;
			case Minus :
				if (*curTextPtr == '-') {
					curTextPtr++;
					ps->token = PostDecrement;	// let the parser decide later
				}
				else {
					if (*curTextPtr == '>') {
						curTextPtr++;
						ps->token = Arrow;
					}
					else {
						if (*curTextPtr == '=') {
							curTextPtr++;
							ps->token = MinusEqual;
						}
						else
							ps->token = Minus;
					}
				}
				ps->value = curTextPtr - startPtr;
				return curTextPtr;
			case Plus :
				if (*curTextPtr == '+') {
					curTextPtr++;
					ps->token = PostIncrement;	// let the parser decide later
				}
				else {
					if (*curTextPtr == '=') {
						curTextPtr++;
						ps->token = PlusEqual;
					}
					else
						ps->token = Plus;
				}
				ps->value = curTextPtr - startPtr;
				return curTextPtr;
			case Equal :
				if (*curTextPtr == '=') {
					curTextPtr++;
					ps->token = EqualEqual;
				}
				else {
					ps->token = Equal;
				}
				ps->value = curTextPtr - startPtr;
				return curTextPtr;
			case Multiply :
				if (*curTextPtr == '=') {
					curTextPtr++;
					ps->token = MultiplyEqual;
				}
				else {
					ps->token = Multiply;
				}
				ps->value = curTextPtr - startPtr;
				return curTextPtr;
			case Remainder :
				if (*curTextPtr == '=') {
					curTextPtr++;
					ps->token = RemainderEqual;
				}
				else {
					ps->token = Remainder;
				}
				ps->value = curTextPtr - startPtr;
				return curTextPtr;
			case And :
				if (*curTextPtr == '=') {
					curTextPtr++;
					ps->token = AndEqual;
				}
				else {
					if (*curTextPtr == '&') {
						curTextPtr++;
						ps->token = AndAnd;
					}
					else
						ps->token = And;
				}
				ps->value = curTextPtr - startPtr;
				return curTextPtr;
			case Or :
				if (*curTextPtr == '=') {
					curTextPtr++;
					ps->token = OrEqual;
				}
				else {
					if (*curTextPtr == '|') {
						curTextPtr++;
						ps->token = OrOr;
					}
					else
						ps->token = Or;
				}
				ps->value = curTextPtr - startPtr;
				return curTextPtr;
			case Xor :
				if (*curTextPtr == '=') {
					curTextPtr++;
					ps->token = XorEqual;
				}
				else {
					ps->token = Xor;
				}
				ps->value = curTextPtr - startPtr;
				return curTextPtr;
			case Divide :
				if (*curTextPtr == '/') {
					while ((*curTextPtr != 0)
							 && (*curTextPtr != '\n')
							 && (*curTextPtr != '\r')) {
						curTextPtr++;
						if (curTextPtr == macroEndPtr) 
							curTextPtr = popMacro(curTextPtr);
					}
/*					
					if (gReturnWhitespace) {
						ps->token = Whitespace;						
						ps->value = curTextPtr - startPtr;
						return curTextPtr;
					}
*/					
				}
				else {
					if (*curTextPtr == '*') {
						curTextPtr++;
						while (true) {
							if (*curTextPtr == '\n') gLineCount++;
							if (*curTextPtr == 0) {
								sourceError("End of file found in comment", startPtr);
								ps->token = Token_Error;
								return curTextPtr;
							}
							if (*curTextPtr == '*') {
								curTextPtr++;
								if (curTextPtr == macroEndPtr) 
									curTextPtr = popMacro(curTextPtr);
								if (*curTextPtr == 0) {
									sourceError("End of file found in comment", startPtr);
									ps->token = Token_Error;
									return curTextPtr;
								}
								if (*curTextPtr == '/') {
									curTextPtr++;
									if (curTextPtr == macroEndPtr) 
										curTextPtr = popMacro(curTextPtr);
									break;
								}
							}		
							else {
								curTextPtr++;						
								if (curTextPtr == macroEndPtr) 
									curTextPtr = popMacro(curTextPtr);
							}
						}
/*						
						if (gReturnWhitespace) {
							ps->token = Whitespace;						
							ps->value = curTextPtr - startPtr;
							return curTextPtr;
						}
*/						
					}
					else {
						if (*curTextPtr == '=') {
							curTextPtr++;
							ps->token = DivideEqual;
							ps->value = curTextPtr - startPtr;
							return curTextPtr;
						}
						else {
							ps->token = Divide;
							ps->value = curTextPtr - startPtr;
							return curTextPtr;
						}
					}
				}
				break;
			case Quote : {
					value = 0;
					tFlag = false;
					i = 0;
					while (((charTokenMap[(UChar)(*curTextPtr)] != Quote) || (tFlag))
							&& (*curTextPtr != '\n')
							&& (*curTextPtr != '\r')
							&& (charTokenMap[(UChar)(*curTextPtr)] != NotACharacter)) {
						if (tFlag) {
							value <<= 8;
							switch (*curTextPtr) {
								case '\\' : 
									value |= '\\';
									curTextPtr++;
									break;
								case '\'' : 
									value |= '\'';
									curTextPtr++;
									break;
								case '0' :
								case '1' :
								case '2' :
								case '3' :
								case '4' :
								case '5' :
								case '6' :
								case '7' :
									length = *curTextPtr++ - '0';
									while (charTokenMap[(UChar)(*curTextPtr)] == Digit)
										length = (length << 3) + (*curTextPtr++ - '0');
									value |= length;
									break;
								case 'x' :
									length = 0;
									curTextPtr++;
									while (charTokenMap[(UChar)(*curTextPtr)] == HexDigit) {
										length = (length << 4);
										if (*curTextPtr <= '9')
											length += (*curTextPtr++ - '0');
										else
											if (*curTextPtr > 'F')
												length += (*curTextPtr++ - 'a' + 10);
											else
												length += (*curTextPtr++ - 'A' + 10);		
									}						
									value |= length;
									break;
								case 'n' :
									value |= '\n';
									curTextPtr++;
									break;
								case 'b' :
									value |= '\b';
									curTextPtr++;
									break;
								case 'r' :
									value |= '\n';
									curTextPtr++;
									break;
								case 't' :
									value |= '\t';
									curTextPtr++;
									break;
							}
							tFlag = !tFlag && (*curTextPtr == '\\');
							i++;
						}
						else {
							if (*curTextPtr != '\\') {
								value <<= 8;
								value |= *curTextPtr;
								i++;
							}
							tFlag = !tFlag && (*curTextPtr == '\\');
							curTextPtr++;
						}
						if (curTextPtr == macroEndPtr) 
							curTextPtr = popMacro(curTextPtr);
					}
					if ((charTokenMap[(UChar)(*curTextPtr)] != Quote) 
							|| (i > 4)) {
						ps->token = Token_Error;
						return curTextPtr;
					}
					curTextPtr++;
					ps->value = value;
					ps->token = Digit;
				}
				return curTextPtr;
			case DoubleQuote : {
					tFlag = false;
					length = 0;
					while (((charTokenMap[(UChar)(*curTextPtr)] != DoubleQuote)
									|| (tFlag))
							&& (*curTextPtr != '\n')
							&& (*curTextPtr != '\r')
							&& (charTokenMap[(UChar)(*curTextPtr)] != NotACharacter)) {
						tFlag = !tFlag && (*curTextPtr == '\\');
						curTextPtr++;
						length++;
						if (curTextPtr == macroEndPtr) 
							curTextPtr = popMacro(curTextPtr);
					}
					if (charTokenMap[(UChar)(*curTextPtr)] != DoubleQuote) {
						ps->token = Token_Error;
						return curTextPtr;
					}
					curTextPtr++;
					ps->value = length;
					ps->start = startPtr + 1;
					ps->token = DoubleQuote;
				}
				return curTextPtr;
			case HexDigit :
			case Letter : {
					unsigned char tHash = *startPtr;
					length = 1;
					tFlag = false;
					while (((charTokenMap[(UChar)(*curTextPtr)] == Letter)
									|| (charTokenMap[(UChar)(*curTextPtr)] == HexDigit)
									|| (charTokenMap[(UChar)*curTextPtr] == Digit))) {
						unsigned char c = (unsigned char)*curTextPtr++;
						if (c < 'a') tFlag = true;
						tHash ^= c;
						length++;
						if (curTextPtr == macroEndPtr) 
							curTextPtr = popMacro(curTextPtr);
					}
					tHash &= 0x7F;
					if ((charTokenMap[(UChar)*curTextPtr] == NotACharacter)) {
						ps->token = Token_Error;
						return curTextPtr;
					}
					if (tFlag || !reservedWordHashHit[tHash]) {
						ps->value = length;
						ps->start = startPtr;
						ps->token = Identifier;
						ps->hash = tHash;
					}
					else 
						if(!findReservedWord(startPtr, length, &ps->token)) {
							ps->value = length;
							ps->start = startPtr;
							ps->token = Identifier;
							ps->hash = tHash;
						}
				}
				return curTextPtr;
			case Digit : {					
					value = *startPtr - '0';
					if ((*startPtr == '0') 
							&& ((*curTextPtr == 'x') || (*curTextPtr == 'X'))) {
						curTextPtr++;
						while (((charTokenMap[(UChar)*curTextPtr] == Digit)
									|| (charTokenMap[(UChar)*curTextPtr] == HexDigit))) {
							if (charTokenMap[(UChar)*curTextPtr] == Digit) 
								value = (value << 4) + (*curTextPtr - '0');
							else
								if (*curTextPtr > 'F')
									value = (value << 4) + (*curTextPtr - 'a' + 10);
								else
									value = (value << 4) + (*curTextPtr - 'A' + 10);								
							curTextPtr++;
							if (curTextPtr == macroEndPtr) 
								curTextPtr = popMacro(curTextPtr);
						}
						if ((charTokenMap[(UChar)*curTextPtr] == NotACharacter)) {
							ps->token = Token_Error;
							return curTextPtr;
						}
						ps->token = Digit;
						ps->value = value;
						return curTextPtr;
					}
					else {					
						while ((charTokenMap[(UChar)*curTextPtr] == Digit)) {
							value = (value * 10) + (*curTextPtr - '0');
							curTextPtr++;
							if (curTextPtr == macroEndPtr) 
								curTextPtr = popMacro(curTextPtr);
						}
						if ((charTokenMap[(UChar)*curTextPtr] == Period)) {
							
							curTextPtr = getFPDigit(value, curTextPtr, &ps->token);	// out-of-line because mwerks FP locals go nuts
							
							return curTextPtr;
						}
						if ((charTokenMap[(UChar)*curTextPtr] == NotACharacter)) {
							ps->token = Token_Error;
							return curTextPtr;
						}
						ps->token = Digit;
						ps->value = value;
						return curTextPtr;
					}
				}
			case Operator :
			case NotACharacter :
/*
				if (gReturnWhitespace) {
					ps->value = curTextPtr - startPtr;
					ps->token = Whitespace;
				}
				else
*/				
					ps->token = Token_Error;				
				return curTextPtr;
			case EndOfInput : {
					tFlag = false;
					for (i = 0; i < gIncludeTop; i++) {
						if ((startPtr > includeData[i].includePtr) 
								&& (startPtr < (includeData[i].includePtr + includeData[i].includeLength))) {
							curTextPtr = includeData[i].savedSourcePtr;					
							gLineCount = includeData[i].savedLineCount;
							tFlag = true;
							break;
						}
					}
					if (!tFlag) {
						ps->token = Token_EOF;
						return curTextPtr;
					}
					else
						break;
				}
			case Backslash :
			  if (*curTextPtr == '\n'){
			    curTextPtr++;
			    gLineCount++;
			  }
				else {
					ps->token = Token_Error;
					return curTextPtr;
				}
				break;
			case Whitespace :
				if (curTextPtr[-1] == '\n') gLineCount++;
				if (curTextPtr == macroEndPtr) 
					curTextPtr = popMacro(curTextPtr);
/*
				if (gReturnWhitespace) {
					ps->token = Whitespace;
					ps->value = 1;
					ps->start = startPtr;
					return curTextPtr;
				}
*/				
				break;
			default :
				ps->token = charTokenMap[(UChar)*startPtr];
				ps->value = 1;		
				return curTextPtr;
		}
	}
}

void writeMacro(MacroDef *dest, MacroDef *contents)
{
	if ((gHB != NULL) && (dest >= gHB->macroList) && (dest < (gHB->macroList + gHB->macroCount)))
		DmWrite(gHB, (char *)dest - (char *)(gHB), contents, sizeof(MacroDef));
	else
		DmWrite(macroList, (char *)dest - (char *)(macroList), contents, sizeof(MacroDef));
}

CharPtr removeMacro(CharPtr curTextPtr)
{
	ParseState ps;
	MacroDef *md;
	MacroDef *prevmd;
	MacroDef localmd;

	curTextPtr = getNextLexeme(curTextPtr, &ps);
	if (ps.token != Identifier) {
		sourceError("Expected identifier after '#undef'", curTextPtr);
		return NULL;
	}
	md = (gMacroHashTable[ps.hash] == -1) ? NULL : macroList + gMacroHashTable[ps.hash];
	prevmd = NULL;
	while (md != NULL) {
		if (ps.value == md->length) {
			int x;
			for (x = 0; x < ps.value; x++) {
				if (ps.start[x] != md->name[x])
					break;
			}
			if (x == ps.value) {
				if (prevmd == NULL)
					gMacroHashTable[ps.hash] = md->nextMacro;
				else {
					localmd = *prevmd;
					localmd.nextMacro = md->nextMacro;
					writeMacro(prevmd, &localmd);
				}
				localmd = *md;
				localmd.length = -1;
				writeMacro(md, &localmd);
				return curTextPtr;
			}
		}
		prevmd = md;
		md = (md->nextMacro == -1) ? NULL : macroList + md->nextMacro;
	}
	if (gHB != NULL) {
		md = (gHB->macroHashTable[ps.hash] == -1) ? NULL : gHB->macroList + gHB->macroHashTable[ps.hash];
		prevmd = NULL;
		while (md != NULL) {
			if (ps.value == md->length) {
				int x;
				for (x = 0; x < ps.value; x++) {
					if (ps.start[x] != md->name[x])
						break;
				}
				if (x == ps.value) {
					if (prevmd == NULL)
						gHB->macroHashTable[ps.hash] = md->nextMacro;
					else {
						localmd = *prevmd;
						localmd.nextMacro = md->nextMacro;
						writeMacro(prevmd, &localmd);
					}
					localmd = *md;
					localmd.length = -1;
					writeMacro(md, &localmd);
					return curTextPtr;
				}
			}
			prevmd = md;
			md = (md->nextMacro == -1) ? NULL : gHB->macroList + md->nextMacro;
		}
	}
	return curTextPtr;
}


/*
 * The macro processing functions and the lookup table used to access them.  
 * Macro function numbers are held in the MacroFunctions enumeration.
 */

static MacroDef* simpleMacroFunc( MacroDef *md );
static MacroDef* lineMacroFunc( MacroDef *md );
static MacroDef* fileMacroFunc( MacroDef *md );


typedef MacroDef *(MacroFunc)( MacroDef * );

static MacroFunc *gMacroFuncLookup[] = { simpleMacroFunc, 
                                  lineMacroFunc, 
                                  fileMacroFunc };


/*
 * Just a pass through, a normal macro not attached to any special compiler
 * behavior
 */

static MacroDef* simpleMacroFunc(MacroDef *md)
{
  return md;
}


/*
 * The temporary storage, returned macro definition, and processing function
 * invoked when the __LINE__ macro is used.
 */

static char lineMacroBuff[20];

static MacroDef gLineMacroDef = {
  "__LINE__",                      // name
  8,                               // length
  0,                               // parameterCount
  '_'^'_'^'L'^'I'^'N'^'E'^'_'^'_', //hash
};


static MacroDef* lineMacroFunc(MacroDef *md) 
{
  StrPrintF(lineMacroBuff,"%u",gLineCount);
  gLineMacroDef.macroStart=lineMacroBuff;
  gLineMacroDef.macroEnd=lineMacroBuff+StrLen(lineMacroBuff);
  return &gLineMacroDef;
}


/*
 * The temporary storage, returned struct, and function used when the __FILE__
 * macro is invoked.
 */

static char fileMacroBuff[20];

static MacroDef gFileMacroDef = {
  "__FILE__", // name
  8,             // length
  0,             // parameterCount
  '_'^'_'^'F'^'I'^'L'^'E'^'_'^'_', // hash
};


static MacroDef* fileMacroFunc(MacroDef *md)
{
  StrPrintF(fileMacroBuff,"\"%s\"",
            gProjectHeader.source[gSourceIndex].sourceDB);
  gFileMacroDef.macroStart=fileMacroBuff;
  gFileMacroDef.macroEnd=fileMacroBuff+StrLen(fileMacroBuff);
  return &gFileMacroDef;
}


/*
 * Add a macro with some associated special processing.  This is done for the
 * __LINE__ and __FILE__ macros, which are sensative to the current compile
 * state in order to be evaluated properly.
 */

CharPtr addSpecialMacro(CharPtr curTextPtr, MacroFunctions function )
{
	ParseState ps;
	CharPtr macroEnd;
	CharPtr savePtr = curTextPtr;
	CharPtr macroName;
	int length;
	MacroDef *mdp;
	MacroDef md;
	int macroIndex = 0;
	unsigned char hash;
	MacroDef *previousMacro = NULL;

	curTextPtr = getNextLexeme(curTextPtr, &ps);
	if (ps.token != Identifier) {
		sourceError("Expected identifier after '#define'", savePtr);
		return NULL;
	}
	macroName = ps.start;
	length = ps.value;
	hash = ps.hash;

	mdp = findMacro(macroName, length, hash);
	
	if (mdp == NULL) {
		for (macroIndex = 0; macroIndex < macroListTop; macroIndex++) {
			if (macroList[macroIndex].length == -1)
				break;
		}
		if (macroIndex == macroListTop) {
			if (macroListTop == maxMacros) {
				maxMacros += 32;
				MemHandleUnlock(macroH);
				macroH = DmResizeRecord(gOutputDB, macroRecordIndex, maxMacros * sizeof(MacroDef));
				if (macroH == NULL) {
					sourceError("Too many macros", savePtr);
					return NULL;
				}
				macroList = MemHandleLock(macroH);
			}
			macroListTop++;
		}
		md.name = macroName;
		md.length = length;
		md.hash = hash;
		md.nextMacro = gMacroHashTable[hash];
		gMacroHashTable[hash] = macroIndex;
		if (gMacroHashTable[hash] == md.nextMacro) error("macro cycle");
		mdp = &md;
	}
	else {
		previousMacro = mdp;
		md = *mdp;
		mdp = &md;
	}
	mdp->parameterCount = 0;

	savePtr = curTextPtr;
	if (charTokenMap[(int) *curTextPtr] != Whitespace) {
		curTextPtr = getNextLexeme(curTextPtr, &ps);
		if (ps.token == LeftParen) {
			while (true) {
				savePtr = curTextPtr;
				curTextPtr = getNextLexeme(curTextPtr, &ps);
				if (ps.token != Identifier) {
					sourceError("Macro parameter name expected", savePtr);
					return NULL;
				}
				if (md.parameterCount == MaxMacroParameters) {
					sourceError("Too many macro parameters", savePtr);
					return NULL;
				}
				mdp->parameter[mdp->parameterCount] = ps.start;
				mdp->parameterHash[mdp->parameterCount] = ps.hash;
				mdp->parameterLength[mdp->parameterCount] = ps.value;
				mdp->parameterCount++;
				savePtr = curTextPtr;
				curTextPtr = getNextLexeme(curTextPtr, &ps);
				if (ps.token != Comma) {
					if (ps.token != RightParen) {
						sourceError("Comma or right paren expected in macro parameter list", savePtr);
						return NULL;
					}
					else
						break;
				}
			}
		}
		else
			curTextPtr = savePtr;
	}
		
	mdp->macroStart = curTextPtr;
	macroEnd = curTextPtr;	
	while ((*macroEnd != '\n') && (*macroEnd != 0)) {
		if ((*macroEnd == '/') && (macroEnd[1] == '/'))
			break;
		if ((*macroEnd == '\\') && (macroEnd[1] == '\n'))
			macroEnd++;
		macroEnd++;
	}
	curTextPtr = macroEnd;
	while ((charTokenMap[(UChar)*(macroEnd - 1)] == Whitespace) && (macroEnd > md.macroStart))
		macroEnd--;
	
	mdp->macroEnd = macroEnd;
	mdp->function = function;

	if (previousMacro != NULL){
		writeMacro(previousMacro, mdp);
	}
	else{
		writeMacro(&macroList[macroIndex], mdp);
	}

	return curTextPtr;
}


/*
 * Add a macro with the default processing.  Any user defined macro coming from
 * source files calls this addMacro().
 */

CharPtr addMacro(CharPtr curTextPtr)
{
  return addSpecialMacro(curTextPtr,MFSimple);
}

MacroDef *evalMacro(CharPtr macroName, int length, unsigned char hash)
{
  MacroDef *md;
  md = findMacro(macroName,length,hash);

  if(NULL!=md)
    return (gMacroFuncLookup[md->function])( md );
  else
    return NULL;

}

MacroDef *findMacro(CharPtr macroName, int length, unsigned char hash)
{
	MacroDef *md;
	
	md = (gMacroHashTable[hash] == -1) ? NULL : macroList + gMacroHashTable[hash];
	while (md != NULL) {
		if (length == md->length) {
			int x;
			for (x = 0; x < length; x++) {
				if (macroName[x] != md->name[x])
					break;
			}
			if (x == length) {
				return md;
			}
		}
		md = (md->nextMacro == -1) ? NULL : macroList + md->nextMacro;
	}
	if (gHB != NULL) {
		MacroDef *md = (gHB->macroHashTable[hash] == -1) ? NULL : gHB->macroList + gHB->macroHashTable[hash];
		while (md != NULL) {
			if (length == md->length) {
				int x;
				for (x = 0; x < length; x++) {
					if (macroName[x] != md->name[x])
						break;
				}
				if (x == length) {
					return md;
				}
			}
			md = (md->nextMacro == -1) ? NULL : gHB->macroList + md->nextMacro;
		}
	}
	
	return NULL;
}

CharPtr expandMacro(MacroDef *md, CharPtr curTextPtr)
{
	ParseState ps;
	CharPtr savePtr = curTextPtr;
	int argCount = 0;
	int i;
	Boolean recursiveParameter;


	macroEndPtrStack[textStackTop] = macroEndPtr;
	macroListTopStack[textStackTop] = macroListTop;
	
	curTextPtr = getNextLexeme(curTextPtr, &ps);
	if (ps.token == LeftParen) {
		while (true) {
		  // Go thru each argument, and make a newMacro named <param name> which evals to <param value>
		  // this newMacro gets added to the macro list, so each time the arg is encountered during
		  // evaluation of the (real)macro, it's  value is substituted in.
		  // Presumeably the newMacro gets removed after the end of the (real)macro is reached
		  // Boz 31/07/2003 (only my impression of what's happening...)

			MacroDef newMacro;
			CharPtr expansionStart = curTextPtr;
			int parenCount = 0;
			while (*curTextPtr != 0) {
				// assume matching parens inside the argument, yeurk!
				if (*curTextPtr == '(')	// balance ')'
					parenCount++;
				// else commented out, as it's entirely redundant.
				// Boz 31/07/2003
				//				else {

				// balance '('
				if (*curTextPtr == ')') {
				  if (parenCount == 0)
				    break;
				  parenCount--;
				}
				//				}
				if ((*curTextPtr == ',')	// only break for a ',' if we're not in parens
						&& (parenCount == 0))
					break;
				curTextPtr++;
			}
			if (*curTextPtr == 0) {
				sourceError("End of input in macro expansion", savePtr);
				return NULL;
			}
			if (macroListTop == maxMacros) {
				maxMacros += 32;
				MemHandleUnlock(macroH);
				macroH = DmResizeRecord(gOutputDB, macroRecordIndex, maxMacros * sizeof(MacroDef));
				if (macroH == NULL) {
					sourceError("Too many macros (during expansion)", savePtr);
					return NULL;
				}
				macroList = MemHandleLock(macroH);

//				sourceError("Too many macros (during expansion)", savePtr);
//				return NULL;
			}
			/*	
				if the expansion text for a parameter is the same as the parameter, don't do it!
				we'll end up recursing forever
			*/
			recursiveParameter = true;
			for (i = 0; i < md->parameterLength[argCount]; i++) {
				if (md->parameter[argCount][i] != expansionStart[i]) {
					recursiveParameter = false;
					break;
				}
			}
			if (!recursiveParameter) {
				newMacro.name = md->parameter[argCount];
				newMacro.length = md->parameterLength[argCount];
				newMacro.hash = md->parameterHash[argCount];
				newMacro.parameterCount = 0;
				newMacro.macroStart = expansionStart;
				newMacro.macroEnd = curTextPtr;
				newMacro.nextMacro = gMacroHashTable[newMacro.hash];
				newMacro.function = MFSimple;
				DmWrite(macroList, macroListTop * sizeof(MacroDef), &newMacro, sizeof(MacroDef));
				gMacroHashTable[newMacro.hash] = macroListTop;
				if (gMacroHashTable[newMacro.hash] == newMacro.nextMacro) error("macro cycle");
				macroListTop++;
			}
			argCount++;
			
			if (argCount == md->parameterCount) {
				if (*curTextPtr != ')') {
					sourceError("Too many arguments in macro expansion", savePtr);
					return NULL;
				}
				else {
					curTextPtr++;		// skip over the ')'
					break;
				}
			}
			else {
				if (*curTextPtr == ')') {
					sourceError("Too few arguments in macro expansion", savePtr);
					return NULL;
				}
				else
					curTextPtr++;		// skip over the ','
			}
		}		
		curTextPtrStack[textStackTop] = curTextPtr;
		textStackTop++;
		macroEndPtr = md->macroEnd;
		return md->macroStart;	
	}
	else {
		sourceError("Expected argument list for macro expansion", savePtr);
		return NULL;
	}
}

enum { NoID, DefineID, IfdefID, IfndefID, EndifID, IncludeID, PragmaID, UndefID };

int matchPreprocessorId(long value, unsigned char hash, char *startPtr)
{
	if ((value == DefineLength)
			&& (hash == DefineHash)) {
		int x;
		for (x = 0; x < DefineLength; x++) {
			if (startPtr[x] != DefineName[x])
				break;
		}
		if (x == DefineLength)
			return DefineID;
	}
	if ((value == IncludeLength)
			&& (hash == IncludeHash)) {
		int x;
		for (x = 0; x < IncludeLength; x++) {
			if (startPtr[x] != IncludeName[x])
				break;
		}
		if (x == IncludeLength)
			return IncludeID;
	}
	if ((value == IfdefLength)
			&& (hash == IfdefHash)) {
		int x;
		for (x = 0; x < IfdefLength; x++) {
			if (startPtr[x] != IfdefName[x])
				break;
		}
		if (x == IfdefLength)
			return IfdefID;
	}
	if ((value == IfndefLength)
			&& (hash == IfndefHash)) {
		int x;
		for (x = 0; x < IfndefLength; x++) {
			if (startPtr[x] != IfndefName[x])
				break;
		}
		if (x == IfndefLength)
			return IfndefID;
	}
	if ((value == EndifLength)
			&& (hash == EndifHash)) {
		int x;
		for (x = 0; x < EndifLength; x++) {
			if (startPtr[x] != EndifName[x])
				break;
		}
		if (x == EndifLength)
			return EndifID;
	}
	if ((value == UndefLength)
			&& (hash == UndefHash)) {
		int x;
		for (x = 0; x < UndefLength; x++) {
			if (startPtr[x] != UndefName[x])
				break;
		}
		if (x == UndefLength)
			return UndefID;
	}
	if ((value == PragmaLength)
			&& (hash == PragmaHash)) {
		int x;
		for (x = 0; x < PragmaLength; x++) {
			if (startPtr[x] != PragmaName[x])
				break;
		}
		if (x == PragmaLength)
			return PragmaID;
	}
	return NoID;
}

Boolean macroIsNumber(CharPtr macroStart, CharPtr macroEnd, long *retval)
{	
	ParseState ps;
	Boolean isNeg = false;
	Boolean needParen = false;
	CharPtr result = getNextLexeme(macroStart, &ps);
	if (result && (ps.token == LeftParen) && (result < macroEnd)) {
		result = getNextLexeme(result, &ps);
		needParen = true;
	}
	if (result && (ps.token == Minus) && (result < macroEnd)) {
		result = getNextLexeme(result, &ps);
		isNeg = true;
	}
	if (result && (ps.token == Digit)) {
		*retval = (isNeg) ? -ps.value : ps.value;
		if (needParen) {
			if (result < macroEnd) {
				result = getNextLexeme(result, &ps);
				if (result && (ps.token != RightParen)) return false;
			}
			else
				return false;
		}
		return true;
	}
	return false;
}

CharPtr handleConditional(int id, CharPtr result)
{
	ParseState ps;
	Boolean notdef = false;
	
/*	
	char buffer[10];
	StrPrintF(buffer, "%d", id);
	setStatus("Handling...", buffer, StrLen(buffer));
	
*/	
	switch (id) {
		default :
			return false;
		case IfndefID :
			notdef = true;
			// fall thru...
		case IfdefID :
			if (gSkipState[gSkipTop] == DontSkip) {
				result = getNextLexeme(result, &ps);
				if (ps.token == Identifier) {
				  MacroDef *md = evalMacro(ps.start, ps.value, ps.hash);
					if ((md != NULL) == (!notdef)) {
						gSkipTop++;
						if (gSkipTop >= MAXSKIP) {
							error("#ifdef nesting too deep");
							return NULL;
						}
						gSkipState[gSkipTop] = DontSkip;
					}
					else {
						gSkipTop++;
						if (gSkipTop >= MAXSKIP) {
							error("#ifdef nesting too deep");
							return NULL;
						}
						gSkipState[gSkipTop] = SkipToElse;
					}
				}
				else {
					if (notdef)
						error("expected macro name after #ifndef");
					else
						error("expected macro name after #ifdef");
					return NULL;
				}
			}
			else {
				gSkipTop++;
				if (gSkipTop >= MAXSKIP) {
					error("#ifdef nesting too deep");
					return NULL;
				}
				gSkipState[gSkipTop] = SkipToEnd;
			}
			return result;
		case If :
			if (gSkipState[gSkipTop] == DontSkip) {
				result = getNextLexeme(result, &ps);
				if (ps.token == Digit) {
					if (ps.value != 0) {
						gSkipTop++;
						if (gSkipTop >= MAXSKIP) {
							error("#ifdef nesting too deep");
							return NULL;
						}
						gSkipState[gSkipTop] = DontSkip;
					}
					else {
						gSkipTop++;
						if (gSkipTop >= MAXSKIP) {
							error("#ifdef nesting too deep");
							return NULL;
						}
						gSkipState[gSkipTop] = SkipToElse;
					}
				}
				else
					if (ps.token == Identifier) {
						MacroDef *md = NULL;
						long value;
						md = evalMacro(ps.start, ps.value, ps.hash);
						if ((md != NULL) && (md->parameterCount == 0) && macroIsNumber(md->macroStart, md->macroEnd, &value))  {							
							if (value != 0) {
								gSkipTop++;
								if (gSkipTop >= MAXSKIP) {
									error("#ifdef nesting too deep");
									return NULL;
								}
								gSkipState[gSkipTop] = DontSkip;
							}
							else {
								gSkipTop++;
								if (gSkipTop >= MAXSKIP) {
									error("#ifdef nesting too deep");
									return NULL;
								}
								gSkipState[gSkipTop] = SkipToElse;
							}
						}
						else {
							error("sorry, only supporting simple constants for #if");
							return NULL;
						}
					}
					else {
						error("sorry, only supporting simple constants for #if");
						return NULL;
					}
			}
			else {
				gSkipTop++;
				if (gSkipTop >= MAXSKIP) {
					error("#ifdef nesting too deep");
					return NULL;
				}
				gSkipState[gSkipTop] = SkipToEnd;
			}
			return result;
		case Else :
			if (gSkipTop == 0) {
				error("unexpected #else");
				return NULL;
			}
			if (gSkipState[gSkipTop] == SkipToElse)
				gSkipState[gSkipTop] = DontSkip;
			else
				if (gSkipState[gSkipTop] == DontSkip)
					gSkipState[gSkipTop] = SkipToEnd;
			return result;
		case EndifID :
			if (gSkipTop == 0) {
				error("unexpected #endif");
				return NULL;
			}
			gSkipTop--;
			return result;
	}
}

int gCurSegment;

CharPtr handlePreprocessor(int id, CharPtr result, CharPtr savePtr)
{	
	ParseState ps;
	
	switch (id) {
		default :
			return NULL;
		case PragmaID :
			result = getNextLexeme(result, &ps);
			if ((ps.token != Identifier) || (StrCompare(ps.start, "segment") == 0)) {
				sourceError("Expected 'segment' after '#pragma'", savePtr);
				return NULL;
			}
			result = getNextLexeme(result, &ps);
			if (ps.token != Digit) {
				sourceError("Expected segment number", savePtr);
				return NULL;
			}
			addTokenToOutput(Token_Code);
			gCurSegment = ps.value;
			addNumberToOutput(ps.value);
			return result;
		case UndefID :
			return removeMacro(result);
		case DefineID :
			return addMacro(result);
		case IncludeID : {
				int i;
				// search the include stack for an include trigger
				// that matches the current text pointer - if we
				// find it, then we're re-visiting a #include that
				// we've already seen (because of pushback) and so
				// we can ignore it.
				for (i = 0; i < gIncludeTop; i++) {
					if (includeData[i].includeTrigger == savePtr) {
						result = includeData[gIncludeTop - 1].includePtr;
						break;
					}
				}
				if (i == gIncludeTop) {
					if (gIncludeTop == MaxInclude) {
						error("Too many #includes");
						return NULL;
					}
					result = getNextLexeme(result, &ps);
					if (ps.token != DoubleQuote) {
						sourceError("#include must be followed by name", savePtr);
						return NULL;
					}
					includeData[gIncludeTop].savedSourcePtr = result;
					includeData[gIncludeTop].includeTrigger = savePtr;							
					includeData[gIncludeTop].savedLineCount = gLineCount;
					gLineCount = 0;							
					result = getIncludeRecord(ps.start, ps.value);
					if (result == NULL) {
						sourceErrorWithLength("can't find #include source", ps.start, ps.value);
						return NULL;
					}
					gIncludeTop++;
				}
			}
			return result;
	}
}

int gHasPushBackToken;
int gPushBackToken[MAX_PUSHBACK];
long gPushBackValue[MAX_PUSHBACK];
unsigned char gPushBackHash[MAX_PUSHBACK];
CharPtr gPushBackStart[MAX_PUSHBACK];

void pushBack(ParseState *ps)
{
	if (gHasPushBackToken == MAX_PUSHBACK) {
		error("Exceeded maximum pushback");
	}
	gPushBackToken[gHasPushBackToken] = ps->token;
	gPushBackValue[gHasPushBackToken] = ps->value;
	gPushBackStart[gHasPushBackToken] = ps->start;
	gPushBackHash[gHasPushBackToken] = ps->hash;
	gHasPushBackToken++;
}

CharPtr getNextToken(CharPtr curTextPtr, ParseState *ps)
{
	CharPtr result = curTextPtr;

	if (gHasPushBackToken > 0) {
		gHasPushBackToken--;
		ps->token = gPushBackToken[gHasPushBackToken];
		ps->value = gPushBackValue[gHasPushBackToken];
		ps->start = gPushBackStart[gHasPushBackToken];
		ps->hash = gPushBackHash[gHasPushBackToken];
		return curTextPtr;
	}
	
	while (true) {
		CharPtr savePtr = result;
		result = getNextLexeme(result, ps);
		while (result == macroEndPtr) {
			int i;
			textStackTop--;
			result = curTextPtrStack[textStackTop];
			macroEndPtr = macroEndPtrStack[textStackTop];
			for (i = macroListTop - 1; i >= macroListTopStack[textStackTop]; i--) {
				gMacroHashTable[macroList[i].hash] = macroList[i].nextMacro;
				if (gMacroHashTable[macroList[i].hash] != -1)
					if ((macroList + gMacroHashTable[macroList[i].hash])->nextMacro == gMacroHashTable[macroList[i].hash])
						error("macro cycle");
			}
			macroListTop = macroListTopStack[textStackTop];
		}		
		if (ps->token == Identifier) {
			if ((gMacroHashTable[ps->hash] != -1)
					|| (gHB && (gHB->macroHashTable[ps->hash] != -1))) {
			  MacroDef *md = evalMacro(ps->start, ps->value, ps->hash);
				if (md != NULL) {
					if (md->macroEnd > md->macroStart) {
						if (md->parameterCount == 0) {
							curTextPtrStack[textStackTop] = result;
							macroEndPtrStack[textStackTop] = macroEndPtr;
							macroListTopStack[textStackTop] = macroListTop;
							textStackTop++;
							result = md->macroStart;
							macroEndPtr = md->macroEnd;
						}
						else {
							result = expandMacro(md, result);
							if (result == NULL) {
								ps->token = Token_Error;
								return savePtr;
							}
						}
					}
					// else, an empty macro, just swing around for the next token
				}
				else
					break;
			}
			else
				break;
		}
		else {
			if (ps->token == Number) {
				CharPtr numberPosition = result;
				result = getNextLexeme(result, ps);
				if ((ps->token == Identifier) || (ps->token == Else) || (ps->token == If)) {
					int id = (ps->token == Identifier) ? matchPreprocessorId(ps->value, ps->hash, ps->start) : ps->token;
					if (id != NoID) {
						if ((id == Else) || (id == IfdefID) || (id == IfndefID) || (id == EndifID) || (id == If)) {
							result = handleConditional(id, result);
							if (result == NULL) {
								ps->token = Token_Error;
								return savePtr;
							}
							else {
								while (gSkipState[gSkipTop] != DontSkip) {
									result = getNextLexeme(result, ps);
									if (ps->token == Token_EOF) {
										error("EOF encountered while scanning for #endif or #else");
										ps->token = Token_Error;
										return savePtr;
									}
									if (result == NULL) {
										ps->token = Token_Error;
										return savePtr;
									}
									else {
										if (ps->token == Number) {
											result = getNextLexeme(result, ps);
											if ((ps->token == Identifier) || (ps->token == Else)) {
												int id = (ps->token == Identifier) ? matchPreprocessorId(ps->value, ps->hash, ps->start) : Else;
												if ((id == Else) || (id == IfdefID) || (id == IfndefID) || (id == EndifID)) {
													result = handleConditional(id, result);
													if (result == NULL) {
														ps->token = Token_Error;
														return savePtr;
													}
												}
											}
										}
									}
								}
								// continue around for token after conditional
							}
						} 
						else {
							result = handlePreprocessor(id, result, savePtr);
							if (result == NULL) {
								ps->token = Token_Error;
								return savePtr;
							}
							// continue around for next token 
						}
					}
					else {
						ps->token = Number;
						ps->start = savePtr;
						ps->value = 1;
						return numberPosition;			// wasn't a preprocessor symbol, return the '#' position
					}
				}
				else {
					ps->token = Number;
					ps->start = savePtr;
					ps->value = 1;
					return numberPosition;			// wasn't a preprocessor symbol, return the '#' position
				}
			}
			else
				break;
		}
	}
	return result;
}

void setHeaderPointers(HeaderBlock *pHB, char *basePtr, char *stringBase)
{
	pHB->base = basePtr;
	pHB->stringBase = stringBase;
	pHB->macroList = (MacroDef *)(basePtr + sizeof(HeaderBlock));
	pHB->globalList = (Declaration *)(basePtr + sizeof(HeaderBlock)
										 + pHB->macroCount * sizeof(MacroDef));
	pHB->sysTrapList = (Declaration *)(stringBase + pHB->stringSize);
/*
	pHB->sysTrapList = (Declaration *)(basePtr + sizeof(HeaderBlock) 
										+ pHB->macroCount * sizeof(MacroDef)
										+ pHB->globalCount * sizeof(Declaration));
*/
	pHB->argList = (Declaration *)(basePtr + sizeof(HeaderBlock) 
										+ pHB->macroCount * sizeof(MacroDef)
//										+ pHB->sysTrapCount * sizeof(Declaration)
										+ pHB->globalCount * sizeof(Declaration));
	pHB->fieldList = (Declaration *)(basePtr + sizeof(HeaderBlock) 
										+ pHB->macroCount * sizeof(MacroDef)
										+ pHB->globalCount * sizeof(Declaration)
//										+ pHB->sysTrapCount * sizeof(Declaration)
										+ pHB->argCount * sizeof(Declaration));
	pHB->inlineList = (int *)(basePtr + sizeof(HeaderBlock) 
										+ pHB->macroCount * sizeof(MacroDef)
										+ pHB->globalCount * sizeof(Declaration)
//										+ pHB->sysTrapCount * sizeof(Declaration)
										+ pHB->argCount * sizeof(Declaration)
										+ pHB->fieldCount * sizeof(Declaration));
	pHB->typeList = (Type *)(basePtr + sizeof(HeaderBlock) 
										+ pHB->macroCount * sizeof(MacroDef)
										+ pHB->globalCount * sizeof(Declaration)
//										+ pHB->sysTrapCount * sizeof(Declaration)
										+ pHB->argCount * sizeof(Declaration)
										+ pHB->fieldCount * sizeof(Declaration)
										+ pHB->inlineCount * sizeof(int));
}

void thawHeader(char *stringBase)
{
	HeaderBlock hb = *gHB;
	//char *oldBase = gHB->base;
	char *oldStringBase = gHB->stringBase;
	char *headerBase = (char *)gHB;
	int headerOffset = sizeof(HeaderBlock);
	//long adjust = headerBase - oldBase;
	long stringAdjust = stringBase - oldStringBase;
	int i;
	
	setHeaderPointers(&hb, headerBase, stringBase);
	DmWrite(gHB, 0, &hb, sizeof(HeaderBlock));
	
/*
	for (i = 0; i < 128; i++) {
		char *md = (char *)(gHB->macroHashTable[i]);
		if (md != NULL) {
			md += adjust;
			DmWrite(gHB, (char *)(&gHB->macroHashTable[i]) - headerBase, &md, sizeof(MacroDef *));
		}
	}	
*/		
	for (i = 0; i < gHB->macroCount; i++) {
		int p;
		MacroDef md = gHB->macroList[i];
		md.name += stringAdjust;
		md.macroStart += stringAdjust;
		md.macroEnd += stringAdjust;
//		if (md.nextMacro != NULL)
//			md.nextMacro = (MacroDef *)((char *)(md.nextMacro) + adjust);
		for (p = 0; p < md.parameterCount; p++) {
			md.parameter[p] += stringAdjust;
		}
		DmWrite(headerBase, headerOffset, &md, sizeof(MacroDef));
		headerOffset += sizeof(MacroDef);
	}
	for (i = 0; i < gHB->globalCount; i++) {
		char *namePtr = gHB->globalList[i].name;
		namePtr += stringAdjust;
		DmWrite(headerBase, (char *)(&gHB->globalList[i].name) - headerBase, &namePtr, sizeof(char *));
	}
	for (i = 0; i < gHB->sysTrapCount; i++) {
		char *namePtr = gHB->sysTrapList[i].name;
		namePtr += stringAdjust;
//		DmWrite(headerBase, (char *)(&gHB->sysTrapList[i].name) - headerBase, &namePtr, sizeof(char *));
		DmWrite(stringBase, (char *)(&gHB->sysTrapList[i].name) - stringBase, &namePtr, sizeof(char *));
	}
	for (i = 0; i < gHB->argCount; i++) {
		char *namePtr = gHB->argList[i].name;
		namePtr += stringAdjust;
		DmWrite(headerBase, (char *)(&gHB->argList[i].name) - headerBase, &namePtr, sizeof(char *));
	}
	for (i = 0; i < gHB->fieldCount; i++) {
		char *namePtr = gHB->fieldList[i].name;
		namePtr += stringAdjust;
		DmWrite(headerBase, (char *)(&gHB->fieldList[i].name) - headerBase, &namePtr, sizeof(char *));
	}
}

/*
void assureOutput(ULong length)
{
	if ((outputTop + length) >= outputSize) {
		MemHandleUnlock(outputHandle);
		DmResizeRecord(gAsmDB, outputRecordNumber, outputTop);
		DmReleaseRecord(gAsmDB, outputRecordNumber, true);
		totalOutputSize += outputTop;
		outputTop = 0;
		outputRecordNumber++;
//
// don't do this; we start outputSize at 4096 and don't want to increase it from there
//	since that's the record size we set in the DOC header
//
//		if (length > outputSize) outputSize = length + 1000;
//
		outputHandle = DmNewRecord(gAsmDB, &outputRecordNumber, outputSize);
		if (outputHandle == NULL)  {
			error("Out of memory for output buffer");
			return;
		}
		outputBuffer = MemHandleLock(outputHandle);
	}
}
void addToOutput(CharPtr contents, ULong length)
{
	while (length > 4096) {
		assureOutput(4096);
		DmWrite(outputBuffer, outputTop, contents, 4096);
		contents += 4096;
		outputTop += 4096;
		length -= 4096;
	}
	assureOutput(length);
	DmWrite(outputBuffer, outputTop, contents, length);
	outputTop += length;
}

*/
void assureOutput(ULong length)
{
	if ((outputTop + length) >= outputSize) {
		MemHandleUnlock(outputHandle);
		outputSize = outputTop + length + 1000;
		outputHandle = DmResizeRecord(gAsmDB, outputRecordNumber, outputSize);
		if (outputHandle == NULL)  {
			outputHandle = DmResizeRecord(gAsmDB, outputRecordNumber, outputSize);
			if (outputHandle == NULL)  {
				error("Couldn't grow output buffer");
				return;
			}
		}
		outputBuffer = MemHandleLock(outputHandle);
	}
}

void addToOutput(CharPtr contents, ULong length)
{
	assureOutput(length);
	DmWrite(outputBuffer, outputTop, contents, length);
	outputTop += length;
}

void addTokenToOutput(AsmToken token)
{
	addToOutput((CharPtr)&token, 1);
}

void addNumberToOutput(long value)
{
	if (value == 0)
		addTokenToOutput(Asm_Zero);
	else {
		addTokenToOutput(Asm_Digit);
		addToOutput((char *)&value, 4);
	}
}

void addStringToOutput(CharPtr str, int length)
{
	char tBuf[4];	// buf may be being used to handle the string, urk
	
	tBuf[0] = Asm_DoubleQuote;
	tBuf[1] = (length >> 8) & 0xFF;
	tBuf[2] = length & 0xFF;
	addToOutput(tBuf, 3);
	addToOutput(str, length);
//	addTokenToOutput(Asm_Whitespace);
}

void addIdentifierToOutput(CharPtr name, int length, int depth, Boolean isStatic)
{
	char tBuf[4];	// buf may be being used to handle the identifier, urk
	
	tBuf[0] = Asm_Identifier;
	tBuf[1] = (depth == -1) ? length : length + 3;
	if (isStatic) tBuf[1] += 3;
	if (depth < -1) tBuf[1] += 3;		// local static
	addToOutput(tBuf, 2);
	assureOutput(length + 6);		// make sure the (worst case) name isn't split
	
	if (depth >= -1) {
		addToOutput(name, length);
		tBuf[0] = '_';
		tBuf[1] = (depth >> 3) + '0';
		tBuf[2] = (depth & 0x7) + '0';
		addToOutput(tBuf, 3);
	}
	else {
		if (depth < -1) {
			int functionIndex = -depth - 2;
			addToOutput(name, length);
			addToOutput("_f_", 3);
			tBuf[0] = '_';
			tBuf[1] = (functionIndex >> 3) + '0';
			tBuf[2] = (functionIndex & 0x7) + '0';
			addToOutput(tBuf, 3);
		}
		else
			addToOutput(name, length);
	}
	if (isStatic) {
		tBuf[0] = '_';
		tBuf[1] = (gStaticIndex >> 3) + '0';
		tBuf[2] = (gStaticIndex & 0x7) + '0';
		addToOutput(tBuf, 3);
	}
//	addTokenToOutput(Asm_Whitespace);
}

void writeToDataBuffer(char *ptr, int length)
{
	if ((gDataTop + length) >= 4096) {
		UInt at = -1;
		gDataContentSizeStack[gDataContentStackTop] = gDataTop;
		gDataContentStackTop++;
		if (gDataContentStackTop == 8) {
			error("Data buffer overrun");
			return;
		}
		gDataContentHStack[gDataContentStackTop] = DmNewRecord(gOutputDB, &at, 4096);
		if (gDataContentHStack[gDataContentStackTop] == NULL) {
			error("Failed to allocate additional data buffer");
			return; 
		}
		gDataContentIndexStack[gDataContentStackTop] = at;
		gDataContentStack[gDataContentStackTop] = MemHandleLock(gDataContentHStack[gDataContentStackTop]);
		gDataTop = 0;
	}
	DmWrite(gDataContentStack[gDataContentStackTop], gDataTop, ptr, length);
	gDataTop += length;
}

void writeTokenToDataBuffer(AsmToken token)
{
	char t = token;
	writeToDataBuffer(&t, 1);
}

/*
Boolean emitLibRoutine(int id)
{
	if (gEmitTokenStream) {
		if (id == LMULString)
			addToOutput((char *)LMUL_tokenStream, sizeof(LMUL_tokenStream));
		else
		if (id == LDIVString)
			addToOutput((char *)LDIV_tokenStream, sizeof(LDIV_tokenStream));
		else
		if (id == LUDIVString)
			addToOutput((char *)LUDIV_tokenStream, sizeof(LUDIV_tokenStream));
		else
		if (id == LREMString)
			addToOutput((char *)LREM_tokenStream, sizeof(LREM_tokenStream));
		else
		if (id == LUREMString)
			addToOutput((char *)LUREM_tokenStream, sizeof(LUREM_tokenStream));
	}
	else {
		Handle textH = DmGetResource('tSTR', id);
		CharPtr p;
		int length;
		if (textH == NULL) {
			error("Bad Lib string resource");
			return false;
		}
		length = MemHandleSize(textH) - 1;
		p = MemHandleLock(textH);
		addToOutput(p, length);
		MemHandleUnlock(textH);
		DmReleaseResource(textH);
	}
	return true;
}
*/

Int comparF(void *A, void *B, Long other)
{

	Declaration *dA;
	Declaration *dB;

	int iA = *((int *)A);
	int iB = *((int *)B);
	if (iA >= (fullGlobalCount * MaxGlobal)) {
		iA -= fullGlobalCount * MaxGlobal;
		dA = &globalTable[iA];
	}
	else {
		int tbl = iA / MaxGlobal;
		iA -= tbl * MaxGlobal;
		dA = &fullGlobalTable[tbl][iA];
	}
	if (iB >= (fullGlobalCount * MaxGlobal)) {
		iB -= fullGlobalCount * MaxGlobal;
		dB = &globalTable[iB];
	}
	else {
		int tbl = iB / MaxGlobal;
		iB -= tbl * MaxGlobal;
		dB = &fullGlobalTable[tbl][iB];
	}
	
	return (dA->hash - dB->hash);
}

#define SysTrapString 1000

void deleteHeader()
{
	if (gHB != NULL) {
		UInt num, i;
		MemHandleUnlock(headerData);
		num = DmNumRecords(gOutputDB);
		for (i = 0; i < num; i++) {
			DmReleaseRecord(gOutputDB, 0, false);
			DmRemoveRecord(gOutputDB, 0);
		}							
	}
	gHB = NULL;
	freezeHeader = true;
}

void freezeHeaderBlock()
{
	int i,j;
	long stringOffset;
	long headerOffset;
	char *headerBase;
	long headerSize = sizeof(HeaderBlock);
	long stringSize = 0;
	int hashIndex;
	int curHash;
	HeaderBlock hb;
	int *tableIndex;
	char *stringBase;
	
	hb.version = HeaderVersion;
	headerSize += macroListTop * sizeof(MacroDef);
	for (i = 0; i < macroListTop; i++) {
		stringSize += (macroList[i].macroEnd - macroList[i].name) + 1;
	}
	hb.macroCount = macroListTop;
	headerSize += globalTop * sizeof(Declaration);
	for (i = 0; i < globalTop; i++) {
		stringSize += globalTable[i].length;
	}
	hb.globalCount = globalTop;
	for (i = 0; i < fullGlobalCount; i++) {
		for (j = 0; j < MaxGlobal; j++) {
			stringSize += fullGlobalTable[i][j].length;
		}
		hb.globalCount += MaxGlobal;
		headerSize += MaxGlobal * sizeof(Declaration);
	}
//	headerSize += gSysTrapTop * sizeof(Declaration);
	for (i = 0; i < gSysTrapTop; i++) {
		stringSize += gSysTrapList[i].length;
	}
	hb.sysTrapCount = gSysTrapTop;
	headerSize += argumentListTop * sizeof(Declaration);
	for (i = 0; i < argumentListTop; i++) {
		stringSize += argumentList[i].length;
	}
	hb.argCount = argumentListTop;
	headerSize += fieldListTop * sizeof(Declaration);
	for (i = 0; i < fieldListTop; i++) {
		stringSize += fieldList[i].length;
	}
	hb.fieldCount = fieldListTop;
	headerSize += inlineListTop * sizeof(int);
	hb.inlineCount = inlineListTop;
	headerSize += typeIndex * sizeof(Type);
	hb.typeCount = typeIndex;
	
	headerData = DmResizeRecord(gOutputDB, 0, headerSize);
	headerData = DmGetRecord(gOutputDB, 0);
//
	if (stringSize & 1) stringSize++;
	hb.stringSize = stringSize;
//
	stringData = DmResizeRecord(gOutputDB, 1, hb.stringSize + gSysTrapTop * sizeof(Declaration));
	stringData = DmGetRecord(gOutputDB, 1);
	gHB = MemHandleLock(headerData);
	headerBase = (char *)gHB;
	stringBase = (char *)MemHandleLock(stringData);
	setHeaderPointers(&hb, headerBase, stringBase);
	
	tableIndex = MemPtrNew(sizeof(int) * hb.globalCount);
	for (i = 0; i < hb.globalCount; i++) {
		tableIndex[i] = i;
	}

	SysQSort(tableIndex, hb.globalCount, sizeof(int), comparF, 0xDEADBEEF);
	
	hb.hashEntryIndex[0] = 0;
	hashIndex = 1;
	curHash = 0;
	for (i = 0; i < hb.globalCount; i++) {
		int index = tableIndex[i];
		if (index >= (fullGlobalCount * MaxGlobal)) {
			index -= fullGlobalCount * MaxGlobal;
			if (globalTable[index].hash != curHash) {
				curHash = globalTable[index].hash;
				while (hashIndex <= curHash) {
					hb.hashEntryIndex[hashIndex] = i;
						// this will be the index as the sorted records are written 
					hashIndex++;
				}
			}
		}
		else {
			int tbl = index / MaxGlobal;
			index -= tbl * MaxGlobal;
			if (fullGlobalTable[tbl][index].hash != curHash) {
				curHash = fullGlobalTable[tbl][index].hash;
				while (hashIndex <= curHash) {
					hb.hashEntryIndex[hashIndex] = i;
						// this will be the index as the sorted records are written 
					hashIndex++;
				}
			}
		}
	}
	for (i = 0; i < 128; i++) {
		hb.macroHashTable[i] = gMacroHashTable[i];
/*		
		if (gMacroHashTable[i] == NULL)
			hb.macroHashTable[i] = NULL;
		else
			hb.macroHashTable[i] = hb.macroList + (gMacroHashTable[i] - macroList);
*/
	}
	
	DmWrite(gHB, 0, &hb, sizeof(HeaderBlock));
	stringOffset = 0;
	headerOffset = sizeof(HeaderBlock);
	
	for (i = 0; i < macroListTop; i++) {
		int p;
		MacroDef md = macroList[i];
		char *macroBase = md.name;
		int macroLength = (md.macroEnd - macroBase);
		DmWrite(stringBase, stringOffset, md.name, macroLength);
		p = 0;
		DmWrite(stringBase, stringOffset + macroLength, &p, 1);
		md.name = stringBase + stringOffset;
		md.macroStart = (md.macroStart - macroBase) + stringBase + stringOffset;
		md.macroEnd = (md.macroEnd - macroBase) + stringBase + stringOffset;
//		if (md.nextMacro != NULL)
//			md.nextMacro = gHB->macroList + (md.nextMacro - macroList);
		for (p = 0; p < md.parameterCount; p++) {
			md.parameter[p] = (md.parameter[p] - macroBase) + stringBase + stringOffset;
		}
		stringOffset += macroLength + 1;
		DmWrite(gHB, headerOffset, &md, sizeof(MacroDef));
		headerOffset += sizeof(MacroDef);
	}
	
	for (i = 0; i < hb.globalCount; i++) {
		int index = tableIndex[i];
		Declaration d;
		if (index >= (fullGlobalCount * MaxGlobal)) {
			index -= fullGlobalCount * MaxGlobal;
			d = globalTable[index];
		}
		else {
			int tbl = index / MaxGlobal;
			index -= tbl * MaxGlobal;
			d = fullGlobalTable[tbl][index];
		}
		DmWrite(stringBase, stringOffset, d.name, d.length);
		d.name = stringBase + stringOffset;
		stringOffset += d.length;
		DmWrite(gHB, headerOffset, &d, sizeof(Declaration));
		headerOffset += sizeof(Declaration);
	}
/*		
	for (i = 0; i < globalTop; i++) {
		int index = tableIndex[i];		
		Declaration d = globalTable[index];
		DmWrite(stringBase, stringOffset, d.name, d.length);
		d.name = stringBase + stringOffset;
		stringOffset += d.length;
		DmWrite(gHB, headerOffset, &d, sizeof(Declaration));
		headerOffset += sizeof(Declaration);
	}
*/	
	for (i = 0; i < gSysTrapTop; i++) {
		Declaration d = gSysTrapList[i];
		DmWrite(stringBase, stringOffset, d.name, d.length);
		d.name = stringBase + stringOffset;
		stringOffset += d.length;
//		DmWrite(gHB, headerOffset, &d, sizeof(Declaration));
//		headerOffset += sizeof(Declaration);
		DmWrite(stringBase, stringSize + (i * sizeof(Declaration)), &d, sizeof(Declaration));
	}
	for (i = 0; i < argumentListTop; i++) {
		Declaration d = argumentList[i];
		DmWrite(stringBase, stringOffset, d.name, d.length);
		d.name = stringBase + stringOffset;
		stringOffset += d.length;
		DmWrite(gHB, headerOffset, &d, sizeof(Declaration));
		headerOffset += sizeof(Declaration);
	}
	for (i = 0; i < fieldListTop; i++) {
		Declaration d = fieldList[i];
		DmWrite(stringBase, stringOffset, d.name, d.length);
		d.name = stringBase + stringOffset;
		stringOffset += d.length;
		DmWrite(gHB, headerOffset, &d, sizeof(Declaration));
		headerOffset += sizeof(Declaration);
	}
	DmWrite(gHB, headerOffset, inlineList, inlineListTop * sizeof(int));
	headerOffset += inlineListTop * sizeof(int);
	DmWrite(gHB, headerOffset, typeTable, typeIndex * sizeof(Type));
	headerOffset += typeIndex * sizeof(Type);
	
	MemPtrFree(tableIndex);
}

void reallocateOutputDBRecords()
/*
	blow off all but the header records at #0,#1
	and re-assign the string and data records
*/
{
	int i;
	UInt at;
	
	int num = DmNumRecords(gOutputDB);
	if (num != 2) {		
		if (num > 2) {
			for (i = 0; i < (num - 2); i++) {
				DmRemoveRecord(gOutputDB, 2);
			}
		}
		else {	// must be preparing to build and freeze the header blocks
			Handle h;
			if (num == 1) DmRemoveRecord(gOutputDB, 0);
			// build a couple of small records at 0 and 1, we'll resize them when building the
			// header blocks, but we need to reserve slots 0 and 1 prior to allocating the rest
			// of the records.
			at = 0;
			h = DmNewRecord(gOutputDB, &at, 2);
			DmReleaseRecord(gOutputDB, at, true);
			at = 1;
			h = DmNewRecord(gOutputDB, &at, 2);
			DmReleaseRecord(gOutputDB, at, true);
		}
	}

	at = 2;
	gDataContentStackTop = 0;
	gDataContentHStack[0] = DmNewRecord(gOutputDB, &at, 4096);
	if (gDataContentHStack[0] == NULL) {
		error("Failed to allocate data buffer");
		return;
	}
	gDataContentIndexStack[0] = at;
	gDataContentStack[0] = MemHandleLock(gDataContentHStack[0]);
	gDataContentSizeStack[0] = gDataTop;
	gDataTop = 0;

	at++;
	maxMacros = 32;
	macroH = DmNewRecord(gOutputDB, &at, maxMacros * sizeof(MacroDef));
	macroRecordIndex = at;	
	if (macroH == NULL) {
		error("Couldn't allocate macro table space");
	}
	else {
		at++;
		stringPoolSize = 2048;
		stringPoolContents = DmNewRecord(gOutputDB, &at, stringPoolSize);
		stringPoolRecordIndex = at;
		if (stringPoolContents == NULL) {
			error("Couldn't allocate stringPool contents");
		}
		else {
			at++;
			codeStringPoolSize = 2048;
			codeStringPoolContents = DmNewRecord(gOutputDB, &at, codeStringPoolSize);
			codeStringPoolRecordIndex = at;
			if (codeStringPoolContents == NULL) {
				error("Couldn't allocate codeStringPool contents");
			}
		}		
	}		
}

char *SysTrapTypedefName = "SysTrapNumber";
unsigned char SysTrapTypedefHash = 'S' ^ 'y' ^ 's' ^ 'T' ^ 'r' ^ 'a' ^ 'p' ^ 'N' ^ 'u' ^ 'm' ^ 'b' ^ 'e' ^ 'r';
unsigned char SysTrapHash = 's' ^ 'y' ^ 's' ^ 'T' ^ 'r' ^ 'a' ^ 'p';
#define SysTrapTypedefNameLength 13

UInt buildSysTrapData(DmOpenRef outputDB, CharPtr sysTrapSource)
{
	CharPtr curTextPtr = sysTrapSource;
	ParseState ps;
	UInt at = -1;
	int enumValue = sysTrapBase;

	int theEnumType = newEnumType();
	Declaration *theEnumTypeDef = makeDeclaration(SysTrapTypedefName, 
													SysTrapTypedefNameLength, 
													theEnumType, 
													SysTrapTypedefName, 
													GlobalScope, 
													SysTrapTypedefHash, false);
	setTypeDef(theEnumTypeDef, GlobalScope, true);

	gSysTrapSize = 1000;
	gSysTrapTop = 0;
	gSysTrapHandle = DmNewRecord(outputDB, &at, gSysTrapSize * sizeof(Declaration));
	gSysTrapList = MemHandleLock(gSysTrapHandle);
	
	while (true) {
		curTextPtr = getNextToken(curTextPtr, &ps);
		if (ps.token != Identifier) {
			break;
		}
		else {
			Declaration d;
			if (gSysTrapTop >= gSysTrapSize) {
				MemHandleUnlock(gSysTrapHandle);
				gSysTrapSize += 100;
				gSysTrapHandle = DmResizeRecord(outputDB, at, gSysTrapSize * sizeof(Declaration));
				gSysTrapList = MemHandleLock(gSysTrapHandle);
			} 
			d.name = ps.start;
			d.length = ps.value;
			d.hash = ps.hash ^ SysTrapHash;
			d.isTypedef = false;
			d.typeNameSpace = false;
			d.bitWidth = true;
			d.depth = 0;
			d.typeIndex = theEnumType;
			d.data.value = enumValue++;
			DmWrite(gSysTrapList, gSysTrapTop * sizeof(Declaration), &d, sizeof(Declaration));
			gSysTrapTop++;

		}
	}
	
	return at;
}

Boolean doCompile(Handle sourceH, char *sourceName, CharPtr appName, Boolean forDebug)
{
	//UInt at = -1;
	CharPtr headerP;
	Boolean result = true;
//	int length;
//	LocalID asmDBId;

	gIncludeTop = 0;
	
/*
	length = StrLen(appName);
	asmDBId = openDOCAsmFile(length, appName);
	if (!gEmitTokenStream) addToOutput("; ", 2);
	length = StrLen(sourceName);
	if (!gEmitTokenStream) addToOutput(sourceName, length);
	if (gEmitTokenStream)
		addTokenToOutput(Token_Prc);
	else
		addToOutput("\nprc ", 5);
	length = StrLen(appName);
*/

//		addIdentifierToOutput(appName, length, -1);
	addTokenToOutput(Token_Code);

//	dataBuffer = MemHandleLock(dataContents);
//	dataTop = 0;
	stringPoolBuffer = MemHandleLock(stringPoolContents);
	stringPoolTop = 0;
	codeStringPoolBuffer = MemHandleLock(codeStringPoolContents);
	codeStringPoolTop = 0;
	macroListTop = 0;
	macroList = MemHandleLock(macroH);

	resetCompiler();
	
	if (freezeHeader) {
		UInt cardNo;
		LocalID dbID;
		DmSearchStateType searchState;
		Err err;
		DmOpenRef assemblerDB;
		Handle sysTrapH;
		CharPtr sysTrapSource;
		UInt sysTrapListAt;

   		Handle headerH;
		UInt at = -1;
		gHeaderDBID = DmFindDatabase(0, HeaderName);

		setStatus("Finding header file...", "", 0);

		if (gHeaderDBID == 0) {
			error("Can't find header source");
			return false;
		}
		
		if (gRecordLengthList != NULL) MemHandleFree(gRecordLengthList);
		gRecordLengthList = NULL;

		headerH = copyDOCSource(DmOpenDatabase(0, gHeaderDBID, dmModeReadOnly), gOutputDB, &at, &gRecordLengthList);
   		if (headerH == NULL) {
			error("Can't find header source");
   			return false;
   		}
		headerP = MemHandleLock(headerH);

		err = DmGetNextDatabaseByTypeCreator(true, &searchState, 'appl', 'OnBA', true, &cardNo, &dbID);
		if (err != 0) {
			error("Can't find OnBoard Assembler");
			MemHandleUnlock(headerH);
			return false;
		}
		assemblerDB = DmOpenDatabase(cardNo, dbID, dmModeReadOnly);
		sysTrapH = DmGetResource('tSTR', SysTrapString);
		if (sysTrapH == NULL) {
			error("Can't find systrap string, maybe corrupt OnBoard Assembler");
			return false;
		}
		sysTrapSource = MemHandleLock(sysTrapH);
		sysTrapListAt = buildSysTrapData(gOutputDB, sysTrapSource);

		setStatus("Building header...", "", 0);

		gErrorInHeader = true;
		gMainSourceBasePtr = headerP;
		result = compile(headerP, forDebug);
		gErrorInHeader = false;

		// Moved into setDefaultMacros(), called from resetCompiler(), to match other internal macros
		//		addOnBoardMacro();

		if (result) {
			freezeHeaderBlock();			
			resetCompiler();
		}
		MemHandleUnlock(gSysTrapHandle);
		DmReleaseRecord(gOutputDB, sysTrapListAt, true);
		gSysTrapHandle = NULL;
		
		MemHandleUnlock(sysTrapH);
		DmReleaseResource(sysTrapH);
		DmCloseDatabase(assemblerDB);

		MemHandleUnlock(headerH);
		
		if (result) freezeHeader = false;
		if (gRecordLengthList != NULL) MemHandleFree(gRecordLengthList);
		gRecordLengthList = NULL;
	}
	if (result) {
		gMainSourceBasePtr = MemHandleLock(sourceH);
#if 1
		result = compile(gMainSourceBasePtr, forDebug);
#else
		{	// used for timing Lexer/Parser in isolation
			ParseState ps;
			CharPtr curTextPtr = gMainSourceBasePtr;
			while (true) {
//				curTextPtr = getNextLexeme(curTextPtr, &ps);
				curTextPtr = getNextToken(curTextPtr, &ps);
				if ((curTextPtr == NULL) || (ps.token == Token_EOF))
					break;
			}
			setStatus("Done", "", 0);
			result = true;
		}
#endif		
		
		MemHandleUnlock(sourceH);
/*		
		if (result) {
			if (longMultiplyUsed)
				result = emitLibRoutine(LMULString);
			if (result)
				if (longUnsignedDivideUsed)
					result = emitLibRoutine(LUDIVString);
			if (result)
				if (longSignedDivideUsed)
					result = emitLibRoutine(LDIVString);				
			if (result)
				if (longUnsignedRemainderUsed)
					result = emitLibRoutine(LUREMString);
			if (result)
				if (longSignedRemainderUsed)
					result = emitLibRoutine(LREMString);

		}
*/
		if (result) {
			int i;
			addTokenToOutput(Token_Data);
			addToOutput(stringPoolBuffer, stringPoolTop);
			gDataContentSizeStack[gDataContentStackTop] = gDataTop;
			for (i = 0; i <= gDataContentStackTop; i++) {
				addToOutput(gDataContentStack[i], gDataContentSizeStack[i]);
			}
//			addToOutput(dataBuffer, dataTop);
			addTokenToOutput(Asm_Token_EOF);
		}
	}
		
	MemHandleUnlock(macroH);
	MemHandleUnlock(stringPoolContents);
	MemHandleUnlock(codeStringPoolContents);
//	MemHandleUnlock(dataContents);
	
	if (bodyContents != NULL) {
		MemHandleFree(bodyContents);
		bodyContents = NULL;
	}

   	if (typeHandle != NULL) {
   		MemHandleUnlock(typeHandle);
   		MemHandleFree(typeHandle);
   		typeHandle = NULL;
   	}

	if (argumentListData != NULL) {
		MemHandleUnlock(argumentListData);
		MemHandleFree(argumentListData);
		argumentListData = NULL;
	}

	if (fieldListData != NULL) {
		MemHandleUnlock(fieldListData);
		MemHandleFree(fieldListData);
		fieldListData = NULL;
	}

	if (tempNameBase != NULL) {
		TempNamePool *t = tempNameBase;
		while (t != NULL) {
			TempNamePool *next = t->next;
			MemPtrFree(t);
			t = next;
		}
		tempNameBase = NULL;
	}

	releaseIncludeRecords();
	releaseSourceRecord();

	if (!result) {
		reallocateOutputDBRecords();
	}
	
	gStaticIndex++;
	return result;
}

/*
static Boolean mainListEventHandler(EventPtr event)
{
	Boolean handled = false;
	switch (event->eType) {
  		case frmOpenEvent: {
  				int fldIndex;
				FormPtr frm = FrmGetActiveForm();
				if (gPrefs.memoRecords) {
					FrmSetControlGroupSelection(frm, OnBoardCFormGroupID, MainListMemoPushButton);
					collectCTextNamesFromMemoDB();
				}
				else {
					if (gPrefs.smartEdit)
						FrmSetControlGroupSelection(frm, OnBoardCFormGroupID, MainListSmartEditPushButton);
					else
						if (gPrefs.zDoc)
							FrmSetControlGroupSelection(frm, OnBoardCFormGroupID, MainListZDocPushButton);
						else
							FrmSetControlGroupSelection(frm, OnBoardCFormGroupID, MainListDOCPushButton);
	   				collectCTextNamesFromDOC();	   				
				}
				fldIndex = FrmGetObjectIndex(frm, MainListSaveASMCheckbox);
				CtlSetValue(FrmGetObjectPtr(frm, fldIndex), gPrefs.saveASM);
				fldIndex = FrmGetObjectIndex(frm, MainListExecuteCheckbox);
				CtlSetValue(FrmGetObjectPtr(frm, fldIndex), gPrefs.execute);
				setStatus("", "", 0);
				FrmDrawForm(frm);
				setListSelection(gPrefs.sourceName);
				handled = true;
  			}
  			break;
  			
		case menuEvent:
			switch (event->data.menu.itemID) {
				case OptionsDeleteHeader: {
						if (gHB != NULL) {
							UInt num, i;
							MemHandleUnlock(headerData);
							num = DmNumRecords(gOutputDB);
							for (i = 0; i < num; i++) {
								DmReleaseRecord(gOutputDB, 0, false);
								DmRemoveRecord(gOutputDB, 0);
							}							
						}
						gHB = NULL;
						freezeHeader = true;
					}
					break;
				case OptionsAboutOnBoardC: {
						FormPtr frm = FrmInitForm(AboutForm);
						FrmDoDialog(frm);
			 			FrmDeleteForm(frm);
					}
					break;
			}
			handled = true;
  			break;

	   	case ctlSelectEvent:
	   		switch (event->data.ctlSelect.controlID) {
				case MainListSaveASMCheckbox :
					gPrefs.saveASM = event->data.ctlSelect.on;
	   				break;
				case MainListExecuteCheckbox :
					gPrefs.execute = event->data.ctlSelect.on;
	   				break;
	   			case MainListMemoPushButton : {
	   					if (event->data.ctlSelect.on) {
							collectCTextNamesFromMemoDB();
							gPrefs.memoRecords = true;
	   					}
	   				}
	   				break;
	   			case MainListDOCPushButton : {
	   					if (event->data.ctlSelect.on) {
			   				collectCTextNamesFromDOC();
			   				gPrefs.memoRecords = false;
			   				gPrefs.smartEdit = false;
			   				gPrefs.zDoc = false;
			   			}
	   				}
	   				break;
	   			case MainListSmartEditPushButton : {
	   					if (event->data.ctlSelect.on) {
			   				collectCTextNamesFromDOC();
			   				gPrefs.memoRecords = false;
			   				gPrefs.smartEdit = true;
			   				gPrefs.zDoc = false;
			   			}
	   				}
	   				break;
	   			case MainListZDocPushButton : {
	   					if (event->data.ctlSelect.on) {
			   				collectCTextNamesFromDOC();
			   				gPrefs.memoRecords = false;
			   				gPrefs.smartEdit = false;
			   				gPrefs.zDoc = true;
			   			}
	   				}
	   				break;
	   			case MainListBuildButton : {
	   					Handle sourceH;
	   					reallocateOutputDBRecords();
						sourceH = getSourceText();
						if (sourceH != NULL) {
								CharPtr sourceName = getSourceName();
								CharPtr appName = getAppName(sourceName);
								setStatus("Compiling...", "", 0);
								doCompile(sourceH, sourceName, appName);
								MemPtrFree(appName);
//							}
						}
					   	handled = true;
	   				}
	   				break;
	   		}
	   		break;
	}
	return handled;
}
*/
void startParser()
{
	int error;
	//UInt at = 2;		// records #0,#1 will be the frozen header
/*
	Word prefSize = sizeof(PrefsData);
	
	if ((PrefGetAppPreferences('OnBC', 1000, &gPrefs, &prefSize, false) == noPreferenceFound) 
			|| (prefSize != sizeof(PrefsData))) {
		gPrefs.memoRecords = false;
		gPrefs.smartEdit = false;
		gPrefs.zDoc = false;
		gPrefs.sourceName[0] = 0;
		gPrefs.execute = true;
		gPrefs.saveASM = false;
	}
*/	
	gMatchReservedWords = true;
	gSkipTop = 0;
	gSkipState[gSkipTop] = DontSkip;		
	
	gHB = NULL;

	gOutputDB = DmOpenDatabaseByTypeCreator('DATA', 'OnBC', dmModeReadWrite);
	if (gOutputDB == NULL) {
		error = DmCreateDatabase(0, "CompilerData.OnBC", 'OnBC', 'DATA', false);
		gOutputDB = DmOpenDatabaseByTypeCreator('DATA', 'OnBC', dmModeReadWrite);
		gHB = NULL;
		freezeHeader = true;
	}
	else {
		UInt num = DmNumRecords(gOutputDB);
		if (num > 1) {
			UInt i;
			headerData = DmGetRecord(gOutputDB, 0);
			gHB = (HeaderBlock *)MemHandleLock(headerData);
			if (gHB->version != HeaderVersion) {
				DmReleaseRecord(gOutputDB, 0, false);
				gHB = NULL;
				freezeHeader = true;
				for (i = 0; i < num; i++) {
					DmRemoveRecord(gOutputDB, 0);
				}
			}
			else {
				freezeHeader = false;
				stringData = DmGetRecord(gOutputDB, 1);
				thawHeader(MemHandleLock(stringData));
				for (i = 0; i < (num - 2); i++) {
					DmRemoveRecord(gOutputDB, 2);
				}
			}
		}
		else
			freezeHeader = true;
	}
	
	initReservedWordMap();
}

void stopParser()
{
	if (gHB != NULL) {
		UInt num;
		int i;
		MemHandleUnlock(headerData);
		DmReleaseRecord(gOutputDB, 0, true);
		MemHandleUnlock(stringData);
		DmReleaseRecord(gOutputDB, 1, true);
		num = DmNumRecords(gOutputDB);
		if (num > 2) {
			for (i = 0; i < (num - 2); i++) {
				DmRemoveRecord(gOutputDB, 2);
			}
		}
	}
	DmCloseDatabase(gOutputDB);
}

