/***********************************************************************
 *
 * This file is part of the OnBoard C package.  For more information on 
 * using OnBoard C, see http://groups.yahoo.com/group/onboardc.  For 
 * more information on developing OnBoard C (including submission of 
 * bug reports), see http://sourceforge.net/projects/onboardc.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 ***********************************************************************/

#include <PalmOS.h>
#include <PalmCompatibility.h>
#include <SystemMgr.h>

#include "OnBoardC.h"
#include "OnBoardC_res.h"

/* Multi-Segment Defines */
#include "sourceFiles-seg.h"

/* End of Multi-segment defines */


UInt gSourceRecordIndex;			// the record number for the copy of the DOC source in the outputDB

CharPtr gMainSourceBasePtr;
Handle gRecordLengthList = NULL;	// list of record lengths for main source file, if DOC format.
									// Needed to re-discover the record number an error occured in
									// if we're launching an app on error-goto that needs the record
									// number and offset.
									
Boolean gErrorInHeader = false;		// flag to force gotoEditor to jump to error in OnBoardHeader DB
LocalID gHeaderDBID;

void setInclude(LocalID dbID, CharPtr name, int nameLength, UInt at, Handle copy, SourceKind kind)
{
	includeData[gIncludeTop].includeDBid = dbID;
	includeData[gIncludeTop].includeName = name;
	includeData[gIncludeTop].includeNameLength = nameLength;
	includeData[gIncludeTop].includeIndex = at;
	includeData[gIncludeTop].includeHandle = copy;
	includeData[gIncludeTop].includePtr = MemHandleLock(includeData[gIncludeTop].includeHandle);
	includeData[gIncludeTop].includeLength = MemHandleSize(includeData[gIncludeTop].includeHandle);
	includeData[gIncludeTop].includeKind = kind;
	markInclude(&includeData[gIncludeTop]);
}

CharPtr getIncludeRecord(CharPtr name, int nameLength)
{
	UInt at = -1;
	DmOpenRef includeDB;
	Handle copy;
	char dbName[dmDBNameLength];
	int i;
	LocalID dbID;
	
	for (i = 0; i < nameLength; i++) dbName[i] = name[i];
	dbName[nameLength] = '\0';
	
	dbID = DmFindDatabase(0, dbName);
	if (dbID == 0) {
		int k;
		buildMemoSourceList();
		for (k = 0; k < memoSourceCount; k++) {
			if (StrCompare(memoSourceList[k], dbName) == 0) {
				copy = copyMemoSource(memoSourceIndexList[k], gOutputDB, &at, &includeData[gIncludeTop].recordLengthList);
				setInclude(memoSourceIndexList[k], name, nameLength, at, copy, MemoSource);
				return includeData[gIncludeTop].includePtr;
			}
		}
		buildPedit32SourceList();
		for (k = 0; k < pedit32SourceCount; k++) {
			if (StrCompare(pedit32SourceList[k], dbName) == 0) {
				copy = copyPedit32Source(pedit32SourceIndexList[k], gOutputDB, &at, &includeData[gIncludeTop].recordLengthList);
				setInclude(pedit32SourceIndexList[k], name, nameLength, at, copy, Pedit32Source);
				return includeData[gIncludeTop].includePtr;
			}
		}
		return NULL;
	}
	else {
		// if the database is not the standard PalmDoc database, returns NULL. - C061
		UInt32 dbType, dbCrid;
		DmDatabaseInfo(0, dbID, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, &dbType, &dbCrid);
		if ((dbCrid != 'REAd') && (dbType != 'TEXt'))
			return NULL;
		// C061 fix end here

		includeDB = DmOpenDatabase(0, dbID, dmModeReadOnly);
		copy = copyDOCSource(includeDB, gOutputDB, &at, &includeData[gIncludeTop].recordLengthList);
		setInclude(dbID, name, nameLength, at, copy, DocSource);
		return includeData[gIncludeTop].includePtr;
	}
}

Handle getSourceText(LocalID dbID)
{
	DmOpenRef sourceDB = DmOpenDatabase(0, dbID, dmModeReadOnly);
	gSourceRecordIndex = -1;
	gRecordLengthList = NULL;
	gLineCount = 0;
	return copyDOCSource(sourceDB, gOutputDB, &gSourceRecordIndex, &gRecordLengthList);
}

Handle getMemoText(int recordNum)
{
	gLineCount = 0;
	gSourceRecordIndex = -1;
	gRecordLengthList = NULL;
	return copyMemoSource(recordNum, gOutputDB, &gSourceRecordIndex, &gRecordLengthList);
}

Handle getPedit32Text(int recordNum)
{
	gLineCount = 0;
	gSourceRecordIndex = -1;
	gRecordLengthList = NULL;
	return copyPedit32Source(recordNum, gOutputDB, &gSourceRecordIndex, &gRecordLengthList);
}

void releaseIncludeRecords()
{
	int i;
	for (i = 0; i < gIncludeTop; i++) {
		MemHandleUnlock(includeData[i].includeHandle);
		DmReleaseRecord(gOutputDB, includeData[i].includeIndex, false);
		if (includeData[i].recordLengthList != NULL)
			MemHandleFree(includeData[i].recordLengthList);
	}
}

void releaseSourceRecord()
{
	DmReleaseRecord(gOutputDB, gSourceRecordIndex, false);
	if (gRecordLengthList != NULL) MemHandleFree(gRecordLengthList);
}

#define COUNT_BITS 3

Int decompress(unsigned char *inPtr, Int inLength, unsigned char *outPtr) 
{
	
	Int in = 0;
	Int out = 0;

	while (in < inLength) {
		Int ch = inPtr[in++];
	
		if ((ch > 0) && (ch < 9)) {	// n chars echoed from in to out
		    while (ch-- > 0) outPtr[out++] = inPtr[in++];
	    }
		else {
			if (ch < 0x80) {			// literal
				outPtr[out++] = ch;
			}
			else {
				if (ch >= 0xc0) {		// space preceded literal, high bit off
				    outPtr[out++] = ' ';
				    outPtr[out++] = ch & 0x7F;
				}
				else {					// duplication - <0b10><11 bit back displacement><3 bit length (plus 3)>
					Int orig;
					Int count;
				    ch <<= 8;
				    ch |= inPtr[in++];
				    orig = out - ((ch & 0x3fff) >> 3);
				    count = (ch & 0x7) + 3;
				    while (count--)	outPtr[out++] = outPtr[orig++];
				}
			}
		}
    }

    return out;
}

typedef struct {
	int compressedSize;
	int uncompressedSize;
} RecordLengthInfo;

Handle copyDOCSource(DmOpenRef sourceDB, DmOpenRef theCopyDB, UIntPtr at, Handle *recordLengthList)
{
/*
	Build a zero terminated single record out of all the records (index > 0)
*/
	Handle copy;
	Handle original;
	CharPtr p;
	int i;
	int *rllp;
	unsigned long size;
	unsigned long offset;
	DOCHeader *header;
	unsigned char *decompressBuffer = NULL;
	RecordLengthInfo rli;
	
	original = DmGetRecord(sourceDB, 0);
	if (original == NULL) {
		DmCloseDatabase(sourceDB);
		return NULL;
	}
	header = (DOCHeader *)MemHandleLock(original);
	if ((header->flag & 0xFF) == 2) decompressBuffer = MemPtrNew(6144);
	size = header->size;
	MemHandleUnlock(original);
	DmReleaseRecord(sourceDB, 0, false);
	
	copy = DmNewRecord(theCopyDB, at, size + 1);
	if (copy == NULL) {
		DmCloseDatabase(sourceDB);
		error("Couldn't allocate record for copy of source");
		return NULL;
	}
	p = MemHandleLock(copy);
	offset = 0;

// save the record count in recordLengthList[0] - which is unused otherwise
//
	if (recordLengthList != NULL) {
		*recordLengthList = MemHandleNew(sizeof(RecordLengthInfo) * (header->count + 1));
		rli.compressedSize = (header->count + 1);
		rli.uncompressedSize = rli.compressedSize;
		rllp = MemHandleLock(*recordLengthList);
		DmWrite(rllp, 0, &rli, sizeof(RecordLengthInfo));
		MemHandleUnlock(*recordLengthList);
	}

	for (i = 1; i <= header->count ; i++) {
		Int decomSize;
		original = DmGetRecord(sourceDB, i);
		if (original == NULL) {
			DmCloseDatabase(sourceDB);
			return NULL;
		}
		size = MemHandleSize(original);
		rli.compressedSize = size;
		if ((header->flag & 0xFF) == 2) {
			decomSize = decompress(MemHandleLock(original), size, decompressBuffer);
			DmWrite(p, offset, decompressBuffer, decomSize);
			offset += decomSize;
		}
		else {
			decomSize = size;
			DmWrite(p, offset, MemHandleLock(original), size);
			offset += size;
		}
		if (recordLengthList != NULL) {
			rllp = MemHandleLock(*recordLengthList);
			rli.uncompressedSize = decomSize;
			DmWrite(rllp, i * sizeof(RecordLengthInfo), &rli, sizeof(RecordLengthInfo));
			MemHandleUnlock(*recordLengthList);
		}
		MemHandleUnlock(original);
		DmReleaseRecord(sourceDB, i, false);
	}
	
	if (decompressBuffer) MemPtrFree(decompressBuffer);
	
	DmWrite(p, offset, "", 1);
	DmCloseDatabase(sourceDB);

	MemHandleUnlock(copy);
	DmReleaseRecord(theCopyDB, *at, true);
	copy = DmGetRecord(theCopyDB, *at);

	return copy;
}

void launchEditor(ULong edType, GoToParamsType *pBlock, Word cmd, char *editorName)
{
	UInt cardNo;
	LocalID dbID = 0;
	DmSearchStateType searchState;
	Err err = 0;

	if ((edType == 'pn10') || (edType == 'pn32'))
		err = DmGetNextDatabaseByTypeCreator(true, &searchState, 'appl', 'pnPr', true, &cardNo, &dbID);
		
	if (err || (dbID == 0))
		err = DmGetNextDatabaseByTypeCreator(true, &searchState, 'appl', edType, true, &cardNo, &dbID);
	if (dbID) {
		err = SysUIAppSwitch(cardNo, dbID, cmd, (Ptr)pBlock);
		ErrNonFatalDisplayIf(err, "Could not launch Editor");
	}
	else {
		char buf[64];
		StrPrintF(buf, "Couldn't find %s", editorName);
		error(buf);
	}
}

#define goToPedit04OnSysAppLaunchCmdGoTo          (UInt32) 0x19480000L
#define goToPedit32OnSysAppLaunchCmdGoTo      (UInt32) 0x20680000L

void gotoMemoPad(long sourceOffset, int length, int recordNum)
{
	GoToParamsType *pBlock;

	pBlock = (GoToParamsType *)MemPtrNew(sizeof(GoToParamsType));
	MemSet(pBlock, sizeof(GoToParamsType), 0);
	MemPtrSetOwner(pBlock, 0);
	
	pBlock->recordNum = recordNum;		// index of record that contain a match
	pBlock->matchPos = sourceOffset;	// postion in record of the match.	
	
	if (gPrefs.memoEditor == Pedit) {
		UInt cardNo;
		LocalID dbID;
		DmSearchStateType searchState;
		Err err;

        pBlock->matchCustom = length;
        pBlock->matchCustom += goToPedit04OnSysAppLaunchCmdGoTo; 

		err = DmGetNextDatabaseByTypeCreator(true, &searchState, 'DATA', 'memo', true, &cardNo, &dbID);
		pBlock->dbID = dbID;

		launchEditor('pn10', pBlock, sysAppLaunchCmdGoTo, "pedit");
	}
	else {
		pBlock->searchStrLen = length;		// length of search string.
		if (gPrefs.memoEditor == WordSmith_Memo)
			launchEditor('WrdS', pBlock, sysAppLaunchCmdGoTo, "WordSmith");
		else
			launchEditor('memo', pBlock, sysAppLaunchCmdGoTo, "MemoPad !");
	}
}

void gotoPedit32(long sourceOffset, int length, int recordNum)
{
	UInt cardNo;
	LocalID dbID;
	DmSearchStateType searchState;
	Err err;
	GoToParamsType *pBlock;

	pBlock = (GoToParamsType *)MemPtrNew(sizeof(GoToParamsType));
	MemSet(pBlock, sizeof(GoToParamsType), 0);
	MemPtrSetOwner(pBlock, 0);

	pBlock->recordNum = recordNum;		// index of record that contain a match
	pBlock->matchPos = sourceOffset;	// postion in record of the match.	

	err = DmGetNextDatabaseByTypeCreator(true, &searchState, 'DATA', 'pn32', true, &cardNo, &dbID);
	pBlock->dbID = dbID;

    pBlock->matchCustom = length;
    pBlock->matchCustom += goToPedit32OnSysAppLaunchCmdGoTo; 

	launchEditor('pn32', pBlock, sysAppLaunchCmdGoTo, "pedit32");
}

int getSourceOffset(CharPtr sourcePtr)
{
	if ((gIncludeTop > 0) &&
			(sourcePtr > includeData[gIncludeTop - 1].includePtr) 
				&& (sourcePtr < (includeData[gIncludeTop - 1].includePtr + includeData[gIncludeTop - 1].includeLength)))
		return sourcePtr - includeData[gIncludeTop - 1].includePtr;
	else
		return sourcePtr - gMainSourceBasePtr;
}

/***********************************************************************
 *
 *   OnBoardC is free software as specified in the GNU General Public License
 *   listed in the file COPYING or http://www.gnu.org/licenses/gpl.txt
 *
 ***********************************************************************/

void gotoEditor(CharPtr sourcePtr, int length, LocalID selectedDBid, char *selectedDBname)
{
	GoToParamsType *pBlock;
	UInt cardNo;
	LocalID dbID;
	DmSearchStateType searchState;
	int i = gPrefs.docEditor;
	
	//The following 2 lines have been commented out as they forced the launched
	//editor to be SrcEdit (LedX) if it was found to be installed on the palm.
	//This is no longer needed as you can now choose SrcrEdit in the Choose
	//Editor Form.  At some point these lines can and should be deleted.
	//Change made by dmoore330 on 08/27/2002.
	
//	DmGetNextDatabaseByTypeCreator(true, &searchState, 'appl', 'LedX', true, &cardNo, &dbID);	
//	if (dbID != 0) i = ProgED;

	switch (i) {

		case QuickWord :
			pBlock = (GoToParamsType *)MemPtrNew(sizeof(GoToParamsType));
			MemPtrSetOwner(pBlock, 0);
			
			pBlock->searchStrLen = length;		// length of search string.
			pBlock->dbCardNo = 0;				// card number of the database
			if (selectedDBid) {
				pBlock->searchStrLen = 0;		// flag we don't have a selection - just re-open at saved selection
				pBlock->matchPos = 0;
				pBlock->matchFieldNum = 1;
				pBlock->dbID = selectedDBid;
			}
			else {
				if ((gIncludeTop > 0) &&
						(sourcePtr > includeData[gIncludeTop - 1].includePtr) 
							&& (sourcePtr < (includeData[gIncludeTop - 1].includePtr + includeData[gIncludeTop - 1].includeLength))) {
					pBlock->matchPos = sourcePtr - includeData[gIncludeTop - 1].includePtr;			// postion in record of the match.
					pBlock->dbID = includeData[gIncludeTop - 1].includeDBid;
				}
				else {
					pBlock->matchPos = sourcePtr - gMainSourceBasePtr;
					if (gErrorInHeader)
						pBlock->dbID = gHeaderDBID;
					else
						pBlock->dbID = getSourceDBid();
				}			
				pBlock->matchFieldNum = 1;
			}
			pBlock->matchCustom = 0;		// application specific info
			
			launchEditor('Qwrd', pBlock, sysAppLaunchCmdGoTo, "QuickWord");
			break;
		
		case WordSmith_Doc :
			launchEditor('WrdS', NULL, sysAppLaunchCmdNormalLaunch, "WordSmith");
/*
			pBlock = (GoToParamsType *)MemPtrNew(sizeof(GoToParamsType));
			MemPtrSetOwner(pBlock, 0);
			
			pBlock->searchStrLen = length;		// length of search string.
			pBlock->dbCardNo = 0;				// card number of the database
			if (selectedDBid) {
				pBlock->searchStrLen = -1;		// flag we don't have a selection - just re-open at saved selection
				pBlock->matchPos = 0;
				pBlock->matchFieldNum = 1;
				pBlock->dbID = selectedDBid;
			}
			else {
				if ((gIncludeTop > 0) &&
						(sourcePtr > includeData[gIncludeTop - 1].includePtr) 
							&& (sourcePtr < (includeData[gIncludeTop - 1].includePtr + includeData[gIncludeTop - 1].includeLength))) {
					pBlock->matchPos = sourcePtr - includeData[gIncludeTop - 1].includePtr;			// postion in record of the match.
					pBlock->dbID = includeData[gIncludeTop - 1].includeDBid;
				}
				else {
					pBlock->matchPos = sourcePtr - gMainSourceBasePtr;
					if (gErrorInHeader)
						pBlock->dbID = gHeaderDBID;
					else
						pBlock->dbID = getSourceDBid();
				}			
				pBlock->matchFieldNum = 1;
			}
			pBlock->matchCustom = 0;		// application specific info
			
			launchEditor('WrdS', pBlock, sysAppLaunchCmdGoTo, "WordSmith");
*/			
			break;
		
		case SrcEdit :
			pBlock = (GoToParamsType *)MemPtrNew(sizeof(GoToParamsType));
			MemPtrSetOwner(pBlock, 0);
			
			pBlock->searchStrLen = length;		// length of search string.
			pBlock->dbCardNo = 0;				// card number of the database
			if (selectedDBid) {
				pBlock->searchStrLen = -1;		// flag we don't have a selection - just re-open at saved selection
				pBlock->matchPos = 0;
				pBlock->matchFieldNum = 0;
				pBlock->dbID = selectedDBid;
			}
			else {
				if ((gIncludeTop > 0) &&
						(sourcePtr > includeData[gIncludeTop - 1].includePtr) 
							&& (sourcePtr < (includeData[gIncludeTop - 1].includePtr + includeData[gIncludeTop - 1].includeLength))) {
					pBlock->matchPos = sourcePtr - includeData[gIncludeTop - 1].includePtr;			// postion in record of the match.
					pBlock->dbID = includeData[gIncludeTop - 1].includeDBid;
				}
				else {
					pBlock->matchPos = sourcePtr - gMainSourceBasePtr;
					if (gErrorInHeader)
						pBlock->dbID = gHeaderDBID;
					else
						pBlock->dbID = getSourceDBid();
				}			
			}
			pBlock->matchCustom = 0;		// application specific info
			
			launchEditor('LedX', pBlock, sysAppLaunchCmdGoTo, "SrcEdit");
			break;
		
		case SmartEdit :		
			pBlock = (GoToParamsType *)MemPtrNew(sizeof(GoToParamsType));
			MemPtrSetOwner(pBlock, 0);
			
			pBlock->searchStrLen = length;		// length of search string.
			pBlock->dbCardNo = 0;				// card number of the database	
			if (selectedDBid) {
				pBlock->matchPos = 0;
				pBlock->matchFieldNum = 1;
				pBlock->dbID = selectedDBid;
			}
			else {
				if ((gIncludeTop > 0) &&
						(sourcePtr > includeData[gIncludeTop - 1].includePtr) 
							&& (sourcePtr < (includeData[gIncludeTop - 1].includePtr + includeData[gIncludeTop - 1].includeLength))) {
					RecordLengthInfo *rllp;					
					long sourceOffset = sourcePtr - includeData[gIncludeTop - 1].includePtr;			// postion in record of the match.
					pBlock->dbID = includeData[gIncludeTop - 1].includeDBid;
					rllp = (RecordLengthInfo *)MemHandleLock(includeData[gIncludeTop - 1].recordLengthList);
					for (i = 1; i < rllp[0].uncompressedSize; i++) {
						if ((sourceOffset - rllp[i].uncompressedSize) < 0) {
							pBlock->matchFieldNum = i;
							pBlock->matchPos = sourceOffset;
							break;
						}
						else
							sourceOffset -= rllp[i].uncompressedSize;
					}
					MemHandleUnlock(includeData[gIncludeTop - 1].recordLengthList);
				}
				else {
					RecordLengthInfo *rllp;
					long sourceOffset = sourcePtr - gMainSourceBasePtr;
					if (gErrorInHeader)
						pBlock->dbID = gHeaderDBID;
					else
						pBlock->dbID = getSourceDBid();
					rllp = MemHandleLock(gRecordLengthList);
					for (i = 1; i < rllp[0].uncompressedSize; i++) {
						if ((sourceOffset - rllp[i].uncompressedSize) < 0) {
							pBlock->matchFieldNum = i;
							pBlock->matchPos = sourceOffset;
							break;
						}
						else
							sourceOffset -= rllp[i].uncompressedSize;
					}
					MemHandleUnlock(gRecordLengthList);
				}			
			}
			pBlock->matchCustom = 0;		// application specific info
			
			launchEditor('Smrt', pBlock, sysAppLaunchCmdGoTo, "SmartEdit");
			break;
		case QED : {
				// Get the version of Qed
				UInt16 version = 0;
				DmOpenRef QEDdb = NULL;
				MemHandle h;
				
				DmGetNextDatabaseByTypeCreator(true, &searchState, 'appl', 'QEDi', true, &cardNo, &dbID);
				if (dbID == 0) {
					error("Couldn't find QED");
					break;
				}
				DmDatabaseInfo(0, dbID, 0, 0, &version, 0, 0, 0, 0, 0, 0, 0, 0);		
				
				// use the 'tver' resource to find version number (not the same as the version in the prefs)
				QEDdb = DmOpenDatabase(0, dbID, dmModeReadOnly);
				h = DmGet1Resource('tver', 1);
				if (h != NULL) {
					CharPtr p = MemHandleLock(h);
					if ((p[0] == 'V') && (p[2] == '.')) {
						version = (p[1] - '0') * 100 + (p[3] - '0') * 10 + (p[4] - '0');
					}
					else if (p[1] == '.')
					{
						version = (p[0] - '0') * 100 + (p[2] - '0') * 10 + (p[3] - '0');
					}
					else
					{
						version = 9999; // Maybe the future version?
					}
					MemHandleUnlock(h);
					DmReleaseResource(h);
				}
				DmCloseDatabase(QEDdb);

				// Launch the new version of Qed which supports Goto.
				if (version > 262)
				{
					pBlock = (GoToParamsType *)MemPtrNew(sizeof(GoToParamsType));
					MemPtrSetOwner(pBlock, 0);
	
					if (selectedDBid) {
						pBlock->searchStrLen = 0;
						pBlock->dbID = selectedDBid;
						pBlock->recordNum = - 1;
						pBlock->matchPos = 0;
						pBlock->matchFieldNum = 0;
						pBlock->matchCustom = 0;
					}
					else {
						pBlock->searchStrLen = 0;
						pBlock->recordNum = 0;
						pBlock->matchFieldNum = 0;
						pBlock->matchCustom = 0;
						if ((gIncludeTop > 0) &&
								(sourcePtr > includeData[gIncludeTop - 1].includePtr) 
									&& (sourcePtr < (includeData[gIncludeTop - 1].includePtr + includeData[gIncludeTop - 1].includeLength))) {
							pBlock->matchPos = sourcePtr - includeData[gIncludeTop - 1].includePtr;			// postion in record of the match.
							pBlock->dbID = includeData[gIncludeTop - 1].includeDBid;
						}
						else {
							pBlock->matchPos = sourcePtr - gMainSourceBasePtr;
							if (gErrorInHeader)
								pBlock->dbID = gHeaderDBID;
							else
								pBlock->dbID = getSourceDBid();
						}			
					}
					launchEditor('QEDi', pBlock, sysAppLaunchCmdGoTo, "QED");
				}
				else
				{
				// The following is to launch the old version of Qed.
				// QED doesn't support Goto so we launch it instead and
				// change it's savedPrefs record to the name of the error DB,
				// and set the cursor position in the appInfoBlock of that DB
				QEDPrefsData QEDPrefs;
				QEDPrefsData2 QEDPrefs2;
				QEDPrefsData156 QEDPrefs156;
				char sourceName[dmDBNameLength];
				DmOpenRef savedPrefsDB = NULL;
				DmOpenRef theSourceDB = NULL, QEDdb = NULL;
				LocalID appInfoID;
				long cursorPos;
				UInt version = 0, prefsVersion = 0;
				Handle h;
				
				int i;
				if (selectedDBid) {
					StrCopy(sourceName, selectedDBname);
					cursorPos = 0;
					theSourceDB = NULL;
				}
				else {
					for (i = 0; i < gIncludeTop; i++) {
						if ((sourcePtr > includeData[i].includePtr) 
								&& (sourcePtr < (includeData[i].includePtr + includeData[i].includeLength))) {
							int x;
							LocalID dbID = includeData[i].includeDBid;
							theSourceDB = DmOpenDatabase(0, dbID, dmModeReadWrite);
							cursorPos = sourcePtr - includeData[i].includePtr;
							for (x = 0; x < includeData[i].includeNameLength; x++)
								sourceName[x] = includeData[i].includeName[x];
							sourceName[includeData[i].includeNameLength] = 0;
							break;
						}
					}
					if (i == gIncludeTop) {
						if (gErrorInHeader) {
							theSourceDB = DmOpenDatabase(0, gHeaderDBID, dmModeReadWrite);
							StrCopy(sourceName, HeaderName);
						}
						else {
							theSourceDB = DmOpenDatabase(0, getSourceDBid(), dmModeReadWrite);
							StrCopy(sourceName, getSourceName());
						}
						cursorPos = sourcePtr - gMainSourceBasePtr;
	/*
	
			QED seems to be able to handle getting the uncompressed source-offset ?
	
	*/
					}
				}
						
				DmGetNextDatabaseByTypeCreator(true, &searchState, 'appl', 'QEDi', true, &cardNo, &dbID);
				if (dbID == 0) {
					error("Couldn't find QED");
					break;
				}
				DmDatabaseInfo(0, dbID, 0, 0, &version, 0, 0, 0, 0, 0, 0, 0, 0);		
				
				// use the 'tver' resource to find version number (not the same as the version in the prefs)
				QEDdb = DmOpenDatabase(0, dbID, dmModeReadOnly);
				h = DmGet1Resource('tver', 1);
				if (h != NULL) {
					CharPtr p = MemHandleLock(h);
					if ((p[0] == 'V') && (p[2] == '.')) {
						version = (p[1] - '0') * 100 + (p[3] - '0') * 10 + (p[4] - '0');
					}
					MemHandleUnlock(h);
					DmReleaseResource(h);
				}
				DmCloseDatabase(QEDdb);

				// then get the prefs version #
				savedPrefsDB = DmOpenDatabaseByTypeCreator('sprf', 'psys', dmModeReadOnly);
				if (savedPrefsDB != 0) {
					h = DmGet1Resource('QEDi', 1);
					if (h != NULL) {
						CharPtr p = MemHandleLock(h);
						prefsVersion = *((short *)p);
						MemHandleUnlock(h);
						DmReleaseResource(h);
					}
					DmCloseDatabase(savedPrefsDB);
				}

				if (version >= 200) {
					Word size = sizeof(QEDPrefsData2);
					PrefGetAppPreferences('QEDi', 1, &QEDPrefs2, &size, true);
					StrCopy(QEDPrefs2.name, sourceName);
					PrefSetAppPreferences('QEDi', 1, prefsVersion, &QEDPrefs2, sizeof(QEDPrefsData2), true);
				}
				else {
					if (version >= 156) {
						Word size = sizeof(QEDPrefsData156);
						PrefGetAppPreferences('QEDi', 1, &QEDPrefs156, &size, true);
						StrCopy(QEDPrefs156.name, sourceName);
						PrefSetAppPreferences('QEDi', 1, prefsVersion, &QEDPrefs156, sizeof(QEDPrefsData156), true);
					}
					else {
						Word size = sizeof(QEDPrefsData);
						PrefGetAppPreferences('QEDi', 1, &QEDPrefs, &size, true);
						StrCopy(QEDPrefs.name, sourceName);
						PrefSetAppPreferences('QEDi', 1, prefsVersion, &QEDPrefs, sizeof(QEDPrefsData), true);
					}
				}
				
				if (theSourceDB != NULL) {
					appInfoID = DmGetAppInfoID(theSourceDB);
					if ((appInfoID != 0)) {
						long pageStart = cursorPos;
						long *p = MemLocalIDToLockedPtr(appInfoID, 0);
						int i = 0;
						// try peeking backwards to get the previous line as the one at the top
						// but don't go too far
						while ((*sourcePtr-- != '\n') && (i < 64)) i++;
						pageStart = cursorPos - i;
						if (pageStart < 0) pageStart = 0;
						if (version > 150) {
							if (version >= 200) {
								if (version >= 205) {
									typedef struct {
										long qedversion;		// 'Q205'
										long pageTopOffset;
										long cursorOffset;
										unsigned short pageWidth;
										long flags;
									} QEDAppInfo;
									DmWrite(p, 4, &pageStart, sizeof(long));
									DmWrite(p, 8, &cursorPos, sizeof(long));
/*
									if (((QEDAppInfo *)p)->pageWidth > 800) {
										// older version coming forward
										((QEDAppInfo *)p)->pageWidth = 160;
										((QEDAppInfo *)p)->flags = 0x0101;
										DmWrite(p, 2, &((QEDAppInfo *)p)->pageWidth, sizeof(short) + sizeof(long));
									}
*/									
								}
								else {
									typedef struct {
										long pageTopOffset;
										long cursorOffset;
										unsigned short pageWidth;
										long flags;
									} QEDAppInfo;
									DmWrite(p, 0, &pageStart, sizeof(long));
									DmWrite(p, 4, &cursorPos, sizeof(long));
									if (((QEDAppInfo *)p)->pageWidth > 800) {
										// older version coming forward
										((QEDAppInfo *)p)->pageWidth = 160;
										((QEDAppInfo *)p)->flags = 0x0101;
										DmWrite(p, 2, &((QEDAppInfo *)p)->pageWidth, sizeof(short) + sizeof(long));
									}
								}
							}
							else {
								typedef struct {
									short dontKnow;
									long pageTopOffset;
									long cursorOffset;
								} QEDAppInfo;
								DmWrite(p, 2, &pageStart, sizeof(long));				
								DmWrite(p, 6, &cursorPos, sizeof(long));				
							}
						}
						else {
							DmWrite(p, 0, &pageStart, sizeof(long));
							DmWrite(p, 4, &cursorPos, sizeof(long));
						}
						if (MemLocalIDKind(appInfoID) == memIDHandle) {
							MemHandleUnlock(MemPtrRecoverHandle(p));
						}
					}
					DmCloseDatabase(theSourceDB);
				}

				launchEditor('QEDi', NULL, sysAppLaunchCmdNormalLaunch, "QED");
				}
			}
			break;
			
		case ZDoc :
/* currently zDoC isn't doing a find thing (or much else) so we just launch it */
			launchEditor('ZURK', NULL, sysAppLaunchCmdNormalLaunch, "ZDoc");
			break;
	}
}

Handle copyRecordSource(int recordNum, DmOpenRef theSourceDB, DmOpenRef theCopyDB, UIntPtr at, Handle *recordLengthList)
{
	Handle h = DmGetRecord(theSourceDB, recordNum);
	UInt size = MemHandleSize(h);
	Handle copy = DmNewRecord(theCopyDB, at, size + 1);
	
	CharPtr p = MemHandleLock(copy);
	DmWrite(p, 0, MemHandleLock(h), size);
	DmWrite(p, size, "\0", 1);
	
	MemHandleUnlock(h);
	MemHandleUnlock(copy);
	DmReleaseRecord(theSourceDB, recordNum, false);
	
	*recordLengthList = NULL;
	
	DmReleaseRecord(theCopyDB, *at, true);
	copy = DmGetRecord(theCopyDB, *at);

	return copy;
}

Handle copyMemoSource(int recordNum, DmOpenRef theCopyDB, UIntPtr at, Handle *recordLengthList)
{
	DmOpenRef memoDB = DmOpenDatabaseByTypeCreator('DATA', 'memo', dmModeReadOnly);
	Handle copy = copyRecordSource(recordNum, memoDB, theCopyDB, at, recordLengthList);
	DmCloseDatabase(memoDB);
	return copy;
}

Handle copyPedit32Source(int recordNum, DmOpenRef theCopyDB, UIntPtr at, Handle *recordLengthList)
{
	DmOpenRef pedit32DB = DmOpenDatabaseByTypeCreator('DATA', 'pn32', dmModeReadOnly);
	Handle copy = copyRecordSource(recordNum, pedit32DB, theCopyDB, at, recordLengthList);
	DmCloseDatabase(pedit32DB);
	return copy;
}



