/*
 *	File:		DEBUG.C
 *	Author:		Wade Guthrie
 *	Created:	22 Mar 93
 *
 *	Description:	Debug library.
 *
 *	Inputs:		None.
 *
 *	Outputs:	Diagnostic messages.
 *
 *	Notes:		None.
 */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <stdarg.h>

#include "general.h"	// for isTrue and flags
#include "debug.h"

/*
 * NOTE * NOTE * NOTE * NOTE * NOTE * NOTE * NOTE * NOTE * NOTE * NOTE *
 *
 *          You must define VSNPRINTF correctly for your system!
 *
 * NOTE * NOTE * NOTE * NOTE * NOTE * NOTE * NOTE * NOTE * NOTE * NOTE *
 */

#define	SAFE_SPRINTF

//#define	VSNPRINTF	_vsnprintf	// Microsoft
#define		VSNPRINTF	vsnprintf	// This works fine for GCC

//#define	HIDE_DEBUG_INFO

#define		STRING_SIZE	512

static char	*doBuildError (		char	*format, 
					va_list	ap);
static char	*doBuildWarning (	int	warningThreshold, 
					char	*format, 
					va_list	ap);

/*
 *	The following stuff is for debug output.
 */

static FILE	*debug_file	= NULL;

static int	debugLevel	= 0,
		warningLevel	= 0,
		errorCount	= 0,
		warningCount	= 0;



	void
initializeDebug (
	char	*name,
	char	*debugMode,	// should be "w" or "a"
	int	newDebugLevel,
	int	newWarningLevel)
{
#	define	FUNCTION	"initializeDebug"

	//struct tm	*stime;
	//time_t		ltime;

	/*
	 * open the debug file if there is one
	 */

	if (name != NULL){
		char	*mode	= (debugMode == 0) ? "w" : debugMode;

		// okay to overwrite
		if ((debug_file = fopen (name, mode)) == 0) {
			fprintf (stderr, "** %s: couldn't open %s\n", 
				FUNCTION, name);
			perror (FUNCTION);
		} else {
			// set output to file to be unbuffered
			setvbuf (debug_file, NULL, _IONBF, 0) ;	

			//(void)time(&ltime);
			//stime = localtime(&ltime);
			//fprintf (debug_file, 
			//	"This file was created automatically on %s\n",
			//	asctime(stime));
		}
	}

	errorCount	= 0;
	warningCount	= 0;
	setDebugLevel (newDebugLevel);
	setWarningLevel (newWarningLevel);
#	undef	FUNCTION
}

	int
getErrorCount (void)
{
#	define	FUNCTION	"getErrorCount"
	return errorCount;
#	undef	FUNCTION
}

	int
getWarningCount (void)
{
#	define	FUNCTION	"getWarningCount"
	return warningCount;
#	undef	FUNCTION
}

	void
printDebugTally (void)
{
#	define	FUNCTION	"printDebugTally"

	fprintf (stderr, "\t%d Error(s) were reported\n", errorCount);
	fprintf (stderr, "\t%d Warning(s) were reported\n", warningCount);
	fflush (stderr);

	/* if we're doing debug to a file, output it there too */

	if (debug_file != NULL) {
		fprintf (debug_file, "\t%d Error(s) were reported\n", 
				errorCount);
		fprintf (debug_file, "\t%d Warning(s) were reported\n", 
				warningCount);
		fflush (debug_file);
	}
#	undef	FUNCTION
}

	void
finishDebug (void)
{
#	define	FUNCTION	"finishDebug"
	//fclose (debug_file);
	//debug_file	= 0;
#	undef	FUNCTION
}

	int
getDebugLevel (void)
{
#	define	FUNCTION	"getDebugLevel"
	return debugLevel;
#	undef	FUNCTION
}

	void	
setDebugLevel (int newDebugLevel)
{
#	define	FUNCTION	"setDebugLevel"
	debugLevel	= newDebugLevel;
#	undef	FUNCTION
}

	int	
getWarningLevel (void)
{
#	define	FUNCTION	"getWarningLevel"
	return warningLevel;
#	undef	FUNCTION
}


	void	
setWarningLevel (int newWarningLevel)
{
#	define	FUNCTION	"setWarningLevel"
	warningLevel	= newWarningLevel;
#	undef	FUNCTION
}




/*
 * debug_printf
 */

	void
debug_printf (int debugThreshold, char *format, ...)
{
#	define	FUNCTION	"debug_printf"

	va_list	ap;

	va_start (ap, format);

	if (debugLevel >= debugThreshold) {

		if (debug_file != NULL) {
			vfprintf(debug_file, format, ap);	/* -> to file */
			fflush(debug_file);

#			ifndef	HIDE_DEBUG_INFO
				vfprintf(stdout, format, ap);	/* -> screen */
				fflush (stdout);
#			endif
		} else {
			vfprintf(stdout, format, ap);		/* -> screen */
			fflush (stdout);
		}
	}
#	undef	FUNCTION
}

	void
warningPrintf (int warningThreshold, char *format, ...)
{
#	define	FUNCTION	"warningPrintf"

	va_list	ap;
	char	*string	= 0;

	va_start (ap, format);

	/* output to the display */

	if ((string = doBuildWarning (warningThreshold, format, ap)) != 0) {
		fputs (string, stderr);
		fflush (stderr);
	}
#	undef	FUNCTION
}


	char	*
buildWarning (int warningThreshold, char *format, ...)
{
#	define	FUNCTION	"buildWarning"

	va_list		ap;
	// static char	string[STRING_SIZE];
	// char		*answer	= string;

	va_start (ap, format);

	return doBuildWarning (warningThreshold, format, ap);
#	undef	FUNCTION
}

	static char	*
doBuildWarning (int warningThreshold, char *format, va_list ap)
{
#	define	FUNCTION	"doBuildWarning"

	static char	string[STRING_SIZE];
	char		*answer	= string;

	/* output to the display */

	if (warningLevel >= warningThreshold) {

#		ifdef SAFE_SPRINTF
			VSNPRINTF (string, sizeof(string), format, ap);
#		else
			vsprintf (string, format, ap);
#		endif

		string[sizeof(string)-1] = '\0';
		answer	= string;

		warningCount++;

		if (debug_file != NULL) {
			fputs (string, debug_file);
			fflush (debug_file);
		}
	} else {
		answer	= 0;
	}

	return answer;
#	undef	FUNCTION
}



/*
 * Error stuff
 */

	void
errorPrintf (char *format, ...)
{
#	define	FUNCTION	"errorPrintf"

	va_list	ap;
	char	*string	= 0;

	va_start (ap, format);

	/* output to the display */

	if ((string = doBuildError (format, ap)) != 0) {
		fputs (string, stderr);
		fflush (stderr);

	}
#	undef	FUNCTION
}


	char	*
buildError (char *format, ...)
{
#	define	FUNCTION	"buildError"

	// static char	string[STRING_SIZE];
	va_list		ap;

	va_start (ap, format);

	return doBuildError (format, ap);
#	undef	FUNCTION
}

	static char	*
doBuildError (char *format, va_list ap)
{
#	define	FUNCTION	"doBuildError"

	static char	string[STRING_SIZE];

#	ifdef SAFE_SPRINTF
		VSNPRINTF (string, sizeof(string), format, ap);
#	else
		vsprintf (string, format, ap);
#	endif

	string[sizeof(string)-1] = '\0';

	if (debug_file != NULL) {
		fputs (string, debug_file);
		fflush (debug_file);
	}

	errorCount++;

	return string;
#	undef	FUNCTION
}

#ifdef USING_PROFILING

/*
 * Rough Profiling Tools -- this stuff is currently not very well tested.
 * Don't use it without working with it a bit first.
 */

struct timeval time_reference;

	void
reset_time (
	char	*string,
	int	debugThreshold )
{
#	define	FUNCTION	"reset_time"

	struct timezone	tz;

	if (debugLevel >= debugThreshold)
	{
		if (string == 0)
			string = FUNCTION;

		(void) gettimeofday (&time_reference, &tz);
		debugPrintf ((0, "%s: Resetting Time at %ld s %ld us\n", 
			string, time_reference.tv_sec, time_reference.tv_usec));
	}
#	undef	FUNCTION
}

	void
mark_time (
	char	*string,
	int	debugThreshold )
{
#	define	FUNCTION	"mark_time"

	struct timeval	tv;
	struct timezone	tz;

	if (debugLevel >= debugThreshold) {

		if (string == 0)
			string = FUNCTION;

		(void) gettimeofday (&tv, &tz);
		debugPrintf ((0, 
			"%s: Delta Time at %ld s, %ld us (T= %ld s %ld us)\n", 
			string,
			tv.tv_sec - time_reference.tv_sec, 
			tv.tv_usec - time_reference.tv_usec,
			time_reference.tv_sec, 
			time_reference.tv_usec));
	}
#	undef	FUNCTION
}
#endif	/* USING_PROFILING */

void	doIgnore (const int intentionallyIgnored) {}

/*
 * This file will open a file only if either 1) it doesn't already exist or 
 * 2) if 'FlagOverwriteFile' is set.
 *
 * The idea with this function, is that you call it (presumably with a "w" 
 * mode) in case you don't want to overwrite an existing file.
 */

	FILE	*
openNewFile (char *filename, char *mode)
{
#	define	FUNCTION	"openNewFile"
	FILE	*file	= 0;

	// if we can overwrite the file even if it exists, go ahead and open
	// it

	if (isTrue(FlagOverwriteFile))
	{
		if ((file = fopen (filename, mode)) == 0)
		{
			errorPrintf (
				"** ERROR (%s): Problem opening file '%s', %s\n"
				FUNCTION, filename, strerror (errno));
		}
	}

	// check to see if the file is there. . .
	else if ((file = fopen (filename, "r")) == 0)
	{
		/*
		 * There's a shorter way of writing the following stuff but it
		 * isn't NEARLY as clear.
		 */

		// the file isn't there -- go ahead and open it
		if (errno == ENOENT)
		{
			if ((file = fopen (filename, mode)) == 0)
			{
				errorPrintf (
					"** ERROR (%s): Problem opening "
					"file '%s', %s\n"
					FUNCTION, filename, strerror (errno));
			}
		}

		// we couldn't open the file for other reasons
		else
		{
			errorPrintf (
				"** ERROR (%s): Problem opening file '%s', %s\n"
				FUNCTION, filename, strerror (errno));
		}
	}

	// ...oops, the file *IS* there
	else
	{
		errorPrintf (
			"** ERROR (%s): Not overwriting exist file '%s' -- "
			"try using --overwrite flag\n",
			FUNCTION, filename);

		fclose (file);
		file	= 0;
	}

	return file;
#	undef	FUNCTION
}
