/*
 * This code is from MakeDoc.  The source was found, unattributed, on the net.
 *
 * This file converts between PalmOS's DOC format and text files on the host
 */

#ifdef sparc
#	ifndef UNIX
#	define UNIX 1
#	endif
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>	// for strerror
#include <errno.h>	// for errno
#include "debug.h"
#include "general.h"

/*
 * Typedefs and defines
 */

typedef unsigned char byte;

#define DISP_BITS 11
#define COUNT_BITS 3

// all numbers in these structs are big-endian, MAC format
typedef struct tDocHeader 
{
	char sName[32];		// 32 bytes	0-1f
	tIntU32 dwUnknown1;	// 36		20-23
	tIntU32 dwTime1;	// 40		24-27
	tIntU32 dwTime2;	// 44		28-2b
	tIntU32 dwTime3;	// 48		2c-2f
	tIntU32 dwLastSync;	// 52		30-33
	tIntU32 ofsSort;	// 56		34-37
	tIntU32 ofsCatagories;	// 60		38-3b
	tIntU32 dwCreator;	// 64		3c-3f
	tIntU32 dwType;		// 68		40-43
	tIntU32 dwUnknown2;	// 72		44-47
	tIntU32 dwUnknown3;	// 76		48-4b
	tIntU16  wNumRecs;	// 78		4c-4d
} tDocHeader;

// Some compilers pad structures out to tIntU32 boundaries so using sizeof()
// doesn't give the intended result.
#define DOCHEADSZ 78

typedef struct tDocRecord0 
{
	tIntU16 wVersion;	// 4e-4f	1=plain text, 2=compressed
	tIntU16 wSpare;		// 50-51
	tIntU32 dwStoryLen;	// 52-55	in bytes, when decompressed
	tIntU16 wNumRecs; 	// 56-57	text records only; equals tDocHeader.wNumRecs-1
	tIntU16 wRecSize;	// 58-59	usually 0x1000
	tIntU32 dwSpare2;	// 60-63
} tDocRecord0;

typedef struct tBuf 
{
	byte		*buf;
	unsigned	len;

}tBuf;

/*
 * Globals
 */

tIntU16 (*Swap16)(tIntU16 r) = NULL;
tIntU32 (*Swap32)(tIntU32 r) = NULL;

/*
 * Prototypes
 */

static tIntU16	SwapWord21	(tIntU16 r);
static tIntU16	SwapWord12	(tIntU16 r);
static tIntU32	SwapLong4321	(tIntU32 r);
static tIntU32	SwapLong1234	(tIntU32 r);
static byte	*memfind	(byte* t, int t_len, byte* m, int m_len);

void		Doc2Text	(char	*src, char* dest, int bBinary);
extern void	Doc2Text_wrap	(char	*nameOnPalm,
				 char	*nameOnHost);

void		Text2Doc	(char	*txtFileName, 
				 char	*pdbFileName, 
				 char	*docFileName,
				 int	bDecomp,
				 int	bBinary,
				 int	bReport,
				 int	bCompress);
int		SwapChoose	(void);

static void	out_word	(short w, FILE* fout);
static void	out_long	(long d, FILE* fout);

static tBuf	*newtBuf	() ;
static void	deletetBuf	(tBuf *that);

static unsigned	Issue		(tBuf *that, byte src, int *bSpace);
static unsigned	RemoveBinary	(tBuf *that);
static unsigned	Decompress	(tBuf *that);
static unsigned	Compress	(tBuf *that);
static unsigned	DuplicateCR	(tBuf *that);
static void	Clear		(tBuf *that);


	static tIntU16 
SwapWord21(tIntU16 r)
{
#	define	FUNCTION	"SwapWord21"
	return (r>>8) + (r<<8);
#	undef	FUNCTION
}


	static tIntU16 
SwapWord12(tIntU16 r)
{
#	define	FUNCTION	"SwapWord12"
	return r;  
#	undef	FUNCTION
}



	static tIntU32 
SwapLong4321(tIntU32 r)
{
#	define	FUNCTION	"SwapLong4321"
	return  ((r>>24) & 0xFF) 
			+ (r<<24) 
			+ ((r>>8) & 0xFF00) 
			+ ((r<<8) & 0xFF0000);
#	undef	FUNCTION
}



	static tIntU32 
SwapLong1234(tIntU32 r)
{
#	define	FUNCTION	"SwapLong1234"
	return r;
#	undef	FUNCTION
}



// copy bytes into a word and double word and see how they fall,
// then choose the appropriate swappers to make things come out
// in the right order.

	int 
SwapChoose(void)
{
#	define	FUNCTION	"SwapChoose"
  union { char b[2]; tIntU16 w; } w;
  union { char b[4]; tIntU32 d; } d;

  strncpy(w.b, "\1\2", 2);
  strncpy(d.b, "\1\2\3\4", 4);

  if (w.w == 0x0201)
    Swap16 = SwapWord21;
  else if (w.w == 0x0102)
    Swap16 = SwapWord12;
  else
    return 0;

  if (d.d == 0x04030201)
    Swap32 = SwapLong4321;
  else if (d.d == 0x01020304)
    Swap32 = SwapLong1234;
  else
    return 0;
  
  return 1;
#	undef	FUNCTION
}  



// replacement for strstr() which deals with 0's in the data
	static byte* 
memfind(byte* t, int t_len, byte* m, int m_len)
{
#	define	FUNCTION	"memfind"
	int i;

	for (i = t_len - m_len + 1 ; i>0; i--, t++)
	{
		if (t[0]==m[0] && memcmp(t,m,m_len)==0)
		{
			return t;
		}
	}

	return 0;
#	undef	FUNCTION
}





	static tBuf *
newtBuf() 
{
#	define	FUNCTION	"newtBuf"
	tBuf	*that	= 0;

	if ((that = (tBuf *) malloc (sizeof (tBuf))) == 0)
	{
		errorPrintf ("** ERROR (%s): couldn't allocate space\n", 
			FUNCTION);
		return 0;
	}

	that->len = 6000;
	if ((that->buf = (byte *) malloc (sizeof(byte) * 6000)) == 0)
	{
		errorPrintf ("** ERROR (%s): couldn't allocate space\n", 
			FUNCTION);
		return 0;
	}

	return that;
#	undef	FUNCTION
};

	static void
deletetBuf(tBuf *that)
{
#	define	FUNCTION	"deletetBuf"
	if (that != 0)
	{
		if (that->buf) 
			free (that->buf); 
		free (that);
	}
#	undef	FUNCTION
}

//
// Issue()
//
// action: handle the details of writing a single
//		character to the compressed stream
//

	static unsigned
Issue(tBuf *that, byte src, int *bSpace)
{
#	define	FUNCTION	"Issue"

	int	iDest;
	byte	*dest = 0;

	if (that == 0)
		return 0;

	iDest	= that->len;
	dest	= that->buf;

	// if there is an outstanding space char, see if
	// we can squeeze it in with an ASCII char
	if (*bSpace != 0)
	{
		if ((src >= 0x40) && (src <= 0x7F))
		{
			dest[iDest++] = src ^ 0x80;
		}
		else
		{
			// couldn't squeeze it in, so issue the space char by itself
			// most chars go out simple, except the range 1...8,0x80...0xFF
			dest[iDest++] = ' ';
			if (src<0x80 && (src==0 || src>8) )
				dest[iDest++] = src;
			else
				dest[iDest++] = 1, dest[iDest++] = src;
		}
		// knock down the space flag
		*bSpace = 0;
	}
	else
	{
		// check for a space char
		if (src==' ')
			*bSpace = 1;
		else
		{
			if (src<0x80 && (src==0 || src>8))
				dest[iDest++] = src;
			else
				dest[iDest++] = 1, dest[iDest++] = src;

		}
	}
	that->len = iDest;
	return iDest;
#	undef	FUNCTION
}

//
// Compress
//
// params: 	none
//
// action:	takes the given buffer,
//					and compresses
//					the original data down into a second buffer
//
// comment:	This version make heavy use of walking pointers.
//
	static unsigned 
Compress(tBuf *that)
{
#	define	FUNCTION	"Compress"

	int	i; //,j;
	int	k;
	int	bSpace		= 0;

	// run through the input buffer
	byte	*pBuffer	= 0;	// points to the input buffer
	byte	*pHit		= 0;		// points to a walking test hit; works upwards on successive matches
	byte	*pPrevHit	= 0;	// previous value of pHit
	byte	*pTestHead	= 0;	// current test string
	byte	*pTestTail	= 0;	// current walking pointer; one past the current test buffer
	byte	*pEnd		= 0;		// 1 past the end of the input buffer

	if (that == 0)
		return 0;

	pHit		= pPrevHit = pTestHead = pBuffer = that->buf;
	pTestTail	= pTestHead+1;
	pEnd		= that->buf + that->len;

	//printf("pointers %x %x",pTestTail, pEnd);
	//printf("\nstart compression buf len=%d",that->len);

	// make a dest buffer and reassign the local buffer
	if ((that->buf = (byte *) malloc (sizeof (byte) * 6000)) == 0)
	{
		errorPrintf ("** ERROR (%s): couldn't allocate space\n", FUNCTION);
		return 0;
	}
	that->len = 0;		// used to walk through the output buffer

	// loop, absorbing one more char from the input buffer on each pass
	for (; pTestHead != pEnd; pTestTail++)
	{
		//printf("\npointers pTestHead %x pTestTail %x pTestHead[]=%x %x",pTestHead, pTestTail, pTestHead[0], pTestHead[1]);
		// establish where the scan can begin
		if (pTestHead - pPrevHit > ((1<<DISP_BITS)-1))
			pPrevHit = pTestHead - ((1<<DISP_BITS)-1);

		// scan in the previous data for a match
		pHit = memfind(pPrevHit, pTestTail - pPrevHit, pTestHead, pTestTail - pTestHead);

		if (pHit==0)
			errorPrintf("** ERROR (%s): source %x%x%x, dest %x%x%x, %d bytes",	
				FUNCTION,
				pPrevHit[0],
				pPrevHit[1],
				pPrevHit[2],
				pTestHead[0],
				pTestHead[1],	
				pTestHead[2],	
				pTestTail-pTestHead);

		// on a mismatch or end of buffer, issued codes
		if (pHit==0
			|| pHit==pTestHead
			|| pTestTail-pTestHead>(1<<COUNT_BITS)+2
			|| pTestTail==pEnd)
		{
			// issued the codes
			// first, check for short runs
			if (pTestTail-pTestHead < 4)
			{
				//printf("\nissue a char %x",pTestHead[0]);
				Issue(that, pTestHead[0], &bSpace);
				pTestHead++;
			}
			// for longer runs, issue a run-code
			else
			{
				unsigned int dist;
				unsigned int compound;

				// issue space char if required
				if (bSpace) 
					that->buf[that->len++] = ' ', bSpace = 0;

				dist		= pTestHead - pPrevHit;
				compound	= (dist << COUNT_BITS) + pTestTail-pTestHead - 4;

				if (dist>=(1<<DISP_BITS)) 
					errorPrintf(
						"** ERROR (%s): dist overflow\n",
						FUNCTION);

				if (pTestTail-pTestHead-4>7) 
					errorPrintf(
						"** ERROR (%s): dist overflow\n",
						FUNCTION);

				that->buf[that->len++] = 0x80 + (compound>>8);
				that->buf[that->len++] = compound & 0xFF;
				//printf("\nissuing code for sequence len %d <%c%c%c>",
				//	pTestTail-pTestHead-1,pTestHead[0],
				//	pTestHead[1],pTestHead[2]);
				//printf("\n          <%x%x>",pOut[-2],pOut[-1]);
				// and start again
				pTestHead = pTestTail-1;
			}
			// start the search again
			pPrevHit = pBuffer;
		}
		// got a match
		else
		{
			pPrevHit = pHit;
		}
		//printf("pointers %x %x %x",pTestHead, pTestTail, pPrevHit);
		// when we get to the end of the buffer, don't inc past the end
		// this forces the residue chars out one at a time
		if (pTestTail==pEnd) 
			pTestTail--;
	}

	// clean up any dangling spaces
	if (bSpace) 
		that->buf[that->len++] = ' ';


	// final scan to merge consecutive high chars together
	for (i=k=0; (unsigned) i < that->len; i++,k++)
	{
		that->buf[k] = that->buf[i];
		// skip the run-length codes
		if (that->buf[k]>=0x80 && that->buf[k]<0xC0)
		{
			that->buf[++k] = that->buf[++i];
		}
		// if we hit a high char marker, look ahead for another
		else if (that->buf[k]==1)
		{
			that->buf[k+1] = that->buf[i+1];
			while	(((unsigned) (i+2) < that->len)
				&& (that->buf[i+2] == 1)
				&& (that->buf[k] < 8))
			{
				that->buf[k]++;
				that->buf[k+that->buf[k]] = that->buf[i+3];
				i+=2;
			}
			k += that->buf[k]; i++;
		}
	}

	// delete original buffer
	free (pBuffer);
	that->len = k;

	return k;
#	undef	FUNCTION
}
/*
	Decompress

	params:	none

	action: make a new buffer
					run through the source data
					check the 4 cases:
						0,9...7F represent self
						1...8		escape n chars
						80...bf reference earlier run
						c0...ff	space+ASCII

*/
	static unsigned
Decompress(tBuf *that)
{
#	define	FUNCTION	"Decompress"

	// we "know" that all decompresses fit within 4096, right?
	byte		*pOut		= (byte *) malloc ( sizeof (byte) * 6000);
	byte		*out_buf	= pOut;
	byte		*in_buf		= 0;
	int		i,j;
	unsigned int	c;

	if (that == 0)
	{
		errorPrintf ("** ERROR (%s): NULL parameter\n", FUNCTION);
		return 0;
	}
	else if (pOut == 0)
	{
		errorPrintf ("** ERROR (%s): couldn't allocate space\n", FUNCTION);
		return 0;
	}
	else if (that->buf == 0)
	{
		errorPrintf ("** ERROR (%s): that->buf is NULL\n", FUNCTION);
		return 0;
	}

	in_buf		= that->buf;

	for (j=i=0; (unsigned) j < that->len; )
	{
		// take a char from the input buffer
		c = in_buf[j++];

		// separate the char into zones: 0, 1...8, 9...0x7F, 0x80...0xBF, 0xC0...0xFF

		// codes 1...8 mean copy that many bytes; for accented chars & binary
		if (c>0 && c<9)
		{
			while(c--) 
			{
				out_buf[i++] = in_buf[j++];
			}
		}

		// codes 0, 9...0x7F represent themselves
		else if (c<0x80)
		{
			out_buf[i++] = c;
		}

		// codes 0xC0...0xFF represent "space + ascii char"
		else if (c>=0xC0)
		{
				out_buf[i++] = ' ', out_buf[i++] = c ^ 0x80;
		}

		// codes 0x80...0xBf represent sequences
		else
		{
			int m,n;
			c <<= 8;
			c += in_buf[j++];
			m = (c & 0x3FFF) >> COUNT_BITS;
			n = c & ((1<<COUNT_BITS) - 1);
			n += 3;
			while (n--)
			{
				out_buf[i] = out_buf[i-m];
				i++;
			}
		}
	}
	free (that->buf);
	that->buf = pOut;
	that->len = i;

	return i;
#	undef	FUNCTION
}

	static unsigned 
DuplicateCR(tBuf *that)
{
#	define	FUNCTION	"DuplicateCR"

	byte* pBuf = 0;
	int k,j;

	if (that == 0)
	{
		errorPrintf ("** ERROR (%s): NULL parameter\n", FUNCTION);
		return 0;
	}
	else if (that->buf == 0)
	{
		errorPrintf ("** ERROR (%s): N->buf is NULL\n", FUNCTION);
		return 0;
	}
	else if ((pBuf = (byte *) malloc (sizeof (byte) *2*that->len)) == 0)
	{
		errorPrintf ("** ERROR (%s): Couldn't allocate memory\n", 
			FUNCTION);
		return 0;
	}
	else
	{
		for (j=k=0; (unsigned) j < that->len; j++, k++)
		{
			pBuf[k] = that->buf[j];
			if (pBuf[k]==0x0A) 
			{
				pBuf[k++] = 0x0D, pBuf[k] = 0x0A;
			}
		}
		free (that->buf);
		that->buf = pBuf;
		that->len = k;
	}
	return k;
#	undef	FUNCTION
}

	void		
Doc2Text_wrap	(char	*nameOnPalm,
		 char	*nameOnHost)
{
#	define	FUNCTION	"Doc2Text_wrap"

	int	bBinary	= 0;

	debugPrintf ((DEBUG_TRIVIAL, "%s: turning '%s' into '%s'\n",
		FUNCTION,
		nameOnPalm,
		nameOnHost));

	if ((nameOnPalm != 0) && (nameOnHost != 0))
		Doc2Text (nameOnPalm, nameOnHost, bBinary);

#	undef	FUNCTION
}

	void 
Doc2Text (char* src, char* dest, int bBinary)
{
#	define	FUNCTION	"Doc2Text"

	FILE		*fin	= 0;
	FILE		*fout	= 0;
	tIntU32		dwPos;
	tIntU32		dwLen;
	tIntU32		dwRecLen;
	tIntU16		nRecs;
	tIntU16		bCompressed;
	tDocHeader	head;
	tDocRecord0	rec0;
	tBuf		*t = newtBuf();
	int		i;
	int		n;

	if (t == 0)
	{
		errorPrintf("** ERROR (%s): Couldn't allocate space\n", 
			FUNCTION);
		return;
	}
	else if ((fin = fopen(src,"rb")) == 0)
	{
		errorPrintf("** ERROR (%s): Problem opening source file '%s'\n", 
			FUNCTION, src);
		deletetBuf(t);
		return;
	}
	else if (fread(&head, 1, DOCHEADSZ, fin) != DOCHEADSZ)
	{
		errorPrintf("** ERROR (%s): Problem reading from file, %s\n", 
			FUNCTION, strerror(errno));
		return;
	}

	// just holds the first few bytes of the file

	if (strncmp((char *)&head.dwType, "REAd", 4) != 0
		|| strncmp((char *)&head.dwCreator, "TEXt", 4) != 0)
	{
		errorPrintf("** ERROR (%s): File '%s' is bad format with "
			"creator '%.4s', type '%.4s'\n", 
			FUNCTION,
			src,
			(char *)&head.dwCreator, 
			(char *)&head.dwType);
		deletetBuf(t);
		return;
	}

	// point to start of index
	if (fseek(fin, 0x4E, SEEK_SET) != 0)
	{
		errorPrintf("** ERROR (%s): Problem with fseek, %s\n", 
			FUNCTION, strerror(errno));
		return;
	}

	// read the location of the first record
	if (fread(&dwPos, 4, 1, fin) != 1)
	{
		errorPrintf("** ERROR (%s): Problem reading from file, %s\n", 
			FUNCTION, strerror(errno));
		return;
	}

	dwPos = Swap32(dwPos);
	if (fseek(fin, dwPos, SEEK_SET) != 0)
	{
		errorPrintf("** ERROR (%s): Problem with fseek, %s\n", 
			FUNCTION, strerror(errno));
		return;
	}

	if (fread(&rec0, sizeof(rec0), 1, fin) != 1)
	{
		errorPrintf("** ERROR (%s): Problem reading from file, %s\n", 
			FUNCTION, strerror(errno));
		return;
	}

	bCompressed = Swap16(rec0.wVersion);
	if (bCompressed!=1 && bCompressed!=2)
	{
		warningPrintf(1, "** WARNING: unknown file compression type:%d\n",
			bCompressed);
	}
	bCompressed--;

	if ((fout = openNewFile(dest,"wb")) == 0)
	{
		// 'openNewFile' already generated the error message
		//errorPrintf("** ERROR (%s): Problem opening output file '%s'\n",
		//	FUNCTION, dest);
		deletetBuf(t);
		return;
	}

	if (fseek(fin,0,SEEK_END) != 0)
	{
		errorPrintf("** ERROR (%s): Problem with fseek, %s\n", 
			FUNCTION, strerror(errno));
		return;
	}

	dwLen = ftell(fin);

	nRecs = Swap16(head.wNumRecs) - 1;

	// this is the main record buffer
	// it knows how to stretch to accomodate the decompress
	for (i=0; i<nRecs; i++)
	{
		// read the record offset
		if (fseek(fin, 0x56 + 8*i, SEEK_SET) != 0)
		{
			errorPrintf("** ERROR (%s): Problem with fseek, %s\n", 
				FUNCTION, strerror(errno));
			return;
		}

		if (fread(&dwPos, 4, 1, fin) != 1)
		{
			errorPrintf("** ERROR (%s): Problem reading from file, %s\n", 
				FUNCTION, strerror(errno));
			return;
		}

		dwPos = Swap32(dwPos);

		// read start of next record
		if (fseek(fin, 0x5E + 8*i, SEEK_SET) != 0)
		{
			errorPrintf("** ERROR (%s): Problem with fseek, %s\n", 
				FUNCTION, strerror(errno));
			return;
		}

		if (fread(&dwRecLen, 4, 1, fin) != 1)
		{
			errorPrintf("** ERROR (%s): Problem reading from file, %s\n", 
				FUNCTION, strerror(errno));
			return;
		}

		dwRecLen = Swap32(dwRecLen);

		// for the last, use the file len
		if (i==nRecs-1) 
			dwRecLen = dwLen;

		dwRecLen -= dwPos;

		if (fseek(fin,dwPos,SEEK_SET) != 0)
		{
			errorPrintf("** ERROR (%s): Problem with fseek, %s\n", 
				FUNCTION, strerror(errno));
			return;
		}

		if ((n = fread(t->buf, 1, dwRecLen, fin)) != dwRecLen)
		{
			errorPrintf("** ERROR (%s): Problem reading from file, %s\n", 
				FUNCTION, strerror(errno));
			return;
		}

		t->len = n;
		if(bCompressed)
			Decompress(t);

		// check for CR insert
		if (!bBinary)
			DuplicateCR(t);

		debugPrintf((DEBUG_MODERATE, 
			"reconverting %s: record %d of %d\n",
			head.sName,i,nRecs));

		if (fwrite(t->buf, 1, t->len, fout) != t->len)
		{
			errorPrintf("** ERROR (%s): Problem writing to file, %s\n", 
				FUNCTION, strerror(errno));
			return;
		}

	}

	fclose(fin);
	fclose(fout);
	deletetBuf(t);

#	undef	FUNCTION
}

	void
Text2Doc	(char	*txtFileName, 
		 char	*pdbFileName, 
		 char	*docFileName,
		 int	bDecomp,
		 int	bBinary,
		 int	bReport,
		 int	bCompress)
{
#	define	FUNCTION	"Text2Doc"

	tDocHeader	head1;
	tDocRecord0	rec0;

	FILE		*fin		= 0;
	FILE		*fout		= 0;
	tBuf		*buf		= newtBuf();
	tIntU32		storySize	= 0;
	tIntU32		x		= 0;
	tIntU32		z		= 0,
			numRecs		= 0;
	long		recSize		= 4096;
	unsigned long	index		= 0;
	int		n		= 0;
	int		recNum		= 0;
	int		i		= 0;

	if ((fin = fopen(txtFileName, "rb")) == 0)
	{
		errorPrintf("** ERROR (%s): Problem opening '%s', %s\n", 
			FUNCTION, txtFileName, strerror(errno));
		deletetBuf(buf);
		return;
	}
	else if ((fout = openNewFile(pdbFileName, "wb")) == 0)
	{
		// 'openNewFile' already generated the error message
		//errorPrintf("** ERROR (%s): Problem opening '%s', %s\n", 
		//	FUNCTION, pdbFileName, strerror (errno));
		deletetBuf(buf);
		return;
	}

	if (fseek(fin,0,SEEK_END) != 0)
	{
		errorPrintf("** ERROR (%s): Problem with fseek, %s\n", 
			FUNCTION, strerror(errno));
		return;
	}

	storySize = ftell(fin);
	if (fseek(fin,0,SEEK_SET) != 0)
	{
		errorPrintf("** ERROR (%s): Problem with fseek, %s\n", 
			FUNCTION, strerror(errno));
		return;
	}

	for (i=0; i<32; i++)	// WDG
		head1.sName[i] = 0;

	sprintf(head1.sName,"%.31s", docFileName);
	head1.sName[31] = 0;
	debugPrintf((DEBUG_MODERATE, "saving to %s as <%s>,%s%s compressed",
		pdbFileName,
		docFileName,
		bBinary ? " binary mode," : "",
		bCompress ? "" : " not"));

	head1.dwUnknown1 = 0;
	strncpy((char *)&head1.dwTime1, "\x06\xD1\x44\xAE", 4);
	strncpy((char *)&head1.dwTime2, "\x06\xD1\x44\xAE", 4);
	head1.dwTime3 = 0;
	head1.dwLastSync = 0;
	head1.ofsSort = 0;
	head1.ofsCatagories = 0;
	strncpy((char *)&head1.dwCreator, "TEXt", 4);	// database creator
	strncpy((char *)&head1.dwType, "REAd", 4);	// database type
	head1.dwUnknown2 = 0;
	head1.dwUnknown3 = 0;

	z = (int) (storySize/(long) recSize);
	if (((tIntU32) ((long) z * recSize)) < storySize)
		z ++;

	numRecs = z;
	z ++;

	head1.wNumRecs = Swap16(z);		//  the number of records to follow

	if (fwrite(&head1,1,DOCHEADSZ,fout) != DOCHEADSZ)
	{
		errorPrintf("** ERROR (%s): Problem writing to file, %s\n", 
			FUNCTION, strerror(errno));
		return;
	}

	/*
	 *
	 */

	index = 0x406F8000;		// the pattern for attributes=dirty + unique_id=0x6f8000
	x = 0x50L + (long) z * 8;

	out_long(x,fout);		// start writing the record offsets
	out_long(index,fout);
	x += 0x0010L;

	index++;
	z--;

	while(z--) 
	{
		out_long(x,fout);		//more record offsets
		out_long(index++,fout);		// the attributes + ID's
		x += 0x1000L;
	}
	// one more word.....
	out_word(0,fout);


	rec0.wVersion	= Swap16(bCompress ? 2 : 1);
	rec0.wSpare	= 0;
	rec0.dwStoryLen	= Swap32(storySize);
	rec0.wNumRecs	= Swap16(Swap16(head1.wNumRecs) - 1);
	rec0.wRecSize	= Swap16(recSize);
	rec0.dwSpare2	= 0;

	if (fwrite(&rec0,1,sizeof(rec0),fout) != sizeof(rec0))
	{
		errorPrintf("** ERROR (%s): Problem writing to file, %s\n", 
			FUNCTION, strerror(errno));
		return;
	}

	/*
	 *
	 */

	n = recSize;
	// dump the whole story into the new file
	recNum = 0;

	while(((tIntU32)recNum) < numRecs)
	{
		long pos;
		int nOrg;

		pos = ftell(fout);
		fseek(fout, 0x56 + 8*recNum, SEEK_SET);
		if (((tIntU32) recNum) != numRecs) 
		{
			out_long(pos,fout);
		}
		fseek(fout, pos, SEEK_SET);

		Clear(buf);

		if ((nOrg = n = fread(buf->buf,1,4096,fin)) != 4096)
		{
			errorPrintf("** ERROR (%s): Problem reading from file, %s\n", 
				FUNCTION, strerror(errno));
			return;
		}

		buf->len	= n;

		if (n==0) 
			break;

		if (!bBinary)
			RemoveBinary(buf);
		if (bCompress)
			Compress(buf);
		if ((n = fwrite(buf->buf,1,buf->len,fout)) != buf->len)
		{
			errorPrintf("** ERROR (%s): Problem writing to file, %s\n", 
				FUNCTION, strerror(errno));
			return;
		}

		debugPrintf((DEBUG_MODERATE, "Converting record %d of %lu\n",
			recNum+1,numRecs));

		if (bReport && n && bCompress)
			debugPrintf((DEBUG_MODERATE, 
				"original %d bytes, compressed to %d bytes, ratio: %f5.1\n",
				nOrg, n, 100. * n / nOrg));
		recNum++;
	}

	fclose(fin);
	fclose(fout);
	deletetBuf(buf);
#	undef	FUNCTION
}

// this nasty little beast removes really low ASCII and 0's
// and handles the CR problem
//
// if a cr appears before a lf, then remove the cr
// if a cr appears in isolation, change to a lf
	static unsigned 
RemoveBinary(tBuf *that)
{
#	define	FUNCTION	"RemoveBinary"
	byte	*in_buf		= 0;
	byte	*out_buf	= 0;
	int	k,j;

	if (that == 0)
		return 0;

	in_buf = that->buf;
	if ((out_buf = (byte *) malloc ( sizeof (byte) * that->len)) == 0)
	{
		errorPrintf ("** ERROR (%s): couldn't allocate space\n", FUNCTION);
		return 0;
	}

	for (j=k=0; (unsigned) j < that->len; j++,k++)
	{
		// copy each byte
		out_buf[k] = in_buf[j];

		// throw away really low ASCII
		if (/*out_buf[k]>=0 &&*/ out_buf[k]<9) 
			k--;

		// for CR
		if (out_buf[k]==0x0D)
		{
			// if next is LF, then drop it
			if (((unsigned) j < that->len-1) && (in_buf[j+1]==0x0A))
				k--;
			else // turn it into a LF
				out_buf[k] = 0x0A;
		}
	}
	free (that->buf);
	that->buf = out_buf;
	that->len = k;
	return k;
#	undef	FUNCTION
}

	static void 
out_word(short w, FILE* fout)
{
#	define	FUNCTION	"out_word"
	short m = Swap16(w);
	if (fwrite(&m,2,1,fout) != 2)
	{
		errorPrintf("** ERROR (%s): Problem writing to file, %s\n", 
			FUNCTION, strerror(errno));
		return;
	}

#	undef	FUNCTION
}

	static void 
out_long(long d, FILE* fout)
{
#	define	FUNCTION	"out_long"
	long d1 = Swap32(d);
	if (fwrite(&d1,4,1,fout) != 4)
	{
		errorPrintf("** ERROR (%s): Problem writing to file, %s\n", 
			FUNCTION, strerror(errno));
		return;
	}

#	undef	FUNCTION
}

	static void 
Clear(tBuf *that) 
{
#	define	FUNCTION	"Clear"
	if (that != 0)
	{
		if (that->buf != 0)
			free (that->buf); 
		that->len = 6000;
		if ((that->buf = (byte *) malloc (sizeof (byte) * that->len)) == 0)
		{
			errorPrintf ("** ERROR (%s): couldn't allocate space\n", FUNCTION);
			return;
		}
	}
#	undef	FUNCTION
}

/*
 * Documents have the same name on the Palm and the Host so we'll just return
 * that.
 */

	char	*
DocHostName	(char	*nameOnPalm, 
		 char	*databaseName)
{
#	define	FUNCTION	"DocHostName"
	ignore ((int) databaseName);
	return databaseName;
#	undef	FUNCTION
}

