/* **********************************************************************
 *
 *   LED/SrcEdit is free software as specified in the GNU General public License
 *   listed in the file COPYING or http://www.gnu.org/licenses/gpl.txt
 *
 ********************************************************************** */

#ifndef __OBC__
#include <PalmOS.h>
#include <PalmCompatibility.h>
#include <SoundMgr.h>
#endif

#include "LedGlobals.h"

/****************************************************************************/

typedef enum {
    NoAction,
    PasteAction,
    DeleteAction
} Action;

typedef struct {
    Handle       handle;
    Word         length;
    Word         collapsedLength;
    CharPtr      text;
    SegmentKind  kind;
    Word         depth;
} TextChunk;

typedef struct {
    Word        chunkNumber;
    Word        offset;
} LineStart;

/****************************************************************************/

static char toLower[256];

Word gMaxLines;             // Holds the total number of lines
SegmentData gResultData;
LineStart *gLineStarts;
Word gTopChunk = 0;         // the index of the final, guardian chunk

TextChunk *gChunkList = NULL;// the chunks

DmOpenRef gTempDB;          // the (resource) database used for working storage

Word gSelectionLine;        // the line containing the start of the selection
Word gSelectionOffset;      // the offset WITHIN the line for the selection start
Word gSelectionEndLine;     // ditto (end)
Word gSelectionEndOffset;   // ditto (end)
Boolean gHasSelection;

Word gUndoSelectionLine;
Word gUndoSelectionOffset;
Word gUndoSelectionEndLine;
Word gUndoSelectionEndOffset;
Action gUndoAction;

Boolean gContinueExtend;

TextChunk *gUndoChunkList = NULL;
Word gUndoChunkCount = 0;
TextChunk *gClipChunkList = NULL;
Word gClipChunkCount = 0;

/****************************************************************************/

#define TEMP_DB_NAME "LED.tempDB"
#define TEMP_DB_TYPE 'DATA'
#define TEMP_DB_CREATOR 'LEDR'

#define MAX_CHUNKS 64
#define MAX_UNDO_CHUNKS 32
#define MAX_CLIP_CHUNKS 32
#define MAX_LINES 4096
#define MAX_CHUNK_LENGTH 4096
#define DOC_RECORD_LENGTH 4096

#define TextType 'Data'
#define UndoType 'Undo'
#define ClipType 'Clip'
#define TempType 'Temp'

/****************************************************************************/
#define TDEBUG

#ifdef TDEBUG
#define VALIDATE        validateChunks()
#define VALIDATE_LINES  validateLineStarts()
#else
#define VALIDATE
#define VALIDATE_LINES
#endif

/******************************************************************************
 * Function Prototypes for Exported Functions
 ******************************************************************************/
unsigned char* mem_find(unsigned char*t, int t_len, unsigned char *m, int m_len ) INTEXTCODE;
void error(CharPtr message) INTEXTCODE;
// Boolean insertChars(CharPtr cp, Word length, Word *startChunkP, Word *newOffsetP) INTEXTCODE;

/******************************************************************************
 * Function Prototypes for static Functions
 ******************************************************************************/
static void validateLineStarts() INTEXTCODE;
static Boolean copySelectionToList(TextChunk *chunkList, Word *chunkCount, ULong type) INTEXTCODE;
static void deleteListChunks(TextChunk *chunkList, Word chunkCount, ULong type) INTEXTCODE;
static void getChunkAndOffset(Word line, Word offset, Word *chunkIndexP, Word *chunkOffsetP) INTEXTCODE;
static void getEndChunkAndOffset(Word line, Word offset, Word *chunkIndexP, Word *chunkOffsetP) INTEXTCODE;
static void validateChunks() INTEXTCODE;
static Boolean copyFromText(CharPtr text, ULong type, TextChunk *chunkList,
	Word *chunkCountP, Word length, SegmentKind kind) INTEXTCODE;
static char *findStr(char *searchText, char *sourceText, char *moreSource, Boolean ignoreCase) INTEXTCODE;



void error(CharPtr message)
{
    FrmCustomAlert(TextErrorAlert, message, NULL, NULL);
//  int *p = (int *)3;
//  *p = 4;
}

Boolean initText()
{
    gTempDB = DmOpenDatabaseByTypeCreator(TEMP_DB_TYPE, TEMP_DB_CREATOR, dmModeReadWrite);
    if (gTempDB == NULL) {
        int err = DmCreateDatabase(0, TEMP_DB_NAME, TEMP_DB_CREATOR, TEMP_DB_TYPE, true);
        if (err == 0) {
            gTempDB = DmOpenDatabaseByTypeCreator(TEMP_DB_TYPE, TEMP_DB_CREATOR, dmModeReadWrite);
            if (gTempDB == 0) {
                error("second open failed");
                return false;
            }
        }
        else {
            error("Create failed");
            return false;
        }
    }
    else {
        int numResources = DmNumResources(gTempDB);
        for (int i = 0; i < numResources; i++) {
            DmRemoveResource(gTempDB, 0);
        }
    }

    Handle h = (Handle)MemHandleNew(sizeof(TextChunk) * MAX_CHUNKS);
    gChunkList = (TextChunk *)MemHandleLock(h);

    h = (Handle)MemHandleNew(sizeof(TextChunk) * MAX_UNDO_CHUNKS);
    gUndoChunkList = (TextChunk *)MemHandleLock(h);

    h = (Handle)MemHandleNew(sizeof(TextChunk) * MAX_CLIP_CHUNKS);
    gClipChunkList = (TextChunk *)MemHandleLock(h);

    h = (Handle)MemHandleNew(sizeof(LineStart) * MAX_LINES);
    gLineStarts = (LineStart *)MemHandleLock(h);

    gSelectionLine = 0;
    gSelectionOffset = 0;
    gSelectionEndLine = 0;
    gSelectionEndOffset = 0;
    gHasSelection = false;

    gUndoChunkCount = 0;
    gClipChunkCount = 0;

    gUndoAction = NoAction;
    gContinueExtend = false;

    for (int i = 0; i < 255; i++) {
        toLower[i] = i;
    }
    toLower[255] = 0;
    StrToLower(&toLower[1], &toLower[1]);
    toLower[255] = 255;

    return true;
}

void termText()
{
    if (gTempDB != 0) {
        if (gTopChunk > 0) {
            deleteListChunks(gChunkList, gTopChunk + 1, TextType);
        }
        deleteListChunks(gUndoChunkList, gUndoChunkCount, UndoType);
        // Apparently there are other resources left in this db, blow them off here
        // but I'd rather understand what they are.
        int numResources = DmNumResources(gTempDB);
        for (int i = 0; i < numResources; i++) {
            DmRemoveResource(gTempDB, 0);
        }
        DmCloseDatabase(gTempDB);
    }

    LocalID lid = DmFindDatabase(0, TEMP_DB_NAME);
    if (lid) {
        DmDeleteDatabase(0, lid);
    }

    if (gChunkList) {
        Handle h = (Handle)MemPtrRecoverHandle(gChunkList);
        MemHandleUnlock(h);
        MemHandleFree(h);
    }
    if (gUndoChunkList) {
        Handle h = (Handle)MemPtrRecoverHandle(gUndoChunkList);
        MemHandleUnlock(h);
        MemHandleFree(h);
    }
    if (gClipChunkList) {
        Handle h = (Handle)MemPtrRecoverHandle(gClipChunkList);
        MemHandleUnlock(h);
        MemHandleFree(h);
    }
    if (gLineStarts) {
        Handle h = (Handle)MemPtrRecoverHandle(gLineStarts);
        MemHandleUnlock(h);
        MemHandleFree(h);
    }
}

SegmentData* getLineContents(Word lineNumber)
{
    TEXT_ASSERT(lineNumber <= gMaxLines, "getLineContents, Line number beyond max");

    MemHandle ErrorStrH;
    char* ErrorStr;

    int chunkIndex = gLineStarts[lineNumber].chunkNumber;
    TextChunk *chunkP = &gChunkList[chunkIndex];
    TextChunk *nextChunk;
    if(lineNumber==gMaxLines) {
        nextChunk = &gChunkList[gTopChunk];
    }
    else {
        nextChunk = &gChunkList[gLineStarts[lineNumber + 1].chunkNumber];
    }
    Segment *segP = &gResultData.segment[0];
    if (chunkP == nextChunk) {
        // degenerate case
        gResultData.count = 1;
        segP->kind = chunkP->kind;
        segP->text = chunkP->text + gLineStarts[lineNumber].offset;
/*
        if (gLineStarts[lineNumber + 1].offset == gLineStarts[lineNumber].offset)
            segP->length = 0;       // wackiness that comes from having a fake eol at the end
        else
        */
            segP->length = gLineStarts[lineNumber + 1].offset
                                                - gLineStarts[lineNumber].offset - 1;   // skip the <CR>
    }
    else {
        Word segCount = 0;
        Word offset = gLineStarts[lineNumber].offset;
        while (chunkP != nextChunk) {
            segP->kind = chunkP->kind;
            segP->text = chunkP->text + offset;
            segP->length = chunkP->length - offset;

            TEXT_ASSERT((chunkP->length > 0), "empty chunk");
            TEXT_ASSERT((segP->length > 0), "empty segment");

            offset = 0; // after the initial chunk, we use all the contents of the chunks
            segCount++;
            TEXT_ASSERT(segCount < MAX_SEGMENT, "Too many segments per line");
            segP++;
            chunkIndex++;
            chunkP = &gChunkList[chunkIndex];
        }
        /*
            if the next line begins at the start of the chunk
            there's no point including anything from it. But
            that implies there was a <CR> at the end of the
            last segment which we wish to exclude.
        */
        if (gLineStarts[lineNumber + 1].offset == 0) {
            if (gChunkList[chunkIndex - 1].text[gChunkList[chunkIndex - 1].length - 1] == '\n') {
                gResultData.segment[segCount - 1].length--;
            }
        }
        else { // take all of the chunk up to the start of the next line, allowing for the <CR>
            segP->kind = chunkP->kind;
            segP->text = chunkP->text;
            segP->length = gLineStarts[lineNumber + 1].offset - 1;
            segCount++;
            TEXT_ASSERT(segCount < MAX_SEGMENT, "Too many segments per line");
        }
        gResultData.count = segCount;
    }

    // Long Line guardian
    // #846827 (S043) Long Line Hang
    if (!(segP->length <= MAX_CHARS_PER_LINE))
    {
        ErrorStrH = MemHandleNew(100);
        ErrorStr = (char*)MemHandleLock(ErrorStrH);
        StrPrintF(ErrorStr, "Too many chars per line! Line number: %i", lineNumber+1);

        TEXT_ASSERT((segP->length <= MAX_CHARS_PER_LINE), ErrorStr);

        MemHandleUnlock(ErrorStrH);
        MemHandleFree(ErrorStrH);
    }
    return &gResultData;
}

static void buildLineStartsData()
{
    VALIDATE;

    Word chunkIndex = 0;
    Word offset = 0;
    TextChunk *chunkP = &gChunkList[0];
    CharPtr cp = chunkP->text;
    Word startChunk = chunkIndex;
    Word startOffset = offset;
    int lineNumber = 0;
    while (chunkIndex != gTopChunk) {
        if (gChunkList[chunkIndex].kind != CollapsedText) {
            while (*cp != '\n') {           // there is always one at the end, so this will stop
                cp++;
                offset++;
                if (offset == chunkP->length) {
                    chunkIndex++;
                    chunkP = &gChunkList[chunkIndex];
                    while (chunkP->kind == CollapsedText) {
                        chunkIndex++;
                        chunkP = &gChunkList[chunkIndex];
                    }
                    cp = chunkP->text;
                    offset = 0;
                }
            }
            gLineStarts[lineNumber].chunkNumber = startChunk;
            gLineStarts[lineNumber].offset = startOffset;
            TEXT_ASSERT((startOffset < gChunkList[startChunk].length), "bad offset");
            lineNumber++;
            TEXT_ASSERT(lineNumber < MAX_LINES, "Too many lines");
            if (chunkIndex != gTopChunk) {
                offset++;           // step past the <CR> to start the next line
                cp++;
                if (offset == chunkP->length) {
                    chunkIndex++;
                    chunkP = &gChunkList[chunkIndex];
                    cp = chunkP->text;
                    offset = 0;
                }
            }
            startChunk = chunkIndex;
            startOffset = offset;
        }
        else {
            chunkIndex++;
            chunkP = &gChunkList[chunkIndex];
            cp = chunkP->text;
            offset = 0;
        }
    }
    /*
        we add a guardian empty line to make 'getLineContents' easier...
    */
    gLineStarts[lineNumber].chunkNumber = gTopChunk;
    gLineStarts[lineNumber].offset = 0;
    gMaxLines = lineNumber;                     // ...but don't tell

    VALIDATE_LINES;
}

void validateLineStarts()
{
    for (Word i = 0; i < gMaxLines; i++) {
        TEXT_ASSERT((gLineStarts[i].chunkNumber <= gTopChunk), "invalid chunk from lineStart");
        TEXT_ASSERT((gLineStarts[i].offset < gChunkList[gLineStarts[i].chunkNumber].length), "invalid offset from lineStart");
        TEXT_ASSERT(((gLineStarts[i].offset <= gLineStarts[i + 1].offset) || (gLineStarts[i].chunkNumber < gLineStarts[i + 1].chunkNumber)),
                        "out-of-order lineStart");
    }
}

unsigned char* mem_find(unsigned char*t, int t_len, unsigned char *m, int m_len )
// replacement for strstr() which deals with 0's in the data
{
    int i;
    for ( i = t_len - m_len + 1; i > 0; --i, ++t )
        if ( *t == *m && !MemCmp( t, m, m_len ))
            return t;
    return 0;
}

/*
static void put_byte(unsigned char *b, int *pos, unsigned char c, Boolean *space )
 // Put a byte into a buffer.
{
    if ( *space ) {
        *space = false;

        //  There is an outstanding space char: see if we can squeeze it
        //  in with an ASCII char.

        if ((c >= 0x40)&&(c <= 0x7F)) {
            b[*pos++] = c ^ 0x80;
            return;
        }
        b[*pos++] = ' ';    // couldn't squeeze it in
    } else if ( c == ' ' ) {
        *space = true;
        return;
    }

    if ( c >= 1 && c <= 8 || c >= 0x80 )
        b[*pos++] = '\1';

    b[*pos++] = c;
}
*/

#define BUFFER_SIZE 6000    /* why this is this value, I don't know */
#define COUNT_BITS  3   /* ditto */
#define DISP_BITS   11  /* ditto */

// #define NEW_BUFFER(b)    (b) = (char*)MemPtrNew(BUFFER_SIZE)

static int compress(unsigned char *b, int inLength)
 // Replace the given buffer with a compressed version of itself.
{
/*
    int i, j;
    Boolean space = false;

    unsigned char *buf_orig;
    unsigned char *p;   // walking test hit; works up on successive matches
    unsigned char *p_prev;
    unsigned char *head;    // current test string
    unsigned char *tail;    // 1 past the current test buffer //
    unsigned char *end;     // 1 past the end of the input buffer //

    p = p_prev = head = buf_orig = b;
    tail = head + 1;
    end = b + inLength;

    MemPtrResize(b, BUFFER_SIZE);
    int outLength = 0;

    // loop, absorbing one more char from the input buffer on each pass
    while ( head != end ) {
        // establish where the scan can begin
        if ( head - p_prev > (( 1 << DISP_BITS )-1) )
            p_prev = head - (( 1 << DISP_BITS )-1);

        // scan in the previous data for a match
        p = mem_find( p_prev, tail - p_prev, head, tail - head );

        // on a mismatch or end of buffer, issued codes
        if ( !p || p == head || tail - head > ( 1 << COUNT_BITS ) + 2
            || tail == end
        ) {
            // issued the codes
            // first, check for short runs
            if ( tail - head < 4 )
                put_byte( b, &outLength, *head++, &space );
            else {
                unsigned dist = head - p_prev;
                unsigned compound = (dist << COUNT_BITS)
                    + tail - head - 4;

                if ( dist >= ( 1 << DISP_BITS ) ||tail - head - 4 > 7)
                    error("Compress error: dist overflow");

                // for longer runs, issue a run-code
                // issue space char if required
                if ( space )
                    b[outLength++] = ' ', space = false;

                b[outLength++] = 0x80 + ( compound >> 8 );
                b[outLength++] = compound & 0xFF;
                head = tail - 1;    // and start again
            }
            p_prev = buf_orig;      // start search again
        } else
            p_prev = p;             // got a match

        // when we get to the end of the buffer, don't inc past the
        // end; this forces the residue chars out one at a time
        if ( tail != end )
            ++tail;
    }
    // free( buf_orig );

    if ( space )
        b[outLength++] = ' ';   // add left-over space

    // final scan to merge consecutive high chars together
    for (i = j = 0; (Int32)i < outLength; ++i, ++j ) {
        b[j] = b[i];

        // skip run-length codes
        if ( b[j] >= 0x80 && b[j] < 0xC0 )
            b[++j] = b[++i];

        // if we hit a high char marker, look ahead for another
        else if ( b[j] == '\1' ) {
            b[j + 1] = b[i + 1];
            while ((Int32)(i + 2) < outLength && b[i + 2] == 1 && b[j] < 8) {
                b[j]++;
                b[j+ b[j]] = b[i + 3];
                i += 2;
            }
            j += b[j];
            ++i;
        }
    }
    return j;
*/
return 0;
}

static int decompress(unsigned char *inPtr, int inLength, unsigned char *outPtr)
{
    int in = 0;
    int out = 0;

    while (in < inLength) {
        int ch = inPtr[in++];

        if ((ch > 0) && (ch < 9)) { // n chars echoed from in to out
            while (ch-- > 0) {
                outPtr[out++] = inPtr[in++];
            }
        }
        else {
            if (ch < 0x80) {            // literal
                outPtr[out++] = ch;
            }
            else {
                if (ch >= 0xc0) {       // space preceded literal, high bit off
                    outPtr[out++] = ' ';
                    outPtr[out++] = ch & 0x7F;
                }
                else {                  // duplication - <0b10><11 bit back displacement><3 bit length (plus 3)>
                    int orig;
                    int count;
                    ch <<= 8;
                    ch |= inPtr[in++];
                    orig = out - ((ch & 0x3fff) >> 3);
                    count = (ch & 0x7) + 3;
                    while (count--) {
                        outPtr[out++] = outPtr[orig++];
                    }
                }
            }
        }
    }
    return out;
}

typedef struct {
    int     flag;           // 0x0001 if data is uncompressed, 0x0002 if compressed
    int     dontKnow;      // Unknown... set to 0 upon creation
    long    size;           // uncompressed total size
    int     count;          // # records
    int     recSize;        // uncompressed size per record (Usually 4096)
    long    position;       // currently viewed position in the document (NOT IN USE)
} DOCHeader;

static Boolean buildEmptyChunk(int chunkIndex)
{
    gChunkList[chunkIndex].length = 1;
    Handle localH = (Handle)DmNewResource(gTempDB, TextType, chunkIndex, 1);
    if (localH == NULL) {
        error("Couldn't allocate temp space");
        return false;
    }
    CharPtr localP = (CharPtr)MemHandleLock(localH);
    gChunkList[chunkIndex].text = localP;
    gChunkList[chunkIndex].handle = localH;
    gChunkList[chunkIndex].kind = PlainText;
    gChunkList[chunkIndex].depth = 0;
    DmWrite(localP, 0, "\n", 1);
    return true;
}

Boolean newDocFile(CharPtr dbName)
{
  UInt att;
    LocalID dbID = DmFindDatabase(0, dbName);
    if (dbID != 0) {
        error("Duplicate DB name"); // XXX should delete existing with dialog confirmation thing
        return false;
    }
    int err = DmCreateDatabase(0, dbName, 'REAd', 'TEXt', false);
    if (err != 0) {
        error("Failed to create new DB");
        return false;
    }

    dbID = DmFindDatabase(0, dbName);
    DmDatabaseInfo(0, dbID, NULL, &att,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL);
    att |= dmHdrAttrBackup;
    DmSetDatabaseInfo(0,dbID,NULL,&att,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL);

    if (gTopChunk > 0) {
        deleteListChunks(gChunkList, gTopChunk + 1, TextType);
    }
    deleteListChunks(gUndoChunkList, gUndoChunkCount, UndoType);
    gUndoChunkCount = 0;

    gTopChunk = 1;
    gMaxLines = 1;
    buildEmptyChunk(0);
    buildEmptyChunk(1);

    buildLineStartsData();

    gUndoAction = NoAction;
    gContinueExtend = false;

    markSelection(0, 0, 0, 0);

    return true;
}

Boolean loadDocFile(CharPtr dbName)
{
    MemHandle localH;
    CharPtr localP;
    Boolean result = false;
    UInt16 attr;

    LocalID dbID = DmFindDatabase(0, dbName);
    if (dbID == 0) {
//      error("Couldn't open DOC file");        <-- not necessarily fatal
        return false;
    }
    DmOpenRef docDB = DmOpenDatabase(0, dbID, dmModeReadOnly);
    if (docDB == 0) {
        error("Couldn't open DOC file");
        return false;
    }
    DmDatabaseInfo(0,dbID,NULL,&attr,0,0,0,0,0,0,0,0,0);

    MemHandle headerH = (Handle)DmGetRecord(docDB, 0);
    if (headerH == NULL) {
        error("Couldn't get record #0");
        DmCloseDatabase(docDB);
        return false;
    }
    DOCHeader *hdr = (DOCHeader *)MemHandleLock(headerH);

    // #621840 (S032) SrcEdit document categories
    UInt16 recAttr;
    DmRecordInfo(docDB, 0, &recAttr, NULL, NULL);

    if (gTopChunk > 0) {
        deleteListChunks(gChunkList, gTopChunk + 1, TextType);
        gTopChunk = 0;
    }
    deleteListChunks(gUndoChunkList, gUndoChunkCount, UndoType);
    gUndoChunkCount = 0;

    unsigned char *decompressionBuffer = NULL;
    if (hdr->flag == 0x2) {
        decompressionBuffer = (unsigned char *)MemPtrNew(6144);     // I don't know the origin of this number, but it seems to work
    }
//  DbgBreak();
    for (int i = 1; i <= hdr->count; i++) {
        Handle docH = (Handle)DmGetRecord(docDB, i);
        if (docH == NULL) {
            error("Couldn't get record");
            goto exit;      // it is memory leaky, isn't it ?
        }

        CharPtr docP = (CharPtr)MemHandleLock(docH);

        int size = MemHandleSize(docH);
        if (hdr->flag == 0x2) {
            size = decompress((unsigned char *)docP, size, decompressionBuffer);
            localH = (Handle)DmNewResource(gTempDB, TextType, gTopChunk, size);
            if (localH == NULL) {
                error("Couldn't allocate temp space");
                goto exit;      // should free what was allocated first...
            }
            localP = (CharPtr)MemHandleLock(localH);
            DmWrite(localP, 0, decompressionBuffer, size);
        } else {
            localH = (Handle)DmNewResource(gTempDB, TextType, gTopChunk, size);
            if (localH == NULL) {
                error("Couldn't allocate temp space");
                goto exit;
            }
            localP = (CharPtr)MemHandleLock(localH);
            DmWrite(localP, 0, docP, size);
        }
        gChunkList[gTopChunk].length = size;
        gChunkList[gTopChunk].text = localP;
        gChunkList[gTopChunk].handle = localH;
        gChunkList[gTopChunk].kind = PlainText;
        gChunkList[gTopChunk].depth = 0;
        gTopChunk++;
        TEXT_ASSERT(gTopChunk < MAX_CHUNKS, "Too many chunks");

        MemHandleUnlock(docH);
        DmReleaseRecord(docDB, i, false);
    }
    {
        // make sure there's a '\n' at the end of the last chunk, it's
        // just easier that way.
        TextChunk *chunkP = &gChunkList[gTopChunk - 1];
        if (chunkP->text[chunkP->length - 1] != '\n') {
            MemHandleUnlock(chunkP->handle);
            Err err = MemHandleResize(chunkP->handle, chunkP->length + 1);
            if (err) {
                error("Couldn't grow final handle");
                goto exit;      // it is memory leaky, isn't it ?
            }
            chunkP->text = (char *)MemHandleLock(chunkP->handle);
            DmWrite(chunkP->text, chunkP->length, "\n", 1);
            chunkP->length++;
        }
    }


    // when we're all done reading the docDB, add a final
    // guardian chunk with a bogus '\n'
    result = buildEmptyChunk(gTopChunk);

    // #621840 (S032) SrcEdit document categories
    if ((attr & dmHdrAttrReadOnly) == false){
        DmSetRecordInfo(docDB, 0, &recAttr, NULL);
        DmSetDatabaseInfo(0,dbID,NULL,&attr, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);  // (S055) File Info and Attributes
    }
exit:
    if (hdr->flag == 0x2)
        MemPtrFree(decompressionBuffer);
    MemHandleUnlock(headerH);
    DmReleaseRecord(docDB, 0, true);
    DmCloseDatabase(docDB);

    if (result)
        buildLineStartsData();

    gUndoAction = NoAction;
    gContinueExtend = false;
    if (gHighlightMode != NOMODE)
        setHighlightMode(dbName);

    return result;
}

Boolean insertChars(CharPtr cp, Word length, Word *startChunkP, Word *newOffsetP)
{
    Err err;

    // calculate the per-chunk offsets for selection start
    Word startOffset;
    Word startChunk;
    getChunkAndOffset(gSelectionLine, gSelectionOffset, &startChunk, &startOffset);

    if (startChunk == gTopChunk) {  // don't want to allow this - it can
                                    // only occur for an insert right at the
                                    // end.
        startChunk--;
        startOffset = gChunkList[startChunk].length;
    }

    TextChunk *chunkP = &gChunkList[startChunk];

    if (chunkP->kind == CollapsedText) {
        // might need to create a new chunk, but see if we can extend the previous one
        if ((startChunk == 0) || (gChunkList[startChunk - 1].kind == CollapsedText)) {

        }
        else {
            startChunk--;
            startOffset = gChunkList[startChunk].length;
        }
    }

    MemHandleUnlock(chunkP->handle);
    err = MemHandleResize(chunkP->handle, chunkP->length + length);

    if (err) {
        error("Couldn't resize handle for insert");
        return false;
    }
    CharPtr p = (CharPtr)MemHandleLock(chunkP->handle);
    chunkP->text = p;

    DmWrite(p, startOffset + length, p + startOffset, chunkP->length - startOffset);
    DmWrite(p, startOffset, cp, length);

    chunkP->length += length;

    *startChunkP = startChunk;
    *newOffsetP = startOffset;

    VALIDATE;

    return true;
}

Boolean insertCharacters(CharPtr cp, Word length, Boolean updateSelection)
{
    Word startChunk;
    Word newOffset;

    if (gHasSelection) {
        if (!isEmptySelection()) {
            if (!deleteSelection())     // this moves any current non-empty selection to the undo paste contents
                return false;
        }
        else {
            if (gContinueExtend) {
                gUndoSelectionEndOffset += length;
            }
            else {
                gUndoAction = DeleteAction;
                gUndoSelectionLine = gSelectionLine;
                gUndoSelectionOffset = gSelectionOffset;
                gUndoSelectionEndLine = gSelectionLine;
                gUndoSelectionEndOffset = gSelectionOffset + length;
            }
        }

        if (!insertChars(cp, length, &startChunk, &newOffset)) {
            return false;
        }
        if (updateSelection) {
            // move the selection offset to the end of the text just inserted
            gSelectionOffset += length;
            gSelectionEndOffset = gSelectionOffset;
        }
        // all lines in this chunk after the selected line
        // now start 'length' bytes further away
        Word line = gSelectionLine + 1;
        while ((gLineStarts[line].chunkNumber == startChunk) && (line <= gMaxLines)) {
            gLineStarts[line].offset += length;
            line++;
        }
        // We need to rebuild the line starts! /jw
        buildLineStartsData();
        // VALIDATE_LINES; // Don't need this! It's done in 'buildLineStarts()'

        gContinueExtend = true;
    }
    return true;
}

Boolean insertNewLine()
{
    Word startChunk;
    Word newOffset;

    if (!deleteSelection()) {   // this moves any current non-empty selection to the undo paste contents
        return false;
    }

    if (!insertChars("\n", 1, &startChunk, &newOffset)) { // returns the chunk/offset into which the character was inserted
        return false;
    }

    if ((gMaxLines + 1) >= MAX_LINES) {
        error("Too many lines after insert NL");
        return false;
    }

    // all lines shuffle down to make room for the new line
    MemMove(gLineStarts + (gSelectionLine + 2),
            gLineStarts + (gSelectionLine + 1),
            (gMaxLines - gSelectionLine) * sizeof(LineStart));  // remember we have a line beyond gMaxLines

    // set the new line info (gSelectionOffset has already been pushed forward to follow
    // the character just inserted)
    if (gChunkList[startChunk].length == (newOffset + 1)) {

        VALIDATE_LINES;
        // this can only happen when inserting at the very end - where a line
        // has already been imputed - so leave immediately
        return true;

    }
    else {
        gLineStarts[gSelectionLine + 1].chunkNumber = startChunk;
        gLineStarts[gSelectionLine + 1].offset = newOffset + 1;     // starts after the newly inserted <CR>
    }

    // step the selection to the start of the new line
    gSelectionLine++;
    gSelectionOffset = 0;
    gSelectionEndLine = gSelectionLine;
    gSelectionEndOffset = gSelectionOffset;


    // all the other lines in this chunk begin 1 character further down
    Word line = gSelectionLine + 1;
    gMaxLines++;
    while ((gLineStarts[line].chunkNumber == startChunk) && (line <= gMaxLines)) {
        gLineStarts[line].offset++;
        line++;
    }
    // We need to rebuild the line starts! /jw
    buildLineStartsData();
    // VALIDATE; Don't need this! It's done in 'buildLineStarts()'
    // VALIDATE_LINES; Don't need this! It's done in 'buildLineStarts()'

    return true;
}

void unmarkSelection()
{
    gHasSelection = false;
}

void markSelection(Word fromLine, Word fromOffset, Word toLine, Word toOffset)
{
    TEXT_ASSERT(fromLine <= gMaxLines, "markSelection fromLine, Line number beyond max");
    TEXT_ASSERT(toLine <= gMaxLines, "markSelection toLine, Line number beyond max");
    TEXT_ASSERT(fromLine <= toLine, "Negative selection range");
    TEXT_ASSERT((fromLine < toLine) || (fromOffset <= toOffset), "Negative selection range");

    gSelectionLine = fromLine;
    gSelectionOffset = fromOffset;
    gSelectionEndLine = toLine;
    gSelectionEndOffset = toOffset;

    gHasSelection = true;
    gContinueExtend = false;
}

void getSelection(Word *fromLine, Word *fromOffset, Word *toLine, Word *toOffset)
{
    *fromLine = gSelectionLine;
    *fromOffset = gSelectionOffset;
    *toLine = gSelectionEndLine;
    *toOffset = gSelectionEndOffset;
}
/*
    removes 'length' characters from the given chunk at 'offset'
    assumes undo contents & lineStarts data will be adjusted accordingly
*/
static Boolean deleteFromChunk(TextChunk *chunkP, Word offset, Word length)
{
// even deleting all of a chunk is bad here
    TEXT_ASSERT((length < chunkP->length), "Delete all or more of chunk");

    Err err;
    Word endDeleteOffset = offset + length;
    DmWrite(chunkP->text, offset, chunkP->text + endDeleteOffset, chunkP->length - endDeleteOffset);

    MemHandleUnlock(chunkP->handle);
    chunkP->length -= length;

    err = MemHandleResize(chunkP->handle, chunkP->length);
    if (err) {
        error("Couldn't resize handle for delete");
        return false;
    }
    CharPtr p = (CharPtr)MemHandleLock(chunkP->handle);
    chunkP->text = p;

    return true;
}

Boolean isEmptySelection()
{
    return ((gSelectionLine == gSelectionEndLine) && (gSelectionOffset == gSelectionEndOffset));
}

Boolean hasSelection()
{
    return gHasSelection;
}

static int getLineLength(Word line)
{
    Word chunkIndex = gLineStarts[line].chunkNumber;
    if (gLineStarts[line + 1].chunkNumber == chunkIndex)
        return gLineStarts[line + 1].offset - gLineStarts[line].offset;
    else {
        int result = gChunkList[chunkIndex].length - gLineStarts[line].offset;
        chunkIndex++;
        while (chunkIndex != gLineStarts[line + 1].chunkNumber) {
            // weird icky case where the intervening chunk contains just the end-of-line
            // just prior to the end of document (the next line is the bogus sentinel)
            if ((gChunkList[chunkIndex].length == 1)
                    && (gChunkList[chunkIndex].text[0] == '\n'))
                break;
            result += gChunkList[chunkIndex].length;
            chunkIndex++;
        }
        result += gLineStarts[line + 1].offset;
        return result;
    }
}

static void shuffleChunks(Word startChunk, Word newLocation)
{
    long i;
    Handle h;

    Word count = (gTopChunk - startChunk + 1);      // remember to include the topChunk itself
    // have to be careful of the direction when we re-assign id's
    // since we may be over-writing id's we need to access
    if (startChunk < newLocation) {
        // moving from lower id to higher id
        for (i = (long)gTopChunk; i >= (long)startChunk; i--)
        {
            int index = DmFindResource(gTempDB, TextType, (Word)i, NULL);
            TEXT_ASSERT((index != -1), "Failed to find resource 1");
            h = (Handle)DmGetResourceIndex(gTempDB, index);
            TEXT_ASSERT((h == gChunkList[i].handle), "bad prior handle");

            UInt newID = newLocation + (i - startChunk);
            Err err = DmSetResourceInfo(gTempDB, index, NULL, &newID);
            TEXT_ASSERT((err == 0), "Error on set info");
            DmResourceInfo(gTempDB, index, NULL, &newID, NULL);
            TEXT_ASSERT((newID == (newLocation + (i - startChunk))), "bad setinfo");
            h = (Handle)DmGetResourceIndex(gTempDB, index);
            TEXT_ASSERT((h == gChunkList[i].handle), "bad set handle");
        }
    }
    else {
        for (i = (long)startChunk; i <= (long)gTopChunk; i++)
        {
            Int16 index = DmFindResource(gTempDB, TextType, i, NULL);
            TEXT_ASSERT((index != -1), "Failed to find resource 2");
            h = (Handle)DmGetResourceIndex(gTempDB, index);
            TEXT_ASSERT((h == gChunkList[i].handle), "bad prior handle");

            UInt newID = newLocation + (i - startChunk);
            Err err = DmSetResourceInfo(gTempDB, index, NULL, &newID);
            TEXT_ASSERT((err == 0), "Error on set info");
            DmResourceInfo(gTempDB, index, NULL, &newID, NULL);
            TEXT_ASSERT((newID == (newLocation + (i - startChunk))), "bad setinfo");

            TEXT_ASSERT((index == (signed)DmFindResource(gTempDB, TextType, newID, NULL)), "bad index set");

            h = (Handle)DmGetResourceIndex(gTempDB, index);
            TEXT_ASSERT((h == gChunkList[i].handle), "bad set handle");
        }
    }

    MemMove(gChunkList + newLocation, gChunkList + startChunk, count * sizeof(TextChunk));

    // partial validation on the newly modified chunks:

    for (i = 0; i < (long)count; i++) {
        int index = DmFindResource(gTempDB, TextType, newLocation + i, NULL);
        TEXT_ASSERT((index != -1), "Failed to find resource");
        ULong type;
        UInt id;
        DmResourceInfo(gTempDB, index, &type, &id, NULL);
        TEXT_ASSERT((type == TextType), "bad type");
        TEXT_ASSERT((id == (newLocation + i)), "bad id");
        h = (Handle)DmGetResourceIndex(gTempDB, index);
        TEXT_ASSERT((h == gChunkList[newLocation + i].handle), "Mismatched handle");
        if (gChunkList[newLocation + i].kind == PlainText)
            TEXT_ASSERT( (MemHandleSize(h) == gChunkList[newLocation + i].length)  , "Wrong plain handle length");
        else
            if (gChunkList[newLocation + i].kind == CollapsedText)
                TEXT_ASSERT( (MemHandleSize(h) == gChunkList[newLocation + i].collapsedLength)  , "Wrong collapsed handle length");
            else
                TEXT_ASSERT( (false)  , "Bad chunk kind");
    }

}

static void validateChunks()
{
    for (Word i = 0; i <= gTopChunk; i++) {
        int index = DmFindResource(gTempDB, TextType, i, NULL);
        TEXT_ASSERT((index != -1), "Failed to find resource");
        ULong type;
        UInt id;
        DmResourceInfo(gTempDB, index, &type, &id, NULL);
        TEXT_ASSERT((type == TextType), "bad type");
        TEXT_ASSERT((id == i), "bad id");
        Handle h = (Handle)DmGetResourceIndex(gTempDB, index);
        TEXT_ASSERT((h == gChunkList[i].handle), "Mismatched handle");
        if (gChunkList[i].kind == PlainText)
            TEXT_ASSERT( (MemHandleSize(h) == gChunkList[i].length)  , "Wrong plain handle length");
        else
            if (gChunkList[i].kind == CollapsedText)
                TEXT_ASSERT( (MemHandleSize(h) == gChunkList[i].collapsedLength)  , "Wrong collapsed handle length");
            else
                TEXT_ASSERT( (false)  , "Bad chunk kind");
    }
}

Boolean backspace()
{
    /*
        // (S033xtra) Check if capsLock is on, if so change to autoCaps and bail out.
        // THIS IS NOT NORMAL PALM BEHAVIOUR. THEREFOR NOT NOT IMPLEMENTED! /jw
        Boolean capsLock;
        GrfGetState(&capsLock, NULL, NULL, NULL);
        if (capsLock){
            GrfSetState(false, false, true);
            return true;
        }
    */

    if (gHasSelection) {
        if (!isEmptySelection()) {
            deleteSelection();
            return true;  // To avoid troubble....
        }
        else { // delete the character before the insertion point

            if ((gSelectionOffset == 0) && (gSelectionLine == 0)) return true;  // can't go there

            Word startOffset;
            Word startChunk;
            // get the end chunk/offset because we actually care about the character BEFORE
            // the insertion point
            getEndChunkAndOffset(gSelectionLine, gSelectionOffset, &startChunk, &startOffset);

            TextChunk *chunkP = &gChunkList[startChunk];

            // catch all the really bad cases first and re-build the lineStarts from
            // scratch. It may be that some of these are more frequent and could be
            // handled less dramatically.

            // watching for :
            // - backspacing through start of line
            // - backspacing through start of chunk
            // - deleting the only character in a chunk

            char deletedCh;

            if ((startOffset == 0) || (chunkP->length == 1)) {
                if (startOffset == 0) {
                    TEXT_ASSERT((startChunk > 0), "backspace into void?");
                    startChunk--;
                    chunkP = &gChunkList[startChunk];
                    startOffset = chunkP->length;
                }
                if (chunkP->kind == CollapsedText) {
                    // set it up as a selection instead
                    if (gSelectionOffset == 0) {
                        gSelectionLine--;
                        gSelectionOffset = getLineLength(gSelectionLine) - 1;
                    }
                    else
                        gSelectionOffset--;
                    return deleteSelection();
                }
                else {
                    deletedCh = chunkP->text[startOffset - 1];
                    if (gSelectionOffset == 0) {
                        gSelectionLine--;
                        gSelectionOffset = getLineLength(gSelectionLine) - 1;   // get this before they all change
                    }
                    else {
                        gSelectionOffset--;
                        gSelectionEndOffset = gSelectionOffset;
                    }
                    gSelectionEndLine = gSelectionLine;
                    gSelectionEndOffset = gSelectionOffset;
                    if (chunkP->length == 1) {  // nuts, a tiny chunk is getting totally blown away
                        MemHandleUnlock(chunkP->handle);
                        int index = DmFindResource(gTempDB, TextType, startChunk, NULL);
                        TEXT_ASSERT((index != -1), "Failed to find resource");
                        TEXT_ASSERT((DmGetResourceIndex(gTempDB, index) == chunkP->handle), "Mismatched handle");
                        DmRemoveResource(gTempDB, index);
                        chunkP->handle = NULL;
                        shuffleChunks(startChunk + 1, startChunk);
                        gTopChunk--;
                    }
                    else {
                        if (!deleteFromChunk(chunkP, startOffset - 1, 1))
                            return false;
                    }
                }
                buildLineStartsData();
            }
            else {
                // phew, slightly more normal; but still watch for
                // backspacing through start of line
                deletedCh = chunkP->text[startOffset - 1];
                if (!deleteFromChunk(chunkP, startOffset - 1, 1))
                    return false;
                if (gSelectionOffset == 0) {
                    gSelectionLine--;
                    gSelectionOffset = getLineLength(gSelectionLine) - 1;   // get this before they all change
                    gSelectionEndLine = gSelectionLine;
                    gSelectionEndOffset = gSelectionOffset;
                    buildLineStartsData();      // COULD BE BETTER - we've just deleted the line start for
                                                // the original gSelectionLine; so bring all the others down
                                                // in the linestarts array and subtract 1. All chunks remain.
                }
                else {
                    if (gLineStarts[gSelectionLine].offset == chunkP->length) { // deleted first character in line and it
                                                                                // was at the very end of the chunk
                        gLineStarts[gSelectionLine].chunkNumber++;
                        gLineStarts[gSelectionLine].offset = 0;
                    }
                    Word line = gSelectionLine + 1;
                    while ((gLineStarts[line].chunkNumber == startChunk)
                                && (line < gMaxLines)) {
                        gLineStarts[line].offset--;
                        line++;
                    }
                    gSelectionOffset--;
                    gSelectionEndOffset = gSelectionOffset;
                    // We need to rebuild the line starts! /jw
                    buildLineStartsData();

                    // VALIDATE; Don't need this! It's done in 'buildLineStarts()'
                    // VALIDATE_LINES; Don't need this! It's done in 'buildLineStarts()'
                }
            }
            deleteListChunks(gUndoChunkList, gUndoChunkCount, UndoType);
            gUndoChunkCount = 0;
            if (!copyFromText(&deletedCh, UndoType, gUndoChunkList, &gUndoChunkCount, 1, PlainText))
                return false;
            gUndoAction = PasteAction;
            // set an insertion point for the undo
            gUndoSelectionLine = gSelectionLine;
            gUndoSelectionOffset = gSelectionOffset;
            gUndoSelectionEndLine = gSelectionLine;
            gUndoSelectionEndOffset = gSelectionOffset;
        }
    }
    gContinueExtend = false;
    return true;
}

static Boolean deleteRange(Word chunkIndex, Word startOffset, Word endChunkIndex, Word endOffset, Boolean rebuildLineStarts)
{
    Word length = endOffset - startOffset;  // only valid for same chunk, but we use it
                                            // first under that condition below

    TextChunk *chunkP = &gChunkList[chunkIndex];
    if ((chunkIndex == endChunkIndex) && (length < chunkP->length)) {   // watch for deleting entire chunk

        if (!deleteFromChunk(chunkP, startOffset, length))
            return false;
        // optimize for deleting from a single line, adjust all the other lines
        // in this chunk by the deleted amount
        if (rebuildLineStarts) {    // this is only true for deleting the selection,
                                    //  hardly a clean API though
            if (gSelectionLine == gSelectionEndLine) {
                Word line = gSelectionLine + 1;
                while ((gLineStarts[line].chunkNumber == chunkIndex)&& (line < gMaxLines)) {
                    gLineStarts[line].offset -= length;
                    line++;
                }

                // We need to rebuild the line starts! /jw
                buildLineStartsData();

                // VALIDATE;  Don't need this! It's done in 'buildLineStarts()'
                // VALIDATE_LINES; Don't need this! It's done in 'buildLineStarts()'
            }
            else {
                // need to find all lines deleted and remove from the lineStarts array
                // stitching the final line into the start line.
                buildLineStartsData();  // a cop-out for now
            }
        }
    }
    else {
        // a cross-chunk delete; delete all of this chunk from the start
        // and then all the intervening chunks through to the endChunk.
        // Assume worst case and re-build the lineStarts from scratch

        // watch for deleting all of the initial chunk
        if (startOffset != 0) {
            if (!deleteFromChunk(chunkP, startOffset, chunkP->length - startOffset))
                return false;
            chunkIndex++;
            chunkP = &gChunkList[chunkIndex];
        }
        int deleteCount = 0;
        int startDelete = chunkIndex;
        while (true) {
            // deleting a whole chunk means that the chunkList is
            // shuffled down, so the lineStarts HAS to be updated
            // for new chunk numbers - rebuilding the lineStarts
            // is a simple solution
            if (chunkIndex == endChunkIndex)
                if (endOffset < chunkP->length)
                    break;
            MemHandleUnlock(chunkP->handle);
            int index = DmFindResource(gTempDB, TextType, chunkIndex, NULL);
            TEXT_ASSERT((index != -1), "Failed to find resource");
            DmRemoveResource(gTempDB, index);
            chunkP->handle = NULL;
            deleteCount++;
            if (chunkIndex == endChunkIndex) break;
            chunkIndex++;
            chunkP = &gChunkList[chunkIndex];
        }
        if (endOffset < chunkP->length)
            if (!deleteFromChunk(chunkP, 0, endOffset))
                return false;
        if (deleteCount > 0) {
            shuffleChunks(startDelete + deleteCount, startDelete);
            gTopChunk -= deleteCount;
        }
        if (rebuildLineStarts) buildLineStartsData();
    }
    return true;
}

Boolean deleteSelection()
{
    if (gHasSelection && !isEmptySelection()) {
        deleteListChunks(gUndoChunkList, gUndoChunkCount, UndoType);
        gUndoChunkCount = 0;
        if (!copySelectionToList(gUndoChunkList, &gUndoChunkCount, UndoType))
            return false;
        gUndoAction = PasteAction;
        // set just an insertion point for the undo
        gUndoSelectionLine = gSelectionLine;
        gUndoSelectionOffset = gSelectionOffset;
        gUndoSelectionEndLine = gSelectionLine;
        gUndoSelectionEndOffset = gSelectionOffset;


        // calculate the per-chunk offsets for selection start and end
        Word startOffset;
        Word chunkIndex;
        getChunkAndOffset(gSelectionLine, gSelectionOffset, &chunkIndex, &startOffset);
        Word endOffset;
        Word endChunkIndex;
        getEndChunkAndOffset(gSelectionEndLine, gSelectionEndOffset, &endChunkIndex, &endOffset);

        if (!deleteRange(chunkIndex, startOffset, endChunkIndex, endOffset, true))
            return false;

        // reduce the selection to the insertion point at the start
        gSelectionEndLine = gSelectionLine;
        gSelectionEndOffset = gSelectionOffset;

        // We need to rebuild the line starts! /jw
        buildLineStartsData();
    }
    return true;
}

//  Copy from chunkP, the given offset and length into a new
//  chunk in the given chunkList of the given type. Updating
//  the chunk count for that list via the count pointer
//
//
static Boolean copyFromChunk(TextChunk *chunkP, ULong type, TextChunk *chunkList,
                                        Word *chunkCountP, Word length, Word offset)
{
    TEXT_ASSERT((length <= chunkP->length), "bad length for copy chunk");
    TEXT_ASSERT((offset < chunkP->length), "bad offset for copy chunk");
    TEXT_ASSERT((chunkP->kind != CollapsedText), "partial copy from collapse chunk");

    int chunkCount = *chunkCountP;
    Handle h = (Handle)DmNewResource(gTempDB, type, chunkCount, length);
    if (h == NULL) {
        error("Couldn't allocate for copy selection");
        return false;
    }
    CharPtr p = (CharPtr)MemHandleLock(h);
    DmWrite(p, 0, chunkP->text + offset, length);
    chunkList[chunkCount].kind = chunkP->kind;
    chunkList[chunkCount].length = length;
    chunkList[chunkCount].text = p;
    chunkList[chunkCount].handle = h;
    chunkList[chunkCount].depth = chunkP->depth;
    chunkCount++;
    *chunkCountP = chunkCount;
    return true;
}

static Boolean copyAllChunk(TextChunk *chunkP, ULong type, TextChunk *chunkList, Word *chunkCountP)
{
    int length = (chunkP->kind == CollapsedText) ? chunkP->collapsedLength : chunkP->length;
    int chunkCount = *chunkCountP;
    Handle h = (Handle)DmNewResource(gTempDB, type, chunkCount, length);
    if (h == NULL) {
        error("Couldn't allocate for copy selection");
        return false;
    }
    CharPtr p = (CharPtr)MemHandleLock(h);
    DmWrite(p, 0, chunkP->text, length);
    chunkList[chunkCount].kind = chunkP->kind;
    if (chunkP->kind == CollapsedText) {
        chunkList[chunkCount].collapsedLength = length;
        chunkList[chunkCount].length = 1;
    }
    else
        chunkList[chunkCount].length = length;
    chunkList[chunkCount].text = p;
    chunkList[chunkCount].handle = h;
    chunkList[chunkCount].depth = chunkP->depth;
    chunkCount++;
    *chunkCountP = chunkCount;
    return true;
}

static Boolean copyFromText(CharPtr text, ULong type, TextChunk *chunkList,
        Word *chunkCountP, Word length, SegmentKind kind)
{
    int chunkCount = *chunkCountP;
    Handle h = (Handle)DmNewResource(gTempDB, type, chunkCount, length);
    if (h == NULL) {
        error("Couldn't allocate for copyText selection");
        return false;
    }
    CharPtr p = (CharPtr)MemHandleLock(h);
    DmWrite(p, 0, text, length);
    chunkList[chunkCount].kind = kind;
    chunkList[chunkCount].length = length;
    chunkList[chunkCount].text = p;
    chunkList[chunkCount].handle = h;
    chunkList[chunkCount].depth = 0;
    chunkCount++;
    *chunkCountP = chunkCount;
    return true;
}

// deleting all chunks from the list
static void deleteListChunks(TextChunk *chunkList, Word chunkCount, ULong type)
{
    if (chunkCount > 0) {
        for (Word i = 0; i < chunkCount; i++) {
            MemHandleUnlock(chunkList[i].handle);
            int index = DmFindResource(gTempDB, type, i, NULL);
            TEXT_ASSERT((index != -1), "Failed to find resource for chunk delete");
            DmRemoveResource(gTempDB, index);
            chunkList[i].handle = NULL;
        }
    }
}

static Boolean copyRangeToList(Word chunkIndex, Word startOffset, Word endChunkIndex, Word endOffset,
        TextChunk *chunkList, Word *chunkCountP, ULong type)
{
    TextChunk *chunkP = &gChunkList[chunkIndex];
    if (chunkIndex == endChunkIndex) {
        Word length = endOffset - startOffset;
        if (length == chunkP->length) {
            if (!copyAllChunk(chunkP, type, chunkList, chunkCountP))
                return false;
        }
        else
            if (!copyFromChunk(chunkP, type, chunkList, chunkCountP, length, startOffset))
                return false;
    }
    else {
    /*
        copy from the first chunk at selection start thru to the end of the chunk
    */
        Word length = chunkP->length - startOffset;
        if (!copyFromChunk(chunkP, type, chunkList, chunkCountP, length, startOffset))
            return false;

        chunkIndex++;
        chunkP = &gChunkList[chunkIndex];
    /*
        then all of the intervening chunks
    */
        while (chunkIndex != endChunkIndex) {
            if (!copyAllChunk(chunkP, type, chunkList, chunkCountP))
                return false;
            chunkIndex++;
            chunkP = &gChunkList[chunkIndex];
        }
    /*
        and finally from the start to the end of selection in the last chunk
    */
        if (endOffset > 0) {
            length = endOffset;
            if (!copyFromChunk(chunkP, type, chunkList, chunkCountP, length, 0))
                return false;
        }
    }
    return true;
}

static Boolean copySelectionToList(TextChunk *chunkList, Word *chunkCountP, ULong type)
{
    if (gHasSelection && !isEmptySelection()) {
        // calculate the per-chunk offsets for selection start and end
        Word startOffset;
        Word chunkIndex;
        getChunkAndOffset(gSelectionLine, gSelectionOffset, &chunkIndex, &startOffset);
        Word endOffset;
        Word endChunkIndex;
        getEndChunkAndOffset(gSelectionEndLine, gSelectionEndOffset, &endChunkIndex, &endOffset);
        return copyRangeToList(chunkIndex, startOffset, endChunkIndex, endOffset, chunkList, chunkCountP, type);
    }
    return true;
}

Boolean copySelection()
{
    if (gHasSelection && !isEmptySelection()) {
        deleteListChunks(gClipChunkList, gClipChunkCount, ClipType);
        gClipChunkCount = 0;
        if (!copySelectionToList(gClipChunkList, &gClipChunkCount, ClipType))
            return false;
    }
    return true;
}

CharPtr getClipboard(Word *length)
{
    CharPtr result = NULL;
    if (gClipChunkCount) {
        long size = 0;
        for (Word i = 0; i < gClipChunkCount; i++) {
            size += gClipChunkList[i].length;
        }
        *length = size;
        result = (char *)MemPtrNew(size);
        long offset = 0;
        for (Word i = 0; i < gClipChunkCount; i++) {
            MemMove(result + offset, gClipChunkList[i].text, gClipChunkList[i].length);
            offset += gClipChunkList[i].length;
        }
    }
    return result;
}

Boolean setClipboard(CharPtr text, Word length)
{
    deleteListChunks(gClipChunkList, gClipChunkCount, ClipType);
    gClipChunkCount = 0;
    return copyFromText(text, ClipType, gClipChunkList, &gClipChunkCount, length, PlainText);
}

Boolean cutSelection()
{
/*
    just like delete except we copy to the clipboard first
*/
    if (!copySelection())
        return false;

    return deleteSelection();   // reduces the selection to the insertion point
}

/*
    find the line number and offset for an insertion
    point at the start of the given chunk
*/
static void setSelectionAtChunkStart(Word targetChunk)
{
    for (Word i = 0; i < gMaxLines; i++) {
    //
    //  a problem is that it could be that no lines begin in the target chunk
    //  so we have to find the line that starts in the nearest next chunk
    //
        if (gLineStarts[i].chunkNumber >= targetChunk) {
            if ((gLineStarts[i].offset == 0) && (gLineStarts[i].chunkNumber == targetChunk)) {
                gSelectionLine = i;
                gSelectionOffset = 0;
                gSelectionEndLine = gSelectionLine;
                gSelectionEndOffset = gSelectionOffset;
            }
            else {
                gSelectionLine = i - 1;
                // this line may cross many chunks, but we need to set the
                // offset as the position of the character from it that is in
                // the target chunk.
                Word chunkIndex = gLineStarts[i - 1].chunkNumber;
                Word lineLength = gChunkList[chunkIndex].length - gLineStarts[i - 1].offset;
                chunkIndex++;
                while (chunkIndex != targetChunk) {
                    lineLength += gChunkList[chunkIndex].length;
                    chunkIndex++;
                }
                gSelectionOffset = lineLength;
                gSelectionEndLine = gSelectionLine;
                gSelectionEndOffset = gSelectionOffset;
            }
            return;
        }
    }
    // get here if the insertion point is in the last line (somewhere)
    gSelectionLine = gMaxLines - 1;
    Word chunkIndex = gLineStarts[gSelectionLine].chunkNumber;
    gSelectionOffset = gChunkList[chunkIndex].length - gLineStarts[gSelectionLine].offset;
    chunkIndex++;
    while (chunkIndex != targetChunk) {
        gSelectionOffset += gChunkList[chunkIndex].length;
        chunkIndex++;
    }
    gSelectionEndLine = gSelectionLine;
    gSelectionEndOffset = gSelectionOffset;
}

static void getChunkAndOffset(Word line, Word offset, Word *chunkIndexP, Word *chunkOffsetP)
{
    Word chunkIndex = gLineStarts[line].chunkNumber;
    if ((gLineStarts[line].offset + offset) < gChunkList[chunkIndex].length) {
        *chunkOffsetP = gLineStarts[line].offset + offset;
    }
    else {
        offset -= (gChunkList[chunkIndex].length - gLineStarts[line].offset);
        chunkIndex++;
        while (offset >= gChunkList[chunkIndex].length) {
            offset -= gChunkList[chunkIndex].length;
            chunkIndex++;
        }
        *chunkOffsetP = offset;
    }
    TEXT_ASSERT((chunkIndex != gTopChunk), "offset overrun");
    *chunkIndexP = chunkIndex;
}

//  The difference between this and the above is what happens on chunk boundaries -
//      if we're looking for the end offset, and that coincides with a chunk end
//      then we want that chunk, if we wanted a starting offset we need the next
//      chunk.
//
static void getEndChunkAndOffset(Word line, Word offset, Word *chunkIndexP, Word *chunkOffsetP)
{
    Word chunkIndex = gLineStarts[line].chunkNumber;
    if ((gLineStarts[line].offset + offset) <= gChunkList[chunkIndex].length) {
        *chunkOffsetP = gLineStarts[line].offset + offset;
    }
    else {
        offset -= (gChunkList[chunkIndex].length - gLineStarts[line].offset);
        chunkIndex++;
        while (offset > gChunkList[chunkIndex].length) {
            offset -= gChunkList[chunkIndex].length;
            chunkIndex++;
        }
        *chunkOffsetP = offset;
    }
    *chunkIndexP = chunkIndex;
}

static Boolean insertFromChunkList(Word *startChunk, Word startOffset, TextChunk *fromChunkList, Word fromChunkCount)
{
    // make room in the chunk list for the incoming chunks
    // plus 1 (since we'll be splitting the current chunk
    // at the insertion point)

    // we don't split iff the insertion point just happens
    // to be at a chunk start
    Word newChunks = fromChunkCount;
    if (startOffset > 0) {
        newChunks++;    // we'll split startChunk into two
        (*startChunk)++;    // so we move chunks above it up to make room
    }
    if ((newChunks + gTopChunk) >= MAX_CHUNKS) {
        error("Too many chunks in paste");
        return false;
    }
    // make room for the incoming chunks
    VALIDATE;
    shuffleChunks(*startChunk, *startChunk + newChunks);
    // split the target chunk
    if (startOffset > 0) {
        TextChunk *chunkToSplit = &gChunkList[*startChunk - 1];
        Word chunkLoc = *startChunk + fromChunkCount;
        copyFromChunk(chunkToSplit, TextType, gChunkList, &chunkLoc, chunkToSplit->length - startOffset, startOffset);
        deleteFromChunk(chunkToSplit, startOffset, chunkToSplit->length - startOffset);
    }
    // now copy the clip contents into the intervening new slots
    Word chunkLoc = *startChunk;
    for (Word i = 0; i < fromChunkCount; i++) {
        TextChunk *clipChunk = &fromChunkList[i];
        copyAllChunk(clipChunk, TextType, gChunkList, &chunkLoc);   // chunkLoc gets incremented
    }
    gTopChunk += newChunks;
    buildLineStartsData();
    return true;
}

static Boolean prvPasteSelection(TextChunk *fromChunkList, Word fromChunkCount)
{
    // if there is a non-empty selection, it becomes the undo paste contents
    // otherwise there is just an undo delete

    if (gHasSelection) {
        if (!isEmptySelection()) {
        // COULD BE BETTER
        // - the delete selection is blowing away chunks we could
        // reuse AND is re-building line starts only to re-build
        // them again below.
        //
            if (!deleteSelection())     // current selection becomes the undo, action is paste
                return false;
        }
        else
            // if there's nothing to replace with, the undo is just a delete
            gUndoAction = DeleteAction;

        // need to set the undoSelection to cover the text being inserted
        // i.e from the original gSelectionLine/Offset to the eventual
        // new version thereof.
        gUndoSelectionLine = gSelectionLine;
        gUndoSelectionOffset = gSelectionOffset;

        Word startOffset;
        Word startChunk;
        getChunkAndOffset(gSelectionLine, gSelectionOffset, &startChunk, &startOffset);
        if (!insertFromChunkList(&startChunk, startOffset, fromChunkList, fromChunkCount))
            return false;

        // startChunk is modified to reflect the start of the insertion chunks
        setSelectionAtChunkStart(startChunk + fromChunkCount);
        // can now set undoEndSelection
        gUndoSelectionEndLine = gSelectionLine;
        gUndoSelectionEndOffset = gSelectionOffset;

        // Need to rebuild the line starts... /jw
        buildLineStartsData();
    }
    return true;
}

Boolean pasteSelection()
{
    if (gClipChunkCount > 0)
        return prvPasteSelection(gClipChunkList, gClipChunkCount);
    else
        return true;
}

Boolean undo()
{
    if (gUndoAction == NoAction) return true;

    gHasSelection = true;
    gSelectionLine = gUndoSelectionLine;
    gSelectionOffset = gUndoSelectionOffset;
    gSelectionEndLine = gUndoSelectionEndLine;
    gSelectionEndOffset = gUndoSelectionEndOffset;

    Boolean result = false;

    switch (gUndoAction) {
        case DeleteAction :
            result = deleteSelection();
            break;
        case PasteAction : {
                Handle h = (Handle)MemHandleNew(sizeof(TextChunk) * gUndoChunkCount);
                TextChunk *tempChunkList = (TextChunk *)MemHandleLock(h);
                Word tempChunkCount = 0;
                for (Word i = 0; i < gUndoChunkCount; i++)
                    if (!copyAllChunk(&gUndoChunkList[i], TempType, tempChunkList, &tempChunkCount))
                        return false;

                result = prvPasteSelection(tempChunkList, tempChunkCount);

                deleteListChunks(tempChunkList, tempChunkCount, TempType);
            }
            break;
        default :
            error("Illegal undo action");
            return false;
    }
    return result;
}

Boolean commentSelection()
{
    // #846291 (S053) Comment-out selected region

    UInt16 originalSelectionLine = gSelectionLine;
    UInt16 originalSelectionEndLine = gSelectionEndLine;
    UInt16 originalSelectionOffset = gSelectionOffset;
    UInt16 originalSelectionEndOffset = gSelectionEndOffset;
    if (originalSelectionEndOffset > (UInt8)(getLineLength(originalSelectionEndLine)-1))
            originalSelectionEndOffset = (getLineLength(originalSelectionEndLine)-1);

    UInt16 i = originalSelectionLine;
    UInt16 y = originalSelectionEndLine;
    Boolean selectionStartsWithWhitespace = false;

    Word startOffset;
    Word startChunk;
    getChunkAndOffset(originalSelectionLine, originalSelectionOffset, &startChunk, &startOffset);
    char *p = gChunkList[startChunk].text + startOffset;
    if (isWhitespace(p[0]))
        selectionStartsWithWhitespace = true;

    // check Mode
    if (gHighlightMode == CMODE){
        // C
        Boolean needEndingNewLine = false;
        markSelection(i, originalSelectionOffset, i, originalSelectionOffset);
        if (i != y){ // Block comment needed
            if (originalSelectionOffset !=0){
                insertNewLine(); // This handles the Chunks &gMaxLines
                i++;
                y++;
                originalSelectionOffset = 0;
            }
            if (originalSelectionEndOffset < (UInt8)(getLineLength(originalSelectionEndLine)-1) && (originalSelectionEndLine != gMaxLines -1))
                needEndingNewLine = true;

            insertCharacters( gCComments[0].start, gCComments[0].startLen, true );
            markSelection(i, gCComments[0].startLen, i, gCComments[0].startLen);
            insertNewLine();
            i++;
            y++;
            markSelection(y, originalSelectionEndOffset, y, originalSelectionEndOffset);
            if (originalSelectionEndOffset != 0)
                insertNewLine();
            insertCharacters(gCComments[0].end, gCComments[0].endLen, true );

            if (needEndingNewLine)
                insertNewLine();

            originalSelectionLine =i;
            originalSelectionEndLine =y;
            }
        if (i == y){ // Lets insert a line comment
            // insert the comment indicator
            markSelection(i, originalSelectionOffset, i, originalSelectionOffset);
            insertCharacters(gCComments[1].start, gCComments[1].startLen, true);
            // Only insert the space if needed
            if (!selectionStartsWithWhitespace)
                insertCharacters(" ", 1, true);
            // fix the Selection to be...
            originalSelectionOffset += gCComments[1].startLen +1;
            originalSelectionEndOffset += gCComments[1].startLen + 1;
            }
    }
    if (gHighlightMode == ASMMODE){
        // ASM
        if (originalSelectionEndLine == i+1 && originalSelectionEndOffset == 0)
            y = i;
        for (UInt16 x = i; x< y+1 ; x++) {
            if (x == i)
                markSelection(x, originalSelectionOffset, x, originalSelectionOffset);
            else {
                markSelection(x, 0, x, 0);
            }
            insertCharacters( gASMComments[0].start, gASMComments[0].startLen, true );
            insertCharacters( " ", 1, true );
        }
        originalSelectionOffset += gASMComments[0].startLen +1;
        if (originalSelectionEndOffset != 0)
            originalSelectionEndOffset += gASMComments[0].startLen +1;
    }
    if (gHighlightMode == HTMLMODE){
        // HTML
        markSelection(i, originalSelectionOffset, i, originalSelectionOffset);
        insertCharacters( gHTMLComments[0].start, gHTMLComments[0].startLen, true );
        if (i == y)
            markSelection(y, originalSelectionEndOffset + gHTMLComments[0].startLen, y, originalSelectionEndOffset + gHTMLComments[0].startLen);
        else
            markSelection(y, originalSelectionEndOffset, y, originalSelectionEndOffset);

        insertCharacters( gHTMLComments[0].end, gHTMLComments[0].endLen, true );

        originalSelectionOffset += gHTMLComments[0].startLen;
        originalSelectionEndOffset += gHTMLComments[0].startLen;
    }
    markSelection(originalSelectionLine, originalSelectionOffset, originalSelectionEndLine, originalSelectionEndOffset);
    return true;
}

Boolean findCharsForwards(Word searchStartLine, Word searchStartOffset, char* t, Word maxDistance)
{
    Word startOffset;
    Word startChunk;
    getChunkAndOffset(searchStartLine, searchStartOffset, &startChunk, &startOffset);
    if (startChunk == gTopChunk) return false;

    char *p = gChunkList[startChunk].text + startOffset;
    UInt16 tLen = StrLen(t);

    while (true) {
        while (*p && (StrNCompare(p, t, tLen) != 0)) p++;
        if (StrNCompare(p, t, tLen) == 0) {
            long docOffset = 0;
            for (Word i = 0; i < startChunk; i++)
                docOffset += gChunkList[i].length;
            setSelectionFromCharacterOffsetRange(docOffset + (p - gChunkList[startChunk].text), 1);
            return true;
        }
        else {
            startChunk++;
            if (startChunk == gTopChunk) break;
            p = gChunkList[startChunk].text;
            if (--maxDistance <= 0) return false;
        }
    }
    return false;
}

Boolean findCharsBackwards(Word searchStartLine, Word searchStartOffset, char* t, Word maxDistance)
{
    Word startOffset;
    Word startChunk;
    getChunkAndOffset(searchStartLine, searchStartOffset, &startChunk, &startOffset);

    char *p = gChunkList[startChunk].text + startOffset - 1;
    UInt16 tLen = StrLen(t);

    while (true) {
        if (p < gChunkList[startChunk].text) {
            if (startChunk == 0) break;
            startChunk--;
            p = gChunkList[startChunk].text + gChunkList[startChunk].length - 1;
        }

        while (StrNCompare(p, t, tLen) !=0) {
            p--;
            if (--maxDistance <= 0) return false;
            if (p < gChunkList[startChunk].text)
                break;
        }
        if (StrNCompare(p, t, tLen) == 0) {
            long docOffset = 0;
            for (Word i = 0; i < startChunk; i++)
                docOffset += gChunkList[i].length;
            setSelectionFromCharacterOffsetRange(docOffset + (p - gChunkList[startChunk].text), tLen);
            return true;
        }
    }
    return false;
}

Boolean uncommentSelectionStart()
{
    // #846291 (S053) Comment-out selected region

//    Handles theese cases:
//        1. User Selects only text (The indicator is last on the previous line)
//        2. User Selects only text (The indicator is someware on the same line)
//        3. User Selects comment indicator + text
//        4. User Semi selects comment indicator + text

    UInt16 originalSelectionLine = gSelectionLine;
    UInt16 originalSelectionEndLine = gSelectionEndLine;
    UInt16 originalSelectionOffset = gSelectionOffset;
    UInt16 originalSelectionEndOffset = gSelectionEndOffset;
    UInt16 i = originalSelectionLine;

    Boolean hasEnd = false; // indicates if we have to search for a comment end
    UInt8 commentType = NULL; // indicate what type of comment end we have to search for
    UInt8 space = 0; // indicates if a space were found imediatly after the comment start indicator
    Boolean handledAllLines = false; // indicates if all selected lines contained a comment indicator (C: line comments, ASM: comments)
    char* p = NULL;

    UInt16 startOffset = 0, startChunk = 0, originalOffset = 0, originalChunk = 0;
    getChunkAndOffset(originalSelectionLine, originalSelectionOffset, &originalChunk, &originalOffset);
    // check Mode
    if (gHighlightMode == CMODE){
        UInt8 numCharsToSearch = originalSelectionOffset+3;  // Hey.. need to change this!
        if (i !=0)
            numCharsToSearch += 2+1;
        // Selection could contain the comment indicator
        for (Int8 x = 0; x<gNumCComments; x++){
        // for (Int8 x = gNumCComments-1; x>-1; x--){ // This will make sure to look for a line comment indicator before looking for a block comment indicator
            if(findCharsBackwards(i, originalSelectionOffset+gCComments[x].startLen, gCComments[x].start, numCharsToSearch)){
                if (StrCompare(gCComments[x].start, "/*") == 0){
                    // It's a block-comment
                    getChunkAndOffset(gSelectionLine, gSelectionOffset, &startChunk, &startOffset);
                    p = gChunkList[startChunk].text + startOffset;
                    if (p[gCComments[x].startLen] == '\n'){
                        // error("New Line detected");              // Use only for testing
                        if(gSelectionOffset !=0){
                            originalSelectionOffset = 0;
                            // error("Start indicator NOT alone");  // Use only for testing
                            }
                        else{
                            // remove the newline
                            gSelectionEndLine = gSelectionLine + 1;
                            gSelectionEndOffset = 0;
                            originalSelectionEndLine -=1;
                            // error("Start indicator alone");      // Use only for testing
                        }

                        if((originalOffset - startOffset)< gCComments[x].startLen){
                            // User selected (or semi selected) the comment indicator
                            // Fix the selection to be
                            originalSelectionOffset = 0;
                            // error("Selected indicator");         // Use only for testing
                        }
                        else{
                            if(originalSelectionEndOffset !=0)
                                originalSelectionEndOffset += gCComments[x].startLen;
                            // error("Not selected indicator");     // Use only for testing
                            if (i>gSelectionLine){
                                // Fix the selection to be
                                if(gSelectionOffset ==0)
                                    originalSelectionLine -=1;
                                // error("Selection starts on the follwing line");  // Use only for testing
                            }
                            else{
                                originalSelectionOffset = 0;
                                // error("Selection starts on the same line"); // Use only for testing
                            }
                        }
                    }
                    else{
                        // error("No New line");                                // Use only for testing
                        if(originalOffset != startOffset){
                            // User did NOT select the comment indicator
                            // Fix the selection to be
                            originalSelectionOffset -= gCComments[x].startLen;
                            // error("Not selected indicator");                 // Use only for testing
                            if ((originalOffset - startOffset)< gCComments[x].startLen){
                                // special Case: User semi selected comment indicator
                                // Fix the selection to be
                                originalSelectionOffset +=(originalOffset - startOffset);
                                // error("Semi selected indicator");            // Use only for testing
                            }
                        }
                    }
                    hasEnd = true;
                }
                else{
                    // It's a line-comment
                    // Line comments are only valid for a line at a time...
                    // Fix the selection to be
                    if(originalSelectionOffset>gCComments[x].startLen)
                        originalSelectionOffset -= gCComments[x].startLen;
                    p = gChunkList[startChunk].text + originalOffset;
                    if (!isWhitespace(p[0]) || (originalOffset == startOffset)){
                        // ........... or User selected the comment indicator
                        getChunkAndOffset(gSelectionLine, gSelectionOffset, &startChunk, &startOffset);
                        p = gChunkList[startChunk].text + startOffset;
                        if (p[gCComments[x].startLen]==' '){
                            // The char after the comment indicator is a 'space'
                            space = 1;
                            gSelectionEndOffset +=space;
                            // Fix the selection to be
                            if(originalSelectionOffset>0)
                                originalSelectionOffset -=space;
                        }
                    }
                    handledAllLines = true;
                    hasEnd = false;  // Since we're dealing with a line comment we don't have to search for a comment end
                }
                deleteSelection();

                // Check if wee have to deal with more lines with line comment start indicators
                if ((StrCompare(gCComments[x].start, "//")==0) && originalSelectionEndLine > i && !((originalSelectionEndLine == i+1) && (originalSelectionEndOffset == 0))){ // Are there more lines selected?
                    // error("More lines selected");                               // Use only for testing
                    for(UInt16 line = i+1; line<originalSelectionEndLine+1; line++){ // takes care of any following lines..
                        numCharsToSearch = getLineLength(line); // search the line
                        space = 0;
                        handledAllLines = false;
                        if(findCharsForwards(line, 0, gCComments[x].start, numCharsToSearch)){
                            getChunkAndOffset(gSelectionLine, gSelectionOffset, &startChunk, &startOffset);
                            p = gChunkList[startChunk].text + startOffset;
                            // error(p);                                            // Use only for testing
                            if (p[gCComments[x].startLen] == ' '){
                                // error("Space char detected");                    // Use only for testing
                                space = 1;
                            }
                            handledAllLines = true;
                            gSelectionEndOffset = gSelectionOffset + gCComments[x].startLen +space;
                            deleteSelection();
                        }
                        else
                            break;  // Don't search if the next line wasn't commented
                    }
/*
                    if(handledAllLines == false)
                          error("Selected lines with NO comment indicator.");
*/
                }

                if ((handledAllLines == true) && (originalSelectionLine == originalSelectionEndLine) && (originalSelectionEndOffset !=0))
                    // Fix the selection to be
                    originalSelectionEndOffset -=gCComments[x].startLen+space;
                commentType = x;
                goto exit;
            }
        }
        error("Could not find C comment start indicator.");
    }
    if (gHighlightMode == ASMMODE){
        // ASM
        space = 0;

        UInt8 numCharsToSearch = originalSelectionOffset+1;
        if (i !=0)
            numCharsToSearch += getLineLength(i-1);

        for (UInt8 x = 0; x<gNumASMComments; x++){
            handledAllLines = false;
            UInt16 searchOffset = originalSelectionOffset +gASMComments[x].startLen;
            if(findCharsBackwards(i, searchOffset, gASMComments[x].start, numCharsToSearch)){
                getChunkAndOffset(gSelectionLine, gSelectionOffset, &startChunk, &startOffset);
                p = gChunkList[startChunk].text + startOffset;
                if (p[gASMComments[x].startLen] == ' ')
                    space = 1;
                gSelectionEndOffset += space;
                originalSelectionOffset -= gASMComments[x].startLen+space;
                deleteSelection();
                handledAllLines = true;
                if (originalSelectionEndLine > i && !((originalSelectionEndLine == i+1) && (originalSelectionEndOffset == 0))){ // Are there more lines selected?
                    searchOffset = 0;
                    // error("More than one line selected");                        // Use only for testing
                    for(UInt16 line = i+1; line<originalSelectionEndLine+1; line++){ // takes care of any following lines..
                        numCharsToSearch = getLineLength(line); // search the line
                        handledAllLines = false;
                        space = 0;
                        if(findCharsForwards(line, searchOffset, gASMComments[x].start, numCharsToSearch)){
                            getChunkAndOffset(gSelectionLine, gSelectionOffset, &startChunk, &startOffset);
                            p = gChunkList[startChunk].text + startOffset;
                            if (p[gASMComments[x].startLen] == ' ')
                                space = 1;
                            gSelectionEndOffset += space;
                            deleteSelection();
                            handledAllLines = true;
                        }
                    }
                }
            if (originalSelectionEndOffset !=0 && (handledAllLines==true))
                originalSelectionEndOffset -=gASMComments[x].startLen+space;
            hasEnd = false; // ASM comments don't have a comment end
            commentType = x;
            goto exit;
            }
        }
        error("Could not find ASM comment start indicator.");
    }
    if (gHighlightMode == HTMLMODE){
        // HTML
        UInt8 numCharsToSearch = originalSelectionOffset+1;
        if (i !=0)
            numCharsToSearch += getLineLength(i-1);

        for (UInt8 x = 0; x<gNumHTMLComments; x++){
            if(findCharsBackwards(i, originalSelectionOffset+gHTMLComments[x].startLen, gHTMLComments[x].start, numCharsToSearch)){
                if (gSelectionLine == originalSelectionLine){
                    originalSelectionOffset -= gHTMLComments[x].startLen;
                    originalSelectionEndOffset -=gHTMLComments[x].startLen;
                    }
                deleteSelection();
                hasEnd = true; // HTML comments does have a comment end
                commentType = x;
                goto exit;
            }
        }
        error("Could not find HTML comment start indicator.");
    }
    exit:
    markSelection(originalSelectionLine, originalSelectionOffset, originalSelectionEndLine, originalSelectionEndOffset);

    if (hasEnd)
        uncommentSelectionEnd(commentType);

    return true;
}

Boolean uncommentSelectionEnd(UInt8 commentType)
{
    // #846291 (S053) Comment-out selected region

//    Handles theese cases:
//        1. User Selects only text (The indicator is someware on the next line)
//        2. User Selects only text (The indicator is someware on the same line)
//        3. User Selects comment indicator + text
//        4. User semi Selects comment indicator + text

    UInt16 originalSelectionLine = gSelectionLine;
    UInt16 originalSelectionEndLine = gSelectionEndLine;
    UInt16 originalSelectionOffset = gSelectionOffset;
    UInt16 originalSelectionEndOffset = gSelectionEndOffset;

    UInt16 y = originalSelectionEndLine;
    char* p = NULL;
    UInt16 startOffset = 0, startChunk = 0;

    // check Mode
    if (gHighlightMode == CMODE){
        // C Mode
        if (StrCompare(gCComments[commentType].end, "\n") == 0){
            // It's a line-comment
            error("Doesn't know of a 'line comment end indicator'");
            goto exit;
        }

        UInt8 numCharsToSearch = getLineLength(y); // - originalSelectionEndOffset-1;
        if (y+1<gMaxLines)
            numCharsToSearch += getLineLength(y+1);

            // Selection could contain the comment indicator
            if(findCharsForwards(y, getLineLength(y), gCComments[commentType].end, numCharsToSearch)){
            // if(findCharsForwards(y-1, getLineLength(y-1), gCComments[commentType].end, numCharsToSearch)){
                // Get Chunk & Offset for indicator start
                getChunkAndOffset(gSelectionLine, gSelectionOffset, &startChunk, &startOffset);
                p = gChunkList[startChunk].text + startOffset;

                UInt8 overSelection = 0;
                if (originalSelectionEndLine == gSelectionLine)
                    overSelection = (originalSelectionEndOffset - gSelectionOffset);
/*
                MemHandle H = MemHandleNew(30);
                char* s = (char*)MemHandleLock(H);
                StrIToA(s, (Int8)overSelection);
                error(s);
                MemHandleUnlock(H);
*/
                if (p[gCComments[commentType].endLen] == '\n'){
                    if(overSelection){
                        // User selected (or semi seleced) the comment end indicator
                        // Fix the selection to be
                        originalSelectionEndOffset =0;
                        // error("End indicator selected");
                    }
                    // make sure the newline is removed
                    if (gSelectionEndLine == gMaxLines -1){
                        gSelectionLine -=1;
                        gSelectionOffset = getLineLength(gSelectionLine)-1;
                        gSelectionEndOffset +=1 ;
                        // error("Comment end indicator on the Last line...");
                    }

                    else{
                        gSelectionEndLine +=1;
                        gSelectionEndOffset =0 ;
                    }
                    // error("New Line detected");
                }
                else{
                    // It's NOT a \n after the comment indicator
                    gSelectionEndOffset +=1;
                    // error("No new Line");
                    if(overSelection){
                        // if comment indicator was selected (or semi selected)
                        // Fix the selection to be
                        switch (overSelection){
                            case 1:
                            case 2:
                                originalSelectionEndOffset -= overSelection;
                                // error("End indicator selected");
                                break;
                            default:
                                error("Comment end over selected");
                                break;
                        }
                    }
                }
                deleteSelection();
                goto exit;
            }
        error("Could not find C comment end indicator.");
    }
    if (gHighlightMode == ASMMODE){
        error("Deoesn't know of a 'ASM comment end indicator'.");
        goto exit;
    }
    if (gHighlightMode == HTMLMODE){
        // HTML
        UInt8 numCharsToSearch = originalSelectionOffset+1;
        if (y !=0)
            numCharsToSearch += getLineLength(y-1);

        for (UInt8 x = 0; x<gNumHTMLComments; x++){
            if(findCharsBackwards(y, originalSelectionOffset+gHTMLComments[x].startLen, gHTMLComments[x].start, numCharsToSearch)){
                if (gSelectionLine == originalSelectionLine){
                    originalSelectionOffset -= gHTMLComments[x].startLen;
                    originalSelectionEndOffset -=gHTMLComments[x].startLen;
                    }
                deleteSelection();
                goto exit;
            }
        }
        error("Could not find HTML comment start indicator.");
    }
    exit:
    markSelection(originalSelectionLine, originalSelectionOffset, originalSelectionEndLine, originalSelectionEndOffset);
    return true;
}

Boolean stitchBlocks()
{
    for (Word i = 0; i < (gTopChunk - 1); i++) {
        if ((gChunkList[i].kind == gChunkList[i + 1].kind)
                && ((gChunkList[i].kind == PlainText) || ((gChunkList[i].depth == gChunkList[i + 1].depth))) ) {

            Word newLength;
            if (gChunkList[i].kind == PlainText)
                newLength = gChunkList[i].length + gChunkList[i + 1].length;
            else
                newLength = gChunkList[i].collapsedLength + gChunkList[i + 1].collapsedLength;

            if (newLength <= MAX_CHUNK_LENGTH) {
                // add the second chunk's contents onto the first's
                MemHandleUnlock(gChunkList[i].handle);
                Err err = MemHandleResize(gChunkList[i].handle, newLength);
                if (err) {
                    error("Couldn't resize handle for stitch");
                    return false;
                }
                CharPtr p = gChunkList[i].text = (CharPtr)MemHandleLock(gChunkList[i].handle);
                if (gChunkList[i].kind == PlainText)
                    DmWrite(p, gChunkList[i].length, gChunkList[i + 1].text, gChunkList[i + 1].length);
                else
                    DmWrite(p, gChunkList[i].collapsedLength, gChunkList[i + 1].text, gChunkList[i + 1].collapsedLength);

                MemHandleUnlock(gChunkList[i + 1].handle);
                int index = DmFindResource(gTempDB, TextType, (i + 1), NULL);
                TEXT_ASSERT((index != -1), "Failed to find resource");
                DmRemoveResource(gTempDB, index);
                gChunkList[i + 1].handle = NULL;
                //  now we re-build the line starts -
                //      all lines in the deleted chunk get transferred to the previous chunk
                //      lines in chunks beyond get moved down by one
                for (Word j = 0; j <= gMaxLines; j++) {
                    if (gLineStarts[j].chunkNumber == (i + 1)) {
                        gLineStarts[j].chunkNumber--;
                        if (gChunkList[i].kind == PlainText)
                            gLineStarts[j].offset += gChunkList[i].length;
                        else
                            gLineStarts[j].offset += gChunkList[i].collapsedLength;
                    }
                    else
                        if (gLineStarts[j].chunkNumber > (i + 1))
                            gLineStarts[j].chunkNumber--;
                }
                if (gChunkList[i].kind == PlainText)
                    gChunkList[i].length = newLength;
                else
                    gChunkList[i].collapsedLength = newLength;
                shuffleChunks(i + 2, i + 1);
                gTopChunk--;
                VALIDATE_LINES;
                VALIDATE;
                return true;        // true return implies lineStarts has changed, so update is necessary
            }
        }
    }
    return false;
}

#if 0
static Boolean collapseCore(Word startChunk, Word startOffset, char openDelimiter, char closeDelimiter)
{
    Word depth = 0;
    Word beginCollapseChunk = startChunk;
    Word beginCollapseOffset = startOffset - 1;     // the character AFTER the insertion point is not included
    Word endCollapseChunk = startChunk;
    Word endCollapseOffset = startOffset;

    if (gChunkList[beginCollapseChunk].kind == CollapsedText) { // starting at a collapsed chunk
        beginCollapseOffset = -1;                               // force immediate failure in the loop below
        depth = gChunkList[beginCollapseChunk].depth;
        endCollapseOffset = gChunkList[endCollapseChunk].length;
    }

    while (true) {
        int delimiterDepth = 0;
        char *p = gChunkList[beginCollapseChunk].text;
        while ((beginCollapseOffset != -1)  /* sic, because it's unsigned */
                        && ((p[beginCollapseOffset] != openDelimiter)
                                || (delimiterDepth > 0))) {
            if (p[beginCollapseOffset] == closeDelimiter)
                delimiterDepth++;
            if (p[beginCollapseOffset] == openDelimiter)
                delimiterDepth--;
            beginCollapseOffset--;
        }
        if (beginCollapseOffset != -1) {
            delimiterDepth = 0;
            while (true) {
                p = gChunkList[endCollapseChunk].text;
                while ((endCollapseOffset < gChunkList[endCollapseChunk].length)
                            && ((p[endCollapseOffset] != closeDelimiter)
                                    || (delimiterDepth > 0))) {
                    if (p[endCollapseOffset] == openDelimiter)
                        delimiterDepth++;
                    if (p[endCollapseOffset] == closeDelimiter)
                        delimiterDepth--;
                    endCollapseOffset++;
                }
                if (p[endCollapseOffset] == closeDelimiter) {

                // step the endCollapseOffset up by 1 to include the closeDelimiter

                    if (endCollapseOffset < (gChunkList[endCollapseChunk].length - 1))
                        endCollapseOffset++;
                    else {
                        endCollapseChunk++;
                        endCollapseOffset = 0;
                    }

                // we cut the delimited text into a temp chunk list and then paste it back
                // as CollapsedText, there are undoubtedly faster ways of doing this.

                    Handle h = (Handle)MemHandleNew(sizeof(TextChunk) * MAX_CLIP_CHUNKS);
                    TextChunk *tempChunkList = (TextChunk *)MemHandleLock(h);
                    Word tempChunkCount = 0;

                    if (!copyRangeToList(beginCollapseChunk, beginCollapseOffset,
                                        endCollapseChunk, endCollapseOffset,
                                        tempChunkList, &tempChunkCount, TempType))
                        return false;

                    if (!deleteRange(beginCollapseChunk, beginCollapseOffset,
                                        endCollapseChunk, endCollapseOffset, false))
                        return false;

                    for (int i = 0; i < tempChunkCount; i++) {
                        if (tempChunkList[i].kind != CollapsedText) {
                            tempChunkList[i].kind = CollapsedText;
                            tempChunkList[i].depth = depth + 1;
                            tempChunkList[i].collapsedLength = tempChunkList[i].length;
                            tempChunkList[i].length = 1;
                        }
                    }

                    // if we chopped off part of the original beginChunk, we actually do the
                    // insert at the start of the next chunk. This will happen whenever the
                    // text being removed spans more than 1 chunk.
                    if (beginCollapseChunk != endCollapseChunk) {
                        beginCollapseChunk++;
                        beginCollapseOffset = 0;
                    }

                    // this last step also rebuilds the line starts
                    if (!insertFromChunkList(&beginCollapseChunk, beginCollapseOffset, tempChunkList, tempChunkCount))
                        return false;

                    deleteListChunks(tempChunkList, tempChunkCount, TempType);

                    // set the selection at the first collapsed chunk (beginCollapseChunk has been updated to reflect this)
                    setSelectionAtChunkStart(beginCollapseChunk);

                    return true;
                }
                else {
                    while (true) {
                        endCollapseChunk++;
                        if (endCollapseChunk < gTopChunk) {
                            if (gChunkList[endCollapseChunk].kind == CollapsedText) {
                                if (gChunkList[endCollapseChunk].depth > depth)
                                    depth = gChunkList[endCollapseChunk].depth;
                            }
                            else {
                                endCollapseOffset = 0;
                                break;
                            }
                        }
                        else
                            return false;
                    }
                }
            }
        }
        else {
            while (true) {
                if (beginCollapseChunk == 0)
                    return false;
                else {
                    beginCollapseChunk--;
                    if (gChunkList[beginCollapseChunk].kind == CollapsedText) {
                        if (gChunkList[beginCollapseChunk].depth > depth)
                            depth = gChunkList[beginCollapseChunk].depth;
                    }
                    else {
                        beginCollapseOffset = gChunkList[beginCollapseChunk].length - 1;
                        break;
                    }
                }
            }
        }
    }
    return true;
}

Boolean collapseAll(char openDelimiter, char closeDelimiter)
{
    int chunk = 0;
    Boolean collapsedAny = false;
    while (true) {
        Boolean foundOne = false;
        if (gChunkList[chunk].kind != CollapsedText) {
            int length = gChunkList[chunk].length;
            char *p = gChunkList[chunk].text;
            for (int i = 0; i < length; i++) {
                if (p[i] == openDelimiter) {
                    if (!collapseCore(chunk, i + 1, openDelimiter, closeDelimiter)) return collapsedAny;
                    VALIDATE_LINES;
                    VALIDATE;
                    foundOne = true;
                    collapsedAny = true;
                    break;
                }
            }
        }
        if (foundOne) {
            chunk = 0;
        }
        else {
            chunk++;
            if (chunk == gTopChunk) break;
        }
    }
    return collapsedAny;
}
/*
    for an empty selection, we scan backwards & forwards from the selection point looking for the nearest
    of the given delimiter. The enclosed text becomes a collapsed chunk
*/

Boolean collapseText(char openDelimiter, char closeDelimiter)
{
    if (hasSelection()) {
        if (isEmptySelection()) {
            Word startChunk;
            Word startOffset;
            getChunkAndOffset(gSelectionLine, gSelectionOffset, &startChunk, &startOffset);
            Boolean result = collapseCore(startChunk, startOffset, openDelimiter, closeDelimiter);
            VALIDATE_LINES;
            VALIDATE;
            return result;
        }
    }
    return true;
}

void uncollapseText(Word line, Word offset)
{
    Word startChunk;
    Word startOffset;
    getChunkAndOffset(gSelectionLine, gSelectionOffset, &startChunk, &startOffset);

    if (gChunkList[startChunk].kind != CollapsedText) {
        // look at the next chunk, if we're at the end of one
        if ((startOffset == (gChunkList[startChunk].length - 1))
                && (gChunkList[startChunk + 1].kind == CollapsedText))
            startChunk++;
        else
            return;
    }

    Word depth = gChunkList[startChunk].depth;
    // simply re-mark all collapsed text as plain and rebuild the line starts
    while (gChunkList[startChunk].kind == CollapsedText) {
        if (gChunkList[startChunk].depth == depth) {
            gChunkList[startChunk].kind = PlainText;
            gChunkList[startChunk].length = gChunkList[startChunk].collapsedLength;
            gChunkList[startChunk].depth = 0;
        }
        startChunk++;
    }
    buildLineStartsData();
    int len = getLineLength(gSelectionLine);
    if (gSelectionOffset < (len - 1))
        gSelectionOffset++;
    else {
        gSelectionLine++;
        gSelectionOffset = 0;
    }
    gSelectionEndLine = gSelectionLine;
    gSelectionEndOffset = gSelectionOffset;

}
#endif

Boolean findMatchingParenBackwards(Word searchStartLine, Word searchStartOffset, char open_paren, char close_paren)
{
    Word startOffset;
    Word startChunk;
    getChunkAndOffset(searchStartLine, searchStartOffset, &startChunk, &startOffset);

    int depth = 0;
    char *p = gChunkList[startChunk].text + startOffset - 1;

    while (true) {
        if (p < gChunkList[startChunk].text) {
            if (startChunk == 0) break;
            startChunk--;
            p = gChunkList[startChunk].text + gChunkList[startChunk].length - 1;
        }

        while ((*p != open_paren) && (*p != close_paren)) {
            p--;
            if (p < gChunkList[startChunk].text)
                break;
        }
        if (*p == open_paren) {
            if (depth == 0) {
                long docOffset = 0;
                for (Word i = 0; i < startChunk; i++)
                    docOffset += gChunkList[i].length;
                setSelectionFromCharacterOffsetRange(docOffset + (p - gChunkList[startChunk].text), 1);
                return true;
            }
            else {
                depth--;
                p--;
            }
        }
        else {
            if (*p == close_paren) {
                depth++;
                p--;
            }
        }
    }
    return false;
}

Boolean findMatchingParen(Word searchStartLine, Word searchStartOffset, char open_paren, char close_paren)
{
    Word startOffset;
    Word startChunk;
    getChunkAndOffset(searchStartLine, searchStartOffset, &startChunk, &startOffset);

    int depth = 0;
    char *p = gChunkList[startChunk].text + startOffset;

    while (true) {
        while (*p && (*p != open_paren) && (*p != close_paren)) p++;
        if (*p == close_paren) {
            if (depth == 0) {
                long docOffset = 0;
                for (Word i = 0; i < startChunk; i++)
                    docOffset += gChunkList[i].length;
                setSelectionFromCharacterOffsetRange(docOffset + (p - gChunkList[startChunk].text), 1);
                return true;
            }
            else {
                depth--;
                p++;
            }
        }
        else {
            if (*p == open_paren) {
                depth++;
                p++;
            }
            else {
                startChunk++;
                if (startChunk == gTopChunk) break;
                p = gChunkList[startChunk].text;
            }
        }
    }
    return false;
}

static char *findStr(char *searchText, char *sourceText, char *moreSource, Boolean ignoreCase)
{
    while (*sourceText) {

        char *s = sourceText;
        char *f = searchText;

        Boolean inMoreSource = false;

        while (*f) {
            if (*s == 0) {
                if (inMoreSource) break;
                inMoreSource = true;
                s = moreSource;
            }
            if (*s != *f) {
                if (ignoreCase) {
                    if (toLower[*s] != toLower[*f])
                        break;
                }
                else
                    break;
            }
            s++;
            f++;
        }
        if (*f == 0) return sourceText;

        sourceText++;
    }
    return NULL;
}

Boolean search(Word searchStartLine, Word searchStartOffset, char *searchText, Boolean ignoreCase, Boolean beep)
{
    Word startOffset;
    Word startChunk;
    getChunkAndOffset(searchStartLine, searchStartOffset, &startChunk, &startOffset);
    long docOffset = 0;
    for (Word i = 0; i < startChunk; i++)
        docOffset += gChunkList[i].length;

    while (1) {
        char *p = gChunkList[startChunk].text + startOffset;
        char *result = findStr(searchText, p, gChunkList[startChunk + 1].text, ignoreCase);
        if (result) {
            setSelectionFromCharacterOffsetRange(docOffset + (result - gChunkList[startChunk].text),
                                                    StrLen(searchText));
            return true;
        }
        else {
            docOffset += gChunkList[startChunk].length;
            startChunk++;
            if (startChunk == gTopChunk) break;
            startOffset = 0;
        }
    }
    if (beep){
        //Task (S038), play a beep if no findy, revision by Nick
        SndCommandType snd;
        snd.cmd = sndCmdNoteOn;
        snd.param1 = BEEP_PITCH; //pitch==high treble g
        snd.param2 = BEEP_DURATION;  //500 half a sec should do
        snd.param3 = (UInt16)PrefGetPreference(prefSysSoundVolume);  // #869072 let the Beeping follow SysPrefs
        // snd.param3=120;

        SndDoCmd(NULL,&snd,false);
    }
    return false;
}

Boolean saveDocFile(CharPtr dbName, Boolean compressFlag)
{

    LocalID dbID = DmFindDatabase(0, dbName);
    if (dbID == 0) {
        error("Couldn't find DOC file");
        return false;
    }

    DmOpenRef docDB = DmOpenDatabase(0, dbID, dmModeReadWrite);
    if (docDB == 0) {
        Err err;
        err = DmGetLastErr();
        if(err==dmErrReadOnly || err==dmErrROMBased)
            return false;
        error("Couldn't open DOC file");
        return false;
    }

    UInt16 numRecords = DmNumRecords(docDB);

    // #621840 (S032) SrcEdit document categories
    UInt16 recAttr;
    if (numRecords >0)
      DmRecordInfo(docDB, 0, &recAttr, NULL, NULL);

  for (UInt16 i = 0; i <numRecords; i++)
         DmRemoveRecord(docDB, 0);

    UInt at = 0;
    Handle headerH = (Handle)DmNewRecord(docDB, &at, sizeof(DOCHeader));
    if (headerH == NULL) {
        error("Couldn't create header record");
        return false;
    }
    at++;
/*
    if (compressFlag) {

    }
    else {
*/
        for (Word i = 0; i < gTopChunk; i++) {
            if (gChunkList[i].kind == CollapsedText) {
                gChunkList[i].kind = PlainText;
                gChunkList[i].length = gChunkList[i].collapsedLength;
                gChunkList[i].depth = 0;
            }
        }

        while (stitchBlocks()) ;

        long totalSize = 0;

        if (compressFlag){
            for (Word i = 0; i < gTopChunk; i++) {
                if (compressFlag){
                    gChunkList[i].kind = PlainText;
                    totalSize += gChunkList[i].length;
                    gChunkList[i].length = compress((unsigned char*)gChunkList[i].text, gChunkList[i].collapsedLength);
                    gChunkList[i].depth = 0;

                }
            }
            error("Compression: Done!");
        }

        Word currentChunk = 0;
        Word bytesUsed = 0;     // from the current chunk
        while (currentChunk < gTopChunk) {
            Handle h = (Handle)DmNewRecord(docDB, &at, DOC_RECORD_LENGTH);
            if (h == NULL) {
                error("Couldn't create doc record");
                return false;
            }
            long bytesWritten = 0;  // to the current record
            CharPtr p = (CharPtr)MemHandleLock(h);
            while (bytesWritten < DOC_RECORD_LENGTH) {
                Word bytesToWrite = DOC_RECORD_LENGTH - bytesWritten;
                if (bytesToWrite > (gChunkList[currentChunk].length - bytesUsed))
                    bytesToWrite = (gChunkList[currentChunk].length - bytesUsed);
                DmWrite(p, bytesWritten, gChunkList[currentChunk].text + bytesUsed, bytesToWrite);
                bytesWritten += bytesToWrite;
                if(compressFlag == false)
                    totalSize += bytesToWrite;
                bytesUsed += bytesToWrite;
                if (bytesUsed == gChunkList[currentChunk].length) {
                    currentChunk++;
                    if (currentChunk == gTopChunk) break;
                    bytesUsed = 0;
                }
            }
            MemHandleUnlock(h);
            DmResizeRecord(docDB, at, bytesWritten);
            DmReleaseRecord(docDB, at, true);
            at++;
        }


        DOCHeader docHdr = {compressFlag+1, 0, totalSize, (at - 1), MAX_CHUNK_LENGTH, 0 };

        DmWrite(MemHandleLock(headerH), 0, &docHdr, sizeof(DOCHeader));
        MemHandleUnlock(headerH);

        DmReleaseRecord(docDB, 0, true);
    // }
    // #621840 (S032) SrcEdit document categories
    if (DmNumRecords(docDB) >0){
        //LocalID uniqueID;
        //DmRecordInfo(docDB, 0, NULL, &uniqueID, NULL);

        DmSetRecordInfo(docDB, 0, &recAttr, NULL);
    }
    DmCloseDatabase(docDB);

    return true;
}

static void convertOffsetToLineOffset(Word offset, Word *line, Word *lineOffset)
{

    Word docOffset = 0;
    for (Word i = 0; i < gTopChunk; i++) {
        if (offset < (docOffset + gChunkList[i].length)) {
            for (Word j = 0; j < gMaxLines; j++) {      // a faster mapping from chunk to first line
                                                        // in chunk would be good.
                if (gLineStarts[j].chunkNumber == i) {
                    offset -= docOffset;    // just a chunk offset now
                    while ((j < (gMaxLines - 1)) && (offset >= gLineStarts[j + 1].offset)) j++;
                    *line = j;
                    *lineOffset = offset - gLineStarts[j].offset;
                    return;
                }
            }
        }
        docOffset += gChunkList[i].length;
    }
/*
    int lineLength = 0;
    Word docOffset = 0;
    for (int j = 0; j < gMaxLines; j++) {
        lineLength = getLineLength(j);
        if (offset < (docOffset + lineLength)) {
            *line = j;
            *lineOffset = offset - docOffset;
            return;
        }
        docOffset += lineLength;        // the <CR>
    }
    *line = gMaxLines - 1;
    *lineOffset = lineLength;
*/
    *line = gMaxLines - 1;
    *lineOffset = getLineLength(gMaxLines - 1);
}

void setSelectionFromCharacterOffsetRange(Word offset, Word length)
{
    convertOffsetToLineOffset(offset, &gSelectionLine, &gSelectionOffset);
    convertOffsetToLineOffset(offset + length, &gSelectionEndLine, &gSelectionEndOffset);
    gHasSelection = true;
}

Boolean findChar(Word searchStartLine, Word searchStartOffset, char t)
{
    Word startOffset;
    Word startChunk;
    getChunkAndOffset(searchStartLine, searchStartOffset, &startChunk, &startOffset);
    if (startChunk == gTopChunk) return false;

    char *p = gChunkList[startChunk].text + startOffset;

    while (true) {
        while (*p && (*p != t)) p++;
        if (*p == t) {
            long docOffset = 0;
            for (Word i = 0; i < startChunk; i++)
                docOffset += gChunkList[i].length;
            setSelectionFromCharacterOffsetRange(docOffset + (p - gChunkList[startChunk].text), 1);
            return true;
        }
        else {
            startChunk++;
            if (startChunk == gTopChunk) break;
            p = gChunkList[startChunk].text;
        }
    }
    return false;
}

Boolean findCharBackwards(Word searchStartLine, Word searchStartOffset, char t, Word maxDistance)
{
    Word startOffset;
    Word startChunk;
    getChunkAndOffset(searchStartLine, searchStartOffset, &startChunk, &startOffset);

    char *p = gChunkList[startChunk].text + startOffset - 1;

    while (true) {
        if (p < gChunkList[startChunk].text) {
            if (startChunk == 0) break;
            startChunk--;
            p = gChunkList[startChunk].text + gChunkList[startChunk].length - 1;
        }

        while (*p != t) {
            p--;
            if (--maxDistance <= 0) return false;
            if (p < gChunkList[startChunk].text)
                break;
        }
        if (*p == t) {
            long docOffset = 0;
            for (Word i = 0; i < startChunk; i++)
                docOffset += gChunkList[i].length;
            setSelectionFromCharacterOffsetRange(docOffset + (p - gChunkList[startChunk].text), 1);
            return true;
        }
    }
    return false;
}

static char *backup(char *p, Word *chunk)
{
    int tc = *chunk;
    if (p > gChunkList[tc].text)
        return --p;
    if (tc == 0) {
        *chunk = (Word)-1;
        return p;
    }
    tc--;
    *chunk = tc;
    return gChunkList[tc].text + gChunkList[tc].length - 1;
}
/*

Not used for now... Uncomment before using this !
static char *forward(char *p, Word *chunk)
{
    Word tc = *chunk;
    if (p < (gChunkList[tc].text + gChunkList[tc].length))
        return ++p;
    tc++;
    if (tc == gTopChunk) {
        *chunk = (Word)-1;
        return p;
    }
    *chunk = tc;
    return gChunkList[tc].text;
}
*/

Boolean isWordLetter(char ch)
{
    return ((ch >= 'A') && (ch <= 'Z'))
                                || ((ch >= '0') && (ch <= '9'))
                                || ((ch >= 'a') && (ch <= 'z'))
                                || (ch == '_');
}

Boolean isWhitespace(char ch)
{
    return ((ch == ' ') || (ch == 0x0D) || (ch == 0x0A) || (ch == 0x09));
}

Boolean findPreviousWord(Word searchStartLine, Word searchStartOffset)
{
    Word startOffset;
    Word startChunk;
    getChunkAndOffset(searchStartLine, searchStartOffset, &startChunk, &startOffset);

    char *p = gChunkList[startChunk].text + startOffset - 1;

    while (true) {
        while (isWhitespace(*p)) {
            p = backup(p, &startChunk);
            if (startChunk == (Word)-1) return false;
        }
        if (isWordLetter(*p)) {
            char *wordEnd = p;
            Word wordEndChunk = startChunk;

            while (isWordLetter(*p)) {
                p = backup(p, &startChunk);
                if (startChunk == (Word)-1) break;
            }
            if (startChunk != (Word)-1) p++;

            long docOffset = 0;
            for (Word i = 0; i < startChunk; i++)
                docOffset += gChunkList[i].length;
            convertOffsetToLineOffset(docOffset + (p - gChunkList[startChunk].text), &gSelectionLine, &gSelectionOffset);

            docOffset = 0;
            for (Word i = 0; i < wordEndChunk; i++)
                docOffset += gChunkList[i].length;
            convertOffsetToLineOffset(docOffset + (wordEnd - gChunkList[wordEndChunk].text), &gSelectionEndLine, &gSelectionEndOffset);
            gHasSelection = true;
            return true;
        }
        else
            break;
    }
    return false;
}

/***********************************************

    DEBUG

************************************************/

void buildLineStartsList(ListPtr lst)
{
    char **list = (char **)MemPtrNew(sizeof(char *) * gMaxLines);
    for (Word i = 0; i < gMaxLines; i++)
    {
        char buffer[64];
        StrPrintF(buffer, "Chunk %d, offset %d", gLineStarts[i].chunkNumber, gLineStarts[i].offset);
        list[i] = (char *)MemPtrNew(StrLen(buffer) + 1);
        StrCopy(list[i], buffer);
    }
    LstSetListChoices(lst, list, gMaxLines);
}









