/* -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=- */
/*                                                                   */
/*    AV3D - OS-Independent wrapper with OpenGL support              */
/*    Copyright (c)2001 Galactica Software                           */
/*    ALL RIGHTS RESERVED.                                           */
/*                                                                   */
/*    Programmers: Adam Majer (adamm@galacticasoftware.com)          */
/*                                                                   */
/*    This source code and related compilations are distributed      */
/*    under the terms of LPGL. For license see "license.txt" enclosed*/
/*    or go to "http://www.gnu.org/copyleft/lesser.html"             */
/*
    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
/*                                                                   */
/* -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.- */

#include <av3d.hpp>
#include <math.h>

////////////////////////////////////////////////////////////
AV_Camera :: AV_Camera()
{
  ZeroAll();

  Matrix[12 + 0] =  Matrix[12 + 1] = Matrix[12 + 2] = 0;
  Matrix[12+ 3] = 1;
  Matrix[0+3] = Matrix[4+3] = Matrix[8+3] = 0; // no scalling
}
////////////////////////////////////////////////////////////
void AV_Camera::CalculateNewMatrix()
{
  fNewMatrix = false;
  
  double TX = RQuat[1]*RQuat[1];
  double TY = RQuat[2]*RQuat[2];
  double TZ = RQuat[3]*RQuat[3];
  double TQ = TY+TZ;
  double TK;
  
  if (TQ + TX + RQuat[0]*RQuat[0] != 0) TK = 2 / (TQ + TX + RQuat[0]*RQuat[0]);
  else TK = 0;
  Matrix[0 + 0]   =        1 - TK*TQ;
  Matrix[4 + 1]   =        1 - TK*(TX + TZ);
  Matrix[8 + 2]   =        1 - TK*(TX + TY);
  TX              =       TK*RQuat[1];
  TY              =       TK*RQuat[2];
  TQ              =       (TK*RQuat[3])*RQuat[0]; 
  TK              =       TX*RQuat[2];
  Matrix[4 + 0]   =       TK - TQ;
  Matrix[0 + 1]   =       TK + TQ;
  TQ              =       TY*RQuat[0];
  TK              =       TX*RQuat[3];
  Matrix[8 + 0]   =       TK+TQ;
  Matrix[0 + 2]   =       TK-TQ;
  TQ              =       TX*RQuat[0];
  TK              =       TY*RQuat[3];
  Matrix[8 + 1]   =       TK - TQ;
  Matrix[4 + 2]   =       TK + TQ;
}    
////////////////////////////////////////////////////////////
void AV_Camera::RotateAngle1(Direction dir, double Offset)
{
  /* Rotates to the new coordinates in a specific direction */
  
  // gets Xi and Theta angles of rotation from the vector
  // FIXME: make table lookup here for faster calculations - resolution <1deg (1/512 revs
  
  //  if(Offset < 0) Offset += 512.0;
  //  int A = static_cast<int>(Offset / 360.0 * 512.0)&511;
  double A = Offset / 360.0 * M_PI;
  double Q[4];
  double s = sin(A);
  
  Q[0] = cos(A);

  if(dir & VERTICAL)
    // rotates the vector up along the plane that marks vertical in FOV
    Q[1] = s;
  else Q[1] = 0;

  if(dir & HORIZONTAL)
    // rotates the vector in the horizontal directiony
    Q[2] = s;
  else Q[2] = 0;
  
  if(dir & ROLL)
    // rotates along the new z axis -- easiest
    Q[3] = s;
  else Q[3] = 0;
  

  // computes the new quatrilion
  double Q1, Q2, Q3, Q4;
  
  Q1  =  Q[0] * RQuat[0] -  Q[1] * RQuat[1] -  Q[2] * RQuat[2] -  Q[3] * RQuat[3];
  Q2  =  Q[1] * RQuat[0] +  Q[0] * RQuat[1] -  Q[3] * RQuat[2] +  Q[2] * RQuat[3];
  Q3  =  Q[2] * RQuat[0] +  Q[3] * RQuat[1] +  Q[0] * RQuat[2] -  Q[1] * RQuat[3];
  Q4  =  Q[3] * RQuat[0] -  Q[2] * RQuat[1] +  Q[1] * RQuat[2] +  Q[0] * RQuat[3];
  
  // normalizes the rotation
  double nor = Q1*Q1 + Q2*Q2 + Q3*Q3 + Q4*Q4;
  Q1/=nor; Q2/=nor; Q3/=nor; Q4/=nor;
  
  RQuat[0] = Q1;   RQuat[1] = Q2;
  RQuat[2] = Q3;   RQuat[3] = Q4;
  
  fNewMatrix = true;
  
  // calculates new "normal" vectors
  N[0] = 2.0*Q2*Q4 - 2.0*Q1*Q3;
  N[1] = 2.0*Q3*Q4 + 2.0*Q1*Q2;
  N[2] = Q4*Q4 + Q1*Q1 - Q3*Q3 - Q2*Q2;
  
  Y[0] = - 2.0*Q2*Q3 - 2.0*Q1*Q4;
  Y[1] = - Q1*Q1 - Q3*Q3 + Q2*Q2 + Q4*Q4;
  Y[2] = + 2.0*Q1*Q2 - 2.0*Q3*Q4;
  
  X[0] = Q2*Q2 + Q1*Q1 - Q4*Q4 - Q3*Q3;
  X[1] = 2.0*Q2*Q3 - 2.0*Q1*Q4;
  X[2] = 2.0*Q2*Q4 + 2.0*Q1*Q3;

  // normalizes all of these new "AXIS"
  /*
    nor = (N[0]*N[0] + N[1]*N[1] + N[2]*N[2]); 
    N[0]/=nor; N[1]/=nor; N[2]/=nor;
    
    nor = (Y[0]*Y[0] + Y[1]*Y[1] + Y[2]*Y[2]); 
    Y[0]/=nor; Y[1]/=nor; Y[2]/=nor;
    
    nor = (X[0]*X[0] + X[1]*X[1] + X[2]*X[2]); 
    X[0]/=nor; X[1]/=nor; X[2]/=nor;
  */
}
