/* -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=- */
/*                                                                   */
/*    AV3D - OS-Independent wrapper with OpenGL support              */
/*    Copyright (c)2001 Galactica Software                           */
/*    ALL RIGHTS RESERVED.                                           */
/*                                                                   */
/*    Programmers: Adam Majer (adamm@galacticasoftware.com)          */
/*                                                                   */
/*    This source code and related compilations are distributed      */
/*    under the terms of LPGL. For license see "license.txt" enclosed*/
/*    or go to "http://www.gnu.org/copyleft/lesser.html"             */
/*
    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
/*                                                                   */
/* -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.- */


#ifndef __AV3D_H
#define __AV3D_H

#include <stdio.h>
#include <iostream.h>
#include <key.h>

//#ifdef __MAIN_INTERFACE
//#define EXTERN
//#else
#define EXTERN extern 
//#endif

#ifdef __X11
 #include <mikmod.h>
#endif

#ifdef __WINDOWS
 #include <libmikmod_win32.h>
#endif

/******     OS Specific Defines    ********/
#if !defined(__WINDOWS) && !defined(__X11)
#error "Must define a target - Supported [__WINDOWS, __X11]"
#endif

/* You may also define __RTL__ if you want to use Real Time Linux
 */

/******** Platform specific defines *******/
#if defined(___DIRECTX) || defined(__WINDOWS)
#define DIRECTINPUT_VERSION  0x0700
#include <windows.h>
#include <dinput.h>   // this includes keyboard scancodes in DirectX
namespace _av3d_{
  extern LPDIRECTINPUT7 DirectInput;
};
extern HINSTANCE hInstance;
#endif // platform specific defines

// includes the main OpenGL header
#include <GL/gl.h>

/*******      Global Functions     ********/
class AV_ZBuffer; class AV_Video;

/*******     Global Structures     ********/
struct AV_FontStruct{
  const char *FontFace;
  int Size, Width;               // size in pixels
  bool fItalic, fBold;
};

/*******  Font Classes  ********/
struct OpenGLData; struct Font2Struct;
class AV_Font{
  friend AV_Video;
protected:
  AV_Video &V;
  Font2Struct *Data;
  
  void Write(int,int,const char*,int);
  void WriteFormatted(int,int,const char*,int);
  
  AV_Font(AV_Video&);
public:
  AV_Font(AV_Font&);
  AV_Font(AV_Video&, AV_FontStruct&);
  virtual ~AV_Font();
};

#define N_STEPS 0xFF
/*******     Video Mode Class    ********/
struct OpenGLData;
class AV_Video{
  
protected:
  OpenGLData *Data;
  bool fFullScreen;
  int w,h,d;
  AV_Font *Font;
  bool fSet;
  
public:
  AV_Video(const char*, int, int, int, bool=false);
  ~AV_Video();

  void ClearScreen();
  void FlipPages();
  int  GetHeight(){return w;};
  int  GetWidth(){return h;};
  void RestoreVideoMode();
  void SetFont(AV_Font&);
  void SetVideoMode(const char*, int, int, int, bool=false);
  
  void Write(int, int, const char *String, int Length=0);
  void WriteFormatted(int, int, const char *String, int Length=0);

  OpenGLData const* GetPrivateData(); // returns OS specific data to a friend class
};

/*******   3D Class Definitions    ********/
class AV_Camera{  // rotation of --- ALL in degrees
private:
  GLdouble Matrix[16];  // used in OpenGL
  GLdouble RQuat[4];    // keeps track of rotation
  double x,y,z;         // translation from the origin

  double N[3], X[3], Y[3];    // current "normal", x and y vectors [normal into screen]
  bool fNewMatrix;      // true if new matrix from Quadtriolion is to be calculated

  void CalculateNewMatrix();
public:
  enum Direction{VERTICAL=0x01, HORIZONTAL=0x02, ROLL=0x04};
  AV_Camera();

  void Translate(double dx, double dy, double dz){x-=dx, y-=dy, z-=dz;};
  void ZeroAll(){
    x=y=z=0; N[0]=N[1]=X[2]=X[1]=Y[0]=Y[2]=0;
    N[2]=X[0]=Y[1]=1;
    RQuat[1] = RQuat[2] = RQuat[3] = 0;
    RQuat[0] = 1;
    fNewMatrix = true;
  };
  
  // rotates the FOV in one specific direction - this is "first person" rotation
  void RotateAngle1(Direction, double Offset);

  // translates rotated FOV
  void TranslateR(double dx, double dy, double dz){
    if(dz!=0){// forward movement
      z+=dz*N[2]; y+=dz*N[1]; x+=dz*N[0];
    }if(dy!=0){// vertical movement
      z-=dy*Y[2]; y-=dy*Y[1]; x-=dy*Y[0];
    }if(dx!=0){// horizontal movement
      z-=dx*X[2]; y-=dx*X[1]; x-=dx*X[0];
    }
  };
  
  void Go(){
    if(fNewMatrix)
      CalculateNewMatrix();
    // loads the matrix MODEL_VIEW matix into GL
    glLoadMatrixd(Matrix);
    glTranslated(x,y,z);
  };
};

/*******     Interface Classes     ********/
typedef int ControllerData;
typedef bool(* EnumControllerProc)(ControllerData, int, const char*, const char*);

// Type allowed
#define MOUSE      0x0001
#define JOYSTICK   0x0002
#define KEYBOARD   0x0003
#define GAMEPAD    0x0004
#define WHEEL      0x0005
#define UNKNOWN    0x0006

// time in ms between updates on devices
#define MIN_INTERFACE_UPDATE_TIME 50

struct ButtonStatus{
  bool KeyStatus[256];
};

struct JoystickState{
  int  x, y, z;
  int  Rx, Ry, Rz;
  int  Sliders[2];
  int  POVs[4];
  bool Buttons[256];
};

struct MouseState{
  int  x,y,z;   // z is usually wheel
  bool Buttons[256];
};

class AV_Controller{
protected:
  AV_Video &V;
public:

  enum Controller_Status{OK, FAIL, NODRIVER_OR_DEVICE};

  AV_Controller(AV_Video&);
  virtual ~AV_Controller();
  
  virtual void Center()=0;
  virtual ButtonStatus& GetButtons()=0;
  virtual int  GetX()=0;
  virtual int  GetY()=0;
  virtual void Reset()=0;
};

struct JoyStruct;
class AV_Joystick : public AV_Controller{
protected:
  JoyStruct *js;

  void ___GetState();
public:
  AV_Joystick(AV_Video&, ControllerData nJoy);
  ~AV_Joystick();
  
  virtual void Center();
  virtual ButtonStatus& GetButtons();
  virtual int  GetX();
  virtual int  GetY();
  virtual void Reset();
  
  static int EnumDevices(EnumControllerProc);
  const JoystickState& GetState();
};

struct KeyboardStruct;
class AV_Keyboard : public AV_Controller{
private:
  KeyboardStruct *kb; // used in windows ONLY
  void __GetState();
public:
  AV_Keyboard(AV_Video&);
  virtual ~AV_Keyboard();

  virtual void Center();
  virtual ButtonStatus& GetButtons();
  virtual int  GetX();
  virtual int  GetY();
  virtual void Reset();
};

struct MouseStruct;
class AV_Mouse : public AV_Controller{
private:
  MouseStruct *ms;

  void __GetState();
public:
  AV_Mouse(AV_Video&);
  virtual ~AV_Mouse();
  
  virtual void Center();
  virtual ButtonStatus& GetButtons();
  virtual int  GetX();
  virtual int  GetY();
  virtual void Reset();

  const MouseState& GetState();
};

/********  SFX & Music Interface    ********/
class AV_MusicMixer{
public:
  virtual ~AV_MusicMixer(){};

  virtual void MixMusic(short int* Buffer, int Len)=0;
  virtual bool fStop()=0;
};

class AV_ModuleMixer : public AV_MusicMixer{
public:
  // Supported formats.....
  enum AV_ModuleType{AV_S3M, AV_669, AV_AMF, AV_DSM, AV_FAR, AV_GDM, AV_MED, AV_M15, AV_MOD,AV_MTM, AV_OKT, AV_STM, AV_STX, AV_ULT};
protected:
  void *Data;

  void CheckVersion();
  void CreateNewThread(void *);
  bool RegisterLoader(AV_ModuleType);
public:
  AV_ModuleMixer(char *fn, AV_ModuleType);
  AV_ModuleMixer(FILE *, AV_ModuleType);
  virtual ~AV_ModuleMixer();
  
  virtual void MixMusic(short int*, int){};
  virtual bool fStop();
};

typedef struct{
  SAMPLE Sample;
  SBYTE  VoiceNumber;
}AV_Sample;

class AV_Sound{
private:
  AV_Camera &Camera;
  int MaxChannels;

  AV_MusicMixer *Music;
public:
  AV_Sound(AV_Video&, AV_Camera&, int Channels=32);
  ~AV_Sound();
  
  void PlayMusic(const char *); // plays a S3M or MP3 file
  int SFX(SAMPLE *Data, int x=0, int y=0, int z=0);
  void Stop(int); // stop a particular sound = -1 means music
  void StopAll(); // stops all sounds! - even music!

  SAMPLE* LoadSample(FILE *in){return Sample_LoadFP(in);};
  int  GetSampleFrequency(SAMPLE *s){return s->speed;};
  void ChangeSampleFrequency(SAMPLE *s, int f){s->speed=f;};
  void FreeSample(SAMPLE *s){Sample_Free(s);};
};

/*******    Miscelenous Classes    ********/
class AV_Timer{
private:
  static int Count; // used internally
  int Time;
public:
  AV_Timer();
  ~AV_Timer();

  unsigned long GetTime();
  void ResetFrame();
};

/*******      Exception Class      ********/
class AV_Exception{
public:
  enum Av_ExceptionNo{
    OUT_OF_MEMORY,
    CANT_INITIALIZE_DEVICE,
    NO_SOUND,

    /* Module problems */
    BAD_MODULE_TYPE,
    CANT_LOAD_MODULE,
    BAD_MIKMOD_VERSION,

    /* Timer Problems */
    CANNOT_INITIALIZE_TIMER,

    /* Keyboard problems */
    CANNOT_GRAB_KEYBOARD,

    END_OF_EXCEPTIONS
  };

private:
  Av_ExceptionNo a;

public:
  AV_Exception(Av_ExceptionNo q){a=q;};
  AV_Exception(AV_Exception const&e){a=e.a;};

  const char*operator()();
};

/*******     Global Variables      ********/
EXTERN AV_Timer          GlobalTimer;

/*******     Global Functions      ********/
//int AV3D_Main(const char *args);
ostream& operator<<(ostream&, AV_Exception&);

/*******   User Defined Functions  ********/
extern "C"{
  bool AV3D_InitProc();
  bool AV3D_MainLoop();
  bool AV3D_EndProc();
};

#endif
