/* -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=- */
/*                                                                   */
/*    AV3D - OS-Independent wrapper with OpenGL support              */
/*    Copyright (c)2001 Galactica Software                           */
/*    ALL RIGHTS RESERVED.                                           */
/*                                                                   */
/*    Programmers: Adam Majer (adamm@galacticasoftware.com)          */
/*                                                                   */
/*    This source code and related compilations are distributed      */
/*    under the terms of LPGL. For license see "license.txt" enclosed*/
/*    or go to "http://www.gnu.org/copyleft/lesser.html"             */
/*
    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
/*                                                                   */
/* -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.- */

#include <windows.h>
#include <gl/gl.h>
#pragma hdrstop
#include <av3d.hpp>

struct JoyStruct{
  _GUID JoyNum;
  LPDIRECTINPUTDEVICE7 JoyDev;
  JoystickState State;
};

extern HINSTANCE hInstance;
namespace __AV_Joystick_namespace{
  int nDevices;
}

struct OpenGLData{
  HDC    hDC;
  HGLRC  hRC;
  HWND   hWnd;
};

////////////////////////////////////////////////////////////
BOOL CALLBACK ___InternalJoyCallback(LPCDIDEVICEINSTANCE lpdi, LPVOID proc)
{
  EnumControllerProc EnumProc = (EnumControllerProc)proc;
  
  int Type;
  if(((lpdi->dwDevType>>8)&0xFF) == DIDEVTYPEJOYSTICK_GAMEPAD)
    Type = GAMEPAD;
  else if(((lpdi->dwDevType>>8)&0xFF) == DIDEVTYPEJOYSTICK_WHEEL)
    Type = WHEEL;
  else
    Type = JOYSTICK;

  ControllerData data = (int)new int[sizeof(_GUID)/sizeof(int)+1];
  memcpy((void*)data, &lpdi->guidInstance, sizeof(_GUID));
  __AV_Joystick_namespace::nDevices ++;
  bool ret = EnumProc(data, Type, lpdi->tszInstanceName, 
		      lpdi->tszProductName);
  //delete[]data;
  return ((ret)?(DIENUM_CONTINUE):(DIENUM_STOP));
}
////////////////////////////////////////////////////////////
AV_Joystick :: AV_Joystick(AV_Video &v, ControllerData nJoy)
  : AV_Controller(v)
{
  // allocates structure
  js = new JoyStruct;
  memcpy(&js->JoyNum, (void*)nJoy, sizeof(_GUID));

  // creates handle
  if(_av3d_::DirectInput->CreateDeviceEx(js->JoyNum, IID_IDirectInputDevice7,
					 (void**)&js->JoyDev, NULL) != DI_OK)
    throw AV_Exception(AV_Exception::CANT_INITIALIZE_DEVICE);

  js->JoyDev->SetDataFormat(&c_dfDIJoystick);
  js->JoyDev->SetCooperativeLevel(v.GetPrivateData()->hWnd, DISCL_NONEXCLUSIVE | DISCL_FOREGROUND);
 
  DIPROPRANGE diprg; 
  
  diprg.diph.dwSize       = sizeof(diprg); 
  diprg.diph.dwHeaderSize = sizeof(diprg.diph); 
  diprg.diph.dwObj        = 0;
  diprg.diph.dwHow        = DIPH_DEVICE; 
  diprg.lMin              = -1000; 
  diprg.lMax              = +1000; 
  js->JoyDev->SetProperty(DIPROP_RANGE, &diprg.diph);

  int i;
  for(i=0;i<256;++i)
    js->State.Buttons[i] = false;
}
////////////////////////////////////////////////////////////
AV_Joystick :: ~AV_Joystick()
{
  // releases joystick
  js->JoyDev->Unacquire();
  js->JoyDev->Release();
  delete js;
}
////////////////////////////////////////////////////////////
void AV_Joystick :: Center()
{
  // does nothin here
}
////////////////////////////////////////////////////////////
int AV_Joystick :: EnumDevices(EnumControllerProc EnumProc)
{
  /* Enumerates all of the joystick devices until EnumProc 
   * returns FALSE! Or until no more devices are to be
   * enumerated -- then EnumDevices returns the total number of
   * devices that were enumerated by EnumProc.
   */
  
  // enumerates all devices here
  __AV_Joystick_namespace::nDevices = 0;
  _av3d_::DirectInput->EnumDevices(DIDEVTYPE_JOYSTICK, ___InternalJoyCallback,
				   EnumProc,
				   DIEDFL_ATTACHEDONLY);
  return __AV_Joystick_namespace::nDevices;
}
////////////////////////////////////////////////////////////
ButtonStatus& AV_Joystick :: GetButtons()
{
  ___GetState();
  return *(ButtonStatus*)js->State.Buttons;
}
////////////////////////////////////////////////////////////
void AV_Joystick :: ___GetState()
{
  /* Gets current position of the joystick device.. This is
   * updated max. 20 times per second to speed up things.
   */

  DIJOYSTATE JoyState;

  switch(js->JoyDev->Poll())
    {
    case DIERR_NOTACQUIRED:
    case DIERR_INPUTLOST:
      js->JoyDev->Acquire();
      break;
    case DI_NOEFFECT:
    case DI_OK:
      js->JoyDev->GetDeviceState(sizeof(DIJOYSTATE), &JoyState);

      // stores data in joystick memory
      js->State.x = JoyState.lX;
      js->State.y = JoyState.lY;
      js->State.z = JoyState.lZ;

      js->State.Rx = JoyState.lRx;
      js->State.Ry = JoyState.lRy;
      js->State.Rz = JoyState.lRz;

      js->State.Sliders[0] = JoyState.rglSlider[0];
      js->State.Sliders[1] = JoyState.rglSlider[1];

      int i;
      for(i=0;i<4;++i)
	js->State.POVs[i] = JoyState.rgdwPOV[i];
      for(i=0;i<32;++i)
	js->State.Buttons[i] = JoyState.rgbButtons[i]!=0;
    }
}
////////////////////////////////////////////////////////////
const JoystickState& AV_Joystick :: GetState()
{
  ___GetState();
  return js->State;
}
////////////////////////////////////////////////////////////
int AV_Joystick :: GetX()
{
  ___GetState();
  return js->State.x;
}
////////////////////////////////////////////////////////////
int AV_Joystick :: GetY()
{
  ___GetState();
  return js->State.y;
}
////////////////////////////////////////////////////////////
void AV_Joystick :: Reset()
{
  ___GetState();
}
