/* MikMod sound library
   (c) 1998, 1999, 2000 Miodrag Vallat and others - see file AUTHORS for
   complete list.
   
   This library is free software; you can redistribute it and/or modify
   it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of
   the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.
*/

/*==============================================================================
  
  $Id: drv_ds.c,v 1.2 2001/09/27 23:27:59 adamm Exp $
  
  Driver for output on win32 platforms using DirectSound
  
  ==============================================================================*/

/*
  
  Written by Brian McKinney <Brian.McKinney@colorado.edu>
  
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mikmod_internals.h"

#ifdef DRV_DS

#include <string.h>

#define INITGUID
#include <windows.h>
#include <mmsystem.h>
#include <dsound.h>

/* DSBCAPS_CTRLALL is not defined anymore with DirectX 7. Of course DirectSound
   is a coherent, backwards compatible API... */
#ifndef DSBCAPS_CTRLALL
#define DSBCAPS_CTRLALL ( DSBCAPS_CTRLPOSITIONNOTIFY | DSBCAPS_CTRLVOLUME | DSBCAPS_CTRLPAN | DSBCAPS_CTRLFREQUENCY | DSBCAPS_CTRL3D )
#endif

/* size of each buffer */
#define FRAGSIZE 14
/* buffer count */
#define UPDATES  32

static LPDIRECTSOUND pSoundCard;
static LPDIRECTSOUNDBUFFER pPrimarySoundBuffer,pSoundBuffer;
static LPDIRECTSOUNDNOTIFY pSoundBufferNotify;

static HANDLE notifyUpdateHandle,updateBufferHandle;
static DWORD updateBufferThreadID,soundBufferCurrentPosition,blockBytes1,blockBytes2;
static LPVOID pBlock1,pBlock2;
static DWORD sleepInterval;
volatile static BOOL threadInUse;
static int fragsize=1<<FRAGSIZE;
static DWORD controlflags = DSBCAPS_CTRLALL & ~DSBCAPS_GLOBALFOCUS;
HANDLE DRV_DS_Mutex; // prevents release of memory when playing etc....

#pragma argsused
static DWORD WINAPI updateBufferProc(LPVOID lpParameter)
{
  int t;
  LPWORD data;
  int LockSize;
  int LastWritePosition = fragsize;
  while(threadInUse) {
    WaitForSingleObject(notifyUpdateHandle, INFINITE);
    
    pSoundBuffer->lpVtbl->GetStatus(pSoundBuffer, &data);
    // Thread termination is controlled here -- DRV_DS_Mutex is locked
    // externally then this thread quits
    {
      DWORD Res;
      Res = WaitForSingleObject(DRV_DS_Mutex, 0);
      if(Res==WAIT_ABANDONED){
	ReleaseMutex(DRV_DS_Mutex);
	return 0;
      }
    }
    
    pSoundBuffer->lpVtbl->GetCurrentPosition
      (pSoundBuffer,&soundBufferCurrentPosition,NULL);
    
    t = LastWritePosition; // holds the initial lock buffer position
    if(soundBufferCurrentPosition < LastWritePosition){ // looped so lock to end
      LockSize = 2*fragsize - LastWritePosition + 1;
      LastWritePosition = 0;
    } else{
      LockSize = soundBufferCurrentPosition - LastWritePosition;
      LastWritePosition = soundBufferCurrentPosition;
    }
    
    if(pSoundBuffer->lpVtbl->Lock
       (pSoundBuffer,t,LockSize,&pBlock1,&blockBytes1,
	&pBlock2,&blockBytes2,0)==DSERR_BUFFERLOST) {	
      pSoundBuffer->lpVtbl->Restore(pSoundBuffer);
      pSoundBuffer->lpVtbl->Lock
	(pSoundBuffer,t,LockSize,&pBlock1,&blockBytes1,
	 &pBlock2,&blockBytes2,0);
    }
    MUTEX_LOCK(vars);
    VC_WriteBytes((SBYTE*)pBlock1,(ULONG)blockBytes1);
    if(pBlock2)
      VC_WriteBytes((SBYTE*)pBlock2,(ULONG)blockBytes2);
    MUTEX_UNLOCK(vars);
    
    pSoundBuffer->lpVtbl->Unlock
      (pSoundBuffer,pBlock1,blockBytes1,pBlock2,blockBytes2);

    ReleaseMutex(DRV_DS_Mutex);
  }
  return 0;
}

static void DS_CommandLine(CHAR *cmdline)
{
  CHAR *ptr=MD_GetAtom("buffer",cmdline,0);
  
  if(ptr) {
    int buf=atoi(ptr);
    
    if((buf<12)||(buf>19)) buf=FRAGSIZE;
    fragsize=1<<buf;
    
    free(ptr);
  }
  
  if ((ptr=MD_GetAtom("globalfocus",cmdline,1)) != NULL) {
    controlflags |= DSBCAPS_GLOBALFOCUS;
    free(ptr);
  } else
    controlflags &= ~DSBCAPS_GLOBALFOCUS;
}

static BOOL DS_IsPresent(void)
{
  if(DirectSoundCreate(NULL,&pSoundCard,NULL)!=DS_OK)
    return 0;
  if(pSoundCard){
    pSoundCard->lpVtbl->Release(pSoundCard);
    pSoundCard = NULL;
  }
  return 1;
}

static BOOL DS_Init(void)
{
  int i;
  DSBUFFERDESC soundBufferFormat;
  WAVEFORMATEX pcmwf;
  DSBPOSITIONNOTIFY positionNotifications[UPDATES];
  LPVOID Block1, Block2;
  DWORD lBlock1, lBlock2;
  
  if(DirectSoundCreate(NULL,&pSoundCard,NULL)!=DS_OK) {
    _mm_errno=MMERR_OPENING_AUDIO;
    fprintf(stderr, "Cannot open audio\n");
    return 1;
  }
  
  if(pSoundCard->lpVtbl->SetCooperativeLevel
     (pSoundCard,GetForegroundWindow(),DSSCL_PRIORITY)!=DS_OK) {
    _mm_errno=MMERR_DS_PRIORITY;
    fprintf(stderr, "Cooperative problem\n");
    return 1;
  }
  
  memset(&pcmwf,0,sizeof(PCMWAVEFORMAT));
  pcmwf.wFormatTag     =WAVE_FORMAT_PCM;
  pcmwf.nChannels      =(md_mode&DMODE_STEREO)?2:1;
  pcmwf.nSamplesPerSec =md_mixfreq;
  pcmwf.nBlockAlign    =4;
  pcmwf.nAvgBytesPerSec=pcmwf.nSamplesPerSec*pcmwf.nBlockAlign;
  pcmwf.wBitsPerSample =(md_mode&DMODE_16BITS)?16:8;
  
  memset(&soundBufferFormat,0,sizeof(DSBUFFERDESC));
  soundBufferFormat.dwSize       =sizeof(DSBUFFERDESC);
  soundBufferFormat.dwFlags      =DSBCAPS_PRIMARYBUFFER;
  soundBufferFormat.dwBufferBytes=0;                                
  soundBufferFormat.lpwfxFormat  =NULL; 
  
  if(pSoundCard->lpVtbl->CreateSoundBuffer
     (pSoundCard,&soundBufferFormat,&pPrimarySoundBuffer,NULL)!=DS_OK) {
    _mm_errno=MMERR_DS_BUFFER;
    fprintf(stderr, "Buffer error\n");
    return 1;
  }
  
  if(pPrimarySoundBuffer->lpVtbl->SetFormat
     (pPrimarySoundBuffer,&pcmwf)!=DS_OK) {
    _mm_errno=MMERR_DS_FORMAT;
    fprintf(stderr, "Format issue\n");
    return 1;
  }
  pPrimarySoundBuffer->lpVtbl->Play(pPrimarySoundBuffer,0,0,DSBPLAY_LOOPING);
  
  memset(&pcmwf,0,sizeof(PCMWAVEFORMAT));
  pcmwf.wFormatTag     =WAVE_FORMAT_PCM;
  pcmwf.nChannels      =(md_mode&DMODE_STEREO)?2:1;
  pcmwf.nSamplesPerSec =md_mixfreq;
  pcmwf.nBlockAlign    =4;
  pcmwf.nAvgBytesPerSec=pcmwf.nSamplesPerSec*pcmwf.nBlockAlign;
  pcmwf.wBitsPerSample =(md_mode&DMODE_16BITS)?16:8;
  
  sleepInterval=((fragsize)/pcmwf.nAvgBytesPerSec)*1000;
  
  memset(&soundBufferFormat,0,sizeof(DSBUFFERDESC));
  soundBufferFormat.dwSize       =sizeof(DSBUFFERDESC);
  soundBufferFormat.dwFlags      =controlflags|DSBCAPS_GETCURRENTPOSITION2 ;
  soundBufferFormat.dwBufferBytes=fragsize*2;
  soundBufferFormat.lpwfxFormat  =&pcmwf;
  
  if(pSoundCard->lpVtbl->CreateSoundBuffer
     (pSoundCard,&soundBufferFormat,&pSoundBuffer,NULL)!=DS_OK) {
    fprintf(stderr, "Secondary problem\n");
    _mm_errno=MMERR_DS_BUFFER;
    return 1;
  }

  // set secondary  buffer to 0 to create silence...
  if(pSoundBuffer->lpVtbl->Lock(pSoundBuffer, 0, 0, &Block1, &lBlock1, &Block2, &lBlock2, DSBLOCK_ENTIREBUFFER) == DS_OK){
    fprintf(stderr, "Secondary Locked\n");
    if(lBlock1 && Block1) ZeroMemory(Block1, lBlock1);
    //    if(lBlock2 && Block1) ZeroMemory(Block2, lBlock2);
    pSoundBuffer->lpVtbl->Unlock(pSoundBuffer, Block1, lBlock1, Block2, lBlock2);
  }
  
  pSoundBuffer->lpVtbl->QueryInterface
    (pSoundBuffer,&IID_IDirectSoundNotify,(LPVOID*)&pSoundBufferNotify);
  if(!pSoundBufferNotify) {
    _mm_errno=MMERR_DS_NOTIFY;
    fprintf(stderr, "Notify Error\n");
    return 1;
  }
  
  notifyUpdateHandle=CreateEvent
    (NULL,FALSE,FALSE,"libmikmod DirectSound Driver positionNotify Event");
  if(!notifyUpdateHandle) {
    fprintf(stderr, "Notify Event problem\n");
    _mm_errno=MMERR_DS_EVENT;
    return 1;
  }
  
  DRV_DS_Mutex = CreateMutex(NULL, FALSE, "libmikmod DirectSound Driver Mutes");
  if(!DRV_DS_Mutex){
    fprintf(stderr, "Mutex Error\n");
    _mm_errno=MMERR_DS_THREAD;
    return 1;
  }
  ReleaseMutex(DRV_DS_Mutex);

  
  updateBufferHandle=CreateThread
    (NULL,0,updateBufferProc,NULL,CREATE_SUSPENDED,&updateBufferThreadID);
  if(!updateBufferHandle) {
    _mm_errno=MMERR_DS_THREAD;
    fprintf(stderr, "Thread creation problem\n");
    return 1;
  }
  SetThreadPriority(updateBufferHandle, THREAD_PRIORITY_HIGHEST);
  //SetThreadPriorityBoost(updateBufferHandle, TRUE);

  memset(&positionNotifications,0,UPDATES*sizeof(DSBPOSITIONNOTIFY));
  for(i=0;i<UPDATES;++i){
    positionNotifications[i].dwOffset    = i*fragsize*2/UPDATES;
    positionNotifications[i].hEventNotify= notifyUpdateHandle;
  }
  if(pSoundBufferNotify->lpVtbl->SetNotificationPositions(pSoundBufferNotify,UPDATES,&positionNotifications)!=DS_OK) {
    _mm_errno=MMERR_DS_UPDATE;
    fprintf(stderr, "Update events problem\n");
    return 1;
  }
  
  if(VC_Init())
    return 1;
  
  return 0;
}

static void DS_PlayStop(void);

static void DS_Exit(void)
{
  DWORD statusInfo;

  DS_PlayStop();

  if(updateBufferHandle) {
    threadInUse = 0;
    //    ResumeThread(updateBufferHandle);
    //    WaitForSingleObject(updateBufferHandle, INFINITE);
    // ExitThread(updateBufferHandle);
    CloseHandle((HANDLE)updateBufferThreadID);
    CloseHandle(updateBufferHandle);
  }
  CloseHandle(notifyUpdateHandle);
  
  VC_Exit();
  
  if(pSoundBufferNotify){
    pSoundBufferNotify->lpVtbl->Release(pSoundBufferNotify);
    pSoundBufferNotify=NULL;
  }
  if(pSoundBuffer) {
    if(pSoundBuffer->lpVtbl->GetStatus(pSoundBuffer,&statusInfo)==DS_OK)
      if(statusInfo&DSBSTATUS_PLAYING)
	pSoundBuffer->lpVtbl->Stop(pSoundBuffer);
    pSoundBuffer->lpVtbl->Release(pSoundBuffer);
    pSoundBuffer=NULL;
  }
  
  if(pPrimarySoundBuffer) {
    if(pPrimarySoundBuffer->lpVtbl->GetStatus
       (pPrimarySoundBuffer,&statusInfo)==DS_OK)
      if(statusInfo&DSBSTATUS_PLAYING)
	pPrimarySoundBuffer->lpVtbl->Stop(pSoundBuffer);
    pPrimarySoundBuffer->lpVtbl->Release(pPrimarySoundBuffer);
    pPrimarySoundBuffer=NULL;
  }
  
  if(pSoundCard){
    pSoundCard->lpVtbl->Release(pSoundCard);
    pSoundCard = NULL;
  }
  
  

  pSoundCard=0;
  pPrimarySoundBuffer=pSoundBuffer=0;
  pSoundBufferNotify=0;
  
  notifyUpdateHandle=updateBufferHandle=0;
  updateBufferThreadID=soundBufferCurrentPosition=blockBytes1=blockBytes2=0;
  pBlock1=pBlock2=0;
  sleepInterval=0;
  threadInUse=0;
  fragsize=1<<FRAGSIZE;
  controlflags = DSBCAPS_CTRLALL & ~DSBCAPS_GLOBALFOCUS;  

  CloseHandle(DRV_DS_Mutex);
  DRV_DS_Mutex = 0;
}

static void DS_Update(void)
{
  // ResumeThread(updateBufferHandle);
  // updateBufferProc(NULL);
  return;
}

static void DS_PlayStop(void)
{
  
  WaitForSingleObject(DRV_DS_Mutex, INFINITE);
  //threadInUse=0;
  SuspendThread(updateBufferHandle);
  //Sleep(sleepInterval);
  //  if(notifyUpdateHandle && PulseEvent(notifyUpdateHandle) == TRUE && updateBufferHandle)
  ReleaseMutex(DRV_DS_Mutex);
  //  WaitForSingleObject(updateBufferHandle, INFINITE);
  //  CloseHandle(updateBufferHandle);
  
  pSoundBuffer->lpVtbl->Stop(pSoundBuffer);
  VC_PlayStop();

  // set secondary  buffer to 0 to create silence...
  {
    LPVOID Block1, Block2;
    DWORD lBlock1, lBlock2;
    if(pSoundBuffer->lpVtbl->Lock(pSoundBuffer, 0, 0, &Block1, &lBlock1, &Block2, &lBlock2, DSBLOCK_ENTIREBUFFER) == DS_OK){
      fprintf(stderr, "Secondary Locked\n");
      if(lBlock1 && Block1) ZeroMemory(Block1, lBlock1);
      //    if(lBlock2 && Block1) ZeroMemory(Block2, lBlock2);
      pSoundBuffer->lpVtbl->Unlock(pSoundBuffer, Block1, lBlock1, Block2, lBlock2);
    }
  }

}

static BOOL DS_PlayStart(void)
{
  HRESULT r;
  threadInUse=1;
  VC_PlayStart();
  ResumeThread(updateBufferHandle);
  r = pSoundBuffer->lpVtbl->Play(pSoundBuffer,0,0,DSBPLAY_LOOPING);
  fprintf(stderr, "Starting playing.%c\n", ((r==DS_OK)?('Y'):('N')));
  return 0;
}

//MIKMODAPI MDRIVER drv_ds=
MDRIVER drv_ds=
{
  NULL,
  "DirectSound",
  "DirectSound Driver (DX7) v0.1b",
  0,255,
  "ds",
  
  DS_CommandLine,
  DS_IsPresent,
  VC_SampleLoad,
  VC_SampleUnload,
  VC_SampleSpace,
  VC_SampleLength,
  DS_Init,
  DS_Exit,
  NULL,
  VC_SetNumVoices,
  DS_PlayStart,
  DS_PlayStop,
  DS_Update,
  NULL,
  VC_VoiceSetVolume,
  VC_VoiceGetVolume,
  VC_VoiceSetFrequency,
  VC_VoiceGetFrequency,
  VC_VoiceSetPanning,
  VC_VoiceGetPanning,
  VC_VoicePlay,
  VC_VoiceStop,
  VC_VoiceStopped,
  VC_VoiceGetPosition,
  VC_VoiceRealVolume
};

#else

MISSING(drv_ds);

#endif

/* ex:set ts=4: */
