/* -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=- */
/*                                                                   */
/*    AV3D - OS-Independent wrapper with OpenGL support              */
/*    Copyright (c)2001 Galactica Software                           */
/*    ALL RIGHTS RESERVED.                                           */
/*                                                                   */
/*    Programmers: Adam Majer (adamm@galacticasoftware.com)          */
/*                                                                   */
/*    This source code and related compilations are distributed      */
/*    under the terms of LPGL. For license see "license.txt" enclosed*/
/*    or go to "http://www.gnu.org/copyleft/lesser.html"             */
/*
    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
/*                                                                   */
/* -.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.-.- */


/* NOTES: All of the video redefinitions used in the Windows OS       */
/*        under OpenGL are defined here. DirectX is not used because  */
/*        I don't like it! :)                                         */

#include <av3d.hpp>
#include <GL/glu.h>
#include <GL/glx.h>
#include <math.h>

// also defined in keyboard, joystick and mouse, and font
struct OpenGLData{
  Window win;
  Display *disp;
};
const char AV3DApp[] = "AV3D Application";

extern Display *disp;
extern Window CurWindow;

////////////////////////////////////////////////////////////
AV_Video :: AV_Video(const char* a, int b, int c, int d, bool e=false)
{
  /* Initialize video mode to the current specifications
   */
  fSet = false;
  Data = new OpenGLData;

  Data->disp = disp;
  Data->win = 0;
  Font = NULL;
  
  SetVideoMode(a,b,c,d,e);
  Font = new AV_Font(*this);
}
////////////////////////////////////////////////////////////
AV_Video :: ~AV_Video()
{
  /* Kills the AV video */
  if(Font != NULL) delete Font;
  delete Data;
}  
////////////////////////////////////////////////////////////
void AV_Video :: ClearScreen()
{
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  glLoadIdentity();
}
////////////////////////////////////////////////////////////
void AV_Video :: FlipPages()
{ 
  glFinish();
  glXSwapBuffers(disp, Data->win);
}
////////////////////////////////////////////////////////////
OpenGLData const* AV_Video :: GetPrivateData()
{	
  // returns OpenGLData pointer
  return Data;
}
////////////////////////////////////////////////////////////
void AV_Video :: RestoreVideoMode()
{
  if(!fSet) return;
  XUnmapWindow(disp, Data->win);
  XDestroyWindow(disp, Data->win);
  
  fSet = false;
  CurWindow = 0;
}
////////////////////////////////////////////////////////////
void AV_Video :: SetFont(AV_Font &f)
{
  delete Font;
  Font = new AV_Font(f);
}
////////////////////////////////////////////////////////////
void AV_Video :: SetVideoMode(const char*n, int W, int H, int, bool f)
{

  // clears video mode first
  RestoreVideoMode();

  // new video mode set now
  w=W;
  h=H;
  d=0;

  int attrib[] = { GLX_RGBA,
		     GLX_RED_SIZE, 1,
		     GLX_GREEN_SIZE, 1,
		     GLX_BLUE_SIZE, 1,
		     GLX_DOUBLEBUFFER,
		     None };
  int scrnum;
  XSetWindowAttributes attr;
  unsigned long mask;
  Window root;
  GLXContext ctx;
  XVisualInfo *visinfo;
  
  scrnum = DefaultScreen( disp );
  root = RootWindow( disp, scrnum );
  visinfo = glXChooseVisual( disp, scrnum, attrib );
  if (!visinfo) {
    printf("Error: couldn't get an RGB, Double-buffered visual\n");
    exit(0);
  }
  
  /* window attributes */
  attr.background_pixel = 0;
  attr.border_pixel = 0;
  attr.colormap = XCreateColormap( disp, root, visinfo->visual, AllocNone);
  attr.event_mask = StructureNotifyMask | ExposureMask;
  mask = CWBackPixel | CWBorderPixel | CWColormap | CWEventMask;
  
  CurWindow = Data->win = XCreateWindow( disp, root, 0, 0, W, H,
					 0, visinfo->depth, InputOutput,
					 visinfo->visual, mask, &attr );
  
  {
    const char *ProgramName = n;
    XSizeHints sizehints;
    sizehints.x = 0;
    sizehints.y = 0;
    sizehints.width  = W;
    sizehints.height = H;
    sizehints.flags = USSize | USPosition;
    XSetNormalHints(disp, Data->win, &sizehints);
    XSetStandardProperties(disp, Data->win, ProgramName, ProgramName,
			   None, (char **)NULL, 0, &sizehints);
  }
  
  ctx = glXCreateContext( disp, visinfo, NULL, true );
  
  bool fcur = glXMakeCurrent( disp, Data->win, ctx );

  //  if(!ctx || !fcur)
    fprintf(stderr, "Status: CTX-%d   fcur-%d", (int)ctx, (int)fcur);
  fSet = true;

  XMapWindow( disp, Data->win );
}
////////////////////////////////////////////////////////////
void AV_Video :: Write(int x, int y, const char *s, int len)
{
  if(!Font)
    Font = new AV_Font(*this);
  Font->Write(x,y,s,len);
}
////////////////////////////////////////////////////////////
void AV_Video :: WriteFormatted(int x, int y, const char *s, int len)
{
  if(!Font)
    Font = new AV_Font(*this);
  Font->WriteFormatted(x,y,s,len);
}
