/******************************************************************************
 *                                                                            *
 * File:   base64  .c         Version  1.01            Date: 05-MAR-1993      *
 *                                                                            *
 * Copyright (C) 1993 by Kosta Kostis - this is freeware!                     *
 * Written kosta@blues.sub.de (Kosta Kostis)                                  *
 *                                                                            *
 ******************************************************************************
 *                                                                            *
 * Function:                                                                  *
 *        - function handling conversion from/to base64 and binary            *
 *                                                                            *
 ******************************************************************************
 *                                                                            *
 * History:                                                                   *
 *    05-MAR-1993: KK V1.01                                                   *
 *        - added translation table support                                   *
 *    01-FEB-1993: KK V1.00                                                   *
 *        - initial coding                                                    *
 *                                                                            *
 *****************************************************************************/

#include <stdio.h>
#include <string.h>

#include "base64.h"

/******************************************************************************

	base64 character table with pad character appended

******************************************************************************/

static  char    Base64Chars [] =
{
	'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H',         /* 00-07 */
	'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P',         /* 08-0F */
	'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X',         /* 10-17 */
	'Y', 'Z', 'a', 'b', 'c', 'd', 'e', 'f',         /* 18-1F */
	'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n',         /* 20-27 */
	'o', 'p', 'q', 'r', 's', 't', 'u', 'v',         /* 28-2F */
	'w', 'x', 'y', 'z', '0', '1', '2', '3',         /* 30-37 */
	'4', '5', '6', '7', '8', '9', '+', '/',         /* 38-3F */
	'='                                             /* 40    */
} ;

/******************************************************************************

	basic "base64" functions

******************************************************************************/

/******************************************************************************

	Function:
		convert base64 character to base64 value

	Parameter:
		int     ascii           character to be decoded

	Returns:
		BASE64_CODE_INVALID     for an invalid character
		BASE64_CODE_NULL        for the pad character
		0..63                   the base64 value

******************************************************************************/

static  int     CharToB64
(
	int     ascii
)
{
	static  char    *value ;

	/**********************************************************************

		first handle the exception to the rule

	**********************************************************************/

	if (ascii == BASE64_CHAR_PAD)
		return (BASE64_CODE_NULL) ;

	/**********************************************************************

		Now try to figure out the code independend from the
		character encoding used on the machine because there's
		more than ASCII (e. g. EBCDIC) in the world

		I think efficiency is not that important here

	**********************************************************************/

	value = strchr (Base64Chars, ascii) ;
	if (value != (char *) NULL)
		return (value - Base64Chars) ;

	return (BASE64_CODE_INVALID) ;
}

/******************************************************************************

	Function:
		convert base64 value to base64 character

	Parameter:
		int     value           value to be encoded

	Returns:
		BASE64_CODE_INVALID     for an invalid value (not in 0..64)
		int                     the base64 character for the value

******************************************************************************/

static  int     B64ToChar
(
	int     value
)
{
	if ((value < BASE64_CODE_NULL) || (value > BASE64_CODE_PAD))
		return (BASE64_CODE_INVALID) ;

	return ((int) Base64Chars [value]) ;
}

/******************************************************************************

	8-bit octet tripple functions

******************************************************************************/

/******************************************************************************

	Function:
		read 8-bit octet tripple (paded)

	Parameters:
		FILE    *fin            file being read
		int     *tripple        tripple 8-bit octet buffer
		int     *table          pointer to translation table

	Returns:
		BASE64_OK               if all is OK
		EOF                     if EOF has been read

******************************************************************************/

static  int     ReadTripple
(
	FILE    *fin,
	int     *tripple,
	int     *table
)
{
	int     ch ;
	int     i ;

	/**********************************************************************

		first pad the tripple

	**********************************************************************/

	for (i = 0 ; i < BASE64_INTERN ; ++i)
		tripple [i] = BASE64_CODE_INVALID ;

	/**********************************************************************

		now read the actual octet stream

	**********************************************************************/

	for (i = 0 ; i < BASE64_INTERN ; ++i)
	{
		ch = fgetc (fin) ;
		if (ch == EOF)
			return (EOF) ;

		if (table != (int *) NULL)
			ch = table [ch] ;

		tripple [i] = ch ;
	}

	return (BASE64_OK) ;
}

/******************************************************************************

	Function:
		write 8-bit octet tripple converting from qword

	Parameters:
		FILE    *fout   file being written
		char    *qw     quad-word buffer
		int     *table  pointer to translation table

	Returns:
		BASE64_OK       if all is OK
		BASE64_WRITE    if there's a write error

******************************************************************************/

static  int     WriteTripple
(
	FILE    *fout,
	char    *qw,
	int     *table
)
{
	int     b1 ;
	int     b2 ;
	int     b3 ;

	/**********************************************************************

		this logic is hardcoded but that shouldn't harm a lot

	**********************************************************************/

	b1 = (CharToB64 (qw [0]) << 2) | (CharToB64 (qw [1]) >> 4) ;
	b2 = ((CharToB64 (qw [1]) & 0xF) << 4) | (CharToB64 (qw [2]) >> 2) ;
	b3 = ((CharToB64 (qw [2]) & 0x3) << 6) | CharToB64 (qw [3]) ;

	/**********************************************************************

		if a translation table is supplied, use it

	**********************************************************************/

	if (table != (int *) NULL)
	{
		b1 = table [b1] ;
		b2 = table [b2] ;
		b3 = table [b3] ;
	}

	/**********************************************************************

		the first octet is for sure not empty

	**********************************************************************/

	if (fputc (b1, fout) != b1)
		return (BASE64_WRITE) ;

	/**********************************************************************

		the other octets may be paded

	**********************************************************************/

	if ((qw [1] != BASE64_CHAR_PAD) && (qw [2] != BASE64_CHAR_PAD))
		if (fputc (b2, fout) != b2)
			return (BASE64_WRITE) ;

	if ((qw [2] != BASE64_CHAR_PAD) && (qw [3] != BASE64_CHAR_PAD))
		if (fputc (b3, fout) != b3)
			return (BASE64_WRITE) ;

	return (BASE64_OK) ;
}

/******************************************************************************

	base64 qword functions

******************************************************************************/

/******************************************************************************

	Function:
		read a base64 quad-word

	Parameters:
		FILE    *fin    file being read
		char    *qword  qword buffer

	Returns:
		BASE64_OK       if all is OK
		EOF             if EOF has been read

******************************************************************************/

static  int     ReadQWord
(
	FILE    *fin,
	char    *qword
)
{
	int     ch ;
	int     i ;

	/**********************************************************************

		pad qword buffer

	**********************************************************************/

	for (i = 0 ; i < BASE64_EXTERN ; ++i)
		qword [i] = B64ToChar (BASE64_CODE_NULL) ;

	/**********************************************************************

		skip EOLN markers

	**********************************************************************/

	while ((ch = fgetc (fin)) == '\n') ;

	/**********************************************************************

		read a qword - keep in mind the first char is already read

	**********************************************************************/

	for (i = 0 ; i < BASE64_EXTERN ; ++i)
	{
		if (i)                          /* keep first character */
			ch = fgetc (fin) ;

		if ((ch == EOF) || ((i == 0) && (ch == BASE64_CHAR_PAD)))
			return (EOF) ;

		qword [i] = (char) ch ;
	}

	return (BASE64_OK) ;
}

/******************************************************************************

	Function:
		write base64 qword converting from 8-bit octet tripple

	Parameters:
		FILE    *fout           file being written
		int     *tripple        8-bit octet tripple (stored as int)

	Returns:
		BASE64_OK               if all is OK
		BASE64_WRITE            if there's a write error

******************************************************************************/

static  int     WriteQWord
(
	FILE    *fout,
	int     *tripple
)
{
	int     qword   [BASE64_EXTERN] ;
	int     i ;
	int     ch ;

	/**********************************************************************

		first pad qword

	**********************************************************************/

	for (i = 2 ; i < BASE64_EXTERN ; ++i)
		qword [i] = BASE64_CODE_PAD ;

	/**********************************************************************

		the first 8-bit octet is for sure not paded, so convert it

	**********************************************************************/

	qword [0] = tripple [0] >> 2 ;
	qword [1] = (tripple [0] & 0x3) << 4 ;

	/**********************************************************************

		now check the second 8-bit octet

	**********************************************************************/

	if (tripple [1] != BASE64_CODE_INVALID)
	{
		qword [1] |= (tripple [1] >> 4) ;
		qword [2]  = (tripple [1] & 0xF) << 2 ;

		/**************************************************************

			last but not least check the third 8-bit octet

		**************************************************************/

		if (tripple [2] != BASE64_CODE_INVALID)
		{
			qword [2] |= (tripple [2]) >> 6 ;
			qword [3]  = tripple [2] & 0x3F ;
		}
	}

	/**********************************************************************

		now finally convert the base64 values to characters
		and add them to the output stream

	**********************************************************************/

	for (i = 0 ; i < BASE64_EXTERN ; ++i)
	{
		ch = B64ToChar (qword [i]) ;
		if (fputc (ch, fout) != ch)
			return (BASE64_WRITE) ;
	}

	return (BASE64_OK) ;
}

/******************************************************************************

	public functions

******************************************************************************/

/******************************************************************************

	Function:
		convert 8bit to base64

	Parameters:
		FILE    *fin    file being read
		FILE    *fout   file being written
		int     *table  pointer to translation table

	Returns:
		BASE64_OK       if all goes well
		BASE64_WRITE    if there's a write error

******************************************************************************/

int     Copy8BitToBase64
(
	FILE    *fin,
	FILE    *fout,
	int     *table
)
{
	int     tripple [BASE64_INTERN] ;
	int     Status  = BASE64_OK ;
	int     qpl     = 0 ;                   /* qwords per line */

	/**********************************************************************

		read tripples, write qwords - the whole job

	**********************************************************************/

	while (ReadTripple (fin, tripple, table) != EOF)
	{
		Status = WriteQWord (fout, tripple) ;
		if (Status != BASE64_OK)
			return (Status) ;

		/**************************************************************

			automagic line wrap at BASE64_CPL (=chars per line)

		**************************************************************/

		if (++qpl >= BASE64_CPL / 4)    /* 4 chars make a qword */
		{
			qpl = 0 ;
			if (fputc ('\n', fout) != '\n')
				return (BASE64_WRITE) ;
		}
	}

	/**********************************************************************

		the last tripple might be padded, check this out

	**********************************************************************/

	if (tripple [0] != BASE64_CODE_INVALID)
		Status = WriteQWord (fout, tripple) ;

	if (++qpl < BASE64_CPL / 4)
		if (fputc ('\n', fout) != '\n')
			return (BASE64_WRITE) ;

	return (Status) ;
}

/******************************************************************************

	Function:
		convert base64 to 8bit

	Parameters:
		FILE    *fin    file being read
		FILE    *fout   file being written
		int     *table  pointer to translation table

	Returns:
		BASE64_OK       if all goes well
		BASE64_WRITE    if there's a write error

******************************************************************************/

int     CopyBase64To8Bit
(
	FILE    *fin,
	FILE    *fout,
	int     *table
)
{
	char    qword   [BASE64_EXTERN] ;
	int     Status ;

	/**********************************************************************

		read qwords, write tripples - the whole job
		note: qwords are *always* padded, so don't doubt EOF

	**********************************************************************/

	while (ReadQWord (fin, qword) != EOF)
	{
		Status = WriteTripple (fout, qword, table) ;
		if (Status != BASE64_OK)
			return (Status) ;
	}

	return (BASE64_OK) ;
}
