/******************************************************************************
 *                                                                            *
 * File:   decode  .c         Version  1.20            Date: 14-AUG-1993      *
 *                                                                            *
 * Copyright (C) 1993 by Kosta Kostis - this is freeware!                     *
 * Written by kosta@blues.sub.de (Kosta Kostis)                               *
 *                                                                            *
 ******************************************************************************
 *                                                                            *
 * Function:                                                                  *
 *        - convert quoted printables to 8-bit text                           *
 *        - convert base64 to binary                                          *
 *                                                                            *
 ******************************************************************************
 *                                                                            *
 * History:                                                                   *
 *    14-AUG-1993: KK V1.20                                                   *
 *        - optional file name implemented                                    *
 *    11-JUN-1993: KK V1.11                                                   *
 *        - accept unknown Codepages                                          *
 *    29-MAY-1993: KK V1.10                                                   *
 *        - support for ISO 8859-1/2/5/7/9 character sets                     *
 *    21-MAR-1993: KK V1.03                                                   *
 *        - changes due to changes in mime.c                                  *
 *    05-MAR-1993: KK V1.02                                                   *
 *        - included translation table                                        *
 *    01-MAR-1993: KK V1.01                                                   *
 *        - cosmetic changes                                                  *
 *    01-FEB-1993: KK V1.00                                                   *
 *        - initial coding                                                    *
 *                                                                            *
 *****************************************************************************/

#include <stdio.h>
#include <string.h>

#include "rfc.h"
#include "strccmp.h"
#include "readline.h"
#include "mime.h"
#include "patch.h"
#include "rstatic.h"
#include "binary.h"
#include "qp.h"
#include "base64.h"
#include "codepage.h"

#include "decode.h"

/******************************************************************************

	include ISO 8859-x to IBM Code Page y translation tables here

******************************************************************************/

#define BAD             255
#define CHARNUM         256
#include "isox850.tab"
#include "isox852.tab"
#include "isox857.tab"
#include "isox866.tab"
#include "isox869.tab"

static  int     *c_isoxibm      = (int *) NULL ;

/******************************************************************************

	Function:
		display program message

	Parameter:
		none

	Returns:
		nothing

******************************************************************************/

static  void    Hello (void)
{
	fprintf (stderr, MSG_COPYRIGHT) ;
	fprintf (stderr, MSG_AUTHOR) ;
	fprintf (stderr, MSG_VERSION, PRGNAM) ;
}

/******************************************************************************

	Function:
		copy header and interpret MIME-field C-T-E

	Parameters:
		FILE    *fin            file being read
		FILE    *fout           file being written

	Returns:
		MIME_CODE_ERROR

		MIME_CODE_NONE
		MIME_CODE_UNKNOWN

		MIME_CODE_7BIT
		MIME_CODE_8BIT
		MIME_CODE_BINARY

		MIME_CODE_QP
		MIME_CODE_BASE64

******************************************************************************/

static  int     CopyHeader
(
	FILE    *fin,
	FILE    *fout
)
{
	char    Line    [RFC822_LINE_LENGTH + 1] ;
	int     Status ;
	int     Encoding        = MIME_CODE_NONE ;
	char    *EncType ;

	do
	{
		Status = ReadLine (fin, Line, RFC822_LINE_LENGTH) ;
		if (Status == READLINE_LONG)
			return (MIME_CODE_ERROR) ;

		if (*Line != '\0')
		{
			EncType = strccmp (Line, MIME_TAG_ENCODING) ;
			if (EncType != (char *) NULL)
			{
				Encoding = MIME_TagToCode (EncType) ;
				switch (Encoding)
				{
					case    MIME_CODE_BASE64:
					case    MIME_CODE_QP:

						sprintf (Line,
							MIME_HEAD_ENCODING,
							MIME_TAG_ENCODING,
							MIME_TAG_8BIT) ;
						break ;
				}
			}

			fprintf (fout, "%s\n", Line) ;
		}
	}
	while ((*Line != '\0') && (Status == READLINE_OK)) ;

	return (Encoding) ;
}

/*****************************************************************************/

int     main
(
	int     argc,
	char	*argv	[]
)
{
	FILE	*SrcFile	= stdin ;
	FILE	*DstFile	= stdout ;
	char	TmpName	[MAX_CMDLEN + 1] ;
	char	*Help ;
	int     Encoding ;
	int     Status ;
	char    *Translate ;
	int	cp ;

	/**********************************************************************

		first, check the syntax

	**********************************************************************/

	if (argc > 2)
	{
		Hello () ;
		fprintf (stderr, ERR_MSG_SYNTAX, PRGNAM) ;
		return (ERR_SYNTAX) ;
	}

	/**********************************************************************

		open / create files if necessary

	**********************************************************************/

	if (argc > 1)
	{
		strcpy (TmpName, argv [1]) ;

		Help = strrchr (TmpName, '\\') ;
		if (Help != (char *) NULL)
			*++Help = '\0' ;
		else
			strcpy (TmpName, "") ;

		strcat (TmpName, tmpnam ((char *) NULL)) ;

		SrcFile = fopen (argv [1], "r") ;
		if (SrcFile == (FILE *) NULL)
		{
			fprintf (stderr, ERR_MSG_OPEN, PRGNAM, argv [1]) ;
			return (ERR_OPEN) ;
		}

		DstFile = fopen (TmpName, "w") ;
		if (DstFile == (FILE *) NULL)
		{
			fprintf (stderr, ERR_MSG_CREATE, PRGNAM, TmpName) ;
			fclose (SrcFile) ;
			return (ERR_CREATE) ;
		}
	}

	/**********************************************************************

		determine current IBM Code Page (and ISO 8859 equivalent)

	**********************************************************************/

	cp = CodePage () ;
	switch (cp)
	{
		case	437:
		case	850:
			c_isoxibm = c_isox850 ;
			break ;

		case	852:
			c_isoxibm = c_isox852 ;
			break ;

		case	857:
			c_isoxibm = c_isox857 ;
			break ;

		case	866:
			c_isoxibm = c_isox866 ;
			break ;

		case	869:
			c_isoxibm = c_isox869 ;
			break ;

		default:
			c_isoxibm = (int *) NULL ;
	}

	/**********************************************************************

		shall we translate?

	**********************************************************************/

	Translate = rstatic (STATIC_MIME_ISOxIBM) ;
	if (Translate != (char *) NULL)
		if (strcmp (Translate, VAL_NO) == 0)
			c_isoxibm = (int *) NULL ;

	/**********************************************************************

		first we copy and interpret the original header

	**********************************************************************/

	Encoding = CopyHeader (SrcFile, DstFile) ;

	switch (Encoding)
	{
		case    MIME_CODE_QP:

			fprintf (DstFile, "X-Remark: %s converted to %s\n\n",
				MIME_CodeToTag (MIME_CODE_QP),
				MIME_CodeToTag (MIME_CODE_8BIT)) ;
			Status = CopyQPTo8Bit (SrcFile, DstFile, c_isoxibm) ;
			break ;

		case    MIME_CODE_BASE64:

			fprintf (DstFile, "X-Remark: %s converted to %s\n\n",
				MIME_CodeToTag (MIME_CODE_BASE64),
				MIME_CodeToTag (MIME_CODE_8BIT)) ;
			Status = CopyBase64To8Bit (SrcFile, DstFile, c_isoxibm) ;
			break ;

		default:

			fputc ('\n', DstFile) ;
			Status = CopyBinary (SrcFile, DstFile, c_isoxibm) ;
			break ;
	}

	if (Status)
		fprintf (stderr, ERR_MSG_WRITE, PRGNAM) ;

	if (argc > 1)
	{
		fclose (SrcFile) ;
		fclose (DstFile) ;

		if (remove (argv [1]))
		{
			fprintf (stderr, ERR_MSG_REMOVE, PRGNAM, argv [1]) ;
			return (ERR_REMOVE) ;
		}

		if (rename (TmpName, argv [1]))
		{
			fprintf (stderr, ERR_MSG_RENAME,
				PRGNAM, TmpName, argv [1]) ;
			return (ERR_RENAME) ;
		}
	}

	return (Status) ;
}
